//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package com.dstc.security.x509.extns;

import com.dstc.security.asn1.Asn1;
import com.dstc.security.asn1.Sequence;
import com.dstc.security.asn1.Asn1Exception;
import com.dstc.security.asn1.Oid;
import com.dstc.security.asn1.Boolean;
import com.dstc.security.asn1.OctetString;
import com.dstc.security.asn1.BitString;
import com.dstc.security.provider.OID;
import com.dstc.security.x509.Extension;

/**
 * <p>Class representing the ASN.1 structure BasicConstraints.
 *
 * <pre>
 *     BasicConstraints ::= SEQUENCE {
 *          cA                      BOOLEAN DEFAULT FALSE,
 *          pathLenConstraint       INTEGER (0..MAX) OPTIONAL }
 * </pre>
 *
 * @author Ming Yung
 */ 
public class BasicConstraints extends Extension
{
  private IBasicConstraints constr;

  /**
   * Constructs a BasicConstraints extension
   */
  public BasicConstraints(boolean isCritical, boolean isCA, int len)
  {
    this.extnID = OID.basicConstraints;
    this.addComponent(new Oid(OID.basicConstraints));
 
    if (isCritical)
    {
      this.critical = isCritical;
      this.addComponent(new Boolean(1));
    }

    this.constr = new IBasicConstraints(isCA, len);
    this.addComponent(new OctetString(constr.encode()));
  }

  /**
   * Constructs a BasicConstraints extension from a DER encoding
   */
  public BasicConstraints(byte encoded[]) throws Asn1Exception
  {
    super(encoded);
    this.constr = new IBasicConstraints(this.value);
  }

  public String toString()
  {
    StringBuffer sb = new StringBuffer();
    sb.append("BasicConstraints extension:\n");

    if (constr.isCA)
      sb.append("  CA, ");

    sb.append("Path Length Constraint = ");
    if (constr.len > 0)
      sb.append(constr.len + "\n");
    else
      sb.append("unspecified\n");
    
    return sb.toString();
  }

  //Inner class
  protected class IBasicConstraints extends Sequence
  {
    protected int len = -1;
    protected boolean isCA = false;

    protected IBasicConstraints(boolean isCA, int len)
    {
      if (isCA)
      {
        this.isCA = true;
        addComponent(new Boolean(1));
      }

      if (len > 0)
      {
       this.len = len;
       addComponent(new com.dstc.security.asn1.Integer(len));
      }
    }

    protected IBasicConstraints(byte[] encoded) throws Asn1Exception
    {
      doDecode(encoded);
      int i=0;

      if (i >= components.size())
        return;

      try
      {
        this.isCA = ((Boolean)components.elementAt(i)).getTruth();
        i++;
      }
      catch (ClassCastException e)
      {
        //Ignore
      }

      if (i >= components.size())
        return;

      this.len 
        = ((com.dstc.security.asn1.Integer)components.elementAt(i)).getInt();
    }
  }
}
