//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package javax.crypto.spec;

import java.security.InvalidKeyException;
import java.security.spec.KeySpec;

/**
 * <p>This class specifies a DES key. 
 *
 * @see java.security.Key
 * @see java.security.KeyFactory
 * @see java.security.spec.KeySpec
 *
 * @version 0.98, 98/07/01
 * @author Ming Yung
 *
 */
public class DESKeySpec implements KeySpec
{
  private byte keyBytes[];

  /**
   * Uses the first 8 bytes in key as the key material for the DES key.
   *
   * @param key the buffer with the DES key material.
   *
   * @exception java.security.InvalidKeyException if the given key
   * material is shorter than 8 bytes.
   */
  public DESKeySpec(byte[] key) throws InvalidKeyException
  {
    if (key.length < 8) throw new InvalidKeyException("Key too short");

    keyBytes = new byte[8];
    System.arraycopy(key, 0, keyBytes, 0, 8);
  }

  /**
   * Uses the first 8 bytes in key, beginning at offset, as the key 
   * material for the DES key.
   *
   * @param key the buffer with the DES key material.
   * @param offset the offset in key, where the DES key material starts.
   *
   * @exception java.security.InvalidKeyException if the given key
   * material is shorter than 8 bytes.
   */
  public DESKeySpec(byte[] key, int offset) throws InvalidKeyException
  {
    if (key.length - offset < 8) 
      throw new InvalidKeyException("Key too short");

    keyBytes = new byte[8];
    System.arraycopy(key, offset, keyBytes, 0, 8);
  }

  /**
   * Returns the DES key material.
   *
   * @return the DES key material.
   */
  public byte[] getKey()
  {
    return keyBytes;
  }

  /**
   * Checks if the given DES key material, starting at offset, 
   * is parity-adjusted.
   *
   * @return true if the given DES key material is parity-adjusted,
   * false otherwise.
   *
   * @exception java.security.InvalidKeyException if the given key
   * material, starting at offset, is shorter than 8 bytes.
   */
  public static boolean isParityAdjusted(byte[] key, int offset) 
    throws InvalidKeyException
  {
    int score;
    for (int j=offset; j < key.length; j++)
    {
      score = 0;
      for (int i=7; i>=0; i--)
      {
        score += (key[j] >>> i) & 0x1;
      }
      if (0 == score % 2) return false;
    }
    return true;
  }

  /**
   * Checks if the given DES key material is weak or semi-weak.
   *
   * @return true if the given DES key material is weak or semi-weak,
   * false otherwise.
   *
   * @exception java.security.InvalidKeyException if the given key
   * material, starting at offset, is shorter than 8 bytes.
   */
  public static boolean isWeak(byte[] key, int offset) 
    throws InvalidKeyException
  {
    return true;
  }
}
