//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package javax.net;

import java.io.*;
import java.net.*;

/**
 * <p> This class creates sockets. It may be subclassed by other factories,
 * which create particular subclasses of sockets and thus provide a
 * general framework for the addition of public socket-level
 * functionality. This is done flexibly, without affecting JVM-wide
 * security policy or involving an API targetted at C code (SocketImpl)
 * for standard socket functionality.
 *
 * <p> Socket factories are a simple way to capture a variety of policies
 * related to the sockets being constructed, producing such sockets in a
 * way which does not require special configuration of the code which
 * asks for the sockets:
 *   * Due to polymorphism of both factories and sockets, different kinds
 *     of sockets can be used by the same application code just by
 *     passing it different kinds of factories.
 *   * Factories can themselves be customized with parameters used in
 *     socket construction. So for example, factories could be customized
 *     to return sockets with different networking timeouts or security
 *     parameters already configured.
 *   * The sockets returned to the application can be subclasses of
 *     java.net.Socket, so that they can directly expose new APIs for
 *     features such as compression, security, record marking, statistics
 *     collection, or firewall tunneling.
 *
 * <p> Factory classes are specified by environment-specific configuration
 * mechanisms. For example, the getDefault method could return a factory
 * that was appropriate for a particular user or applet, and a framework
 * could use a factory customized to its own purposes.
 *
 * @version 0.98, 98/09/10
 * @author Ming Yung
 *
 */

public abstract class SocketFactory 
{
  /**
   * Constructor is used only by subclasses.
   */
  protected SocketFactory()
  {
  }

  /**
   * Returns a copy of the environment's default socket factory.
   */
  public static SocketFactory getDefault()
  {
    return null;
  }

  /**
   * Returns a socket connected to a ServerSocket on the named host, at the
   * given port. This socket is configured using the socket options
   * established for this factory.
   *
   * @param host  the server host
   * @param port  the server port
   *
   * @throw IOException if the connection can't be established
   * @throw UnknownHostException if the host is not known
   */
  public abstract Socket createSocket(String host, int port) 
    throws IOException, UnknownHostException;

  /**
   * Returns a socket connected to a ServerSocket on the named host, at the
   * given port. The client address address is the specified host
   * and port. This socket is configured using the socket options
   * established for this factory.
   *
   * @param host  the server host
   * @param port  the server port
   * @param clientHost  the client host
   * @param clientPort  the client port
   *
   * @throw IOException if the connection can't be established
   * @throw UnknownHostException if the host is not known
   */
  public abstract Socket createSocket(String host, int port,
                                      InetAddress clientHost, int clientPort) 
    throws IOException, UnknownHostException;

  /**
   * Returns a socket connected to a ServerSocket at the specified network
   * address and port. This socket is configured using the socket
   * options established for this factory.
   *
   * @param host  the server host
   * @param port  the server port
   *
   * @throw IOException if the connection can't be established
   */
  public abstract Socket createSocket(InetAddress host, int port) 
    throws IOException;

  /**
   * Returns a socket connected to a ServerSocket at the specified network
   * address and port. The client is bound to the specified network
   * address and port, and the socket is configured using the socket
   * options established for this factory.
   * 
   * @param address  the server network address
   * @param port  the server port
   * @param clientAddress  the client network address
   * @param clientPort  the client port
   *
   * @throw IOException if the connection can't be established
   */
  public abstract Socket createSocket(InetAddress address, int port,
                                     InetAddress clientAddress,
                                     int clientPort) 
    throws IOException;

}
