%{

// **********************************************************************
//
// Copyright (c) 1999
// Object Oriented Concepts, Inc.
// Billerica, MA, USA
//
// All Rights Reserved
//
// **********************************************************************

#include <OB/CORBA.h>
#include <Types.h>
#include <Util.h>
#include <GramUtil.h>

#include <stdlib.h>
#include <ctype.h>

#ifdef WIN32
#   include <malloc.h>
#ifndef BORLAND
#   define alloca _alloca
#endif
#endif

#ifdef __DECCXX
#   include <alloca.h>
#endif

void yyerror(const char* s);
int yylex();

static void TranslateAny(const CORBA_Any&,
			 CORBA_ULong&, CORBA_Boolean&,
			 CORBA_Long&, CORBA_Boolean&,
			 CORBA_Double&, CORBA_Boolean&);

//
// The current line number and the filename stack
//
long IdlLineno;
IdlStringSeq IdlFileStack;

//
// The current documentation comment
//
CORBA_String_var IdlDocComment;

//
// The ORB
//
CORBA_ORB_ptr IdlORB;

//
// The repository that is to be feeded
//
CORBA_Repository_ptr IdlRepository;

//
// All repository ids of all definitions in the top-level IDL file
//
CORBA_RepositoryIdSeq* IdlTopFileContent;

//
// All IDL files that are directly included from the top-level IDL file
//
IdlStringSeq* IdlIncludes;

//
// All comments for all repository ids
//
IdlCommentSeq* IdlComments;

//
// The stack for the containers
//
IdlContainerSeq IdlContStack;

//
// A stack of stacks for the prefixes
// 
IdlStringSeqSeq IdlPrefStackStack;

//
// This is needed for unions
//
CORBA_IDLType_ptr IdlUnionSwitchType;

//
// Sequence with interface that have been forward declared
//
IdlStringSeq IdlFwdInterfaces;

%}

//
// Expect one shift/reduce
//
%expect 1

/*** Return type ***/

%union
{
    //
    // General types
    //
    char* string;
    IdlStringSeq* stringSeq;

    CORBA_Long longVal;
    CORBA_ULong uLongVal;
    CORBA_Char charVal;
    CORBA_Double doubleVal;
    CORBA_Boolean booleanVal;

    CORBA_Any* any;
    IdlAnySeq* anySeq;

    //
    // Special types
    //
    IdlDeclarator* declarator;
    IdlDeclaratorSeq* declaratorSeq;

    //
    // Types from the IR
    //
    CORBA_IDLType_ptr idlType;
    CORBA_StructDef_ptr structDef;
    CORBA_UnionDef_ptr unionDef;
    CORBA_ExceptionDef_ptr exceptionDef;
    CORBA_AttributeMode attributeMode;
    CORBA_StructMemberSeq* structMemberSeq;
    CORBA_UnionMemberSeq* unionMemberSeq;
    CORBA_UnionMember* unionMember;
    CORBA_InterfaceDefSeq* interfaceDefSeq;
    CORBA_OperationMode operationMode;
    CORBA_ParameterMode parameterMode;
    CORBA_ParameterDescription* parameterDescription;
    CORBA_ParDescriptionSeq* parDescriptionSeq;
    CORBA_ExceptionDefSeq* exceptionDefSeq;
};

/*** Token definitions ***/

%token				IDL_ANY
%token				IDL_ATTRIBUTE
%token				IDL_BOOLEAN
%token				IDL_CASE
%token				IDL_CHAR
%token				IDL_CONST
%token				IDL_CONTEXT
%token				IDL_DEFAULT
%token				IDL_DOUBLE
%token				IDL_ENUM
%token				IDL_EXCEPTION
%token				IDL_FLOAT
%token				IDL_IN
%token				IDL_INOUT
%token				IDL_INTERFACE
%token				IDL_LONG
%token				IDL_MODULE
%token				IDL_OCTET
%token				IDL_ONEWAY
%token				IDL_OUT
%token				IDL_RAISES
%token				IDL_READONLY
%token				IDL_SEQUENCE
%token				IDL_SHORT
%token				IDL_STRING
%token				IDL_STRUCT
%token				IDL_SWITCH
%token				IDL_TYPEDEF
%token				IDL_UNSIGNED
%token				IDL_UNION
%token				IDL_VOID
%token				IDL_LEFT_SHIFT
%token				IDL_RIGHT_SHIFT
%token				IDL_SCOPE_DELIMITOR

%token <string>			IDL_IDENTIFIER
%token <uLongVal>		IDL_INTEGER_LITERAL
%token <string>			IDL_STRING_LITERAL
%token <charVal>		IDL_CHARACTER_LITERAL
%token <doubleVal>		IDL_FLOATING_PT_LITERAL
%token <booleanVal>		IDL_FALSE
%token <booleanVal>		IDL_TRUE

%token <string>			IDL_PRAGMA

/*** Types definitions ***/

%type <any>			literal
%type <booleanVal>		boolean_literal
%type <stringSeq>		COMMA_string_literal_STAR
%type <string>			scoped_name
%type <stringSeq>		COMMA_scoped_name_STAR
%type <string>			enumerator
%type <stringSeq>		COMMA_enumerator_STAR
%type <uLongVal>		positive_int_const
%type <uLongVal>		fixed_array_size
%type <interfaceDefSeq>		inheritance_spec
%type <interfaceDefSeq>		inheritance_spec_OPT

%type <idlType>			type_spec
%type <idlType>			const_type
%type <idlType>			simple_type_spec
%type <idlType>			param_type_spec
%type <idlType>			base_type_spec
%type <idlType>			template_type_spec
%type <idlType>			constr_type_spec
%type <idlType>			floating_pt_type
%type <idlType>			integer_type
%type <idlType>			signed_int
%type <idlType>			signed_long_int
%type <idlType>			signed_short_int
%type <idlType>			unsigned_int
%type <idlType>			unsigned_long_int
%type <idlType>			unsigned_short_int
%type <idlType>			char_type
%type <idlType>			boolean_type
%type <idlType>			octet_type
%type <idlType>			any_type
%type <idlType>			sequence_type
%type <idlType>			string_type
%type <structDef>		struct_type
%type <unionDef>		union_type
%type <idlType>			enum_type
%type <idlType>			switch_type_spec
%type <idlType>			op_type_spec

%type <attributeMode>		readonly_OPT

%type <declaratorSeq>		declarators
%type <declaratorSeq>		COMMA_declarator_STAR
%type <declarator>		declarator
%type <declarator>		simple_declarator
%type <declaratorSeq>		COMMA_simple_declarator_STAR
%type <declarator>		complex_declarator
%type <declarator>		array_declarator
%type <declarator>		fixed_array_size_PLUS

%type <structMemberSeq>		member_list
%type <structMemberSeq>		member_PLUS
%type <structMemberSeq>		member_STAR
%type <structMemberSeq>		member
%type <structMemberSeq>		member_NO_SEM

%type <unionMemberSeq>		switch_body
%type <unionMemberSeq>		case_PLUS
%type <unionMemberSeq>		case
%type <unionMemberSeq>		case_NO_SEM
%type <anySeq>			case_label_PLUS
%type <any>			case_label
%type <unionMember>		element_spec

%type <operationMode>		op_attribute
%type <operationMode>		op_attribute_OPT
%type <parDescriptionSeq>	COMMA_param_dcl_STAR
%type <parDescriptionSeq>	parameter_dcls
%type <parameterDescription>	param_dcl
%type <parameterMode>		param_attribute

%type <exceptionDefSeq>		raises_expr
%type <exceptionDefSeq>		raises_expr_OPT

%type <stringSeq>		context_expr
%type <stringSeq>		context_expr_OPT

%type <any>			const_exp
%type <any>			or_expr
%type <any>			xor_expr
%type <any>			and_expr
%type <any>			shift_expr
%type <any>			add_expr
%type <any>			mult_expr
%type <any>			unary_expr
%type <charVal>			unary_operator
%type <any>			primary_expr

%%

start		: specification
		{
		}
		| /* empty */
		{
		}
		;

specification	:
                {
		    //
		    // Push repository on the container stack
		    //
		    IdlContStack.append(CORBA_Repository::
					_duplicate(IdlRepository));

		    //
		    // Push new empty prefix on the prefix stacks
		    //
		    for(CORBA_ULong i = 0 ; i < IdlPrefStackStack.length() ;
			i++)
			IdlPrefStackStack[i].append(CORBA_string_dup(""));
                }
                definition_PLUS
		{
		    //
		    // Pop container from the stack
		    //
		    assert(IdlContStack.length() > 0);
		    IdlContStack.length(IdlContStack.length() - 1);
		    assert(yynerrs > 0 || IdlContStack.length() == 0);

		    //
		    // Pop prefix from the prefix stacks
		    //
		    for(CORBA_ULong i = 0 ; i < IdlPrefStackStack.length() ;
			i++)
		    {
			assert(IdlPrefStackStack[i].length() > 0);
			IdlPrefStackStack[i].length(
			    IdlPrefStackStack[i].length() - 1);
			assert(yynerrs > 0 ||
			       IdlPrefStackStack[i].length() == 0);
		    }
		}
		;

definition_PLUS	: definition
		{
		}
		| definition definition_PLUS
		{
		}
		;

definition	: definition_NO_SEM ';'
		{
		}
		| definition_NO_SEM
		{
		    IdlError("`;' missing after definition");
		    yynerrs++;
		}
		| error ';'
		{
		    yyerrok;
		}
		| IDL_PRAGMA
		{
		    CORBA_String_var s = $1;
		    if(IdlParsePragma(s) == false)
		    {
			yynerrs++;
		    }
		}
		;

definition_NO_SEM: type_dcl
		{
		}
		| const_dcl
		{
		}
		| except_dcl
		{
		}
		| interface
		{
		}
		| module
		{
		}
		;

module		: IDL_MODULE IDL_IDENTIFIER
		{
		    //
		    // Get container from container stack
		    //
		    assert(IdlContStack.length() > 0);
		    CORBA_Container_var container =
			IdlContStack[IdlContStack.length() - 1];

		    //
		    // Get name, id and version
		    //
		    CORBA_Identifier_var name = $2;
		    CORBA_RepositoryId_var id = IdlGetRepositoryId(name);
		    CORBA_VersionSpec_var version = CORBA_string_dup("1.0");

		    //
		    // Check if name already exists
		    //
		    CORBA_Contained_var c;
		    try
		    {
			c = container -> lookup(name);
		    }
		    catch(const CORBA_INTF_REPOS& ex)
		    {
			IdlPrintException(ex.minor(), id, name);
			yynerrs++;
			YYERROR;
		    }

		    CORBA_ModuleDef_var module;
		    if(!CORBA_is_nil(c))
		    {
			//
			// Name exists, check if it is an module declaration
			//
			module = CORBA_ModuleDef::_narrow(c);

			if(CORBA_is_nil(module))
			{	
			    CORBA_String_var err = CORBA_string_dup("`");
			    err += name;
			    err += "' is not a module";
			    IdlError(err);
			    yynerrs++;
			    YYERROR;
			}
		    }
		    else
		    {
			//
			// Create new module
			//
			try
			{
			    module =
				container -> create_module(id, name, version);
			}
			catch(const CORBA_INTF_REPOS& ex)
			{
			    IdlPrintException(ex.minor(), id, name);
			    yynerrs++;
			    YYERROR;
			}
		    }

		    assert(!CORBA_is_nil(module));

		    //
		    // Push new container on the stack
		    //
		    IdlContStack.append(CORBA_ModuleDef::_duplicate(module));

		    //
		    // Push new prefix on the prefix stacks
		    //
		    for(CORBA_ULong i = 0 ; i < IdlPrefStackStack.length() ;
			i++)
		    {
			CORBA_String_var prefix =
			    IdlPrefStackStack[i][IdlPrefStackStack[i].length()
						- 1];
			prefix += name;
			prefix += "/";
			IdlPrefStackStack[i].append(prefix);
		    }

		    //
		    // Add ID to IdlTopFileContent and IdlComments
		    //
		    IdlAddID(id);
		}
		'{' definition_PLUS '}'
		{
		    //
		    // Pop container from the stack
		    //
		    assert(IdlContStack.length() > 0);
		    IdlContStack.length(IdlContStack.length() - 1);

		    //
		    // Pop prefix from the prefix stacks
		    //
		    for(CORBA_ULong i = 0 ; i < IdlPrefStackStack.length() ;
			i++)
		    {
			assert(IdlPrefStackStack[i].length() > 0);
			IdlPrefStackStack[i].length(
			    IdlPrefStackStack[i].length() - 1);
		    }
		}
		;

interface	: interface_dcl
		{
		}
		| forward_dcl
		{
		}
		;

interface_dcl	: interface_header '{' interface_body '}'
		{
		    //
		    // Pop container from the stack
		    // Pop prefixes from the stack
		    //
		    assert(IdlContStack.length() > 0);
		    IdlContStack.length(IdlContStack.length() - 1);

		    //
		    // Pop prefix from the prefix stacks
		    //
		    for(CORBA_ULong i = 0 ; i < IdlPrefStackStack.length() ;
			i++)
		    {
			assert(IdlPrefStackStack[i].length() > 0);
			IdlPrefStackStack[i].length(
			    IdlPrefStackStack[i].length() - 1);
		    }
		}
		;

forward_dcl	: IDL_INTERFACE IDL_IDENTIFIER
		{
		    //
		    // Get container from container stack
		    //
		    assert(IdlContStack.length() > 0);
		    CORBA_Container_var container =
			IdlContStack[IdlContStack.length() - 1];

		    //
		    // Get name, id and version
		    //
		    CORBA_Identifier_var name = $2;
		    CORBA_RepositoryId_var id = IdlGetRepositoryId(name);
		    CORBA_VersionSpec_var version = CORBA_string_dup("1.0");

		    //
		    // Check if name already exists
		    //
		    CORBA_Contained_var c;
		    try
		    {
			c = container -> lookup(name);
		    }
		    catch(const CORBA_INTF_REPOS& ex)
		    {
			IdlPrintException(ex.minor(), id, name);
			yynerrs++;
			YYERROR;
		    }

		    if(!CORBA_is_nil(c))
		    {
			//
			// Check for ID equivalence
			//
// 			CORBA_RepositoryId_var oldId = c -> id();
// 			if(strcmp(oldId, id) != 0)
// 			{
// 			    CORBA_String_var err =
// 				CORBA_string_dup("conflicting repository ids "
// 						 "for `interface ");
// 			    err += name;
// 			    err += "'";
// 			    IdlError(err);
// 			    yynerrs++;
// 			    YYERROR;
// 			}

			//
			// Name exists, check if it is an interface
			//
			CORBA_InterfaceDef_var i =
			    CORBA_InterfaceDef::_narrow(c);

			if(CORBA_is_nil(i))
			{
			    CORBA_String_var err =
				CORBA_string_dup("conflicting types for "
						 "`interface ");
			    err += name;
			    err += "'";
			    IdlError(err);
			    yynerrs++;
			    YYERROR;
			}
		    }
		    else
		    {
			//
			// Create "empty" interface as forward
			// declaration, do not put it on the container
			// stack, but put it in the list of forward
			// declared interfaces
			//
			CORBA_InterfaceDefSeq seq;

			try
			{
			    CORBA_InterfaceDef_var interf =
				container -> create_interface(id, name,
							      version, seq);
			    IdlFwdInterfaces.append(interf -> absolute_name());
		        }
			catch(const CORBA_INTF_REPOS& ex)
			{
			    IdlPrintException(ex.minor(), id, name);
			    yynerrs++;
			    YYERROR;
			}
		    }
		}
		;

interface_header : IDL_INTERFACE IDL_IDENTIFIER inheritance_spec_OPT
		{
		    //
		    // Get container from container stack
		    //
		    assert(IdlContStack.length() > 0);
		    CORBA_Container_var container =
			IdlContStack[IdlContStack.length() - 1];

		    //
		    // Get name, id and version
		    //
		    CORBA_Identifier_var name = $2;
		    CORBA_RepositoryId_var id = IdlGetRepositoryId(name);
		    CORBA_VersionSpec_var version = CORBA_string_dup("1.0");

		    //
		    // Get inheritance specification sequence
		    //
		    CORBA_InterfaceDefSeq_var seq = $3;
		    
		    //
		    // Check if name already exists
		    //
		    CORBA_Contained_var c;
		    try
		    {
			c = container -> lookup(name);
		    }
		    catch(const CORBA_INTF_REPOS& ex)
		    {
			IdlPrintException(ex.minor(), id, name);
			yynerrs++;
			YYERROR;
		    }

		    CORBA_InterfaceDef_var interf;
		    if(!CORBA_is_nil(c))
		    {
			//
			// Check for ID equivalence
			//
// 			CORBA_RepositoryId_var oldId = c -> id();
// 			if(strcmp(oldId, id) != 0)
// 			{
// 			    CORBA_String_var err =
// 				CORBA_string_dup("conflicting repository ids "
// 						 "for `interface ");
// 			    err += name;
// 			    err += "'";
// 			    IdlError(err);
// 			    yynerrs++;
// 			    YYERROR;
// 			}

			//
			// Name exists, check if it is an interface
			// forward declaration
			//
			interf = CORBA_InterfaceDef::_narrow(c);
			
			if(CORBA_is_nil(interf))
			{	
			    CORBA_String_var err =
				CORBA_string_dup("conflicting types for "
						 "`interface ");
			    err += name;
			    err += "'";
			    IdlError(err);
			    yynerrs++;
			    YYERROR;
			}

			//
			// Don't use ID for comparison, because it's
			// possible that the ID was changed with
			// #pragma
			//
			CORBA_ScopedName_var abs = interf -> absolute_name();
			CORBA_ULong i;
			for(i = 0 ; i < IdlFwdInterfaces.length() ; i++)
			{
			    if(strcmp(abs, IdlFwdInterfaces[i]) == 0)
				break;
			}

			if(i >= IdlFwdInterfaces.length())
			{
			    CORBA_String_var err =
				CORBA_string_dup("redefinition of "
						 "`interface ");
			    err += name;
			    err += "'";
			    IdlError(err);
			    yynerrs++;
			    YYERROR;
			}
			
			IdlFwdInterfaces.remove(i);

			try
			{
			    //
			    // Fill interface
			    //
			    interf -> id(id);
			    // Not necessary, move() sets new name and
			    // version:
			    //interf -> name(name);
			    //interf -> version(version);
			    interf -> base_interfaces(seq);
			    
			    //
			    // Move interface (this will make this interface
			    // the last entry in the container)
			    //
			    interf -> move(container, name, version);
			}
			catch(const CORBA_INTF_REPOS& ex)
			{
			    IdlPrintException(ex.minor(), id, name);
			    yynerrs++;
			    YYERROR;
			}
		    }
		    else
		    {
			//
			// Push new container on the stack
			//
			try
			{
			    interf = container ->
				create_interface(id, name, version, seq);
			}
			catch(const CORBA_INTF_REPOS& ex)
			{
			    IdlPrintException(ex.minor(), id, name);
			    yynerrs++;
			    YYERROR;
			}
		    }
		    
		    //
		    // Push filled interface on the stack
		    //
		    IdlContStack.append(CORBA_InterfaceDef::
					_duplicate(interf));
		    
		    //
		    // Push new prefix on the prefix stacks
		    //
		    for(CORBA_ULong i = 0 ; i < IdlPrefStackStack.length() ;
			i++)
		    {
			CORBA_String_var prefix =
			    IdlPrefStackStack[i][IdlPrefStackStack[i].length()
						- 1];
			prefix += name;
			prefix += "/";
			IdlPrefStackStack[i].append(prefix);
		    }
		    
		    //
		    // Add ID to IdlTopFileContent and IdlComments
		    //
		    IdlAddID(id);
		}
		;

inheritance_spec_OPT: inheritance_spec
		{
		    $$ = $1;
		}
		| /* empty */
		{
		    $$ = new CORBA_InterfaceDefSeq;
		}
		;

interface_body	: export_STAR
		{
		}
		;

export_STAR	: export export_STAR
		{
		}
		| /* empty */
		{
		}
		;

export		: export_NO_SEM ';'
		{
		}
		| export_NO_SEM
		{
		    IdlError("`;' missing after definition");
		    yynerrs++;
		}
		| error ';'
		{
		    yyerrok;
		}
		| IDL_PRAGMA
		{
		    CORBA_String_var s = $1;
		    if(!IdlParsePragma(s))
		    {
			yynerrs++;
		    }
		}
		;

export_NO_SEM	: type_dcl
		{
		}
		| const_dcl
		{
		}
		| except_dcl
		{
		}
		| attr_dcl
		{
		}
		| op_dcl
		{
		}
		;

inheritance_spec: ':' scoped_name COMMA_scoped_name_STAR
		{
		    IdlStringSeq_var scopedNameSeq = $3;
		    scopedNameSeq -> insert($2);
		
		    //
		    // Get container from container stack
		    //
		    assert(IdlContStack.length() > 0);
		    CORBA_Container_var container =
			IdlContStack[IdlContStack.length() - 1];

		    //
		    // Create inheritance specification sequence
		    //
		    CORBA_InterfaceDefSeq_var iDefSeq =
			new CORBA_InterfaceDefSeq(scopedNameSeq -> length());

		    for(CORBA_ULong i = 0 ;
			i < scopedNameSeq -> length() ;	i++)
		    {
			//
			// Find scoped name
			//
			const char* scoped = scopedNameSeq[i];
			CORBA_IRObject_var irObj =
			    IdlFindScopedName(container, scoped);

			if(CORBA_is_nil(irObj))
			{
			    yynerrs++;
			    YYERROR;
			}
			
			while(true)
			{
			    CORBA_AliasDef_var aliasDef =
				CORBA_AliasDef::_narrow(irObj);
			    
			    if(CORBA_is_nil(aliasDef))
				break;
			    
			    irObj = aliasDef -> original_type_def();
			}

			CORBA_InterfaceDef_var interfaceDef =
			    CORBA_InterfaceDef::_narrow(irObj);

			if(CORBA_is_nil(interfaceDef))
			{	
			    CORBA_String_var err = CORBA_string_dup("`");
			    err += scoped;
			    err += "' is not an interface type";
			    IdlError(err);
			    yynerrs++;
			    YYERROR;
			}

			CORBA_ScopedName_var abs =
			    interfaceDef -> absolute_name();
			for(CORBA_ULong j = 0 ;
			    j < IdlFwdInterfaces.length() ; j++)
			{
			    if(strcmp(abs, IdlFwdInterfaces[j]) == 0)
			    {
				CORBA_String_var err =
				    CORBA_string_dup("can't inherit from "
						     "undefined interface `");
				err += scoped;
				err += "'";
				IdlError(err);
				yynerrs++;
				YYERROR;
			    }
			}

			iDefSeq -> append(interfaceDef);
		    }

		    $$ = iDefSeq._retn();
		}
		;

COMMA_scoped_name_STAR: ',' scoped_name COMMA_scoped_name_STAR
		{
		    $3 -> insert($2);
		    $$ = $3;
		}
		| /* empty */
		{
		    $$ = new IdlStringSeq;
		}
		;

scoped_name	: IDL_IDENTIFIER
		{
		    $$ = $1;
		}
		| IDL_SCOPE_DELIMITOR IDL_IDENTIFIER
		{
		    CORBA_Identifier_var ident = $2;

		    $$ = CORBA_string_alloc(2 + strlen(ident));
		    strcpy($$, "::");
		    strcat($$, ident);
		}
		| scoped_name IDL_SCOPE_DELIMITOR IDL_IDENTIFIER
		{
		    CORBA_Identifier_var scoped = $1;
		    CORBA_Identifier_var ident = $3;

		    $$ = CORBA_string_alloc(strlen(scoped) + 2 +
					    strlen(ident));
		    strcpy($$, scoped);
		    strcat($$, "::");
		    strcat($$, ident);
		}
		;

const_dcl	: IDL_CONST const_type IDL_IDENTIFIER '=' const_exp
		{
		    //
		    // Get type and value
		    //
		    CORBA_IDLType_var idlType = $2;
		    CORBA_Any_var any = $5;

		    //
		    // Get name, id and version
		    //
		    CORBA_Identifier_var name = $3;
		    CORBA_RepositoryId_var id = IdlGetRepositoryId(name);
		    CORBA_VersionSpec_var version = CORBA_string_dup("1.0");

		    //
		    // Check if conversion is necessary
		    //
		    CORBA_TypeCode_var tc1 = idlType -> type();
		    tc1 = OBGetOrigType(tc1);
		    CORBA_TypeCode_var tc2 = any -> type();

		    if(!tc1 -> equal(tc2))
		    {
			CORBA_ULong ul;
			CORBA_Boolean ulb;
			CORBA_Long l;
			CORBA_Boolean lb;
			CORBA_Double d;
			CORBA_Boolean db;
			TranslateAny(any, ul, ulb, l, lb, d, db);
			
			const char* err =
			    "wrong type for constant initialization";
		    
			//
			// Convert value type
			//
			switch(tc1 -> kind())
			{
			case CORBA_tk_short:
			    
			    if(ulb)
				any <<= (CORBA_Short)ul;
			    else if(lb)
				any <<= (CORBA_Short)l;
			    else if(db)
			    {
				IdlWarning("converting floating point to "
					   "short");
				any <<= (CORBA_Short)d;
			    }
			    else
			    {
				IdlError(err);
				yynerrs++;
				YYERROR;
			    }
			    
			    break;
			    
			case CORBA_tk_long:
			    
			    if(ulb)
				any <<= (CORBA_Long)ul;
			    else if(lb)
				any <<= (CORBA_Long)l;
			    else if(db)
			    {
				IdlWarning("converting floating point to "
					   "long");
				any <<= (CORBA_Long)d;
			    }
			    else
			    {
				IdlError(err);
				yynerrs++;
				YYERROR;
			    }
			    
			    break;
			    
			case CORBA_tk_ushort:
			    
			    if(ulb)
				any <<= (CORBA_UShort)ul;
			    else if(lb)
			    {
				if(l < 0)
				    IdlWarning("converting negative value to "
					       "unsigned short");
				any <<= (CORBA_UShort)l;
			    }
			    else if(db)
			    {
				IdlWarning("converting floating point to "
					   "unsigned short");
				any <<= (CORBA_UShort)d;
			    }
			    else
			    {
				IdlError(err);
				yynerrs++;
				YYERROR;
			    }
			    
			    break;
			    
			case CORBA_tk_ulong:
			    
			    if(ulb)
				any <<= (CORBA_ULong)ul;
			    else if(lb)
			    {
				if(l < 0)
				    IdlWarning("converting negative value to "
					       "unsigned short");
				any <<= (CORBA_ULong)l;
			    }
			    else if(db)
			    {
				IdlWarning("converting floating point to "
					   "unsigned long");
				any <<= (CORBA_ULong)d;
			    }
			    else
			    {
				IdlError(err);
				yynerrs++;
				YYERROR;
			    }
			    
			    break;
			    
			case CORBA_tk_float:
			    
			    if(ulb)
				any <<= (CORBA_Float)ul;
			    else if(lb)
				any <<= (CORBA_Float)l;
			    else if(db)
				any <<= (CORBA_Float)d;
			    else
			    {
				IdlError(err);
				yynerrs++;
				YYERROR;
			    }
			    
			    break;
			    
			case CORBA_tk_double:
			    
			    if(ulb)
				any <<= (CORBA_Double)ul;
			    else if(lb)
				any <<= (CORBA_Double)l;
			    else if(db)
				any <<= (CORBA_Double)d;
			    else
			    {
				IdlError(err);
				yynerrs++;
				YYERROR;
			    }
			    
			    break;

			case CORBA_tk_string:
			{
			    char* s;
			    if(any >>= s)
			    {
				CORBA_ULong len = tc1 -> length();
				CORBA_Any::from_string from(s, len);
				any <<= from;
			    }
			    break;
			}

			default:
			    IdlError(err);
			    yynerrs++;
			    YYERROR;
			    break;
			}
		    }

		    //
		    // Get container from container stack
		    //
		    assert(IdlContStack.length() > 0);
		    CORBA_Container_var container =
			IdlContStack[IdlContStack.length() - 1];

		    //
		    // Create constant definition
		    //
		    try
		    {
			CORBA_release(container ->
				      create_constant(id, name, version,
						      idlType, any));
		    }
		    catch(const CORBA_INTF_REPOS& ex)
		    {
			IdlPrintException(ex.minor(), id, name);
			yynerrs++;
			YYERROR;
		    }

		    //
		    // Add ID to IdlTopFileContent and IdlComments
		    //
		    IdlAddID(id);
		}
		;

const_type	: integer_type
		{
		    $$ = $1;
		}
		| char_type
		{
		    $$ = $1;
		}
		| boolean_type
		{
		    $$ = $1;
		}
		| floating_pt_type
		{
		    $$ = $1;
		}
		| string_type
		{
		    $$ = $1;
		}
		| scoped_name
		{
		    CORBA_ScopedName_var scoped = $1;

		    //
		    // Get container from container stack
		    //
		    assert(IdlContStack.length() > 0);
		    CORBA_Container_var container =
			IdlContStack[IdlContStack.length() - 1];
			
		    //
		    // Find scoped name
		    //
		    CORBA_IRObject_var irObj =
			IdlFindScopedName(container, scoped);

		    if(CORBA_is_nil(irObj))
		    {
			yynerrs++;
			YYERROR;
		    }

		    bool valid = false;

		    CORBA_IDLType_var idlType = CORBA_IDLType::_narrow(irObj);
		    if(!CORBA_is_nil(idlType))
		    {
			CORBA_TypeCode_var tc = idlType -> type();
			tc = OBGetOrigType(tc);
			
			CORBA_TCKind k = tc -> kind();
			
			if(k == CORBA_tk_short || k == CORBA_tk_ushort ||
			   k == CORBA_tk_long || k == CORBA_tk_ulong ||
			   k == CORBA_tk_float || k == CORBA_tk_double ||
			   k == CORBA_tk_char || k == CORBA_tk_boolean ||
			   (k == CORBA_tk_string && tc -> length() == 0))
			    valid = true;
		    }

		    if(!valid)
		    {
			CORBA_String_var err = CORBA_string_dup("`");
			err += scoped;
			err += "' is not a valid constant type";
			IdlError(err);
			yynerrs++;
			YYERROR;
		    }

		    $$ = idlType._retn();
		}
		;

const_exp	: or_expr
		{
		    $$ = $1;
		}
		;

or_expr		: xor_expr
		{
		    $$ = $1;
		}
		| or_expr '|' xor_expr
		{
		    CORBA_Any_var left = $1;
		    CORBA_Any_var right = $3;
		    CORBA_Any_var result = new CORBA_Any;

		    CORBA_ULong uLongL, uLongR;
		    CORBA_Boolean uLongLBool, uLongRBool;
		    CORBA_Long longL, longR;
		    CORBA_Boolean longLBool, longRBool;
		    CORBA_Double doubleL, doubleR;
		    CORBA_Boolean doubleLBool, doubleRBool;

		    TranslateAny(left,
				 uLongL, uLongLBool,
				 longL, longLBool,
				 doubleL, doubleLBool);

		    TranslateAny(right,
				 uLongR, uLongRBool,
				 longR, longRBool,
				 doubleR, doubleRBool);

		    if((uLongLBool || longLBool) &&
		       (uLongRBool || longRBool))
		    {
			if(longLBool || longRBool)
			    result <<= (longL | longR);
			else
			    result <<= (uLongL | uLongR);
		    }
		    else
		    {
			IdlError("invalid operand for binary operator |");
			yynerrs++;
			YYERROR;
		    }

		    $$ = result._retn();
		}
		;

xor_expr	: and_expr
		{
		    $$ = $1;
		}
		| xor_expr '^' and_expr
		{
		    CORBA_Any_var left = $1;
		    CORBA_Any_var right = $3;
		    CORBA_Any_var result = new CORBA_Any;

		    CORBA_ULong uLongL, uLongR;
		    CORBA_Boolean uLongLBool, uLongRBool;
		    CORBA_Long longL, longR;
		    CORBA_Boolean longLBool, longRBool;
		    CORBA_Double doubleL, doubleR;
		    CORBA_Boolean doubleLBool, doubleRBool;

		    TranslateAny(left,
				 uLongL, uLongLBool,
				 longL, longLBool,
				 doubleL, doubleLBool);

		    TranslateAny(right,
				 uLongR, uLongRBool,
				 longR, longRBool,
				 doubleR, doubleRBool);

		    if((uLongLBool || longLBool) &&
		       (uLongRBool || longRBool))
		    {
			if(longLBool || longRBool)
			    result <<= (longL ^ longR);
			else
			    result <<= (uLongL ^ uLongR);
		    }
		    else
		    {
			IdlError("invalid operand for binary operator ^");
			yynerrs++;
			YYERROR;
		    }

		    $$ = result._retn();
		}
		;

and_expr	: shift_expr
		{
		    $$ = $1;
		}
		| and_expr '&' shift_expr
		{
		    CORBA_Any_var left = $1;
		    CORBA_Any_var right = $3;
		    CORBA_Any_var result = new CORBA_Any;

		    CORBA_ULong uLongL, uLongR;
		    CORBA_Boolean uLongLBool, uLongRBool;
		    CORBA_Long longL, longR;
		    CORBA_Boolean longLBool, longRBool;
		    CORBA_Double doubleL, doubleR;
		    CORBA_Boolean doubleLBool, doubleRBool;

		    TranslateAny(left,
				 uLongL, uLongLBool,
				 longL, longLBool,
				 doubleL, doubleLBool);

		    TranslateAny(right,
				 uLongR, uLongRBool,
				 longR, longRBool,
				 doubleR, doubleRBool);

		    if((uLongLBool || longLBool) &&
		       (uLongRBool || longRBool))
		    {
			if(longLBool || longRBool)
			    result <<= (longL & longR);
			else
			    result <<= (uLongL & uLongR);
		    }
		    else
		    {
			IdlError("invalid operand for binary operator &");
			yynerrs++;
			YYERROR;
		    }

		    $$ = result._retn();
		}
		;

shift_expr	: add_expr
		{
		    $$ = $1;
		}
		| shift_expr IDL_RIGHT_SHIFT add_expr
		{
		    CORBA_Any_var left = $1;
		    CORBA_Any_var right = $3;
		    CORBA_Any_var result = new CORBA_Any;

		    CORBA_ULong uLongL, uLongR;
		    CORBA_Boolean uLongLBool, uLongRBool;
		    CORBA_Long longL, longR;
		    CORBA_Boolean longLBool, longRBool;
		    CORBA_Double doubleL, doubleR;
		    CORBA_Boolean doubleLBool, doubleRBool;

		    TranslateAny(left,
				 uLongL, uLongLBool,
				 longL, longLBool,
				 doubleL, doubleLBool);

		    TranslateAny(right,
				 uLongR, uLongRBool,
				 longR, longRBool,
				 doubleR, doubleRBool);

		    if((uLongLBool || longLBool) &&
		       (uLongRBool || longRBool))
		    {
			if(!(0 <= longR && longR < 32))
			{
			    IdlError("right operand of binary operator >> "
				     "must be in the range 0 <= right "
				     "operand < 32");
			    yynerrs++;
			    YYERROR;
			}

			if(longLBool)
			    result <<= (longL >> longR);
			else
			    result <<= (uLongL >> longR);
		    }
		    else
		    {
			IdlError("invalid operand for binary operator >>");
			yynerrs++;
			YYERROR;
		    }

		    $$ = result._retn();
		}
		| shift_expr IDL_LEFT_SHIFT add_expr
		{
		    CORBA_Any_var left = $1;
		    CORBA_Any_var right = $3;
		    CORBA_Any_var result = new CORBA_Any;

		    CORBA_ULong uLongL, uLongR;
		    CORBA_Boolean uLongLBool, uLongRBool;
		    CORBA_Long longL, longR;
		    CORBA_Boolean longLBool, longRBool;
		    CORBA_Double doubleL, doubleR;
		    CORBA_Boolean doubleLBool, doubleRBool;

		    TranslateAny(left,
				 uLongL, uLongLBool,
				 longL, longLBool,
				 doubleL, doubleLBool);

		    TranslateAny(right,
				 uLongR, uLongRBool,
				 longR, longRBool,
				 doubleR, doubleRBool);

		    if((uLongLBool || longLBool) &&
		       (uLongRBool || longRBool))
		    {
			if(!(0 <= longR && longR < 32))
			{
			    IdlError("right operand of binary operator << "
				     "must be in the range 0 <= right "
				     "operand < 32");
			    yynerrs++;
			    YYERROR;
			}

			if(longLBool)
			    result <<= (longL << longR);
			else
			    result <<= (uLongL << longR);
		    }
		    else
		    {
			IdlError("invalid operand for binary operator <<");
			yynerrs++;
			YYERROR;
		    }

		    $$ = result._retn();
		}
		;

add_expr	: mult_expr
		{
		    $$ = $1;
		}
		| add_expr '+' mult_expr
		{
		    CORBA_Any_var left = $1;
		    CORBA_Any_var right = $3;
		    CORBA_Any_var result = new CORBA_Any;

		    CORBA_ULong uLongL, uLongR;
		    CORBA_Boolean uLongLBool, uLongRBool;
		    CORBA_Long longL, longR;
		    CORBA_Boolean longLBool, longRBool;
		    CORBA_Double doubleL, doubleR;
		    CORBA_Boolean doubleLBool, doubleRBool;

		    TranslateAny(left,
				 uLongL, uLongLBool,
				 longL, longLBool,
				 doubleL, doubleLBool);

		    TranslateAny(right,
				 uLongR, uLongRBool,
				 longR, longRBool,
				 doubleR, doubleRBool);

		    if((uLongLBool || longLBool || doubleLBool) &&
		       (uLongRBool || longRBool || doubleRBool))
		    {
			if(doubleLBool || doubleRBool)
			    result <<= (doubleL + doubleR);
			else if(longLBool || longRBool)
			    result <<= (longL + longR);
			else
			    result <<= (uLongL + uLongR);
		    }
		    else
		    {
			IdlError("invalid operand for binary operator +");
			yynerrs++;
			YYERROR;
		    }

		    $$ = result._retn();
		}
		| add_expr '-' mult_expr
		{
		    CORBA_Any_var left = $1;
		    CORBA_Any_var right = $3;
		    CORBA_Any_var result = new CORBA_Any;

		    CORBA_ULong uLongL, uLongR;
		    CORBA_Boolean uLongLBool, uLongRBool;
		    CORBA_Long longL, longR;
		    CORBA_Boolean longLBool, longRBool;
		    CORBA_Double doubleL, doubleR;
		    CORBA_Boolean doubleLBool, doubleRBool;

		    TranslateAny(left,
				 uLongL, uLongLBool,
				 longL, longLBool,
				 doubleL, doubleLBool);

		    TranslateAny(right,
				 uLongR, uLongRBool,
				 longR, longRBool,
				 doubleR, doubleRBool);

		    if((uLongLBool || longLBool || doubleLBool) &&
		       (uLongRBool || longRBool || doubleRBool))
		    {
			if(doubleLBool || doubleRBool)
			    result <<= (doubleL - doubleR);
			else if(longLBool || longRBool)
			    result <<= (longL - longR);
			else
			{
			    if(uLongR > uLongL)
				result <<= ((CORBA_Long)uLongL -
					    (CORBA_Long)uLongR);
			    else
				result <<= (uLongL - uLongR);
			}
		    }
		    else
		    {
			IdlError("invalid operand for binary operator -");
			yynerrs++;
			YYERROR;
		    }

		    $$ = result._retn();
		}
		;

mult_expr	: unary_expr
		{
		    $$ = $1;
		}
		| mult_expr '*' unary_expr
		{
		    CORBA_Any_var left = $1;
		    CORBA_Any_var right = $3;
		    CORBA_Any_var result = new CORBA_Any;

		    CORBA_ULong uLongL, uLongR;
		    CORBA_Boolean uLongLBool, uLongRBool;
		    CORBA_Long longL, longR;
		    CORBA_Boolean longLBool, longRBool;
		    CORBA_Double doubleL, doubleR;
		    CORBA_Boolean doubleLBool, doubleRBool;

		    TranslateAny(left,
				 uLongL, uLongLBool,
				 longL, longLBool,
				 doubleL, doubleLBool);

		    TranslateAny(right,
				 uLongR, uLongRBool,
				 longR, longRBool,
				 doubleR, doubleRBool);

		    if((uLongLBool || longLBool || doubleLBool) &&
		       (uLongRBool || longRBool || doubleRBool))
		    {
			if(doubleLBool || doubleRBool)
			    result <<= (doubleL * doubleR);
			else if(longLBool || longRBool)
			    result <<= (longL * longR);
			else
			    result <<= (uLongL * uLongR);
		    }
		    else
		    {
			IdlError("invalid operand for binary operator *");
			yynerrs++;
			YYERROR;
		    }

		    $$ = result._retn();
		}
		| mult_expr '/' unary_expr
		{
		    CORBA_Any_var left = $1;
		    CORBA_Any_var right = $3;
		    CORBA_Any_var result = new CORBA_Any;

		    CORBA_ULong uLongL, uLongR;
		    CORBA_Boolean uLongLBool, uLongRBool;
		    CORBA_Long longL, longR;
		    CORBA_Boolean longLBool, longRBool;
		    CORBA_Double doubleL, doubleR;
		    CORBA_Boolean doubleLBool, doubleRBool;

		    TranslateAny(left,
				 uLongL, uLongLBool,
				 longL, longLBool,
				 doubleL, doubleLBool);

		    TranslateAny(right,
				 uLongR, uLongRBool,
				 longR, longRBool,
				 doubleR, doubleRBool);

		    if((uLongLBool || longLBool || doubleLBool) &&
		       (uLongRBool || longRBool || doubleRBool))
		    {
			if(doubleLBool || doubleRBool)
			    result <<= (doubleL / doubleR);
			else if(longLBool || longRBool)
			    result <<= (longL / longR);
			else
			    result <<= (uLongL / uLongR);
		    }
		    else
		    {
			IdlError("invalid operand for binary operator /");
			yynerrs++;
			YYERROR;
		    }

		    $$ = result._retn();
		}
		| mult_expr '%' unary_expr
		{
		    CORBA_Any_var left = $1;
		    CORBA_Any_var right = $3;
		    CORBA_Any_var result = new CORBA_Any;

		    CORBA_ULong uLongL, uLongR;
		    CORBA_Boolean uLongLBool, uLongRBool;
		    CORBA_Long longL, longR;
		    CORBA_Boolean longLBool, longRBool;
		    CORBA_Double doubleL, doubleR;
		    CORBA_Boolean doubleLBool, doubleRBool;

		    TranslateAny(left,
				 uLongL, uLongLBool,
				 longL, longLBool,
				 doubleL, doubleLBool);

		    TranslateAny(right,
				 uLongR, uLongRBool,
				 longR, longRBool,
				 doubleR, doubleRBool);

		    if((uLongLBool || longLBool) &&
		       (uLongRBool || longRBool))
		    {
			if(longLBool || longRBool)
			    result <<= (longL % longR);
			else
			    result <<= (uLongL % uLongR);
		    }
		    else
		    {
			IdlError("invalid operand for binary operator %");
			yynerrs++;
			YYERROR;
		    }

		    $$ = result._retn();
		}
		;

unary_expr	: unary_operator primary_expr
		{
		    CORBA_Any_var result = $2;
		    
		    const char* err = 0;
		    
		    if($1 == '~')
			err = "wrong argument type for bit-complement";
		    else if($1 == '+')
			err = "wrong argument type for unary plus";
		    else if($1 == '-')
			err = "wrong argument type for unary minus";
		    else
			assert(false);

		    CORBA_ULong ul;
		    CORBA_Boolean ulb;
		    CORBA_Long l;
		    CORBA_Boolean lb;
		    CORBA_Double d;
		    CORBA_Boolean db;
		    TranslateAny(result, ul, ulb, l, lb, d, db);

		    if(ulb)
		    {
			if($1 == '-')
			{
			    l = -(CORBA_Long)ul;
			    result <<= l;
			}
			else if($1 == '~')
			{
			    ul = ~ul;
			    result <<= ul;
			}
		    }
		    else if(lb)
		    {
			if($1 == '-')
			{
			    l = -l;
			    result <<= l;
			}
			else if($1 == '~')
			{
			    l = ~l;
			    result <<= l;
			}
		    }
		    else if(db)
		    {
			if($1 == '-')
			{
			    d = -d;
			    result <<= d;
			}
			else if($1 == '~')
			{
			    IdlError(err);
			    yynerrs++;
			    YYERROR;
			}
		    }
		    else
		    {
			IdlError(err);
			yynerrs++;
			YYERROR;
		    }

		    $$ = result._retn();
		}
		| primary_expr
		{
		    $$ = $1;
		}
		;

unary_operator	: '-'
		{
		    $$ = '-';
		}
		| '+'
		{
		    $$ = '+';
		}
		| '~'
		{
		    $$ = '~';
		}
		;

primary_expr	: scoped_name
		{
		    CORBA_ScopedName_var scoped = $1;
		    $$ = 0;
		    
		    //
		    // Get container from container stack
		    //
		    assert(IdlContStack.length() > 0);
		    CORBA_Container_var container =
			IdlContStack[IdlContStack.length() - 1];
		    
		    if(!CORBA_is_nil(IdlUnionSwitchType))
		    {
			//
			// Special treatment for unions with enums
			//
			CORBA_IDLType_var switchType =
			    CORBA_IDLType::_duplicate(IdlUnionSwitchType);
			
			while(true)
			{
			    CORBA_AliasDef_var aliasDef =
				CORBA_AliasDef::_narrow(switchType);
			    
			    if(CORBA_is_nil(aliasDef))
				break;
			    
			    switchType = aliasDef -> original_type_def();
			}
			
			CORBA_EnumDef_var enumDef =
			    CORBA_EnumDef::_narrow(switchType);
			
			if(!CORBA_is_nil(enumDef))
			{
			    CORBA_Long memberNum =
				IdlFindEnumMember(container, enumDef, scoped);
			    
			    if(memberNum == -1)
			    {
				yynerrs++;
				YYERROR;
			    }
			    
			    CORBA_TypeCode_var tc = enumDef -> type();
			    CORBA_ULong* p = new CORBA_ULong;
			    *p = (CORBA_ULong)memberNum;
			    $$ = new CORBA_Any;
			    $$ -> replace(tc, p, true);
			}
		    }

		    if($$ == 0)
		    {
			//
			// Find scoped name
			//
			CORBA_IRObject_var irObj =
			    IdlFindScopedName(container, scoped);
			
			if(CORBA_is_nil(irObj))
			{
			    yynerrs++;
			    YYERROR;
			}
			
			CORBA_ConstantDef_var co =
			    CORBA_ConstantDef::_narrow(irObj);
			if(CORBA_is_nil(co))
			{
			    CORBA_String_var err = CORBA_string_dup("`");
			    err += scoped;
			    err += "' is not a constant";
			    IdlError(err);
			    yynerrs++;
			    YYERROR;
			}
			
			$$ = co -> value();
		    }
		}
		| literal
		{
		    $$ = $1;
		}
		| '(' const_exp ')'
		{
		    $$ = $2;
		}
		;

literal		: IDL_INTEGER_LITERAL
		{
		    CORBA_Any* any = new CORBA_Any;
		    *any <<= $1;
		    $$ = any;
		}
		| IDL_STRING_LITERAL
		{
		    CORBA_String_var origString = CORBA_string_dup($1);

		    if(!IdlRemoveEscapes($1, true))
		    {
			CORBA_String_var err = CORBA_string_dup("`");
			err += origString;
			err += "' contains \\0 character";
			IdlError(err);
			yynerrs++;
			YYERROR;
		    }

		    CORBA_Any* any = new CORBA_Any;
		    *any <<= $1;
		    $$ = any;
		}
		| IDL_CHARACTER_LITERAL
		{
		    CORBA_Any* any = new CORBA_Any;
		    *any <<= CORBA_Any::from_char($1);
		    $$ = any;
		}
		| IDL_FLOATING_PT_LITERAL
		{
		    CORBA_Any* any = new CORBA_Any;
		    *any <<= $1;
		    $$ = any;
		}
		| boolean_literal
		{
		    CORBA_Any* any = new CORBA_Any;
		    *any <<= CORBA_Any::from_boolean($1);
		    $$ = any;
		}
		;

boolean_literal	: IDL_TRUE
		{
		    assert($1 == CORBA_TRUE);
		    $$ = true;
		}
		| IDL_FALSE
		{
		    assert($1 == CORBA_FALSE);
		    $$ = false;
		}
		;

positive_int_const: const_exp
		{
		    CORBA_Any_var any = $1;
		    CORBA_TypeCode_var tc = any -> type();
		    tc = OBGetOrigType(tc);
		    
		    switch(tc -> kind())
		    {
		    case CORBA_tk_short:
		    {
			CORBA_Short v;
			any >>= v;
			if(v <= 0)
			{
			    IdlError("integer constant must be > 0");
			    yynerrs++;
			    YYERROR;
			}
			$$ = (CORBA_ULong)v;
			break;
		    }

		    case CORBA_tk_long:
		    {
			CORBA_Long v;
			any >>= v;
			if(v <= 0)
			{
			    IdlError("integer constant must be > 0");
			    yynerrs++;
			    YYERROR;
			}
			$$ = (CORBA_ULong)v;
			break;
		    }
		    
		    case CORBA_tk_ushort:
		    {
			CORBA_UShort v;
			any >>= v;
			$$ = (CORBA_ULong)v;
			break;
		    }

		    case CORBA_tk_ulong:
		    {
			CORBA_ULong v;
			any >>= v;
			$$ = (CORBA_ULong)v;
			break;
		    }
		    
		    default:
		    {
			IdlError("not an integer constant");
			yynerrs++;
			YYERROR;
			break;
		    }
		    }
		}
		;

type_dcl	: IDL_TYPEDEF type_declarator
		{
		}
		| struct_type
		{
		    CORBA_release($1);
		}
		| union_type
		{
		    CORBA_release($1);
		}
		| enum_type
		{
		    CORBA_release($1);
		}
		;

type_declarator	: type_spec declarators
		{
		    //
		    // Get container from container stack
		    //
		    assert(IdlContStack.length() > 0);
		    CORBA_Container_var container =
			IdlContStack[IdlContStack.length() - 1];

		    CORBA_IDLType_var simpleIDLType = $1;
		    IdlDeclaratorSeq_var declarators = $2;

		    for(CORBA_ULong i = 0 ; i < declarators -> length() ; i++)
		    {
			IdlDeclarator declarator = declarators[i];

			//
			// Get name, id and version
			//
			CORBA_Identifier_var name = declarator.identifier;
			CORBA_RepositoryId_var id = IdlGetRepositoryId(name);
			CORBA_VersionSpec_var version =
			    CORBA_string_dup("1.0");

			//	
			// Handle arrays
			//
			CORBA_IDLType_var idlType = simpleIDLType;

			for(CORBA_Long j = declarator.sizes.length() - 1 ;
			    j >= 0 ; j--)
			{
			    CORBA_ArrayDef_var arrayDef = IdlRepository ->
				create_array(declarator.sizes[j], idlType);

			    idlType = CORBA_IDLType::_duplicate(arrayDef);
			}

			//
			// Create alias
			//
			try
			{
			    CORBA_release(container ->
					  create_alias(id, name, version,
						       idlType));
			}
			catch(const CORBA_INTF_REPOS& ex)
			{
			    IdlPrintException(ex.minor(), id, name);
			    yynerrs++;
			    YYERROR;
			}

			//
			// Add ID to IdlTopFileContent and IdlComments
			//
			IdlAddID(id);
		    }
		}
		;

type_spec	: simple_type_spec
		{
		    $$ = $1;
		}
		| constr_type_spec
		{
		    $$ = $1;
		}
		;

simple_type_spec: base_type_spec
		{
		    $$ = $1;
		}
		| template_type_spec
		{
		    $$ = $1;
		}
		| scoped_name
		{
		    CORBA_ScopedName_var scoped = $1;

		    //
		    // Get container from container stack
		    //
		    assert(IdlContStack.length() > 0);
		    CORBA_Container_var container =
			IdlContStack[IdlContStack.length() - 1];
			
		    //
		    // Find scoped name
		    //
		    CORBA_IRObject_var irObj =
			IdlFindScopedName(container, scoped);

		    if(CORBA_is_nil(irObj))
		    {
			yynerrs++;
			YYERROR;
		    }

		    CORBA_IDLType_var idlType = CORBA_IDLType::_narrow(irObj);
		    if(CORBA_is_nil(idlType))
		    {
			CORBA_String_var err = CORBA_string_dup("`");
			err += scoped;
			err += "' is not a type specification";
			IdlError(err);
			yynerrs++;
			YYERROR;
		    }

		    $$ = idlType._retn();
		}
		;

base_type_spec	: floating_pt_type
		{
		    $$ = $1;
		}
		| integer_type
		{
		    $$ = $1;
		}
		| char_type
		{
		    $$ = $1;
		}
		| boolean_type
		{
		    $$ = $1;
		}
		| octet_type
		{
		    $$ = $1;
		}
		| any_type
		{
		    $$ = $1;
		}
		;

template_type_spec: sequence_type
		{
		    $$ = $1;
		}
		| string_type
		{
		    $$ = $1;
		}
		;

constr_type_spec: struct_type
		{
		    $$ = $1;
		}
		| union_type
		{
		    $$ = $1;
		}
		| enum_type
		{
		    $$ = $1;
		}
		;

declarators	: declarator COMMA_declarator_STAR
		{
		    IdlDeclarator_var declarator = $1;
		    $2 -> insert(declarator);
		    $$ = $2;
		}
		;

COMMA_declarator_STAR: ',' declarator COMMA_declarator_STAR
		{
		    IdlDeclarator_var declarator = $2;
		    $3 -> insert(declarator);
		    $$ = $3;
		}
		| /* empty */
		{
		    $$ = new IdlDeclaratorSeq;
		}
		;

declarator	: simple_declarator
		{
		    $$ = $1;
		}
		| complex_declarator
		{
		    $$ = $1;
		}
		;

simple_declarator: IDL_IDENTIFIER
		{
		    $$ = new IdlDeclarator;
		    $$ -> identifier = $1;
		}
		;

COMMA_simple_declarator_STAR: ',' simple_declarator
			      COMMA_simple_declarator_STAR
		{
		    IdlDeclarator_var declarator = $2;
		    $3 -> insert(declarator);
		    $$ = $3;
		}
		| /* empty */
		{
		    $$ = new IdlDeclaratorSeq;
		}
		;

complex_declarator: array_declarator
		{
		    $$ = $1;
		}
		;

floating_pt_type: IDL_FLOAT
		{
		    $$ = IdlRepository -> get_primitive(CORBA_pk_float);
		}
		| IDL_DOUBLE
		{
		    $$ = IdlRepository -> get_primitive(CORBA_pk_double);
		}
		;

integer_type	: signed_int
		{
		    $$ = $1;
		}
		| unsigned_int
		{
		    $$ = $1;
		}
		;

signed_int	: signed_long_int
		{
		    $$ = $1;
		}
		| signed_short_int
		{
		    $$ = $1;
		}
		;

signed_long_int	: IDL_LONG
		{
		    $$ = IdlRepository -> get_primitive(CORBA_pk_long);
		}
		;

signed_short_int: IDL_SHORT
		{
		    $$ = IdlRepository -> get_primitive(CORBA_pk_short);
		}
		;

unsigned_int	: unsigned_long_int
		{
		    $$ = $1;
		}
		| unsigned_short_int
		{
		    $$ = $1;
		}
		;

unsigned_long_int: IDL_UNSIGNED IDL_LONG
		{
		    $$ = IdlRepository -> get_primitive(CORBA_pk_ulong);
		}
		;

unsigned_short_int: IDL_UNSIGNED IDL_SHORT
		{
		    $$ = IdlRepository -> get_primitive(CORBA_pk_ushort);
		}
		;

char_type	: IDL_CHAR
		{
		    $$ = IdlRepository -> get_primitive(CORBA_pk_char);
		}
		;

boolean_type	: IDL_BOOLEAN
		{
		    $$ = IdlRepository -> get_primitive(CORBA_pk_boolean);
		}
		;

octet_type	: IDL_OCTET
		{
		    $$ = IdlRepository -> get_primitive(CORBA_pk_octet);
		}
		;

any_type	: IDL_ANY
		{
		    $$ = IdlRepository -> get_primitive(CORBA_pk_any);
		}
		;

struct_type	: IDL_STRUCT IDL_IDENTIFIER
                {
		    //
		    // Get container from container stack
		    //
		    assert(IdlContStack.length() > 0);
		    CORBA_Container_var container =
			IdlContStack[IdlContStack.length() - 1];

		    //
		    // Get name, id and version
		    //
		    const char* name = $2; // No _var here!
		    CORBA_RepositoryId_var id = IdlGetRepositoryId(name);
		    CORBA_VersionSpec_var version = CORBA_string_dup("1.0");

		    //
		    // Create empty member sequence
		    //
		    CORBA_StructMemberSeq_var members =
		        new CORBA_StructMemberSeq;

		    //
		    // Create struct
		    //
		    try
                    {
			$<structDef>$ = container ->
			    create_struct(id, name, version, members);
		    }
		    catch(const CORBA_INTF_REPOS& ex)
		    {
			IdlPrintException(ex.minor(), id, name);
			yynerrs++;
			YYERROR;
		    }

		    //
		    // Add ID to IdlTopFileContent and IdlComments
		    //
		    IdlAddID(id);
                }
                '{' member_list '}'
		{
		    $$ = $<structDef>3;

		    //
		    // Get name and id
		    //
		    CORBA_Identifier_var name = $2;
		    CORBA_RepositoryId_var id = IdlGetRepositoryId(name);

		    //
		    // Get member sequence
		    //
		    CORBA_StructMemberSeq_var members = $5;

		    //
		    // Set the member sequence
		    //
		    try
                    {
			$$ -> members(members);
		    }
		    catch(const CORBA_INTF_REPOS& ex)
		    {
			IdlPrintException(ex.minor(), id, $2);
			yynerrs++;
			YYERROR;
		    }
                }
		;

member_list	: member_PLUS
		{
		    $$ = $1;
		}
		;

member_PLUS	: member
		{
		    $$ = $1;
		}
		| member member_PLUS
		{
		    CORBA_StructMemberSeq_var members1 = $1;
		    CORBA_StructMemberSeq_var members2 = $2;

		    for(CORBA_ULong i = 0 ; i < members2 -> length() ; i++)
		    {
			//
			// Check for duplicate struct members
			//
			for(CORBA_ULong j = 0 ; j < members1 -> length() ; j++)
			{
			    if(strcmp(members2[i].name, members1[j].name) == 0)
			    {
				CORBA_String_var err =
				    CORBA_string_dup("duplicate struct "
						     "member `");
				err += members2[i].name;
				err += "'";
				IdlError(err);
				yynerrs++;
				YYERROR;
			    }
			}

			members1 -> append(members2[i]);
		    }

		    $$ = members1._retn();
		}
		;

member		: member_NO_SEM ';'
		{
		    $$ = $1;
		}
		| member_NO_SEM
		{
		    $$ = $1;
		    IdlError("`;' missing after struct member");
		    yynerrs++;
		}
		;

member_NO_SEM	: type_spec declarators
		{
		    CORBA_IDLType_var typeSpec = $1;
		    IdlDeclaratorSeq_var declarators = $2;

		    CORBA_StructMemberSeq_var memberSeq =
			new CORBA_StructMemberSeq(declarators -> length());

		    for(CORBA_ULong i = 0 ; i < declarators -> length() ; i++)
		    {
			IdlDeclarator declarator = declarators[i];

			//	
			// Handle arrays
			//
			CORBA_IDLType_var idlType = typeSpec;

			for(CORBA_Long j = declarator.sizes.length() - 1 ;
			    j >= 0 ; j--)
			{
			    CORBA_ArrayDef_var arrayDef = IdlRepository ->
				create_array(declarator.sizes[j], idlType);

			    idlType = CORBA_IDLType::_duplicate(arrayDef);
			}

			//
			// Create member
			//
			CORBA_StructMember member;
			member.name = declarator.identifier;
			member.type = idlType -> type();
			member.type_def = idlType;
			
			memberSeq -> append(member);
		    }

		    $$ = memberSeq._retn();
		}
		;

union_type	: IDL_UNION IDL_IDENTIFIER IDL_SWITCH '(' switch_type_spec ')'
		{
		    //
		    // Get container from container stack
		    //
		    assert(IdlContStack.length() > 0);
		    CORBA_Container_var container =
			IdlContStack[IdlContStack.length() - 1];

		    //
		    // Get name, id and version
		    //
		    const char* name = $2; // No _var here!
		    CORBA_RepositoryId_var id = IdlGetRepositoryId(name);
		    CORBA_VersionSpec_var version = CORBA_string_dup("1.0");

		    //
		    // Get union switch type
		    //
		    CORBA_IDLType_ptr disc = $5; // No _var here!

		    //
		    // Create empty member sequence
		    //
		    CORBA_UnionMemberSeq_var members =
		        new CORBA_UnionMemberSeq;

		    //
		    // Create union
		    //
		    try
                    {
			$<unionDef>$ = container ->
			    create_union(id, name, version, disc, members);
		    }
		    catch(const CORBA_INTF_REPOS& ex)
		    {
			IdlPrintException(ex.minor(), id, name);
			yynerrs++;
			YYERROR;
		    }

		    //
		    // Add ID to IdlTopFileContent and IdlComments
		    //
		    IdlAddID(id);

		    //
		    // Set IdlUnionSwitchType
		    //
		    // assert(CORBA_is_nil(IdlUnionSwitchType));
		    // ^^^ Don't do this. In case of an syntax error
		    // IdlUnionSwitchType will not be set to nil.
		    //
		    IdlUnionSwitchType = CORBA_IDLType::_duplicate(disc);
		}
		'{' switch_body '}'
		{
		    $$ = $<unionDef>7;

		    //
		    // Reset IdlUnionSwitchType
		    //
		    CORBA_release(IdlUnionSwitchType);
		    IdlUnionSwitchType = CORBA_IDLType::_nil();
		    
		    //
		    // Get name and id
		    //
		    CORBA_Identifier_var name = $2;
		    CORBA_RepositoryId_var id = IdlGetRepositoryId(name);

		    //
		    // Get union switch type
		    //
		    CORBA_IDLType_var disc = $5;

		    //
		    // Get member sequence
		    //
		    CORBA_UnionMemberSeq_var members = $9;

		    bool haveDefault = false;
		    for(CORBA_ULong i = 0 ; i < members -> length() ; i++)
		    {
			//
			// Check for multiple default labels
			//
			CORBA_Octet oct;
			CORBA_Boolean b =
			    members[i].label >>= CORBA_Any::to_octet(oct);

			if(b)
			{
			    assert(oct == 0);

			    if(haveDefault)
			    {
				IdlError("multiple default labels");
				yynerrs++;
				YYERROR;
			    }

			    haveDefault = true;
			}
			else
			{
			    //
			    // Check for duplicate labels
			    //
			    for(CORBA_ULong j = 0 ;
				j < members -> length() ; j++)
			    {
				if(i == j)
				    continue;

				CORBA_TypeCode_var tc1 =
				    members[i].label.type();
				CORBA_TypeCode_var tc2 =
				    members[j].label.type();
				
				tc1 = OBGetOrigType(tc1);
				tc2 = OBGetOrigType(tc2);

				if(tc2 -> kind() == CORBA_tk_octet)
				    continue;

				assert(tc1 -> equal(tc2)); 

				const void* v1 = members[i].label.value();
				const void* v2 = members[j].label.value();
				
				switch(tc1 -> kind())
				{
				case CORBA_tk_short:
				    if(*(CORBA_Short*)v1 ==
				       *(CORBA_Short*)v2)
				    {
					CORBA_String_var err =
					    CORBA_string_dup("duplicate label "
							     "`");
					err += *(CORBA_Short*)v1;
					err += "'";
					IdlError(err);
					yynerrs++;
					YYERROR;
				    }
				    break;

				case CORBA_tk_ushort:
				    if(*(CORBA_UShort*)v1 ==
				       *(CORBA_UShort*)v2)
				    {
					CORBA_String_var err =
					    CORBA_string_dup("duplicate label "
							     "`");
					err += *(CORBA_UShort*)v1;
					err += "'";
					IdlError(err);
					yynerrs++;
					YYERROR;
				    }
				    break;

				case CORBA_tk_long:
				    if(*(CORBA_Long*)v1 ==
				       *(CORBA_Long*)v2)
				    {
					CORBA_String_var err =
					    CORBA_string_dup("duplicate label "
							     "`");
					err += *(CORBA_Long*)v1;
					err += "'";
					IdlError(err);
					yynerrs++;
					YYERROR;
				    }
				    break;

				case CORBA_tk_ulong:
				    if(*(CORBA_ULong*)v1 ==
				       *(CORBA_ULong*)v2)
				    {
					CORBA_String_var err =
					    CORBA_string_dup("duplicate label "
							     "`");
					err += *(CORBA_ULong*)v1;
					err += "'";
					IdlError(err);
					yynerrs++;
					YYERROR;
				    }
				    break;

				case CORBA_tk_enum:
				    if(*(CORBA_ULong*)v1 ==
				       *(CORBA_ULong*)v2)
				    {
					CORBA_String_var err =
					    CORBA_string_dup("duplicate label "
							     "`");
					CORBA_String_var s =
					    tc1 ->
					    member_name(*(CORBA_ULong*)v1);
					err += "'";
					IdlError(err);
					yynerrs++;
					YYERROR;
				    }
				    break;
				    
				case CORBA_tk_char:
				    if(*(CORBA_Char*)v1 ==
				       *(CORBA_Char*)v2)
				    {
					CORBA_String_var err =
					    CORBA_string_dup("duplicate label "
							     "`");
					err += (char)*(CORBA_Char*)v1;
					err += "'";
					IdlError(err);
					yynerrs++;
					YYERROR;
				    }
				    break;

				case CORBA_tk_boolean:
				    if(*(CORBA_Boolean*)v1 ==
				       *(CORBA_Boolean*)v2)
				    {
					CORBA_String_var err =
					    CORBA_string_dup("duplicate label "
							     "`");
					err += *(CORBA_Boolean*)v1 ?
					    "TRUE" : "FALSE";
					err += "'";
					IdlError(err);
					yynerrs++;
					YYERROR;
				    }
				    break;
				    
				default:
				    assert(false);
				    break;
				}
			    }
			}
		    }

		    //
		    // Check for illegal default label
		    //
		    if(haveDefault)
		    {
			CORBA_TypeCode_var tc = disc -> type();
			tc = OBGetOrigType(tc);
			
			if((tc -> kind() == CORBA_tk_char &&
			    members -> length() > 256) ||
			   (tc -> kind() == CORBA_tk_boolean &&
			    members -> length() > 2) ||
			   (tc -> kind() == CORBA_tk_enum &&
			    members -> length() > tc -> member_count()))
			{
			    IdlError("No value available for an explicit "
				     "default label");
			    yynerrs++;
			    YYERROR;
			}
		    }

		    //
		    // Set the member sequence
		    //
		    try
                    {
			$$ -> members(members);
		    }
		    catch(const CORBA_INTF_REPOS& ex)
		    {
			IdlPrintException(ex.minor(), id, $2);
			yynerrs++;
			YYERROR;
		    }
		}
		;

switch_type_spec: integer_type
		{
		    $$ = $1;
		}
		| char_type
		{
		    $$ = $1;
		}
		| boolean_type
		{
		    $$ = $1;
		}
		| enum_type
		{
		    $$ = $1;
		}
		| scoped_name
		{
		    CORBA_ScopedName_var scoped = $1;

		    //
		    // Get container from container stack
		    //
		    assert(IdlContStack.length() > 0);
		    CORBA_Container_var container =
			IdlContStack[IdlContStack.length() - 1];
			
		    //
		    // Find scoped name
		    //
		    CORBA_IRObject_var irObj =
			IdlFindScopedName(container, scoped);

		    if(CORBA_is_nil(irObj))
		    {
			yynerrs++;
			YYERROR;
		    }

		    bool valid = false;

		    CORBA_IDLType_var idlType = CORBA_IDLType::_narrow(irObj);
		    if(!CORBA_is_nil(idlType))
		    {
			CORBA_TypeCode_var tc = idlType -> type();
			tc = OBGetOrigType(tc);
			
			CORBA_TCKind k = tc -> kind();
			
			if(k == CORBA_tk_short || k == CORBA_tk_ushort ||
			   k == CORBA_tk_long || k == CORBA_tk_ulong ||
			   k == CORBA_tk_char || k == CORBA_tk_boolean ||
			   k == CORBA_tk_enum)
			    valid = true;
		    }

		    if(!valid)
		    {
			CORBA_String_var err = CORBA_string_dup("`");
			err += scoped;
			err += "' is not a valid union discriminator type";
			IdlError(err);
			yynerrs++;
			YYERROR;
		    }

		    $$ = idlType._retn();
		}
		;

switch_body	: case_PLUS
		{
		    $$ = $1;
		}
		;

case_PLUS	: case
		{
		    $$ = $1;
		}
		| case case_PLUS
		{
		    CORBA_UnionMemberSeq_var members1 = $1;
		    CORBA_UnionMemberSeq_var members2 = $2;

		    //
		    // Check for duplicate union members
		    //
		    for(CORBA_ULong i = 0 ; i < members2 -> length() ; i++)
		    {
			//
			// It's only necessary to compare against the
			// first element of members1, since all
			// elements in members1 are equal.
			//
			if(strcmp(members2[i].name, members1[0].name) == 0)
			{
			    CORBA_String_var err =
				CORBA_string_dup("duplicate union member `");
			    err += members2[i].name;
			    err += "'";
			    IdlError(err);
			    yynerrs++;
			    YYERROR;
			}

			members1 -> append(members2[i]);
		    }

		    $$ = members1._retn();
		}
		;

case		: case_NO_SEM ';'
		{
		    $$ = $1;
		}
		| case_NO_SEM
		{
		    $$ = $1;
		    IdlError("`;' missing after union member");
		    yynerrs++;
		}
                ;

case_NO_SEM	: case_label_PLUS element_spec
		{
		    IdlAnySeq_var anySeq = $1;
		    CORBA_UnionMember_var member = $2;

		    CORBA_UnionMemberSeq_var memberSeq =
			new CORBA_UnionMemberSeq(anySeq -> length());
		    memberSeq -> length(anySeq -> length());

		    for(CORBA_ULong i = 0 ; i < anySeq -> length() ; i++)
		    {
			memberSeq[i] = member;
			memberSeq[i].label = anySeq[i];
		    }

		    $$ = memberSeq._retn();
		}
		;

case_label_PLUS	: case_label
		{
		    CORBA_Any_var any = $1;
		    $$ = new IdlAnySeq;
		    $$ -> insert(any);
		}
		| case_label case_label_PLUS
		{
		    CORBA_Any_var any = $1;
		    $2 -> insert(any);
		    $$ = $2;
		}
		;

case_label	: IDL_CASE const_exp ':'
		{
		    CORBA_Any_var result = $2;

		    //
		    // Check if discriminator type is matched
		    //
		    assert(!CORBA_is_nil(IdlUnionSwitchType));
		    CORBA_TypeCode_var tc1 = IdlUnionSwitchType -> type();
		    tc1 = OBGetOrigType(tc1);
		    CORBA_TypeCode_var tc2 = result -> type();
		    tc2 = OBGetOrigType(tc2);

		    if(!tc1 -> equal(tc2))
		    {
			//
			// Check if conversion is possible
			//
			CORBA_ULong ul;
			CORBA_Boolean ulb;
			CORBA_Long l;
			CORBA_Boolean lb;
			CORBA_Double d;
			CORBA_Boolean db;
			TranslateAny(result, ul, ulb, l, lb, d, db);
			
			const char* err =
			    "case label type doesn't "
			    "match the discriminator type";
			
			//
			// Convert value type
			//
			switch(tc1 -> kind())
			{
			case CORBA_tk_short:
			    
			    if(ulb)
				result <<= (CORBA_Short)ul;
			    else if(lb)
				result <<= (CORBA_Short)l;
			    else
			    {
				IdlError(err);
				yynerrs++;
				YYERROR;
			    }
			    
			    break;
			    
			case CORBA_tk_long:
			    
			    if(ulb)
				result <<= (CORBA_Long)ul;
			    else if(lb)
				result <<= (CORBA_Long)l;
			    else
			    {
				IdlError(err);
				yynerrs++;
				YYERROR;
			    }
			    
			    break;
			    
			case CORBA_tk_ushort:
			    
			    if(ulb)
				result <<= (CORBA_UShort)ul;
			    else
			    {
				IdlError(err);
				yynerrs++;
				YYERROR;
			    }
			    
			    break;
			    
			case CORBA_tk_ulong:
			    
			    if(ulb)
				result <<= (CORBA_ULong)ul;
			    else
			    {
				IdlError(err);
				yynerrs++;
				YYERROR;
			    }
			    
			    break;
			    
			default:
			    IdlError(err);
			    yynerrs++;
			    YYERROR;
			    break;
			}
		    }

		    $$ = result._retn();
		}
		| IDL_DEFAULT ':'
		{
		    $$ = new CORBA_Any;
		    *$$ <<= CORBA_Any::from_octet(0);
		}
		;

element_spec	: type_spec declarator
		{
		    CORBA_IDLType_var typeSpec = $1;
		    IdlDeclarator_var declarator = $2;

		    //	
		    // Handle arrays
		    //
		    CORBA_IDLType_var idlType = typeSpec;
		    
		    for(CORBA_Long j = declarator -> sizes.length() - 1 ;
			j >= 0 ; j--)
		    {
			CORBA_ArrayDef_var arrayDef = IdlRepository ->
			    create_array(declarator -> sizes[j], idlType);
			
			idlType = CORBA_IDLType::_duplicate(arrayDef);
		    }
		    
		    //
		    // Create member
		    //
		    $$ = new CORBA_UnionMember;
		    $$ -> name = declarator -> identifier;
		    $$ -> type = idlType -> type();
		    $$ -> type_def = idlType;
		}
		;

enum_type	: IDL_ENUM IDL_IDENTIFIER '{' enumerator
		  COMMA_enumerator_STAR '}'
		{
		    //
		    // Get container from container stack
		    //
		    assert(IdlContStack.length() > 0);
		    CORBA_Container_var container =
			IdlContStack[IdlContStack.length() - 1];

		    //
		    // Get member sequence
		    //
		    CORBA_EnumMemberSeq_var members = $5;
		    members -> insert($4);

		    //
		    // Get name, id and version
		    //
		    CORBA_Identifier_var name = $2;
		    CORBA_RepositoryId_var id = IdlGetRepositoryId(name);
		    CORBA_VersionSpec_var version = CORBA_string_dup("1.0");

		    //
		    // Create enumerator
		    //
		    try
		    {
			$$ = container -> create_enum(id, name, version,
						      members);
		    }
		    catch(const CORBA_INTF_REPOS& ex)
		    {
			IdlPrintException(ex.minor(), id, name);
			yynerrs++;
			YYERROR;
		    }

		    //
		    // Add ID to IdlTopFileContent and IdlComments
		    //
		    IdlAddID(id);
		}
		;

COMMA_enumerator_STAR: ',' enumerator COMMA_enumerator_STAR
		{
		    CORBA_Identifier_var member = $2;
		    CORBA_EnumMemberSeq_var members = $3;
		    
		    //
		    // Check for duplicate enum members
		    //
		    for(CORBA_ULong i = 0 ; i < members -> length() ; i++)
		    {
			if(strcmp(members[i], member) == 0)
			{
			    CORBA_String_var err =
				CORBA_string_dup("duplicate enum member `");
			    err += member;
			    err += "'";
			    IdlError(err);
			    yynerrs++;
			    YYERROR;
			}
		    }

		    members -> insert(member);
		    $$ = members._retn();
		}
		| /* empty */
		{
		    $$ = new CORBA_EnumMemberSeq;
		}
		| ','
		{
		    IdlError("unexpected `,'");
		    yynerrs++;
		    YYERROR;
		}
		;

enumerator	: IDL_IDENTIFIER
		{
		    $$ = $1;
		}
		;

sequence_type	: IDL_SEQUENCE '<' simple_type_spec ',' positive_int_const '>'
		{
		    CORBA_IDLType_var idlType = $3;
		    CORBA_ULong bound = $5;

		    //
		    // Create sequence
		    //
		    $$ = IdlRepository -> create_sequence(bound, idlType);
		}
		| IDL_SEQUENCE '<' simple_type_spec '>'
		{
		    CORBA_IDLType_var idlType = $3;

		    //
		    // Create sequence
		    //
		    $$ = IdlRepository -> create_sequence(0, idlType);
		}
		;

string_type	: IDL_STRING '<' positive_int_const '>'
		{
		    CORBA_ULong bound = $3;

		    //
		    // Create string
		    //
		    $$ = IdlRepository -> create_string(bound);
		}
		| IDL_STRING
		{
		    //
		    // Get string primitive
		    //
		    $$ = IdlRepository -> get_primitive(CORBA_pk_string);
		}
		;

array_declarator: IDL_IDENTIFIER fixed_array_size_PLUS
		{
		    $$ = $2;
		    $$ -> identifier = $1;
		}
		;

fixed_array_size_PLUS: fixed_array_size
		{
		    $$ = new IdlDeclarator;
		    $$ -> sizes.insert($1);
		}
		| fixed_array_size fixed_array_size_PLUS
		{
		    $$ = $2;
		    $$ -> sizes.insert($1);
		}
		;

fixed_array_size: '[' positive_int_const ']'
		{
		    $$ = $2;
		}
		;

attr_dcl	: readonly_OPT IDL_ATTRIBUTE param_type_spec
		  simple_declarator COMMA_simple_declarator_STAR
		{
		    //
		    // Get container from container stack
		    //
		    assert(IdlContStack.length() > 0);
		    CORBA_Container_var container =
			IdlContStack[IdlContStack.length() - 1];

		    CORBA_InterfaceDef_var interf =
			CORBA_InterfaceDef::_narrow(container);
		    assert(!CORBA_is_nil(interf));

		    //
		    // Get attribute mode
		    //
		    CORBA_AttributeMode mode = $1;

		    //
		    // Get parameter/return type
		    //
		    CORBA_IDLType_var idlType = $3;

		    //
		    // Get declarators
		    //
		    IdlDeclarator_var declarator = $4;
		    IdlDeclaratorSeq_var declarators = $5;
		    declarators -> insert(declarator);

		    for(CORBA_ULong i = 0 ; i < declarators -> length() ; i++)
		    {
			assert(declarators[i].sizes.length() == 0);

			//
			// Get name, id and version
			//
			CORBA_Identifier_var name = declarators[i].identifier;
			CORBA_RepositoryId_var id = IdlGetRepositoryId(name);
			CORBA_VersionSpec_var version =
			    CORBA_string_dup("1.0");

			//
			// Create attribute
			//
			try
			{
			    CORBA_release(interf ->
					  create_attribute(id, name, version,
							   idlType, mode));
			}
			catch(const CORBA_INTF_REPOS& ex)
			{
			    IdlPrintException(ex.minor(), id, name);
			    yynerrs++;
			    YYERROR;
			}

			//
			// Add ID to IdlTopFileContent and IdlComments
			//
			IdlAddID(id);
		    }
		}
		;

readonly_OPT	: IDL_READONLY
		{
		    $$ = CORBA_ATTR_READONLY;
		}
		| /* empty */
		{
		    $$ = CORBA_ATTR_NORMAL;
		}
		;

except_dcl	: IDL_EXCEPTION IDL_IDENTIFIER
                {
		    //
		    // Get container from container stack
		    //
		    assert(IdlContStack.length() > 0);
		    CORBA_Container_var container =
			IdlContStack[IdlContStack.length() - 1];

		    //
		    // Get name, id and version
		    //
		    const char* name = $2; // No _var here!
		    CORBA_RepositoryId_var id = IdlGetRepositoryId(name);
		    CORBA_VersionSpec_var version = CORBA_string_dup("1.0");

		    //
		    // Create empty member sequence
		    //
		    CORBA_StructMemberSeq_var members =
		        new CORBA_StructMemberSeq;

		    //
		    // Create exception
		    //
		    try
                    {
			$<exceptionDef>$ = container ->
			    create_exception(id, name, version, members);
		    }
		    catch(const CORBA_INTF_REPOS& ex)
		    {
			IdlPrintException(ex.minor(), id, name);
			yynerrs++;
			YYERROR;
		    }

		    //
		    // Add ID to IdlTopFileContent and IdlComments
		    //
		    IdlAddID(id);
                }
                '{' member_STAR '}'
		{
		    CORBA_ExceptionDef_var ex = $<exceptionDef>3;

		    //
		    // Get name and id
		    //
		    CORBA_Identifier_var name = $2;
		    CORBA_RepositoryId_var id = IdlGetRepositoryId(name);

		    //
		    // Get member sequence
		    //
		    CORBA_StructMemberSeq_var members = $5;

		    //
		    // Set the member sequence
		    //
		    try
                    {
			ex -> members(members);
		    }
		    catch(const CORBA_INTF_REPOS& ex)
		    {
			IdlPrintException(ex.minor(), id, $2);
			yynerrs++;
			YYERROR;
		    }
                }
		;

member_STAR	: member member_STAR
		{
		    CORBA_StructMemberSeq_var members1 = $1;
		    CORBA_StructMemberSeq_var members2 = $2;

		    for(CORBA_ULong i = 0 ; i < members2 -> length() ; i++)
		    {
			//
			// Check for duplicate exception members
			//
			for(CORBA_ULong j = 0 ; j < members1 -> length() ; j++)
			{
			    if(strcmp(members2[i].name, members1[j].name) == 0)
			    {
				CORBA_String_var err =
				    CORBA_string_dup("duplicate exception "
						     "member `");
				err += members2[i].name;
				err += "'";
				IdlError(err);
				yynerrs++;
				YYERROR;
			    }
			}

			members1 -> append(members2[i]);
		    }

		    $$ = members1._retn();
		}
		| /* empty */
		{
		    $$ = new CORBA_StructMemberSeq;
		}
		;

op_dcl		: op_attribute_OPT op_type_spec IDL_IDENTIFIER
		  parameter_dcls raises_expr_OPT context_expr_OPT
		{
		    //
		    // Get container from container stack
		    //
		    assert(IdlContStack.length() > 0);
		    CORBA_Container_var container =
			IdlContStack[IdlContStack.length() - 1];

		    CORBA_InterfaceDef_var interf =
			CORBA_InterfaceDef::_narrow(container);
		    assert(!CORBA_is_nil(interf));

		    //
		    // Get operation mode
		    //
		    CORBA_OperationMode mode = $1;

		    //
		    // Get return type
		    //
		    CORBA_IDLType_var result = $2;

		    //
		    // Get member sequence
		    //
		    CORBA_ParDescriptionSeq_var params = $4;

		    //
		    // Get exceptions definition sequence
		    //
		    CORBA_ExceptionDefSeq_var exceptions = $5;

		    //
		    // Get context ID sequence
		    //
		    CORBA_ContextIdSeq_var contexts = $6;

		    //
		    // Get name, id and version
		    //
		    CORBA_Identifier_var name = $3;
		    CORBA_RepositoryId_var id = IdlGetRepositoryId(name);
		    CORBA_VersionSpec_var version = CORBA_string_dup("1.0");

		    //
		    // Check for oneway constraints
		    //
		    if(mode == CORBA_OP_ONEWAY)
		    {
			CORBA_TypeCode_var resultTC = result -> type();
			if(resultTC -> kind() != CORBA_tk_void)
			{
			    IdlError("oneway operations cannot have "
				     "return value");
			    yynerrs++;
			    YYERROR;
			}

			for(CORBA_ULong i = 0 ; i < params -> length() ; i++)
			    if(params[i].mode != CORBA_PARAM_IN)
			    {
				IdlError("oneway operations can only have "
					 "`in' parameters");
				yynerrs++;
				YYERROR;
			    }

			if(exceptions -> length() > 0)
			{
			    IdlError("oneway operations cannot have "
				     "user defined exceptions");
			    yynerrs++;
			    YYERROR;
			}
		    }

		    //
		    // Create operation
		    //
		    try
		    {
			CORBA_release(interf ->
				      create_operation(id, name, version,
						       result, mode, params,
						       exceptions, contexts));
		    }
		    catch(const CORBA_INTF_REPOS& ex)
		    {
			IdlPrintException(ex.minor(), id, name);
			yynerrs++;
			YYERROR;
		    }

		    //
		    // Add ID to IdlTopFileContent and IdlComments
		    //
		    IdlAddID(id);
		}
		;

op_attribute_OPT: op_attribute
		{
		     $$ = $1;
		}
		| /* empty */
		{
		     $$ = CORBA_OP_NORMAL;
		}
		;

raises_expr_OPT	: raises_expr
		{
		    $$ = $1;
		}
		| /* empty */
		{
		    $$ = new CORBA_ExceptionDefSeq;
		}
		;

context_expr_OPT: context_expr
		{
		    $$ = $1;
		}
		| /* empty */
		{
		    $$ = new CORBA_ContextIdSeq;
		}
		;

op_attribute	: IDL_ONEWAY
		{
		    $$ = CORBA_OP_ONEWAY;
		}
		;

op_type_spec	: param_type_spec
		{
		    $$ = $1;
		}
		| IDL_VOID
		{
		    $$ = IdlRepository -> get_primitive(CORBA_pk_void);
		}
		;

parameter_dcls	: '(' param_dcl COMMA_param_dcl_STAR ')'
		{
		    CORBA_ParameterDescription_var param = $2;
		    $3 -> insert(param);
		    $$ = $3;
		}
		| '(' ')'
		{
		    $$ = new CORBA_ParDescriptionSeq;
		}
		;

COMMA_param_dcl_STAR: ',' param_dcl COMMA_param_dcl_STAR
		{
		    CORBA_ParameterDescription_var param = $2;
		    $3 -> insert(param);
		    $$ = $3;
		}
		| /* empty */
		{
		    $$ = new CORBA_ParDescriptionSeq;
		}
		;

param_dcl	: param_attribute param_type_spec simple_declarator
		{
		    CORBA_ParameterMode mode = $1;
		    CORBA_IDLType_var typeSpec = $2;
		    IdlDeclarator_var declarator = $3;
		    assert(declarator -> sizes.length() == 0);

		    CORBA_ParameterDescription_var param =
			new CORBA_ParameterDescription;

		    param -> name = declarator -> identifier;
		    param -> type = typeSpec -> type();
		    param -> type_def = typeSpec;
		    param -> mode = mode;

		    $$ = param._retn();
		}
		| param_type_spec
		{
		    CORBA_IDLType_var typeSpec = $1;

		    IdlError("`in', `inout' or `out' missing");
		    IdlError("parameter name missing");
		    yynerrs++;
		    YYERROR;
		}
		| param_type_spec simple_declarator
		{
		    CORBA_IDLType_var typeSpec = $1;
		    IdlDeclarator_var declarator = $2;

		    IdlError("`in', `inout' or `out' missing");
		    yynerrs++;
		    YYERROR;
		}
		| param_attribute param_type_spec
		{
		    CORBA_IDLType_var typeSpec = $2;

		    IdlError("parameter name missing");
		    yynerrs++;
		    YYERROR;
		}
		;

param_attribute	: IDL_IN
		{
		    $$ = CORBA_PARAM_IN;
		}
		| IDL_OUT
		{
		    $$ = CORBA_PARAM_OUT;
		}
		| IDL_INOUT
		{
		    $$ = CORBA_PARAM_INOUT;
		}
		;

raises_expr	: IDL_RAISES '(' scoped_name COMMA_scoped_name_STAR ')'
		{
		    IdlStringSeq_var scopedNameSeq = $4;
		    scopedNameSeq -> insert($3);
		
		    //
		    // Get container from container stack
		    //
		    assert(IdlContStack.length() > 0);
		    CORBA_Container_var container =
			IdlContStack[IdlContStack.length() - 1];

		    //
		    // Create exception definition sequence
		    //
		    CORBA_ExceptionDefSeq_var exDefSeq =
			new CORBA_ExceptionDefSeq(scopedNameSeq -> length());

		    for(CORBA_ULong i = 0 ;
			i < scopedNameSeq -> length() ; i++)
		    {
			//
			// Find scoped name
			//
			const char* scoped = scopedNameSeq[i];
			CORBA_IRObject_var irObj =
			    IdlFindScopedName(container, scoped);

			if(CORBA_is_nil(irObj))
			{
			    yynerrs++;
			    YYERROR;
			}

			CORBA_ExceptionDef_var exceptionDef =
			    CORBA_ExceptionDef::_narrow(irObj);

			if(CORBA_is_nil(exceptionDef))
			{	
			    CORBA_String_var err = CORBA_string_dup("`");
			    err += scoped;
			    err += "' is not an exception type";
			    IdlError(err);
			    yynerrs++;
			    YYERROR;
			}

			exDefSeq -> append(exceptionDef);
		    }

		    $$ = exDefSeq._retn();
		}
		;

context_expr	: IDL_CONTEXT '(' IDL_STRING_LITERAL
		  COMMA_string_literal_STAR ')'
		{
		    CORBA_String_var string = $3;
		    CORBA_String_var origString = CORBA_string_dup(string);
		    IdlStringSeq_var result = $4;

		    if(!IdlRemoveEscapes(string.inout(), true))
		    {
			CORBA_String_var err = CORBA_string_dup("`");
			err += origString;
			err += "' contains \\0 character";
			IdlError(err);
			yynerrs++;
			YYERROR;
		    }

		    CORBA_String_var err2 = CORBA_string_dup("`");
		    err2 += origString;
		    err2 += "' is not a valid context name";

		    if(strlen(string) == 0)
		    {
			IdlError(err2);
			yynerrs++;
			YYERROR;
		    }

		    for(CORBA_ULong i = 0 ; i < strlen(string) ; i++)
		    {
			char c = string[i];

			if(isalpha(c))
			    continue;

			if(i != 0 && (isdigit(c) || c == '_' || c == '.'))
			    continue;

			if(i == strlen(string) - 1 && c == '*')
			    continue;

			IdlError(err2);
			yynerrs++;
			YYERROR;
		    }

		    result -> insert(string);
		    $$ = result._retn();
		}
		;

COMMA_string_literal_STAR: ',' IDL_STRING_LITERAL COMMA_string_literal_STAR
		{
		    CORBA_String_var string = $2;
		    CORBA_String_var origString = CORBA_string_dup(string);
		    IdlStringSeq_var result = $3;

		    if(!IdlRemoveEscapes(string.inout(), true))
		    {
			CORBA_String_var err = CORBA_string_dup("`");
			err += origString;
			err += "' contains \\0 character";
			IdlError(err);
			yynerrs++;
			YYERROR;
		    }

		    result -> insert(string);
		    $$ = result._retn();
		}
		| /* empty */
		{
		    $$ = new IdlStringSeq;
		}
		;

param_type_spec	: base_type_spec
		{
		    $$ = $1;
		}
		| sequence_type
		{
		    CORBA_IDLType_var typeSpec = $1;

		    IdlError("typedefs must be used for sequence type "
			     "parameters");
		    yynerrs++;
		    YYERROR;
		}
		| string_type
		{
		    $$ = $1;
		}
		| scoped_name
		{
		    CORBA_ScopedName_var scoped = $1;

		    //
		    // Get container from container stack
		    //
		    assert(IdlContStack.length() > 0);
		    CORBA_Container_var container =
			IdlContStack[IdlContStack.length() - 1];
			
		    //
		    // Find scoped name
		    //
		    CORBA_IRObject_var irObj =
			IdlFindScopedName(container, scoped);

		    if(CORBA_is_nil(irObj))
		    {
			yynerrs++;
			YYERROR;
		    }

		    CORBA_IDLType_var idlType = CORBA_IDLType::_narrow(irObj);
		    if(CORBA_is_nil(idlType))
		    {
			CORBA_String_var err = CORBA_string_dup("`");
			err += scoped;
			err += "' is not a parameter type specification";
			IdlError(err);
			yynerrs++;
			YYERROR;
		    }

		    $$ = idlType._retn();
		}
		;
%%

// ----------------------------------------------------------------------
// Do parsing
// ----------------------------------------------------------------------

int
IdlParse(CORBA_ORB_ptr orb,
	 CORBA_Repository_ptr repository,
	 CORBA_RepositoryIdSeq*& topFileContent,
	 IdlStringSeq*& includes,
	 IdlCommentSeq*& comments)
{
    //
    // Initialize global variables
    //   
    IdlLineno = 1;
    IdlDocComment = CORBA_string_dup("");
    IdlORB = orb;
    IdlRepository = repository;
    IdlTopFileContent = new CORBA_RepositoryIdSeq;
    IdlIncludes = new IdlStringSeq;
    IdlComments = new IdlCommentSeq;
    IdlUnionSwitchType = CORBA_IDLType::_nil();
    IdlFwdInterfaces.length(0);    

    //
    // Set out parameters
    //
    topFileContent = IdlTopFileContent;
    includes = IdlIncludes;
    comments = IdlComments;

    //
    // Add empty file name to file name stack
    // Add empty prefix stack to the prefix stack stack
    //
    IdlFileStack.append(CORBA_string_dup(""));
    IdlPrefStackStack.length(1);
    
    //
    // Parse
    //
    yynerrs = 0;
    int yyparse();
    int status = yyparse();
    assert(yynerrs > 0 || status == 0);

    //
    // Do some cleanup
    //
    IdlFileStack.length(0);
    IdlPrefStackStack.length(0);

    //
    // Check whether there are some interface that have been forward
    // declared but never defined
    //
    for(CORBA_ULong i = 0 ; i < IdlFwdInterfaces.length() ; i++)
    {
        //
        // Don't use IdlError here, because filename and linenumber
        // are not defined
        //
        fflush(stderr);
        fprintf(stderr, "interface `%s' has been declared but not defined\n",
	        IdlFwdInterfaces[i].in() + 2); // Don't show the leading "::"

        yynerrs++;
    }

    //
    // Important note:
    //
    // If IdlParse returns with a status != 0, it is most certain that
    // there are memory leaks. The reason for that is that in case of
    // an error, it is not possible to free any values of the yyval
    // union that have been created with "new".
    //

    //
    // Return the status
    //
    return yynerrs;
}

// ----------------------------------------------------------------------
// Translate an Any into ULong, Long and Double
// ----------------------------------------------------------------------

static void TranslateAny(const CORBA_Any& any,
			 CORBA_ULong& ul, CORBA_Boolean& ulBool,
			 CORBA_Long& l, CORBA_Boolean& lBool,
			 CORBA_Double& d, CORBA_Boolean& dBool)
{
    ulBool = any >>= ul;
    lBool = any >>= l;
    dBool = any >>= d;

    //
    // If it's not a Long, or ULong, or a Double check if it's a short
    // or an unsigned short. If it's either of these then assign the
    // value to the long.
    //
    if(!ulBool && !lBool && !dBool)
    {
	CORBA_Short u;
	CORBA_UShort us;
	if(any >>= u)
	{
	    lBool = true;
	    l = u;
	}
	else if(any >>= us)
	{
	    lBool = true;
	    l = us;
	}
    }

    if(!lBool)
	if(ulBool)
	    l = (CORBA_Long)ul;
			
    if(!dBool)
    {
	if(ulBool)
	    d = (CORBA_Double)ul;
	else if(lBool)
	    d = (CORBA_Double)l;
    }
}

// ----------------------------------------------------------------------
// Error output
// ----------------------------------------------------------------------

void
yyerror(const char* s)
{
    IdlError(s);
}
