// **********************************************************************
//
// Copyright (c) 1999
// Object Oriented Concepts, Inc.
// Billerica, MA, USA
//
// All Rights Reserved
//
// **********************************************************************

#include <OB/CORBA.h>
#include <OB/CosProperty_impl.h>

#include <Config.h> // Make sure that the compiler complains if the
                    // configuration doesn't match

// ----------------------------------------------------------------------

CosPropertyService_PropertySetDef_impl::
CosPropertyService_PropertySetDef_impl(CORBA_ORB_ptr orb)
    : orb_(CORBA_ORB::_duplicate(orb))
{
}

CosPropertyService_PropertySetDef_impl::CosPropertyService_PropertySetDef_impl(
    CORBA_ORB_ptr orb,
    const CosPropertyService_PropertyTypes& allowed_property_types,
    const CosPropertyService_PropertyDefs& allowed_property_defs)
    : orb_(CORBA_ORB::_duplicate(orb)),
      allowedTypes_(allowed_property_types),
      allowedProperties_(allowed_property_defs)
{
}

CosPropertyService_PropertySetDef_impl::CosPropertyService_PropertySetDef_impl(
    CORBA_ORB_ptr orb,
    const CosPropertyService_PropertyDefs& initial_property_defs)
    : orb_(CORBA_ORB::_duplicate(orb)),
      propertySeq_(initial_property_defs)
{
}

void
CosPropertyService_PropertySetDef_impl::
define_property(const char* property_name,
		const CORBA_Any& property_value)
{
    try
    {
	define_property_with_mode(property_name, property_value,
				  CosPropertyService_undefined);
    }
    catch(const CosPropertyService_UnsupportedMode&)
    {
	// This exception can't be thrown in this context
	assert(false);
    }
}

void
CosPropertyService_PropertySetDef_impl::
define_properties(const CosPropertyService_Properties& nproperties)
{
    CosPropertyService_PropertyDefs p(nproperties.length());
    p.length(nproperties.length());

    for(CORBA_ULong i = 0 ; i < nproperties.length() ; i++)
    {
	p[i].property_name = nproperties[i].property_name;
	p[i].property_value = nproperties[i].property_value;
	p[i].property_mode = CosPropertyService_undefined;
    }

    define_properties_with_modes(p);
}

CORBA_ULong
CosPropertyService_PropertySetDef_impl::get_number_of_properties()
{
    return propertySeq_.length();
}

void
CosPropertyService_PropertySetDef_impl::
get_all_property_names(CORBA_ULong how_many,
		       CosPropertyService_PropertyNames*& property_names,
		       CosPropertyService_PropertyNamesIterator_ptr& rest)
{
    CORBA_ULong i;
 
    CORBA_ULong len = propertySeq_.length();
    CORBA_ULong num = len < how_many ? len : how_many;
 
    property_names = new CosPropertyService_PropertyNames(num);
    property_names -> length(num);
 
    for(i = 0 ; i < num ; i++)
	(*property_names)[i] = propertySeq_[i].property_name;

    if(how_many < len)
    {
	CosPropertyService_PropertyNames rem(len - how_many);
        rem.length(len - how_many);
 
        for(i = 0 ; i < len - how_many ; i++)
	    rem[i] = propertySeq_[i + how_many].property_name;
 
        rest = new CosPropertyService_PropertyNamesIterator_impl(orb_, rem);
    }
    else
        rest = CosPropertyService_PropertyNamesIterator::_nil();
}

CORBA_Any*
CosPropertyService_PropertySetDef_impl::
get_property_value(const char* property_name)
{
    if(!strlen(property_name))
	throw CosPropertyService_InvalidPropertyName();

    CosPropertyService_PropertyDef* p = get(property_name);
    if(!p)
	throw CosPropertyService_PropertyNotFound();

    return new CORBA_Any(p -> property_value);
}

CORBA_Boolean
CosPropertyService_PropertySetDef_impl::
get_properties(const CosPropertyService_PropertyNames& property_names,
	       CosPropertyService_Properties*& nproperties)
{
    CORBA_Boolean allNamesAreValid = true;

    nproperties = new CosPropertyService_Properties;
    nproperties -> length(property_names.length());

    for(CORBA_ULong i = 0 ; i < property_names.length() ; i++)
    {
	(*nproperties)[i].property_name = property_names[i];

	CosPropertyService_PropertyDef* p = get(property_names[i]);
	if(p)
	    (*nproperties)[i].property_value = p -> property_value;
	else
	{
	    //
	    // Create property with value tk_void for unknown name
	    //
	    (*nproperties)[i].property_value.replace(CORBA__tc_void, 0);
	    allNamesAreValid = false;
	}
    }

    return allNamesAreValid;
}

void
CosPropertyService_PropertySetDef_impl::
get_all_properties(CORBA_ULong how_many,
		   CosPropertyService_Properties*& nproperties,
		   CosPropertyService_PropertiesIterator_ptr& rest)
{
    CORBA_ULong i;
 
    CORBA_ULong len = propertySeq_.length();
    CORBA_ULong num = len < how_many ? len : how_many;
 
    nproperties = new CosPropertyService_Properties(num);
    nproperties -> length(num);
 
    for(i = 0 ; i < num ; i++)
    {
	(*nproperties)[i].property_name = propertySeq_[i].property_name;
	(*nproperties)[i].property_value = propertySeq_[i].property_value;
    }

    if(how_many < len)
    {
	CosPropertyService_PropertyDefs rem(len - how_many);
        rem.length(len - how_many);
 
        for(i = 0 ; i < len - how_many ; i++)
	    rem[i] = propertySeq_[i + how_many];
 
        rest = new CosPropertyService_PropertiesIterator_impl(orb_, rem);
    }
    else
        rest = CosPropertyService_PropertiesIterator::_nil();
}

void
CosPropertyService_PropertySetDef_impl::
delete_property(const char* property_name)
{
    if(!strlen(property_name))
	throw CosPropertyService_InvalidPropertyName();

    CORBA_ULong pos;
    CosPropertyService_PropertyDef* p = get(property_name, &pos);
    if(!p)
	throw CosPropertyService_PropertyNotFound();

    if(p -> property_mode == CosPropertyService_fixed_normal ||
       p -> property_mode == CosPropertyService_fixed_readonly)
	throw CosPropertyService_FixedProperty();

    propertySeq_.remove(pos);
}

void
CosPropertyService_PropertySetDef_impl::
delete_properties(const CosPropertyService_PropertyNames& property_names)
{
    //
    // Sequence with exceptions
    //
    CosPropertyService_MultipleExceptions mex;

    for(CORBA_ULong i = 0 ; i < property_names.length() ; i++)
    {
	if(!property_names.length())
	{
	    CORBA_ULong len = mex.exceptions.length();
	    mex.exceptions.length(len + 1);
	    mex.exceptions[len].reason =
		CosPropertyService_invalid_property_name;
	    mex.exceptions[len].failing_property_name = property_names[i];
	    continue;
	}

	CORBA_ULong pos;
	CosPropertyService_PropertyDef *p = get(property_names[i], &pos);
	if(!p)
	{
	    CORBA_ULong len = mex.exceptions.length();
	    mex.exceptions.length(len + 1);
	    mex.exceptions[len].reason = CosPropertyService_property_not_found;
	    mex.exceptions[len].failing_property_name = property_names[i];
	    continue;
	}

	if(p -> property_mode == CosPropertyService_fixed_normal ||
	   p -> property_mode == CosPropertyService_fixed_readonly)
	{
	    CORBA_ULong len = mex.exceptions.length();
	    mex.exceptions.length(len + 1);
	    mex.exceptions[len].reason = CosPropertyService_fixed_property;
	    mex.exceptions[len].failing_property_name = property_names[i];
	    continue;
	}

	propertySeq_.remove(pos);
    }

    //
    // Check for exceptions
    //
    if(mex.exceptions.length())
	throw mex;
}

CORBA_Boolean
CosPropertyService_PropertySetDef_impl::delete_all_properties()
{
    //
    // Sequence with remaining (fixed) properties
    //
    CosPropertyService_PropertyDefs fixedSeq;

    for(CORBA_ULong i = 0 ; i < propertySeq_.length() ; i++)
    {
	CosPropertyService_PropertyDef p = propertySeq_[i];
	if(p.property_mode == CosPropertyService_fixed_normal &&
	   p.property_mode == CosPropertyService_fixed_readonly)
	    fixedSeq.append(p);
    }

    propertySeq_ = fixedSeq;

    return propertySeq_.length() == 0;
}

CORBA_Boolean
CosPropertyService_PropertySetDef_impl::
is_property_defined(const char* property_name)
{
    return get(property_name) != 0;
}

void
CosPropertyService_PropertySetDef_impl::
get_allowed_property_types(CosPropertyService_PropertyTypes*& property_types)
{
    property_types =
	new CosPropertyService_PropertyTypes(allowedTypes_);
}

void
CosPropertyService_PropertySetDef_impl::
get_allowed_properties(CosPropertyService_PropertyDefs*& property_defs)
{
    property_defs =
	new CosPropertyService_PropertyDefs(allowedProperties_);
}

void
CosPropertyService_PropertySetDef_impl::
define_property_with_mode(const char* property_name,
			  const CORBA_Any& property_value,
			  CosPropertyService_PropertyModeType property_mode)
{
    //
    // Check name, type and property
    //
    check(property_name, property_value, property_mode);

    CosPropertyService_PropertyDef* p = get(property_name);

    //
    // Overwrite existing property if allowed
    //
    if(p)
    {
	if(p -> property_mode == CosPropertyService_read_only ||
	   p -> property_mode == CosPropertyService_fixed_readonly)
	    throw CosPropertyService_ReadOnlyProperty();

	//
	// Check type
	//
        CORBA_TypeCode_var tc1 = p -> property_value.type();
        CORBA_TypeCode_var tc2 = property_value.type();
        if(!tc1 -> equal(tc2))
	    throw CosPropertyService_ConflictingProperty();

	//
	// Set new value
	//
	p -> property_value = property_value;
    }
    else
    {
	CORBA_ULong len = propertySeq_.length();
	propertySeq_.length(len + 1);
	propertySeq_[len].property_name = property_name;
	propertySeq_[len].property_value = property_value;
	propertySeq_[len].property_mode = property_mode;
    }
}

void
CosPropertyService_PropertySetDef_impl::define_properties_with_modes(
    const CosPropertyService_PropertyDefs& property_defs)
{
    //
    // Sequence with exceptions
    //
    CosPropertyService_MultipleExceptions mex;

    for(CORBA_ULong i = 0 ; i < property_defs.length() ; i++)
    {
	try
	{
	    check(property_defs[i].property_name,
		  property_defs[i].property_value,
		  property_defs[i].property_mode);
	}
	catch(const CosPropertyService_InvalidPropertyName&)
	{
	    CORBA_ULong len = mex.exceptions.length();
	    mex.exceptions.length(len + 1);
	    mex.exceptions[len].reason =
		CosPropertyService_invalid_property_name;
	    mex.exceptions[len].failing_property_name =
		property_defs[i].property_name;
	    continue;
	}
	catch(const CosPropertyService_UnsupportedTypeCode&)
	{
	    CORBA_ULong len = mex.exceptions.length();
	    mex.exceptions.length(len + 1);
	    mex.exceptions[len].reason =
		CosPropertyService_unsupported_type_code;
	    mex.exceptions[len].failing_property_name =
		property_defs[i].property_name;
	    continue;
	}
	catch(const CosPropertyService_UnsupportedProperty&)
	{
	    CORBA_ULong len = mex.exceptions.length();
	    mex.exceptions.length(len + 1);
	    mex.exceptions[len].reason =
		CosPropertyService_unsupported_property;
	    mex.exceptions[len].failing_property_name =
		property_defs[i].property_name;
	    continue;
	}

	CosPropertyService_PropertyDef* p =
	    get(property_defs[i].property_name);

	//
	// Overwrite existing property if allowed
	//
	if(p)
	{
	    if(p -> property_mode == CosPropertyService_read_only ||
	       p -> property_mode == CosPropertyService_fixed_readonly)
	    {
		CORBA_ULong len = mex.exceptions.length();
		mex.exceptions.length(len + 1);
		mex.exceptions[len].reason =
		    CosPropertyService_read_only_property;
		mex.exceptions[len].failing_property_name =
		    property_defs[i].property_name;
		continue;
	    }

	    //
	    // Check type
	    //
            CORBA_TypeCode_var tc1 = p -> property_value.type();
            CORBA_TypeCode_var tc2 = property_defs[i].property_value.type();
            if(!tc1 -> equal(tc2))
	    {
		CORBA_ULong len = mex.exceptions.length();
		mex.exceptions.length(len + 1);
		mex.exceptions[len].reason =
		    CosPropertyService_conflicting_property;
		mex.exceptions[len].failing_property_name =
		    property_defs[i].property_name;
		continue;
	    }

	    //
	    // Set new value
	    //
	    p -> property_value = property_defs[i].property_value;
	}
	else
	{
	    CORBA_ULong len = propertySeq_.length();
	    propertySeq_.length(len + 1);
	    propertySeq_[len].property_name = property_defs[i].property_name;
	    propertySeq_[len].property_value = property_defs[i].property_value;
	    propertySeq_[len].property_mode = property_defs[i].property_mode;
	}
    }

    //
    // Check for exceptions
    //
    if(mex.exceptions.length())
	throw mex;
}

CosPropertyService_PropertyModeType
CosPropertyService_PropertySetDef_impl::
get_property_mode(const char* property_name)
{
    if(!strlen(property_name))
	throw CosPropertyService_InvalidPropertyName();

    CosPropertyService_PropertyDef* p = get(property_name);
    if(!p)
	throw CosPropertyService_PropertyNotFound();

    return p -> property_mode;
}

CORBA_Boolean
CosPropertyService_PropertySetDef_impl::
get_property_modes(const CosPropertyService_PropertyNames& property_names,
		   CosPropertyService_PropertyModes*& property_modes)
{
    bool allDefined = true;

    property_modes = new CosPropertyService_PropertyModes;

    for(CORBA_ULong i = 0 ; i < property_names.length() ; i++)
    {
	CosPropertyService_PropertyMode m;
	m.property_name = property_names[i];

	CosPropertyService_PropertyDef* p = get(property_names[i]);
	if(p)
	    m.property_mode = p -> property_mode;
	else
	{
	    m.property_mode = CosPropertyService_undefined;
	    allDefined = false;
	}

	property_modes -> append(m);
    }

    return allDefined;
}

void
CosPropertyService_PropertySetDef_impl::
set_property_mode(const char* property_name,
		  CosPropertyService_PropertyModeType property_mode)
{
    if(!strlen(property_name))
	throw CosPropertyService_InvalidPropertyName();

    CosPropertyService_PropertyDef* p = get(property_name);
    if(!p)
	throw CosPropertyService_PropertyNotFound();

    if(property_mode == CosPropertyService_undefined)
	throw CosPropertyService_UnsupportedMode();

    p -> property_mode = property_mode;
}

void
CosPropertyService_PropertySetDef_impl::
set_property_modes(const CosPropertyService_PropertyModes& property_modes)
{
    //
    // Sequence with exceptions
    //
    CosPropertyService_MultipleExceptions mex;

    for(CORBA_ULong i = 0 ; i < property_modes.length() ; i++)
    {
	if(!strlen(property_modes[i].property_name))
	{
	    CORBA_ULong len = mex.exceptions.length();
	    mex.exceptions.length(len + 1);
	    mex.exceptions[len].reason =
		CosPropertyService_invalid_property_name;
	    mex.exceptions[len].failing_property_name =
		property_modes[i].property_name;
	    continue;
	}

	CosPropertyService_PropertyDef* p =
	    get(property_modes[i].property_name);
	if(!p)
	{
	    CORBA_ULong len = mex.exceptions.length();
	    mex.exceptions.length(len + 1);
	    mex.exceptions[len].reason =
		CosPropertyService_property_not_found;
	    mex.exceptions[len].failing_property_name =
		property_modes[i].property_name;
	    continue;
	}

	if(property_modes[i].property_mode == CosPropertyService_undefined)
	{
	    CORBA_ULong len = mex.exceptions.length();
	    mex.exceptions.length(len + 1);
	    mex.exceptions[len].reason =
		CosPropertyService_unsupported_mode;
	    mex.exceptions[len].failing_property_name =
		property_modes[i].property_name;
	    continue;
	}

	p -> property_mode = property_modes[i].property_mode;
    }

    //
    // Check for exceptions
    //
    if(mex.exceptions.length())
	throw mex;
}

void
CosPropertyService_PropertySetDef_impl::
check(const char* property_name, const CORBA_Any& property_value,
      CosPropertyService_PropertyModeType property_mode)
{
    CORBA_ULong i;

    //
    // Check name
    //
    if(!strlen(property_name))
	throw CosPropertyService_InvalidPropertyName();

    //
    // Check if property is allowed
    //
    if(allowedProperties_.length())
    {
	bool propertyIsAllowed = false;
	for(i = 0 ; i < allowedProperties_.length() ; i++)
	{
            if(!strcmp(allowedProperties_[i].property_name, property_name))
	    {
		//
		// Check if the type matches
		//
		CORBA_TypeCode_var tc1 =
		    allowedProperties_[i].property_value.type();
		CORBA_TypeCode_var tc2 = property_value.type();
		
		if(!tc1 -> equal(tc2))
		    throw CosPropertyService_UnsupportedTypeCode();
 
		//
		// Mode is only relevant if not undefined
		//
		if(property_mode != CosPropertyService_undefined)
		{
		    if(allowedProperties_[i].property_mode != property_mode)
			continue;
		}

		propertyIsAllowed = true;
		break;
	    }
	}

	if(!propertyIsAllowed)
	    throw CosPropertyService_UnsupportedProperty();
    }

    //
    // Check if TypeCode is allowed
    //
    if(allowedTypes_.length())
    {
	bool typeIsAllowed = false;
	for(i = 0 ; i < allowedTypes_.length() ; i++)
	{
            CORBA_TypeCode_var tc = property_value.type();
            if(allowedTypes_[i] -> equal(tc))
	    {
		typeIsAllowed = true;
		break;
	    }
	}

	if(!typeIsAllowed)
	    throw CosPropertyService_UnsupportedTypeCode();
    }
}

CosPropertyService_PropertyDef*
CosPropertyService_PropertySetDef_impl::get(const char* property_name,
					    CORBA_ULong* pos)
{
    for(CORBA_ULong i = 0 ; i < propertySeq_.length() ; i++)
	if(!strcmp(propertySeq_[i].property_name, property_name))
	{
	    if(pos)
		*pos = i;
	    return &propertySeq_[i];
	}

    return 0;
}

// ----------------------------------------------------------------------

CosPropertyService_PropertyNamesIterator_impl::
CosPropertyService_PropertyNamesIterator_impl(
    CORBA_ORB_ptr orb,
    const CosPropertyService_PropertyNames& names)
    : orb_(CORBA_ORB::_duplicate(orb)),
      idx_(0),
      names_(names)
{
}

void
CosPropertyService_PropertyNamesIterator_impl::reset()
{
    idx_ = 0;
}

CORBA_Boolean
CosPropertyService_PropertyNamesIterator_impl::
next_one(CosPropertyService_PropertyName& property_name)
{
    if(idx_ < names_.length())
    {
        property_name = CORBA_string_dup(names_[idx_++]);
        return true;
    }
    else
    {
	property_name = CORBA_string_dup("");
        return false;
    }
}

CORBA_Boolean
CosPropertyService_PropertyNamesIterator_impl::
next_n(CORBA_ULong how_many, CosPropertyService_PropertyNames*& property_names)
{
    CORBA_ULong len = names_.length();
    CORBA_ULong num = len - idx_ < how_many ? len - idx_ : how_many;

    property_names = new CosPropertyService_PropertyNames(num);

    for(CORBA_ULong i = 0 ; i < num ; i++)
        (*property_names)[i] = CORBA_string_dup(names_[idx_++]);
 
    return how_many <= num;
}

void
CosPropertyService_PropertyNamesIterator_impl::destroy()
{
    CosPropertyService_PropertyNamesIterator_var self = _this();
    orb_ -> disconnect(self);
}

// ----------------------------------------------------------------------

CosPropertyService_PropertiesIterator_impl::
CosPropertyService_PropertiesIterator_impl(
    CORBA_ORB_ptr orb,
    const CosPropertyService_PropertyDefs& properties)
    : orb_(CORBA_ORB::_duplicate(orb)),
      idx_(0),
      properties_(properties)
{
}

void
CosPropertyService_PropertiesIterator_impl::reset()
{
    idx_ = 0;
}

CORBA_Boolean
CosPropertyService_PropertiesIterator_impl::
next_one(CosPropertyService_Property*& aproperty)
{
    aproperty = new CosPropertyService_Property;

    if(idx_ < properties_.length())
    {
	aproperty -> property_name = properties_[idx_].property_name;
	aproperty -> property_value = properties_[idx_].property_value;
	idx_++;
        return true;
    }
    else
        return false;
}

CORBA_Boolean
CosPropertyService_PropertiesIterator_impl::
next_n(CORBA_ULong how_many, CosPropertyService_Properties*& nproperties)
{
    CORBA_ULong len = properties_.length();
    CORBA_ULong num = len - idx_ < how_many ? len - idx_ : how_many;

    nproperties = new CosPropertyService_Properties(num);
    nproperties -> length(num);

    for(CORBA_ULong i = 0 ; i < num ; i++)
    {
	(*nproperties)[i].property_name = properties_[idx_].property_name;
	(*nproperties)[i].property_value = properties_[idx_].property_value;
	idx_++;
    }
 
    return how_many <= num;
}

void
CosPropertyService_PropertiesIterator_impl::destroy()
{
    CosPropertyService_PropertiesIterator_var self = _this();
    orb_ -> disconnect(self);
}

// ----------------------------------------------------------------------

CosPropertyService_PropertySet_ptr
CosPropertyService_PropertySetFactory_impl::create_propertyset()
{
    return new CosPropertyService_PropertySetDef_impl(orb_);
}

CosPropertyService_PropertySet_ptr
CosPropertyService_PropertySetFactory_impl::
create_constrained_propertyset(
    const CosPropertyService_PropertyTypes& allowed_property_types,
    const CosPropertyService_Properties& allowed_properties)
{
    CosPropertyService_PropertyDefs p;
    p.length(allowed_properties.length());

    for(CORBA_ULong i = 0 ; i < allowed_properties.length() ; i++)
    {
	p[i].property_name = allowed_properties[i].property_name;
	p[i].property_value = allowed_properties[i].property_value;
	p[i].property_mode = CosPropertyService_normal;
    }

    return new CosPropertyService_PropertySetDef_impl(orb_,
						      allowed_property_types,
						      p);
}

CosPropertyService_PropertySet_ptr
CosPropertyService_PropertySetFactory_impl::create_initial_propertyset(
    const CosPropertyService_Properties& initial_properties)
{
    //
    // Sequence with exceptions
    //
    CosPropertyService_MultipleExceptions mex;

    CosPropertyService_PropertyDefs p;
    p.length(initial_properties.length());

    for(CORBA_ULong i = 0 ; i < initial_properties.length() ; i++)
    {
	if(!strlen(initial_properties[i].property_name))
	{
	    CORBA_ULong len = mex.exceptions.length();
	    mex.exceptions.length(len + 1);
	    mex.exceptions[len].reason =
		CosPropertyService_invalid_property_name;
	    mex.exceptions[len].failing_property_name =
		initial_properties[i].property_name;
	    continue;
	}

	p[i].property_name = initial_properties[i].property_name;
	p[i].property_value = initial_properties[i].property_value;
	p[i].property_mode = CosPropertyService_normal;
    }

    //
    // Check for exceptions
    //
    if(mex.exceptions.length())
	throw mex;
 
    return new CosPropertyService_PropertySetDef_impl(orb_, p);
}

// ----------------------------------------------------------------------

CosPropertyService_PropertySetDef_ptr
CosPropertyService_PropertySetDefFactory_impl::create_propertysetdef()
{
    return new CosPropertyService_PropertySetDef_impl(orb_);
}

CosPropertyService_PropertySetDef_ptr
CosPropertyService_PropertySetDefFactory_impl::
create_constrained_propertysetdef(
    const CosPropertyService_PropertyTypes& allowed_property_types,
    const CosPropertyService_PropertyDefs& allowed_property_defs)
{
    return new CosPropertyService_PropertySetDef_impl(orb_,
						      allowed_property_types,
						      allowed_property_defs);
}

CosPropertyService_PropertySetDef_ptr
CosPropertyService_PropertySetDefFactory_impl::create_initial_propertysetdef(
    const CosPropertyService_PropertyDefs& initial_property_defs)
{
    //
    // Sequence with exceptions
    //
    CosPropertyService_MultipleExceptions mex;

    for(CORBA_ULong i = 0 ; i < initial_property_defs.length() ; i++)
    {
	if(!strlen(initial_property_defs[i].property_name))
	{
	    CORBA_ULong len = mex.exceptions.length();
	    mex.exceptions.length(len + 1);
	    mex.exceptions[len].reason =
		CosPropertyService_invalid_property_name;
	    mex.exceptions[len].failing_property_name =
		initial_property_defs[i].property_name;
	    continue;
	}
    }

    //
    // Check for exceptions
    //
    if(mex.exceptions.length())
	throw mex;
 
    return new CosPropertyService_PropertySetDef_impl(orb_,
						      initial_property_defs);
}
