/*************************************************************************
**
**  SCSI Tape Utility program - STU.C
**
**      This program displays general information about the specified
**      device, based on SCSI Inquiry data from the device.  For the
**      TZ85, TZ86, TZ87, TZ87XT, TZ88, TZ89 tape products, it also can get
**      and display detailed subsystem revision information.
**
**      It also gives access to some of the unique features of the TZ8x
**      products, which include:
**
**      1. Lets the user change the ForceDensity EEPROM parameter,
**
**      2. Lets the user reformat the current media to a specified
**         density (media must be loaded, and the ForceDensity param
**         must be set to zero.)
**
**         NOTE: the VMS INIT command (up to V5.5-2 at least) will cause
**         the device to go to its default density (for the TZ86 that
**         means TZ86 format) since it (or the MKDriver actually) issues
**         a Mode Select w/ density field set to 0 before doing a write
**         from BOT.  Only the EEPROM ForceDensity parameter (see 2 above)
**         will over-ride this.
**
**      3. Lets the user display error logs and EEROM parameters (TZ87)
**
**  Disclaimer: Quantum Corp. makes no guarantees, express or
**  implied, about the correct or expected operation of this program,
**  and accepts no liabilities related to its use.
**
**  However, it has been tested, and is believed to behave as documented.
**  The software contained in this file is being provided as a convenience
**  to users, and can be considered "FreeWare".  This source file can be
**  modified, enhanced, and copied without permission or limit.
**
**************************************************************************
**  Update History:     (Note: Update revision info in printfs in main)
**
**  v1.22   Ralf-Peter Rohbeck      08-May-1999
**      Merged Dick Stone's HP-UX version, added Linux support and threw
**      in some untested Solaris code.
**      Added command line options.
**      Added set_any_eerom_param(), load(), RecoverDir()
**      Please let me know about any corrections - mailto:rrohbeck@qntm.com
**
**  v1.21   Hitesh Trivedi          22-Feb-1996
**      Fixed forcedensity for the DLT7000
**
**  v1.20   Hitesh Trivedi          17-Nov-1995
**      Added support for the DLT7000 -- same as the DLT6000
**
**  v1.19   Hitesh Trivedi           2-Nov-1995
**      Increased the Timeout for a drivecup because it takes a while longer
**          on the DLT6000
**
**  v1.18   Hitesh Trivedi          11-Aug-1995
**      Well took out large mem model 'cause we're not gonna do it anymore
**
**  v1.17   Hitesh Trivedi          13-Jul-1995
**      Added code to be able to deal with TZ89 large memory models as
**          well as small memory model
**
**  v1.16   Michelle Hall           24-Mar-1995
**      Change the way drive type is identified.  Key off of the product family
**          code if it exists in the Inquiry Data, else key off of the drive
**          software revision level.
**
**  v1.15   Tim Johnson             02-Mar-1995
**      Made changes to allow the image validation for TZ89.
**
**  v1.14   Weston Clarke           16-Feb-1995
**      Change the way drive type is identified.  Key off the drive name
**          found in the inquiry data.
**
**  v1.13   Hitesh Trivedi          11-Nov-1994
**      Added code in so that we could stu'in DLT6000 code. This required
**          code space changes and locations.
**
**  v1.12   Mike O'Brien            31-Oct-1994
**      Added "#pragma nomember_alignment". This was for the VAX AXP
**          C compiler. prior to this, the AXP compiler would align things
**          on long word boundries, padding out where needed. This created
**          invalid CDB lengths.
**
**  v1.10   Dave Cressman           26-Sep-1994
**      Updated v1.7 change to decode LibraryDrive value & always print
**        it (before, it only got printed if in an embedded loader!).
**
**  v1.9    Hitesh Trivedi          7-Jul-1994
**      N Error Log Page again. Made it compatible with older versions of
**          controller code and the tz88.
**
**  v1.8    Hitesh Trivedi          29-Jun-1994
**      Made sure that the Page_cnt in Get_Logdata was initialized before
**          getting the last n errors. This was found when the eerom page
**          was cleared and we still thought that there were XXXX bytes still
**          left and reported them.
**
**  v1.7    Dave Cressman           10-Jun-1994
**      Display new Inquiry data byte for LibraryDrive code.
**
**  v1.6    Hitesh Trivedi          2-May-1994
**      Fixed the n error log page so that STU is able to correctly receive
**          n error log pages as well as for version 20 and greater where
**          there was significant work done to reformat the data.
**
**  v1.5    Hitesh Trivedi          25-Mar-1994
**      Modified the way we report n error log because a change was made
**      in the controller code to report it so that it conforms to the SCSI-2
**      spec a little better (lists were being truncated).
**
**  v1.4                            22-Mar-1994
**      Instead of using a fixed "sleep" time while waiting for FUP to
**        complete, poll unit every second until done.
**      Check EDC values to verify success/failure of a FUP.
**      Tweek FUP process display info.
**      Misc cleanup & fixes to update new edc/rev globals properly from
**        Inquiry data.
**
**  v1.3                            18-Mar-1994
**      Don't byte-swap TZ87 image on fup, if version > v10.
**      Slightly increase fup sleep time, due to fup timeout manuf saw.
**      Enhance display of fup process info (for manuf mostly).
**
**  v1.2                            03-Mar-1994
**      Enhanced the CUP validation errors so that we know a little
**          more as to why it may have failed.
**
**  V1.1    First Release           27-Jan-1994
**      This code used to timeout a QIO, if a firmware update to the TZ87
**        was done which included a servo code update, as well as the
**        SCSI code.  Fixed this bug, and tested.
**
**  V1.0    First Release           24-Jan-1994
**      Built from in-house tools.  Pre-releases did not decode the TZ87
**        serial number properly.  This bug has been fixed & tested.
**
**************************************************************************
*/

#define DEBUG 0 /* Debug level, the higher the more dumps */

/* Find out where we are
*/
#ifdef hpux
#define HPUX 1
#endif
#ifdef __hpux
#define HPUX 1
#endif
#ifdef __hpux__
#define HPUX 1
#endif
#ifdef __HPUX__
#define HPUX 1
#endif

#ifdef __linux__
#define LINUX 1
#endif
#ifdef linux
#define LINUX 1
#endif

#ifdef SUN
#define SUNOS 1
#endif

/*
**
**      include files and definitions
**
*/
#ifdef VMS

#include <stdlib.h>
#include <starlet.h>
#include <stdio.h>
#include <ctype.h>
#include <ssdef.h>
#include <iodef.h>
#include <descrip.h>
#include <dvidef.h>
#include <string.h>
#include <time.h>
#include <unixio.h>
#include <file.h>

#pragma nomember_alignment

#else /* ifdef VMS */

#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <sys/file.h>
#include <sys/fcntl.h>
#include <sys/ioctl.h>
#include <sys/signal.h>
#include <sys/errno.h>
#include <sys/types.h>
#include <sys/time.h>
#ifdef HPUX
#include <sys/scsi.h>
#endif /* ifdef HPUX */
#ifdef LINUX
#include <scsi/scsi.h>
#include <scsi/sg.h>
#endif /* ifdef LINUX */
#ifdef SUNOS
#include <memory.h>
#include <sys/scsi/scsi.h>
#endif /* ifdef SUNOS */

#endif /* ifdef VMS */

/* For some reason, the HP-UX version didn't use function prototypes,
   neither did it use struct initializations.
   Maybe some idiosyncrasy of the compiler...
*/
#ifdef HPUX
#undef FUNCTION_PROTOTYPES
#define NO_AUTO_STRUCT_INIT
#else
#define FUNCTION_PROTOTYPES
#undef NO_AUTO_STRUCT_INIT
#endif

#ifdef FUNCTION_PROTOTOTYPES
#define FUNC_PROTO(x) (x)
#define FUNC_PROTO2(x1,x2) (x1,x2)
#define FUNC_PROTO3(x1,x2,x3) (x1,x2,x3)
#define FUNC_PROTO5(x1,x2,x3,x4,x5) (x1,x2,x3,x4,x5)
#else /* #ifdef FUNCTION_PROTOTOTYPES */
#define FUNC_PROTO(x) ()
#define FUNC_PROTO2(x1,x2) ()
#define FUNC_PROTO3(x1,x2,x3) ()
#define FUNC_PROTO5(x1,x2,x3,x4,x5) ()
#endif /* #ifdef FUNCTION_PROTOTOTYPES */

#if defined(HPUX)
#define LITTLEENDIAN 1
#endif

#if defined(LINUX)
#include <endian.h>
#if !defined(__LITTLE_ENDIAN) || __LITTLE_ENDIAN==1234
#define LITTLEENDIAN 1
#endif /* if !defined(__LITTLE_ENDIAN) || __LITTLE_ENDIAN==1234 */
/* For old Linuces with buffer size restrictions */
#define LINUX_CHECK_FOR_BUFFER_MAX 1
#ifndef SG_BIG_BUFF
#define SG_BIG_BUFF 4096 /* default size for old versions */
#endif /* ifdef SG_BIG_BUFF */
#define MAX_IO_SIZE (SG_BIG_BUFF-sizeof(struct sg_header)-12)
#else /* if defined(LINUX) */
#define LINUX_CHECK_FOR_BUFFER_MAX 0
#endif /* if defined(LINUX) */

#undef SCSI_AUTOSENSE
/* Does the SCSI passthrough driver perform automatic SCSI Request Sense? */
#if defined(HPUX) || defined(LINUX)
#define SCSI_AUTOSENSE 1
#endif


/*
**  Define the SCSI commands and modifiers
*/
#define SUCCESS 1
#define TRUE 1
#define FALSE 0

#define OPCODE              0
#define FLAGS               1
#define COMMAND_ADDRESS     2
#define COMMAND_LENGTH      3
#define DATA_ADDRESS        4
#define DATA_LENGTH         5
#define PAD_LENGTH          6
#define PHASE_TIMEOUT       7
#define DISCONNECT_TIMEOUT  8

#define FLAGS_WRITE         0
#define FLAGS_READ          1
#define FLAGS_DISCONNECT    2

#define GK_EFN 1
#define SCSI_STATUS_MASK 0X3E

#define MAX_INQ_DATA        0xff
#ifdef LINUX
#define MAX_RQS_DATA        16 /* Linux only likes 16 bytes */
#else
#define MAX_RQS_DATA        0xff
#endif
#define MAX_LOG_DATA        0xff
#define Log_K_ParamLength   0x3fff
#define MAXNERRPARAMS       28
/*
 * SCSI Inquiry Device types.
 */
#define DISK_DEV    0
#define TAPE_DEV    1
#define LOADR_DEV   8

/*
 * Definitions used for Firmware Update routines.
 */
#define TZA_IMAGE_SIZE      0x40000
#define TZA_RELEASE_LOC     (TZA_IMAGE_SIZE - 44 + 5)
#define TZA_MINOR_LOC       (TZA_IMAGE_SIZE - 44 + 6)
#define TZA_MAJOR_LOC       (TZA_IMAGE_SIZE - 44 + 7)
#define TZA_TIME_STAMP_LOC  (TZA_IMAGE_SIZE - 44 + 12)
#define TZA_EDC_LOC         (TZA_IMAGE_SIZE - 44 + 40)

/*
 * EDC calculation stops in different spots for different products.
 * The IMAGE_SIZE tells us where to stop the EDC calculation. The
 * entire size of the image is the IMAGE_SIZE plus the EDC_OFFSET
 * and the DRIVE_CODE_SIZE. The TZ87 begins with an EDC_OFFSET of
 * zero.
 */
#define TZC_DRIVE_CODE_SIZE (0x10000)
#define TZC_IMAGE_SIZE      (0x80000 + TZC_DRIVE_CODE_SIZE)
#define TZC_RELEASE_LOC     (TZC_IMAGE_SIZE - 44 + 5)
#define TZC_MINOR_LOC       (TZC_IMAGE_SIZE - 44 + 6)
#define TZC_MAJOR_LOC       (TZC_IMAGE_SIZE - 44 + 7)
#define TZC_TIME_STAMP_LOC  (TZC_IMAGE_SIZE - 44 + 12)
#define TZC_EDC_LOC         (TZC_IMAGE_SIZE - 44 + 40)

#define TZE_EDC_OFFSET      8
#define TZE_IMAGE_SIZE      ((0x80000 - TZE_EDC_OFFSET) + TZC_DRIVE_CODE_SIZE)
#define TZE_RELEASE_LOC     (TZE_IMAGE_SIZE - 44 + 5)
#define TZE_MINOR_LOC       (TZE_IMAGE_SIZE - 44 + 6)
#define TZE_MAJOR_LOC       (TZE_IMAGE_SIZE - 44 + 7)
#define TZE_TIME_STAMP_LOC  (TZE_IMAGE_SIZE - 44 + 12)
#define TZE_EDC_LOC         (TZE_IMAGE_SIZE - 44 + 40)

#define TZF_EDC_OFFSET      4
#define TZF_DRIVE_CODE_SIZE (0x10000)
#define TZF_IMAGE_SIZE      ((0x80000 - TZF_EDC_OFFSET) + TZF_DRIVE_CODE_SIZE)
#define TZF_RELEASE_LOC     (TZF_IMAGE_SIZE - 44 + 5)
#define TZF_MINOR_LOC       (TZF_IMAGE_SIZE - 44 + 6)
#define TZF_MAJOR_LOC       (TZF_IMAGE_SIZE - 44 + 7)
#define TZF_TIME_STAMP_LOC  (TZF_IMAGE_SIZE - 44 + 12)
#define TZF_EDC_LOC         (TZF_IMAGE_SIZE - 44 + 40)

#define TZG_DRIVE_CODE_SIZE (0x20000)
#define TZG_ZONE_SIZE       (0x8000)    /* Special area for drive rev info. */
#define TZG_EDC_OFFSET      16
#define TZG_IMAGE_SIZE      ((0x100000 - TZG_EDC_OFFSET) \
                                + TZG_DRIVE_CODE_SIZE + TZG_ZONE_SIZE)

#define TZG_RELEASE_LOC     (TZG_IMAGE_SIZE - 44 + 5)
#define TZG_MINOR_LOC       (TZG_IMAGE_SIZE - 44 + 6)
#define TZG_MAJOR_LOC       (TZG_IMAGE_SIZE - 44 + 7)
#define TZG_TIME_STAMP_LOC  (TZG_IMAGE_SIZE - 44 + 12)
#define TZG_EDC_LOC         (TZG_IMAGE_SIZE - 44 + 40)

#define TZGS_DRIVE_CODE_SIZE (0x20000)
#define TZGS_ZONE_SIZE       (0x2000)    /* Special area for drive rev info. */
#define TZGS_EDC_OFFSET      16

/*
#define TZGS_IMAGE_SIZE      ((0x100000 - TZGS_EDC_OFFSET) \
                                + TZGS_DRIVE_CODE_SIZE + TZGS_ZONE_SIZE)
*/

#define TZGS_IMAGE_SIZE      ((0x80000 - TZGS_EDC_OFFSET) \
                                + TZGS_DRIVE_CODE_SIZE + TZGS_ZONE_SIZE)

#define TZGS_RELEASE_LOC     (TZGS_IMAGE_SIZE - 44 + 5)
#define TZGS_MINOR_LOC       (TZGS_IMAGE_SIZE - 44 + 6)
#define TZGS_MAJOR_LOC       (TZGS_IMAGE_SIZE - 44 + 7)
#define TZGS_TIME_STAMP_LOC  (TZGS_IMAGE_SIZE - 44 + 12)
#define TZGS_EDC_LOC         (TZGS_IMAGE_SIZE - 44 + 40)

/*
 * Product Family Codes - from the Vendor Unique Byte Offset 36   Upper 4 bits
 */
#define TK85_DRIVE              0x01    /* (i.e. TF85,TZ85,TF857,TZ857)     */
#define TK86_DRIVE              0x02    /* (i.e. TF86,TZ86,TF867,TZ867)     */
#define TK87_DRIVE              0x03    /* (i.e. TZ87,TZ877,DLT2000)        */
/*                              0x04                                        */
#define TK88_DRIVE              0x05    /* (i.e. DLT4000, DLT4500)          */
#define TK87XT_DRIVE            0x06    /* (i.e. DLT2000XT )                */
#define TK89_DRIVE              0x07    /* (i.e. DLT6000)                   */
#define LATEST_DRIVE            0x07    /* (latest drive supported by STU)  */

#define InqDat_V_ProdFam        0x04    /* Bit Position of Product Family   */
#define InqDat_M_ProdFam        0xF0    /* Product Family Code Mask         */
#define InqDat_M_RelOpt         0x0F    /* Release Option Field Mask        */
/*
 * Drive Density: these are defined to be equal to the corresponding
 *              ForceDensity parameter's possible values
 */
#define TK85_DENSITY      1
#define TK86_DENSITY      2
#define TK87_DENSITY      3
#define TK88_DENSITY      4
#define TK89_DENSITY      5

#define MIN_TK86_ROM_REV    0x40
#define MIN_TK87_ROM_REV    0x80
#define MIN_TK88_ROM_REV    0xC0
#define MIN_TK89_ROM_REV    0x100

#ifdef VMS
/*
 * Define QIO commands and modifiers.
 */
#define DEV$V_MNT       0x00080000
#define DEV$V_ALL       0x00800000
#define DEV$V_FOR       0x01000000
#define DEV$V_SWL       0x02000000
#endif /* ifdef VMS */

#define EF_PASS         4           /* Event flag               */
#define EF_PASS_M       0x010       /* Event flag-4 mask        */

#define CHUNK_SIZE_BIG      0x8000
#define CHUNK_SIZE_SMALL    0x2000

#define MEM_MODEL_LARGE     0
#define MEM_MODEL_SMALL     1

#ifdef VMS
/*
 * Item list descriptor structure
 */
struct dsc$descriptor_il {
    unsigned short int  dsc$w_blength;
    unsigned short int  dsc$w_item;
    unsigned char       *dsc$a_buffer_b;
    unsigned int        *dsc$a_rlength_l;
    unsigned int        dsc$l_end;
};
#endif /* ifdef VMS */

/*
 * SCSI Command Descriptor Block structure definitions.
 */
struct TUR {                        /* Test Unit Ready Command structure */
    unsigned char Opcode;
    unsigned char Rsvd1;
    unsigned char Rsvd2;
    unsigned char Rsvd3;
    unsigned char Rsvd4;
    unsigned char Cntrl;
};
struct RQS {                        /* Request Sense Command structure */
    unsigned char Opcode;
    unsigned char Rsvd1;
    unsigned char Rsvd2;
    unsigned char Rsvd3;
    unsigned char AllocLen;
    unsigned char Cntrl;
};

struct INQ {                        /* Inquiry Command structure */
    unsigned char Opcode;
    unsigned char EVPD;
    unsigned char PageCode;
    unsigned char Rsvd3;
    unsigned char AllocLen;
    unsigned char Cntrl;
};

struct WRB {                        /* Write Buffer Command structure */
    unsigned char Opcode;
    unsigned char Mode;
    unsigned char Buff_ID;
    unsigned char BufferOffset[3];
    unsigned char AllocLen[3];
    unsigned char Cntrl;
};
struct MSE {                        /* 6 byte Mode Sense Command structure */
    unsigned char Opcode;
    unsigned char LUN;
    unsigned char PageCode;
    unsigned char Rsvd3;
    unsigned char AllocLen;
    unsigned char Cntrl;
};

struct MSE10 {                      /* 10 byte Mode Sense Command structure */
    unsigned char Opcode;
    unsigned char LUN;
    unsigned char PageCode;
    unsigned char Rsvd3;
    unsigned char Rsvd4;
    unsigned char Rsvd5;
    unsigned char Rsvd6;
#ifdef LITTLEENDIAN
    unsigned char AllocLen1;
    unsigned char AllocLen0;
#else /* ifdef LITTLEENDIAN */
    unsigned short int AllocLen;
#endif /* ifdef LITTLEENDIAN */
    unsigned char Cntrl;
};

struct MSE10DATA {                  /* Mode Page Data structure */
#ifdef LITTLEENDIAN
    unsigned char ModeLen1;
    unsigned char ModeLen0;
#else /* ifdef LITTLEENDIAN */
    unsigned short ModeLen;
#endif /* ifdef LITTLEENDIAN */
    unsigned char MedTyp;
    unsigned char CshWp;
    unsigned char Rsvd4;
    unsigned char Rsvd5;
#ifdef LITTLEENDIAN
    unsigned char DesLen1;
    unsigned char DesLen0;
#else /* ifdef LITTLEENDIAN */
    unsigned short DesLen;
#endif /* ifdef LITTLEENDIAN */
    unsigned char PgCode;
    unsigned char PgLen;
    unsigned char Params[(0xFFFF-(0x08+2))];    /* total allocation - header */
};
struct MSE10DATA ModeSense_Page;  /* Mode Sense Page Data area. */

struct LSE {                        /* Log Sense Command structure */
    unsigned char Opcode;
    unsigned char Rsvd1;
    unsigned char PageCode;
    unsigned char Rsvd3;
    unsigned char Rsvd4;
    unsigned char Rsvd5;
    unsigned char ParPtr;
#ifdef LITTLEENDIAN
    unsigned char AllocLen1;
    unsigned char AllocLen0;
#else /* ifdef LITTLEENDIAN */
    unsigned short int AllocLen;
#endif /* ifdef LITTLEENDIAN */
    unsigned char Cntrl;
};

struct LSEDATAPGHDR {
    unsigned char PageCode;
    unsigned char Rsvd1;
    unsigned char PageLengthH;
    unsigned char PageLengthL;
};

struct LSEDATAPRMHDR {              /* Log Param Header         */
#ifdef LITTLEENDIAN
    unsigned char ParamNum1;
    unsigned char ParamNum0;
#else /* ifdef LITTLEENDIAN */
    unsigned short int ParamNum ;
#endif /* ifdef LITTLEENDIAN */
    unsigned char Flags;
    unsigned char ParamLength ;
};

struct LSEDATAOLD {                     /* Log Page Data structure(<V20)*/
    unsigned char PageCode;             /* Log Page Header starts here  */
    unsigned char Rsvd1;
    unsigned char PageLengthH;
    unsigned char PageLengthL;
    unsigned char ParamCodeH;           /* Log Parameter starts here    */
    unsigned char ParamCodeL;
    unsigned char Flags;
    unsigned char ParamLength;
    unsigned char ParamValue[Log_K_ParamLength];
};

struct LSEDATA {                        /* Log Page Data structure (>V20)*/
    struct LSEDATAPGHDR header;
    unsigned char ParamValue[(0x3FFF-(0x08+2))];    /* total allocation - header */
};

union U_LSEDATA {                       /* Log Page Data Structure      */
    struct LSEDATA LogSense_Page;
    struct LSEDATAOLD LogSense_PageOld;
} LogSense;

struct DRIVE_DATA {
    int     DriveType;
    char    *DriveStr;
};

struct LOAD {                        /* 6 byte Load Command structure */
    unsigned char Opcode;
    unsigned char LUN;
    unsigned char Rsvd1;
    unsigned char Rsvd2;
    unsigned char EOTRetenLoad;
    unsigned char Cntrl;
};

struct LOCATE {                        /* 10 byte Load Command structure */
    unsigned char Opcode;
    unsigned char LUN;
    unsigned char Rsvd1;
    unsigned char BlockAddress[4];
    unsigned char Rsvd2;
    unsigned char Partition;
    unsigned char Cntrl;
};

struct READ_POS {                        /* 10 byte Read Position Command structure */
    unsigned char Opcode;
    unsigned char LUN;
    unsigned char Rsvd[7];
    unsigned char Cntrl;
};

/*
**  Function Prototype Definitions
**  FUNC_PROTO(x) expands either to (x) or ()
*/
int  chk_drv FUNC_PROTO(char *);
int  inquiry_info();
int  get_drivetype_from_string();
void print_revinfo();
int  curr_density();
int  set_tape_format();
int  get_eerom_param();
int  set_eerom_param();
int  set_any_eerom_param();
int  load FUNC_PROTO(int);
int  RecoverDir();

void get_logdata();
int  get_sense_data();
int  scsi_qio FUNC_PROTO5(unsigned char *, int,  unsigned char*, int, int);
#ifdef SUNOS
static int call_uscsi(int fd, caddr_t cdb, unsigned char cdblen,
                      caddr_t buf, unsigned int buflen,
                      int rdwr);
#endif /* ifdef SUNOS */
void do_code_update();
char *get_image  FUNC_PROTO(int);
int  validate_image  FUNC_PROTO2(int, char *);
int  transmit_image FUNC_PROTO2(int, char *);
void wait_for_cup_completion ();
void show_timestamp  FUNC_PROTO2(int, char *);
void show_new_version  FUNC_PROTO2(int, char *);
void show_current_version ();

void do_byte_swap  FUNC_PROTO2(int, char *);
unsigned long int reverse_bytes  FUNC_PROTO(char *);
unsigned short int get_word FUNC_PROTO(unsigned char*);
unsigned long  int get_long FUNC_PROTO(unsigned char*);
void my_gets FUNC_PROTO2(char *,unsigned buflen);
#if DEBUG
#define MAX_DUMP 128 /* maximum number of bytes in data dump */
void dump FUNC_PROTO3(char *,unsigned char *,unsigned);
#endif

/*
**  Global parameter declarations for the drives
*/

unsigned char   inquiry_data[0xff];
unsigned char   vpd00_data[0xff];
unsigned char   vpd80_data[0xff];
unsigned char   vpdC0_data[0xff];

unsigned char   tur_cmd[6] =  {0x0, 0, 0, 0, 0, 0 };        /* Test Unit Rdy */

unsigned char   mode_data[0xff];
unsigned char   read_blklim_data[6];

int tape_chan;              /* I/O channel number */
int device_type;            /* Set to SCSI Inquiry device type */
int curr_drive_type = 0;    /* Setup based on Inquiry data. Init to invalid. */

unsigned char cnt_rev;      /* Setup based on Inquiry data */
unsigned char drive_rev;    /* Setup based on Inquiry data */
unsigned long cnt_edc;
unsigned short drive_edc;
unsigned char image_cnt_rev;
unsigned long image_cnt_edc;
unsigned char image_drive_rev;    /* TZ87/88 only: drive rev from image */
unsigned short image_drive_edc;

unsigned char write_prot;   /* Write Protect flag */

char scsi_status;               /* Status of last SCSI command */
unsigned char scsi_sense;       /* Sense code */
unsigned char scsi_ASC;         /* ASC */
unsigned char scsi_ASCQ;        /* ASCQ */
int scsi_quiet=0;               /* Don't display error messages, just fill in
                                   status if != 0 */

char print_busy;                /* display BUSY errpr message if != 0 */
int chunk_size;
int mem_model;

int verbose=0;                  /* Display status as we move along, the higher
                                   the more */

#ifdef SCSI_AUTOSENSE
int bypass_req_sens_cmd = FALSE;/* Don't exec REQUEST SENE -
                                   get sense data from the array */
unsigned char sense_data[MAX_RQS_DATA];
#endif


/*++
***********************************************************************
**
**  **-main
**
**      SCSI Tape Utility (STU) - Main function
**
**  FUNCTIONAL DESCRIPTION:
**      Oversees operation of this program, mostly handling the menu
**      interface, calling sub-functions to process the user's requests.
**
**  FORMAL PARAMETERS:
**      Program expects only the VMS device name on the command line, of
**      the target SCSI device (ex: MKB500).
**
**  RETURN VALUE:
**      Nothing special unless error (see standard "exit" function)
**
***********************************************************************
--*/
#ifdef FUNCTION_PROTOTYPES
main (int argc, char *argv[])
#else
main (argc,argv)
int argc;
char *argv[];
#endif
{
    char input_line[80];
    char *cp;
    int choice;
    int i;
    int quiet=0;
    int myargc;
    char **myargv;


    myargc=argc;
    myargv=argv;
    if(myargc>=2&&myargv[1][0]=='-') /* optional parameters ? */
    { ++myargv; --myargc;
      for(cp=myargv[0]+1;*cp;++cp)
      { switch(*cp)
        { case 'q':
            quiet=1;
            break;
          case 'Q':
            quiet=2;
            break;
          case 'v':
            ++verbose;
            break;  
          case 'h':
          case 'H':
          case '?':
            puts( "DLT(tm) SCSI tape utility for "
#ifndef VAXC
#ifdef VMS
                  "VMS"
#endif
#ifdef HPUX
                  "HP-UX"
#endif
#ifdef SOLARIS
                  "Solaris"
#endif
#ifdef LINUX
                  "Linux"
#endif
#ifdef LITTLEENDIAN
                  " little-endian"
#else
                  " big-endian"
#endif
                  "\n"
                  "Run with tape device name as argument"
#ifdef LINUX
                  " (use sg device with Linux)"
#endif
                  "\nOptions:\n"
                  "-h   Help\n"
                  "-q   Quiet, don't display header\n"
                  "-Q   Very quiet, don't display menu\n"
                  "-v   verbosity level (can be used multiple times to increase verbosity)\n"
                  "\n"
                  "Normally, a menu is displayed.\n"
#endif
                );
            break;
          default:
            printf("%c not understood.\n",*cp);
            break;
        }
      }
    }
    if(!quiet)
    { printf(  "  *************************************************************");
      printf("\n  **    SCSI Tape Utility - STU.EXE   V1.22 May 1999          *");
      printf("\n  **                                                          *");
      printf("\n  **    Invocation: STU <device name>                         *");
      printf("\n  *************************************************************");
      printf("\n");
    }

    if (myargc < 2) {
#ifdef VAXC
	printf("Usage:\n$ STU [-option...] <DLTdevicename>\nSCSI tape utility for DLT(tm) tape drives\n"
#else
        printf("Usage:\n"
# ifdef VMS
               "$ MCR "
# endif
               "%s [-option...] <DLTdevicename>\n"
# ifdef VMS
               "Or\n"
               "$ STU:=$%s\n"
               "$ STU [-option...] <DLTdevicename>\n"
# endif
               "SCSI tape utility for DLT(tm) tape drives\n"
               ,argv[0]
# ifdef VMS
               ,argv[0]
# endif
#endif
              );
        exit(1);
    }
    /*
     *  Parse the command line parameters
     */
    tape_chan = chk_drv (*++myargv);
    if (tape_chan == -1) {
        printf ("\n Unable to open a channel to the drive ... Aborting");
        printf ("\n    Is drive already in use? Do you have access right? ");
        printf ("Check drive %s\n", *myargv);
        exit (1);
    }
    print_busy = TRUE;      /* initialize */

    /*
     * Get any pending Sense Data
     */
    for (i = 0; i < 5; i++) {
        /* get up to 5 queued Unit Attentions. */
        if (get_sense_data() == TRUE) {
            break;
        }
    }
    /*
     * Get the necessary Inquiry information.
     */
    inquiry_info();

    /* Report unknown drive if drive type not found. */
    if (curr_drive_type == 0) {
        printf("Sorry, STU is unfamiliar with this drive type.");
        return(SUCCESS);
    }

    while (TRUE) {
        chunk_size = CHUNK_SIZE_SMALL;      /* chunks size for data transfer */
        mem_model = MEM_MODEL_SMALL;
        if(quiet<2)
        { 
          printf ( "\n\n");
          printf ("  SCSI Tape Utility Options Menu\n");
          printf ("-----------------------------------------\n");
          printf ("    1  Print Revision Information.\n");
          printf ("    2  Display Current tape format.\n");
          printf ("    3  Change Current tape format.\n");
          printf ("    4  Display EEROM Parameters\n");
          printf ("    5  Change ForceDensity EEROM Parameter.\n");
          printf ("    6  Display Last N Error Logs.\n");
          printf ("    7  Perform a Drive Firmware Update.\n");
          printf ("    8  Change any EEROM Parameter.\n");
          printf ("    9  Load.\n");
          printf ("   10  Unload.\n");
          printf ("   11  Recover directory.\n");
          printf ("   99  Exit.\n");
          printf ("\nEnter # of your choice: ");
        }
        my_gets (input_line,sizeof(input_line));
        choice = atoi (input_line);
        switch (choice) {
            case 1:
                print_revinfo();
                break;
            case 2:
                if (scsi_qio(tur_cmd, 6, read_blklim_data, 0, FLAGS_READ)) {
                    curr_density(); /* Get and display current density */
                    if (write_prot) {
                        printf("\n  Current Media is write protected\n");
                    }
                }
                break;
            case 3:
                set_tape_format();
                break;
            case 4:
                get_eerom_param();
                break;
            case 5:
                set_eerom_param();
                break;
            case 6:
                get_logdata();
                break;
            case 7:
                do_code_update ();
                break;
            case 8:
                set_any_eerom_param();
                break;
            case 9:
                load(1);
                break;
            case 10:
                load(0);
                break;
            case 11:
                RecoverDir();
                break;
            case 99:
                return;
            default:
                printf ("Invalid Choice, try again.\n");
                break;
        }       
    }
} /* main */

/*++
***************************************************************************
**
**  **-chk_drv
**
**  FUNCTIONAL DESCRIPTION:
**      chk_drv assigns the channel to the selected tape drive
**      and checks if it is mounted foreign.  If either fails
**      program will exit.
**
**  FORMAL PARAMETERS:
**      name:
**          pointer to a character string containg the name of the
**          tape drive.
**
**  RETURN VALUE:
**      int channel - the channel assigned to the tape drive under test.
**                    -1 returned if any of the requirements are not
**                       satisfied.
**
***************************************************************************
--*/
int chk_drv(name)
char *name;
{
#ifdef VMS
    struct dsc$descriptor_s tape_desc;
    struct dsc$descriptor_il tape_char;
#endif
    unsigned char tape_cbuff[4];
    unsigned int *tape_cbp;
    int tape_rlen;
    int channel;
    int local_status;
    int dvi_iosb[2];


#ifdef VMS
    /*  Build name descriptor and assign channel    */
    tape_desc.dsc$a_pointer = name;
    tape_desc.dsc$b_class = DSC$K_CLASS_S;
    tape_desc.dsc$b_dtype = DSC$K_DTYPE_T;
    tape_desc.dsc$w_length = strlen( name );

    local_status = sys$assign( &tape_desc, &(channel), 0, NULL );
    if (local_status != SS$_NORMAL) {
        printf("ERROR-T0- Assign channel failed\n");
        channel = -1;
        return (channel);
    }

    /*  Check device charareristics     */
    tape_char.dsc$w_item = DVI$_DEVCHAR;
    tape_char.dsc$w_blength = 4;
    tape_char.dsc$a_buffer_b = tape_cbuff;
    tape_char.dsc$a_rlength_l = (unsigned int *) &tape_rlen;
    tape_char.dsc$l_end = 0;
    tape_cbp = (unsigned int *) tape_cbuff;

    local_status = sys$getdviw( NULL, channel, NULL, &tape_char,
                          dvi_iosb, NULL, NULL, NULL );

    if (local_status != SS$_NORMAL) {
        printf("ERROR-T0- Get drive info failed\n");
        channel = -1;
    }

    /*  Check that drive is mounted with a "/FOREIGN" qualifier */
#if FALSE
    if (((DEV$V_MNT | DEV$V_FOR) & *tape_cbp) == 0) {
        printf("ERROR-T0- Tape %s is not mounted Foreign\n", name );
        channel = -1;
        return (channel);
    }

    if ((DEV$V_SWL & *tape_cbp) != 0) {
        printf("ERROR - Tape %s is WRITE_LOCKED\n", name );
        channel = -1;
        return (channel);
    }
#endif
#else /* ifdef VMS */
#if defined(HPUX) || defined (SOLARIS)
  if ((channel = open(name,O_RDWR|O_NDELAY)) < 0)
#endif
#if defined (LINUX)
  if ((channel = open(name,O_RDWR)) < 0)
#endif
    {
    printf("Cannot access \'%s\'\n",name);
    channel = -1;
    }
#endif
    /*  All requirements have been satisfied, continue  */
    return (channel);
} /* chk_drv */

/*++
*************************************************************************
**
**  **-inquiry_info
**
**  FUNCTIONAL DESCRIPTION:
**      inquiry_info() sends various flavors of the SCSI Inquiry cmd
**      to get standard and DEC vendor unique info about the device.
**
**      The following globals are updated:
**          inquiry_data
**          cnt_rev
**          drive_rev
**          curr_drive_type
**          device_type
**          vpd80_data
**          vpdC0_data
**          cnt_edc
**          drive_edc
**
**  Drive Software Revs:
**  Type (Code)| SW Revision Range
**  =============================
**  TX85 (A)        0000 - 003F
**  TX86 (B)        0040 - 007F
**  TX87 (C)        0080 - 00BF
**  TX88 (E)        00C0 - 00FF
**
**  FORMAL PARAMETERS: none
**
**  RETURN VALUE: TRUE if success, else FALSE
**
*************************************************************************
--*/
int inquiry_info()
{
    int success;
    int index, inq_index;
    struct INQ *Inquiry_Ptr;      /* Pointer to the CDB */
    struct INQ Inquiry_CDB;       /* Allocation of CDB */

    Inquiry_Ptr = &Inquiry_CDB;
    Inquiry_Ptr->Opcode = 0x12;
    Inquiry_Ptr->EVPD = 0;
    Inquiry_Ptr->PageCode = 0;
    Inquiry_Ptr->Rsvd3 = 0;
    Inquiry_Ptr->AllocLen = MAX_INQ_DATA;
    Inquiry_Ptr->Cntrl = 0;

    /*
     * Get standard and VU Inquiry data bytes:
     */
    success = scsi_qio((char *)&Inquiry_CDB, sizeof(struct INQ),
                inquiry_data, MAX_INQ_DATA, FLAGS_READ);
    if (!success) {
        success = scsi_qio((char *)&Inquiry_CDB, sizeof(struct INQ),
                            inquiry_data, MAX_INQ_DATA, FLAGS_READ);
        if (!success) {
            printf("\n\t*** Inquiry cmd failed twice, aborting. ***");
            exit(1);
        }
    }
    /*
     * Initialize the SCSI Device_Type and the Drive_Rev globals.
     */
    device_type = (inquiry_data[0] & 0x1f);

    /* Set the Drive_rev global. */
    if ((inquiry_data[32] >= 'A') && (inquiry_data[32] <= 'F')) {
        drive_rev = (inquiry_data[32] - '7') << 4;
    } else {
        drive_rev = (inquiry_data[32] - '0') << 4;
    }
    if ((inquiry_data[33] >= 'A') && (inquiry_data[33] <= 'F')) {
        drive_rev = drive_rev | (inquiry_data[33] - '7');
    } else {
        drive_rev = drive_rev | (inquiry_data[33] - '0');
    }

    /*
     * Getting Current Drive Type:
     * If the product family code is availabe and non-zero, use it.
     * Else check the string (needed for early rev's of DLT2000/4000/6000)
     * Else assume older product and decipher from the Low byte drive software
     * revision level.
     */
    if ((inquiry_data[4] >= (36-4)) &&
        (curr_drive_type = ((inquiry_data[36]&InqDat_M_ProdFam)
                    >>InqDat_V_ProdFam))) {        /* available and non-zero */

    } else if (curr_drive_type = get_drivetype_from_string()) { /*check string*/

    } else {
        if (drive_rev < MIN_TK86_ROM_REV) {         /* TZ85 Drive? */
            curr_drive_type = TK85_DRIVE;
        } else if (drive_rev < MIN_TK87_ROM_REV) {  /* TZ86 Drive? */
            curr_drive_type = TK86_DRIVE;
        } else if (drive_rev < MIN_TK88_ROM_REV) {
            curr_drive_type = TK87_DRIVE;           /* TZ87 */
        } else {
            curr_drive_type = TK88_DRIVE;           /* TZ88 */
        }
    }

    cnt_rev = inquiry_data[37];
    /*
     * Now try to get the Vital Product Data Pages:
     */
    Inquiry_Ptr->EVPD |= 1;
    success = scsi_qio((char *)&Inquiry_CDB, sizeof(struct INQ),
                        vpd00_data, MAX_INQ_DATA, FLAGS_READ);
    if (success == FALSE) {
        printf("BAD_STATUS: Inquiry VPD-0 failed\n");
    }

    Inquiry_Ptr->PageCode = 0x80;
    success = scsi_qio((char *)&Inquiry_CDB, sizeof(struct INQ),
                        vpd80_data, MAX_INQ_DATA, FLAGS_READ);
    if (success == FALSE) {
        printf("BAD_STATUS: Inquiry VPD-80 failed\n");
    }

    Inquiry_Ptr->PageCode = 0xC0;
    success = scsi_qio((char *)&Inquiry_CDB, sizeof(struct INQ),
                        vpdC0_data, MAX_INQ_DATA, FLAGS_READ);
    if (success == FALSE) {
        printf("BAD_STATUS: Inquiry VPD-C0 failed\n");
    }
    drive_edc = get_word(&vpdC0_data[4]);
    cnt_edc = get_long(&vpdC0_data[8]);
    return(SUCCESS);
} /* inquiry_info */

/*++
*************************************************************************
**
**  **-get_drivetype_from_string
**
**  FUNCTIONAL DESCRIPTION:
**
**  FORMAL PARAMETERS: none
**
**  RETURN VALUE: drive type/
**
*************************************************************************
--*/
int get_drivetype_from_string() {

    char Drive_Str[5];
    int index, inq_index;
    int type = 0;
    /* The drive strings to check for.  No need to specify TZ857 etc. */
    /* NOTE : Drive type must match those defined above. */
#ifndef NO_AUTO_STRUCT_INIT
    struct DRIVE_DATA Drive[] = {TK85_DRIVE, "TZ85", /* Drive type and string */
                                 TK86_DRIVE, "TZ86",
                                 TK87_DRIVE, "TZ87",
                                 TK88_DRIVE, "TZ88",
                                 TK89_DRIVE, "TZ89",
                                 TK87_DRIVE, "DLT2",
                                 TK88_DRIVE, "DLT4",
                                 TK89_DRIVE, "DLT6",
                                 TK89_DRIVE, "DLT7",
                                 0, ""};
#else /* ifndef NO_AUTO_STRUCT_INIT */
  struct DRIVE_DATA Drive[10];
  Drive[0].DriveType = TK85_DRIVE;
  Drive[0].DriveStr  = "TZ85"; /* Drive type and string */
  Drive[1].DriveType = TK86_DRIVE;
  Drive[1].DriveStr  = "TZ86";
  Drive[2].DriveType = TK87_DRIVE;
  Drive[2].DriveStr  = "TZ87";
  Drive[3].DriveType = TK88_DRIVE;
  Drive[3].DriveStr  = "TZ88";
  Drive[4].DriveType = TK89_DRIVE;
  Drive[4].DriveStr  = "TZ89";
  Drive[5].DriveType = TK87_DRIVE;
  Drive[5].DriveStr  = "DLT2";
  Drive[6].DriveType = TK88_DRIVE;
  Drive[6].DriveStr  = "DLT4";
  Drive[7].DriveType = TK89_DRIVE;
  Drive[7].DriveStr  = "DLT6";
  Drive[8].DriveType = TK89_DRIVE;
  Drive[8].DriveStr  = "DLT7";
  Drive[9].DriveType = 0;
  Drive[9].DriveStr  = "";
#endif /* ifndef NO_AUTO_STRUCT_INIT */

    /*
     * Determine the drive type.
     */
    /* Copy the 1st four characters of drive string from the inquiry data. */
    for (index = 0, inq_index = 16; index < 4; index++, inq_index++) {
        Drive_Str[index] = inquiry_data[inq_index];
    }

    /* Terminate the drive string with a null. */
    Drive_Str[4] = '\0';

    index = 0;          /* Reset the array index. */

    /* Search for the drive type in the drive data table. */
    do {
        /* If there is a match, set the type and break out. */
        if (!(strncmp(Drive_Str, Drive[index].DriveStr, 4))) {
            type = Drive[index].DriveType;
            break;
        }
        index++;
    } while (Drive[index].DriveStr[0] != '\0');
    return type;   /* return 0 if didn't find a match */

}


/*++
*************************************************************************
**
**  **-print_revinfo
**
**  FUNCTIONAL DESCRIPTION:
**      print_revinfo displays the data from the Inquiry Command in a
**      formated display on the output device.
**
**  FORMAL PARAMETERS: none
**
**  RETURN VALUE: void
**
*************************************************************************
--*/
void print_revinfo()
{
#ifndef NO_AUTO_STRUCT_INIT
    unsigned char read_blklim_cmd[6] = {0x05, 0, 0, 0, 0, 0 };
#else /* ifndef NO_AUTO_STRUCT_INIT */
  unsigned char read_blklim_cmd[6];
  read_blklim_cmd[0] = 0x05;
  read_blklim_cmd[1] = 0;
  read_blklim_cmd[2] = 0;
  read_blklim_cmd[3] = 0;
  read_blklim_cmd[4] = 0;
  read_blklim_cmd[5] = 0;
#endif /* ifndef NO_AUTO_STRUCT_INIT */
    unsigned char drive_rev;
    char input_line[80];
    char *s;

    inquiry_info();                 /* Get up-to-date info */
    if (device_type == TAPE_DEV) {
        scsi_qio(read_blklim_cmd, 6, read_blklim_data, 6, FLAGS_READ);
    }
    if (device_type == DISK_DEV) {
        s = "Random Access (Disk)";
    } else if (device_type == TAPE_DEV) {
        s = "Sequential Access (Tape)";
    } else if (device_type == LOADR_DEV) {
        s = "Medium Changer (Loader)";
    } else {
        s = "(Unknown)";
    }
    printf("\n\n    Vendor     SCSI Product ID     Rev    Device Type\n");
    printf(    "  ----------  ------------------  ------ ---------------\n");
    printf(    "  '%.8s'  '%.16s'  '%.4s'  %.1X = %s\n\n",
        &inquiry_data[8],
        &inquiry_data[16],
        &inquiry_data[32],
        device_type,
        s);
    if(inquiry_data[3]&0x40) printf (  "\t Term IOP Supported\n");
    if(inquiry_data[3]&0x80) printf (  "\t AENC Supported\n");
    if(inquiry_data[6]&0x40) printf (  "\t Medium Changer supported\n");
    if(inquiry_data[7]&0x01) printf (  "\t Soft Reset Supported\n");
    if(inquiry_data[7]&0x02) printf (  "\t CmdQue Supported\n");
    if(inquiry_data[7]&0x08) printf (  "\t Linked Cmds Supported\n");
    if(inquiry_data[7]&0x10) printf (  "\t Sync Xfers Supported\n");
    if(inquiry_data[7]&0x20) printf (  "\t WBus16 Supported\n");
    if(inquiry_data[7]&0x40) printf (  "\t WBus32 Supported\n");
    if(inquiry_data[7]&0x80) printf (  "\t RelAdr Supported\n");

    if (device_type == TAPE_DEV) {
        printf("\t Block Sizes Supported: min = %-u,  max = %-u\n",
            get_word(&read_blklim_data[4]),
            get_long(&read_blklim_data[0]));
    } else {
        printf("\n");
    }
    if (inquiry_data[4] <= (49-4))  {    /* VU data available? */
        return;     /* nope, don't decode garbage */
    }
    if (inquiry_data[36]&InqDat_M_RelOpt) {     /* Released Firmware? */
        /* Yes: T or V code */
        if (inquiry_data[38] == 0) {
            printf("\n\t FEPROM:: V%03u/", cnt_rev);
        } else {
            printf("\n\t FEPROM:: T%03u-%d/", cnt_rev, inquiry_data[38]);
        }
    } else {
        printf("\n\t FEPROM:: X%03u-%d/", cnt_rev, inquiry_data[38]);
    }
    printf("%08X", cnt_edc);
    printf(" (%-.24s)", &vpdC0_data[12]);
    printf( "   Pers: %01u-%01u\n",inquiry_data[41],inquiry_data[42]);

    printf("\t Dir=%03u EE=%03u.%03u HW=%03u\n",
        inquiry_data[43], inquiry_data[39],
        inquiry_data[40], inquiry_data[44]);

    if ((inquiry_data[32] >= 'A') && (inquiry_data[32] <= 'F')) {
        drive_rev = (inquiry_data[32] - '7') << 4;
    } else {
        drive_rev = (inquiry_data[32] - '0') << 4;
    }

    if ((inquiry_data[33] >= 'A') && (inquiry_data[33] <= 'F')) {
        drive_rev = drive_rev | (inquiry_data[33] - '7');
    } else {
        drive_rev = drive_rev | (inquiry_data[33] - '0');
    }
    printf("\t Drive:: SWrev=%03u/%04X EErev=%03u/%02X%02X HWrev=%03u\n",
          drive_rev , drive_edc,
          inquiry_data[45],
          vpdC0_data[6], vpdC0_data[7],
          inquiry_data[46]);

    printf("\t Drive-Library Type:: ");
    if (inquiry_data[51] == 2) {
        printf("for ATL/TL820 Library only");
    } else if (inquiry_data[51] == 0) {
        printf("standard");
    } else if (inquiry_data[51] == 1) {
        printf("5 Cart. integrated loader only");
    } else {
        printf("unknown");
    }
    printf(" (%0X)\n", inquiry_data[51] );

    if (inquiry_data[50]) {     /* Media Loader Present? */
        printf("\t Ldr:: SW/HW/ME=%03u/%03u/%03u \n",
              inquiry_data[47], inquiry_data[48], inquiry_data[49] );
    }
    /*
     * Display serial number(s)
     */
    if (vpd80_data[3] == 0) {
        printf("\t Controller/Drive Serial Numbers not available.\n");
    } else {
        printf("\t Controller SN:: %.10s\n", &vpd80_data[4]);
        if (vpd80_data[3] > 10) {
            printf("\t Drive SN:: %.10s\n", &vpd80_data[14]);
        }
    }
    printf("\t Product Family Code:  %02d\n",inquiry_data[36]>>InqDat_V_ProdFam);
    printf ("\n\t     (hit 'return' to continue)\n");
    my_gets (input_line,sizeof(input_line));
} /* print_revinfo */

/*++
************************************************************************
**
**  **-curr_density
**
**  FUNCTIONAL DESCRIPTION:
**      Does a Mode Sense to get and display the currently
**      selected density of the device.
**
**  FORMAL PARAMETERS: none
**
**  RETURN VALUE: TRUE if success, else FALSE
**
************************************************************************
--*/
int curr_density()
{
    unsigned char density;
#ifndef NO_AUTO_STRUCT_INIT
    unsigned char mode_sense_cmd[6] = {0x1A, 0, 0, 0, 12, 0 };
#else /* ifndef NO_AUTO_STRUCT_INIT */
  unsigned char mode_sense_cmd[6];
  mode_sense_cmd[0] = 0x1A;
  mode_sense_cmd[1] = 0;
  mode_sense_cmd[2] = 0;
  mode_sense_cmd[3] = 0;
  mode_sense_cmd[4] = 12;
  mode_sense_cmd[5] = 0;
#endif /* ifndef NO_AUTO_STRUCT_INIT */
    unsigned char mode_sense_data[12];

    if (scsi_qio(mode_sense_cmd, 6, mode_sense_data, 12,FLAGS_READ) == FALSE) {
        printf("\nMode Sense cmd failed.");
        return(FALSE);
    }
    write_prot = mode_sense_data[2] & 0x80;
    density = mode_sense_data[4];
    printf("\n\t*** Current Media Density is: %2Xh = ", density);
    switch (density) {
        case 0x0A:
            printf("TK50");
            break;
        case 0x16:
            printf("TK70");
            break;
        case 0x17:
            printf("TK85");
            break;
        case 0x18:
            printf("TK86");
            break;
        case 0x19:
            printf("TK87");
            break;
        case 0x1A:
            printf("TK88");
            break;
         case 0x1B:
            printf("TK89");
            break;
        case 0x80:
            printf("TK87 - Not Compressed");
            break;
        case 0x81:
            printf("TK87 - Compressed");
            break;
        case 0x82:
            printf("TK88 - Not Compressed");
            break;
        case 0x83:
            printf("TK88 - Compressed");
            break;
        case 0x84:
            printf("TK89 - Not Compressed");
            break;
        case 0x85:
            printf("TK89 - Compressed");
            break;
        case 0x00:
            printf("default for device");
            break;
        default:
            printf("(unknown)");
            break;
    }
    printf(" ***\n");
    return(TRUE);
}


/*++
*************************************************************************
**
**  **-set_tape_format
**
**  FUNCTIONAL DESCRIPTION:
**      set_tape_format() ask the user what tape format they want (TK85
**      TK86, TK87, TK88, or TK89).  Note that each drive cannot write more
**      recent formats, but can do older ones.  In other words, the TZ85 drive
**      can only write TK85 format, the TZ86 drive can write TK85 and TK86,
**      while the TZ87 drive can write all three formats (densities).
**
**      Next, a Mode Select command is issued to tell the device the
**      selected density, then a write from BOT is done to force the media
**      to be (re)formated to the specified density.
**
**  FORMAL PARAMETERS: none
**
**  RETURN VALUE: TRUE if success, else FALSE
**
*************************************************************************
--*/
int set_tape_format()
{
#ifndef NO_AUTO_STRUCT_INIT
    unsigned char rewind_cmd[6] =  {0x1, 0, 0, 0, 0, 0 };
    unsigned char write_cmd[6] = {0x0A, 0, 0, 0, 200, 0 };
    unsigned char writefm_cmd[6] = {0x10, 0, 0, 0, 1, 0 };
    unsigned char reserve_cmd[6] = {0x16, 0, 0, 0, 0, 0 };
    unsigned char release_cmd[6] = {0x17, 0, 0, 0, 0, 0 };
    unsigned char mode_select_cmd[6] = {0x15, 0, 0, 0, 12, 0 };
#else /* ifndef NO_AUTO_STRUCT_INIT */
  unsigned char rewind_cmd[6];
  unsigned char write_cmd[6];
  unsigned char writefm_cmd[6];
  unsigned char reserve_cmd[6];
  unsigned char release_cmd[6];
  unsigned char mode_select_cmd[6];

  rewind_cmd[0] = 0x1;
  rewind_cmd[1] = 0;
  rewind_cmd[2] = 0;
  rewind_cmd[3] = 0;
  rewind_cmd[4] = 0;
  rewind_cmd[5] = 0;

  write_cmd[0] = 0x0A;
  write_cmd[1] = 0;
  write_cmd[2] = 0;
  write_cmd[3] = 0;
  write_cmd[4] = 200;
  write_cmd[5] = 0;

  writefm_cmd[0] = 0x10;
  writefm_cmd[1] = 0;
  writefm_cmd[2] = 0;
  writefm_cmd[3] = 0;
  writefm_cmd[4] = 1;
  writefm_cmd[5] = 0;

  reserve_cmd[0] = 0x16;
  reserve_cmd[1] = 0;
  reserve_cmd[2] = 0;
  reserve_cmd[3] = 0;
  reserve_cmd[4] = 0;
  reserve_cmd[5] = 0;

  release_cmd[0] = 0x17;
  release_cmd[1] = 0;
  release_cmd[2] = 0;
  release_cmd[3] = 0;
  release_cmd[4] = 0;
  release_cmd[5] = 0;

  mode_select_cmd[0] = 0x15;
  mode_select_cmd[1] = 0;
  mode_select_cmd[2] = 0;
  mode_select_cmd[3] = 0;
  mode_select_cmd[4] = 12;
  mode_select_cmd[5] = 0;
#endif /* ifndef NO_AUTO_STRUCT_INIT */
    unsigned char new_den;
    char reply[80];

    if (device_type != TAPE_DEV) {
        printf("\n\tFunctionality not supported for non-Tape Devices.\n");
        return(TRUE);
    }
    if (curr_drive_type == TK85_DRIVE) {
        printf("\n\tTZ85 Drive supports only TK85 format.\n");
        return(TRUE);
    }
    /*
     * Check to see if the unit is Ready?
     */
    if (!scsi_qio(tur_cmd, 6, read_blklim_data, 0, FLAGS_READ)) {
        printf("\n\tNo media loaded to reformat ?\n");
        return(TRUE);
    }
    printf("\n\n\t*** Warning: the density selection will destroy any  ***");
    printf(  "\n\t***     data on the tape!  Quit below if necessary.  ***");
    printf(  "\n\t***                                                  ***");
    printf(  "\n\t***     Note: ForceDensity must be = 0 (default) to  ***");
    printf(  "\n\t***     allow this utility to change the current     ***");
    printf(  "\n\t***     media to a different user specified density. ***");

    printf("\n\n\tSelect format: \n");
    printf("\t 1 = TK85\n");
    printf("\t 2 = TK86\n");
    if ((curr_drive_type == TK87_DRIVE) || (curr_drive_type == TK88_DRIVE) ||
        (curr_drive_type == TK89_DRIVE) || (curr_drive_type == TK87XT_DRIVE)) {
        printf("\t 3 = TK87\n");
    }
    if ((curr_drive_type == TK88_DRIVE) || (curr_drive_type == TK89_DRIVE)) {
        printf("\t 4 = TK88\n");
    }
    if (curr_drive_type == TK89_DRIVE ) {
        printf("\t 5 = TK89\n");
    }
    printf("\t other = quit\n");
    printf("\n\tEnter selection: ");
    my_gets(reply,sizeof(reply));
    if (*reply == '1') {
        new_den = 0x17;
    } else if (*reply == '2') {
        new_den = 0x18;
    } else if (*reply == '3') {
        new_den = 0x19;
    } else if (*reply == '4') {
        new_den = 0x1A;
    } else if (*reply == '5') {
        new_den = 0x1B;
    } else {
        return(TRUE);
    }
    mode_select_cmd[4] = 12;    /* # bytes mode data */
    mode_data[0] = 0;       /* zero Sense data length (from Mode Sense) */
    mode_data[3] = 8;       /* Block Desc length */
    mode_data[4] = new_den;

    if (scsi_qio(reserve_cmd, 6, inquiry_data, 0, 0) == FALSE) {
        printf("\nReservation of unit failed: unit active from other host.");
        return(FALSE);
    }
    printf("\n\t    Rewinding unit to BOT for density change...");
    if (scsi_qio(rewind_cmd, 6, mode_data, 0, 0) == FALSE) {
        printf("\nRewind failed.");
        return(FALSE);
    }
    printf("\n\t    Selecting specified density...");
    if (scsi_qio(mode_select_cmd, 6, mode_data, 12, FLAGS_WRITE) == FALSE) {
        printf("\nMode Select for density set failed.");
        return(FALSE);
    }
    printf("\n\t    Forcing media to selected density (write from BOT)...");
    printf("\n\t    ...(a density change takes 2 to 5 minutes).\n");
    if (scsi_qio(write_cmd, 6, inquiry_data, 200, FLAGS_WRITE) == FALSE) {
        printf("\nWrite failed.");
        return(FALSE);
    }
    scsi_qio(writefm_cmd, 6, inquiry_data, 0, FLAGS_WRITE);
    /* Second write filemark will force device's cache to be flushed. */
    if (scsi_qio(writefm_cmd, 6, inquiry_data, 0, FLAGS_WRITE) == FALSE) {
        printf("\nWrite operations failed. Check unit/try again.");
        return(FALSE);
    }
    scsi_qio(release_cmd, 6, inquiry_data, 0, FLAGS_WRITE);

    if (curr_density() == FALSE)  { /* Get and display current density */
        return(FALSE);      /* failed */
    }
    if (mode_data[4] != new_den) {
        printf("\n Note: current density not equal to that selected!");
    }
    return(TRUE);
} /* set_tape_format */
/*++
*************************************************************************
**
**  **-get_eerom_param
**
**  FUNCTIONAL DESCRIPTION:
**      get_eerom_param() sends a 10 byte Mode Sense Command to get the
**      EEROM parameters (vendor unique) page. TZ87 and higher only.
**
**  FORMAL PARAMETERS: none
**
**  RETURN VALUE: TRUE if success, else FALSE
**
*************************************************************************
--*/
int get_eerom_param()
{
    unsigned long int length;
    int stat;
    struct MSE10 *ModeSense_Ptr;        /* Pointer to the Mode  Sense CDB */
    struct MSE10 ModeSense_CDB;         /* Allocation of the Mode Sense CDB */
/*
    int lines;
    static int lines_per_page=20;
*/

    /*
     * If not TZ87 drive or more recent, stop here.
     */
    if (curr_drive_type < TK87_DRIVE) {
        printf("\n\tEEROM parameters unavailable for this drive type.\n");
        return(FALSE);
    }
    ModeSense_Ptr = &ModeSense_CDB;
    ModeSense_Ptr->Opcode = 0x5A;
    ModeSense_Ptr->LUN = 0x08;              /* Disable Block Descriptor */
    ModeSense_Ptr->PageCode = 0x3E;
    ModeSense_Ptr->Rsvd3 = 0x00;
    ModeSense_Ptr->Rsvd4 = 0x00;
    ModeSense_Ptr->Rsvd5 = 0x00;
    ModeSense_Ptr->Rsvd6 = 0x00;
    length=(unsigned)sizeof(struct MSE10DATA);
#if LINUX_CHECK_FOR_BUFFER_MAX
    if(length>MAX_IO_SIZE)
    { length =MAX_IO_SIZE; /* Can't do 64 K IO :( */
    }
#endif /* if LINUX_CHECK_FOR_BUFFER_MAX */
#ifdef LITTLEENDIAN
    ModeSense_Ptr->AllocLen1 = (unsigned char)(length>>8);
    ModeSense_Ptr->AllocLen0 = (unsigned char)length;
#else /* ifdef LITTLEENDIAN */
    ModeSense_Ptr->AllocLen = length;
#endif /* ifdef LITTLEENDIAN */
    ModeSense_Ptr->Cntrl = 0x00;

    stat = scsi_qio((unsigned char *)&ModeSense_CDB,
                    sizeof (struct MSE10),
                    (unsigned char *)&ModeSense_Page,
                    length,
                    FLAGS_READ);

    /* Get whole page based on the Mode Length field */

#ifdef LITTLEENDIAN
    length = (unsigned long int) get_word(&ModeSense_Page.ModeLen1);
#else /* ifdef LITTLEENDIAN */
    length = (unsigned long int) get_word(&ModeSense_Page.ModeLen);
#endif /* ifdef LITTLEENDIAN */

    length = length - (((int)&ModeSense_Page.Params[0]) -
         ((int)&ModeSense_Page))  + sizeof(unsigned short);

    ModeSense_Page.Params[length]  = '\0';
    printf("\n");
    puts ((char *)&ModeSense_Page.Params[0]);
    return(TRUE);
}

/*++
*************************************************************************
**
**  **-set_eerom_param
**
**  FUNCTIONAL DESCRIPTION:
**      set_eerom_param() allows the user to set the EEROM based
**      ForceDensity parameter to one of the following values,
**      on TZ86 or TZ87 type drives.
**
**      Definition of ForceDensity values:
**          0 = Automatic (host selected)
**          1 = Force to TK85 format on all writes from BOT (ex: INIT)
**          2 = Force to TK86 format on all writes from BOT
**          3 = Force to TK87 format on all writes from BOT (TZ87 only)
**          4 = Force to TK88 format on all writes from BOT (TZ88/TZ89 only)
**          5 = Force to TK89 format on all writes from BOT (TZ89 only)
**
**  FORMAL PARAMETERS: none
**
**  RETURN VALUE: TRUE if success, else FALSE
**
*************************************************************************
--*/
int set_eerom_param()
{
    struct MSE *ModeSense_Ptr;           /* Pointer to the Mode Sense CDB */
    struct MSE ModeSense_CDB;            /* Allocation of the Mode Sense CDB */
#ifndef NO_AUTO_STRUCT_INIT
    unsigned char forcedensity[] =
        {0x00, 0x00, 0x10, 0x00, 0x3E, 0x0F,
        'F','o','r','c','e','D','e','n','s','i','t','y',' ','1', 0x00};
#else /* ifndef NO_AUTO_STRUCT_INIT */
  unsigned char forcedensity[21];
  forcedensity[0] = 0x00;
  forcedensity[1] = 0x00;
  forcedensity[2] = 0x10;
  forcedensity[3] = 0x00;
  forcedensity[4] = 0x3E;
  forcedensity[5] = 0x0F;
  forcedensity[6] = 'F';
  forcedensity[7] = 'o';
  forcedensity[8] = 'r';
  forcedensity[9] = 'c';
  forcedensity[10] = 'e';
  forcedensity[11] = 'D';
  forcedensity[12] = 'e';
  forcedensity[13] = 'n';
  forcedensity[14] = 's';
  forcedensity[15] = 'i';
  forcedensity[16] = 't';
  forcedensity[17] = 'y';
  forcedensity[18] = ' ';
  forcedensity[19] = '1';
  forcedensity[20] = 0x00;
#endif /* ifndef NO_AUTO_STRUCT_INIT */
    char reply[80];
    int status;

    if (device_type != TAPE_DEV) {
        printf("\n\tFunctionality not supported for non-Tape Devices.\n");
        return(TRUE);
    }
    if (curr_drive_type == TK85_DRIVE) {
        printf("\n\tTZ85 Drive supports only one format.\n");
        return(TRUE);
    }
    printf("\n\n\t*** WARNING! if ForceDensity is left set to a lower  ***");
    printf(  "\n\t***     density, all subsequent media will be forced ***");
    printf(  "\n\t***     to that density, if written from BOT.        ***");
    printf(  "\n\t***                                                  ***");
    printf(  "\n\t***     Remember to change ForceDensity back to 0 to ***");
    printf(  "\n\t***     avoid unintentional cartridge capacity loss! ***");

    printf("\n\n\tSelect format: \n");
    printf("\t 0 = Automatic (selected by host software driver),\n");
    printf("\t 1 = TK85 (Force to TK85 format on all writes from BOT, ex: INIT),\n");
    printf("\t 2 = TK86 (Force to TK86 format on all writes from BOT),\n");
    if ((curr_drive_type == TK87_DRIVE) || (curr_drive_type == TK88_DRIVE) ||
        (curr_drive_type == TK89_DRIVE) ||  (curr_drive_type == TK87XT_DRIVE)) {
        printf("\t 3 = TK87 (Force to TK87 format on all writes from BOT),\n");
    }
    if ((curr_drive_type == TK88_DRIVE) || (curr_drive_type == TK89_DRIVE)) {
        printf("\t 4 = TK88 (Force to TK88 format on all writes from BOT),\n");
    }
    if (curr_drive_type == TK89_DRIVE) {
        printf("\t 5 = TK89 (Force to TK89 format on all writes from BOT),\n");
    }
    printf("\t other = quit\n ");
    printf("\n\tEnter selection: ");
    my_gets(reply,sizeof(reply));

    if ((*reply < '0') || (*reply > '5')) {     /* User wants to quit function? */
        return(TRUE);
    }
    forcedensity[19] = *reply;                  /* Set new value */
    ModeSense_Ptr = &ModeSense_CDB;
    ModeSense_Ptr->Opcode = 0x15;
    ModeSense_Ptr->LUN = 0x10;
    ModeSense_Ptr->PageCode = 0x00;
    ModeSense_Ptr->Rsvd3 = 0x00;
    ModeSense_Ptr->AllocLen = 21;
    ModeSense_Ptr->Cntrl = 0x00;

    status = scsi_qio(ModeSense_Ptr, sizeof (struct MSE),
                        forcedensity, 21, FLAGS_WRITE);
    if (status == FALSE) {
        printf("\n\t*** Mode Select for ForceDensity param failed. ***\n");
    } else {
        printf("\n\n\t*** ForceDensity EEPROM Parameter updated. ***\n");
    }
    return(status);

} /* set_eerom_param */
/*++
*************************************************************************
**
**  **-set_any_eerom_param
**
**  FUNCTIONAL DESCRIPTION:
**      set_eerom_param() allows the user to set an EEROM based
**      parameter to any value
**
**  FORMAL PARAMETERS: none
**
**  RETURN VALUE: TRUE if success, else FALSE
**
*************************************************************************
--*/
int set_any_eerom_param()
{
    struct MSE *ModeSense_Ptr;           /* Pointer to the Mode Sense CDB */
    struct MSE ModeSense_CDB;            /* Allocation of the Mode Sense CDB */
    static unsigned char param[255] =
        {0x00, 0x00, 0x10, 0x00, 0x3E, 0x0F };
    char reply[80];
    int status;

    if (device_type != TAPE_DEV) {
        printf("\n\tFunctionality not supported for non-Tape Devices.\n");
        return(TRUE);
    }

    printf("\n\n\tEnter parameter name:\n");
    my_gets(reply,sizeof(reply));
    if(!strlen(reply))
    { return TRUE;
    }
    strncpy((char *)(param+6),reply,sizeof(param)-6);
    if(strchr((char *)(param+6),' ')==NULL) /* Skip value entry if "param value" was entered */
    { strncat((char *)(param+6)," ",sizeof(param)-6-strlen((char *)(param+6))-1);
      printf("\n\n\tEnter value:\n");
      my_gets(reply,sizeof(reply));
      strncat((char *)(param+6),reply,sizeof(param)-6-strlen((char *)(param+6))-1);
    }
    param[5]=strlen((char *)(param+6))+1;

    ModeSense_Ptr = &ModeSense_CDB;
    ModeSense_Ptr->Opcode = 0x15;
    ModeSense_Ptr->LUN = 0x10;
    ModeSense_Ptr->PageCode = 0x00;
    ModeSense_Ptr->Rsvd3 = 0x00;
    ModeSense_Ptr->AllocLen = param[5]+6;
    ModeSense_Ptr->Cntrl = 0x00;

    status = scsi_qio(ModeSense_Ptr, sizeof (struct MSE),
                        param, ModeSense_Ptr->AllocLen, FLAGS_WRITE);
    if (status == FALSE) {
        printf("\n\t*** Mode Select for page 3eh failed. ***\n");
    } else {
        printf("\n\n\t*** EEPROM Parameter updated. ***\n");
    }
    return(status);

} /* set_any_eerom_param */


/*++
*************************************************************************
**
**  **-load
**
**  FUNCTIONAL DESCRIPTION:
**      load and unload cartridge
**
**  FORMAL PARAMETERS: int: 1=load, 0:unload
**
**  RETURN VALUE: TRUE if success, else FALSE
**
*************************************************************************
--*/
#ifdef FUNCTION_PROTOTYPES
int load(int flag)
#else
int load(flag)
int flag;
#endif
{ static struct LOAD Load={0x1b,0,0,0,0,0};
  Load.EOTRetenLoad=flag;
  return scsi_qio(&Load,sizeof(Load),NULL,0,0);
} /* load */


/*++
*************************************************************************
**
**  **-RecoverDir
**
**  FUNCTIONAL DESCRIPTION:
**      Recover trashed cartridge directory
**      Do a LOCATE immediate to block 0x7fffffff, then wait actively
**      for command completion (you never know what SCSI timeouts hosts will accept...)
**
**  FORMAL PARAMETERS: none
**
**  RETURN VALUE: TRUE if success, else FALSE
**
*************************************************************************
--*/
int RecoverDir()
{ static struct LOCATE LocateImm={0x2b,1,0,0x7f,0xff,0xff,0xff,0,0,0};
  static struct READ_POS ReadPos={0x34,0,0,0,0,0,0,0,0,0};
  static char Rewind[]={0x01,0,0,0,0,0};
  unsigned char ReadPosData[20];
  unsigned long pos,old_pos;
  int update_wait;
  int no_of_idle_seconds=120;

  if (!scsi_qio(tur_cmd, 6, NULL, 0, FLAGS_READ)) {
      printf("\n\tNo media loaded.\n");
      return(TRUE);
  }
  if(!scsi_qio(&LocateImm,sizeof(LocateImm),NULL,0,0))
  { printf("LOCATE command failed.\n");
    return FALSE;
  }

  if(verbose)
  { printf("Waiting for command to complete...\n");
  }
  print_busy=FALSE;
  old_pos=(unsigned long)-1;
  for(update_wait=0;update_wait<60;++update_wait)
  { scsi_quiet=1;               /* Suppress error message - this is expected to fail */
    if(scsi_qio(&ReadPos,sizeof(ReadPos),ReadPosData,sizeof(ReadPosData),FLAGS_READ))
    { pos=get_long(ReadPosData+4);
      if(verbose)
      { printf("At block %lx\n",pos);
      }
      if(pos==old_pos&&!--no_of_idle_seconds) /* Stop if tape hasn't moved in ages */
      { break;
      }
      old_pos=pos;
    }
    else
    { if(scsi_status==2 && scsi_sense==8 && scsi_ASC==0 && scsi_ASCQ==5)
         /* Blank Check/EOD? */
      { if(verbose)
        { printf("AT EOD.\n");
        }
        break;
      }
      else
      { printf("READ POSITION FAILED:%02x/%02x/%02x/%02x\n",
               scsi_status,scsi_sense,scsi_ASC,scsi_ASCQ);
        break;
      }
    }
    sleep(1);
  }
  print_busy=TRUE;
  if(no_of_idle_seconds==0)
  { printf("Timeout!\n");
  }
  if(verbose)
  { printf("Rewinding.\n");
  }
  scsi_qio(Rewind,sizeof(Rewind),NULL,0,0);
  if(verbose)
  { printf("Unloading...\n");
  }
  load(0);
  if(verbose)
  { printf("Done...\n");
  }
  return no_of_idle_seconds!=0;
} /* load */


void get_logdataold()
{
    int i;
    int stat;
    struct LSEDATAOLD *LogPage_Ptr;     /* Pointer to the Log Page Data */
    struct LSE *LogSense_Ptr;           /* Pointer to the Log Sense CDB */
    struct LSE LogSense_CDB;            /* Allocationof the Log Sense CDB */
    unsigned short int length;          /* Return length of Log Sense Data */
    unsigned char param;                /* Log Parameter Code */

    /*
     * If not TZ87 drive, stop here.
     */
    if ((curr_drive_type == TK85_DRIVE) || (curr_drive_type == TK86_DRIVE)) {
        printf("\n\tErrors Logs unavailable for this drive type.\n");
        return;
    }
    LogPage_Ptr = &LogSense.LogSense_PageOld;
    LogSense_Ptr = &LogSense_CDB;

    LogSense_Ptr->Opcode = 0x4D;
    LogSense_Ptr->Rsvd1 = 0x00;
    LogSense_Ptr->PageCode = 0x07;
    LogSense_Ptr->Rsvd3 = 0x00;
    LogSense_Ptr->Rsvd4 = 0x00;
    LogSense_Ptr->Rsvd5 = 0x00;
#ifdef LITTLEENDIAN
    LogSense_Ptr->AllocLen1 = (unsigned char)(sizeof (struct LSEDATA) >>8);
    LogSense_Ptr->AllocLen0 = (unsigned char)(sizeof (struct LSEDATA));
#else /* ifdef LITTLELENDIAN */
    LogSense_Ptr->AllocLen = sizeof (struct LSEDATA);
#endif /* ifdef LITTLELENDIAN */
    LogSense_Ptr->Cntrl = 0x00;

    param = 0;
    for (i = 0; i < 14; i++) {
        LogSense_Ptr->ParPtr = param;
        stat = scsi_qio((unsigned char *)&LogSense_CDB,
                        sizeof (struct LSE),
                        (unsigned char *)LogPage_Ptr,
                        sizeof (struct LSEDATA),
                        FLAGS_READ);
        if (stat == FALSE) {
            printf("\n    Last N Errors Log Parameters Unavailable.");
            break;
        }
        length = LogPage_Ptr->PageLengthL;
        length |= (LogPage_Ptr->PageLengthH << 8);
        if (length == 0) {     /* no more Log Parameters */
            break;
        }
        param = LogPage_Ptr->ParamCodeL + 1;
        /*
         * Puts() places another <CR> on the end, so remove it.
         */
        length = length - 5;
        LogPage_Ptr->ParamValue[length] = '\0';
        puts ((char *)(LogPage_Ptr->ParamValue));
    }
}

/*++
*************************************************************************
**
**  **-get_logdata
**
**  FUNCTIONAL DESCRIPTION:
**      get_logdata sends the appropriate scsi commands to get the needed
**      Last N Error Log Sense information.
**
**  FORMAL PARAMETERS: none
**
**  RETURN VALUE: void
**
*************************************************************************
--*/
void get_logdata()
{
    int stat;
    unsigned char   *LogPage_Ptr;       /* Pointer to the Log Page Data */
    struct LSE *LogSense_Ptr;           /* Pointer to the Log Sense CDB */
    struct LSE LogSense_CDB;            /* Allocationof the Log Sense CDB */
    unsigned short int length;          /* Return length of Log Sense Data */
    unsigned char param;                /* Log Parameter Code */
    unsigned char LogParam[2*MAX_LOG_DATA+1];
    int LogParamLength;
    int returnlength;
    unsigned char *temp;
    int i;
    int pagesize;
    /*
     * If not TZ87 drive, stop here.
     */
    if ((curr_drive_type == TK85_DRIVE) || (curr_drive_type == TK86_DRIVE)) {
        printf("\n\tErrors Logs unavailable for this drive type.\n");
        return;
    }

    LogPage_Ptr = (unsigned char *)&LogSense.LogSense_Page;
    LogSense_Ptr = &LogSense_CDB;

    ((struct LSEDATAPGHDR *)LogPage_Ptr)->PageLengthL=0;
    ((struct LSEDATAPGHDR *)LogPage_Ptr)->PageLengthH=0;

    LogSense_Ptr->Opcode = 0x4D;
    LogSense_Ptr->Rsvd1 = 0x00;
    LogSense_Ptr->PageCode = 0x07;
    LogSense_Ptr->Rsvd3 = 0x00;
    LogSense_Ptr->Rsvd4 = 0x00;
    LogSense_Ptr->Rsvd5 = 0x00;
    LogSense_Ptr->ParPtr = 0x00;
#ifdef LITTLEENDIAN
    LogSense_Ptr->AllocLen1 = (unsigned char)(sizeof (struct LSEDATA) >>8);
    LogSense_Ptr->AllocLen0 = (unsigned char)(sizeof (struct LSEDATA));
#else /* ifdef LITTLEENDIAN */
    LogSense_Ptr->AllocLen = sizeof (struct LSEDATA);
#endif /* ifdef LITTLEENDIAN */
    LogSense_Ptr->Cntrl = 0x00;

        /*
         * If before V20 and tz87 use old method to get eerom params
         */

    if ((cnt_rev < 20) &&
        ((curr_drive_type == TK87_DRIVE))){
        get_logdataold();
        return;
    } else {
        stat = scsi_qio((unsigned char *)&LogSense_CDB,
                        sizeof (struct LSE),
                        (unsigned char *)LogPage_Ptr,
                        sizeof (struct LSEDATA),
                        FLAGS_READ);

        pagesize=((struct LSEDATAPGHDR *)LogPage_Ptr)->PageLengthL;
        pagesize|=(unsigned short)(((struct LSEDATAPGHDR *)LogPage_Ptr)->PageLengthH)<<8;

        LogPage_Ptr+=(sizeof(struct LSEDATAPGHDR));    /* GO past the header of Log Page*/

        while (pagesize != 0) {
            if (pagesize<0) {
                printf("there was an error in the byte count \n\n");
                break;
            }
            LogParamLength = ((struct LSEDATAPRMHDR *)LogPage_Ptr)->ParamLength;

            LogPage_Ptr+=sizeof(struct LSEDATAPRMHDR);  /* go past log param hdr */
            pagesize-=sizeof(struct LSEDATAPRMHDR);

            strncpy((char *)LogParam,(char *)LogPage_Ptr,LogParamLength);

            LogPage_Ptr +=LogParamLength;
            pagesize -=LogParamLength;

            temp = &LogParam[LogParamLength];
            LogParamLength = ((struct LSEDATAPRMHDR *)LogPage_Ptr)->ParamLength;

            LogPage_Ptr+=sizeof(struct LSEDATAPRMHDR);  /* go past log param hdr */
            pagesize-=sizeof(struct LSEDATAPRMHDR);  /* go past log param hdr */

            strncpy((char *)temp,(char *)LogPage_Ptr,LogParamLength);
            temp[LogParamLength]='\0';
            puts((char *)LogParam);

            LogPage_Ptr +=LogParamLength;
            pagesize-=LogParamLength;
        }
    }
}

/*++
***************************************************************************
**
**  **-get_sense_data
**
**  FUNCTIONAL DESCRIPTION:
**      Get sense data and if the Sense Key is non-zero, do some decoding
**      of the information and print results.
**
**  FORMAL PARAMETERS:
**
**  RETURN VALUE: void
**
***************************************************************************
--*/
int get_sense_data()
{
    struct RQS *RequestSense_Ptr;
    struct RQS RequestSense_CDB;
#ifndef SCSI_AUTOSENSE
    unsigned char sense_data[MAX_RQS_DATA];
#endif /* ifndef SCSI_AUTOSENSE */

#ifdef SCSI_AUTOSENSE
  if (bypass_req_sens_cmd != TRUE)
    {
#endif /* ifdef SCSI_AUTOSENSE */
    int success;

    RequestSense_Ptr = &RequestSense_CDB;
    RequestSense_Ptr->Opcode = 0x03;
    RequestSense_Ptr->Rsvd1 = 0;
    RequestSense_Ptr->Rsvd2 = 0;
    RequestSense_Ptr->Rsvd3 = 0;
    RequestSense_Ptr->AllocLen = MAX_RQS_DATA;
    RequestSense_Ptr->Cntrl = 0;
    /*
    **  Set up the descriptor with the SCSI information to be sent
    **  to the target, issue the QIO and check the status.
    */
    success = scsi_qio((char *)&RequestSense_CDB,
                sizeof(struct RQS), sense_data, MAX_RQS_DATA, FLAGS_READ);
    if (success == FALSE) {
        return (FALSE);
    }
#ifdef SCSI_AUTOSENSE
    }
  else
    {
    bypass_req_sens_cmd = FALSE;
    }
#endif /* ifdef SCSI_AUTOSENSE */
    scsi_sense = sense_data[2] & 0x0f;
    scsi_ASC = sense_data[12];
    scsi_ASCQ = sense_data[13];

    if(scsi_quiet)
    { return scsi_sense==0;
    }
    if (scsi_sense != 0) {
        if (scsi_sense == 6)  {
            printf("\n\tUnit Attention on device:");
            printf(" ASC = %02xh", scsi_ASC);
            if (scsi_ASC == 0x28) {
                printf(" = Not Ready to Ready Transition");
            } else if (scsi_ASC == 0x29) {
                printf(" = PowerOn/Reset Occured");
            } else if ((scsi_ASC == 0x3F) && (scsi_ASCQ == 1)) {
                printf(" = Device FW has been changed");
            } else {
                printf ("\n");
            }
        } else if (scsi_sense == 2) {
            printf("\n\n\tMedia not ready!");
            if (scsi_ASCQ == 1) {
                printf(": Load/calibration in progress\n");
            } else if (scsi_ASCQ == 2) {
                printf(": Load command needed.\n");
            } else if (scsi_ASCQ == 3) {
                printf(": Manual intervention needed.\n");
            } else {
                printf("\n");
            }
        } else if (scsi_sense == 5) {
            printf ("\n\tSense Key: %02X,", scsi_sense);
            printf ("   ASC: %02Xh,", scsi_ASC);
            printf ("   ASCQ: %02Xh\n", scsi_ASCQ);
            if (scsi_ASC == 0x24) {
                switch (scsi_ASCQ) {
                case 0x82:
                    printf ("\n\t*** Please Remove Media for Code Update\n");
                    break;
                case 0x87:
                    printf("\n\t*** Image size too small: check image file.\n");
                    break;
                case 0x89:
                    printf("\n\t*** Image size too big: check image file.\n");
                    break;
                case 0x8B:
                    printf("\n\t*** Invalid Image: check image file.\n");
                    break;
                case 0x8D:
                    printf("\n\t*** Invalid servo code EDC: ");
                    printf("Make sure new image file is intact.\n");
                    break;
                case 0x8E:
                    printf("\n\t*** Personality mismatch:\n");
                    printf("\n\t      Verify that personality of image and ");
                    printf("current installed firmware are compatible!\n");
                    break;
                case 0x8F:
                    printf("\n\t*** Invalid controller code EDC: ");
                    printf("Make sure new image file is intact.\n");
                    break;
                default:
                    printf("\tIllegal Request Info: %02Xh, field ptr=%02X%02Xh",
                        sense_data[15], sense_data[16], sense_data[17]);
                    break;
                }
                if (sense_data[19]) {
                    printf("\tVU byte 19= %02Xh\n", sense_data[19]);
                } else {
                    printf("\n");
                }
            }
        } else {
            printf ("\n\tSense Key: %02X,", scsi_sense);
            printf ("   ASC: %02Xh,", scsi_ASC);
            printf ("   ASCQ: %02Xh\n", scsi_ASCQ);
            if (sense_data[19]) {
                printf("\tVU byte 19= %02Xh\n", sense_data[19]);
            } else {
                printf("\n");
            }
        }
        return (FALSE);
    } else {
        return(TRUE);
    }
} /* get_sense_data */

/*++
*************************************************************************
**
**  **-scsi_qio
**
**  FUNCTIONAL DESCRIPTION:
**      scsi_qio sets up the specified info in the gk_desc[] array,
**      issues the QIO, and handles various completions.
**
**  RETURN VALUE: TRUE if successful, else FALSE
**
*************************************************************************
--*/
int scsi_qio(cmd_ptr, cmd_size, data_ptr, data_size, i_o)
unsigned char *cmd_ptr;     /* ptr to command bytes */
int cmd_size;               /* # of SCSI command bytes */
unsigned char *data_ptr;    /* ptr to data area, if any */
int data_size;              /* # bytes of SCSI data, or zero */
int i_o;                    /* read or write flag */




#ifdef VMS
{
    int gk_iosb[2];
    int gk_desc[15];
    int local_status;
    int i;

    scsi_status = 0;            /* clear SCSI STATUS global */
    gk_desc[OPCODE] = 1;
    gk_desc[FLAGS] = i_o + FLAGS_DISCONNECT;
    gk_desc[COMMAND_ADDRESS] = (int) cmd_ptr;
    gk_desc[COMMAND_LENGTH] = cmd_size;
    gk_desc[DATA_ADDRESS] = (int)data_ptr;
    gk_desc[DATA_LENGTH] = data_size;
    gk_desc[PAD_LENGTH] = 0;
    gk_desc[PHASE_TIMEOUT] = 10;
    gk_desc[DISCONNECT_TIMEOUT] = 180;
    for (i = 9; i < 15; i++) {
        gk_desc[i] = 0;     /* Clear reserved fields */
    }
    /*
     *  Issue the QIO & check the returned statuses.
     */
    local_status = sys$qiow (GK_EFN, tape_chan, IO$_DIAGNOSE, gk_iosb, 0, 0,
               &gk_desc[0], 15*4, 0, 0, 0, 0);
    if (!(local_status & 1)) {
        printf("\nlocal_status : %u",local_status);
        sys$exit (local_status);
    }
    local_status = gk_iosb[0] & 0xffff;
    if (!(local_status&1)) {
        if (local_status == 0x22c) {        /* Device timeout? */
            return(FALSE);
        }
        if (local_status == 0x54) {        /* Fatal Controller Error? */
            return(FALSE);
        }
        printf("\ngk_iosb[0] = %x", local_status);
        sys$exit (local_status);
    }
    scsi_status = (gk_iosb[1] >> 24) & SCSI_STATUS_MASK;
#endif /* ifdef VMS */




#ifdef HPUX
#error The HP-UX changes haven't been verified since I did several changes.
#error Please check if everything works. - RPR
{
  struct sctl_io scsi_io;
  memset(scsi_io, 0, sizeof(scsi_io));
    if (i_o == FLAGS_READ)
    {
    scsi_io.flags = SCTL_READ;
    }
  scsi_io.cdb_length = (unsigned char) cmd_size;
  memcpy(scsi_io.cdb, cmd_ptr,scsi_io.cdb_length);
  scsi_io.data = (void *)data_ptr;
  scsi_io.data_length = data_size;
  scsi_io.max_msecs = 10000;
  if (ioctl(tape_chan, SIOC_IO, &scsi_io) < 0)
    {
    printf("Tape access failed - %d\n",errno);
    exit(1);
    }
  scsi_status = scsi_io.cdb_status;
  scsi_sense = scsi_io.sense[2] & 0x0f;
  scsi_ASC = scsi_io.sense[12];
  scsi_ASCQ = scsi_io.sense[13];
  memcpy(sense_data,scsi_io.sense,MAX_RQS_DATA);
#endif /* #ifdef HPUX */




#ifdef SUNOS
#error This is entirely untested.
#error If you tested it and it worked, please remove this. - RPR
{
  int local_status;
  local_status=call_uscsi(tape_chan,
                          (char *)cmd_ptr,
                          (unsigned char) cmd_size,
                          (char *) data_ptr,
                          (unsigned int) data_size,
                          i_o == FLAGS_READ ? USCSI_READ : USCSI_WRITE);
  if (local_status<0)
  { printf("Tape access failed - %d\n",-local_status);
    exit(1);
  }
  scsi_status = local_status;
#endif /* #ifdef SUNOS */




#ifdef LINUX
{ static int our_timeout=180 /* secs */ * 100; /* in units of 10 msec */
  int old_timeout; /* Save device timeout value. You never know. */
  int local_status;
  char *buffer,*in_buffer; /* buffers for data to driver (write) and from driver (read) */
  unsigned int buffer_size,in_buffer_size;
#if DEBUG>4
  printf("scsi_qio:cmd_ptr=%p,cmd_size=%u,data_ptr=%p,data_size=%u,i_o=%d.\n",
         cmd_ptr,cmd_size,data_ptr,data_size,i_o);
#endif

  /* Save original timeout, set our own timeout value */
  ioctl(tape_chan,SG_GET_TIMEOUT,&old_timeout);
  ioctl(tape_chan,SG_SET_TIMEOUT,&our_timeout);

  /* Find out how much space we need.
     With DATA OUT, we need header+CDB+data, otherwise it's only header+CDB. */
  buffer_size=sizeof(struct sg_header)+cmd_size+(i_o==FLAGS_WRITE?data_size:0);
  /* With DATA IN, we get header+data, otherwise it's only header. */
  in_buffer_size=sizeof(struct sg_header)+(i_o==FLAGS_READ?data_size:0);

#if LINUX_CHECK_FOR_BUFFER_MAX /* Can be ignored now ? */
  /* Can a larger buffer be sent to the sg driver in chunks?
     I did't try but it doesn't look like that in the source. */
  if(buffer_size>SG_BIG_BUFF)
  { printf("Write buffer size %u exceeds maximum of %u bytes.\n",buffer_size,SG_BIG_BUFF);
    exit(1);
  }
  if(in_buffer_size>SG_BIG_BUFF)
  { printf("Read buffer size %u exceeds maximum of %u bytes.\n",in_buffer_size,SG_BIG_BUFF);
    exit(1);
  }
#endif
  /* Allocate buffers */
  if((buffer=(char *)malloc(buffer_size))==NULL ||
     (in_buffer=(char *)malloc(in_buffer_size))==NULL)
  { printf("Couldn't allocate %u byte buffers.\n",buffer_size+in_buffer_size);
    exit(1);
  }
  memset((struct sg_header *)buffer,0,sizeof(struct sg_header)); /* be safe */
  ((struct sg_header *)buffer)->reply_len=in_buffer_size;
  ((struct sg_header *)buffer)->twelve_byte=cmd_size==12;
  ((struct sg_header *)buffer)->result=0;
  /* Put CDB into request buffer */
  memcpy(buffer+sizeof(struct sg_header),cmd_ptr,cmd_size);
  if(i_o==FLAGS_WRITE&&data_size!=0)
  { memcpy(buffer+sizeof(struct sg_header)+cmd_size,data_ptr,data_size);
  }
#if DEBUG
  putchar('\n');
  dump("CDB",cmd_ptr,cmd_size);
#if DEBUG>=2
  if(i_o==FLAGS_WRITE)
  { dump("DATA OUT",buffer+sizeof(struct sg_header)+cmd_size,data_size);
  }
#endif
#if DEBUG>=3
  dump("Header out",buffer,sizeof(struct sg_header));
#endif
#endif
  if(write(tape_chan,buffer,buffer_size)!=buffer_size ||
     read(tape_chan,in_buffer,in_buffer_size)!=in_buffer_size)
  { if(errno==EBUSY)
    { scsi_status=8;
    }
    else
    { printf("Tape access failed - errno=%d\n",errno);
      free(in_buffer);
      free(buffer);
      exit(1);
    }
  }
  else
  {
#if DEBUG
    printf("errno=%d,result=%d\n",errno,((struct sg_header *)buffer)->result);
#if DEBUG>=4
    dump("Header out",buffer,sizeof(struct sg_header));
#endif
#if DEBUG>=3
    dump("Header in",in_buffer,sizeof(struct sg_header));
#endif
#if DEBUG>=2
    if(i_o==FLAGS_READ)
    { dump("Data in",in_buffer+sizeof(struct sg_header),data_size);
    }
#endif
    dump("SENSE",((struct sg_header *)in_buffer)->sense_buffer,
                 sizeof(((struct sg_header *)in_buffer)->sense_buffer));
#endif
    if(i_o==FLAGS_READ)
    { memcpy(data_ptr,in_buffer+sizeof(struct sg_header),data_size);
    }
    memcpy(sense_data,((struct sg_header *)in_buffer)->sense_buffer,sizeof(((struct sg_header *)in_buffer)->sense_buffer));
    scsi_status=0; /* Reported from Linux 2.2 on? */
    if(scsi_status==0 && sense_data[2] & 0x0f)
    { scsi_status=2; /* Set to CHECK CONDITION if we have sense data... 
                        Pre-2.2 Linux doesn't report status */
    }
    scsi_sense = sense_data[2] & 0x0f;
    scsi_ASC = sense_data[12];
    scsi_ASCQ = sense_data[13];
#if DEBUG
    printf("Status=%x,sense=%x,ASC=%x,ASCQ=%x\n",
           scsi_status,scsi_sense,scsi_ASC,scsi_ASCQ);
#endif    
  }
  free(in_buffer);
  free(buffer);
  ioctl(tape_chan,SG_SET_TIMEOUT,&old_timeout);
#endif /* #ifdef LINUX */
    if (scsi_status) {
        if(scsi_quiet)
        { if(scsi_status==2)
          {
#ifdef SCSI_AUTOSENSE
            bypass_req_sens_cmd = TRUE;
#endif /* ifdef SCSI_AUTOSENSE */
            get_sense_data();
          }
          return FALSE;
        }
        if ((*cmd_ptr == 0) && (scsi_status == 2)) { /* Test Unit Ready cmd? */
#ifdef SCSI_AUTOSENSE
            bypass_req_sens_cmd = TRUE;
#endif /* ifdef SCSI_AUTOSENSE */
            get_sense_data();                        /* yes */

        } else if (scsi_status == 0x08) {    /* SCSI Busy ? */
            if (print_busy) {
                printf ("\n\tSCSI Status: %02x", scsi_status);
                printf(" = Busy");
            }
        } else {
            printf ("\n\tSCSI Status: %02x", scsi_status);
            if ((*cmd_ptr == 3) && (scsi_status != 0x08)) {
                /*
                 * Request Sense cmd failed and not busy.
                 */
                printf("\n\tRequest Sense cmd failed!");

            } else if (scsi_status == 0x02) {    /* SCSI Check Condition? */
                printf(" = Check Condition");
#ifdef SCSI_AUTOSENSE
                bypass_req_sens_cmd = TRUE;
#endif /* ifdef SCSI_AUTOSENSE */
                get_sense_data();
            }
        }
        return(FALSE);
    } else {
        return(TRUE);
    }
}

#ifdef SUNOS
/* call_uscsi provided by Bruce K. Hillyer of AT&T Bell Labs, slightly modified by RPR. */
static int call_uscsi(int fd, caddr_t cdb, unsigned char cdblen,
                    caddr_t buf, unsigned int buflen,
                    int rdwr)
{ struct uscsi_cmd scmd;

  memset((void *) &scmd, 0, sizeof(scmd));
  scmd.uscsi_cdb = cdb;
  scmd.uscsi_cdblen = cdblen;
  scmd.uscsi_bufaddr = buf;
  scmd.uscsi_buflen = buflen;
  scmd.uscsi_flags = /* USCSI_DIAGNOSE | */ rdwr;
  scmd.uscsi_timeout = 180; /* 180 secs */
  if (ioctl(fd, USCSICMD, &scmd) == -1
      && errno!=EIO /* pass through EIO ... what happens on e.g. sel timeout? */
     ) {
    return -errno;
  }
  return scmd.uscsi_status;
}
#endif /* ifdef SUNOS */

/*++
***************************************************************************
**
**  **-do_code_update
**
**  FUNCTIONAL DESCRIPTION:
**
**
**  FORMAL PARAMETERS: none
**
**  RETURN VALUE: Void
**
***************************************************************************
--*/
void do_code_update()
{
    char *image_ptr;
    int image_size;
    char input_line[80];
    int i;

    switch (curr_drive_type) {
        case TK87_DRIVE:
            printf ("\n\n  *** Starting TZ87 Firmware Update.");
            image_size = TZC_IMAGE_SIZE;
            break;

        case TK87XT_DRIVE:
            printf ("\n\n  *** Starting DLT2000XT Firmware Update.");
            image_size = TZF_IMAGE_SIZE + TZF_EDC_OFFSET;
            break;

        case TK88_DRIVE:
            printf ("\n  *** Starting TZ88 Firmware Update.");
            image_size = TZE_IMAGE_SIZE + TZE_EDC_OFFSET;
            break;

        case TK89_DRIVE:
#if FALSE
            do {
                printf("\n\n\n  \t\tWhich Code model 1 - large\n");
                printf("    \t\t                 2 - small  ");
                my_gets(input_line,sizeof(input_line));
                i = atoi(input_line);
               } while ((i != 1) && (i !=2));
            if (i == 1) {
                mem_model = MEM_MODEL_LARGE;
                image_size = TZG_IMAGE_SIZE + TZG_EDC_OFFSET;
                chunk_size = CHUNK_SIZE_BIG;
            } else {
#endif
                image_size = TZGS_IMAGE_SIZE + TZGS_EDC_OFFSET;
#if FALSE
            }
#endif
            printf ("\n  *** Starting TZ89 Firmware Update.");
            break;

        default:
            printf ("\n  *** Starting TZ85/TZ86 Firmware Update.");
            image_size = TZA_IMAGE_SIZE;
            break;
    }
    /*
     * Get the image.
     */
    if ((image_ptr = get_image (image_size)) != NULL) {
        printf("\n  *** Image capture completed. ");
    } else {
        printf("\n\n  *** Image capture failed.\n");
        return;
    }
    /*
     * Validate the image.
     */
    if (validate_image (image_size, image_ptr) == TRUE) {
        printf("Validation complete.");
    } else {
        printf("\n     Image validation failed.\n");
        free(image_ptr);
        return;
    }
    /*
     * Display the image information.
     */
    show_current_version ();
    show_new_version (image_size, image_ptr);
    /*show_timestamp (image_size, image_ptr); */
    /*
     * The TZ87 (v10 and earlier) image requires a word-wide byte swap of
     * image data before sending. (After V10-- for the *installed* code--
     * the TZ87 code knows how to do the byte swap itself.
     */
    printf ("\n");
    if ((curr_drive_type == TK87_DRIVE) && (cnt_rev < 15)) {
        printf ("  *** TZ87 pre-V15 code running; byte-swapping code image...\n");
        do_byte_swap (image_size, image_ptr);
    }
    /*
     * Verify all Unit Attentions are cleared before transmitting.
     */
    for (i = 0; i < 5; i++) {
        if (get_sense_data() == TRUE) {
            break;
        }
    }
    /*
     * Transmit the image.
     */
    printf ("  *** Downloading Image... ");
    if ((transmit_image(image_size, image_ptr)) == TRUE) {
        printf ("Completed!\n");
        printf ("  *** Device is reprogramming Flash EEPROMs");
        wait_for_cup_completion ();     /* & get & show current versions */
    } else {
        printf ("\n  *** Image transmission failed.\n");
    }
    free(image_ptr);
    if ((cnt_edc != image_cnt_edc) ||
        ((curr_drive_type >= TK87_DRIVE) &&
                (drive_edc != image_drive_edc)) )  {
        printf ("\n  *** Warning! Current tape unit's EDCs don't match ");
        printf ("image EDCs! ***\n\n\t\t*** Update Failed!! ****\n");
    } else {
        printf ("\n  *** Update Verified! Drive EDCs now match image.\n");
    }
    printf ("\n\t     (hit 'return' to continue)\n");
    my_gets (input_line,sizeof(input_line));
}

/*++
***************************************************************************
**
**  **-get_image
**
**  FUNCTIONAL DESCRIPTION:
**
**
**  FORMAL PARAMETERS:
**
**
**  RETURN VALUE:
**      True if transmission successful, else False
**
***************************************************************************
--*/
char *get_image (image_size)
int image_size;
{
    int status;
    char input_line[80];
    char *image_file_name;
    char *image_ptr;
    FILE *fp;

    printf("\n  *** Enter Filename: ");
    my_gets (input_line,sizeof(input_line));
    image_file_name = input_line;
    /*
     * Open the requested image file.
     */
    if ((fp = fopen(image_file_name, "r")) == NULL) {
        printf("  *** Unable to open input file.\n");
        return (NULL);
    }
    /*
     * Allocate a buffer large enough for entire image.
     */
    image_ptr = (char *) malloc(image_size);
    if (image_ptr == NULL) {
        printf("  *** Unable to allocate buffer for image.\n");
        return (NULL);
    }
    /*
     * Read in the requested image file.
     */
    status = fread (image_ptr, image_size, 1, fp);
    if (status == 0) {
        printf("  *** Unable to read input image file.\n");
        free(image_ptr);
        fclose (fp);
        return (NULL);
    }

    status = fread (input_line, 1, 1, fp);
    if (status != 0) {
        printf("  ***input image file too big.\n");
        free(image_ptr);
        fclose (fp);
        return (NULL);
    }

    fclose (fp);
    return (image_ptr);
}

/*++
***************************************************************************
**
**  **-validate_image
**
**  FUNCTIONAL DESCRIPTION:
**
**
**  FORMAL PARAMETERS:
**
**
**  RETURN VALUE:
**     True if code's EDC calcuation was zero, else False.
**
***************************************************************************
--*/
int validate_image (image_size, image_ptr)
int image_size;
char *image_ptr;
{
    unsigned long int i;
    unsigned long int edc_residue;

    /*
     * If this image contains the drive code as well, point past it.
     */
    switch (curr_drive_type) {
        case TK87_DRIVE:
            image_ptr = image_ptr + TZC_DRIVE_CODE_SIZE;
            image_size = TZC_IMAGE_SIZE;
            image_size = image_size - TZC_DRIVE_CODE_SIZE;
            break;
        case TK87XT_DRIVE:
            image_ptr = image_ptr + TZF_DRIVE_CODE_SIZE;
            image_size = TZF_IMAGE_SIZE;
            image_size = image_size - TZF_DRIVE_CODE_SIZE;
            break;
        case TK88_DRIVE:
            image_ptr = image_ptr + TZC_DRIVE_CODE_SIZE;
            image_size = TZE_IMAGE_SIZE;
            image_size = image_size - TZC_DRIVE_CODE_SIZE;
            break;
        case TK89_DRIVE:
            if (mem_model == MEM_MODEL_LARGE ) {
                image_ptr = image_ptr + TZG_DRIVE_CODE_SIZE + TZG_ZONE_SIZE;
/* used to be if falsed */
                image_size = TZG_IMAGE_SIZE;
/**/
                image_size = image_size - TZG_DRIVE_CODE_SIZE - TZG_ZONE_SIZE;
            } else { /* small model */
                image_ptr = image_ptr + TZGS_DRIVE_CODE_SIZE + TZGS_ZONE_SIZE;
/* used to be if falsed */
                image_size = TZGS_IMAGE_SIZE;
/**/
                image_size = image_size - TZGS_DRIVE_CODE_SIZE - TZGS_ZONE_SIZE;

            }
            break;
        default:
            break;
    }

    edc_residue = 0x45;
    for (i = 0; i < (image_size / 4); i++) {
        edc_residue ^= reverse_bytes (image_ptr);
        image_ptr += 4;
        edc_residue = (edc_residue << 1) | (edc_residue >> 31);
        edc_residue *= 3;
    }
    if (edc_residue != 0) {
        return (FALSE);
    } else {
        return (TRUE);
    }
}

/*++
***************************************************************************
**
**  **-transmit_image
**
**  FUNCTIONAL DESCRIPTION:
**
**
**  FORMAL PARAMETERS:
**
**
**  RETURN VALUE:
**      True if transmission successful, else False
**
***************************************************************************
--*/
int transmit_image (image_size, image_ptr)
int image_size;
char *image_ptr;
{
    int i;
    int status;
    struct WRB *WriteBuff_Ptr;      /* Pointer to the Write Buffer CDB */
    struct WRB WriteBuff_CDB;       /* Allocation of Write Buffer CDB */

    WriteBuff_Ptr = &WriteBuff_CDB;

    WriteBuff_Ptr->Opcode = 0x3B;
    WriteBuff_Ptr->Mode = 4;
    WriteBuff_Ptr->Buff_ID = 0;
    WriteBuff_Ptr->BufferOffset[0] = 0;
    WriteBuff_Ptr->BufferOffset[1] = 0;
    WriteBuff_Ptr->BufferOffset[2] = 0;
    WriteBuff_Ptr->AllocLen[0] = (unsigned char)(chunk_size>>16);   /* msb */
    WriteBuff_Ptr->AllocLen[1] = (unsigned char)(chunk_size>>8);    /* middle */
    WriteBuff_Ptr->AllocLen[2] = (unsigned char)(chunk_size);       /* lsb */
    WriteBuff_Ptr->Cntrl = 0;

    for (i = 0; i < (image_size / chunk_size); i++) {
        if (i == ((image_size / chunk_size) - 1)) {
            WriteBuff_Ptr->Mode = 5;                    /* download with save */
        }
        WriteBuff_Ptr->BufferOffset[0] = (i * chunk_size) / 0x10000;
        WriteBuff_Ptr->BufferOffset[1] = ((i * chunk_size) & 0xFFFF) / 0X100;
        WriteBuff_Ptr->BufferOffset[2] = 0;
        status = scsi_qio((unsigned char *)&WriteBuff_CDB,
                        sizeof (struct WRB),
                        (unsigned char *)image_ptr,
                        chunk_size,
                        FLAGS_WRITE);
        image_ptr = image_ptr + chunk_size;
        if (status != TRUE) {
            break;
        }
    }
    return (status);
}

/*++
***************************************************************************
**
**  **-wait_for_cup_completion
**
**  FUNCTIONAL DESCRIPTION:
**
**
**  FORMAL PARAMETERS: none
**
**  RETURN VALUE: Void
**
***************************************************************************
--*/
void wait_for_cup_completion ()
{
    int update_wait = 0;
    int drive_wait = 60;

    do {
        sleep (1);
        if (get_sense_data() == TRUE) {
            if (curr_drive_type >= TK87_DRIVE) { /* If drive fw updateable...*/
                while ((drive_edc != image_drive_edc) && --drive_wait){
                    sleep(1);       /* Wait for unit to get & store... */
                    inquiry_info(); /* get lastest info from the device */
                }                   /* ... & report lastest drive fw rev */
            }
            printf ("\n  *** Tape Drive re-boot completed. ***");
            show_current_version ();
            break;
        } else if (scsi_status == 8) {      /* Did drive report BUSY? */
            if (print_busy) {
                print_busy = FALSE;
            } else {
                printf (".");
            }
        } else if (update_wait&1) {     /* count = odd? */
            printf (".");   /* print every other second */
        }
    } while (update_wait++ < 300);       /* wait up to 5 minutes */

    print_busy = TRUE;          /* Restore */
    if (update_wait >= 300) {
        printf( "\n\n  *** Failed getting information from new system. ***\n");
    }
}

/*++
***************************************************************************
**
**  **-show_timestamp
**
**  FUNCTIONAL DESCRIPTION:
**
**
**  FORMAL PARAMETERS:
**
**
**  RETURN VALUE:
**      Void
**
***************************************************************************
--*/
void show_timestamp (image_size, image_ptr)
int image_size;
char *image_ptr;
{
    unsigned long int vms_timestamp[2];
    int status;
    static char time_string[24];
#ifdef VMS
    $DESCRIPTOR( timbuf_desc, time_string );
#else /* ifdef VMS */
#endif /* ifdef VMS */
    unsigned short strlen;

    if (image_size == TZA_IMAGE_SIZE) {
        vms_timestamp[0] = reverse_bytes (image_ptr+TZA_TIME_STAMP_LOC+4);
        vms_timestamp[1] = reverse_bytes (image_ptr+TZA_TIME_STAMP_LOC);

    } else {
        switch (curr_drive_type) {
            case TK87_DRIVE:
                vms_timestamp[0] = reverse_bytes(image_ptr+TZC_TIME_STAMP_LOC+4);
                vms_timestamp[1] = reverse_bytes(image_ptr+TZC_TIME_STAMP_LOC);
                break;
            case TK87XT_DRIVE:
                vms_timestamp[0] = reverse_bytes(image_ptr+TZF_TIME_STAMP_LOC+4);
                vms_timestamp[1] = reverse_bytes(image_ptr+TZF_TIME_STAMP_LOC);
                break;
            case TK88_DRIVE:
                vms_timestamp[0] = reverse_bytes(image_ptr+TZE_TIME_STAMP_LOC+4);
                vms_timestamp[1] = reverse_bytes(image_ptr+TZE_TIME_STAMP_LOC);
                break;
            case TK89_DRIVE:
                if (mem_model == MEM_MODEL_LARGE)  {
                    vms_timestamp[0] = reverse_bytes(image_ptr+TZG_TIME_STAMP_LOC+4);
                    vms_timestamp[1] = reverse_bytes(image_ptr+TZG_TIME_STAMP_LOC);
                } else { /*small */
                    vms_timestamp[0] = reverse_bytes(image_ptr+TZGS_TIME_STAMP_LOC+4);
                    vms_timestamp[1] = reverse_bytes(image_ptr+TZGS_TIME_STAMP_LOC);
                }
                break;
        }
    }
#ifdef VMS
    status = sys$asctim( &strlen, &timbuf_desc, vms_timestamp, 0 );
    time_string[strlen-3]= '\0';

    printf ("  *** New Image creation date:\t\t%s\n", time_string);
#else /* ifdef VMS */
  printf ("  *** New Image creation date: %X\n", vms_timestamp);
#endif /* ifdef VMS */
}
/*++
***************************************************************************
**
**  **-show_new_version
**
**  FUNCTIONAL DESCRIPTION:
**      Extract from the image data specified, the controller (and drive
**      if TZ87/88) revision and release information.  Print to display.
**
**      The following global(s) set setup:
**          image_cnt_rev
**          image_drive_rev
**
**  RETURN VALUE: Void
**
***************************************************************************
--*/
void show_new_version (image_size, image_ptr)
int image_size;
char *image_ptr;
{
    unsigned int release;
    unsigned int major;
    unsigned int minor;

    if (image_size == TZA_IMAGE_SIZE) {
        release = (unsigned int) *(image_ptr + TZA_RELEASE_LOC);
        major = (unsigned int) *(image_ptr + TZA_MAJOR_LOC);
        minor = (unsigned int) *(image_ptr + TZA_MINOR_LOC);
        image_cnt_edc = reverse_bytes(image_ptr + TZA_EDC_LOC);
    } else {
        if (curr_drive_type == TK87_DRIVE) {
            release = (unsigned int) *(image_ptr + TZC_RELEASE_LOC);
            major = (unsigned int) *(image_ptr + TZC_MAJOR_LOC);
            minor = (unsigned int) *(image_ptr + TZC_MINOR_LOC);
            image_cnt_edc = reverse_bytes(image_ptr + TZC_EDC_LOC);
        } else if (curr_drive_type == TK88_DRIVE) {
            release = (unsigned int) *(image_ptr + TZE_RELEASE_LOC);
            major = (unsigned int) *(image_ptr + TZE_MAJOR_LOC);
            minor = (unsigned int) *(image_ptr + TZE_MINOR_LOC);
            image_cnt_edc = reverse_bytes(image_ptr + TZE_EDC_LOC);
        } else if (curr_drive_type == TK87XT_DRIVE) {
            release = (unsigned int) *(image_ptr + TZF_RELEASE_LOC);
            major = (unsigned int) *(image_ptr + TZF_MAJOR_LOC);
            minor = (unsigned int) *(image_ptr + TZF_MINOR_LOC);
            image_cnt_edc = reverse_bytes(image_ptr + TZF_EDC_LOC);
        } else { /* tz89 */
            if (mem_model == MEM_MODEL_LARGE) {
                release = (unsigned int) *(image_ptr + TZG_RELEASE_LOC);
                major = (unsigned int) *(image_ptr + TZG_MAJOR_LOC);
                minor = (unsigned int) *(image_ptr + TZG_MINOR_LOC);
                image_cnt_edc = reverse_bytes(image_ptr + TZG_EDC_LOC);
            } else { /* small */
                release = (unsigned int) *(image_ptr + TZGS_RELEASE_LOC);
                major = (unsigned int) *(image_ptr + TZGS_MAJOR_LOC);
                minor = (unsigned int) *(image_ptr + TZGS_MINOR_LOC);
                image_cnt_edc = reverse_bytes(image_ptr + TZGS_EDC_LOC);
            }
       }
    }

    printf( "  *** Image versions:  \tCnt= ");
    if (release == 0) {
        printf( "X" );
    } else {
        if (minor == 0) {
            printf( "V" );
        } else {
            printf( "T" );
        }
    }
    image_cnt_rev = major;
    printf ("%03u-%d/ %08X", major, minor, image_cnt_edc);
    /*
     * If this is a TZ87/87xt/88/89 image, also display the Drive code info.
     */
    if (curr_drive_type >= TK87_DRIVE) {
        printf( "\tDrive= ");
        image_drive_rev = *(image_ptr+2);
        image_drive_edc = get_word((image_ptr+4));
        printf ("%d/ %04X (%c-%d)\n",
            image_drive_rev, image_drive_edc,
            (unsigned char) *(image_ptr + 0),   /* alternate nomenclature */
            (unsigned char) *(image_ptr + 1) );
    }
}

/*++
***************************************************************************
**
**  **-show_current_version
**
**  FUNCTIONAL DESCRIPTION:
**
**
**  FORMAL PARAMETERS: none
**
**  RETURN VALUE: Void
**
***************************************************************************
--*/
void show_current_version ()
{
    inquiry_info();     /* Make sure we have up-to-date info */
    printf( "\n\n  *** Current versions:\tCnt= ");
    if ((inquiry_data[36]&InqDat_M_RelOpt) == 0) {
        printf( "X" );
    } else {
        if (inquiry_data[38] == 0) {
            printf("V");
        } else {
            printf("T");
        }
    }
    printf ("%03u-%d/ %08X", cnt_rev, inquiry_data[38], cnt_edc);
    printf( "\tDrive= ");
    printf("%03u/ %04X\n", drive_rev , drive_edc);
}

/**********************************************************************/
#ifdef FUNCTION_PROTOTYPES
unsigned long int reverse_bytes (char *source_ptr)
#else /* ifdef FUNCTION_PROTOTYPES */
unsigned long int reverse_bytes (source_ptr)
char *source_ptr;
#endif /* ifdef FUNCTION_PROTOTYPES */
{
#ifdef LITTLEENDIAN
    int i;
    unsigned char result[4];

    for (i = 0; i < 4; i++)  {
        result[i]= *(source_ptr+3-i);
    }
    return( *(unsigned long int *)result );
#else /* ifdef LITTLEENDIAN */
    return( *(unsigned long int *)source_ptr );
#endif /* ifdef LITTLEENDIAN */
}

/**********************************************************************/
unsigned short int get_word(dptr)
unsigned char *dptr;
{
    unsigned short int data;

    data = (*dptr++)<<8;
    data |= *dptr;
    return(data);
}

/**********************************************************************/
unsigned long int get_long(dptr)
unsigned char *dptr;
{
    unsigned long int data;

    data = (*dptr++)<<24;
    data |= (*dptr++)<<16;
    data |= (*dptr++)<<8;
    data |= *dptr;
    return(data);
}

/**********************************************************************/
void do_byte_swap (image_size, image_ptr)
int image_size;
char *image_ptr;
{
    int i;
    char *dst_ptr;
    char hi_byte;
    char lo_byte;

    dst_ptr = image_ptr;
    for (i = 0; i < image_size; i+=2) {
        hi_byte =  *(image_ptr);
        lo_byte =  *(image_ptr + 1);
        *(dst_ptr) = lo_byte;
        *(dst_ptr + 1) = hi_byte;
        image_ptr += 2;
        dst_ptr += 2;
    }
}

void my_gets(char *buffer,unsigned buflen)
{ int len;
  fgets(buffer,buflen,stdin);
  len=strlen(buffer);
  if(len&&buffer[len-1]=='\n')
  { buffer[len-1]='\0';
  }
}

#if DEBUG
#define BYTES_PER_LINE 16
void dump(char *what,unsigned char *data,unsigned bytes)
{ unsigned index,i;
  unsigned char *cp1,*cp2,*cp3;
  printf("%s - %u bytes\n",what,bytes);
  if(bytes>MAX_DUMP)
  { bytes=MAX_DUMP;
  }
  i=0;
  cp1=data;
  for(index=0;index<bytes;)
  { printf("%04x",index);
    cp3=cp2=cp1;
    index+=BYTES_PER_LINE;
    if(index>bytes)
    { i=3*(index-bytes);
      index=bytes;
    }
    cp1=data+index;
    while(cp2<cp1)
    { printf(" %02x",*cp2++);
    }
    for(++i;i;--i)
    { putchar(' ');
    }
    for(;cp3<cp1;++cp3)
    { if(*cp3>=' '&&*cp3<127)
      { putchar(*cp3);
      }
      else
      { putchar(' ');
      }
    }
    putchar('\n');
  }
}
#endif
