/*
 * clddsetup.c - setup and control cldd devices
 *
 * Copyright 1999 Allan Latham <alatham@flexsys-group.com>
 *
 */

#include "clddmount.h"
#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <getopt.h>
#include <linux/fs.h>

#include <sys/ioctl.h>
static char *progname;


static int closedown_cldd(const char *device)
{
	int 				fd;

	sync();

	if ((fd = open(device, O_RDONLY)) < 0) {
		CLDDERROR(101)
		return 1;
	}

	if (ioctl(fd, CLDD_CLR_MASTER, 0) < 0) {
	    CLDDERROR(102)
	    return 1;
	}

	close(fd);
	sync();
	return 0;
}

static int disable_cldd(const char *device)
{
	int 				fd;

	sync();

	if ((fd = open(device, O_RDONLY)) < 0) {
		CLDDERROR(103)
		return 1;
	}

	if (ioctl(fd, CLDD_CLR_SLAVE, 0) < 0) {
	    CLDDERROR(104)
	    return 1;
	}

	close(fd);
	sync();
	return 0;
	return 0;
}

static int enable_cldd(const char *device)
{
	int 				fd;

	sync();

	if ((fd = open(device, O_RDONLY)) < 0) {
		CLDDERROR(105)
		return 1;
	}

	if (ioctl(fd, CLDD_SET_SLAVE, 0) < 0) {
	    CLDDERROR(106)
	    return 1;
	}

	close(fd);
	sync();
	return 0;
	return 0;
}

static int setup_cldd(const char *device, const char *file)
{
	struct cldd_info 		info;
	int				fd, ffd;
        
 
	if (file[0] != '/') {
	    fprintf(stderr,"real device name must begin with \"/\"\n\n");
		return 1;
	}

	if ((ffd = open (file, O_RDONLY)) < 0) {
		CLDDERROR(107)
		return 1;
	}

	if ((fd = open (device, O_RDONLY)) < 0) {
		CLDDERROR(108)
		close (ffd);
		return 1;
	}

        if (ioctl(fd, CLDD_SET_MASTER, ffd) < 0) {
		CLDDERROR(109)
		close (ffd);
		close (fd);
	  	return 1;
	}

	if (ioctl(fd, CLDD_GET_STATUS, &info) < 0) {
		CLDDERROR(110)
		ioctl(fd, CLDD_CLR_MASTER, 0);
		close (ffd);
		close (fd);
	  	return 1;
	}

	if (info.driver_version != CLDD_VERSION) {
		fprintf(stderr,"Warning: kernel driver is version %d.%d \n\
       but I am version %d.%d !!!\n\n",
			info.driver_version/10, info.driver_version%10,
			CLDD_VERSION/10, CLDD_VERSION%10);
	}

	memset(&info, 0, sizeof(info));
	strncpy(info.name, file, CLDD_NAME_SIZE);
	info.name[CLDD_NAME_SIZE-1] = 0;

	if (ioctl(fd, CLDD_SET_STATUS, &info) < 0) {
		CLDDERROR(111)
		ioctl(fd, CLDD_CLR_MASTER, 0);
		close (ffd);
		close (fd);
	  	return 1;
	}

	close(ffd);
	close(fd);
        sync();
	return (0);
}

static int show_cldd(const char *device)
{
	struct	cldd_info	info;
	int			fd;
	long			size;
	char			*date;
	char			*status;
	char			*master = "Slave is not enabled.";
	char			*slaved = "Slave is enabled.";
	char			*synch  = "Synch is in progress.";
	char			*other  = "Status is unknown.";

	
	if ((fd = open(device, O_RDONLY)) < 0) {
		CLDDERROR(112)
		return 1;
	}
	if (ioctl(fd, CLDD_GET_STATUS, &info) < 0) {
		CLDDERROR(113)
		close(fd);
		return 1;
	}
	if (ioctl(fd, BLKGETSIZE, &size) < 0) {
		CLDDERROR(114)
		close(fd);
		return 1;
	}
	switch (info.status) {
		case CLDD_STATUS_MASTER:
			status = master;
			break;
		case CLDD_STATUS_SLAVED:
			status = slaved;
			break;
		case CLDD_STATUS_SYNCH:
			status = synch;
			break;
		default:
			status = other;
	}

	printf("\nInformation for %s:\n\n", device);
	printf("\tReal device is %s.\n", info.name);
	printf("\tSize is %ld bytes, (%ld blocks).\n", size*512, size/2);
	printf("\tDriver version is %01d.%01d.\n\n",
		info.driver_version/10, info.driver_version%10);
	date = ctime(&info.mtv.tv_sec);
	printf("\tMaster was set up\t%s", date);
	printf("\t%s\n", status);
	date = ctime(&info.stv.tv_sec);
	if (info.status == CLDD_STATUS_MASTER) {
		if (info.stv.tv_sec) {
			printf("\tSlave was last enabled\t%s", date);
		} else {
			printf("\tSlave has never been enabled.\n");
		}
	} else {
		printf("\tSlave was enabled\t%s", date);
	}
	printf("\n");
	close(fd);
	return 0;
}


static int usage(void)
{
	fprintf(stderr, "usage:\n\
  %s    cldd_master_device               # show details\n\
  %s -c cldd_master_device               # closedown master device\n\
  %s -d cldd_slave_device                # disable slave\n\
  %s -e cldd_slave_device                # enable slave\n\
  %s -m cldd_master_device real_device   # setup master device\n\n",
			progname,progname,progname,progname,progname);
	exit(1);
}

int main(int argc, char **argv)
{
	int c;
	int closedown = 0;
	int disable = 0;
	int setup = 0;
	int enable = 0;
	int count = 0;

	progname = "clddsetup";

	if (cldd_intro(progname)) usage();

	while ((c = getopt(argc,argv,"cdem")) != EOF) {
		switch (c) {
			case 'c':
				closedown = 1;
                                count++;
				break;
			case 'd':
				disable = 1;
                                count++;
				break;
			case 'e':
				enable = 1;
                                count++;
				break;
			case 'm':
				setup = 1;
                                count++;
				break;
			default:
				usage();
		}
	}
	if (count > 1) usage();
	if ((count == 0) && (argc !=2)) usage();
	if (enable && (argc != 3)) usage();
	if (disable && (argc != 3)) usage();
	if (closedown && (argc != 3)) usage();
	if (setup && (argc != 4)) usage();

	if (closedown) exit (closedown_cldd(argv[optind]));
	if (disable) exit (disable_cldd(argv[optind]));
	if (enable) exit (enable_cldd(argv[optind]));
	if (setup) exit (setup_cldd(argv[optind],argv[optind+1]));

	exit (show_cldd(argv[optind]));
}

