#include <stdio.h>
#include <math.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <X11/Intrinsic.h>
#include <X11/Core.h>
#include <X11/StringDefs.h>
#include "color.h"
#include "list.h"

#define	DEF_BLACK	BlackPixel(dsp, DefaultScreen(dsp))
#define	DEF_WHITE	WhitePixel(dsp, DefaultScreen(dsp))


extern void Duplicate();
extern void MakeVisList();
extern void Rotate();
extern void marbleTriangle();
extern void woodTriangle();
extern void graniteTriangle();
extern void fireTriangle();
extern void GetShades();
extern void InitCHash();
extern void FreeCHash();
extern void AddCHash();
extern int FindCHash();
extern void NewPts();

extern struct col_rec	Colors[];
extern struct col_rec	BaseColors[];
extern int	Npts, Npolys;
extern int	*AllPolys;
extern int	*AllPixels;
extern float	*AllProbs;
extern float	*AllPts;
extern float	view_dist;
extern int	NumCells;
extern int	BaseCells;
extern int	BaseCount;
extern int	ShadeType;
extern int	Changes;
extern int	Erode;
extern Bool	Nomap;
extern Display	*dsp;
extern Window	Dwindow;
extern GC	DrawGC;
extern Colormap	Cmap;
extern Widget	drawingcore;
extern unsigned int Width, Height;
extern Bool	Perspective;
extern int	NoBlack;
extern int	NoWhite;
extern struct node_rec *TreeRoot;
extern float	Old_distance;

int count;
struct col_rec DynColors[256];


/*
 * Find the closest matching color for this polygon in the colormap.
 * This is expensive, so first chech the hash table to see if you looked
 * it up before.
 */
int
GetColor(list_ptr, colornum, red, green, blue)
	struct list_rec *list_ptr;
	int colornum;
	int red, green, blue;
{
	int indx;
	register int dist, tmp, greatest;
	register int i;
	register int b_red, b_green, b_blue;
	register int rd, gd, bd;
	struct col_rec *cptr;

	indx = FindCHash(red, green, blue);
	if (indx != -1)
	{
		return(indx);
	}

	b_red = red;
	b_green = green;
	b_blue = blue;


	/*
	 * Find the closest pixel in the current colormap
	 * that matches the color we want.
	 */
	if (list_ptr->pixel == -1)
	{
		indx = colornum;
	}
	else
	{
		indx = list_ptr->pixel;
	}
	dist = 3 * 65536;
	greatest = 65536;
	/*
	 * greatest denotes the shrinking bounds of a bounding cude that
	 * is used to reduce the search space, and try and make this
	 * function faster.
	 */
	cptr = DynColors;
	for (i=0; i<NumCells; i++, cptr++)
	{
		if ((NoBlack)&&((cptr->red == 0)&&
				(cptr->green == 0)&&
				(cptr->blue == 0)))
		{
			continue;
		}
		if ((NoWhite)&&((cptr->red == (NumCells - 1))&&
				(cptr->green == (NumCells - 1))&&
				(cptr->blue == (NumCells - 1))))
		{
			continue;
		}
		rd = (b_red - cptr->red);
		rd = rd * rd;
		if (rd > greatest)
		{
			continue;
		}

		gd = (b_green - cptr->green);
		gd = gd * gd;
		if (gd > greatest)
		{
			continue;
		}

		bd = (b_blue - cptr->blue);
		bd = bd * bd;
		if (bd > greatest)
		{
			continue;
		}

		tmp = rd + gd + bd;
		if (tmp < dist)
		{
			indx = i;
			dist = tmp;
			if (dist == 0)
			{
				return(indx);
			}
			if (tmp < greatest)
			{
				greatest = tmp;
			}
		}
	}

	return(indx);
}



/*
 * Given a base color for this polygon, shade it based on the base
 * color, and its angle to the light.
 */
void
BaseShade(list_ptr, colornum, points)
	struct list_rec *list_ptr;
	int colornum;
	XPoint *points;
{
	int b_red, b_green, b_blue;
	int i, tmp, indx, dist;

	/*
	 * If there is no base color, set a gray shade,
	 * otherwise pick the right shade of the base color.
	 */
	if (list_ptr->pixel == -1)
	{
		b_red = colornum;
		b_green = colornum;
		b_blue = colornum;
	}
	else
	{
		b_red = DynColors[list_ptr->pixel].red;
		b_green = DynColors[list_ptr->pixel].green;
		b_blue = DynColors[list_ptr->pixel].blue;
		b_red = b_red * colornum / NumCells;
		b_green = b_green * colornum / NumCells;
		b_blue = b_blue * colornum / NumCells;
	}

	if ((NoBlack)&&((b_red == 0)&&
			(b_green == 0)&&
			(b_blue == 0)))
	{
		b_red = 1;
		b_green = 1;
		b_blue = 1;
	}

	if ((NoWhite)&&((b_red == (NumCells - 1))&&
			(b_green == (NumCells - 1))&&
			(b_blue == (NumCells - 1))))
	{
		b_red = NumCells - 2;
		b_green = NumCells - 2;
		b_blue = NumCells - 2;
	}

	indx = GetColor(list_ptr, colornum, b_red, b_green, b_blue);

	/*
	 * If the colormap is not full, and the closest color is not
	 * exactly the color we asked for, then we should be able tp
	 * allocate exactly the color we want.
	 */
	if ((count < NumCells) &&
		((DynColors[indx].red != b_red)||
		(DynColors[indx].green != b_green)||
		(DynColors[indx].blue != b_blue)))
	{
		XColor tmpcolor;
		tmpcolor.flags = DoRed|DoGreen|DoBlue;
		tmpcolor.red = b_red * 65536 / NumCells;
		tmpcolor.green = b_green * 65536 / NumCells;
		tmpcolor.blue = b_blue * 65536 / NumCells;
		tmpcolor.pixel = indx;
		if (Nomap == True)
		{
			XAllocColor(dsp, Cmap, &tmpcolor);
			if (tmpcolor.pixel != indx)
			{
				DynColors[tmpcolor.pixel].red = b_red;
				DynColors[tmpcolor.pixel].green = b_green;
				DynColors[tmpcolor.pixel].blue = b_blue;
			}
		}
		else
		{
			tmpcolor.pixel = count;
			XStoreColor(dsp, Cmap, &tmpcolor);
			DynColors[tmpcolor.pixel].red = b_red;
			DynColors[tmpcolor.pixel].green = b_green;
			DynColors[tmpcolor.pixel].blue = b_blue;
		}
		indx = tmpcolor.pixel;
	}
	/*
	 * This is a trick that just increments count 99.9% of the
	 * time, but if the color needed was black, and there was not
	 * already a black in the colormap.  Since  unused cells
	 * are initialized to black, it would have taken the next
	 * unused cell without executing the if above.  In that
	 * case, this if increments count to save that black so it
	 * won't be thought of as unused and get written over later.
	 */
	if ((Nomap == False)&&(indx >= count))
	{
		count = indx + 1;
	}

	/*
	 * Add computed color to the hash table so we can reuse it if needed
	 */
	AddCHash(b_red, b_green, b_blue, indx);

	XSetForeground(dsp, DrawGC, indx);
	if ((points[0].x == points[1].x)&&(points[0].x == points[2].x)&&
	    (points[0].y == points[1].y)&&(points[0].y == points[2].y))
	{
		XDrawPoint(dsp, Dwindow, DrawGC,
			points[0].x, points[0].y);
	}
	else
	{
		XFillPolygon(dsp, Dwindow, DrawGC, points, 3,
			Convex, CoordModeOrigin);
	}
}



/*
 * Draw the triangle pointed to by list_ptr.
 */
void
drawTriangleFrame(list_ptr, distance)
	struct list_rec *list_ptr;
	float distance;
{
	int i, draw;
	int xscale, yscale, xoffset, yoffset;
	int colornum, indx, tmp, dist;
	float xp1, yp1, xp2, yp2, xp3, yp3;
	float xu1, xu2, xu3, yu1, yu2, yu3, zu1, zu2, zu3;
	float *pa, *pb, *pc;
	XPoint points[4];

	draw = 1;
	/*
	 * scale points so distance on 1.0 equals 1/2 window size
	 */
	xscale = xoffset = (Width) / 2;
	yscale = yoffset = (Height) / 2;

	/*
	 * get the three verticies of the triangle
	 *              pa
	 *             / \
	 *            /   \
	 *          pc-----pb
	 */
	pa = list_ptr->a;
	pb = list_ptr->b;
	pc = list_ptr->c;

	/*
	 * get the x, y, z coordinates of the verticies
	 *         xu1,yu1,zu1
	 *             / \
	 *            /   \
	 * xu3,yu3,zu3-----xu2,yu2,zu2
	 */
	xu1 = *pa;
	yu1 = *(pa + 1);
	zu1 = *(pa + 2) - distance;
	xu2 = *pb;
	yu2 = *(pb + 1);
	zu2 = *(pb + 2) - distance;
	xu3 = *pc;
	yu3 = *(pc + 1);
	zu3 = *(pc + 2) - distance;
	
	/*
	 * clip triangles that are partially behind your "eye" 
	 */
	if ((zu1 >= view_dist)||(zu2 >= view_dist)||(zu3 >= view_dist))
	{
		draw = 0;
	}

	/*
	 * convert to x, y coordinates based on perpective projection
	 *           xp1,yp1
	 *             / \
	 *            /   \
	 *     xp3,yp3-----xp2,yp2
	 */
	if (Perspective == True)
	{
		xp1 = (xu1 * view_dist) / (view_dist - zu1);
		yp1 = (yu1 * view_dist) / (view_dist - zu1);
		xp2 = (xu2 * view_dist) / (view_dist - zu2);
		yp2 = (yu2 * view_dist) / (view_dist - zu2);
		xp3 = (xu3 * view_dist) / (view_dist - zu3);
		yp3 = (yu3 * view_dist) / (view_dist - zu3);
	}
	else
	{
/*
		xp1 = xu1;
		yp1 = yu1;
		xp2 = xu2;
		yp2 = yu2;
		xp3 = xu3;
		yp3 = yu3;
*/
		xp1 = (xu1 * view_dist) / (view_dist + distance);
		yp1 = (yu1 * view_dist) / (view_dist + distance);
		xp2 = (xu2 * view_dist) / (view_dist + distance);
		yp2 = (yu2 * view_dist) / (view_dist + distance);
		xp3 = (xu3 * view_dist) / (view_dist + distance);
		yp3 = (yu3 * view_dist) / (view_dist + distance);
	}

	/*
	 * scale x coordinates to window space
	 */
	points[0].x = (int)(xp1 * xscale) + xoffset;
	points[1].x = (int)(xp2 * xscale) + xoffset;
	points[2].x = (int)(xp3 * xscale) + xoffset;
	points[3].x = points[0].x;

	/*
	 * scale y coordinates to window space
	 */
	points[0].y = yoffset - (int)(yp1 * yscale);
	points[1].y = yoffset - (int)(yp2 * yscale);
	points[2].y = yoffset - (int)(yp3 * yscale);
	points[3].y = points[0].y;

	/*
	 * calculate colornum which maps to the intensity of the
	 * light reflecting off this triangle.
	 * All triangles have at least 1/10 intensity.
	 */
	colornum = (int)(NumCells * list_ptr->cos_theta);
	colornum += (NumCells / 10);
	if (colornum >= NumCells)
	{
		colornum = NumCells - 1;
	}

	/*
	 * If no colormap, find closest color intensity in Colors[] array
	 */
	if (Nomap == True)
	{
		indx = colornum;
		dist = (int)sqrt((double)3 * 256 * 256);
		for (i=0; i<NumCells; i++)
		{
			if ((NoBlack)&&((Colors[i].red == 0)&&
					(Colors[i].green == 0)&&
					(Colors[i].blue == 0)))
			{
				continue;
			}
			if ((NoWhite)&&((Colors[i].red == (NumCells - 1))&&
					(Colors[i].green == (NumCells - 1))&&
					(Colors[i].blue == (NumCells - 1))))
			{
				continue;
			}
			tmp = (int)sqrt((double)(colornum - Colors[i].red) *
				(colornum - Colors[i].red) +
				(colornum - Colors[i].green) *
				(colornum - Colors[i].green) +
				(colornum - Colors[i].blue) *
				(colornum - Colors[i].blue));
			if (tmp < dist)
			{
				indx = i;
				dist = tmp;
			}
		}
		colornum = indx;
	}

	/*
	 * figure out whether or not to draw triangle based on
	 * opacity probability.
	 */
	if (draw == 1)
	{
		if (list_ptr->prob == 0.0)
		{
			/* Opacity of 0.0 is a special translucent polygon */
			draw = 2;
		}
		else if (list_ptr->prob == 1.0)
		{
			draw = 1;
		}
		else if (((float)(rand() % 32767) / 32767.0) < list_ptr->prob)
		{
			draw = 1;
		}
		else
		{
			draw = 0;
		}
	}

	/* Wireframe */
	if ((ShadeType == 0)&&((draw == 1)||(draw == 2)))
	{
		XSetForeground(dsp, DrawGC, DEF_WHITE);
/*
for (i=0; i<4; i++)
{
printf("(%d, %d) ", points[i].x, points[i].y);
}
printf("\n");
*/
		XDrawLines(dsp, Dwindow, DrawGC ,points, 4,
			CoordModeOrigin);
	}
	/* GrayShade */
	else if ((ShadeType == 1)&&(draw == 1))
	{
		XSetForeground(dsp, DrawGC, colornum);
		XFillPolygon(dsp, Dwindow, DrawGC, points, 3,
			Convex, CoordModeOrigin);
	}
	/* Fire */
	else if ((ShadeType == 2)&&(draw == 1))
	{
		fireTriangle(xu1, yu1, zu1, xu2, yu2, zu2, xu3, yu3, zu3, list_ptr->cos_theta);
	}
	/* Granite */
	else if ((ShadeType == 3)&&(draw == 1))
	{
		graniteTriangle(xu1, yu1, zu1, xu2, yu2, zu2, xu3, yu3, zu3, list_ptr->cos_theta);
	}
	/* Marble */
	else if ((ShadeType == 4)&&(draw == 1))
	{
		marbleTriangle(xu1, yu1, zu1, xu2, yu2, zu2, xu3, yu3, zu3, list_ptr->cos_theta);
	}
	/* Wood */
	else if ((ShadeType == 5)&&(draw == 1))
	{
		woodTriangle(xu1, yu1, zu1, xu2, yu2, zu2, xu3, yu3, zu3, list_ptr->cos_theta);
	}
	/* Base Color */
	else if ((ShadeType == 6)&&((draw == 1)||(draw == 2)))
	{
		int pix;

		if (list_ptr->pixel == -1)
		{
			pix = 0;
		}
		else
		{
			pix = list_ptr->pixel;
		}
		XSetForeground(dsp, DrawGC, pix);
		XFillPolygon(dsp, Dwindow, DrawGC, points, 3,
			Convex, CoordModeOrigin);
	}
	/* Base Color Shaded */
	else if ((ShadeType == 7)&&(draw == 1))
	{
		BaseShade(list_ptr, colornum, points);
	}
	/* Translucent (Not on Wire Frame or Base Color) */
	else if ((ShadeType != 0)&&(ShadeType != 6)&&(draw == 2))
	{
		int cx, cy;
		XImage *tmpimage;

		cx = (points[0].x + points[1].x + points[2].x) / 3;
		cy = (points[0].y + points[1].y + points[2].y) / 3;

		/*
		 * Get the color of the Pixel behind this translucent
		 * triangle.  Note this is highly unpredictable code since
		 * the window could be obscured, or otherwise screwed up
		 * and give me the wrong pixel value.
		 */
		tmpimage = XGetImage(dsp, Dwindow, cx, cy,
			(unsigned int)1, (unsigned int)1, AllPlanes, ZPixmap);
		XFlush(dsp);
		indx = XGetPixel(tmpimage, 0, 0);
		XDestroyImage(tmpimage);

		/*
		 * Translucent triangles must have a base color, or they are
		 * worthless
		 */
		if (list_ptr->pixel != -1)
		{
			int b_red, b_green, b_blue;

			b_red = DynColors[indx].red;
			b_green = DynColors[indx].green;
			b_blue = DynColors[indx].blue;

			/*
			 * The Base Color of the translucent triangle
			 * is used as a filter to determine how much of
			 * the underlying color to let through
			 */
			b_red = b_red * BaseColors[list_ptr->pixel].red /
				(BaseCells - 1);
			b_green = b_green * BaseColors[list_ptr->pixel].green /
				(BaseCells - 1);
			b_blue = b_blue * BaseColors[list_ptr->pixel].blue /
				(BaseCells - 1);

			/*
			 * Find nearest color to wanted color in current
			 * colormap
			 */
			dist = (int)sqrt((double)3 * 256 * 256);
			for (i=0; i<NumCells; i++)
			{
				tmp = (int)sqrt((double)
					(b_red - DynColors[i].red) *
					(b_red - DynColors[i].red) +
					(b_green - DynColors[i].green) *
					(b_green - DynColors[i].green) +
					(b_blue - DynColors[i].blue) *
					(b_blue - DynColors[i].blue));
				if (tmp < dist)
				{
					indx = i;
					dist = tmp;
				}
			}

			/*
			 * If we are in Base Shade mode you might be able to
			 * allocate the color you want.
			 */
			if ((ShadeType == 7) && (count < NumCells) &&
				((DynColors[indx].red != b_red)||
				(DynColors[indx].green != b_green)||
				(DynColors[indx].blue != b_blue)))
			{
				XColor tmpcolor;
				tmpcolor.flags = DoRed|DoGreen|DoBlue;
				tmpcolor.red = b_red * 65536 / NumCells;
				tmpcolor.green = b_green * 65536 / NumCells;
				tmpcolor.blue = b_blue * 65536 / NumCells;
				tmpcolor.pixel = indx;
				if (Nomap == True)
				{
					XAllocColor(dsp, Cmap, &tmpcolor);
					if (tmpcolor.pixel != indx)
					{
						DynColors[tmpcolor.pixel].red
							= b_red;
						DynColors[tmpcolor.pixel].green
							= b_green;
						DynColors[tmpcolor.pixel].blue
							= b_blue;
					}
				}
				else
				{
					tmpcolor.pixel = count;
					XStoreColor(dsp, Cmap, &tmpcolor);
					DynColors[tmpcolor.pixel].red = b_red;
					DynColors[tmpcolor.pixel].green = b_green;
					DynColors[tmpcolor.pixel].blue = b_blue;
				}
				indx = tmpcolor.pixel;
			}
			if ((Nomap == False)&&(indx >= count))
			{
				count = indx + 1;
			}
		}

		XSetForeground(dsp, DrawGC, indx);
		XFillPolygon(dsp, Dwindow, DrawGC, points, 3,
			Convex, CoordModeOrigin);
	}
}


/*
 * Travel the tree drawing polygons as you encounter them.
 * since the tree was sorted based on Z, this draws the object back to front.
 */
void
TravelTree(nptr, dist)
	struct node_rec *nptr;
	float dist;
{
	struct list_rec *lptr;

	lptr = nptr->list;
	while (lptr != NULL)
	{
		if (lptr->left != NULL)
		{
			TravelTree(lptr->left, dist);
			drawTriangleFrame(lptr, dist);
		}
		else
		{
			drawTriangleFrame(lptr, dist);
		}
		lptr = lptr->next;
	}
	if (nptr->next != NULL)
	{
		TravelTree(nptr->next, dist);
	}
}


/*
 * Free up the Z sorted binary tree used for Z buffering.
 */
void
FreeTree(nptr)
	struct node_rec *nptr;
{
	struct list_rec *lptr;
	struct list_rec *lptr2;

	lptr = nptr->list;
	while (lptr != NULL)
	{
		if (lptr->left != NULL)
		{
			FreeTree(lptr->left);
			XtFree(lptr->left);
		}
		lptr2 = lptr;
		lptr = lptr->next;
		XtFree(lptr2);
	}
	if (nptr->next != NULL)
	{
		FreeTree(nptr->next);
		XtFree(nptr->next);
	}
}


/*
 * Draw all triangles after rotating and translating them.
 */
void
doTriangles(delta_xt, delta_yt, delta_zt, distance)
	int delta_xt, delta_yt, delta_zt;
	float distance;
{
	int i;

	if (XtIsManaged(drawingcore))
	{
		Rotate(AllPts, Npts, delta_xt, delta_yt, delta_zt);
		TreeRoot = NULL;
		MakeVisList(AllPolys, AllPts, Npolys, Erode, distance);
		XClearWindow(dsp, Dwindow);	
	
		if (Nomap == True)
		{
			GetShades();
		}
		count = BaseCount;
		for (i=0; i<NumCells; i++)
		{
			DynColors[i].red = Colors[i].red;
			DynColors[i].green = Colors[i].green;
			DynColors[i].blue = Colors[i].blue;
		}
		if (TreeRoot != NULL)
		{
			InitCHash();
			TravelTree(TreeRoot, distance);
			XFlush(dsp);
			FreeTree(TreeRoot);
			FreeCHash();
		}
	}
	else
	{
		Changes = 1;
	}
}

