/*****************************************************************************/
/*
                               HyperSPIxx.c

Basic HyperSPI, plus more items, plus better graphics (hyperSPI++)!

This application generates VMS System Performance Information HTML documents.  
Coupled with 'HyperSpi$agent.c', the data-collection application, it can 
profile, complete with graphics, fundamental system performance indicators as 
CPU usage, memory usage, IO values.

It operates in two distinct modes, text and graphic.  In text mode it returns 
an HTML stream to the browser comprising a hypertext page, with selection 
menu, node performance presentation page, data listing or dump page.  In 
graphic mode it returns a GIF image to the browser, first processing the 
specified data into an in-memory bitmap graph, then sending this image to the 
client via an internal GIF processor. 

A node preformance presentation page may have one or more HTML <IMG...> tags 
within it.  Each of these tags will separately cause the browser to invoke the 
application to generate a graphic from the data specified in the SRC= URL. 

If the /NODES= qualifier specifies a comma-separated list of node names these 
are used in the menus for listing the available nodes.  If this is not 
supplied the data directory is searched for current-day data files, those 
found have the respective node name extacted and these are used to list the 
nodes. 

Page colouration may be specified via the appropriate command-line qualifiers
(or corresponding logical name). Defaults for any not specified.  Specify as
/WHATEVER="" to NOT specify the corresponding colour (i.e. leave it to the
browser). See "Qualifiers" section below, and also about the logical name
"HYPERSPI$PARAM".

An example of changing the page colour to white and the banner to red!

  /PBGCOLOR="#ffffff" /PHBGCOLOR="#ff0000"

The script can format a page in either of two layouts.

  1. Tables are used to create a coloured header and button bar (DEFAULT).
     Default colours are white page with grey heading and button outlines.
  2. Textual header, horizontal rules and a textual button bar.
     No default colours.

Button labels are customizable (potentially to non-English language). They
comprise a label, equate symbol and URL-style path suitable for creating a
link. Multiple buttons are separated using the semicolon. Note that any such
button customization must provide escaped HTML-forbidden characters in the
button label and URI-forbidden characters in the path! The backslash
character, "\", escapes characters, including the button-delimitting "=" and
";". There are defaults, see DEFAULT_BUTTONS.

Here is an example of changing the button labels:

  /BUTTON="Selector;Help=/hyperspi/-/hyperspi.html"

Always have the equivalent of "Close" for the first button!  Additional buttons
may be created by adding "label=path;" elements to the button string.  In this
way an additional information page could be referenced as follows:

  /BUTTON="Selector;Help=/hyperspi/-/hyperspi.html;Other VMS=/vms/"

DIFFICULTY FITTING ALL THESE QUALIFIERS INTO A COMMAND LINE OR LOGICAL?
Use an existing, or create a new, DCL wrapper procedure for the script (same
name and directory) and build up a DCL symbol to provide the information. Up
to 1024 characters can be included in this way. For example:

  $ HYPERSPI$PARAM = "/BUTTON=""Close;Help=/hyperspi/-/hyperspihelp.html"""
  $ HYPERSPI$PARAM = HYPERSPI$PARAM + "/PBGCOLOR/PLINK/PVLINK"
  $ RUN HT_EXE:HYPERSPI


CGILIB
------
Using CGILIB should be suitable for WASD, OSU (DECthreads), VMS Apache (CSWS
V1.0-1 and later), Purveyor, possibly others.


CGI VARIABLES
-------------
FORM_BUFFERED_IO        if non-null then buffered IO requested
FORM_CPU        if non-null then cpu usage requested
FORM_MODES      if non-null then cpu modes usage requested
FORM_COMPUTABLE if non-null then COM+COMO process requested
FORM_LOCKS      if non-null then lock usage requested
FORM_DO         what to do(!): "DUMP", "GRAPH", "LIST", "PAGE", "MENU"
FORM_DAY        the (first) day of the data file(s)
FORM_DIRECT_IO  if non-null then direct IO requested
FORM_EXTRACT    if non-null provide a link to allow graph extraction
FORM_HOUR       the (first) hour of the data
FORM_INCLUDE    additional category: PEAK, TOTAL (only with "user-mode-cpu"),
                                     HARD-FAULTS (only with "faults")
FORM_LIST_NODE  name of node to process data (see also FORM_NODE)
FORM_MINUTE     the (first) minute of the data
FORM_MONTH      the (first) month of the data file(s)
FORM_MSCP_IO    if non-null then MSCP IO requested
FORM_NODE       name of node to process data (used before FORM_LIST_NODE)
FORM_PERIOD     standard period (e.g. "until" now, "business", "today",
                "yesterday", or since a number of days)
FORM_TODAY      the last day of the data file(s)
FORM_TOMINUTE   the last minute of the data
FORM_TOHOUR     the last hour of the data
FORM_TOMONTH    the last month of the data file(s)
FORM_TOYEAR     the last year of the data file(s)
FORM_USER_MODE_CPU      if non-null then user-mode-cpu usage requested
FORM_WHAT       any data category can be placed in this comma-separated list
FORM_XMAG       number of times X axis should be enlarged
FORM_YEAR       the (first) year of the data file(s)
FORM_YMAG       number of times Y axis should be enlarged
FORM_REFRESH    Copy today file  to the directory HyperSpiRefreshDataDirectory


QUALIFIERS
----------
/BUTTONS=       string containing button labels/paths
/CHARSET=       "Content-Type: text/html; charset=...", empty suppresses charset
/DBUG           turns on all "if (Debug)" statements
/DIRECTORY=     directory containing any node description HTML files
/HELP=          help HTML file URL
/NODES=         comma-separated list of node names collecting SPI data
/PBACKGROUND=   <body> background image path
/PBGCOLOR=      <body> background colour
/PBBGCOLOR=     button background color
/PBBORDER=      width of button border
/PHBGCOLOR=     heading background color
/PHBORDER=      width of heading and button-bar border
/PHLOCAL=       **** NOT USED BY THIS SCRIPT ***
/PHTEXT=        heading text colour
/PLAYOUT=       1 is coloured header & buttons, 2 is text & horizontal rules
/PLINK=         <body> link colour
/PTEXT=         <body> text colour
/PVLINK=        <body> visited link colour


LOGICAL NAMES
-------------
HYPERSPI$DATA          locates the data files
HYPERSPI$PARAM         equivalent to (overrides) the command line
                       parameters/qualifiers (define as a system-wide logical)


BUILD DETAILS
-------------
See BUILD_HYPERSPIXX.COM procedure.


COPYRIGHT
---------
hyperSPI++ Copyright (c) 2001 Jean-Franois Pironne  (jfp@altavista.net)
hyperSPI   Copyright (c) 1996-2001 Mark G.Daniel  (mark.daniel@wasd.vsm.com.au)

This program, comes with ABSOLUTELY NO WARRANTY.
This is free software, and you are welcome to redistribute it
under the conditions of the GNU GENERAL PUBLIC LICENSE, version 2.


VERSION HISTORY (update SoftwareID as well!)
---------------
30-MAR-2001  JFP  v2.1.0, updated for later version of GD graphics library
08-JAN-2001  MGD  v2.0.1, interim release
03-NOV-2000  JFP  v2.0.0, hyperSPI++ evolution from WASD hyperSPI,
                          it is dependent on the GD graphics library
12-APR-2000  MGD  v1.6.1, minor changes for CGILIB 1.4
07-AUG-1999  MGD  v1.6.0, use more of the CGILIB functionality
24-APR-1999  MGD  v1.5.0, use CGILIB.C,
                          standard CGI environment (Netscape FastTrack),
                          modify to use CgiVar()
02-OCT-1998  MGD  v1.4.0, provide content-type "; charset=..."
13-AUG-1998  MGD  v1.3.0, accomodations for OSU
                          bugfix; IncludeFile() file name length
13-AUG-1998  MGD  NOTE:   this application is not Y2K compliant (in the sense
                          data is stored in files named with a 2 digit year!)
06-MAY-1998  MGD  v1.2.0, general maintenance, cosmetic changes
11-SEP-1997  MGD  v1.1.1, upped MAX_NODE_COUNT_BEFORE_SELECT to 20
01-AUG-1997  MGD  v1.1.0, added /BODY= qualifier,
                          remove extraneous '/' from <IMG SRC=...>,
                          general maintenance
19-SEP-1995  MGD  v1.0.1, replace <CR><LF> carriage-control single newline,
                          still acceptable for HTTP, slightly more efficient
20-JUN-1995  MGD  v1.0.0, initial development
*/
/*****************************************************************************/

#ifdef __ALPHA
#  define SOFTWAREID "HYPERSPIXX AXP-2.1.0"
#else
#  define SOFTWAREID "HYPERSPIXX VAX-2.1.0"
#endif

/* standard C header files */
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <errno.h>
#include <unixio.h>
#include <string.h>
#include <time.h>
#include <assert.h>
#include <unistd.h>

/* VMS related header files */
#include <descrip.h>
#include <libdef.h>
#include <libdtdef.h>
#include <rmsdef.h>
#include <rms.h>
#include <ssdef.h>
#include <stsdef.h>
#include <starlet.h>
#include <lib$routines.h>
#include <clidef.h>

#define NEWLIB_GDCHART

#include <limits.h>
#define MAXINT INT_MAX
#define MAXSHORT SHRT_MAX

#include "gdc.h"
#include "gdchart.h"
#include "gdcpie.h"

/* application header file */
#include "HyperSpi.h"
#include <cgilib.h>

#ifdef __ALPHA
#   pragma nomember_alignment
#endif

#define boolean int
#define true 1
#define false 0
 
#define VMSok(x) ((x) & STS$M_SUCCESS)
#define VMSnok(x) !(((x) & STS$M_SUCCESS))
 
#define FI_LI __FILE__, __LINE__

#define MAX_SAMPLE_PERIOD_DAYS 31
#define MAX_NODE_COUNT_BEFORE_SELECT 20

#define DEFAULT_BUTTONS "Selector;Help=/hyperspi/-/hyperspi.html"

/* this macro just plugs in some script-specific code into ButtonBar() */
#define SCRIPT_SPECIFIC_BUTTON_CODE \
   ButtonInternal[0] = SelectorPtr;

#define DEFAULT_PS_BGCOLOR        "#ffffff"
#define DEFAULT_PS_TEXT           "#000000"
#define DEFAULT_PS_LINK           "#0000ff"
#define DEFAULT_PS_VLINK          "#0000ff"
#define DEFAULT_PS_HEADBGCOLOR    "#cccccc"
#define DEFAULT_PS_HEADBORDER     "0"
#define DEFAULT_PS_HEADTEXT       "#000000"
#define DEFAULT_PS_BUTTONBGCOLOR  "#ffffff"
#define DEFAULT_PS_BUTTONBORDER   "1"

#define PS_BACKGROUND     0
#define PS_BGCOLOR        1
#define PS_TEXT           2
#define PS_LINK           3
#define PS_VLINK          4
#define PS_HEADBGCOLOR    5
#define PS_HEADTEXT       6
#define PS_HEADBORDER     7
#define PS_BUTTONBGCOLOR  8
#define PS_BUTTONBORDER   9
#define PS_BODYTAG       10
#define PS_LAYOUT        11
#define PS_HEADLOCAL     12
#define PS_HEADPADDING   13

#define LONG_LABEL	  0

char  *PageScheme [16];

char  Utility [] = "HYPERSPIXX";

char  *DayName [] =
      { "", "Sunday", "Monday", "Tuesday", "Wednesday",
            "Thursday", "Friday", "Saturday", };

char  *MonthName [] =
      { "", "January", "February", "March", "April", "May", "June",
            "July", "August", "September", "October", "November", "December" };

boolean  Debug,
         DoDump,
         DoGraph,
         DoList,
         DoComprehensiveMenu,
         DoSummaryMenu,
         DoPresentSummaryPage,
         ErrorReported,
         ExtractGraph,
         GifTransparentBackground = true,
         HttpHasBeenOutput,
         IncludePeak,
         IncludeTotal,
         ProvidePercentModeCPU,
         ProvidePercentModeCPUPie,
         ProvidePercentCPU,
         ProvideBufferedIO,
         ProvidePeakBufferedIO,
         ProvideDirectIO,
         ProvidePeakDirectIO,
         ProvideMemory,
         ProvideMscpIO,
         ProvidePeakMscpIO,
         ProvidePageFaults,
         ProvidePeakPageFaults,
         ProvideHardPageFaults,
         ProvidePeakHardPageFaults,
         ProvideSoftPageFaults,
         ProvidePeakSoftPageFaults,
         ProvideProcesses,
	 ProvideCom,
         ProvideLocks,
         ProvideLocksPie,
	 RefreshTodayData,
         OsuEnvironment,
         StdCgiEnvironment;

int  AvePercentCPU,
     AvePercentModeCPU[K_HyperSpiCpuModeCount],
     AveBufferedIO,
     AveDirectIO,
     AveMscpIO,
     AvePageFaults,
     AveHardPageFaults,
     AveNumberOfProcesses,
     AveNumberOfCom,
     AveLck,
     AveLckLoc,
     AveLckIn,
     AveLckOut,
     AveSystemMemoryPercentInUse,
     AvePageSpacePercentInUse,
     CurrentJulianDate,
     CurrentMinuteFromStartOfPeriod,
     DataFileNameLength,
     DataFileSpecLength,
     DataFilesFoundCount,
     DataFilesProcessedCount,
     DataRecordsReadCount,
     DataRecordsProcessedCount,
     DataMinute,
     DataHour,
     DataDay,
     DataMonth,
     DataYear,
     FromJulianDate,
     FromDay,
     FromHour,
     FromMinute,
     FromMonth,
     FromYear,
     HaveDataY,
     MaxAveBufferedIO,
     MaxAveDirectIO,
     MaxAveCom,
     MaxAveMscpIO,
     MaxAvePageFaults,
     MaxAveHardPageFaults,
     MaxNumberOfProcesses,
     MaxNumberOfCom,
     MaxPageSpacePercentInUse,
     MaxSystemMemoryPercentInUse,
     NodeCount,
     NumberOfCPUs,
     NumberOfDays,
     NumberOfDaysIntoData,
     NumberOfHours,
     NumberOfMinutesIntoData,
     NumberOfProcesses,
     NumberOfCom,
     PageSpaceMBytes,
     PageSpacePercentInUse,
     PeakBufferedIO,
     PeakDirectIO,
     PeakMscpIO,
     PeakPageFaults,
     PeakPercentCPU,
     PeakPercentModeCPU[K_HyperSpiCpuModeCount],
     PeakHardPageFaults,
     PercentCPU,
     PercentModeCPU[K_HyperSpiCpuModeCount],
     RecordSampleRate,
     SizeOfMarginX = 10,
     SizeOfMarginY = 20,
     SizeOfPlotX,
     SizeOfPlotY,
     StartMinuteOnFirstDay,
     StartMinuteOfData,
     SystemMemoryMBytes,
     SystemMemoryPercentInUse,
     ToJulianDate,
     ToMinute,
     ToHour,
     ToDay,
     ToMonth,
     ToYear,
     XMag,
     YMag;
     
char  DefaultButton[] = DEFAULT_BUTTONS,
      *ButtonPtr = DefaultButton,
      *CgiEnvironmentPtr,
      *CgiFormDoPtr,
      *CgiFormIncludePtr,
      *CgiFormNodePtr,
      *CgiFormPeriodPtr,
      *CgiFormWhatPtr,
      *CgiRequestMethodPtr,
      *CgiPathInfoPtr,
      *CgiPathTranslatedPtr,
      *CgiScriptNamePtr,
      *CgiServerNamePtr,
      *CgiServerSoftwarePtr,
      *CliCharsetPtr,
      *HyperSpiDirectoryPtr = "HYPERSPI$DATA:",
      *SelectorPtr;

char  DataFileName [256],
      DataFileSpec [256],
      DataNode [32],
      DateString [32],
      FileSpecification [256],
      HyperSpiUrl [256],
      SoftwareID [48],
      ToDateString [32],
      UnixDateTime [64];

char  SpiNodeNames [256][16] = { "" };

unsigned long  CurrentBinTime [2];
unsigned short  CurrentNumTime [7];

/* this structure is declared in HyperSpi.h */
struct HyperSpiData  SpiRecord;

int  PixelPlotOperation;

static float* Values[6];
static char** Labels;
static GDC_CHART_T ChartType;
static int GraphLinesCnt;

/* required function prototypes */
void GetParameters();
int GetPageParameter();
void SetPageScheme();
void GetRequest();
void ValidateRequest();
void SummaryMenu();
void PresentSummaryPage();
void DumpRecord();
void DumpData();
void GraphData();
void ListProcessedData();
void ListProcessedRecord();
void GraphRecordCPU();
void GraphRecordCPUModes();
void GraphRecordLocks();
void GraphRecordMemory();
void GraphRecordProcesses();
void GraphRecordCom();
void GraphRecordRange();
void SelectNodeNameByDataFileName();
void SummarizeRecord();
void SummarizeData();
char* UniqueNumber();
int IncludeFile(char*);
int ProcessDataFiles();
void PresentCPU();
void PresentCPUModes();
void PresentLocks();
void PresentMemory();
void PresentProcesses();
void PresentCom();
void PresentPageFaults();
void PresentPeakPageFaults();
void PresentHardPageFaults();
void PresentPeakHardPageFaults();
void PresentBufferedIO();
void PresentPeakBufferedIO();
void PresentDirectIO();
void PresentPeakDirectIO();
void PresentMscpIO();
void PresentPeakMscpIO();
void SelectNodeNames ();
void ProcessDataFileRecords();
boolean strsame(char*, char*, int);

int out_graph( short, short, FILE*, GDC_CHART_T, int, char*[], int, ...);

/*****************************************************************************/
/*
'argc/argv' are only required to support OSU, in particular CgiLibOsuInit().
*/

main
(
int argc,
char *argv[]
)
{
   int  status;
   unsigned long  UnixTime;
   struct tm  *UnixTmPtr;

   /*********/
   /* begin */
   /*********/

   sprintf (SoftwareID, "%s (%s)", SOFTWAREID, CgiLibEnvironmentVersion());

   if (getenv ("HYPERSPIXX$DBUG") != NULL) Debug = true;
   CgiLibEnvironmentSetDebug (Debug);

   CgiLibEnvironmentInit (argc, argv, false);

   GetParameters ();

   CgiLibResponseSetBody (PageScheme[PS_BODYTAG]);
   CgiLibResponseSetCharset (CliCharsetPtr);
   CgiLibResponseSetSoftwareID (SoftwareID);
   CgiLibResponseSetErrorMessage ("Reported by HyperSPI++");

   CgiEnvironmentPtr = CgiLibEnvironmentName ();

   SetPageScheme ();

   CgiServerSoftwarePtr = CgiVar ("WWW_SERVER_SOFTWARE");
   CgiRequestMethodPtr = CgiVar ("WWW_REQUEST_METHOD");
   if (strcmp (CgiRequestMethodPtr, "GET"))
   {
      CgiLibResponseHeader (501, "text/html");
      fprintf (stdout, "Not implemented!\n");
      return 1;
   }

   CgiScriptNamePtr = CgiVar ("WWW_SCRIPT_NAME");
   CgiServerNamePtr = CgiVar ("WWW_SERVER_NAME");

   sys$gettim (&CurrentBinTime);
   sys$numtim (&CurrentNumTime, &CurrentBinTime);

   time (&UnixTime);
   UnixTmPtr = localtime (&UnixTime);
   if (!strftime (UnixDateTime, sizeof(UnixDateTime),
                  "%a, %d %b %Y %T", UnixTmPtr))
      strcpy (UnixDateTime, "[error]");
   if (Debug) fprintf (stdout, "UnixDateTime |%s|\n", UnixDateTime);

   GetRequest ();
   ValidateRequest ();

   if (DoSummaryMenu || DoComprehensiveMenu)
   {
      SummaryMenu ();
      exit (SS$_NORMAL);
   }

   /* create file specification for processing data */
   DataFileSpecLength =
         sprintf (DataFileSpec,
                  "%sHYPERSPI_%s_%s_%%%%%%%%%%%%.DAT",
                  HyperSpiDataDirectory, HyperSpiDataVersion, CgiFormNodePtr);
   if (Debug) fprintf (stdout, "DataFileSpec |%s|\n", DataFileSpec);

   if (DoPresentSummaryPage)
      PresentSummaryPage ();
   else   
   if (DoDump)
      DumpData ();
   else   
   if (DoGraph)
      GraphData ();
   else   
   if (DoList)
      ListProcessedData ();

   exit (SS$_NORMAL);
}

/*****************************************************************************/
/*
Get "command-line" parameters, whether from the command-line or from a
configuration logical containing the equivalent.
*/

void GetParameters () {
   static char  CommandLine [256];
   static unsigned long  Flags = 0;

   register int  idx;
   register char  *aptr, *cptr, *clptr, *sptr;

   int  status;
   unsigned short  Length;
   char  ch;
   $DESCRIPTOR (CommandLineDsc, CommandLine);

   /*********/
   /* begin */
   /*********/

   if ((clptr = getenv ("HYPERSPI$PARAM")) == NULL)
   {
      /* get the entire command line following the verb */
      if (VMSnok (status =
          lib$get_foreign (&CommandLineDsc, 0, &Length, &Flags)))
         exit (status);
      (clptr = CommandLine)[Length] = '\0';
   }

   aptr = NULL;
   ch = *clptr;
   for (;;)
   {
      if (aptr != NULL) *aptr = '\0';
      if (!ch) break;

      *clptr = ch;
      if (Debug) fprintf (stdout, "clptr |%s|\n", clptr);
      while (*clptr && isspace(*clptr)) *clptr++ = '\0';
      aptr = clptr;
      if (*clptr == '/') clptr++;
      while (*clptr && !isspace (*clptr) && *clptr != '/')
      {
         if (*clptr != '\"')
         {
            clptr++;
            continue;
         }
         cptr = clptr;
         clptr++;
         while (*clptr)
         {
            if (*clptr == '\"')
               if (*(clptr+1) == '\"')
                  clptr++;
               else
                  break;
            *cptr++ = *clptr++;
         }
         *cptr = '\0';
         if (*clptr) clptr++;
      }
      ch = *clptr;
      if (*clptr) *clptr = '\0';
      if (Debug) fprintf (stdout, "aptr |%s|\n", aptr);
      if (!*aptr) continue;

      if (strsame (aptr, "/BUTTONS=", 4))
      {
         for (cptr = aptr; *cptr && *cptr != '='; cptr++);
         if (*cptr) cptr++;
         ButtonPtr = cptr;
         continue;
      }
      if (strsame (aptr, "/CHARSET=", 4))
      {
         for (cptr = aptr; *cptr && *cptr != '='; cptr++);
         if (*cptr) cptr++;
         CliCharsetPtr = cptr;
         continue;
      }
      if (strsame (aptr, "/DBUG", -1))
      {
         Debug = true;
         continue;
      }
      if (strsame (aptr, "/DIRECTORY=", 3))
      {
         for (cptr = aptr; *cptr && *cptr != '='; cptr++);
         if (*cptr) cptr++;
         HyperSpiDirectoryPtr = cptr;
         continue;
      }
      if (strsame (aptr, "/NODES=", 3))
      {
         for (cptr = aptr; *cptr && *cptr != '='; cptr++);
         if (*cptr) cptr++;
         /* quick and nasty, no checks on array bounds (caveat emptor) */
         idx = 0;
         while (*cptr)
         {
            sptr = SpiNodeNames[idx++];
            while (*cptr && *cptr != ',') *sptr++ = toupper(*cptr++);
            *sptr = '\0';
            if (*cptr) cptr++;
         }
         SpiNodeNames[idx][0] = '\0';
         continue;
      }
      if (GetPageParameter (aptr)) continue;

      if (*aptr != '/')
      {
         fprintf (stdout, "%%%s-E-MAXPARM, too many parameters\n \\%s\\\n",
                  Utility, aptr);
         exit (STS$K_ERROR | STS$M_INHIB_MSG);
      }
      else
      {
         fprintf (stdout, "%%%s-E-IVQUAL, unrecognized qualifier\n \\%s\\\n",
                  Utility, aptr+1);
         exit (STS$K_ERROR | STS$M_INHIB_MSG);
      }
   }
}

/*****************************************************************************/
/*
Get command-line parameters associated with page scheme.
*/

boolean GetPageParameter (char *aptr)

{
   register char  *cptr;

   /*********/
   /* begin */
   /*********/

   if (strsame (aptr, "/PBACKGROUND=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_BACKGROUND] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PBGCOLOR=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_BGCOLOR] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PBBGCOLOR=", 5))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_BUTTONBGCOLOR] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PBBORDER=", 5))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_BUTTONBORDER] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PHBGCOLOR=", 5))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_HEADBGCOLOR] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PHBORDER=", 5))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_HEADBORDER] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PHTEXT=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_HEADTEXT] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PLAYOUT=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_LAYOUT] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PLINK=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_LINK] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PHLOCAL=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_HEADLOCAL] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PTEXT=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_TEXT] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PVLINK=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_VLINK] = cptr;
      return (true);
   }
   return (false);
}

/*****************************************************************************/
/*
Set the page layout and colouration.
*/

void SetPageScheme () {
   int  size;
   char  *sptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SetPageScheme()\n");

   if (PageScheme[PS_LAYOUT] == NULL)
      PageScheme[PS_LAYOUT] = "1";

   if (PageScheme[PS_BACKGROUND] == NULL)
      PageScheme[PS_BACKGROUND] = "";

   if (PageScheme[PS_HEADLOCAL] == NULL)
      PageScheme[PS_HEADLOCAL] = "";

   if (PageScheme[PS_LAYOUT][0] == '2')
   {
      if (PageScheme[PS_BGCOLOR] == NULL) PageScheme[PS_BGCOLOR] = "";
      if (PageScheme[PS_TEXT] == NULL) PageScheme[PS_TEXT] = "";
      if (PageScheme[PS_LINK] == NULL) PageScheme[PS_LINK] = "";
      if (PageScheme[PS_VLINK] == NULL) PageScheme[PS_VLINK] = "";
      if (PageScheme[PS_HEADBGCOLOR] == NULL) PageScheme[PS_HEADBGCOLOR] = "";
      if (PageScheme[PS_HEADBORDER] == NULL) PageScheme[PS_HEADBORDER] = "";
      if (PageScheme[PS_HEADTEXT] == NULL) PageScheme[PS_HEADTEXT] = "";
      if (PageScheme[PS_BUTTONBGCOLOR] == NULL) PageScheme[PS_BUTTONBGCOLOR] = "";
      if (PageScheme[PS_BUTTONBORDER] == NULL) PageScheme[PS_BUTTONBORDER] = "";
   }
   else
   {
      if (PageScheme[PS_BGCOLOR] == NULL)
         PageScheme[PS_BGCOLOR] = DEFAULT_PS_BGCOLOR;
      if (PageScheme[PS_TEXT] == NULL)
         PageScheme[PS_TEXT] = DEFAULT_PS_TEXT;
      if (PageScheme[PS_LINK] == NULL)
         PageScheme[PS_LINK] = DEFAULT_PS_LINK;
      if (PageScheme[PS_VLINK] == NULL)
         PageScheme[PS_VLINK] = DEFAULT_PS_VLINK;
      if (PageScheme[PS_HEADBGCOLOR] == NULL)
         PageScheme[PS_HEADBGCOLOR] = DEFAULT_PS_HEADBGCOLOR;
      if (PageScheme[PS_HEADBORDER] == NULL)
         PageScheme[PS_HEADBORDER] = DEFAULT_PS_HEADBORDER;
      if (PageScheme[PS_HEADTEXT] == NULL)
         PageScheme[PS_HEADTEXT] = DEFAULT_PS_HEADTEXT;
      if (PageScheme[PS_BUTTONBGCOLOR] == NULL)
         PageScheme[PS_BUTTONBGCOLOR] = DEFAULT_PS_BUTTONBGCOLOR;
      if (PageScheme[PS_BUTTONBORDER] == NULL)
         PageScheme[PS_BUTTONBORDER] = DEFAULT_PS_BUTTONBORDER;
   }

   /* <BODY> tag attributes */
   size = strlen(PageScheme[PS_BACKGROUND]) +
          strlen(PageScheme[PS_BGCOLOR]) +
          strlen(PageScheme[PS_TEXT]) +
          strlen(PageScheme[PS_LINK]) +
          strlen(PageScheme[PS_VLINK]);
   if (size)
   {
      if ((sptr = calloc (1, size+64)) == NULL) exit (vaxc$errno);
      PageScheme[PS_BODYTAG] = sptr;
      if (PageScheme[PS_BACKGROUND][0])
         sptr += sprintf (sptr, " BACKGROUND=\"%s\"", PageScheme[PS_BACKGROUND]);
      if (PageScheme[PS_BGCOLOR][0])
         sptr += sprintf (sptr, " BGCOLOR=\"%s\"", PageScheme[PS_BGCOLOR]);
      if (PageScheme[PS_TEXT][0])
         sptr += sprintf (sptr, " TEXT=\"%s\"", PageScheme[PS_TEXT]);
      if (PageScheme[PS_LINK][0])
         sptr += sprintf (sptr, " LINK=\"%s\"", PageScheme[PS_LINK]);
      if (PageScheme[PS_VLINK][0])
         sptr += sprintf (sptr, " VLINK=\"%s\"", PageScheme[PS_VLINK]);
   }
   else
      PageScheme[PS_BODYTAG] = "";

   if (PageScheme[PS_HEADBGCOLOR][0])
   {
      if ((sptr = calloc (1, strlen(PageScheme[PS_HEADBGCOLOR])+16)) == NULL)
         exit (vaxc$errno);
      sprintf (sptr, " BGCOLOR=\"%s\"", PageScheme[PS_HEADBGCOLOR]);
      PageScheme[PS_HEADBGCOLOR] = sptr;
      PageScheme[PS_HEADPADDING] = "10";
   }
   else
      PageScheme[PS_HEADPADDING] = "0";

   if (PageScheme[PS_BUTTONBGCOLOR][0])
   {
      if ((sptr = calloc (1, strlen(PageScheme[PS_BUTTONBGCOLOR])+16)) == NULL)
         exit (vaxc$errno);
      sprintf (sptr, " BGCOLOR=\"%s\"", PageScheme[PS_BUTTONBGCOLOR]);
      PageScheme[PS_BUTTONBGCOLOR] = sptr;
   }
}

/*****************************************************************************/
/*
Provides a divider for top and bottom of the content of the page. This can be
a coloured bar (using <TABLE>) or a horizontal rule depending on the page
layout. "Buttons" providing script-internal and/or additional user-specified
links ('ButtonPtr' string) can be placed with(in) this bar. All button labels
are derived from 'ButtonPtr', with script-internal buttons using link-paths
set up via 'ButtonInternal[]' array, or any user-specified path depending on
requirement. An empty path (i.e. PathPtr[0] == '\0') obviously does not have a
link created, it just displays the button label. For a button-bar at the top
of the document use 1, bottom of the document use 2, and for just a bar with
no buttons at all use 0.
*/

void ButtonBar (int Top1Bottom2)

{
#define MAX_BUTTON_COUNT 8

   static int  ButtonCount = -1;
   static char  *ButtonInternal [MAX_BUTTON_COUNT],
                *ButtonLabel [MAX_BUTTON_COUNT],
                *ButtonPath [MAX_BUTTON_COUNT];

   int  idx;
   char  *PathPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ButtonBar() %d\n", Top1Bottom2);

   if (ButtonCount == -1)
   {
      register char  *cptr, *sptr;

      if (Debug) fprintf (stdout, "|%s|\n", ButtonPtr);
      cptr = ButtonPtr;
      for (ButtonCount = 0;
           ButtonCount < MAX_BUTTON_COUNT && *cptr;
           ButtonCount++)
      {
         for (sptr = cptr; *sptr && *sptr != '=' && *sptr != ';'; sptr++)
            if (*sptr == '\\') memcpy (sptr, sptr+1, strlen(sptr));
         if (*sptr == '=') *sptr++ = '\0';
         ButtonLabel[ButtonCount] = cptr;
         cptr = sptr;
         for (sptr = cptr; *sptr && *sptr != ';'; sptr++)
            if (*sptr == '\\') memcpy (sptr, sptr+1, strlen(sptr));
         if (*sptr) *sptr++ = '\0';

         ButtonPath[ButtonCount] = cptr;
         cptr = sptr;
      }
   }

   if (Top1Bottom2)
   {
      /***********************************/
      /* set up script-specified buttons */
      /***********************************/

      SCRIPT_SPECIFIC_BUTTON_CODE
   }

   if (PageScheme[PS_LAYOUT][0] == '2')
   {
      /************/
      /* format 2 */
      /************/

      if (Top1Bottom2 == 2 || !Top1Bottom2)
      {
         fprintf (stdout, "<HR ALIGN=left SIZE=2 WIDTH=95%%>\n");
         if (!Top1Bottom2) return;
      }

      fprintf (stdout, "<FONT SIZE=-1><NOBR>\n");
      for (idx = 0; idx < ButtonCount; idx++)
      {
         if (ButtonInternal[idx] == NULL)
            PathPtr = ButtonPath[idx];
         else
            PathPtr = ButtonInternal[idx];
         if (idx) fprintf (stdout, "&nbsp;");
         if (PathPtr[0])
            fprintf (stdout, "[<A HREF=\"%s\">%s</A>]\n",
                     PathPtr, ButtonLabel[idx]);
         else
            fprintf (stdout, "[%s]\n", ButtonLabel[idx]);
      }
      fprintf (stdout, "</NOBR></FONT>\n");

      if (Top1Bottom2 == 1)
         fprintf (stdout, "<HR ALIGN=left SIZE=2 WIDTH=95%%>\n");
   }
   else
   {
      /************/
      /* format 1 */
      /************/

      fprintf (stdout,
"<TABLE BORDER=0 CELLPADDING=0 CELLSPACING=0>\
<TR><TD HEIGHT=2></TD></TR>\
</TABLE>\n\
<TABLE BORDER=%s CELLPADDING=%s CELLSPACING=0 WIDTH=100%%>\n\
<TR><TD%s>\n",
         PageScheme[PS_HEADBORDER],
         PageScheme[PS_HEADBORDER],
         PageScheme[PS_HEADBGCOLOR]);

      if (ButtonCount == 0 || !Top1Bottom2)
         fprintf (stdout, "&nbsp;\n");
      else
      {
         fprintf (stdout,
"<TABLE BORDER=%s CELLPADDING=1 CELLSPACING=0>\n",
            PageScheme[PS_BUTTONBORDER]);

         for (idx = 0; idx < ButtonCount; idx++)
         {
            if (ButtonInternal[idx] == NULL)
               PathPtr = ButtonPath[idx];
            else
               PathPtr = ButtonInternal[idx];
            if (PathPtr[0])
               fprintf (stdout,
"<TD ALIGN=center%s><FONT SIZE=-1>\
<NOBR>&nbsp;&nbsp;<A HREF=\"%s\">%s</A>&nbsp;&nbsp;</NOBR></FONT></TD>\n",
                  PageScheme[PS_BUTTONBGCOLOR], PathPtr, ButtonLabel[idx]);
            else
               fprintf (stdout,
"<TD ALIGN=center%s><FONT SIZE=-1>\
<NOBR>&nbsp;&nbsp;%s&nbsp;&nbsp;</NOBR></FONT></TD>\n",
                  PageScheme[PS_BUTTONBGCOLOR], ButtonLabel[idx]);
         }

         fprintf (stdout, "</TR></TABLE>\n");
      }

      fprintf (stdout,
"</TD></TR>\n\
</TABLE>\n\
<TABLE BORDER=0 CELLPADDING=0 CELLSPACING=0>\
<TR><TD HEIGHT=2></TD></TR>\
</TABLE>\n");
   }
}

/*****************************************************************************/
/*
Get the request CGI variables.
*/

void GetRequest () {
   register char  *cptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "GetRequest()\n");

   /* determine what has been requested */

   CgiFormWhatPtr = CgiVar ("WWW_FORM_WHAT");
   for (cptr = CgiFormWhatPtr; *cptr; cptr++) *cptr = toupper(*cptr);


   if (strsame (CgiFormWhatPtr, "CPU", -1)) ProvidePercentCPU = true;
   if (strstr (CgiFormWhatPtr, ",CPU") != NULL) ProvidePercentCPU = true;
   cptr = CgiVar ("WWW_FORM_CPU");
   if (cptr[0]) ProvidePercentCPU = true;

   if (strstr (CgiFormWhatPtr, "MODES") != NULL) ProvidePercentModeCPU = true;
   cptr = CgiVar ("WWW_FORM_MODES");
   if (cptr[0]) ProvidePercentModeCPU = true;

   if (strstr (CgiFormWhatPtr, "MODES_PIE") != NULL)
      ProvidePercentModeCPUPie = true;
   cptr = CgiVar ("WWW_FORM_MODES_PIE");
   if (cptr[0]) ProvidePercentModeCPUPie = true;

   if (strstr (CgiFormWhatPtr, "LOCKS") != NULL) ProvideLocks = true;
   cptr = CgiVar ("WWW_FORM_LOCKS");
   if (cptr[0]) ProvideLocks = true;

   if (strstr (CgiFormWhatPtr, "LOCKS_PIE") != NULL) ProvideLocksPie = true;
   cptr = CgiVar ("WWW_FORM_LOCKS_PIE");
   if (cptr[0]) ProvideLocksPie = true;

   if (strstr (CgiFormWhatPtr, "BUFFERED") != NULL) ProvideBufferedIO = true;
   cptr = CgiVar ("WWW_FORM_BUFFERED_IO");
   if (cptr[0]) ProvideBufferedIO = true;

   if (strstr (CgiFormWhatPtr, "PEAK_BUFFERED") != NULL)
      ProvidePeakBufferedIO = true;
   cptr = CgiVar ("WWW_FORM_PEAK_BUFFERED_IO");
   if (cptr[0]) ProvidePeakBufferedIO = true;

   if (strstr (CgiFormWhatPtr, "DIRECT") != NULL) ProvideDirectIO = true;
   cptr = CgiVar ("WWW_FORM_DIRECT_IO");
   if (cptr[0]) ProvideDirectIO = true;

   if (strstr (CgiFormWhatPtr, "PEAK_DIRECT") != NULL)
      ProvidePeakDirectIO = true;
   cptr = CgiVar ("WWW_FORM_PEAK_DIRECT_IO");
   if (cptr[0]) ProvidePeakDirectIO = true;

   if (strstr (CgiFormWhatPtr, "MSCP") != NULL) ProvideMscpIO = true;
   cptr = CgiVar ("WWW_FORM_MSCP_IO");
   if (cptr[0]) ProvideMscpIO = true;

   if (strstr (CgiFormWhatPtr, "PEAK_MSCP") != NULL) ProvidePeakMscpIO = true;
   cptr = CgiVar ("WWW_FORM_PEAK_MSCP_IO");
   if (cptr[0]) ProvidePeakMscpIO = true;

   if (strstr (CgiFormWhatPtr, "FAULTS") != NULL) ProvidePageFaults = true;
   cptr = CgiVar ("WWW_FORM_FAULTS");
   if (cptr[0]) ProvidePageFaults = true;

   if (strstr (CgiFormWhatPtr, "PEAK_FAULTS") != NULL)
      ProvidePeakPageFaults = true;
   cptr = CgiVar ("WWW_FORM_PEAK_FAULTS");
   if (cptr[0]) ProvidePeakPageFaults = true;

   if (strstr (CgiFormWhatPtr, "HARD_FAULTS") != NULL)
      ProvideHardPageFaults = true;
   cptr = CgiVar ("WWW_FORM_HARD_FAULTS");
   if (cptr[0]) ProvideHardPageFaults = true;

   if (strstr (CgiFormWhatPtr, "PEAK_HARD_FAULTS") != NULL)
      ProvidePeakHardPageFaults = true;
   cptr = CgiVar ("WWW_FORM_PEAK_HARD_FAULTS");
   if (cptr[0]) ProvidePeakHardPageFaults = true;

   if (strstr (CgiFormWhatPtr, "SOFT_FAULTS") != NULL)
      ProvideSoftPageFaults = true;
   cptr = CgiVar ("WWW_FORM_SOFT_FAULTS");
   if (cptr[0]) ProvideSoftPageFaults = true;

   if (strstr (CgiFormWhatPtr, "PEAK_SOFT_FAULTS") != NULL)
      ProvidePeakSoftPageFaults = true;
   cptr = CgiVar ("WWW_FORM_PEAK_SOFT_FAULTS");
   if (cptr[0]) ProvidePeakSoftPageFaults = true;

   if (strstr (CgiFormWhatPtr, "MEMORY") != NULL) ProvideMemory = true;
   cptr = CgiVar ("WWW_FORM_MEMORY");
   if (cptr[0]) ProvideMemory = true;

   if (strstr (CgiFormWhatPtr, "PROCESSES") != NULL) ProvideProcesses = true;
   cptr = CgiVar ("WWW_FORM_PROCESSES");
   if (cptr[0]) ProvideProcesses = true;

   if (strstr (CgiFormWhatPtr, "COMPUTABLE") != NULL) ProvideCom = true;
   cptr = CgiVar ("WWW_FORM_COMPUTABLE");
   if (cptr[0]) ProvideCom = true;

   CgiFormIncludePtr = CgiVar ("WWW_FORM_INCLUDE");
   for (cptr = CgiFormIncludePtr; *cptr; cptr++) *cptr = toupper(*cptr);
   if (strstr (CgiFormIncludePtr, "PEAK") != NULL) IncludePeak = true;
   if (strstr (CgiFormIncludePtr, "TOTAL") != NULL) IncludeTotal = true;

   cptr = CgiVar ("WWW_FORM_FROM");
   sscanf (cptr, "%4d-%2d-%2d %2d:%2d",
           &FromYear, &FromMonth, &FromDay, &FromHour, &FromMinute);

   cptr = CgiVar ("WWW_FORM_TO");
   sscanf (cptr, "%4d-%2d-%2d %2d:%2d",
           &ToYear, &ToMonth, &ToDay, &ToHour, &ToMinute);

   CgiFormDoPtr = CgiVar ("WWW_FORM_DO");

   CgiFormNodePtr = CgiVar ("WWW_FORM_NODE");
   if (!CgiFormNodePtr[0])
      CgiFormNodePtr = CgiVar ("WWW_FORM_LIST_NODE");
   if (CgiFormNodePtr[0])
   {
      /* ensure the node name is in upper case */
      for (cptr = CgiFormNodePtr; *cptr; cptr++) *cptr = toupper(*cptr);
   }

   cptr = CgiVar ("WWW_FORM_EXTRACT");
   if (cptr[0]) ExtractGraph = true;

   CgiFormPeriodPtr = CgiVar ("WWW_FORM_PERIOD");

   cptr = CgiVar ("WWW_FORM_XMAG");
   if (isdigit(cptr[0]))
      XMag = atoi (cptr);
   else
      XMag = 1;
   if (XMag > 4) XMag = 4;
   if (XMag < 1) XMag = 1;

   cptr = CgiVar ("WWW_FORM_YMAG");
   if (isdigit(cptr[0]))
      YMag = atoi (cptr);
   else
      YMag = 1;
   if (YMag > 4) YMag = 4;
   if (YMag < 1) YMag = 1;

   if (strstr (CgiFormWhatPtr, "REFRESH") != NULL) RefreshTodayData = true;
   cptr = CgiVar ("WWW_FORM_REFRESH");
   if (cptr[0]) RefreshTodayData = true;

}

/*****************************************************************************/
/*
Process the request parameters (e.g. time, node name).  Verify the parameters 
are within constraints, particular time.  Process the time components into 
values the application can use.
*/

void ValidateRequest () {
   static long  LibJulianDate = LIB$K_JULIAN_DATE;
   static char  *MonthName [] = 
      { "???", "JAN", "FEB", "MAR", "APR", "MAY", "JUN",
        "JUL", "AUG", "SEP", "OCT", "NOV", "DEC" };

   int  status;
   unsigned long  BinTime [2],
                  DeltaTime [2];
   unsigned short  NumTime [7];
   char  Scratch [256];
   $DESCRIPTOR (TempDsc, "");

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ValidateRequest()\n");

   if (!CgiFormDoPtr[0])
      DoSummaryMenu = true;
   else
   if (toupper(CgiFormDoPtr[0]) == 'G')
      DoGraph = true;
   else
   if (toupper(CgiFormDoPtr[0]) == 'P')
      DoPresentSummaryPage = true;
   else
   if (toupper(CgiFormDoPtr[0]) == 'C')
      DoComprehensiveMenu = true;
   else
   if (toupper(CgiFormDoPtr[0]) == 'M')
      DoSummaryMenu = true;
   else
   if (toupper(CgiFormDoPtr[0]) == 'D')
      DoDump = true;
   else
   if (toupper(CgiFormDoPtr[0]) == 'L')
      DoList = true;
   else
   {
      sprintf (Scratch, "Cannot do \"%s\"", CgiFormDoPtr);
      CgiLibResponseError (FI_LI, 0, Scratch);
      exit (SS$_NORMAL);
   }

   if ((DoPresentSummaryPage || DoGraph || DoList || DoDump) &&
       !CgiFormNodePtr[0])
   {
      CgiLibResponseError (FI_LI, 0, "Node name not specified.");
      exit (SS$_NORMAL);
   }

   /********************/
   /* standard periods */
   /********************/

   if (CgiFormPeriodPtr[0] && toupper(CgiFormPeriodPtr[0]) != 'O')
   {
      /* a standard time period has been specified, other than "other" */
      FromHour = 0;
      FromMinute = 0;
      ToHour = 23;
      ToMinute = 59;
      FromDay = ToDay =  CurrentNumTime[2];
      FromMonth = ToMonth =  CurrentNumTime[1];
      FromYear = ToYear =  CurrentNumTime[0];

      if (toupper(CgiFormPeriodPtr[0]) == 'U')
      {
         /* "until_now" (7am to this hour) */
         FromHour = 7;
         ToHour = CurrentNumTime[3];
      }
      else
      if (toupper(CgiFormPeriodPtr[0]) == 'B')
      {
         /* "business" hours (7am to 7pm) */
         FromHour = 7;
         ToHour = 18;
      }
      else
      if (toupper(CgiFormPeriodPtr[0]) == 'T')
      {
         /* "today" */
         ToHour = 23;
      }
      else
      if (toupper(CgiFormPeriodPtr[0]) == 'S' ||
          toupper(CgiFormPeriodPtr[0]) == 'Y')
      {
         /* "since_yesterday" until now, or all of "yesterday" */
         TempDsc.dsc$a_pointer = "1 00:00:00.00";
         TempDsc.dsc$w_length = 13;
         if (VMSnok (status = sys$bintim (&TempDsc, &DeltaTime)))
         {
            CgiLibResponseError (FI_LI, status, CgiFormPeriodPtr);
            exit (SS$_NORMAL);
         }
         lib$sub_times (&CurrentBinTime, &DeltaTime, &BinTime);
         sys$numtim (&NumTime, &BinTime);
         FromDay = NumTime[2];
         FromMonth = NumTime[1];
         FromYear = NumTime[0];
         if (toupper(CgiFormPeriodPtr[0]) == 'S')
            ToHour = CurrentNumTime[3];
         else
         {
            ToDay = NumTime[2];
            ToMonth = NumTime[1];
            ToYear = NumTime[0];
         }
      }
      else
      if (isdigit(CgiFormPeriodPtr[0]))
      {
         TempDsc.dsc$a_pointer = Scratch;
         TempDsc.dsc$w_length =
            sprintf (Scratch, "%s 00:00:00.00", CgiFormPeriodPtr);
         if (VMSnok (status = sys$bintim (&TempDsc, &DeltaTime)))
         {
            CgiLibResponseError (FI_LI, status, CgiFormPeriodPtr);
            exit (SS$_NORMAL);
         }
         lib$sub_times (&CurrentBinTime, &DeltaTime, &BinTime);
         sys$numtim (&NumTime, &BinTime);
         FromDay = NumTime[2];
         FromMonth = NumTime[1];
         FromYear = NumTime[0];
      }
      else
      {
         CgiLibResponseError (FI_LI, 0, "Periods are \"until_now\"");
         exit (SS$_NORMAL);
      }
   }

   /*******************/
   /* time components */
   /*******************/

   /* non-specified date components default to those of the current day */
   if (!FromDay) FromDay = CurrentNumTime[2];
   if (!FromMonth) FromMonth = CurrentNumTime[1];
   if (!FromYear) FromYear = CurrentNumTime[0];
   if (!ToDay) ToDay =  CurrentNumTime[2];
   if (!ToMonth) ToMonth =  CurrentNumTime[1];
   if (!ToYear) ToYear =  CurrentNumTime[0];

   /* bit of a sanity check (prevents 'MonthName[x]' access violating, etc.) */
   if (FromMonth < 1 || FromMonth > 12) FromMonth = 0;
   if (ToMonth < 1 || ToMonth > 12) ToMonth = 0;

   /* ensure the commencement date/time is acceptable */
   TempDsc.dsc$a_pointer = DateString;
   TempDsc.dsc$w_length = 
      sprintf (DateString, "%d-%s-%d %02.02d:%02.02d",
               FromDay, MonthName[FromMonth], FromYear, FromHour, FromMinute);
   if (Debug) fprintf (stdout, "DateString |%s|\n", DateString);
   if (VMSnok (status = sys$bintim (&TempDsc, &BinTime)))
   {
      sprintf (Scratch, "%02.02d:%02.02d %02.02d/%02.02d/%02.02d",
               FromHour, FromMinute, FromDay, FromMonth, FromYear);
      CgiLibResponseError (FI_LI, status, Scratch);
      exit (SS$_NORMAL);
   }
   /* get the commencement julian date (number of days in epoch) */
   lib$cvt_from_internal_time (&LibJulianDate, &FromJulianDate, &BinTime);

   /* ensure the conclusion date/time is acceptable */
   TempDsc.dsc$a_pointer = ToDateString;
   TempDsc.dsc$w_length = 
      sprintf (ToDateString, "%d-%s-%d %02.02d:%02.02d",
               ToDay, MonthName[ToMonth], ToYear, ToHour, ToMinute);
   if (Debug) fprintf (stdout, "ToDateString |%s|\n", ToDateString);
   if (VMSnok (status = sys$bintim (&TempDsc, &BinTime)))
   {
      sprintf (Scratch, "%02.02d:%02.02d %02.02d/%02.02d/%02.02d",
               ToHour, ToMinute, ToDay, ToMonth, ToYear);
      CgiLibResponseError (FI_LI, status, Scratch);
      exit (SS$_NORMAL);
   }
   /* get the conclusion julian date (number of days in epoch) */
   lib$cvt_from_internal_time (&LibJulianDate, &ToJulianDate, &BinTime);

   /***************************/
   /* calculate period values */
   /***************************/

   /* using julian dates, get number of days and hours sample period covers */
   NumberOfHours = ((ToJulianDate - FromJulianDate) * 24) + ToHour - FromHour;
   NumberOfDays = (NumberOfHours / 24);

   if (DoPresentSummaryPage || DoGraph || DoList || DoDump)
   {
      if (NumberOfHours < 0 || (NumberOfHours == 0 && ToMinute <= FromMinute))
      {
         sprintf (Scratch,
"Beginning of period (%04.04d-%02.02d-%02.02d %02.02d:%02.02d) \
does not precede end (%04.04d-%02.02d-%02.02d %02.02d:%02.02d)!",
         FromYear, FromMonth, FromDay, FromHour, FromMinute,
         ToYear, ToMonth, ToDay, ToHour, ToMinute);
         CgiLibResponseError (FI_LI, 0, Scratch);
         exit (SS$_NORMAL);
      }
      if (NumberOfDays > MAX_SAMPLE_PERIOD_DAYS)
      {
         sprintf (Scratch, "Maximum sample period is %d days.",
                  MAX_SAMPLE_PERIOD_DAYS);
         CgiLibResponseError (FI_LI, 0, Scratch);
         exit (SS$_NORMAL);
      }
   }

   RecordSampleRate = 1;

   /* add one to number of hours, for graphing purposes, re-calculate days */
   NumberOfHours++;
   NumberOfDays = (NumberOfHours / 24);

   /* the number of minutes from midnight data begins being processed */
   StartMinuteOnFirstDay = (FromHour * 60) + FromMinute;

   /* the current minute from the start of the specified period */
   lib$cvt_from_internal_time (&LibJulianDate, &CurrentJulianDate,
                               &CurrentBinTime);
   CurrentMinuteFromStartOfPeriod =
      ((((CurrentJulianDate - FromJulianDate) * 24) +
        (CurrentNumTime[3] - FromHour)) * 60) +
       CurrentNumTime[4];

   if (Debug)
      fprintf (stdout,
"FromJulianDate: %d ToJulianDate: %d\n\
NumberOfDays: %d NumberOfHours: %d RecordSampleRate: %d\n",
      FromJulianDate, ToJulianDate,
      NumberOfDays, NumberOfHours, RecordSampleRate);
}

/*****************************************************************************/
/*
Provide either a standard or comprehensive menu allowing System Performance 
Information system, category and period to be specified.
*/

void SummaryMenu () {
   char  *CheckedPtr,
         *TodayCheckedPtr,
         *UntilNowCheckedPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SummaryMenu()\n");

   if (!DoComprehensiveMenu) CheckedPtr = " CHECKED";

   CgiLibResponseHeader (200, "text/html");

   fprintf (stdout,
"<HTML>\n\
<HEAD>\n\
<META NAME=\"generator\" CONTENT=\"%s\">\n\
<META NAME=\"date\" CONTENT=\"%s\">\n\
<META NAME=\"environment\" CONTENT=\"%s\">\n\
<TITLE>HyperSPI++, Summary Selector</TITLE>\n\
</HEAD>\n\
<BODY%s>\n\
<TABLE BORDER=0 CELLPADDING=%s CELLSPACING=0 WIDTH=100%%%s>\n\
<TR><TD ALIGN=left VALIGN=bottom>\n\
<FONT COLOR=\"%s\">\n\
&nbsp;<FONT SIZE=+1><B>Hyper</B></FONT>\
<FONT SIZE=+3><B>SPI</B><SUP>++</SUP></FONT>\n\
</FONT>\n\
</TD><TD ALIGN=center VALIGN=bottom>\n\
<FONT COLOR=\"%s\" SIZE=+2><B>\n\
Summary Selector\n\
</B></FONT>\n\
</TD><TD ALIGN=right VALIGN=bottom>\n\
<FONT COLOR=\"%s\">\n\
%02.02d %s %d &nbsp;%02.02d:%02.02d\n\
</FONT>\n\
</TD></TR>\n\
</TABLE>\n\
<BLOCKQUOTE>\n\
<P>\n\
<FORM ACTION=\"%s\">\n",
   SoftwareID, UnixDateTime, CgiEnvironmentPtr,
   PageScheme[PS_BODYTAG],
   PageScheme[PS_HEADPADDING],
   PageScheme[PS_HEADBGCOLOR],
   PageScheme[PS_HEADTEXT],
   PageScheme[PS_HEADTEXT],
   PageScheme[PS_HEADTEXT],
   CurrentNumTime[2], MonthName[CurrentNumTime[1]], CurrentNumTime[0],
   CurrentNumTime[3], CurrentNumTime[4],
   CgiScriptNamePtr);

   HttpHasBeenOutput = true;

   SelectNodeNames ();

   if (DoComprehensiveMenu)
   {
      fputs (
"or ... node <INPUT TYPE=text NAME=node SIZE=6 MAXLENGTH=6><BR>",
      stdout);
   }

   if (CurrentNumTime[3] >= 7)
   {
      UntilNowCheckedPtr = " CHECKED";
      TodayCheckedPtr = "";
   }
   else
   {
      UntilNowCheckedPtr = "";
      TodayCheckedPtr = " CHECKED";
   }

   fprintf (stdout,
"<P>\n\
<INPUT TYPE=radio NAME=period VALUE=until_now%s>\
 since 7am <FONT SIZE=-1>(until now)</FONT><BR>\
<INPUT TYPE=radio NAME=period VALUE=today%s> today<BR>\n\
<INPUT TYPE=radio NAME=period VALUE=since_yesterday>\
 since yesterday  <FONT SIZE=-1>(until now)</FONT><BR>\n\
<INPUT TYPE=radio NAME=period VALUE=yesterday> yesterday<BR>\n\
<INPUT TYPE=radio NAME=period VALUE=7> previous week &nbsp;\n\
<INPUT TYPE=radio NAME=period VALUE=14> two weeks &nbsp;\n\
<INPUT TYPE=radio NAME=period VALUE=28> four weeks<BR>\n\
<NOBR>\n\
<INPUT TYPE=radio NAME=period VALUE=other>\
 from\n\
<INPUT TYPE=text SIZE=16 MAXLENGTH=16 NAME=from \
VALUE=\"%04.04d-%02.02d-%02.02d %02.02d:00\">\
 to\n\
<INPUT TYPE=text SIZE=16 MAXLENGTH=16 NAME=to \
VALUE=\"%04.04d-%02.02d-%02.02d 23:59\">\
 &nbsp;<FONT SIZE=-1>(yyyy-mm-dd hh:mm)</FONT>\n\
</NOBR>\n\
<P>\n\
<P><TABLE CELLSPACING=0 CELLPADDING=4>\n\
<TR>\n\
<TD><INPUT TYPE=checkbox NAME=cpu VALUE=yes CHECKED> CPU</TD>\n\
<TD><INPUT TYPE=checkbox NAME=modes VALUE=yes> CPU modes</TD>\n\
<TD><INPUT TYPE=checkbox NAME=locks VALUE=yes> Locking activity</TD>\n\
<TD><INPUT TYPE=checkbox NAME=memory VALUE=yes%s> memory</TD>\n\
</TR><TR>\n\
<TD><INPUT TYPE=checkbox NAME=processes VALUE=yes> processes</TD>\n\
<TD><INPUT TYPE=checkbox NAME=computable VALUE=yes> computable processes</TD>\n",
   UntilNowCheckedPtr, TodayCheckedPtr,
   CurrentNumTime[0], CurrentNumTime[1], CurrentNumTime[2],
   CurrentNumTime[3],
   CurrentNumTime[0], CurrentNumTime[1], CurrentNumTime[2],
   CheckedPtr);

   fprintf (stdout,
"<TD><INPUT TYPE=checkbox NAME=faults VALUE=yes%s> paging</TD>",
   CheckedPtr);
   if (DoComprehensiveMenu)
   {
      fputs (
" <TD><INPUT TYPE=checkbox NAME=peak_faults VALUE=yes> peak</TD>",
      stdout);
   }

   if (DoComprehensiveMenu)
   {
      fputs (
"</TR><TR><TD><INPUT TYPE=checkbox NAME=soft_faults VALUE=yes> paging (soft)</TD> \
 <TD><INPUT TYPE=checkbox NAME=peak_soft_faults VALUE=yes> peak</TD>\n",
      stdout);
      fputs (
"<TD><INPUT TYPE=checkbox NAME=hard_faults VALUE=yes> paging (hard)</TD> \
 <TD><INPUT TYPE=checkbox NAME=peak_hard_faults VALUE=yes> peak</TD>\n",
      stdout);
   }

   fputs ("</TR><TR><TD><INPUT TYPE=checkbox NAME=direct_IO VALUE=yes> disk IO</TD>",
   stdout);
   if (DoComprehensiveMenu)
   {
      fputs (" <TD><INPUT TYPE=checkbox NAME=peak_direct_IO VALUE=yes> peak</TD>",
      stdout);
   }

   fputs (
"<TD><INPUT TYPE=checkbox NAME=buffered_IO VALUE=yes> other IO</TD>",
   stdout);
   if (DoComprehensiveMenu)
   {
      fputs (
" <TD><INPUT TYPE=checkbox NAME=peak_buffered_IO VALUE=yes> peak</TD>",
      stdout);
   }

   fputs (
"<TD><INPUT TYPE=checkbox NAME=mscp_IO VALUE=yes> MSCP IO</TD>",
   stdout);
   if (DoComprehensiveMenu)
   {
      fputs (
" <TD><INPUT TYPE=checkbox NAME=peak_mscp_IO VALUE=yes> peak</TD>",
      stdout);
   }
   fputs ("</TR></TABLE><BR>\n", stdout);

   if (DoComprehensiveMenu)
   {
      fputs (
"<INPUT TYPE=checkbox NAME=include VALUE=peak>\
 <I>include peak plot (where applicable)</I>\n\
<P>\n\
<INPUT TYPE=radio NAME=do VALUE=page CHECKED>graph\n\
 <I>(X,Y magnification:\
<SELECT NAME=Xmag SIZE=1>\n\
<OPTION VALUE=1 SELECTED>1 \n\
<OPTION VALUE=2>2 \n\
<OPTION VALUE=4>4 \n\
</SELECT>,\
<SELECT NAME=Ymag SIZE=1>\n\
<OPTION VALUE=1 SELECTED> 1 \n\
<OPTION VALUE=2> 2 \n\
<OPTION VALUE=4> 4 \n\
</SELECT>)</I><BR>\n\
<INPUT TYPE=radio NAME=do VALUE=list> list<BR>\n\
<INPUT TYPE=radio NAME=do VALUE=dump> dump<BR>\n\
<INPUT TYPE=hidden NAME=extract VALUE=yes>\n\
<P>\n",
      stdout);
   }
   else
   {
      fputs (
"<INPUT TYPE=hidden NAME=do VALUE=page>\n",
      stdout);
   }

   fputs (
"<P>\n\
<INPUT TYPE=submit VALUE=process> \n",
   stdout);

      fputs (
"<INPUT TYPE=checkbox NAME=Refresh VALUE=yes> Refresh today data &nbsp;&nbsp;\n",
         stdout);

   if (!DoComprehensiveMenu)
   { 
      fputs (
"<INPUT TYPE=checkbox NAME=Ymag VALUE=2> double-height graph &nbsp;&nbsp;\n",
         stdout);
   }

   fprintf (stdout,
"<INPUT TYPE=reset VALUE=reset>\n\
</FORM>\n\
</BLOCKQUOTE>\n\
<P>\n");

   SelectorPtr = NULL;

   ButtonBar (2);

   fprintf (stdout,
"</BODY>\n\
</HTML>\n");
}

/*****************************************************************************/
/*
Output a list of node names that data can be selected from.  Get these either 
from a command-line-supplied list or from data files present for the current 
day.
*/ 

void SelectNodeNames () {
   register int  idx;

   char  *CheckedPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SelectNodeNames()\n");

   if (SpiNodeNames[0][0])
   {
      NodeCount = 0;
      for (idx = 0; SpiNodeNames[idx][0]; idx++) NodeCount++;
      if (NodeCount <= MAX_NODE_COUNT_BEFORE_SELECT)
      {
	 int i;
         CheckedPtr = " CHECKED";
	 fprintf(stdout, "<P><TABLE CELLSPACING=0 CELLPADDING=4 BGCOLOR=\"#F3F3F3\">");
         fprintf(stdout, "<TR BGCOLOR=\"#F3F3F3\">");
         for (i = idx = 0; SpiNodeNames[idx][0]; idx++)
         {
	    ++i;
	    if (i > 5) {
	       i = 0;
	       fprintf(stdout, "</TR>");
               fprintf(stdout, "<TR BGCOLOR=\"#F3F3F3\">");
	    }
	    fprintf(stdout, "<TD>");
            fprintf (stdout,
               "<INPUT TYPE=radio NAME=list_node VALUE=\"%s\"%s> %s   \n",
               SpiNodeNames[idx], CheckedPtr, SpiNodeNames[idx]);
            CheckedPtr = "";
	    fprintf(stdout, "</TD>");
         }
	 fprintf(stdout, "</TR></TABLE></P>");
      }
      else
      {
         CheckedPtr = " SELECTED";
         fprintf (stdout, "<SELECT NAME=list_node SIZE=%d>\n",
                  MAX_NODE_COUNT_BEFORE_SELECT);
         for (idx = 0; SpiNodeNames[idx][0]; idx++)
         {
            fprintf (stdout, "<OPTION VALUE=\"%s\"%s> %s\n",
                     SpiNodeNames[idx], CheckedPtr, SpiNodeNames[idx]);
            CheckedPtr = "";
         }
         fputs ("</SELECT>\n", stdout);
      }
   }
   else
   {
      /* create file specification for getting node names (today's files) */
      DataFileSpecLength =
         sprintf (DataFileSpec,
                  "%sHYPERSPI_%s_*_%02.02d%02.02d%02.02d.DAT",
                  HyperSpiDataDirectory, HyperSpiDataVersion,
                  FromDay, FromMonth, FromYear%100);
      if (Debug) fprintf (stdout, "DataFileSpec |%s|\n", DataFileSpec);

      /* count the number of node data files */
      ProcessDataFiles (NULL, false);
      NodeCount = DataFilesFoundCount;

      if (NodeCount <= MAX_NODE_COUNT_BEFORE_SELECT)
         ProcessDataFiles (&SelectNodeNameByDataFileName, false);
      else
      {
         fprintf (stdout, "<SELECT NAME=list_node SIZE=%d>\n",
                  MAX_NODE_COUNT_BEFORE_SELECT);
         ProcessDataFiles (&SelectNodeNameByDataFileName, false);
         fputs ("</SELECT>\n", stdout);
      }
   }
}

/*****************************************************************************/
/*
Called by pointer to function each time a matching data file is returned by 
sys$search(0 in function FindDataFiles().  Output the node name associated 
with the data file.
*/ 

void SelectNodeNameByDataFileName () {
   static char  *CheckedPtr = " CHECKED";
   static char  *SelectedPtr = " SELECTED";

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "SelectNodeNameByFileName() |%s|\n", DataFileName);

   if (NodeCount < MAX_NODE_COUNT_BEFORE_SELECT)
   {
      fprintf (stdout,
         "<INPUT TYPE=radio NAME=list_node VALUE=\"%s\"%s> %s<BR>\n",
         DataNode, CheckedPtr, DataNode);
      CheckedPtr = "";
   }
   else
   {
      fprintf (stdout, "<OPTION VALUE=\"%s\"%s> %s\n",
               DataNode, SelectedPtr, DataNode);
      SelectedPtr = "";
   }
}

/*****************************************************************************/
/*
Generates an HTML page containing information about the selected node, and the 
any additional performance information and graphs.
*/

void PresentSummaryPage () {
   int  PeriodHours;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "PresentSummaryPage()\n");

   CgiLibResponseHeader (200, "text/html");

   fprintf (stdout,
"<HTML>\n\
<HEAD>\n\
<META NAME=\"generator\" CONTENT=\"%s\">\n\
<META NAME=\"date\" CONTENT=\"%s\">\n\
<META NAME=\"environment\" CONTENT=\"%s\">\n\
<TITLE>HyperSPI++ Summary: %s</TITLE>\n\
</HEAD>\n\
<BODY%s>\n\
<TABLE BORDER=0 CELLPADDING=%s CELLSPACING=0 WIDTH=100%%%s>\n\
<TR><TD ALIGN=left VALIGN=bottom>\n\
<FONT COLOR=\"%s\">\n\
&nbsp;<FONT SIZE=+1><B>Hyper</B></FONT>\
<FONT SIZE=+3><B>SPI</B><SUP>++</SUP></FONT>\n\
</FONT>\n\
</TD><TD ALIGN=center VALIGN=bottom>\n\
<FONT COLOR=\"%s\" SIZE=+2><B>\n\
Summary for %s\n\
</B></FONT>\n\
</TD><TD ALIGN=right VALIGN=bottom>\n\
<FONT COLOR=\"%s\">\n\
%02.02d %s %d &nbsp;%02.02d:%02.02d\n\
</FONT>\n\
</TD></TR>\n\
</TABLE>\n\
<BLOCKQUOTE>\n\
<P>\n\
<TABLE BORDER=0 CELLPADDING=2 CELLSPACING=2>\n\
<TR><TH ALIGN=right>Period:</TH><TD>",
   SoftwareID, UnixDateTime, CgiEnvironmentPtr,
   CgiFormNodePtr,
   PageScheme[PS_BODYTAG],
   PageScheme[PS_HEADPADDING],
   PageScheme[PS_HEADBGCOLOR],
   PageScheme[PS_HEADTEXT],
   PageScheme[PS_HEADTEXT],
   CgiFormNodePtr,
   PageScheme[PS_HEADTEXT],
   CurrentNumTime[2], MonthName[CurrentNumTime[1]], CurrentNumTime[0],
   CurrentNumTime[3], CurrentNumTime[4]);

   fflush (stdout);
   HttpHasBeenOutput = true;

   if ((!NumberOfDays && NumberOfHours == 24) || NumberOfDays == 1)
      fputs ("1 day", stdout);
   else
   if (NumberOfDays > 1)
      fprintf (stdout, "%d days", NumberOfDays);

   if ((PeriodHours = (NumberOfHours % 24)) == 1)
      fputs ("1 hour", stdout);
   else
   if (PeriodHours > 1)
      fprintf (stdout, " %d hours", PeriodHours);

   fprintf (stdout,
", from %02.02d %s %d %02.02d:%02.02d to %02.02d %s %d %02.02d:%02.02d\n\
</TD></TR>\n",
   FromDay, MonthName[FromMonth], FromYear, FromHour, FromMinute,
   ToDay, MonthName[ToMonth], ToYear, ToHour, ToMinute);

   if (RefreshTodayData) {
      int currentDay =  CurrentNumTime[2];
      int currentMonth = CurrentNumTime[1];
      int currentYear = CurrentNumTime[0];
      char cmd[512];
      $DESCRIPTOR(cmdD, cmd);
      int flags = CLI$M_NOCLISYM;
      int s;
      unsigned long retPid;
      int completionStatus;

      sprintf(cmd, "copy %s::%sHYPERSPI_%s_%s_%02d%02d%02d.DAT %s", 
	      CgiFormNodePtr, HyperSpiDataDirectory, HyperSpiDataVersion,
	      CgiFormNodePtr,
	      currentDay, currentMonth, currentYear % 100,
	      HyperSpiRefreshDataDirectory);

      cmdD.dsc$w_length = strlen(cmd);
      s = lib$spawn(&cmdD, 0, 0, &flags, 0, &retPid, &completionStatus, 
              0, 0, 0, 0, 0, 0);

   }
   /* summarize the data and calculate required averages */
   SummarizeData ();

   if (DataRecordsProcessedCount)
   {
      IncludeFile (CgiFormNodePtr);

      fprintf (stdout,
"<TR><TH ALIGN=right>CPUs:</TH><TD ALIGN=left>%d</TD></TR>\n\
<TR><TH ALIGN=right>Memory:</TH><TD>%dMb physical, %dMb page space \
(%dMb</TT> total)</TD></TR>\n\
<TR><TH ALIGN=right>Processes:</TH><TD>%d average, %d peak</TD></TR>\n\
<TR><TH ALIGN=right>Computable Processes:</TH><TD>%d average, %d peak</TD></TR>\n\
</TABLE>\n\
<P>\n",
         NumberOfCPUs,
         SystemMemoryMBytes, PageSpaceMBytes,
         SystemMemoryMBytes+PageSpaceMBytes,
         AveNumberOfProcesses, MaxNumberOfProcesses,
         AveNumberOfCom, MaxNumberOfCom);

      if (ProvidePercentCPU) PresentCPU ();
      if (ProvidePercentModeCPU || ProvidePercentModeCPUPie) PresentCPUModes ();
      if (ProvideLocks || ProvideLocksPie) PresentLocks ();
      if (ProvideMemory) PresentMemory ();
      if (ProvideProcesses) PresentProcesses ();
      if (ProvideCom) PresentCom ();
      if (ProvidePageFaults || ProvideSoftPageFaults) PresentPageFaults ();
      if (ProvidePeakPageFaults || ProvidePeakSoftPageFaults) PresentPeakPageFaults ();
      if (ProvideHardPageFaults) PresentHardPageFaults ();
      if (ProvidePeakHardPageFaults) PresentPeakHardPageFaults ();
      if (ProvideBufferedIO) PresentBufferedIO ();
      if (ProvidePeakBufferedIO) PresentPeakBufferedIO ();
      if (ProvideDirectIO) PresentDirectIO ();
      if (ProvidePeakDirectIO) PresentPeakDirectIO ();
      if (ProvideMscpIO) PresentMscpIO ();
      if (ProvidePeakMscpIO) PresentPeakMscpIO ();
   }
   else
      fputs ("No data available.\n", stdout);

   fprintf (stdout,
"</BLOCKQUOTE>\n\
<P>\n");

   SelectorPtr = CgiScriptNamePtr;

   ButtonBar (2);

   fprintf (stdout,
"</BODY>\n\
</HTML>\n");
}

/*****************************************************************************/
/*
Place an HTML link into the HTTP output stream to get a plot-data, GIF image, 
generated dynamically by this application, embedded in the current HTML 
document.  The 'UniqueNumber()' is used to place a unique component into the 
URL of the graph GIF URL, ensuring a cached version is not retrieved.  The 
'WhatPtr' is the name of the performance count to graph.  The optional 
'IncludePtr' allows selected graphs to contain additional information.
*/ 

void GraphImageLink
(
boolean InlineImage,
char *WhatPtr,
char *IncludePtr
)
{
   char  *StartPtr,
         *EndPtr;

   if (Debug) fprintf (stdout, "GraphImageLink()\n");

   if (InlineImage)
   {
      StartPtr = "<IMG ALIGN=top ALT=\"[graph]\" SRC=";
      EndPtr = ">";
   }
   else
   {
      StartPtr = "<I>(<A HREF=";
      EndPtr = ">Extract graph.</A>)</I>";
   }

   fprintf (stdout,
"%s\"%s?do=graph&unique=%s&%s=yes&include=%s\
&Xmag=%d&Ymag=%d&node=%s\
&from=%04.04d-%02.02d-%02.02d%%20%02.02d:%02.02d\
&to=%04.04d-%02.02d-%02.02d%%20%02.02d:%02.02d\
\"%s\n",
   StartPtr,
   CgiScriptNamePtr, UniqueNumber(), WhatPtr, IncludePtr, 
   XMag, YMag, CgiFormNodePtr,
   FromYear, FromMonth, FromDay, FromHour, FromMinute,
   ToYear, ToMonth, ToDay, ToHour, ToMinute,
   EndPtr);
}

/*****************************************************************************/
/*
CPU usage.  Place textual information and a graphic GIF image link into the 
current HTML document being generated.
*/ 

void SectionHeading (char *String)

{
   if (Debug) fprintf (stdout, "SectionHeading() |%s|\n", String);

   fprintf (stdout,
"</BLOCKQUOTE>\n\
<P>\n\
<FONT SIZE=+2>\
&nbsp;&nbsp;<B><U>%s</U></B>\
</FONT>\n\
<P>\n\
<BLOCKQUOTE>\n",
   String);
}

/*****************************************************************************/
/*
CPU usage.  Place textual information and a graphic GIF image link into the 
current HTML document being generated.
*/ 

void PresentCPU () {
   if (Debug) fprintf (stdout, "PresentCPU()\n");

   SectionHeading ("CPU Usage");
   GraphImageLink (true, "cpu", "peak");
   fprintf (stdout,
"<BR>Y axis is percentage. \
Histogram is average, plot is peak \
<I>(total of available %d CPU(s))</I>.\n\
<BR>Overall <TT>%d%%</TT> average, <TT>%d%%</TT> peak \
<I>(\
interrupt <TT>%d%%</TT>, \
MP-sync <TT>%d%%</TT>, \
kernel <TT>%d%%</TT>, \
exec <TT>%d%%</TT>, \
super <TT>%d%%</TT>, \
user <TT>%d%%</TT>\
)</I>.\n",
   NumberOfCPUs,
   AvePercentCPU, PeakPercentCPU,
   AvePercentModeCPU[K_HyperSpiInterruptMode],
   AvePercentModeCPU[K_HyperSpiMultiProcMode],
   AvePercentModeCPU[K_HyperSpiKernelMode],
   AvePercentModeCPU[K_HyperSpiExecutiveMode],
   AvePercentModeCPU[K_HyperSpiSupervisorMode],
   AvePercentModeCPU[K_HyperSpiUserMode]);
   if (ExtractGraph) GraphImageLink (false, "cpu", "peak");
}

/*****************************************************************************/
/*
CPU usage.  Place textual information and a graphic GIF image link into the 
current HTML document being generated.
*/ 

void PresentCPUModes () {
   if (Debug) fprintf (stdout, "PresentCPUModes()\n");

   SectionHeading ("CPU Modes Usage");
   GraphImageLink (true, "modes", "peak");
   fprintf (stdout,
"<BR>Y axis is percentage. \
Histogram is average, plot is peak \
<I>(total of available %d CPU(s))</I>.\n\
<BR>Overall <TT>%d%%</TT> average, <TT>%d%%</TT> peak \
<I>(\
interrupt <TT>%d%%</TT>, \
MP-sync <TT>%d%%</TT>, \
kernel <TT>%d%%</TT>, \
exec <TT>%d%%</TT>, \
super <TT>%d%%</TT>, \
user <TT>%d%%</TT>\
)</I>.\n",
   NumberOfCPUs,
   AvePercentCPU, PeakPercentCPU,
   AvePercentModeCPU[K_HyperSpiInterruptMode],
   AvePercentModeCPU[K_HyperSpiMultiProcMode],
   AvePercentModeCPU[K_HyperSpiKernelMode],
   AvePercentModeCPU[K_HyperSpiExecutiveMode],
   AvePercentModeCPU[K_HyperSpiSupervisorMode],
   AvePercentModeCPU[K_HyperSpiUserMode]);
   if (ExtractGraph) GraphImageLink (false, "modes", "peak");
   GraphImageLink (true, "modes_pie", "");
}

/*****************************************************************************/
/*
Locking activity.  Place textual information and a graphic GIF image link into the 
current HTML document being generated.
*/ 

void PresentLocks () {
   if (Debug) fprintf (stdout, "PresentLocks()\n");

   SectionHeading ("Locking activity (Enq + EnqCvt)");
   GraphImageLink (true, "locks", "peak");
   fprintf (stdout,
"<BR>Y axis is percentage. \
Histogram is average, \
<BR>Overall <TT>%d</TT> average \
<I>(\
incoming <TT>%d/s</TT>, \
outgoing <TT>%d/s</TT>, \
local <TT>%d/s</TT>\
)</I>.\n",
   AveLck, AveLckIn, AveLckOut, AveLckLoc);
   if (ExtractGraph) GraphImageLink (false, "locks", "");
   if (AveLckIn || AveLckOut || AveLckLoc)
      GraphImageLink (true, "locks_pie", "");
}

/*****************************************************************************/
/*
Page space and physical memory usage.  Place textual information and a graphic 
GIF image link into the current HTML document being generated.
*/ 

void PresentMemory () {
   if (Debug) fprintf (stdout, "PresentMemory()\n");

   SectionHeading ("Memory Usage");
   GraphImageLink (true, "memory", "");
   fprintf (stdout,
"<BR>Y axis is percentage. \
Histogram is page space (%dMb) usage, plot is physical memory (%dMb).\n\
<BR>Overall page space <TT>%d%%</TT> average, <TT>%d%%</TT> peak; \
physical memory <TT>%d%%</TT> average, <TT>%d%%</TT> peak.\n",
   PageSpaceMBytes, SystemMemoryMBytes,
   AvePageSpacePercentInUse, MaxPageSpacePercentInUse,
   AveSystemMemoryPercentInUse, MaxSystemMemoryPercentInUse);
   if (ExtractGraph) GraphImageLink (false, "memory", "");
}

/*****************************************************************************/
/*
Number of processes.  Place textual information and a graphic GIF image link 
into the current HTML document being generated.
*/ 

void PresentProcesses () {
   if (Debug) fprintf (stdout, "PresentProcesses()\n");

   SectionHeading ("Number of Processes");
   GraphImageLink (true, "processes", "");
   fprintf (stdout,
"<BR>Y axis is number of processes.\n\
<BR>Overall <TT>%d</TT> average <I>(maximum <TT>%d</TT>)</I>.\n",
   AveNumberOfProcesses, MaxNumberOfProcesses);
   if (ExtractGraph) GraphImageLink (false, "processes", "");
}

/*****************************************************************************/
void PresentCom () {
   if (Debug) fprintf (stdout, "PresentCom()\n");

   SectionHeading ("Number of Computable Processes");
   GraphImageLink (true, "computable", "");
   fprintf (stdout,
"<BR>Y axis is number of computable processes.\n\
<BR>Overall <TT>%d</TT> average <I>(maximum <TT>%d</TT>)</I>.\n",
   AveNumberOfCom, MaxNumberOfCom);
   if (ExtractGraph) GraphImageLink (false, "computable", "");
}


/*****************************************************************************/
/*
Buffered IO (non-disk, non-tape, i.e. network, terminal).  Place textual 
information and a graphic GIF image link into the current HTML document being 
generated.
*/ 

void PresentBufferedIO () {
   char  *ExplainPtr;

   if (Debug) fprintf (stdout, "PresentBufferedIO()\n");

   SectionHeading ("Buffered IO (network, terminal, etc.)");
   if (IncludePeak)
   {
      GraphImageLink (true, "buffered_IO", "peak");
      ExplainPtr = " Histogram is average, plot is peak.";
   }
   else
   {
      GraphImageLink (true, "buffered_IO", "");
      ExplainPtr = "";
   }
   fprintf (stdout,
"<BR>Y axis is IOs per-second.%s\n\
<BR>Overall <TT>%d</TT> average <I>(maximum <TT>%d</TT>)</I>, \
peak <TT>%d</TT>.\n",
   ExplainPtr, AveBufferedIO, MaxAveBufferedIO, PeakBufferedIO);
   if (IncludePeak)
      if (ExtractGraph) GraphImageLink (false, "buffered_IO", "peak"); else;
   else
      if (ExtractGraph) GraphImageLink (false, "buffered_IO", "");
}

/*****************************************************************************/
/*
Peak buffered IO.  Place textual information and a graphic GIF image link 
into the current HTML document being generated.
*/ 

void PresentPeakBufferedIO () {
   if (Debug) fprintf (stdout, "PresentPeakBufferedIO()\n");

   SectionHeading ("Peak Buffered IO (network, terminal, etc.)");
   GraphImageLink (true, "peak_buffered_IO", "");
   fprintf (stdout,
"<BR>Y axis is IOs per-second. \n\
<BR>Peak <TT>%d</TT> per-second.\n",
   PeakBufferedIO);
   if (ExtractGraph) GraphImageLink (false, "peak_buffered_IO", "");
}

/*****************************************************************************/

/*****************************************************************************/
/*
Direct IO (disk, tape, etc).  Place textual information and a graphic GIF 
image link into the current HTML document being generated.
*/ 

void PresentDirectIO () {
   char  *ExplainPtr;

   if (Debug) fprintf (stdout, "PresentDirectIO()\n");

   SectionHeading ("Direct IO (disk, tape, etc.)");
   if (IncludePeak)
   {
      GraphImageLink (true, "direct_IO", "peak");
      ExplainPtr = " Histogram is average, plot is peak.";
   }
   else
   {
      GraphImageLink (true, "direct_IO", "");
      ExplainPtr = "";
   }
   fprintf (stdout,
"<BR>Y axis is IOs per-second.%s\n\
<BR>Overall <TT>%d</TT> average <I>(maximum <TT>%d</TT>)</I>, \
peak <TT>%d</TT>.\n",
   ExplainPtr, AveDirectIO, MaxAveDirectIO, PeakDirectIO);
   if (IncludePeak)
      if (ExtractGraph) GraphImageLink (false, "direct_IO", "peak"); else;
   else
      if (ExtractGraph) GraphImageLink (false, "direct_IO", "");
}

/*****************************************************************************/
/*
Peak direct IO.  Place textual information and a graphic GIF image link 
into the current HTML document being generated.
*/ 

void PresentPeakDirectIO () {
   if (Debug) fprintf (stdout, "PresentPeakDirectIO()\n");

   SectionHeading ("Peak Direct IO (disk, tape, etc.)");
   GraphImageLink (true, "peak_direct_IO", "");
   fprintf (stdout,
"<BR>Y axis is IOs per-second. \n\
<BR>Peak <TT>%d</TT> per-second.\n",
   PeakDirectIO);
   if (ExtractGraph) GraphImageLink (false, "peak_direct_IO", "");
}

/*****************************************************************************/
/*
MSCP IO.  Place textual information and a graphic GIF image link into the 
current HTML document being generated.
*/ 

void PresentMscpIO () {
   char  *ExplainPtr;

   if (Debug) fprintf (stdout, "PresentMscpIO()\n");

   SectionHeading ("MSCP IO (served disk and tape)");
   if (IncludePeak)
   {
      GraphImageLink (true, "mscp_IO", "peak");
      ExplainPtr = " Histogram is average, plot is peak.";
   }
   else
   {
      GraphImageLink (true, "mscp_IO", "");
      ExplainPtr = "";
   }
   fprintf (stdout,
"<BR>Y axis is IOs per-second.%s\n\
<BR>Overall <TT>%d</TT> average <I>(maximum <TT>%d</TT>)</I>, \
peak <TT>%d</TT>.\n",
   ExplainPtr, AveMscpIO, MaxAveMscpIO, PeakMscpIO);
   if (IncludePeak)
      if (ExtractGraph) GraphImageLink (false, "mscp_IO", "peak"); else;
   else
      if (ExtractGraph) GraphImageLink (false, "mscp_IO", "");
}

/*****************************************************************************/
/*
Peak mscp IO.  Place textual information and a graphic GIF image link 
into the current HTML document being generated.
*/ 

void PresentPeakMscpIO () {
   if (Debug) fprintf (stdout, "PresentPeakMscpIO()\n");

   SectionHeading ("Peak MSCP IO (served disk and tape)");
   GraphImageLink (true, "peak_mscp_IO", "");
   fprintf (stdout,
"<BR>Y axis is IOs per-second. \n\
<BR>Peak <TT>%d</TT> per-second.\n",
   PeakMscpIO);
   if (ExtractGraph) GraphImageLink (false, "peak_mscp_IO", "");
}

/*****************************************************************************/
/*
Page faulting IO.  Place textual information and a graphic GIF image link into 
the current HTML document being generated.
*/ 

void PresentPageFaults () {
   char  *ExplainPtr;

   if (Debug) fprintf (stdout, "PresentPageFaults()\n");

   if (ProvidePageFaults) SectionHeading ("Paging");
   else SectionHeading ("Paging (Soft)");
   if (IncludePeak)
   {
      GraphImageLink (true, "faults", "peak");
      ExplainPtr = " Histogram is average, plot is peak.";
   }
   else
   {
      GraphImageLink (true, "faults", "");
      ExplainPtr = "";
   }
   fprintf (stdout,
"<BR>Y axis is page faults per-second.%s\n\
<BR>Overall <TT>%d</TT> average <I>(maximum <TT>%d</TT>)</I>, \
peak <TT>%d</TT>.\n",
   ExplainPtr, AvePageFaults, MaxAvePageFaults, PeakPageFaults);
   if (IncludePeak)
      if (ExtractGraph) GraphImageLink (false, "faults", "peak"); else;
   else
      if (ExtractGraph) GraphImageLink (false, "faults", "");
}

/*****************************************************************************/
/*
Peak paging to disk.  Place textual information and a graphic GIF image link 
into the current HTML document being generated.
*/ 

void PresentPeakPageFaults () {
   if (Debug) fprintf (stdout, "PresentPeakPageFaults()\n");

   SectionHeading ("Peak Paging (Soft)");
   GraphImageLink (true, "peak_faults", "");
   fprintf (stdout,
"<BR>Y axis is page faults per-second. \n\
<BR>Peak <TT>%d</TT> per-second.\n",
   PeakPageFaults);
   if (ExtractGraph) GraphImageLink (false, "peak_faults", "");
}

/*****************************************************************************/
/*
Page faulting IO.  Place textual information and a graphic GIF image link into 
the current HTML document being generated.
*/ 

void PresentHardPageFaults () {
   char  *ExplainPtr;

   if (Debug) fprintf (stdout, "PresentHardPageFaults()\n");

   SectionHeading ("Paging To Disk");
   if (IncludePeak)
   {
      GraphImageLink (true, "hard_faults", "peak");
      ExplainPtr = " Histogram is average, plot is peak.";
   }
   else
   {
      GraphImageLink (true, "hard_faults", "");
      ExplainPtr = "";
   }
   fprintf (stdout,
"<BR>Y axis is IOs per-second.%s\n\
<BR>Overall <TT>%d</TT> average <I>(maximum <TT>%d</TT>)</I>, \
peak <TT>%d</TT>.\n",
   ExplainPtr, AveHardPageFaults, MaxAveHardPageFaults,
   PeakHardPageFaults);
   if (IncludePeak)
      if (ExtractGraph) GraphImageLink (false, "hard_faults", "peak"); else;
   else
      if (ExtractGraph) GraphImageLink (false, "hard_faults", "");
}

/*****************************************************************************/
/*
Peak paging to disk.  Place textual information and a graphic GIF image link 
into the current HTML document being generated.
*/ 

void PresentPeakHardPageFaults () {
   if (Debug) fprintf (stdout, "PresentPeakHardPageFaults()\n");

   SectionHeading ("Peak Paging To Disk");
   GraphImageLink (true, "peak_hard_faults", "");
   fprintf (stdout,
"<BR>Y axis is IOs per-second. \n\
<BR>Peak <TT>%d</TT> per-second.\n",
   PeakHardPageFaults);
   if (ExtractGraph) GraphImageLink (false, "peak_hard_faults", "");
}

/*****************************************************************************/
/*
Read all records for the specified node in the specified time range, 
calculating the average and maximum for each of the data categories.  Assumes 
it is only to be called once as it does not initialize any of the storage.
*/ 

void SummarizeData () {
   int i;

   if (Debug) fprintf (stdout, "SummarizeData()\n");

   ProcessDataFiles (&SummarizeRecord, true);

   /* the averaging must be by the actual sample rate */
   DataRecordsProcessedCount /= RecordSampleRate;

   if (!DataRecordsProcessedCount) return;

   AvePercentCPU /= DataRecordsProcessedCount;
   for (i = 0; i < K_HyperSpiCpuModeCount; ++i)
      AvePercentModeCPU[i] /= DataRecordsProcessedCount;

   AveSystemMemoryPercentInUse /= DataRecordsProcessedCount;
   AveNumberOfProcesses /= DataRecordsProcessedCount;
   AveNumberOfCom /= DataRecordsProcessedCount;
   AvePageSpacePercentInUse /= DataRecordsProcessedCount;
   AvePageFaults /= DataRecordsProcessedCount;
   AveHardPageFaults /= DataRecordsProcessedCount;
   AveBufferedIO /= DataRecordsProcessedCount;
   AveDirectIO /= DataRecordsProcessedCount;
   AveMscpIO /= DataRecordsProcessedCount;
   AveLck /= DataRecordsProcessedCount;
   AveLckLoc /= DataRecordsProcessedCount;
   AveLckIn /= DataRecordsProcessedCount;
   AveLckOut /= DataRecordsProcessedCount;
}

/*****************************************************************************/
/*
Set the various accumulators according to the data in the current 'SpiRecord'.
*/

void SummarizeRecord () {
   static unsigned long  RecordCount = 0,
                         NumberOfProcesses = 0,
                         NumberOfCom = 0,
                         SystemMemoryPercentInUse = 0,
                         PageSpacePercentInUse = 0,
                         PercentCPU = 0,
                         PercentModeCPU[K_HyperSpiCpuModeCount] =
			    {0, 0, 0, 0, 0, 0},
                         BufferedIO = 0,
                         DirectIO = 0,
                         MscpIO = 0,
                         PageFaults = 0,
                         HardPageFaults = 0;

   register unsigned long  tmp;
   int i;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SummarizeRecord()\n");

   /* usually (though not unconditionally) these remain constant! */
   if (SpiRecord.NumberOfCPUs > NumberOfCPUs)
      NumberOfCPUs = SpiRecord.NumberOfCPUs;
   if (SpiRecord.SystemMemoryMBytes > SystemMemoryMBytes)
      SystemMemoryMBytes = SpiRecord.SystemMemoryMBytes;
   if (SpiRecord.PageSpaceMBytes > PageSpaceMBytes)
      PageSpaceMBytes = SpiRecord.PageSpaceMBytes;

   /*
      The percentage CPU values VERY OCCASIONALLY get slightly above
      100%.  I attribute this behaviour to slight inconsistancies
      between obtaining system times and actually obtaining the CPU
      usage data, i.e. to the delta-time used to calculate the percentage.
      I chose to "massage" the data at the display end rather than the
      collection/recording end so that this behaviour could be monitored
      by using the "dump" facility to examine the actual data.
   */
   if (SpiRecord.PercentCPU > 100) SpiRecord.PercentCPU = 100;
   if (SpiRecord.PeakPercentCPU > 100) SpiRecord.PeakPercentCPU = 100;

   for (i = 0; i < K_HyperSpiCpuModeCount; ++i)
      if (SpiRecord.PercentModeCPU[i] > 100)
         SpiRecord.PercentModeCPU[i] = 100;

   /* peaks are always taken as absolutes! */
   if (SpiRecord.PeakPercentCPU > PeakPercentCPU)
      PeakPercentCPU = SpiRecord.PeakPercentCPU;

   if (SpiRecord.PeakBufferedIO > PeakBufferedIO)
      PeakBufferedIO = SpiRecord.PeakBufferedIO;
   if (SpiRecord.PeakDirectIO > PeakDirectIO)
      PeakDirectIO = SpiRecord.PeakDirectIO;
   if (SpiRecord.PeakMscpIO > PeakMscpIO)
      PeakMscpIO = SpiRecord.PeakMscpIO;
   if (SpiRecord.PeakPageFaults > PeakPageFaults)
      PeakPageFaults = SpiRecord.PeakPageFaults;
   if (SpiRecord.PeakHardPageFaults > PeakHardPageFaults)
      PeakHardPageFaults = SpiRecord.PeakHardPageFaults;

   /* this will need to be divided by the number of records processed */
   AvePercentCPU += SpiRecord.PercentCPU;

   /* this will need to be divided by the number of records processed */
   for (i = 0; i < K_HyperSpiCpuModeCount; ++i)
      AvePercentModeCPU[i] += SpiRecord.PercentModeCPU[i];

   /* this will need to be divided by the number of records processed */
   AveNumberOfProcesses += SpiRecord.NumberOfProcesses;
   if (SpiRecord.NumberOfProcesses > MaxNumberOfProcesses)
      MaxNumberOfProcesses = SpiRecord.NumberOfProcesses;

   /* this will need to be divided by the number of records processed */
   AveNumberOfCom += SpiRecord.Computable;
   if (SpiRecord.Computable > MaxNumberOfCom)
      MaxNumberOfCom = SpiRecord.Computable;

   /* this will need to be divided by the number of records processed */
   AveSystemMemoryPercentInUse += SpiRecord.SystemMemoryPercentInUse;
   if (SpiRecord.SystemMemoryPercentInUse > MaxSystemMemoryPercentInUse)
      MaxSystemMemoryPercentInUse = SpiRecord.SystemMemoryPercentInUse;

   /* this will need to be divided by the number of records processed */
   AvePageSpacePercentInUse += SpiRecord.PageSpacePercentInUse;
   if (SpiRecord.PageSpacePercentInUse > MaxPageSpacePercentInUse)
      MaxPageSpacePercentInUse = SpiRecord.PageSpacePercentInUse;

   AveBufferedIO += (tmp = SpiRecord.BufferedIO / 60);
   if (tmp > MaxAveBufferedIO) MaxAveBufferedIO = tmp;

   AveDirectIO += (tmp = SpiRecord.DirectIO / 60);
   if (tmp > MaxAveDirectIO) MaxAveDirectIO = tmp;

   AveMscpIO += (tmp = SpiRecord.MscpIO / 60);
   if (tmp > MaxAveMscpIO) MaxAveMscpIO = tmp;

   AvePageFaults += (tmp = SpiRecord.PageFaults / 60);
   if (tmp > MaxAvePageFaults) MaxAvePageFaults = tmp;

   AveHardPageFaults += (tmp = SpiRecord.HardPageFaults / 60);
   if (tmp > MaxAveHardPageFaults) MaxAveHardPageFaults = tmp;

   AveLck += ((SpiRecord.LckLoc + SpiRecord.LckIn + SpiRecord.LckOut) / 60);
   AveLckLoc += (SpiRecord.LckLoc / 60);
   AveLckIn += (SpiRecord.LckIn / 60);
   AveLckOut += (SpiRecord.LckOut / 60);

}

/*****************************************************************************/
/*
The application is being called to plot a graph of the data of the specified 
node in the specified time range, and generate a GIF image of that graph and 
return it to the browser.
*/

void GraphData () {
   register int  Xcnt, Ycnt, AtX, AtY;
   int i;
   int j;
   int widthHour;
   int graphDeltaHour;
   char* str;
   unsigned long  clr[7]    = /* {0xFF8080, 0x8080FF}; */
	{0xFFCCCC, 0x0000CC, 0x00FFFF, 0xFF66CC, 0x33CC33, 0xFF9900, 0xFFFFFF};
    /* set which slices to explode, and by how much */
    int	expl[] = { 0, 0, 0, 0, 0, 0, 0 };

    /* set missing slices */
    unsigned char missing[] = {FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE};
   int graphWidth;
   int pieGraph = 0;
   char chartTitle[60] = "";
   char pieTitle[60] = "";
   /*********/
   /* begin */
   /*********/

   GDC_title = chartTitle;
   GDCPIE_title = pieTitle;

   if (Debug) fprintf (stdout, "GraphData()\n");

#ifdef GIF_EXPIRED

   CgiLibResponseHeader (200, "image/jpeg",
                         "Expires: Fri, 13 Jan 1978 14:30:00 GMT\n");

#else

   CgiLibResponseHeader (200, "image/jpeg");

#endif
   fflush (stdout);

#ifdef __DECC
      /*+
        Causes records to be written only when explicitly specified by
        a call to fflush, close, or fclose.
      -*/
      if ((stdout = freopen ("SYS$OUTPUT", "w", stdout, "ctx=xplct")) == NULL)
         exit (vaxc$errno);
#endif

   SizeOfPlotX = NumberOfHours * 60 * XMag / RecordSampleRate;
   SizeOfPlotY = 100 + 150 * YMag;
   graphWidth = 800;
   widthHour = SizeOfPlotX / NumberOfHours;
   if (NumberOfHours <= 24) graphDeltaHour = 1;
   else if (NumberOfHours <= 48) graphDeltaHour = 4;
   else if (NumberOfHours <= 192) graphDeltaHour = 12;
   else graphDeltaHour = 24;

   if (Debug) fprintf (stdout, "size %d,%d\n", SizeOfPlotX, SizeOfPlotY);

   HaveDataY = SizeOfMarginY / 6;

   for (i = 0; i < 6; ++i) Values[i] = calloc(SizeOfPlotX, sizeof(float));

#ifdef NEWLIB_GDCHART
   GDC_interpolations = FALSE;
#endif
   Labels = calloc(SizeOfPlotX, sizeof(char*));
   str = calloc(SizeOfPlotX * 12, 1); /* 7 characters maximum for labels */
   for (i = 0; i < SizeOfPlotX; ++i) { 
     Labels[i] = str;
     Labels[i][0] = '\0';
     for (j = 0; j < 6; ++j)
#ifdef NEWLIB_GDCHART
        Values[j][i] = (GDC_interpolations ? GDC_INTERP_VALUE : GDC_NOVALUE);
#else
        Values[j][i] = GDC_NOVALUE;
#endif
     str += 12;
   }   

    {
      int status;
      unsigned long  binTime [2];
      unsigned long  deltaTime [2];
      char  dateString [256];
      $DESCRIPTOR (tempDsc, "");
      static char*  MonthName [] = 
         { "???", "JAN", "FEB", "MAR", "APR", "MAY", "JUN",
           "JUL", "AUG", "SEP", "OCT", "NOV", "DEC" };

      tempDsc.dsc$a_pointer = "1 00:00:00.00";
      tempDsc.dsc$w_length = 13;
      if (VMSnok (status = sys$bintim (&tempDsc, &deltaTime))) {
        CgiLibResponseError (FI_LI, status, CgiFormPeriodPtr);
        exit (SS$_NORMAL);
      }
      tempDsc.dsc$a_pointer = dateString;
      tempDsc.dsc$w_length = 
      sprintf (dateString, "%d-%s-%d %02.02d:%02.02d",
               FromDay, MonthName[FromMonth], FromYear, FromHour, FromMinute);

      if (VMSnok (status = sys$bintim (&tempDsc, &binTime))) {
        sprintf (dateString, "%02.02d:%02.02d %02.02d/%02.02d/%02.02d",
                 FromHour, FromMinute, FromDay, FromMonth, FromYear);
        CgiLibResponseError (FI_LI, status, dateString);
        exit (SS$_NORMAL);
      }
      for (i = 0; i < NumberOfHours;) {
        sprintf(Labels[widthHour * i], "%02d", (FromHour + i) % 24);
        i += graphDeltaHour;
      }

      if (NumberOfHours > 24) {
        unsigned short numTime [7];
	if (FromHour)
          lib$add_times(&binTime, &deltaTime, &binTime);
        for (i = (24 - FromHour) % 24; i < NumberOfHours;) {
          sys$numtim(&numTime, &binTime);
          sprintf(Labels[widthHour * i], "%02d/%02d", numTime[2], numTime[1]);
          lib$add_times(&binTime, &deltaTime, &binTime);
	  i += 24;
	}
      }
    }

   if (ProvidePercentCPU) {
      /* process specified data calling 'GraphRecordCPU()' for each record */
      GraphLinesCnt = 1;
      ChartType = GDC_AREA;
      sprintf(GDC_title, "CPU Usage %s", CgiFormNodePtr);
      ProcessDataFiles (&GraphRecordCPU, true);
   } else if (ProvidePercentModeCPU) {
      /* process specified data calling 'GraphRecordCPU()' for each record */
      GraphLinesCnt = 6;
      ChartType = GDC_AREA;
      sprintf(GDC_title, "CPU Modes Usage %s", CgiFormNodePtr);
      ProcessDataFiles (&GraphRecordCPUModes, true);
   } else if (ProvidePercentModeCPUPie) {
      int i;
      /* process specified data calling 'GraphRecordCPU()' for each record */
      pieGraph = 1;
      GraphLinesCnt = 7;
      sprintf(GDCPIE_title, "CPU Modes Usage %s", CgiFormNodePtr);
      SummarizeData();
      Values[0][0] = AvePercentModeCPU[K_HyperSpiInterruptMode];
      Values[0][1] = AvePercentModeCPU[K_HyperSpiMultiProcMode];
      Values[0][2] = AvePercentModeCPU[K_HyperSpiKernelMode];
      Values[0][3] = AvePercentModeCPU[K_HyperSpiExecutiveMode];
      Values[0][4] = AvePercentModeCPU[K_HyperSpiSupervisorMode];
      Values[0][5] = AvePercentModeCPU[K_HyperSpiUserMode];
      Values[0][6] = 100;
      for (i = 0; i < 6; ++i) {
	if (Values[0][i] == 0) Values[0][i] = 0.1;
        Values[0][6] -= Values[0][i];
      }
      if (Values[0][6] <= 0) Values[0][6]  = 0.1; 
#if LONG_LABEL
      Labels[0] = "Interrupt";
      Labels[1] = "MP-Sync";
      Labels[2] = "Kernel";
      Labels[3] = "Executive";
      Labels[4] = "Supervisor";
      Labels[5] = "User";
      Labels[6] = "NULL";
#else
      Labels[0] = "I";
      Labels[1] = "M";
      Labels[2] = "K";
      Labels[3] = "E";
      Labels[4] = "S";
      Labels[5] = "U";
      Labels[6] = "N";
#endif
   } else if (ProvideLocks) {
      /* process specified data calling 'GraphRecordLocks()' for each record */
      GraphLinesCnt = 3;
      ChartType = GDC_AREA;
      sprintf(GDC_title, "Locking activity (Enq + EnqCvt) %s", CgiFormNodePtr);
      ProcessDataFiles (&GraphRecordLocks, true);
   } else if (ProvideLocksPie) {
      /* process specified data calling 'GraphRecordLocks()' for each record */
      pieGraph = 1;
      GraphLinesCnt = 3;
      sprintf(GDCPIE_title, "Locking activity (Enq + EnqCvt) %s", CgiFormNodePtr);
      SummarizeData();
      Values[0][0] = AveLckIn / (float)(AveLckIn + AveLckOut + AveLckLoc);
      if (Values[0][0] == 0) Values[0][0] = 0.1;
      Values[0][1] = AveLckOut / (float)(AveLckIn + AveLckOut + AveLckLoc);
      if (Values[0][1] == 0) Values[0][1] = 0.1;
      Values[0][2] = 100 - Values[0][0] - Values[0][1];
      Labels[0] = "Incoming";
      Labels[1] = "outgoing";
      Labels[2] = "Local";
   } else if (ProvideMemory) {
      /* process specified data calling 'GraphRecordMemory()' for each record */
      clr[0] = 0xFF8080;
      ChartType = GDC_LINE;
      GraphLinesCnt = 2;
      sprintf(GDC_title, "Memory Usage %s", CgiFormNodePtr);
      ProcessDataFiles (&GraphRecordMemory, true);
   } else if (ProvideBufferedIO || ProvidePeakBufferedIO ||
       ProvideDirectIO ||  ProvidePeakDirectIO ||
       ProvideMscpIO || ProvidePeakMscpIO ||
       ProvidePageFaults || ProvidePeakPageFaults  ||
       ProvideHardPageFaults || ProvidePeakHardPageFaults ||
       ProvideSoftPageFaults || ProvidePeakSoftPageFaults ||
       ProvideProcesses || ProvideCom) {
      /* summarize the data to determine the required Y axis */
      SummarizeData ();

      if (ProvideProcesses) {
         /* process data calling 'GraphRecordProcesses()' for each record */
         GraphLinesCnt = 1;
         ChartType = GDC_LINE;
         sprintf(GDC_title, "Number of Processes %s", CgiFormNodePtr);
         ProcessDataFiles (&GraphRecordProcesses, true);
      } else if (ProvideCom) {
         /* process data calling 'GraphRecordCom()' for each record */
         GraphLinesCnt = 1;
         ChartType = GDC_AREA;
         sprintf(GDC_title, "Number of Computable Processes %s", CgiFormNodePtr);
         ProcessDataFiles (&GraphRecordCom, true);
      } else if (ProvidePageFaults) {
         GraphLinesCnt = 2;
         sprintf(GDC_title, "Paging %s", CgiFormNodePtr);
         ChartType = GDC_AREA;
         ProcessDataFiles (&GraphRecordRange, true);
      } else {
         /* process data calling 'GraphRecordRange()' for each record */
         ChartType = GDC_AREA;
         GraphLinesCnt = 1;
	 if (IncludePeak) GraphLinesCnt = 2;
         ProcessDataFiles (&GraphRecordRange, true);
      }
   } else {
      CgiLibResponseError (FI_LI, 0, "Internal error.");
      exit (SS$_NORMAL);
   }

#ifndef NEWLIB_GDCHART
    GDC_output_type = GDC_JPEG;
#endif

#if 0
    GDC_xtitle = malloc(100);
    sprintf(GDC_xtitle, "%s", "xtitle);
    GDC_ytitle = malloc(100);
#endif

#ifdef __DECC
      if ((stdout = freopen ("SYS$OUTPUT", "wb", stdout, "ctx=xplct")) == NULL)
         exit (vaxc$errno);
#endif
   if (GraphLinesCnt == 1) {
	 clr[0] = 0xFF8080;
	 clr[1] = 0x8080FF;
   }
                              /* ----- call the lib ----- */
   if (pieGraph) {
	/* set options  */
	/* a lot of options are set here for illustration */
	/* none need be - see gdcpie.h for defaults */
	GDCPIE_label_line = TRUE;

	GDCPIE_label_dist = 20;			/* dist. labels to slice edge */
						/* can be negative */
	GDCPIE_LineColor = 0x000000L;
	GDCPIE_label_size = GDC_GIANT;
#if 0
	GDCPIE_3d_depth  = 25;
	GDCPIE_3d_angle  = 45;		/* 0 - 359 */
#endif
	GDCPIE_explode   = expl;	/* default: NULL - no explosion */
	GDCPIE_Color     = clr;
	GDCPIE_BGColor   = 0xFFFFFFL;
	GDCPIE_EdgeColor = 0x000000L;		/* default is GDCPIE_NOCOLOR */
	/* for no edging */
	/* add percentage to slice label */
	/* below the slice label */
	GDCPIE_percent_labels = GDCPIE_PCT_RIGHT;
	GDCPIE_missing   = missing;	/* default: NULL - none missing */
        GDCPIE_percent_labels = GDCPIE_PCT_RIGHT;

	/* call the lib */
	pie_gif(480,			/* width */
		360,			/* height */
		stdout,			/* open file pointer */
		GDC_2DPIE,		/* GDC_3DPIE or GDC_2DPIE */
		GraphLinesCnt,		/* number of slices */
		Labels,			/* slice labels (unlike out_gif(), can be NULL */
		Values[0]);		/* data array */
    } else {
       GDC_BGColor   = 0xFFFFFFL;                  /* backgound color (white) */
       GDC_LineColor = 0x000000L;                  /* line color      (black) */
       GDC_SetColor  = &(clr[0]);                   /* assign set colors */
       GDC_xlabel_spacing = MAXSHORT;
       GDC_requested_ymax = 0;
       GDC_ylabel_density = 20;
       GDC_grid = FALSE;

       /*+
          If all values are 0 out_graph -> floating/decimal divide by zero
       -*/
       for (i = 0; i < SizeOfPlotX; ++i)
#ifdef NEWLIB_GDCHART
          if (Values[0][i] != 0 && Values[0][i] != GDC_NOVALUE &&
	      Values[0][i] != GDC_INTERP_VALUE) break;
#else
          if (Values[0][i] != 0 && Values[0][i] != GDC_NOVALUE) break;
#endif

       if (i >= SizeOfPlotX) {
          Values[0][0] = Values[0][SizeOfPlotX - 1] = 1;
       }

       out_graph(graphWidth, SizeOfPlotY,     /* short       width, height */
              stdout,        /* FILE*       open FILE pointer */
              ChartType,      /* GDC_CHART_T chart type */
              SizeOfPlotX,   /* int         number of points per data set */
              Labels,        /* char*[]     array of X labels */
              GraphLinesCnt,             /* int         number of data sets */
              Values[0],             /* float[]     data set 1 */
              Values[1],             /* float[]     data set 2 */
              Values[2],             /* float[]     data set 3 */
              Values[3],             /* float[]     data set 4 */
              Values[4],             /* float[]     data set 5 */
              Values[5]);           /*  ...        data set 6 */
   }
}

/*****************************************************************************/
/*
Plot a CPU record.  Either total or user-mode usage, as a line from zero to 
the percentage value uasage.  Optionally and additionally, plot peak total or 
user-mode usage, or total usage (for use with the user-mode line plot) as a 
single point.  Plot has a Y axis fixed to representPeak 100 percent.
*/ 

void GraphRecordCPU () {

   if (Debug) fprintf (stdout, "GraphRecordCPU()\n");

   /*
      The percentage CPU values VERY OCCASIONALLY get slightly above
      100%.  I attribute this behaviour to slight inconsistancies
      between obtaining system times and actually obtaining the CPU
      usage data, i.e. to the delta-time used to calculate the percentage.
      I chose to "massage" the data at the display end rather than the
      collection/recording end so that this behaviour could be monitored
      by using the "dump" facility to examine the actual data.
   */
   if (SpiRecord.PercentCPU > 100) SpiRecord.PercentCPU = 100;
   if (SpiRecord.PeakPercentCPU > 100) SpiRecord.PeakPercentCPU = 100;

   if (SpiRecord.PercentUserModeCPU > 100) SpiRecord.PercentUserModeCPU = 100;
   if (SpiRecord.PeakPercentUserModeCPU > 100)
      SpiRecord.PeakPercentUserModeCPU = 100;

   /*********************/
   /* plot the value(s) */
   /*********************/

   {
      /* push these registers onto the stack only if necessary! */
      register int  Xcnt, Ycnt, ToY, AtX, AtY;

      AtX = (NumberOfMinutesIntoData / RecordSampleRate);
      assert(AtX < SizeOfPlotX);
      
      /* no value 100 for a nice graph */
      if (ProvidePercentCPU) {
         Values[0][AtX] = (SpiRecord.PercentCPU >= 100) ? 99.999 : (float)SpiRecord.PercentCPU;
      }      
      return;
   }
}

void GraphRecordCPUModes () {
   int i;

   if (Debug) fprintf (stdout, "GraphRecordCPU()\n");

   /*
      The percentage CPU values VERY OCCASIONALLY get slightly above
      100%.  I attribute this behaviour to slight inconsistancies
      between obtaining system times and actually obtaining the CPU
      usage data, i.e. to the delta-time used to calculate the percentage.
      I chose to "massage" the data at the display end rather than the
      collection/recording end so that this behaviour could be monitored
      by using the "dump" facility to examine the actual data.
   */
   if (SpiRecord.PercentCPU > 100) SpiRecord.PercentCPU = 100;
   if (SpiRecord.PeakPercentCPU > 100) SpiRecord.PeakPercentCPU = 100;

   /* stack values */
   for (i =  1; i < K_HyperSpiCpuModeCount; ++i) {
      SpiRecord.PercentModeCPU[i] += SpiRecord.PercentModeCPU[i - 1];
   }

   /*********************/
   /* plot the value(s) */
   /*********************/

   {
      /* push these registers onto the stack only if necessary! */
      register int  Xcnt, Ycnt, ToY, AtX, AtY;

      AtX = (NumberOfMinutesIntoData / RecordSampleRate);
      assert(AtX < SizeOfPlotX);
      
      /* no value 100 for a nice graph */
      for (i = 0; i < K_HyperSpiCpuModeCount; ++i) {
         Values[i][AtX] = (SpiRecord.PercentModeCPU[i] >= 100) 
			    ? 99.999 
			    : (float)SpiRecord.PercentModeCPU[i];
      }
      return;
   }
}

/*****************************************************************************/
/*
Plot page space usage as a line from zero to percentage used.  Plot physical 
memory as a point at the percentage used.  Plot has a Y axis fixed to 
represent 100 percent.
*/ 

void GraphRecordMemory () {
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "GraphRecordMemory()\n");

   /*******************/
   /* plot the values */
   /*******************/

   {
      /* push these registers onto the stack only if necessary! */
      register int  Xcnt, Ycnt, ToY, AtX, AtY;

      AtX = (NumberOfMinutesIntoData / RecordSampleRate);
      assert(AtX < SizeOfPlotX);


      ToY = SpiRecord.PageSpacePercentInUse;
      AtY = SpiRecord.SystemMemoryPercentInUse;

      Values[0][AtX] = (float)AtY;
      Values[1][AtX] = (float)ToY;
   }
}

/*****************************************************************************/
/*
Plot the number oif processes on the system as plot-points (forming a
semi-continuous line).  Requires a variable range for the Y axis.  That is data
that  can vary considerably in the maximum value represented on the Y axis. 
*/ 

void GraphRecordProcesses () {
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "GraphRecordProcesses()\n");

   /*******************/
   /* plot the value */
   /*******************/

   {
      /* push these registers onto the stack only if necessary! */
      register int  Xcnt, Ycnt, AtX, AtY;

      AtX = (NumberOfMinutesIntoData / RecordSampleRate);

      AtY = SpiRecord.NumberOfProcesses;
      Values[0][AtX] = (float)AtY;
   }
}

/*****************************************************************************/
void GraphRecordCom () {
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "GraphRecordCom()\n");

   /*******************/
   /* plot the value */
   /*******************/

   {
      /* push these registers onto the stack only if necessary! */
      register int  Xcnt, Ycnt, AtX, AtY;

      AtX = (NumberOfMinutesIntoData / RecordSampleRate);

      AtY = SpiRecord.Computable;
      Values[0][AtX] = (float)AtY;
   }
}

/*****************************************************************************/
void GraphRecordLocks () {
   int AtX;
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "GraphRecordLocks()\n");

   /* stack values In, Out, Local*/
   SpiRecord.LckOut += SpiRecord.LckIn;
   SpiRecord.LckLoc += SpiRecord.LckOut;

   /*********************/
   /* plot the value(s) */
   /*********************/


   AtX = (NumberOfMinutesIntoData / RecordSampleRate);
   Values[0][AtX] = (float)(SpiRecord.LckIn / 60);
   Values[1][AtX] = (float)(SpiRecord.LckOut / 60);
   Values[2][AtX] = (float)(SpiRecord.LckLoc / 60);
}

/*****************************************************************************/
/*
Plot values that require a variable range for the Y axis.  That is data that 
can vary considerably in the maximum value represented on the Y axis.
*/

void GraphRecordRange () {
   static unsigned long  StaticPeakValue = 0,
                         StaticValue = 0;

   register unsigned long  PeakValue,
                           Value;
   register unsigned long  PeakValueB,
                           ValueB;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "GraphRecordRange()\n");

   if (ProvideBufferedIO)
   {
      Value = SpiRecord.BufferedIO;
      PeakValue = SpiRecord.PeakBufferedIO;
      sprintf(GDC_title, "Buffered IO %s", CgiFormNodePtr);
   }
   else
   if (ProvidePeakBufferedIO)
   {
      Value = PeakValue = SpiRecord.PeakBufferedIO;
      sprintf(GDC_title, "Peak Buffered IO %s", CgiFormNodePtr);
   }
   else
   if (ProvideDirectIO)
   {
      Value = SpiRecord.DirectIO;
      PeakValue = SpiRecord.PeakDirectIO;
      sprintf(GDC_title, "Direct IO %s", CgiFormNodePtr);
   }
   else
   if (ProvidePeakDirectIO)
   {
      Value = PeakValue = SpiRecord.PeakDirectIO;
      sprintf(GDC_title, "Peak Direct IO %s", CgiFormNodePtr);
   }
   else
   if (ProvideMscpIO)
   {
      Value = SpiRecord.MscpIO;
      PeakValue = SpiRecord.PeakMscpIO;
      sprintf(GDC_title, "MSCP IO %s", CgiFormNodePtr);
   }
   else
   if (ProvidePeakMscpIO)
   {
      Value = PeakValue = SpiRecord.PeakMscpIO;
      sprintf(GDC_title, "Peak MSCP IO %s", CgiFormNodePtr);
   }
   else
   if (ProvidePageFaults)
   {
      Value = SpiRecord.PageFaults;
      PeakValue = SpiRecord.PeakPageFaults;
      ValueB = SpiRecord.HardPageFaults;
      PeakValueB = SpiRecord.PeakHardPageFaults;
      sprintf(GDC_title, "Pages Faults %s", CgiFormNodePtr);
   }
   else
   if (ProvidePeakPageFaults)
   {
      Value = PeakValue = SpiRecord.PeakPageFaults;
      sprintf(GDC_title, "Peak Pages Faults %s", CgiFormNodePtr);
   }
   else
   if (ProvideSoftPageFaults)
   {
      Value = SpiRecord.PageFaults;
      PeakValue = SpiRecord.HardPageFaults;
      sprintf(GDC_title, "Soft Pages Faults %s", CgiFormNodePtr);
   }
   else
   if (ProvidePeakSoftPageFaults)
   {
      Value = PeakValue = SpiRecord.PeakPageFaults;
      sprintf(GDC_title, "Peak Soft Pages Faults %s", CgiFormNodePtr);
   }
   else
   if (ProvideHardPageFaults)
   {
      Value = SpiRecord.HardPageFaults;
      PeakValue = SpiRecord.PeakHardPageFaults;
      sprintf(GDC_title, "Hard Pages Faults %s", CgiFormNodePtr);
   }
   else
   if (ProvidePeakHardPageFaults)
   {
      Value = PeakValue = SpiRecord.PeakHardPageFaults;
      sprintf(GDC_title, "Peak Hard Pages Faults %s", CgiFormNodePtr);
   }

   if (Debug) fprintf (stdout, "Value: %d PeakValue: %d\n", Value, PeakValue);

   /*********************/
   /* plot the value(s) */
   /*********************/

   {
      /* push these registers onto the stack only if necessary! */
      register int  Xcnt, Ycnt, ToY, AtX, AtY;

      AtX = (NumberOfMinutesIntoData / RecordSampleRate);

      if (ProvideBufferedIO ||
          ProvideDirectIO ||
          ProvideMscpIO ||
          ProvideSoftPageFaults ||
          ProvideHardPageFaults)
         Values[0][AtX] = (float)(Value / 60);
      else if (ProvidePageFaults) {
         Values[1][AtX] = (float)(Value / 60);
         Values[0][AtX] = (float)(ValueB / 60);
         return;
      } else
         Values[0][AtX] = PeakValue;

      if (IncludePeak)
         AtY = PeakValue;
      else
      {
         /* no, nothing extra to plot! */
         return;
      }

      Values[0][AtX] = (float)AtY;
   }
}

/*****************************************************************************/
/*
*/

void ListProcessedData () {
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ListProcessedData()\n");

   CgiLibResponseHeader (200, "text/html");

   fprintf (stdout,
"<HTML>\n\
<HEAD>\n\
<META NAME=\"generator\" CONTENT=\"%s\">\n\
<META NAME=\"date\" CONTENT=\"%s\">\n\
<META NAME=\"environment\" CONTENT=\"%s\">\n\
<TITLE>HyperSPI++ Processed Data</TITLE>\n\
</HEAD>\n\
<BODY%s>\n\
<TABLE BORDER=0 CELLPADDING=%s CELLSPACING=0 WIDTH=100%%%s>\n\
<TR><TD ALIGN=left VALIGN=bottom>\n\
<FONT COLOR=\"%s\">\n\
&nbsp;<FONT SIZE=+1><B>Hyper</B></FONT>\
<FONT SIZE=+3><B>SPI</B><SUP>++</SUP></FONT>\n\
</FONT>\n\
</TD><TD ALIGN=center VALIGN=bottom>\n\
<FONT COLOR=\"%s\" SIZE=+2><B>\n\
Processed Data\n\
</B></FONT>\n\
</TD><TD ALIGN=right VALIGN=bottom>\n\
<FONT COLOR=\"%s\">\n\
%02.02d %s %d &nbsp;%02.02d:%02.02d\n\
</FONT>\n\
</TD></TR>\n\
</TABLE>\n",
   SoftwareID, UnixDateTime, CgiEnvironmentPtr,
   PageScheme[PS_BODYTAG],
   PageScheme[PS_HEADPADDING],
   PageScheme[PS_HEADBGCOLOR],
   PageScheme[PS_HEADTEXT],
   PageScheme[PS_HEADTEXT],
   PageScheme[PS_HEADTEXT],
   CurrentNumTime[2], MonthName[CurrentNumTime[1]], CurrentNumTime[0],
   CurrentNumTime[3], CurrentNumTime[4]);

   HttpHasBeenOutput = true;

   ProcessDataFiles (&ListProcessedRecord, true);

   if (DataFilesProcessedCount)
      fputs ("</PRE>\n", stdout);
   else
      fputs ("</PRE>\n<P>No matching data files.\n", stdout);

   fprintf (stdout,
"<P>\n\
<TABLE BORDER=0 CELLPADDING=0 CELLSPACING=2 WIDTH=100%%>\n\
<TR><TD%s>\n\
<TABLE BORDER=0 CELLPADDING=0 CELLSPACING=2>\n\
<TR><TD><FONT SIZE=-1>&nbsp;</FONT></TD></TR>\n\
</TABLE>\n\
</TD></TR>\n\
</TABLE>\n\
</BODY>\n\
</HTML>\n",
      PageScheme[PS_HEADBGCOLOR]);
}

/*****************************************************************************/
/*
*/

void ListProcessedRecord () {
   static int  PrevDay = -1,
               PrevHour = -1;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ListProcessedRecord()\n");

   if (DataDay != PrevDay)
   {
      if (PrevDay != -1) fputs ("</PRE>\n", stdout);
      PrevDay = DataDay;
      fprintf (stdout, 
"<P>\n\
<FONT SIZE=+1><B>%s &nbsp;%02.02d %s %d</B></FONT>\n\
<P><B>%s</B>\n\
<P><PRE>",
      CgiFormNodePtr, DataDay, MonthName[DataMonth], DataYear,
      DataFileName);
   }

   if (SpiRecord.Hour != PrevHour)
   {
      if (PrevHour != -1) fputs ("\n", stdout);
      PrevHour = SpiRecord.Hour;
      fprintf (stdout,
"hh:mm  CPU usr  mem pge  b-IO peak  d-IO peak  mscp peak  \
 flts  peak  hard peak\n\
-----  --- ---  --- ---  ---- ----  ---- ----  ---- ----  \
----- -----  ---- ----\n");
   }

   fprintf (stdout,
"%02.02d:%02.02d  %3d %3d  %3d %3d  \
%4d %4d  %4d %4d  %4d %4d  %5d %5d  %4d %4d\n",
   SpiRecord.Hour, SpiRecord.Minute,
   SpiRecord.PercentCPU, SpiRecord.PercentUserModeCPU,
   SpiRecord.SystemMemoryPercentInUse,
   SpiRecord.PageSpacePercentInUse,
   SpiRecord.BufferedIO / 60, SpiRecord.PeakBufferedIO,
   SpiRecord.DirectIO / 60, SpiRecord.PeakDirectIO,
   SpiRecord.MscpIO / 60, SpiRecord.PeakMscpIO,
   SpiRecord.PageFaults / 60, SpiRecord.PeakPageFaults,
   SpiRecord.HardPageFaults / 60, SpiRecord.PeakHardPageFaults);
}

/*****************************************************************************/
/*
*/

void DumpData () {
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DumpData()\n");

   CgiLibResponseHeader (200, "text/html");

   fprintf (stdout,
"<HTML>\n\
<HEAD>\n\
<META NAME=\"generator\" CONTENT=\"%s\">\n\
<META NAME=\"date\" CONTENT=\"%s\">\n\
<META NAME=\"environment\" CONTENT=\"%s\">\n\
<TITLE>HyperSPI++ Data Dump</TITLE>\n\
</HEAD>\n\
<BODY%s>\n\
<TABLE BORDER=0 CELLPADDING=%s CELLSPACING=0 WIDTH=100%%%s>\n\
<TR><TD ALIGN=left VALIGN=bottom>\n\
<FONT COLOR=\"%s\">\n\
&nbsp;<FONT SIZE=+1><B>Hyper</B></FONT>\
<FONT SIZE=+3><B>SPI</B><SUP>++</SUP></FONT>\n\
</FONT>\n\
</TD><TD ALIGN=center VALIGN=bottom>\n\
<FONT COLOR=\"%s\" SIZE=+2><B>\n\
Data Dump\n\
</B></FONT>\n\
</TD><TD ALIGN=right VALIGN=bottom>\n\
<FONT COLOR=\"%s\">\n\
%02.02d %s %d &nbsp;%02.02d:%02.02d\n\
</FONT>\n\
</TD></TR>\n\
</TABLE>\n",
   SoftwareID, UnixDateTime, CgiEnvironmentPtr,
   PageScheme[PS_BODYTAG],
   PageScheme[PS_HEADPADDING],
   PageScheme[PS_HEADBGCOLOR],
   PageScheme[PS_HEADTEXT],
   PageScheme[PS_HEADTEXT],
   PageScheme[PS_HEADTEXT],
   CurrentNumTime[2], MonthName[CurrentNumTime[1]], CurrentNumTime[0],
   CurrentNumTime[3], CurrentNumTime[4]);

   HttpHasBeenOutput = true;

   ProcessDataFiles (&DumpRecord, true);

   if (DataFilesProcessedCount)
      fputs ("</PRE>\n", stdout);
   else
      fputs ("</PRE>\n<P>No matching data files.\n", stdout);

   fprintf (stdout,
"<P>\n\
<TABLE BORDER=0 CELLPADDING=0 CELLSPACING=2 WIDTH=100%%>\n\
<TR><TD%s>\n\
<TABLE BORDER=0 CELLPADDING=0 CELLSPACING=2>\n\
<TR><TD><FONT SIZE=-1>&nbsp;</FONT></TD></TR>\n\
</TABLE>\n\
</TD></TR>\n\
</TABLE>\n\
</BODY>\n\
</HTML>\n",
      PageScheme[PS_HEADBGCOLOR]);
}

/*****************************************************************************/
/*
*/

void DumpRecord() {
   static int  PrevDay = -1,
               PrevHour = -1;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DumpRecord()\n");

   if (DataDay != PrevDay)
   {
      if (PrevDay != -1) fputs ("</PRE>\n", stdout);
      PrevDay = DataDay;
      fprintf (stdout, 
"<P>\n\
<FONT SIZE=+1><B>%s &nbsp;%02.02d %s %d</B></FONT>\n\
<P><B>%s</B>\n\
<P><PRE>",
      CgiFormNodePtr, DataDay, MonthName[DataMonth], DataYear,
      DataFileName);
   }

   if (SpiRecord.Hour != PrevHour)
   {
      fputs ("\n", stdout);
      PrevHour = SpiRecord.Hour;
      fprintf (stdout,
"hh:mm  prc  CPU  pk  usr  pk  mem pge   buf-IO    peak  \
 dir-IO    peak  mscp-IO    peak   faults    peak   hard  peak Com \
int mps krn exe sup usr Lck-Loc Lck-In  Lck-Out\n\
-----  ---  --- ---  --- ---  --- ---  ------- -------  \
------- -------  ------- -------  ------- -------  ----- ----- --- \
--- --- --- --- --- --- ------- ------- -------\n");
   }

   fprintf (stdout,
"%02.02d:%02.02d  %3d  %3d %3d  %3d %3d  %3d %3d  \
%7d %7d  %7d %7d  %7d %7d  %7d %7d  %5d %5d %3d \
%3d %3d %3d %3d %3d %3d %7d %7d %7d\n",
   SpiRecord.Hour, SpiRecord.Minute,
   SpiRecord.NumberOfProcesses,
   SpiRecord.PercentCPU, SpiRecord.PeakPercentCPU,
   SpiRecord.PercentUserModeCPU, SpiRecord.PeakPercentUserModeCPU,
   SpiRecord.SystemMemoryPercentInUse,
   SpiRecord.PageSpacePercentInUse,
   SpiRecord.BufferedIO, SpiRecord.PeakBufferedIO,
   SpiRecord.DirectIO, SpiRecord.PeakDirectIO,
   SpiRecord.MscpIO, SpiRecord.PeakMscpIO,
   SpiRecord.PageFaults, SpiRecord.PeakPageFaults,
   SpiRecord.HardPageFaults, SpiRecord.PeakHardPageFaults,
   SpiRecord.Computable,
   SpiRecord.PercentModeCPU[K_HyperSpiInterruptMode],
   SpiRecord.PercentModeCPU[K_HyperSpiMultiProcMode],
   SpiRecord.PercentModeCPU[K_HyperSpiKernelMode],
   SpiRecord.PercentModeCPU[K_HyperSpiExecutiveMode],
   SpiRecord.PercentModeCPU[K_HyperSpiSupervisorMode],
   SpiRecord.PercentModeCPU[K_HyperSpiUserMode],
   SpiRecord.LckLoc,
   SpiRecord.LckIn,
   SpiRecord.LckOut);
}

/*****************************************************************************/
/*
The first parameter to this function is the address (a pointer to) the 
function used to process each file name found.
*/ 
 
int ProcessDataFiles(int (*ProcessFileFunction)(), boolean OpenDataFile) {
   register char  *cptr, *sptr;

   int  status;
   char  ExpandedFileName [256],
         Scratch [256];
   struct FAB  SearchFab;
   struct RAB  SearchRab;
   struct NAM  SearchNam;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ProcessDataFiles() |%s|\n", DataFileSpec);

   DataFilesFoundCount = DataFilesProcessedCount =
      DataRecordsReadCount = DataRecordsProcessedCount = 0;

   SearchFab = cc$rms_fab;
   SearchFab.fab$l_fna = DataFileSpec;
   SearchFab.fab$b_fns = DataFileSpecLength;
   SearchFab.fab$l_fop = FAB$M_NAM;
   SearchFab.fab$l_nam = &SearchNam;

   SearchNam = cc$rms_nam;
   SearchNam.nam$l_esa = ExpandedFileName;
   SearchNam.nam$b_ess = sizeof(ExpandedFileName)-1;
   SearchNam.nam$l_rsa = DataFileName;
   SearchNam.nam$b_rss = sizeof(DataFileName)-1;

   if (VMSnok (status = sys$parse (&SearchFab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$parse() %%X%08.08X\n", status);
      CgiLibResponseError (FI_LI, status, DataFileSpec);
      exit (SS$_NORMAL);
   }

   while (VMSok (status = sys$search (&SearchFab, 0, 0)))
   {
      DataFilesFoundCount++;

      SearchNam.nam$l_ver[SearchNam.nam$b_ver] = '\0';
      DataFileNameLength = SearchNam.nam$b_rsl;
      if (Debug) fprintf (stdout, "DataFileName |%s|\n", DataFileName);

      /*
         Pull the node name and time components from the data file name.
         Format: "HYPERSPI_v_node_ddmmyy.DAT"
         ("v" represents the data file version, a single digit number)
      */
      cptr = SearchNam.nam$l_name;
      /* skip "HYPERSPI_version_" */
      while (*cptr && *cptr != '_') cptr++;
      if (*cptr) cptr++;
      while (*cptr && *cptr != '_') cptr++;
      if (*cptr) cptr++;
      /* get the node name this data represents */
      sptr = DataNode;
      while (*cptr && *cptr != '_') *sptr++ = *cptr++;
      *sptr = '\0';
      if (*cptr) cptr++;
      /* get the day, month and year */
      sptr = Scratch;
      if (*cptr) *sptr++ = *cptr++;
      if (*cptr) *sptr++ = *cptr++;
      *sptr = '\0';
      DataDay = atoi (Scratch);
      sptr = Scratch;
      if (*cptr) *sptr++ = *cptr++;
      if (*cptr) *sptr++ = *cptr++;
      *sptr = '\0';
      DataMonth = atoi (Scratch);
      sptr = Scratch;
      if (*cptr) *sptr++ = *cptr++;
      if (*cptr) *sptr++ = *cptr++;
      *sptr = '\0';
      DataYear = atoi (Scratch);
      if (Debug)
         fprintf (stdout, "data-file-name |%s|%d|%d|%d|\n",
                  DataNode, DataDay, DataMonth, DataYear);

      /* filter on year, month and day */
      if (DataYear <= 90)
         DataYear += 2000;
      else
         DataYear += 1900;
      if (DataYear < FromYear || DataYear > ToYear) continue;
      if (DataYear == FromYear)
      {
         if (DataMonth < FromMonth) continue;
         if (DataMonth == FromMonth && DataDay < FromDay) continue;
      }
      if (DataYear == ToYear)
      {
         if (DataMonth > ToMonth) continue;
         if (DataMonth == ToMonth && DataDay > ToDay) continue;
      }

      DataFilesProcessedCount++;

      /* can be used to just count the number of matching files! */
      if (ProcessFileFunction == NULL) continue;

      if (OpenDataFile) {
         ProcessDataFileRecords(ProcessFileFunction);
      } else
         /* by pointer, call the function to process this file name */
         (*ProcessFileFunction) ();
   }
   if (Debug) fprintf (stdout, "sys$search() %%X%08.08X\n", status);

   if (SearchFab.fab$l_sts == RMS$_FNF || SearchFab.fab$l_sts == RMS$_NMF)
      return (SS$_NORMAL);

   CgiLibResponseError (FI_LI, status, DataFileSpec);
   exit (SS$_NORMAL);
}

/*****************************************************************************/
/*
Open the 'DataFileName', read each record and call a specified function to 
process it, then close the file.  The first parameter to this function is the 
address (a pointer to) the function used to process the SPI data record.
*/ 
void ProcessDataFileRecords (int (*ProcessDataFunction)(struct HyperSpiData*)) {
   static long  LibJulianDate = LIB$K_JULIAN_DATE;
   static unsigned short  PrevNumTime [7] = {0,0,0,0,0,0,0};

   int  status;
   unsigned long  JulDate,
                  BinTime [2];
   struct FAB  DataFileFab;
   struct RAB  DataFileRab;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ProcessDataFileRecords() |%s|\n", DataFileName);

   DataFileFab = cc$rms_fab;
   DataFileFab.fab$b_fac = FAB$M_GET;
   DataFileFab.fab$l_fna = DataFileName;  
   DataFileFab.fab$b_fns = DataFileNameLength;
   DataFileFab.fab$b_shr = FAB$M_SHRGET | FAB$M_SHRPUT;

   if (VMSnok (status = sys$open (&DataFileFab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$open() %%X%08.08X\n", status);
      if (status == RMS$_FNF)
      {
         CgiLibResponseError (FI_LI, status, DataNode);
         exit (SS$_NORMAL);
      }
      CgiLibResponseError (FI_LI, status, DataNode);
      exit (SS$_NORMAL);
   }

   DataFileRab = cc$rms_rab;
   DataFileRab.rab$l_fab = &DataFileFab;
   /* 2 buffers and read ahead performance option */
   DataFileRab.rab$b_mbf = 2;
   /*+
      DataFileRab.rab$b_mbf = 5;
      DataFileRab.rab$b_mbc = 127;
   -*/
   DataFileRab.rab$l_rop = RAB$M_RAH;
   DataFileRab.rab$l_ubf = (char *)&SpiRecord;
   DataFileRab.rab$w_usz = sizeof(SpiRecord);

   if (VMSnok (status = sys$connect (&DataFileRab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$connect() %%X%08.08X\n", status);
      sys$close (&DataFileFab, 0, 0);
      CgiLibResponseError (FI_LI, status, DataNode);
      exit (SS$_NORMAL);
   }

   while (VMSok (status = sys$get (&DataFileRab, 0, 0)))
   {
      DataRecordsReadCount++;

      /* filter on hour and minute */
      if (SpiRecord.Day == FromDay)
      {
         if (SpiRecord.Hour < FromHour) continue;
         if (SpiRecord.Hour == FromHour && SpiRecord.Minute < FromMinute)
            continue;
      }
      if (SpiRecord.Day == ToDay)
      {
         if (SpiRecord.Hour > ToHour) continue;
         if (SpiRecord.Hour == ToHour && SpiRecord.Minute > ToMinute) continue;
      }

      if (SpiRecord.Hour != PrevNumTime[3] ||
          SpiRecord.Day != PrevNumTime[2] ||
          SpiRecord.Month != PrevNumTime[1] ||
          SpiRecord.Year != PrevNumTime[0])
      {
         PrevNumTime[3] = SpiRecord.Hour;
         PrevNumTime[2] = SpiRecord.Day;
         PrevNumTime[1] = SpiRecord.Month;
         PrevNumTime[0] = SpiRecord.Year;

         lib$cvt_vectim (&PrevNumTime, &BinTime);
         lib$cvt_from_internal_time (&LibJulianDate, &JulDate, &BinTime);
         NumberOfDaysIntoData = JulDate - FromJulianDate;
         StartMinuteOfData = NumberOfDaysIntoData * 1440;
         if (Debug)
            fprintf (stdout,
               "NumberOfDaysIntoData: %d StartMinuteOfData: %d\n",
               NumberOfDaysIntoData, StartMinuteOfData);
      }
      NumberOfMinutesIntoData = StartMinuteOfData +
                                ((SpiRecord.Hour * 60) + SpiRecord.Minute) -
                                StartMinuteOnFirstDay;

      DataRecordsProcessedCount++;

      /* by pointer, call the function to process this SPI data record */
      (*ProcessDataFunction) (&SpiRecord);
   }

   sys$close (&DataFileFab, 0, 0);

   if (status == RMS$_EOF) status = SS$_NORMAL;
   if (VMSnok (status))
   {
      CgiLibResponseError (FI_LI, status, DataNode);
      exit (SS$_NORMAL);
   }
}

/*****************************************************************************/
/*
The specified parameter is the name of the file, excluding directory and file 
type, just the name.  Construct a full HTML file name and open it, read each 
record just passing the contents into the HTTP output stream, the close the 
file.  If the file cannot be found the status RMS$_FNF is returned, all other 
errors are reported and the image exits.
*/ 

int IncludeFile (char *Name) {
   int  status;
   char  FileName [256],
         Line [256],
         Scratch [256];
   struct FAB  FileFab;
   struct RAB  FileRab;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "IncludeFile() |%s|\n", Name);

   sprintf (FileName, "%s%s.HTML", HyperSpiDirectoryPtr, Name);
   if (Debug) fprintf (stdout, "FileName |%s|\n", FileName);

   FileFab = cc$rms_fab;
   FileFab.fab$b_fac = FAB$M_GET;
   FileFab.fab$l_fna = FileName;  
   FileFab.fab$b_fns = strlen(FileName);  
   FileFab.fab$b_shr = FAB$M_SHRGET;

   if (VMSnok (status = sys$open (&FileFab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$open() %%X%08.08X\n", status);
      if (status == RMS$_FNF || status == RMS$_DNF) return (status);
      CgiLibResponseError (FI_LI, status, Name);
      exit (SS$_NORMAL);
   }

   FileRab = cc$rms_rab;
   FileRab.rab$l_fab = &FileFab;
   /* 2 buffers and read ahead performance option */
   FileRab.rab$b_mbf = 2;
   FileRab.rab$l_rop = RAB$M_RAH;
   FileRab.rab$l_ubf = Line;
   FileRab.rab$w_usz = sizeof(Line)-1;

   if (VMSnok (status = sys$connect (&FileRab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$connect() %%X%08.08X\n", status);
      sys$close (&FileFab, 0, 0);
      CgiLibResponseError (FI_LI, status, Name);
      exit (SS$_NORMAL);
   }

   while (VMSok (status = sys$get (&FileRab, 0, 0)))
   {
      Line[FileRab.rab$w_rsz++] = '\r';
      Line[FileRab.rab$w_rsz++] = '\n';
      Line[FileRab.rab$w_rsz] = '\0';
      if (Debug) fprintf (stdout, "Line |%s|\n", Line);
      fputs (Line, stdout);
   }

   sys$close (&FileFab, 0, 0);

   if (status == RMS$_EOF) return (status);

   CgiLibResponseError (FI_LI, status, Name);
   exit (SS$_NORMAL);
}

/*****************************************************************************/
/*
Return a pointer to a unique number built up of time components.  Used to 
place a unique component into the URL of a graph GIF, ensuring a cached 
version is not retrieved.
*/

char* UniqueNumber ()

{
   static char  String [16];

   unsigned long  BinTime [2];
   unsigned short  NumTime [7];

   sys$gettim (&BinTime);
   sys$numtim (&NumTime, &BinTime);

   sprintf (String, "%d%d%d%d%d%d%d",
            NumTime[0] % 10, NumTime[1], NumTime[2], NumTime[3],
            NumTime[4], NumTime[5], NumTime[6]);
   return (String);
}

/****************************************************************************/
/*
Does a case-insensitive, character-by-character string compare and returns 
true if two strings are the same, or false if not.  If a maximum number of 
characters are specified only those will be compared, if the entire strings 
should be compared then specify the number of characters as 0.
*/ 

boolean strsame
(
register char *sptr1,
register char *sptr2,
register int  count
)
{
   while (*sptr1 && *sptr2)
   {
      if (toupper (*sptr1++) != toupper (*sptr2++)) return (false);
      if (count)
         if (!--count) return (true);
   }
   if (*sptr1 || *sptr2)
      return (false);
   else
      return (true);
}

/*****************************************************************************/

