/*
	Copyright (C) 2004 Jonas Lindholm

	This software was developed by Jonas Lindholm, jlhm@usa.net

	History

	V1.0		Jonas Lindholm	2004-05-14

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <string.h>
#include <stdio.h>

#include <starlet.h>
#include <ssdef.h>
#include <stsdef.h>
#include <psldef.h>
#include <lnmdef.h>
#include <jpidef.h>
#include <descrip.h>

#include <rms.h>
#include <prcdef.h>
#include <prvdef.h>

#include <cli$routines.h>
#include <climsgdef.h>

#include "lgi.h"
#include "lgildap.h"
#include "private.h"

#include "uafdef.h"


/*
	Description:

	This routine ask the user for password.
	Default prompt is "assword (LDAP): " but it can be changed by defining logical
	name LGI_LDAP_PWD_PROMPT.

	INPUT:		LGI argument vector

	OUTPUT:		any status returned by callback routine LGI$ICB_PASSWORD
*/

int do_GetPassword(struct LGI$ARG_VECTOR *av) {
	int		stat;
	char		prompt[256];
	char		userprompt[256] = "Password (LDAP): ";

	stat = do_trnlnm("LGI_LDAP_PWD_PROMPT", userprompt, sizeof(userprompt) - 1, PSL$C_EXEC);

	sprintf(prompt, "\n\r%s", userprompt);

	{
		$DESCRIPTOR(promptdsc, prompt);

		promptdsc.dsc$w_length = strlen(prompt);
		stat = av->LGI$ICB_PASSWORD(-1, &promptdsc);

	}

	return stat;
}


/*
	Description:

	This routine ask and verify the password the user gives for interactive and DECnet
	login.
	If local verification is needed and ok (LGI LDAP Server is not running and logical name
	LGI_LDAP_ALLOW_LOCAL_AUTH is set) LOGINOUT.EXE normal password prompting is done.

	If it is an interactive process the user is prompted for the password.

	If LGI LDAP Server is reachable and password is ok the login is granted by returning
	LGI$_SKIPRELATED.

	If LGI LDAP Server was unreachable and local authentication is ok OR user as SYSPRV
	password verification is done against the password in local SYSUAF.DAT. If the password
	is ok login is granted by returning LGI$_SKIPRELATED.

	If qualifier LOCAL_PASSWORD is requested and supported in LGI_IDENTITY and the user
	has SYSPRV the password is verified against the local SYSUAF.DAT and if it is ok
	LGI$_SKIPRELATED is returned.

	If the account has a secondary password a verification against the secondary password
	hash is done in SYSUAF. There is no support for secondary password on LDAP server.

	Everything else return LGI$_INVPWD.

	INPUT:		LGI argument vector
			pointer to private data

	OUTPUT:		see description above.
*/	                        
           	
int cb_authenticate(struct LGI$ARG_VECTOR *av, CTX_T **ctx) {
	int			stat;

	if (!(*av->LGI$A_ICR_CREPRC_FLAGS & PRC$M_INTER) &&
	    !(*av->LGI$A_ICR_CREPRC_FLAGS & PRC$M_NETWRK)) return SS$_NORMAL;  /* Not interactive nor network, do normal processing */

	if (*av->LGI$A_ICR_CREPRC_FLAGS & PRC$M_NOPASSWORD) return SS$_NORMAL;  /* Invoked as logged in, don't prompt */
 
	if (*av->LGI$A_ICR_SUBPROCESS != 0) return SS$_NORMAL;  /* Don't prompt on subprocesses */

	if ((*av->LGI$A_ICR_CREPRC_FLAGS & PRC$M_INTER) &&
	    (*av->LGI$A_ICR_PWDCOUNT == 0)) { /* This is an interactive login and account has no password */
 
		return SS$_NORMAL;

	}

	if (ctx == NULL || *ctx == NULL || (*ctx)->flags.dolocal) {

		return SS$_NORMAL;

	}

	/*
		We must retreive the password from the user if not a network process.
	*/

	if (*av->LGI$A_ICR_JOB_TYPE != JPI$K_NETWORK) {

		stat = do_GetPassword(av);

	}

	if ((*av->LGI$A_ICR_JOB_TYPE == JPI$K_NETWORK) || $VMS_STATUS_SUCCESS(stat)) {
		char		pwd[av->LGI$A_ICR_PWD1->dsc$w_length + 1];
		LGI_IPC_MSG	lim;

		/*
			In the case password has zero length we must return invalid password
			otherwise the password test below will succeed because without a password
			the LDAP bind will be done anonymously.
		*/

		if (av->LGI$A_ICR_PWD1->dsc$w_length == 0) { return LGI$_INVPWD; }

		strncpy(pwd, av->LGI$A_ICR_PWD1->dsc$a_pointer, av->LGI$A_ICR_PWD1->dsc$w_length);
		pwd[av->LGI$A_ICR_PWD1->dsc$w_length] = 0;

		strcpy(lim.request.authenticate.credential, pwd);

		lim.cmd = IPC_CMD_AUTHENTICATE;

		if (!(*ctx)->flags.localauth) {

			if ($VMS_STATUS_SUCCESS(stat = ipc_command((*ctx)->hndl, &lim))) {

				if ((*av->LGI$A_ICR_CREPRC_FLAGS & PRC$M_INTER) &&
				    *av->LGI$A_ICR_PWDCOUNT == 2) { /* account has secondary password */

					if (av->LGI$ICB_PASSWORD(1) != SS$_NORMAL) {

						return LGI$_INVPWD; /* seconday password failed */

					}
				}

				return LGI$_SKIPRELATED; /* password(s) matched against password in SYSUAF.DAT. */
			}
		}

		/*
			Either the user requested local password verification 
			or something else went wrong with the LDAP password verification.
		*/

		if (((*ctx)->flags.localauth || stat == SS$_INVSECDOMAIN || stat == SS$_WRONGSTATE || stat == SS$_CONNECFAIL) &&
		    ((int unsigned *)(av->LGI$A_ICR_UAF_RECORD)) != NULL) {

			UAF		*uaf = (void *)*((int unsigned *)(av->LGI$A_ICR_UAF_RECORD));
			char		dummy[256];

			if ((*ctx)->flags.localauthok ||
			    uaf->uaf$q_priv & PRV$M_SYSPRV) { /* user must have SYSPRV to use local authentication */

				if ((stat = av->LGI$ICB_CHECK_PASS(av->LGI$A_ICR_PWD1, uaf, 0)) == SS$_NORMAL) {

					if ((*av->LGI$A_ICR_CREPRC_FLAGS & PRC$M_INTER) &&
					    *av->LGI$A_ICR_PWDCOUNT == 2) { /* account has secondary password */

						if (av->LGI$ICB_PASSWORD(1) != SS$_NORMAL) {

							return LGI$_INVPWD; /* secondary password failed */

						}
					}

					return LGI$_SKIPRELATED; /* password(s) matched against password in SYSUAF.DAT. */
				}

			} else {

				return LGI$_NOLOCAUTH;

			}
		}
	}
                      
	return LGI$_INVPWD;
}
