/*
        Copyright (C) 2004 Jonas Lindholm

        This software was developed by Jonas Lindholm, jlhm@usa.net

        History

        V1.0            Jonas Lindholm  2004-05-14

        This program is free software; you can redistribute it and/or modify
        it under the terms of the GNU General Public License as published by
        the Free Software Foundation; either version 2 of the License, or
        (at your option) any later version.

        This program is distributed in the hope that it will be useful,
        but WITHOUT ANY WARRANTY; without even the implied warranty of
        MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
        GNU General Public License for more details.

        You should have received a copy of the GNU General Public License
        along with this program; if not, write to the Free Software
        Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
 
#include <starlet.h>
#include <descrip.h>
#include <ssdef.h>
#include <stsdef.h>
#include <iccdef.h>
#include <builtins.h>

#include "lgildap.h"

typedef struct lgi_ctx_s {
        int unsigned            reqid;
	int unsigned		conn;
        int unsigned		iosb[6];
	int unsigned		alloc[2];
} LGI_CTX;


/*
	Description:

	This routine is invoked when the LGI LDAP Server does not respond in 10 seconds.
	The  ICC connection is terminated and IOSB status is set to SS$_LINKDISCON.
	Also event flag 0 is set.

	INPUT:		context handler

	OUTPUT:		none
*/

void ipc_timeout(LGI_CTX *ctx) {
	int			stat;

	if (ctx->iosb[0] == 0) { /* only disconnect if the original ICC call not complete in time */

		stat = sys$icc_disconnect(ctx->conn, &ctx->iosb, 0, 0, 0, 0);

		ctx->conn = 0;
		ctx->iosb[0] = SS$_CONNECFAIL;

	}

	sys$setef(0);
}


/*
	Description:

	This routine is invoked when an ICC routine complete.
	It only set event flag 0.

	INPUT:		none

	OUTPUT:		none
*/

void set_ef() {

	sys$setef(0);

}


/*
	Description:

	This routine initialize the ICC communication with the LGI LDAP Server process
	on the local node.
	If the connection attempt take more than 10 seconds the connection fail.

	Memory is allocated for the context.

	Why is the context address swapped ? There is no reason more that the returned
	context value looks better when viewed as hex value, i.e. 0x80000057.

	INPUT:		pointer to conext handler

	OUTPUT:		updated context handler
*/

int ipc_init(int unsigned *c) {
	int			stat;
	int unsigned		alloc[2] = { 0, 0 };
	LGI_CTX			*ctx;

	if (_PROBEW(0, sizeof(c), (void *)c) != 1) {

		return SS$_ACCVIO;

	}

	*c = 0;

	stat = sys$expreg((sizeof(LGI_CTX) + 511) >> 9,  &alloc, 0, 0); 

	if ($VMS_STATUS_SUCCESS(stat)) {
		long long		timeout;
		$DESCRIPTOR(timoutdsc, "0 00:00:10");	/* give the LGI LDAP Server process 10 sec to reply */
	        $DESCRIPTOR(assnamdsc, "LGI_LDAP");
		$DESCRIPTOR(locdsc, "");


		ctx = (LGI_CTX *)alloc[0];
		ctx->alloc[0] = alloc[0];
		ctx->alloc[1] = alloc[1];
		ctx->conn = 0;
		ctx->reqid = 0;

		stat = sys$bintim(&timoutdsc, &timeout);

		if ($VMS_STATUS_SUCCESS(stat = sys$setimr(0, &timeout, &ipc_timeout, ctx, 0))) {

	        	stat = sys$icc_connect(&ctx->iosb, &set_ef, 0, ICC$C_DFLT_ASSOC_HANDLE, &ctx->conn, &assnamdsc, &locdsc,
					       0, 0, 0, 0, 0, 0, 0);

			if ($VMS_STATUS_SUCCESS(stat)) {

				stat = sys$synch(0, &ctx->iosb);

			}

			sys$cantim(ctx, 0);
		}

		*c = (int unsigned)((alloc[0] >> 16) | (alloc[0] << 16));

	}

	return stat;
}


int ipc_end(int unsigned c) {
	int			stat;
	LGI_CTX			*ctx = (LGI_CTX *)((c >> 16) | (c << 16));

	if (_PROBEW(0, sizeof(LGI_CTX), (void *)ctx) != 1) {

		return SS$_ACCVIO;

	}

	if (ctx->conn != 0) { sys$icc_disconnect(ctx->conn, &ctx->iosb, 0, 0, 0, 0); }

	return sys$deltva(&ctx->alloc, 0, 0);
}

/*
	Description:

	This routine send a command to the LGI LDAP Server process and wair for a response.
	If no response is received in 10 seconds the wait is interrupted and the connection
	with the LGI LDAP Server process is terminated.

	INPUT:		context value
			message data.

	OUTPUT:		updated message data
			status from message data area
			ICC status if something went wrong.
*/

int ipc_command(int unsigned c, LGI_IPC_MSG *msg) {
	int			stat;
	long long		timeout;
	LGI_CTX			*ctx = (LGI_CTX *)((c >> 16) | (c << 16));
	$DESCRIPTOR(timoutdsc, "0 00:00:10");	/* give the LGILDAP process 10 sec to reply */

	if (_PROBEW(0, sizeof(LGI_CTX), (void *)ctx) != 1 || _PROBEW(0, sizeof(LGI_IPC_MSG), (void *)msg) != 1) {

		return SS$_ACCVIO;

	}

	if (ctx->conn == 0) { return SS$_CONNECFAIL; } /* there is no ICC connection */

	ctx->iosb[0] = 0;

	stat = sys$bintim(&timoutdsc, &timeout);

	if ($VMS_STATUS_SUCCESS(stat = sys$setimr(0, &timeout, &ipc_timeout, ctx, 0))) {

		msg->reqid = ctx->reqid++;
		
		ctx->iosb[2] = (int unsigned)msg;
		ctx->iosb[3] = 0;
		ctx->iosb[4] = sizeof(LGI_IPC_MSG);

		stat = sys$icc_transceive(ctx->conn, &ctx->iosb, &set_ef, 0, (char *)msg, sizeof(LGI_IPC_MSG));

		if ($VMS_STATUS_SUCCESS(stat)) {

			stat = sys$synch(0, &ctx->iosb);

		}

		sys$cantim(ctx, 0);

		if ($VMS_STATUS_SUCCESS(stat)) {

			stat = ctx->iosb[0] & 0x0ffff;

			if ($VMS_STATUS_SUCCESS(stat)) {

				if (msg->rplyid == ctx->reqid - 1) {

					return msg->status;

				}
			}
		}
	}

	msg->cmd = IPC_CMD_ERROR;
	msg->status = SS$_INVSECDOMAIN;

	return stat;
}
