/*
	Copyright (C) 2004 Jonas Lindholm

	This software was developed by Jonas Lindholm, jlhm@usa.net

	History

	V1.0		Jonas Lindholm	2004-05-14

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <string.h>
#include <stdio.h>
#include <ctype.h>

#include <starlet.h>
#include <ssdef.h>
#include <stsdef.h>
#include <psldef.h>
#include <jpidef.h>
#include <uaidef.h>
#include <descrip.h>

#include <rms.h>
#include <rab.h>
#include <rabdef.h>
#include <prcdef.h>

#include <cli$routines.h>
#include <climsgdef.h>

#include "lgi.h"
#include "lgildap.h"
#include "private.h"

#define	UAF$S_USERNAME		32


/*
	Description:

	This routine ask an interactive user for username.
	The default username prompt is "Username (LDAP): " indicating that LDAP is in use.
	The username prompt can be override with logical name LGI_LDAP_USR_PROMPT.

	If the LGI LDAP Server is reachable the username given by the user is translated
	to a real username (uid) using the LDAP server.

	This routine return LGI$_SKIPRELATED as long the input contain
	valid qualifiers and a username.

	Note that username can contain spaces when LGI LDAP Server is reachable and the username
	can be translated to valid VMS username (uid) format.

	The qualifier LOCAL_PASSWORD is used with external authentication and specify if
	the password authentication should be done against the local password hash.
	However, this is only supported when the user has the external authentication flag
	set in SYSUAF. But when that flag is set no LGI callouts are done.
	The workaround is to check for the qualifier after we do the first parsing and
	if it exist remove it from the string and parse again.
	By default this functionality is disabled but can be enabled by defining
	LOCAL_PASSWORD when compiling LGI_IDENTITY.C

	INPUT:		LGI argument vector
			pointer to private data

	OUTPUT:		LGI$_SKIPRELATED as long input is valid.
			CLI$_xxx if the user has tried too many attempts with invalid syntax.
			LGI$_NOTVALID is the user did not input any valid username at all in
				15 retries.
*/

int do_GetUsername(struct LGI$ARG_VECTOR *av, CTX_T **ctx) {
	int		stat;
	int		cnt1 = 15;
	int		cnt2 = 15;
	char		prompt[256];
	char		userprompt[256];
	char		buff[256];
	struct RAB	*rab = av->LGI$A_ICR_INPUT_RAB;

	stat = do_trnlnm("LGI_LDAP_USR_PROMPT", userprompt, sizeof(userprompt) - 1, PSL$C_EXEC);

	sprintf(prompt, "\n\r%s", ($VMS_STATUS_SUCCESS(stat) ? userprompt : "Username (LDAP): "));

	rab->rab$w_usz = sizeof(buff);
	rab->rab$l_ubf = buff;
	rab->rab$b_psz = strlen(prompt);
	rab->rab$l_pbf = prompt;
	rab->rab$v_pmt = 1;
	rab->rab$v_rne = 0;

	while (cnt1-- > 0) {

		memset(buff, 0, sizeof(buff));
		stat = av->LGI$ICB_GET_INPUT(rab, 0);
		rab->rab$v_pta = 0;

		if (rab->rab$w_rsz > 0) {
			char		clistr[128] = "";
			char		parsestr[256];
			char		username[256];
			char		*p;
			LGI_IPC_MSG	lim;

			buff[rab->rab$w_rsz] = 0;

			memset(username, 0, 128);
			strncpy(username, buff, (strchr(buff, '/') == NULL ? strlen(buff) : strchr(buff, '/') - (buff)));

			strcpy(clistr, (buff + strlen(username)));

			for(p = clistr; *p; *p++ = toupper(*p));

			lim.cmd = IPC_CMD_IDENTITY;
			lim.request.identity.jobtype = *av->LGI$A_ICR_JOB_TYPE;

			strcpy(lim.request.identity.identity, username);

			if ($VMS_STATUS_SUCCESS(stat = ipc_command((*ctx)->hndl, &lim))) {

				sprintf(parsestr, "LOGIN %s %s", lim.reply.identity.identity, clistr);

				strncpy(av->LGI$A_ICR_USERNAME->dsc$a_pointer, lim.reply.identity.identity, UAF$S_USERNAME);
				av->LGI$A_ICR_USERNAME->dsc$w_length = strlen(lim.reply.identity.identity);

			} else {

				sprintf(parsestr, "LOGIN %s %s", username, clistr);

				strncpy(av->LGI$A_ICR_USERNAME->dsc$a_pointer, username, UAF$S_USERNAME);
				av->LGI$A_ICR_USERNAME->dsc$w_length = strlen(username);

			}

			$DESCRIPTOR(cliprmdsc, parsestr);
			cliprmdsc.dsc$w_length = strlen(parsestr);

			if ($VMS_STATUS_SUCCESS(stat = av->LGI$ICB_USERPARSE(&cliprmdsc))) {

#ifdef	LOCAL_PASSWORD
				char		*b = NULL;
				$DESCRIPTOR(locathdsc, "LOCAL_PASSWORD");

				/*
					Find position to /L and then the first space, / or EOS.
				*/

				for(p = strstr(clistr, "/L"), b = p; p++ && *p && *p != '/' && *p != ' ';);

				(*ctx)->flags.localauth = (b != NULL);
				if (b == NULL) { b = clistr; p = clistr; }

				/*
					Create new CLI string but without the /L[OCAL_PASSWORD] qualifier.
				*/

				sprintf(clistr, "%*.*s%*.*s", (b - clistr), (b - clistr), clistr,
					strlen(clistr) - (p - clistr), strlen(clistr) - (p - clistr), p);

				/*
					Create new parse string.
				*/

				sprintf(parsestr, "LOGIN %*.*s %s", av->LGI$A_ICR_USERNAME->dsc$w_length, av->LGI$A_ICR_USERNAME->dsc$w_length,
					av->LGI$A_ICR_USERNAME->dsc$a_pointer, clistr);

				/*
					Do new parsing but this time without any /LOCAL_PASSWORD
					qualifier.
				*/

				cliprmdsc.dsc$w_length = strlen(parsestr);
				av->LGI$ICB_USERPARSE(&cliprmdsc);
#endif

				return LGI$_SKIPRELATED;

			}

			if (cnt2-- == 0) { return stat; }
		}
	}

	return LGI$_NOTVALID;
}


/*
	This routine is invoked when verifying username from a DECnet login.

	It will always return LGI$_SKIPRELATED.

	If LGI LDAP Server is reachable the username given by the DEcnet login is target
	for translation to another username (uid) from LDAP server.

	INPUT:		LGI argument vector
			pointer to private data

	OUTPUT:		updated username if LGI LDAP Server is reachable
			LGI$_SKIPRELATED is always returned
*/

int do_NetUsername(struct LGI$ARG_VECTOR *av, CTX_T **ctx) {
	int		stat;
	LGI_IPC_MSG	lim;

	lim.cmd = IPC_CMD_IDENTITY;
	lim.request.identity.jobtype = *av->LGI$A_ICR_JOB_TYPE;

	strncpy(lim.request.identity.identity, av->LGI$A_ICR_USERNAME->dsc$a_pointer, av->LGI$A_ICR_USERNAME->dsc$w_length);
	lim.request.identity.identity[av->LGI$A_ICR_USERNAME->dsc$w_length] = 0;

	if ($VMS_STATUS_SUCCESS(ipc_command((*ctx)->hndl, &lim))) {

		strncpy(av->LGI$A_ICR_USERNAME->dsc$a_pointer, lim.reply.identity.identity, UAF$S_USERNAME);
		av->LGI$A_ICR_USERNAME->dsc$w_length = strlen(lim.reply.identity.identity);

	}

	return LGI$_SKIPRELATED;
}


/*
	Description:

	This routine ask and translate given username for interactive and DECnet login.

	INPUT:		LGI argument vector
			pointer to private data

	OUTPUT:		SS$_NORMAL for non interactive or DECnet login.
			SS$_NORMAL if local verification is requested.
			SS$_NORMAL for proxy login.
			LGI$_NOTVALID is user never gives any username
			LGI$_SKIPRELATED for everything else.
*/

int cb_identify(struct LGI$ARG_VECTOR *av, CTX_T **ctx) {
	int			stat;
	char			username[128];

	if (!(*av->LGI$A_ICR_CREPRC_FLAGS & PRC$M_INTER) &&
	    !(*av->LGI$A_ICR_CREPRC_FLAGS & PRC$M_NETWRK)) return(SS$_NORMAL);  /* Not interactive nor network, do normal processing */

	if (*av->LGI$A_ICR_CREPRC_FLAGS & PRC$M_NOPASSWORD) return(SS$_NORMAL);  /* Invoked as logged in, don't prompt */
 
	if (*av->LGI$A_ICR_SUBPROCESS != 0) return(SS$_NORMAL);  /* Don't prompt on subprocesses */

	if (ctx != NULL && *ctx != NULL && !(*ctx)->flags.dolocal) {

		if (*av->LGI$A_ICR_JOB_TYPE != JPI$K_NETWORK) {

			do_GetUsername(av, ctx);

		} else {

			if (*av->LGI$A_ICR_NETFLAGS & 1) { return SS$_NORMAL; } /* proxy login */

			do_NetUsername(av, ctx);

		}

		return LGI$_SKIPRELATED;

	}

	return SS$_NORMAL;
}
