/*****************************************************************************/
/*
                                PHPwasd.c

                      PHP: Hypertext Preprocessor: WASD

A large element of this code is "monkey-see-monkey-do", but it seems to work!
This is *not* a port or implementation of PHP on VMS.  It is a PHP 'interface',
the part of PHP that talks to the Web server.  WASD CGILIB.C library has not
been used here because this is only an 'interface' to the actual script
processor PHP, and only the CGI variable code would actually be employed, so
this has been provided as a stand-alone (somewhat tailored) function.


PHPWASD COPYRIGHT
-----------------
Copyright (C) 2002-2004 Mark G.Daniel

This code module along with sections of this code are based on code examples,
actual code and of course the underlying PHP engine, which are copyright by The
PHP Group.

   +----------------------------------------------------------------------+
   | PHP version 4.0                                                      |
   +----------------------------------------------------------------------+
   | Copyright (c) 1997-2001 The PHP Group                                |
   +----------------------------------------------------------------------+
   | This source file is subject to version 2.02 of the PHP license,      |
   | that is bundled with this package in the file LICENSE, and is        |
   | available at through the world-wide-web at                           |
   | http://www.php.net/license/2_02.txt.                                 |
   | If you did not receive a copy of the PHP license and are unable to   |
   | obtain it through the world-wide-web, please send a note to          |
   | license@php.net so we can mail you a copy immediately.               |
   +----------------------------------------------------------------------+
   | Authors: various                                                     |
   +----------------------------------------------------------------------+


MAPPING RULES
-------------
There are various ways to employ the WASD PHP interpreter.  It can be used in
vanilla CGI mode, or in persistent CGIplus/RTE mode.  Benchmarking indicates
the CGIplus/RTE use reduces activation time to 10% of CGI (yes, 10x).  There
are subtle differences in the way CGIplus and RTE parse and provide the
PATH_INFO data.  See the "WASD Scripting Overview" for more detail.  The
following rules require the PHP script files to be located in the site
administrator controlled /cgi-bin/ path.  This is of course the most secure.

  # HTTPD$MAP for RTE usage
  # this configuration probable works as well as any
  map /cgi-bin/*.php* /php-bin/*.php*
  exec+ /php-bin/* (cgi-bin:[000000]phpwasd.exe)/cgi-bin/* \
     script=query=relaxed

or

  # HTTPD$MAP for CGI, CGIplus or RTE usage (perhaps for comparison)
  exec+ /php-bin/* (cgi-bin:[000000]phpwasd.exe)/cgi-bin/* \
     script=query=relaxed
  ..
  exec /cgi-bin/* /cgi-bin/*
  exec+ /cgiplus-bin/* /cgi-bin/*

The following rules allow .PHP type files anywhere in the mapped directory
structure to be executed.  This means that any document author can script using
PHP.  This may be what is desired but can be dangerous.  PHP provides for this
type of usage.  Please familiarise yourself with it's requirements and
controls.  As an additional safeguard it is suggested that PHP scripts be
executed under a non-server account using the WASD PERSONA capabilities (see
the Technical Overview if unfamiliar with this functionality and
configuration).

  set /web/*.php* script=as=OTHER-ACCOUNT
  exec+ /web/*.php* /web/*.php*

For scripts requiring extended file specification (and located on ODS-5
volumes) the script path needs to be mapped as ODS-5.

  # HTTPD$MAP for RTE usage for extended file specification
  # (a minimum of WASD 8.4.2 is required for this to work fully)
  exec+ /php-bin/* (cgi-bin:[000000]phpwasd.exe)/cgi-bin/* \
     script=query=relaxed ods=5

The engine will by default chdir() to the a U**x syntax equivalent of the
directory containing the PHP script file.  It also setenv()s the environment
variable PATH to this same string.  This location may be explicitly provided
using the value of CGI variable SCRIPT_DEFAULT and set on a per-script or
general basis using the mapping rule 'script=default=<string>'.  It will accept
either VMS and U**x specifications depending on the requirements of the script
itself.

  set /php-bin/mumble.php* script=default="/mumble_device/000000"
  set /php-bin/mumble.php* script=default="mumble_device:[000000]"



OTHER CONFIGURATION
-------------------

  # HTTPD$CONFIG
  [ScriptRunTime]
  .PHP $CGI-BIN:[000000]PHPWASD.EXE
  [AddType]
  .INI   text/plain  initialization file
  .PHP   text/plain  PHP source
  .PHPS  text/plain  PHP source
  .PHTML text/plain  PHP source


WARNING!
--------
Don't forget that when using persistant environments such as CGIplus/RTE,
especially during development, once changes have been made to source code and
the environment rebuilt any currently executing instances of the previous build
must be purged from the server environment (wish I had a dollar for every time
I'd been caught like this myself!)
 
  $ HTTPD/DO=DCL=PURGE


SPECIAL QUERIES
---------------
The are a number of reserved query strings that may be used for purposes other
than script exection.  These are introduced with a query string prefix of
"!wasd=" followed by a keyword

  o  info ... display PHP information
  o  lint ... run a PHP syntax checker over the script
  o  syntax ... display the script with highlighted syntax

For example

  http://the.host.name/cgi-bin/script.php?!wasd=info
  http://the.host.name/cgi-bin/script.php?!wasd=lint
  http://the.host.name/cgi-bin/script.php?!wasd=syntax

As of PHPWASD v1.1 the environment variable PHPWASD$QUERIES must exist to
enable this facility.  This can be done on a site-wide basis using

  $ DEFINE /SYSTEM PHPWASD$QUERIES 1


AUTHORIZATION GLOBAL VARIABLES
------------------------------
If HTTPD$AUTH is configured to required EXTERNAL authorization the Apache
mod_php authorization variables PHP_AUTH_PW, PHP_AUTH_TYPE and PHP_AUTH_USER
are created from the WASD CGI variables AUTH_PASSWORD, AUTH_TYPE and
REMOTE_USER respectively.  Something like ...

  # HTTPD$AUTH
  [EXTERNAL]
  /cgi-bin/script-name.php* r+w
  # or perhaps triggered with a path
  /cgi-bin/script-name.php/path/trigger/* r+w

or to suppress the server generation of the 401 response field and allow the
script to generate it's own

  # HTTPD$AUTH
  [EXTERNAL]
  /cgi-bin/script-name.php* r+w,param="/NO401"
  # or perhaps triggered with a path
  /cgi-bin/script-name.php/path/trigger/* r+w,param="/NO401"


LOGICAL NAMES
-------------
PHPWASD$DBUG     turns on all "if (Dbug)" statements
PHPWASD$INI      provide an override path for the PHP.INI file
PHPWASD$QUERIES  allows the likes of "?wasd=info" (see above)
PHPWASD$TYPE     comma separated list of allowed PHP script types
                 (e.g. ".PHP" or ".PHP,.PHTML")
PHPWASDSHR       sharable image containing PHP engine


BUILD DETAILS
-------------
$ @BUILD_PHPWASD BUILD  !compile+link
$ @BUILD_PHPWASD LINK   !link-only


VERSION HISTORY (update SOFTWAREVN as well!)
---------------
27-FEB-2004  MGD  V1.2.1, chdir() to script or SCRIPT_DEFAULT location
14-FEB-2004  MGD  v1.2.0, CSWS PHP 1.2 (PHP 4.3.2),
                          minor conditional mods to support IA64
19-APR-2003  MGD  v1.1.2, SapiRegisterCgiVariables() in line with WASD v8.2,
                          SapiSendHeader() save a few cycles with WASD v8.2ff
                          PrePhpError() use "Script-Control:" with WASD v8.2ff
                          remove PhpIniKludge() after PHP 1.1 patch released,
                          bugfix; SapiRegisterCgiVariables() gateway variables
10-JAN-2003  MGD  v1.1.1, PhpIniKludge() - see note with function!
26-DEC-2002  MGD  v1.1.0, build against CSWS PHP v1.1 source
28-JAN-2002  MGD  v1.0.1, SapiSendHeader() buffer and reorder response
                          header fields to make them CGI-compliant for WASD
16-JAN-2002  MGD  v1.0.0, initial development
*/
/*****************************************************************************/

#define SOFTWAREVN "1.2.1"
#define SOFTWARENM "PHPWASD"
#ifdef __ALPHA
#  define SOFTWAREID SOFTWARENM " AXP-" SOFTWAREVN
#endif
#ifdef __ia64
#  define SOFTWAREID SOFTWARENM " IA64-" SOFTWAREVN
#endif
#ifdef __VAX
#  define SOFTWAREID SOFTWARENM " VAX-" SOFTWAREVN
#endif

/************/
/* includes */
/************/

/* standard C header files */

#include <ctype.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <unixlib.h>
#include <unixio.h>

/* VMS-specific header files */

#include <descrip.h>
#include <ssdef.h>

/* PHP-specific includes */

/*
Based on a source kit CSWS_PHP-V0102-SRC.BCK dated 6-NOV-2003.
Just enough of the PHP build environment to compile PHPWASD is used here.
*/

#include "zend.h"
#include "zend_api.h"
#include "zend_compile.h"
#include "zend_highlight.h"
#include "zend_modules.h"

#define PHP_FUNCTION ZEND_FUNCTION
#define PHPAPI
#define TLS_C
#define TLS_CC
typedef zval pval;

#include "info.h"
#include "sapi.h"

/* required prototypes */
int php_body_write(const char *str, uint str_length TSRMLS_DC);
int php_execute_script(zend_file_handle *primary_file TSRMLS_DC);
void php_get_highlight_struct (zend_syntax_highlighter_ini
                               *syntax_highlighter_ini);
int php_handle_auth_data(const char *auth TSRMLS_DC);
void php_handle_aborted_connection(void);
int php_lint_script(zend_file_handle *file TSRMLS_DC);
int php_module_startup (sapi_module_struct *sf,
                        zend_module_entry *additional_modules,
                        unsigned int num_additional_modules);
int php_module_shutdown_wrapper(sapi_module_struct *sapi_globals);
int php_printf(const char *format, ...);
void php_register_variable (char *var, char *val,
                            pval *track_vars_array TSRMLS_DC);
int php_request_startup(TSRMLS_D);
void php_request_shutdown(void *dummy);

#define PUTS(str) do { \
   const char *__str = (str); \
   php_body_write(__str, strlen(__str) TSRMLS_CC); \
} while (0)

/**********/
/* macros */                          
/**********/

#define FI_LI __FILE__, __LINE__

/* comma-separated list of file types allowed to be interpreted */
#define DEFAULT_PHP_TYPE_LIST ".PHP,.PHTML"

/* includes the code for "!wasd=info", etc. */
#define ALLOW_WASD_QUERIES 1

/* includes the code to check for a ".PHP" file type, etc. */
#define CHECK_FILE_TYPE 1

/* size of buffer for PHP output */
#define WRITE_BUFFER_SIZE 4096

/* initial size and increment of response header buffer */
#define SEND_HEADER_BUFFER_CHUNK 1024

/******************/
/* global storage */
/******************/

/* change to 0 to remove all (Dbug) code from executable */
#define DBUG 1
#if DBUG
int  Dbug;
#else
#define Dbug 0
#endif

int  IsCgiPlus,
     ServerVersion,
     UsageCount;

char  *CgiPlusEofPtr,
      *CgiPlusEotPtr,
      *CgiPlusEscPtr,
      *PhpTypeListPtr;

/**************/
/* prototypes */
/**************/

int sys$fao (void*, unsigned short*, void*, ...);
int sys$faol (void*, unsigned short*, void*, unsigned long*);

char* CgiVar (char*);
static void InitRequestInfo ();
unsigned int lib$get_symbol (void*, void*, unsigned short*, unsigned long*);
int ConfiguredFileType (char*);
char* ImageIdent();
void ProcessRequest ();
int PrePhpError (char*, ...);

static int SapiDeactivate (SLS_D);
static void SapiFlush (void*);
static char *SapiReadCookies (SLS_D);
static int SapiReadPost (char*, uint SLS_DC);
static void SapiRegisterVariables (zval* ELS_DC SLS_DC PLS_DC);
static void SapiRegisterCgiVariables (zval* ELS_DC SLS_DC PLS_DC);
static void SapiSendHeader (sapi_header_struct*, void*);
static int SapiWrite (const char*, uint);

/*****************************/
/* PHP SAPI module structure */
/*****************************/

static sapi_module_struct SapiModule =

{
   "phpwasd",                    /* name */
   SOFTWAREID,                   /* pretty name */
   /* cast to void because of an annoying prototype mismatch */
   (void*)php_module_startup,    /* startup */
   php_module_shutdown_wrapper,  /* shutdown */
   NULL,                         /* activate */
   SapiDeactivate,               /* deactivate */
   SapiWrite,                    /* unbuffered write */
   SapiFlush,                    /* flush stdout */
   NULL,                         /* get uid */
   NULL,                         /* getenv */
   NULL,                         /* error handler */
   NULL,                         /* header handler */
   NULL,                         /* send header */
   SapiSendHeader,               /* send header handler */
   SapiReadPost,                 /* read POSTed data */
   SapiReadCookies,              /* read cookies */
   SapiRegisterVariables,        /* register variables */
   NULL,                         /* log message */
   NULL,                         /* php.ini path override */
   NULL,                         /* block interruptions */
   NULL,                         /* unblock interruptions */
   NULL,                         /* default post reader */
};

/*****************************************************************************/
/*
*/

main (int argc, char *argv[])
       
{
   char  *cptr;

   /*********/
   /* begin */
   /*********/

#if DBUG
   if (Dbug = ((getenv ("PHPWASD$DBUG")) != NULL))
      fprintf (stdout, "Content-Type: text/plain\n\n");
#endif

#if CHECK_FILE_TYPE

   if (!(PhpTypeListPtr = getenv ("PHPWASD$TYPE")))
      PhpTypeListPtr = DEFAULT_PHP_TYPE_LIST;

#endif /* CHECK_FILE_TYPE */

   /* if it doesn't look like CGI environment then forget it */
   if (!getenv ("HTTP$INPUT")) exit (SS$_ABORT);

   if (!Dbug)
   {
      /* binary mode to eliminate carriage-control */
      if (!(stdin = freopen ("HTTP$INPUT:", "r", stdin, "ctx=bin")))
         exit (vaxc$errno);
      if (!(stdout = freopen ("SYS$OUTPUT:", "w", stdout, "ctx=bin")))
         exit (vaxc$errno);
      if (!(stderr = freopen ("SYS$OUTPUT:", "w", stderr, "ctx=bin")))
         exit (vaxc$errno);
   }

   IsCgiPlus = ((CgiPlusEofPtr = getenv("CGIPLUSEOF")) != NULL);
   CgiPlusEotPtr = getenv("CGIPLUSEOT");
   CgiPlusEscPtr = getenv("CGIPLUSESC");

   sapi_startup (&SapiModule);

   cptr = getenv("PHPWASD$INI");
   if (Dbug) fprintf (stdout, "PHPWASD$INI |%s|\n", cptr ? cptr : "(null)");
   if (cptr) SapiModule.php_ini_path_override = strdup(cptr);

   if (php_module_startup (&SapiModule, NULL, 0) == FAILURE)
      exit (SS$_BUGCHECK);
   SG(server_context) = (void*)1;

   if (IsCgiPlus)
   {
      for (;;)
      {
         if (Dbug) fprintf (stdout, "Content-Type: text/plain\n\n");
         /* block waiting for the first/next request */
         CgiVar ("");
         ProcessRequest ();
         fflush (stdout);
         fputs (CgiPlusEofPtr, stdout);
         fflush (stdout);
      }
   }
   else
   {
      if (Dbug) fprintf (stdout, "Content-Type: text/plain\n\n");
      ProcessRequest ();
   }

   exit (SS$_NORMAL);
}

/*****************************************************************************/
/*
Process a single CGI/CGIplus/RTE request.
All request processing occurs within this function.
*/

void ProcessRequest ()
       
{
   int  status,
        PHPinfo,
        PHPlint,
        PHPsyntax;
   char  *cptr, *sptr, *zptr,
         *QueryStringPtr,
         *ScriptDefaultPtr,
         *ScriptFileNamePtr;
   const char  *auth;
   char  ScriptDefault [256];
   FILE  *fp;
   zend_file_handle  ZendFileHandle;

   /*********/
   /* begin */
   /*********/

   if (Dbug) fprintf (stdout, "ProcessRequest()\n");

   UsageCount++;

   if (!ServerVersion)
   {
      cptr = CgiVar ("SERVER_SIGNATURE");
      if (cptr)
      {
         /* should be something like "HTTPd-WASD/8.2.1 OpenVMS/AXP SSL" */
         while (*cptr && !isdigit(*cptr)) cptr++;
         ServerVersion = atoi(cptr) * 10000;
         while (*cptr && isdigit(*cptr)) cptr++;
         if (*cptr) cptr++;
         ServerVersion += atoi(cptr) * 100;
         while (*cptr && isdigit(*cptr)) cptr++;
         if (*cptr) cptr++;
         ServerVersion += atoi(cptr);
         /* resulting in a number like 80,201 */
         if (ServerVersion < 80000) ServerVersion = 0;
      }
   }

   ScriptFileNamePtr = CgiVar ("SCRIPT_FILENAME");

#if CHECK_FILE_TYPE

   if (!ConfiguredFileType (ScriptFileNamePtr))
   {
      PrePhpError ("Script file type has not been configured as PHP.");
      return;
   }

#endif /* CHECK_FILE_TYPE */

   if (!(fp = fopen (ScriptFileNamePtr, "r", "shr=get,put")))
   {
      PrePhpError ("Cannot access script, !AZ!AZ.",
                   strerror(errno, vaxc$errno),
                   vaxc$errno == 98962 ? " (protection?)" : ".");
      return;
   }

   ScriptDefaultPtr = CgiVar ("SCRIPT_DEFAULT");
   if (ScriptDefaultPtr)
      status = chdir (cptr = ScriptDefaultPtr);
   else
   {
      zptr = (sptr = ScriptDefault) + sizeof(ScriptDefault)-1;
      for (cptr = ScriptFileNamePtr; *cptr && sptr < zptr; *sptr++ = *cptr++);
      sptr--;
      while (sptr > ScriptDefault && *sptr != ']') sptr--;
      if (*sptr == ']') sptr++;
      *sptr = '\0';
      cptr = decc$translate_vms (ScriptDefault);
      if ((int)cptr > 0)
         status = chdir (cptr);
      else
      {
         status = vaxc$errno;
         cptr = ScriptDefault;
      }
   }
   if (status)
   {
      status = vaxc$errno;
      PrePhpError ("Cannot chdir(&quot;!AZ&quot), !AZ.",
                   cptr, strerror(errno, status));
      return;
   } 
   status = setenv ("PATH", cptr, 1);
   if (status)
   {
      status = vaxc$errno;
      PrePhpError ("Cannot setenv(&quot;PATH&quot;,&quot;!AZ&quot), !AZ.",
                   cptr, strerror(errno, status));
      return;
   }

   QueryStringPtr = CgiVar ("QUERY_STRING");

   SG(request_info).argv0 = ScriptFileNamePtr;		       
   SG(request_info).request_method = CgiVar("REQUEST_METHOD");
   SG(request_info).request_uri = CgiVar("REQUEST_URI");
   SG(request_info).query_string = QueryStringPtr;
   cptr = CgiVar("CONTENT_TYPE");
   SG(request_info).content_type = (cptr ? cptr : "" );
   cptr = CgiVar("CONTENT_LENGTH");
   SG(request_info).content_length = (cptr ? atoi(cptr) : 0);
   /* assume there's a problem to start with */
   SG(sapi_headers).http_response_code = 502;
   
   if (auth = CgiVar("HTTP_AUTHORIZATION"))
      php_handle_auth_data (auth SLS_CC);

   if (php_request_startup (CLS_C ELS_CC PLS_CC SLS_CC) == FAILURE)
   {
      PrePhpError ("PHP request startup has failed!!");
      return;
   }

   PHPinfo = PHPsyntax = PHPlint = 0;

#if ALLOW_WASD_QUERIES

   if (QueryStringPtr[0] == '!')
   {
      if (getenv ("PHPWASD$QUERIES"))
      {
         if (!strncmp (QueryStringPtr, "!wasd=info", 10)) PHPinfo = 1;
         if (!strncmp (QueryStringPtr, "!wasd=lint", 10)) PHPlint = 1;
         if (!strncmp (QueryStringPtr, "!wasd=syntax", 12)) PHPsyntax = 1;
      }
   }

   if (PHPinfo)
   {
      int flag = PHP_INFO_GENERAL | PHP_INFO_CREDITS |
                 PHP_INFO_CONFIGURATION | PHP_INFO_MODULES |
                 PHP_INFO_ENVIRONMENT | PHP_INFO_VARIABLES |
                 PHP_INFO_LICENSE;
      php_print_info (flag);
      php_request_shutdown (NULL);
      return;
   }

#endif /* ALLOW_WASD_QUERIES */

   /* note in [.SAPI]CGI_MAIN.C explains the estrdup() */
   SG(request_info).path_translated = estrdup (ScriptFileNamePtr);

   ZendFileHandle.type = ZEND_HANDLE_FP;
   ZendFileHandle.handle.fd = 0;
   ZendFileHandle.handle.fp = fp;
   ZendFileHandle.filename = SG(request_info).path_translated;
   ZendFileHandle.free_filename = 0;
   ZendFileHandle.opened_path = NULL;

   /* OK, we'll assume everythings OK from here (even if it's not) */
   SG(sapi_headers).http_response_code = 200;

   if (PHPsyntax)
   {
      zend_syntax_highlighter_ini ZendHighlight;

      if (open_file_for_scanning (&ZendFileHandle CLS_CC) == SUCCESS)
      {
         php_get_highlight_struct (&ZendHighlight);
         zend_highlight (&ZendHighlight);
      }
   }
   else
   if (PHPlint)
   {
      cptr = CgiVar("SCRIPT_NAME");
      php_printf ("<HTML>\n<HEAD>\n<TITLE>Lint %s</TITLE>\n</HEAD>\n<BODY>\n",
                  cptr);
      if (php_lint_script (&ZendFileHandle CLS_CC ELS_CC PLS_CC) == SUCCESS)
         PUTS ("<B>lint PASSED</B> ... ");
      else
         PUTS ("<B>lint <FONT COLOR=\"#ff0000\">FAILED</FONT></B> ... ");
      php_printf ("<A HREF=\"%s?!wasd=syntax\">syntax</A>\n</BODY>\n</HTML>",
                  cptr);
   }
   else
   {
      /* restore the *real* translated path (see above) */
      SG(request_info).path_translated = CgiVar("PATH_TRANSLATED");

      php_execute_script (&ZendFileHandle CLS_CC ELS_CC PLS_CC);
   }

   php_request_shutdown (NULL);
}

/*****************************************************************************/
/*
There's gotta be an easier way to get shareable image information than this!
Some quick-and-dirty image analysis.  Open the PHPWASDSHR shareable image and
read selected fields to get the image identification, build, and linking date.
*/ 

char* ImageIdent ()

{
#ifdef __ia64
   /* ever'thin's null-terminated in this brave new world? */
   static $DESCRIPTOR (FormatFaoDsc, "!AZ, !AZ!AZ!%D");
#else
   static $DESCRIPTOR (FormatFaoDsc, "!AC, !AC!AZ!%D");
#endif
   static char ImageIdent [48] = "?";

   int  status;
   unsigned short  ShortLen;
   unsigned long  *ImageDatePtr;
   char  *ImageIdentPtr,
         *ImageNamePtr,
         *PackagePtr;
   char  ImageRecord [512];
   FILE  *ImageFile;
   $DESCRIPTOR (ImageIdentDsc, ImageIdent);

   /*********/
   /* begin */
   /*********/

   if (Dbug) fprintf (stdout, "ImageIdent()\n");

   if (ImageIdent[0] != '?') return (ImageIdent);

   ImageFile = fopen ("PHPWASDSHR", "r", "shr=get");
   if (!ImageFile)
   {
      fclose (ImageFile);
      return ("error: fopen()");
   }
   if (!fread (&ImageRecord, sizeof(ImageRecord), 1, ImageFile))
   {
      fclose (ImageFile);
      return ("error: fread()");
   }

#ifdef __ALPHA
   ImageNamePtr = ImageRecord + 200;
   ImageIdentPtr = ImageRecord + 240;
   ImageDatePtr = (unsigned long*)(ImageRecord + 192);
#endif
#ifdef __ia64
   /* absolutely no idea on IA64 (some would say not only) */
   for (;;)
   {
      if (!fread (&ImageRecord, sizeof(ImageRecord), 1, ImageFile))
      {
         fclose (ImageFile);
         return ("error: fread()");
      }
      /* check this record for some key strings */
      if (memcmp (ImageRecord+24, "IPF/VMS", 7)) continue;
      if (memcmp (ImageRecord+72, "IPF/VMS", 7)) continue;
      ImageNamePtr = ImageRecord + 80;
      ImageIdentPtr = ImageRecord + 128;
      ImageDatePtr = ImageRecord + 168;
      break;
   }
#endif
#ifdef __VAX
   /* and yes I know PHP is not available on VAX! */
   ShortLen = *(unsigned short*)ImageRecord;
   ImageNamePtr = ImageRecord + ShortLen - 80;
   ImageIdentPtr = ImageRecord + ShortLen - 40;
   ImageDatePtr = ImageRecord + ShortLen - 24;
#endif

   fclose (ImageFile);

   status = sys$fao (&FormatFaoDsc, &ShortLen, &ImageIdentDsc,
                     ImageIdentPtr, ImageNamePtr,
                     ImageNamePtr[0] ? ", " : "", ImageDatePtr);
   if (status & 1) ImageIdent[ShortLen] = '\0';
   if (Dbug) fprintf (stdout, "|%s|\n", ImageIdent);
   return (ImageIdent);
}                                   

/*****************************************************************************/
/*
Generate a standard WASD-like error message *before* PHP gets up an running.
WASD 8.2 or later, use "Script-Control:".
*/

int PrePhpError
(
char *FormatString,
...
)
{
   static $DESCRIPTOR (FaoDsc,
"Status: 502\r\n\
\r\n\
<HTML>\n\
<HEAD>\n\
<META NAME=\"generator\" CONTENT=\"!AZ (!AZ)\">\n\
<META NAME=\"environment\" CONTENT=\"!AZ\">\n\
<TITLE>ERROR 502</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<FONT SIZE=+1><B>ERROR 502</B> &nbsp;-&nbsp; Bad Gateway</FONT>\n\
<P>!AZ\n\
!AZ!AZ!AZ\
</BODY>\n</HTML>\n");

   int  argcnt, status;
   unsigned short  ShortLen;
   char  *cptr, *sptr;
   char  Buffer [2048],
         FaoBuffer [1024],
         Format [256];
   $DESCRIPTOR (BufferDsc, Buffer);
   $DESCRIPTOR (FaoBufferDsc, FaoBuffer);
   $DESCRIPTOR (FormatStringDsc, "");
   unsigned long  *vecptr;
   unsigned long  FaoVector [32];
   va_list  argptr;

   /*********/
   /* begin */
   /*********/

   va_count (argcnt);

   if (Dbug) fprintf (stdout, "PrePhpError() %d |%s|\n", argcnt, FormatString);

   if (ServerVersion >= 80200)
   {
      /* WASD 8.2 or later */
      vecptr = FaoVector;
      va_start (argptr, FormatString);
      for (argcnt -= 1; argcnt; argcnt--)
         *vecptr++ = (unsigned long)va_arg (argptr, unsigned long);
      va_end (argptr);

      FormatStringDsc.dsc$a_pointer = FormatString;
      FormatStringDsc.dsc$w_length = strlen(FormatString);
      status = sys$faol (&FormatStringDsc, &ShortLen, &BufferDsc,
                         (unsigned long*)&FaoVector);
      if (Dbug) fprintf (stdout, "sys$fao() %%X%08.08X\n", status);
      if (!(status & 1)) return (status);
      Buffer[ShortLen] = '\0';

      fprintf (stdout,
"Status: 502\r\n\
Script-Control: X-error-text=\"%s\"\r\n\
Script-Control: X-error-module=\"PHPWASD\"\r\n\
Script-Control: X-error-line=%d\r\n\
\r\n",
               Buffer, __LINE__);
   }
   else
   {
      if (!(cptr = CgiVar("SERVER_SOFTWARE"))) cptr = "?";
      sptr = CgiVar ("SERVER_SIGNATURE");
      status = sys$fao (&FaoDsc, &ShortLen, &FaoBufferDsc,
                  SOFTWAREID, ImageIdent(), cptr, FormatString,
                  sptr ? "<P><HR WIDTH=85%% ALIGN=left SIZE=2 NOSHADE>\n" : "",
                  sptr ? sptr : "",
                  sptr ? "\n" : "");
      if (Dbug) fprintf (stdout, "sys$fao() %%X%08.08X\n", status);
      if (!(status & 1)) return (status);
      FaoBuffer[FaoBufferDsc.dsc$w_length = ShortLen] = '\0';

      vecptr = FaoVector;
      va_start (argptr, FormatString);
      for (argcnt -= 1; argcnt; argcnt--)
         *vecptr++ = (unsigned long)va_arg (argptr, unsigned long);
      va_end (argptr);

      status = sys$faol (&FaoBufferDsc, &ShortLen, &BufferDsc,
                         (unsigned long*)&FaoVector);
      if (Dbug) fprintf (stdout, "sys$fao() %%X%08.08X\n", status);
      if (!(status & 1)) return (status);
      Buffer[ShortLen] = '\0';

      fputs (Buffer, stdout);
   }

   return (status);
}

/*****************************************************************************/
/*
Check if the type (extension) of the supplied file name can be found in the
list of types allowed to be interpreted as PHP.  Return true if allowed.
*/

#if CHECK_FILE_TYPE

int ConfiguredFileType (char *FileNamePtr)

{
   char  *cptr, *sptr, *tptr;

   /*********/
   /* begin */
   /*********/

   if (Dbug) fprintf (stdout, "ConfiguredFileType() |%s|\n", FileNamePtr);

   /* is it a permitted file type? */
   for (cptr = FileNamePtr; *cptr; *cptr++);
   while (cptr > FileNamePtr && *cptr != '.') cptr--;
   sptr = PhpTypeListPtr;
   while (*sptr)
   {
      tptr = cptr;
      while (*tptr && *sptr && toupper(*tptr) == toupper(*sptr))
      {
         tptr++;
         sptr++;
      }
      if (!*tptr && (!*sptr || *sptr == ',')) break;
      while (*sptr && *sptr != ',') sptr++;
      if (*sptr) sptr++;
   }   
   if (!*tptr && (!*sptr || *sptr == ',')) return (1);
   return (0);
}

#endif /* CHECK_FILE_TYPE */

/*****************************************************************************/
/*
Called by PHP as the request is finalized.
*/

static int SapiDeactivate (SLS_D)

{
   /*********/
   /* begin */
   /*********/

   if (Dbug) fprintf (stdout, "SapiDeactivate()\n");

   /* flush anything in the SapiWrite() buffer */
   SapiWrite (NULL, 0);

   return (SUCCESS);
}

/*****************************************************************************/
/*
In common with many U**x sourced utilities PHP has a habit of outputting only a
few characters at a time (often a single character).  The DECC RTL turns each
one of these fwrite()s into a $QIO ... slow and inefficient.  Create our own
buffering here.
*/

static int SapiWrite
(
const char *String,
uint StringLength
)
{
   static char  *sptr, *zptr;
   static char  Buffer [WRITE_BUFFER_SIZE];
   
   size_t ret;
   uint cnt;
   char *cptr;
   
   /*********/
   /* begin */
   /*********/

   if (Dbug)
      fprintf (stdout, "SapiWrite() %d %d\n|%*.*s|\n",
               String, StringLength, StringLength, StringLength, String);

   /* initialize static buffer space if necessary */
   if (!zptr) zptr = (sptr = Buffer) + sizeof(Buffer);
   
   if (!String)
   {
      /* explicit flush buffer */
      if (sptr == Buffer) return StringLength;
      ret = fwrite (Buffer, sptr - Buffer, 1, stdout);
      if (!ret) php_handle_aborted_connection();
      zptr = (sptr = Buffer) + sizeof(Buffer);
      return (StringLength);
   }
   
   cptr = (char*)String;
   cnt = StringLength;
   while (cnt)
   {
      if (sptr < zptr)
      {
         *sptr++ = *cptr++;
         cnt--;
         continue;
      }
      /* implicit flush buffer */
      ret = fwrite (Buffer, sptr - Buffer, 1, stdout);
      if (!ret) php_handle_aborted_connection();
      zptr = (sptr = Buffer) + sizeof(Buffer);
   }
   
   return (StringLength);
}
   
/*****************************************************************************/
/*
Flush anything in the SapiWrite() buffer.
*/

void SapiFlush (void *ServerContext)

{
   /*********/
   /* begin */
   /*********/

   if (Dbug) fprintf (stdout, "SapiFlush()\n");

   SapiWrite (NULL, 0);
}

/*****************************************************************************/
/*
As this is a CGI/CGIplus, and WASD (pre-v8.2) can be rather strict about it's
intepretation of CGI response headers, and the PHP engine outputs it response
headers in no particular order, buffer the header fields, then provide to WASD
in an order that is acceptable.  Algorithm is a bit crude but didn't take much
to knock together.  WASD 8.2 is much more liberal in permitted CGI response
field order so save a few CPU cycles by eliminating the above processing.
*/

static void SapiSendHeader
(
sapi_header_struct *shptr,
void *ServerContext
)
{
   static int  BufferLength,
               BufferSize,
               FieldCount;
   static char  *BufferPtr,
                *CurrentPtr;

   int  cnt;
   char  *cptr, *sptr, *zptr;
   
   /*********/
   /* begin */
   /*********/

   if (Dbug) fprintf (stdout, "SapiSendHeader()\n");

   if (ServerVersion >= 80200)
   {
      /* WASD 8.2 or later */
      if (shptr)
      {
         /* write this field (with carriage-control) */
         cptr = shptr->header;
         cnt = shptr->header_len;
         SapiWrite (cptr, cnt);
         SapiWrite ("\r\n", 2);
         FieldCount++;
         return;
      }
      /* end of response header */
      if (!FieldCount) SapiWrite ("Content-Type: text/plain\r\n", 26);
      /* ensure the server processes in stream mode */
      SapiWrite ("Script-control: X-stream-mode\r\n\r\n", 33);
      /* flush the SapiWrite() buffer (the response header) */
      SapiWrite (NULL, 0);
      FieldCount = 0;
      return;
   }

   if (!BufferPtr)
   {
      BufferSize = SEND_HEADER_BUFFER_CHUNK;
      if (!(BufferPtr = calloc (1, BufferSize))) exit (vaxc$errno);
      CurrentPtr = BufferPtr;
   }

   if (shptr)
   {
      /* buffer this header field */
      zptr = BufferPtr + BufferSize - 3;
      sptr = CurrentPtr;
      cptr = shptr->header;
      cnt = shptr->header_len;
      while (cnt--)
      {
         if (sptr < zptr)
         {
            *sptr++ = *cptr++;
            continue;
         }
         /* buffer full, increase it's size, reset the pointers */
         BufferLength = sptr - BufferPtr;
         BufferSize += SEND_HEADER_BUFFER_CHUNK;
         BufferPtr = realloc (BufferPtr, BufferSize);
         if (!BufferPtr) exit (vaxc$errno);
         zptr = BufferPtr + BufferSize - 3;
         sptr = BufferPtr + BufferLength;
         *sptr++ = *cptr++;
      }
      *sptr++ = '\r';
      *sptr++ = '\n';
      *sptr++ = '\0';

      BufferLength = sptr - BufferPtr;
      if (Dbug) fprintf (stdout, "|%s|\n", CurrentPtr);
      CurrentPtr = sptr;
      FieldCount++;
   }
   else
   {
      /* search for the first CGI-compliant response field */
      zptr = NULL;
      cptr = BufferPtr;
      for (cnt = FieldCount; cnt; cnt--)
      {
         for (sptr = cptr; *cptr; cptr++);
         if (!strncasecmp (sptr, "Content-Type:", 13) ||
             !strncasecmp (sptr, "Status:", 7) ||
             !strncasecmp (sptr, "Location:", 9))
         {
            /* note it's address and output */
            SapiWrite (zptr = sptr, cptr - sptr);
            break;
         }
         cptr++;
      }

      /* should never happen, but you never know */
      if (!zptr) SapiWrite ("Content-Type: text/html\r\n", 25);

      /* output all the rest of the fields */
      cptr = BufferPtr;
      for (cnt = FieldCount; cnt; cnt--)
      {
         for (sptr = cptr; *cptr; cptr++);
         /* if it's not the field that's already been output */
         if (sptr != zptr) SapiWrite (sptr, cptr - sptr);
         cptr++;
      }

      /* end of header, ensure the server processes in stream mode */
      SapiWrite ("Script-control: X-stream-mode\r\n\r\n", 33);
      /* flush the SapiWrite() buffer (the response header) */
      SapiWrite (NULL, 0);

      if (IsCgiPlus)
      {
         /* reset the buffer storage */
         CurrentPtr = BufferPtr;
         BufferLength = FieldCount = 0;
      }
   }
}

/*****************************************************************************/
/*
Read the POSTed body from <stdin> (which with WASD is actually HTTP$INPUT).
*/

static int SapiReadPost
(
char *BufferPtr,
uint BufferSize
SLS_DC
)
{
   int  retval,
        BufferCount;
   
   /*********/
   /* begin */
   /*********/

   if (Dbug) fprintf (stdout, "SapiReadPost() %d\n", BufferSize);

   if (BufferSize > (uint)SG(request_info).content_length-SG(read_post_bytes))
      BufferSize = (uint)SG(request_info).content_length-SG(read_post_bytes);
   if (Dbug) fprintf (stdout, "%d\n", BufferSize);

   BufferCount = 0;
   while (BufferCount < BufferSize)
   {
      retval = read (fileno(stdin),
                     BufferPtr + BufferCount,
                     BufferSize - BufferCount);
      if (retval <= 0) break;
      BufferCount += retval;
      if (Dbug) fprintf (stdout, "%d %d\n", BufferCount, BufferSize);
   }

   return (BufferCount);
}

/*****************************************************************************/
/*
Just get any HTTP cookie.
*/

static char *SapiReadCookies (SLS_D)

{
   /*********/
   /* begin */
   /*********/

   if (Dbug) fprintf (stdout, "SapiReadCookies()\n");

   return (CgiVar ("HTTP_COOKIE"));
}

/*****************************************************************************/
/*
Read CGI variables into PHP run-time from the CGIplus stream or from DCL
symbols as appropriate to the operating mode being standard CGI or CGIplus/RTE.
*/

static void SapiRegisterVariables (zval *tvaptr ELS_DC SLS_DC PLS_DC)

{
   char  *cptr, *sptr;
   char  String [32];

   /*********/
   /* begin */
   /*********/

   if (Dbug) fprintf (stdout, "SapiRegisterVariables()\n");

   if (IsCgiPlus)
   {
      /* successively read each CGIplus variable */
      while (sptr = CgiVar("*"))
      {
         /* won't see "FORM_" or "KEY_" with CGI, suppress with CGIplus */
         if (toupper(*sptr) == 'F' && !strncmp (sptr, "FORM_", 5)) continue;
         if (toupper(*sptr) == 'K' && !strncmp (sptr, "KEY_", 4)) continue;
         if (toupper(*sptr) == 'G' && !strncmp (sptr, "GATEWAY_", 8))
         {
            php_register_variable ("GATEWAY_API", SOFTWAREID,
                                   tvaptr ELS_CC PLS_CC);
            php_register_variable ("GATEWAY_PHP", ImageIdent(),
                                   tvaptr ELS_CC PLS_CC);
            sprintf (String, "%d", UsageCount);
            php_register_variable ("GATEWAY_USAGE", String,
                                   tvaptr ELS_CC PLS_CC);
         }
         for (cptr = sptr; *cptr && *cptr != '='; cptr++);
         *cptr = '\0';
         php_register_variable (sptr, cptr+1, tvaptr ELS_CC PLS_CC);
         *cptr = '=';
      }
   }
   else
      SapiRegisterCgiVariables (tvaptr);

   /* only if WASD has configured this realm as EXTERNAL */
   if (cptr = CgiVar ("AUTH_PASSWORD"))
   {
      /* supply these Apache mod_php-expected global variables */
      php_register_variable ("PHP_AUTH_PW", cptr, tvaptr ELS_CC PLS_CC);
      if (sptr = CgiVar ("AUTH_TYPE"))
         php_register_variable ("PHP_AUTH_TYPE", sptr, tvaptr ELS_CC PLS_CC);
      if (sptr = CgiVar ("REMOTE_USER"))
         php_register_variable ("PHP_AUTH_USER", sptr, tvaptr ELS_CC PLS_CC);
   }

   if (!(sptr = CgiVar ("SCRIPT_NAME"))) sptr = "";
   php_register_variable ("PHP_SELF", sptr, tvaptr ELS_CC PLS_CC);
}

/*****************************************************************************/
/*
Standard CGI environment.
Clunky, but what else can we do with DCL symbols?
*/

static void SapiRegisterCgiVariables (zval *tvaptr ELS_DC SLS_DC PLS_DC)

{
   static char  *cgi_var_names [] = {
"AUTH_ACCESS", "AUTH_AGENT", "AUTH_GROUP", "AUTH_PASSWORD",
"AUTH_REALM", "AUTH_REALM_DESCRIPTION", "AUTH_REMOTE_USER",
"AUTH_TYPE", "AUTH_USER",  "CONTENT_LENGTH", "CONTENT_TYPE",
"DOCUMENT_ROOT", "GATEWAY_BG", "GATEWAY_EOF", "GATEWAY_EOT",
"GATEWAY_ESC", "GATEWAY_INTERFACE", "GATEWAY_MRS",
"HTML_BODYTAG", "HTML_FOOTER", "HTML_FOOTERTAG",
"HTML_HEADER", "HTML_HEADERTAG",
"HTTP_ACCEPT", "HTTP_ACCEPT_CHARSET", "HTTP_ACCEPT_ENCODING", 
"HTTP_ACCEPT_LANGUAGE", "HTTP_AUTHORIZATION", "HTTP_CACHE_CONTROL", 
"HTTP_COOKIE", "HTTP_FORWARDED", "HTTP_HOST", "HTTP_IF_NOT_MODIFIED",
"HTTP_PRAGMA", "HTTP_REFERER", "HTTP_USER_AGENT", "HTTP_X_FORWARDED_FOR",
"PATH_INFO", "PATH_ODS", "PATH_TRANSLATED", "QUERY_STRING",
"REMOTE_ADDR", "REMOTE_HOST", "REMOTE_PORT", "REMOTE_USER", 
"REQUEST_CHARSET", "REQUEST_CONTENT_TYPE", "REQUEST_METHOD",
"REQUEST_SCHEME", "REQUEST_TIME_GMT", "REQUEST_TIME_LOCAL", "REQUEST_URI",
"SCRIPT_DEFAULT", "SCRIPT_FILENAME", "SCRIPT_NAME", "SCRIPT_RTE",
"SERVER_ADMIN", "SERVER_ADDR", "SERVER_CHARSET", "SERVER_GMT", "SERVER_NAME",
"SERVER_PROTOCOL", "SERVER_PORT", "SERVER_SOFTWARE", "SERVER_SIGNATURE",
"UNQIUE_ID", NULL };

   static char  *cgi_modssl_names [] = {
"HTTPS", "SSL_PROTOCOL", "SSL_SESSION_ID", "SSL_CIPHER", "SSL_CIPHER_EXPORT",
"SSL_CIPHER_USEKEYSIZE", "SSL_CIPHER_ALGKEYSIZE", "SSL_CLIENT_M_VERSION",
"SSL_CLIENT_M_SERIAL", "SSL_CLIENT_S_DN", "SSL_CLIENT_S_DN_x509",
"SSL_CLIENT_I_DN", "SSL_CLIENT_I_DN_x509", "SSL_CLIENT_V_START",
"SSL_CLIENT_V_END", "SSL_CLIENT_A_SIG", "SSL_CLIENT_A_KEY", "SSL_CLIENT_CERT",
"SSL_SERVER_M_VERSION", "SSL_SERVER_M_SERIAL", "SSL_SERVER_S_DN",
"SSL_SERVER_S_DN_x509", "SSL_SERVER_I_DN", "SSL_SERVER_I_DN_x509",
"SSL_SERVER_V_START", "SSL_SERVER_V_END", "SSL_SERVER_A_SIG",
"SSL_SERVER_A_KEY", "SSL_SERVER_CERT", "SSL_VERSION_INTERFACE",
"SSL_VERSION_LIBRARY", NULL };

   static char  *cgi_purveyor_names [] = {
"SECURITY_STATUS", "SSL_CIPHER", "SSL_CIPHER_KEYSIZE", "SSL_CLIENT_CA",
"SSL_CLIENT_DN", "SSL_SERVER_CA", "SSL_SERVER_DN", "SSL_VERSION", NULL };

   static char  *cgi_x509_names [] = {
"AUTH_X509_CIPHER", "AUTH_X509_FINGERPRINT", "AUTH_X509_ISSUER",
"AUTH_X509_KEYSIZE", "AUTH_X509_SUBJECT", NULL };

   int  idx;
   char  *cptr, *sptr;
   
   /*********/
   /* begin */
   /*********/

   if (Dbug) fprintf (stdout, "SapiRegisterCgiVariables()\n");

   /* standard CGI variables */
   for (idx = 0; cptr = cgi_var_names[idx]; idx++)
   {
      if (!(sptr = CgiVar (cptr))) continue;
      if (toupper(*cptr) == 'G' && !strncmp (cptr, "GATEWAY_", 8))
      {
         php_register_variable ("GATEWAY_API", SOFTWAREID,
                                tvaptr ELS_CC PLS_CC);
         php_register_variable ("GATEWAY_PHP", ImageIdent(),
                                tvaptr ELS_CC PLS_CC);
         php_register_variable ("GATEWAY_USAGE", "0",
                                tvaptr ELS_CC PLS_CC);
      }
      php_register_variable (cptr, sptr, tvaptr ELS_CC PLS_CC);
   }
   
   /* Apache mod_ssl-like SSL CGI variables */
   if (CgiVar ("SSL_VERSION_INTERFACE"))
   {
      for (idx = 0; cptr = cgi_modssl_names[idx]; idx++)
      {
         if (!(sptr = CgiVar (cptr))) continue;
         php_register_variable (cptr, sptr, tvaptr ELS_CC PLS_CC);
      }
   }
   
   /* Purveyor-like SSL CGI variables */
   if (CgiVar ("SECURITY_STATUS"))
   {
      for (idx = 0; cptr = cgi_purveyor_names[idx]; idx++)
      {
         if (!(sptr = CgiVar (cptr))) continue;
         php_register_variable (cptr, sptr, tvaptr ELS_CC PLS_CC);
      }
   }
   
   /* X.509 client certificate authentication CGI variables */
   if (CgiVar ("AUTH_X509_CIPHER"))
   {
      for (idx = 0; cptr = cgi_x509_names[idx]; idx++)
      {
         if (!(sptr = CgiVar (cptr))) continue;
         php_register_variable (cptr, sptr, tvaptr ELS_CC PLS_CC);
      }
   }

   sptr = CgiVar ("SCRIPT_NAME");
   php_register_variable ("PHP_SELF", sptr ? sptr : "", tvaptr ELS_CC PLS_CC);

   SG(request_info).request_uri = CgiVar("SCRIPT_NAME");
}

/*****************************************************************************/
/*
Return the value of a CGI variable regardless of whether it is used in a
standard CGI environment or a WASD CGIplus (RTE) environment.  Also
automatically switches WASD V7.2 and later servers into 'struct' mode for
significantly improved performance.

WASD by default supplies CGI variables prefixed by "WWW_" to differentiate them
from any other DCL symbols (or "env"ironment logicals).  PHP scripts expect CGI
variables without this.  This function is somewhat tailored to the PHP
environment to allow for this, ignoring it with the CGIplus streamed variable,
adding it when access DCL symbols.  Notice also it uses the required PHP
emalloc().

Returns a pointer to an emalloc()ed string if the variable exists, or NULL.
*/

char* CgiVar (char *VarName)

{
#  ifndef CGIVAR_STRUCT_SIZE
#     define CGIVAR_STRUCT_SIZE 8192
#  endif
#  define SOUS sizeof(unsigned short)

   static int  CalloutDone,
               StructLength;
   static char  *NextVarNamePtr;
   static char  StructBuffer [CGIVAR_STRUCT_SIZE];
   static FILE  *CgiPlusIn;
   
   int  status;
   int  Length;
   char  *bptr, *cptr, *sptr;

   /*********/
   /* begin */
   /*********/

   if (Dbug)
      fprintf (stdout, "CgiVar() |%s|\n", !VarName ? "NULL" : VarName);

   if (!VarName || !VarName[0])
   {
      /* initialize */
      StructLength = 0;
      NextVarNamePtr = StructBuffer;
      if (!VarName) return (NULL);
   }

   if (VarName[0])
   {
      /***************************/
      /* return a variable value */
      /***************************/

      if (!IsCgiPlus)
      {
         /* standard CGI environment */
         static char  WwwName [256] = "WWW_";
         static $DESCRIPTOR (NameDsc, WwwName);
         static $DESCRIPTOR (ValueDsc, "");
         unsigned short  ShortLength;
         char  Value [1024];

         /* by default WASD CGI variable names are prefixed by "WWW_", add */
         strncpy (WwwName+4, VarName, sizeof(WwwName)-5);
         NameDsc.dsc$w_length = strlen(WwwName);
         ValueDsc.dsc$a_pointer = Value;
         ValueDsc.dsc$w_length = 1023;
   
         status = lib$get_symbol (&NameDsc, &ValueDsc, &ShortLength, NULL);
         if (status & 1)
         {
            cptr = emalloc (ShortLength+1);
            memcpy (cptr, Value, ShortLength);
            cptr[ShortLength] = '\0';
         }
         else
            cptr = NULL;

         if (Dbug) fprintf (stdout, "CGI |%s|\n", !cptr ? "NULL" : cptr);
         return (cptr);
      }

      /* hmmm, CGIplus not initialized */
      if (!StructLength) return (NULL);

      if (VarName[0] == '*')
      {
         /* return each CGIplus variable in successive calls */
         if (!(Length = *(unsigned short*)NextVarNamePtr))
         {
            NextVarNamePtr = StructBuffer;
            if (Dbug) fprintf (stdout, "CGIplus |NULL|\n");
            return (NULL);
         }
         sptr = (NextVarNamePtr += SOUS);
         NextVarNamePtr += Length;
         if (Dbug) fprintf (stdout, "CGIplus |%s|\n", sptr);
         /* by default WASD CGI variable name are prefixed by "WWW_", ignore */
         return (sptr + 4);
      }

      /* return a pointer to this CGIplus variable's value */
      for (bptr = StructBuffer; Length = *(unsigned short*)bptr; bptr += Length)
      {
         /* by default WASD CGI variable name are prefixed by "WWW_", ignore */
         sptr = (bptr += SOUS) + 4;
         for (cptr = VarName; *cptr && *sptr && *sptr != '='; cptr++, sptr++)
            if (toupper(*cptr) != toupper(*sptr)) break;
         /* if found return a pointer to the value */
         if (!*cptr && *sptr == '=')
         {
            if (Dbug) fprintf (stdout, "CGIplus |%s|\n", sptr+1);
            cptr = emalloc (strlen(sptr));
            strcpy (cptr, sptr+1);
            return (cptr);
         }
      }
      /* not found */
      if (Dbug) fprintf (stdout, "CGIplus |NULL|\n");
      return (NULL);
   }

   /*****************************/
   /* get the CGIplus variables */
   /*****************************/

   /* cannot "sync" in a non-CGIplus environment */
   if (!VarName[0] && !IsCgiPlus) return (NULL);

   /* the CGIPLUSIN stream can be left open */
   if (!CgiPlusIn)
      if (!(CgiPlusIn = fopen (getenv("CGIPLUSIN"), "r")))
         exit (vaxc$errno);

   /* get the starting record (the essentially discardable one) */
   for (;;)
   {
      cptr = fgets (StructBuffer, sizeof(StructBuffer), CgiPlusIn);
      if (!cptr) exit (vaxc$errno);
      /* if the starting sentinal is detected then break */
      if (*(unsigned short*)cptr == '!\0' ||
          *(unsigned short*)cptr == '!\n' ||
          (*(unsigned short*)cptr == '!!' && isdigit(*(cptr+2)))) break;
   }

   /* MUST be done after reading the synchronizing starting record */
   if (Dbug) fprintf (stdout, "Content-Type: text/plain\n\n%s", StructBuffer);

   /* detect the CGIplus "force" record-mode environment variable (once) */
   if (*(unsigned short*)cptr == '!!')
   {
      /********************/
      /* CGIplus 'struct' */
      /********************/

      /* get the size of the binary structure */
      StructLength = atoi(cptr+2);
      if (StructLength <= 0 || StructLength > sizeof(StructBuffer))
         exit (SS$_BUGCHECK);

      if (!fread (StructBuffer, 1, StructLength, CgiPlusIn))
         exit (vaxc$errno);
   }
   else
   {
      /*********************/
      /* CGIplus 'records' */
      /*********************/

      /* reconstructs the original 'struct'ure from the records */
      sptr = (bptr = StructBuffer) + sizeof(StructBuffer);
      while (fgets (bptr+SOUS, sptr-(bptr+SOUS), CgiPlusIn))
      {
         /* first empty record (line) terminates variables */
         if (bptr[SOUS] == '\n') break;
         /* note the location of the length word */
         cptr = bptr;
         for (bptr += SOUS; *bptr && *bptr != '\n'; bptr++);
         if (*bptr != '\n') exit (SS$_BUGCHECK);
         *bptr++ = '\0';
         if (bptr >= sptr) exit (SS$_BUGCHECK);
         /* update the length word */
         *(unsigned short*)cptr = bptr - (cptr + SOUS);
      }
      if (bptr >= sptr) exit (SS$_BUGCHECK);
      /* terminate with a zero-length entry */
      *(unsigned short*)bptr = 0;
      StructLength = (bptr + SOUS) - StructBuffer;
   }

   if (Dbug)
   {
      fprintf (stdout, "%d\n", StructLength);
      for (bptr = StructBuffer; Length = *(unsigned short*)bptr; bptr += Length)
         fprintf (stdout, "|%s|\n", bptr += SOUS);
   }

   if (!CalloutDone)
   {
      /* provide the CGI callout to set CGIplus into 'struct' mode */
      fflush (stdout);
      fputs (CgiPlusEscPtr, stdout);
      fflush (stdout);
      /* the leading '!' indicates we're not going to read the response */
      fputs ("!CGIPLUS: struct", stdout);
      fflush (stdout);
      fputs (CgiPlusEotPtr, stdout);
      fflush (stdout);
      /* don't need to do this again (the '!!' tells us what mode) */
      CalloutDone = 1;
   }

   return (NULL);

#  undef SOUS
}

/*****************************************************************************/
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   