/*****************************************************************************/
/*
                                  cache.h
*/
/*****************************************************************************/

#ifndef CACHE_H_LOADED
#define CACHE_H_LOADED 1

#include "wasd.h"
#include "file.h"

/* number of bytes in each chunk of cache memory */
#define CACHE_CHUNK_SIZE 256
#define CACHE_PERM_CHUNK_SIZE 128

#define CACHE_EXPIRES_NONE   0xffffffff
#define CACHE_EXPIRES_DAY    0xfffffffe
#define CACHE_EXPIRES_HOUR   0xfffffffd
#define CACHE_EXPIRES_MINUTE 0xfffffffc
/* any other value will be considered literal seconds */

/**************/
/* structures */
/**************/

#ifndef __VAX
#   pragma member_alignment __save
#   pragma member_alignment
#endif

typedef struct CacheStruct FILE_CENTRY;

struct CacheStruct
{
   LIST_ENTRY  CacheListEntry;

   struct CacheStruct  *HashCollisionNextPtr,
                       *HashCollisionPrevPtr;

   BOOL  /* cache data is currently being loaded */
         DataLoading,
         /* is a permanent entry */
         EntryPermanent,
         /* cache entry has been invalidated to reclaim memory */
         EntryReclaimed,
         /* cache data is currently being revalidated */
         EntryRevalidating,
         /* data in cache is valid and can be use for requests */
         EntryValid,
         /* loaded via FILE module */
         FromFile,
         /* loaded via NET module */
         FromNet,
         /* loaded via CGI module */
         FromScript,
         /* remove allocated data buffer as soon as not in use */
         Purge,
         /* also remove cache entry */
         PurgeCompletely;

   int  /* some portion of the CGI header is contained in the cache */
        CgiHeaderLength,
        /* analogue to 'rqPathSet.CacheExpiresAfter' */
        ExpiresAfterPeriod,
        /* expires on the change of day, hour or minute */
        ExpiresAfterTime,
        /* server-tick second when the content expires */
        ExpiresTickSecond,
        /* server-tick second where it is still considered recent */
        FrequentTickSecond,
        /* the last guard period set against this entry */
        GuardSeconds,
        /* server-tick second when gaurd period expires */
        GuardTickSecond;

   unsigned short  /* file protection bitmap */
                   Protection,
                   /* file ownership UIC group number */
                   UicGroup,
                   /* file ownership UIC member number */
                   UicMember;

   unsigned long  /* length of data (file contents) pointed to 'ContentPtr' */
                  ContentLength,
                  /* virtual block number of end-of-file */
                  EndOfFileVbn,
                  /* size of the block pointed to by 'ContentPtr' */
                  EntrySize,
                  /* first free byte in end-of-file VBN */
                  FirstFreeByte,
                  /* number of times cache used */
                  HitCount,
                  /* number of times cache hit but file unmodified */
                  HitNotModifiedCount,
                  /* cache entry is in use and should not be modified */
                  InUseCount,
                  /* number of times the entry has been validated */
                  ValidatedCount;

   unsigned long  /* creation quadword */
                  CdtBinTime [2],
                  /* last hit quadword */
                  HitBinTime [2],
                  /* when cache data loaded quadword */
                  LoadBinTime [2],
                  /* last modified quadword */
                  RdtBinTime [2],
                  /* last validated quadword */
                  ValidateBinTime [2];

   char  /* points to entry contents */
         *ContentPtr;

   char  /* entry's MIME content type */
         ContentType [48];

   /* MD5 hash of cached resource (can be path, can be file name) */
   MD5_HASH  Md5Hash;

   ODS_STRUCT  FileOds;

   /* used to buffer the function address of the content handler */
   REQUEST_AST  ContentHandlerFunction;
};

#ifndef __VAX
#   pragma member_alignment __restore
#endif

/***********************/
/* function prototypes */
/***********************/

CacheAcpInfoAst (REQUEST_STRUCT*);
int CacheBegin (REQUEST_STRUCT*);
CacheEnd (REQUEST_STRUCT*);
CacheInit ();
BOOL CacheLoadBegin (REQUEST_STRUCT*, int, char*);
int CacheLoadData (REQUEST_STRUCT*, char*, int);
CacheLoadEnd (REQUEST_STRUCT*);
CacheSearch (REQUEST_STRUCT*);
CacheNext (REQUEST_STRUCT*);
CachePurge (BOOL, int*, int*);
CacheRemoveEntry (FILE_CENTRY*, BOOL);
CacheReport (REQUEST_STRUCT*, REQUEST_AST, BOOL);
CacheReportEntry (REQUEST_STRUCT*, REQUEST_AST, char*);
CacheZeroCounters ();

#endif /* CACHE_H_LOADED */

/*****************************************************************************/
