/*****************************************************************************/
/*
                                  Config.h

Function prototypes for configuration module.

*/
/*****************************************************************************/

#ifndef CONFIG_H_LOADED
#define CONFIG_H_LOADED 1

#ifndef ODSSTRUCT_H_LOADED
#   include "odsstruct.h"
#endif

#ifndef TRACK_H_LOADED
#   include "track.h"
#endif

#include "wasd.h"
#include "metacon.h"

/* must be powers of two! */
#define CONTENT_TYPE_HASH_TABLE_SIZE  128
#define CONTENT_TYPE_HASH_TABLE_MASK  CONTENT_TYPE_HASH_TABLE_SIZE-1
#define ICON_HASH_TABLE_SIZE          64
#define ICON_HASH_TABLE_MASK          ICON_HASH_TABLE_SIZE-1

/**********************************/
/* config authorization structure */
/**********************************/

struct ConfigAuthStruct
{
   BOOL  BasicEnabled,
         DigestEnabled,
         RevalidateLoginCookie,
         SysUafAcceptExpPwd,
         SysUafUseACME;
   int  CacheEntriesMax,
        CacheEntrySize,
        CacheMinutes,
        DigestNonceGetLifeTime,
        DigestNoncePutLifeTime,
        FailureLimit,
        FailurePeriodSeconds,
        FailureTimeoutSeconds,
        RevalidateUserMinutes;
   char  SysUafPwdExpUrl [128];
};

/********************************/
/* config buffer size structure */
/********************************/

struct ConfigBufferStruct
{
   int  SizeDclCommand,
        SizeDclOutput,
        SizeDclCgiHeader,
        SizeDclCgiPlusIn,
        SizeNetRead,
        SizeNetWrite;
};

/**************************/
/* config cache structure */
/**************************/

struct ConfigCacheStruct
{
   BOOL  Enabled;
   int  EntriesMax,
        FileKBytesMax,
        FrequentHits,
        FrequentSeconds,
        GuardSeconds,
        TotalKBytesMax,
        ValidateSeconds;
};

/****************************/
/* config content structure */
/****************************/

typedef struct ContentTypeStruct CONTENT_TYPE;

struct ContentTypeStruct
{
   CONTENT_TYPE  *NextPtr,
                 *HashCollisionPtr;
   BOOL  FromMimeTypes,
         Superceded,
         TypeText,
         TypeUnknown;
   char  FtpMode;
   char  *AutoScriptNamePtr,
         *ContentTypePtr,
         *DescriptionPtr,
         *IconPtr,
         *SuffixPtr;
   /* string data is stored here */
};

typedef struct IconTypeStruct ICON_TYPE;

struct IconTypeStruct
{
   struct IconTypeStruct  *NextPtr,
                          *HashCollisionPtr;
   BOOL  FromMimeTypes,
         Superceded;
   char  *AltTextPtr,
         *ContentTypePtr,
         *IconUriPtr,
         *IconUrlPtr;
   /* string data is stored here */
};

struct ConfigContentStruct
{
   int  CharsetConvertLength,
        ContentTypeStructOverhead,
        HomePageCount,
        IconStructOverhead;
   char  *CharsetConvertPtr,
         *ContentTypeDefaultPtr;
   char  CharsetDefault [32],
         MimeTypesFileName [128];
   char  *HomePageArray [CONFIG_HOME_PAGES_MAX];
   CONTENT_TYPE  *ContentInfoListHeadPtr,
                 *ContentInfoListTailPtr;
   ICON_TYPE  *IconListHeadPtr,
              *IconListTailPtr;
   CONTENT_TYPE  *ContentInfoHashTable [CONTENT_TYPE_HASH_TABLE_SIZE];
   ICON_TYPE  *IconHashTable [ICON_HASH_TABLE_SIZE];
};

/************************/
/* config dir structure */
/************************/

struct ConfigDirStruct
{
   BOOL  Access,
         AccessSelective,
         MetaInfoEnabled,
         NoImpliedWildcard,
         NoPrivIgnore,
         OwnerEnabled,
         PreExpired,
         ReadMeBottom,
         ReadMeTop,
         WildcardEnabled;
   int  DescriptionLines,
        ReadMeFileCount;
   char  DefaultLayout [32],
         BodyTag [256];
   char  *ReadMeFileArray [CONFIG_README_FILES_MAX];
};

/****************************/
/* config logging structure */
/****************************/

struct ConfigLogStruct
{
   BOOL  Enabled,
         PerInstance,
         PerService,
         PerServiceHostOnly;
   int  ExcludeHostsLength,
        ExtendBlocks;
   char  *ExcludeHostsPtr;
   char  FileName [128],
         Format [128],
         Naming [32],
         Period [32];
};

/**********************************/
/* config miscellaneous structure */
/**********************************/

struct ConfigMiscStruct
{
   BOOL  DnsLookupClient,
         MonitorEnabled,
         RegexEnabled;
   int  ActivityNumberOfDays,
        DnsLookupLifeTimeSeconds,
        DnsLookupRetryCount,
        MapUserNameCacheEntries,
        PersonaCacheEntries,
        PutMaxKbytes,
        PutVersionLimit,
        RequestHistory,
        StreamLfConversionMaxKbytes;
};

/**************************/
/* config OPCOM structure */
/**************************/

struct ConfigOpcomStruct
{
   int  Index,
        Messages,
        Target;
};

/**********************************/
/* config proxy serving structure */
/**********************************/

struct ConfigProxyStruct
{
   BOOL  CacheEnabled,
         ReportCacheLog,
         ReportLog,
         ServingEnabled,
         UnknownRequestFields;
   int  CacheDeviceMaxPercent,
        CacheDevicePurgePercent,
        CacheDeviceCheckMinutes,
        CacheDeviceDirOrg,
        CacheFileKBytesMax,
        CacheNoReloadSeconds,
        ForwardedBy,
        HostLookupRetryCount,
        VerifyRecordMax,
        XForwardedFor;
   char  CachePurgeList [64],
         CacheReloadList [64],
         CacheRoutineHourOfDayString [16];
};

/***************************/
/* config report structure */
/***************************/

struct ConfigReportStruct
{
   BOOL  ErrorRecommend,
         BasicOnly,
         MetaInfoEnabled;
   int  ErrorReportPathLength;
   char  ErrorReportPath [128];
};

/******************************/
/* config scripting structure */
/******************************/

struct ConfigRunTimeStruct
{
   int  FileTypeLength;
   char  String [128];   
};

struct ConfigScriptStruct
{
   BOOL  CgiStrictOutput,
         DetachProcess,
         Enabled,
         FullRequest,
         GatewayBg,
         SpawnAuthPriv;
   int  BitBucketTimeout,
        CgiPlusLifeTime,
        CleanupScratchMinutesMax,
        CleanupScratchMinutesOld,
        DECnetConnectListMax,
        DECnetReuseLifeTime,
        DefaultSearchLength,
        DefaultSearchExcludeLength,
        RunTimeCount,
        ScriptProcessHardLimit,
        ScriptProcessSoftLimit,
        ZombieLifeTime;
   char  *DefaultSearchExcludePtr;
   char  DetachProcessPriority [32],
         DefaultSearch [128];
   struct ConfigRunTimeStruct RunTime  [CONFIG_SCRIPT_RUNTIME_MAX];
};

/***************************/
/* config server structure */
/***************************/

struct ConfigServerStruct
{
   int  AcceptHostsLength,
        BusyLimit,
        DefaultPort,
        InstanceMax,
        ListenBacklog,
        RejectHostsLength,
        Signature,
        ServiceLength;
   char  *AcceptHostsPtr,
         *RejectHostsPtr,
         *ServicePtr;
   char  AdminEmail [64],
         AdminBodyTag [256],
         ReportBodyTag [256],
         ServiceNotFoundUrl [128];
};

/*****************************************/
/* config Server Side Includes structure */
/*****************************************/

struct ConfigSsiStruct
{
   BOOL  Enabled,
         AccessesEnabled,
         ExecEnabled;
   int  SizeMax;
};

/****************************/
/* config timeout structure */
/****************************/

struct ConfigTimeoutStruct
{
   int  Input,
        NoProgress,
        Output,
        KeepAlive;
};

/**************************/
/* config track structure */
/**************************/

struct ConfigTrackStruct
{
   BOOL  Enabled,
         MultiSession;
   char  Domain [TRACK_DOMAIN_SIZE];
};

/***************************/
/* configuration structure */
/***************************/

typedef struct ConfigStruct CONFIG_STRUCT;

struct ConfigStruct
{
   struct ConfigAuthStruct    cfAuth;
   struct ConfigBufferStruct  cfBuffer;
   struct ConfigCacheStruct   cfCache;
   struct ConfigContentStruct cfContent;
   struct ConfigDirStruct     cfDir;
   struct ConfigLogStruct     cfLog;
   struct ConfigMiscStruct    cfMisc;
   struct ConfigOpcomStruct   cfOpcom;
   struct ConfigProxyStruct   cfProxy;
   struct ConfigReportStruct  cfReport;
   struct ConfigScriptStruct  cfScript;
   struct ConfigServerStruct  cfServer;
   struct ConfigSsiStruct     cfSsi;
   struct ConfigTimeoutStruct cfTimeout;
   struct ConfigTrackStruct   cfTrack;

   /* basically only used while reading configuration from file */
   BOOL  BracketDirectives,
         IncludeFileDetected,
         EquateDirectives;
   char  DirectiveName [64];
   /* only used when reporting or changing configuration */
   void  *RequestPtr;
};

/***********************/
/* function prototypes */
/***********************/

int ConfigLoad (META_CONFIG**);
BOOL ConfigLoadCallback (META_CONFIG*);
BOOL ConfigAcceptClientHostName (char*, char*);
ConfigAddType (META_CONFIG*, char*);
ConfigAddMimeType (META_CONFIG*, char*);
ConfigAddTypeNow (META_CONFIG*, char*, char*, char*, char*, char, BOOL);
char* ConfigStringList (REQUEST_STRUCT*, char*, char);
char* ConfigContentType (CONTENT_TYPE*, char*);
ConfigContentTypeIcon (META_CONFIG*);
char* ConfigHomePage (int);
ConfigAddIcon (META_CONFIG*, char*, char*, BOOL);
char* ConfigIconFor (char*, char**, char**);
char* ConfigReadMeFile (int);
ConfigReport (REQUEST_STRUCT*, REQUEST_AST, BOOL);
ConfigReportNow (REQUEST_STRUCT*, META_CONFIG*);
ConfigRevise (REQUEST_STRUCT*, REQUEST_AST, BOOL);
ConfigReviseNow (REQUEST_STRUCT*, META_CONFIG*);
BOOL ConfigSameContentType (char*, char*, int);
ConfigSetAccept (META_CONFIG*, char*);
ConfigSetDirReadMe (META_CONFIG*, char*);
ConfigSetDirReadMeFile (META_CONFIG*, char*);
ConfigSetOpcom (META_CONFIG*, char*, char*);
ConfigSetReject (META_CONFIG*, char*);
ConfigSetWelcome (META_CONFIG*, char*);

#endif /* CONFIG_H_LOADED */

/*****************************************************************************/
