/*****************************************************************************/
/*
                                  file.h
*/
/*****************************************************************************/

#ifndef FILE_H_LOADED
#define FILE_H_LOADED 1

#include "wasd.h"

/**********/
/* macros */
/**********/

/* default maximum file size 2048 blocks, ~1MB */
#define FILE_TYPE_DEFAULT_SIZE_MAX 2048*512
/* maximum size of .ISMAP file, 64 blocks, ~32kB */
#define FILE_TYPE_ISMAP_SIZE_MAX 64*512
/* 64 blocks, ~32kB */
#define FILE_TYPE_MENU_SIZE_MAX 64*512
/* 256 blocks, ~128kB */
#define FILE_TYPE_README_SIZE_MAX 256*512
/* 256 blocks, ~128kB */
#define FILE_TYPE_SSI_SIZE_MAX 256*512
/* 16 blocks, ~8kB */
#define FILE_TYPE_URL_SIZE_MAX 16*512

/* ambit dynamically allocated string space when negotiating file language */
#define FILE_ACCEPT_LANG_SIZE 256

/* maximum size of language variant string (larger is obviously not) */
#define FILE_ACCEPT_LANG_VARIANT_MAX 5

/**************/
/* structures */
/**************/

#ifndef __VAX
#   pragma member_alignment __save
#   pragma member_alignment
#endif

typedef struct FileContentsStruct FILE_CONTENT;

struct FileContentsStruct
{
   int  /* count during load, and final length of contents */
        ContentLength,
        /* contents buffer space remaining (complement to 'ContentLength') */
        ContentRemaining,
        /* contents buffer size (total originally allocated memory) */
        ContentSize,
        /* limit on content size */
        ContentSizeMax,
        /* fairly obvious */
        FileNameLength;

   unsigned long  CdtBinTime [2],
                  RdtBinTime [2];

   unsigned short  UicGroup,
                   UicMember,
                   Protection;

   char  /* buffer (pointer to allocated memory, see note below) */
         *ContentPtr,
         /* pointer to current position in buffer */
         *CurrentPtr;

   char  /* file name stored locally */
         FileName [ODS_MAX_FILE_NAME_LENGTH+1];

   REQUEST_AST  /* the address of the function processing this content */
                ContentHandlerFunction,
                /* passes on next task that was passed to FileBegin() */
                NextTaskFunction;

   /*  contents are straight after this and pointed to by 'ContentPtr' */
};

typedef struct FileTaskStruct FILE_TASK;

struct FileTaskStruct
{
   BOOL  /* a lower case alphabetic was detected in the file name */
         AcceptLangLowerCase,
         /* authorize the path to the filename before processing */
         AuthorizePath,
         /* caching of this file is allowed */
         CacheAllowed,
         /* when true HTML-forbidden characters are escaped */
         EscapeHtml,
         /* when true a file's contents are enclosed in <PRE></PRE> tags */
         PreTagFileContents,
         /* when a file's contents have been <PRE>ed, end with </PRE> tag */
         PreTagEndFileContents,
         /* if true task structure has been initialized */
         TaskInitialized;

   int  /* current used from memory allocated */
        AcceptLangTypesLength,
        /* amount of memory allocated */
        AcceptLangTypesSize,
        /* size of buffer for escaping HTML */
        EscapeHtmlSize,
        /* maximum file size allowed when buffering a file's contents */
        FileContentsSizeMax,
        /* fairly obvious */
        FileNameLength,
        /* note when a search-list has been searched for the file */
        SearchListCount,
        /* buffer for most recent task status */
        TaskStatusBuffer;

   char  /* file name stored locally */
         FileName [ODS_MAX_FILE_NAME_LENGTH+1];

   char  /* points to where the language variant wildcard was added */
         *AcceptLangVariantPtr,
         /* when negotiating language contains searched-for types */
         *AcceptLangTypesPtr,
         /* pointer to MIME content-type */
         *ContentTypePtr,
         /* pointer to buffer used to escape HTML-forbidden chars */
         *EscapeHtmlPtr;

   /* MD5 hash of the resource */
   MD5_HASH  Md5Hash;

   /* structure for file system access */
   ODS_STRUCT  FileOds;

   /* success and file-not-found handlers */
   REQUEST_AST  ContentHandlerFunction,
                NextTaskFunction,
                NoSuchFileFunction;
};

#ifndef __VAX
#   pragma member_alignment __restore
#endif

/***********************/
/* function prototypes */
/***********************/

FileAcceptLangBegin (REQUEST_STRUCT*, char*);
FileAcceptLangDefault (REQUEST_STRUCT*);
FileAcceptLangParseAst (struct FAB*);
FileAcceptLangSearchAst (struct FAB*);
FileAcceptLangSelect (REQUEST_STRUCT*);
FileAccessAst (REQUEST_STRUCT*);
FileAcpInfo (REQUEST_STRUCT*);
FileAcpInfoAst (REQUEST_STRUCT*); 
FileAuthorizationAst (REQUEST_STRUCT*); 
FileBegin (REQUEST_STRUCT*, REQUEST_AST, REQUEST_AST, MD5_HASH*, char*, char*);
FileCacheStale (REQUEST_STRUCT*); 
FileConnectAst (struct RAB*); 
FileContentsBuffer (REQUEST_STRUCT*, REQUEST_AST, char*, int);
FileEnd (REQUEST_STRUCT*);
FileFixedRecordNoSpan (REQUEST_STRUCT*);
FileHttpHeader (REQUEST_STRUCT*, int, char*, int);
FileNextBlocks (REQUEST_STRUCT*);
FileNextBlocksAst (REQUEST_STRUCT*); 
FileOpen (FILE_TASK*); 
FileOpenAst (struct FAB*); 
FileParseAst (struct FAB*); 
FileResponseHeader (REQUEST_STRUCT*, char*, int, unsigned long*, RANGE_BYTE*);
void FileSetAuthorizePath (REQUEST_STRUCT*, BOOL);
void FileSetCacheAllowed (REQUEST_STRUCT*, BOOL);
void FileSetContentHandler (REQUEST_STRUCT*, REQUEST_AST, int);
void FileSetEscapeHtml (REQUEST_STRUCT*, BOOL);
void FileSetPreTag (REQUEST_STRUCT*, BOOL);
char* FileSetCharset (REQUEST_STRUCT*, char*);
FileVariableRecord (REQUEST_STRUCT*);

#endif /* FILE_H_LOADED */

/*****************************************************************************/
