/*****************************************************************************/
/*
                                 Net.c

The networking essentials are based on DEC TCP/IP Services for OpenVMS (aka
UCX). The QIO interface was obviously chosen because of its asynchronous,
non-blocking capabilities. Although some functionalilty could have been
implemented using the BSD sockets abstraction and only the asynchronous I/O
done using the QIO I opted for working out the QIOs. It wasn't too difficult
and avoids the (slight) extra overhead of the sockets layer.

With resource wait explicitly enabled all sys$qio()s should wait until
resources become available. The only execption is ASTLM, a fundamental
requirement for this server. If SS$_EXQUOTA is returned from a sys$qio() (i.e.
without wait for completion, and therefore depending on AST delivery) then
EXIT THE SERVER with error status!

The 'ServiceStruct' structure has two groups of fields and performs two basic
roles. First, it provides a linked list which can be traversed to determine
from the destination IP address of a request which host was specified in a
(multi-homed) request. Second, for each unique port specified for the same
server the client components of the structure are used for the accept()ing of
requests.  Note that there may be more 'service' structures in the list than
actual sockets created (and client components used) because only one is
allocated against a given port even though the services may specify multiple
host names using that port.

The server identity is derived from the 'ServerHostName' (gethostname()) name
and the 'ServerPort' (which is the first port specified as a service, or the
port specified in the configuration or /PORT= qualifier).


VERSION HISTORY
---------------
10-APR-2004  MGD  significant modifications to support IPv6
18-FEB-2004  MGD  NetWriteBufferedSizeInit()
30-DEC-2003  MGD  NetTcpIpAgentInfo() mods for IA64
07-JUL-2003  MGD  support response header none/append mappings,
                  cache loading from network output
26-FEB-2003  MGD  disable 'NetMultiHomedHost' (should not be required
                  for modern virtual service processing)
15-JUL-2002  MGD  all 'xray' functionality now performed in RequestScript()
02-JUN-2002  MGD  rework NetCreateService() to allow SS$_IVADDR (invalid
                  media address) service to be supported by using INADDR_ANY
25-APR-2002  MGD  NetAcceptSupervisor() and associated NOIOCHAN redress
31-MAR-2002  MGD  integrate client connection data into request structure,
                  make client host name lookup asynchronous
22-JAN-2002  MGD  bugfix; NetAcceptAst() deassign channel when connection
                  dropped during accept processing (jpp@esme.fr)
18-JAN-2002  MGD  allow ->BindIpAddressString to specify 0.0.0.0 (INADDR_ANY)
29-SEP-2001  MGD  service creation now supports multiple channels to the
                  one listening socket device for multiple per-node instances
04-AUG-2001  MGD  support module WATCHing,
                  CONNECT method connection cancellation is normal behaviour
04-JUL-2001  MGD  bugfix; (completed from 02-JUN, this time for SSL services),
                  also change behaviour, if a bind to INADDR_ANY fails attempt
                  to bind to the resolved host name address
02-JUN-2001  MGD  bugfix; port check when IP address explicitly supplied
18-APR-2001  MGD  rqNet.WriteErrorCount and rqNet.ReadErrorCount,
                  introduce NetTcpIpAgentInfo() (adapted from WATCH.C),
                  bugfix; NetThisVirtualService()
13-FEB-2001  MGD  ntohs() on client port
22-NOV-2000  MGD  rework service creation
17-OCT-2000  MGD  modify SSL initialization so that "sharing" conditions
                  (same port on same IP address) are more easily identified
08-AUG-2000  MGD  client sockets C_SHARE for direct script output to BG:
17-JUN-2000  MGD  modifications for SERVICE.C requirements
10-MAY-2000  MGD  per-service session tracking,
                  per-service listen queue backlog (for Compaq TCP/IP 5.0ff)
29-APR-2000  MGD  proxy authorization
10-NOV-1999  MGD  add IO$_NOWAIT to NetWriteDirect()
25-OCT-1999  MGD  remove NETLIB support
10-OCT-1999  MGD  allow virtual services more latitude,
                  check for request supervisor request timeout,
                  add FULL_DUPLEX_CLOSE,
                  workaround TCPWARE 5.3-3 behaviour (Laishev@SMTP.DeltaTel.RU)
18-AUG-1999  MGD  bugfix; parsing certificate/key from service
11-JUN-1999  MGD  bugfix; NetTooBusy() sys$fao() charset,
                  bugfix; NetDummyReadAst() UCX IOsb by reference
26-MAY-1999  MGD  bugfix; NetShutdownSocket() AST handling
04-APR-1999  MGD  provide HTTP/0.9 header absorbtion (finally!)
15-JAN-1999  MGD  changed AST delivery algorithm
07-NOV-1998  MGD  WATCH facility
24-OCT-1998  MGD  allow SSL certificate to be specified per-service
08-APR-1998  MGD  allow legitimate connects to be CANCELed in NetAcceptAst()
19-JAN-1998  MGD  redesigned NetWriteBuffered()
27-NOV-1997  MGD  hmmm, rationalized AST processing by making network
                  writes always deliver an AST (implicit or explicit)
                  (causing ACCVIOs for the last couple of versions)
25-OCT-1997  MGD  changes to MsgFor() function
06-SEP-1997  MGD  multi-homed hosts and multi-port services
30-AUG-1997  MGD  bugfix; get server host name before starting logging
                  (woops, problem introduced with NETLIB)
09-AUG-1997  MGD  message database
23-JUL-1997  MGD  HTTPD v4.3, MultiNet dropped, using the NETLIB Library
01-FEB-1997  MGD  HTTPd version 4
27-SEP-1996  MGD  add dummy read for early error reporting
12-APR-1996  MGD  observed Multinet disconnection/zero-byte behaviour
                  (request now aborts if Multinet returns zero bytes)
03-JAN-1996  MGD  support for both DEC TCP/IP Services and TGV MultiNet
01-DEC-1995  MGD  NetWriteBuffered() for improving network I/O
20-DEC-1994  MGD  multi-threaded version
20-JUN-1994  MGD  single-threaded version
*/
/*****************************************************************************/

#ifdef WASD_VMS_V6
#undef _VMS_V6_SOURCE
#define _VMS_V6_SOURCE
#undef __VMS_VER
#define __VMS_VER 60000000
#undef __CRTL_VER
#define __CRTL_VER 60000000
#endif

/* standard C header files */
#include <ctype.h>
#include <errno.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>

/* VMS related header files */
#include <descrip.h>
#include <dvidef.h>
#include <iodef.h>
#include <psldef.h>
#include <ssdef.h>
#include <stsdef.h>

/* application-related header files */
#include "wasd.h"

#define WASD_MODULE "NET"

#define NET_TEST 0

/******************/
/* global storage */
/******************/

BOOL  NetAcceptNoIoChan,
      NetMultiHomedHost;

int  ConnectCountTotal,
     InstanceConcurrentMax,
     NetAcceptBytLmRequired,
     NetAcceptNoIoChanCount,
     NetListenBytLmRequired,
     NetReadBufferSize,
     OutputBufferSize,
     ServerHostNameLength;

char  ServerHostName [TCPIP_HOSTNAME_MAX+1],
      ServerHostPort [TCPIP_HOSTNAME_MAX+1+8];

/********************/
/* external storage */
/********************/

#ifdef DBUG
extern BOOL Debug;
#else
#define Debug 0 
#endif

extern BOOL  ControlExitRequested,
             HttpdServerStartup,
             ProtocolHttpsAvailable,
             ProtocolHttpsConfigured,
             TcpIpv6Configured;

extern int  ConfigDnsLookupRetryCount,
            EfnWait,
            EfnNoWait,
            InstanceConnectCurrent,
            InstanceNodeConfig,
            ServiceCount,
            ServerPort;

extern char  CliLogFileName[],
             ControlBuffer[],
             ErrorSanityCheck[],
             HttpdName,
             HttpdVersion,
             Utility[];

extern struct dsc$descriptor TcpIpDeviceDsc;

extern ACCOUNTING_STRUCT  *AccountingPtr;
extern CONFIG_STRUCT  Config;
extern IPADDRESS TcpIpEmptyAddress;
extern MSG_STRUCT  Msgs;
extern SERVICE_STRUCT  *ServiceListHead,
                       *ServiceListTail;
extern TCP_SOCKET_ITEM  TcpIpSocket4,
                        TcpIpSocket6;
extern VMS_ITEM_LIST2  TcpIpFullDuplexCloseOption;
extern VMS_ITEM_LIST2  TcpIpSocketReuseAddrOption;
extern VMS_ITEM_LIST2  TcpIpSocketShareOption;
extern WATCH_STRUCT  Watch;

/*****************************************************************************/
/*
Get local host (server) name.
*/ 

NetGetServerHostName ()

{
   int  status;
   unsigned short  Length;
   char  *cptr;

   /*********/
   /* begin */
   /*********/

   if (WATCH_MODULE(WATCH_MOD_NET))
      WatchThis (NULL, FI_LI, WATCH_MOD_NET, "NetGetServerHostName()");

   if (gethostname (ServerHostName, sizeof(ServerHostName)) != 0)
      ErrorExitVmsStatus (0, strerror(errno), FI_LI);

   /* looks better if its all in lower case (host name is sometimes upper) */
   for (cptr = ServerHostName; *cptr; cptr++) *cptr = tolower(*cptr);
   ServerHostNameLength = cptr - ServerHostName;

   if (!ServerHostNameLength)
      ErrorExitVmsStatus (SS$_BUGCHECK, "could not determine local host name",
                          FI_LI);
}

/*****************************************************************************/
/*
It is only necessary to create one socket for each port port on the same
server, even if that port is specified against a different (multi-home) host
name because the accept()ing socket can be interrogated to determine which
host had been specified.  Then in a infinite loop accept connections from
clients.

Binding services to sockets and allied topics ...

A service (host name and IP address) binds to INADDR_ANY and a port (say 80). 
This allows the socket to accept connections for any address supported by the
interface, and on that port.

Subsequent services (different host name but same IP address, an alias, CNAME
records?) using the same port does not (indeed could not) bind again (to
INADDR_ANY).  It just becomes part of the the existing bound socket's
environment inside the server.  Using the HTTP "Host:" field virtual services
are supported (once the HTTP request header is parsed).  Different ports are of
course bound to different sockets.

Some further subsequent service, this time with a different IP address (i.e.
implying there is a multi-homed system) wishes to establish a service on a
previously used port (say 80).  First it attempts a bind to INADDR_ANY, which
fails because the port is already bound to that (mind you it may not be, in
which case it won't fail).  So the server then retries the bind with it's IP
address, which we'll presume is OK.

Another service, with a third IP address tries to bind to INADDR_ANY on port
80, which fails, but it then successfully is bound using it's (so far) unique
address.

So far, no real site admin intervention.  It seems to simply and easily support
the potential requirements of SSL services, as well as multi-homed standard
services.  All "real" multi-homed services (with autonomous IP addresses) are
always bind against their unqiue address.  "Virtual" services using those
addresses only are ever bound once for each port, subsequent services being
software contrivances.  The [ServiceIpAddress] is still available to
"hard-wire" a service  name to a particular IP address, but the above "cascade"
of binds tends to mean it should be far less important.

Only when a service with an IP address that is not supported by the interface
is attempted to be bound does it fail (with an invalid media address IIRC).
*/ 

NetCreateService ()

{
   int  cnt, status,
        BytLmAfter,
        BytLmBefore,
        IpPort;
   unsigned short  Length,
                   ServerChannel;
   char  *cptr,
         *ProtocolPtr,
         *SocketDevNamePtr;
   IPADDRESS  IpAddress,
              PrimaryIpAddress;
   SOCKADDRIN  *sin4ptr;
   SOCKADDRIN6  *sin6ptr;
   SERVICE_STRUCT  *svptr, *tsvptr;
   TCP_SOCKET_ITEM  *TcpSocketPtr;
   VMS_ITEM_LIST2  SocketNameItem;
   VMS_ITEM_LIST2  *il2ptr;
   $DESCRIPTOR (BgDevNameDsc, "");
   
   /*********/
   /* begin */
   /*********/

   if (WATCH_MODULE(WATCH_MOD_NET))
      WatchThis (NULL, FI_LI, WATCH_MOD_NET,
                 "NetCreateService() !UL", ServiceCount);

   /********************************/
   /* process the list of services */
   /********************************/

   /* block other instances from concurrently attempting to create services */
   if (VMSnok (status = InstanceLock (INSTANCE_NODE_SOCKET)))
      ErrorExitVmsStatus (status, "InstanceLock()", FI_LI);

   /* get the address of the primary (first) service */
   IPADDRESS_COPY (&PrimaryIpAddress,
                   &((SERVICE_STRUCT*)ServiceListHead)->ServerIpAddress);

   for (svptr = ServiceListHead; svptr; svptr = svptr->NextPtr)
   {
      WriteFaoStdout ("%!AZ-I-SERVICE, !AZ//!AZ\n",
                      Utility, svptr->RequestSchemeNamePtr,
                      svptr->ServerHostPort);

      /* if the service has been given a a specific IP address to bind to */
      if (IPADDRESS_IS_SET(&svptr->BindIpAddress))
         IPADDRESS_COPY (&IpAddress, &svptr->BindIpAddress)
      else
      if (IPADDRESS_IS_SAME (&svptr->ServerIpAddress, &PrimaryIpAddress))
      {
         /* zeroing these is the equivalent of setting INADDR_ANY */
         if (IPADDRESS_IS_V4 (&PrimaryIpAddress))
            IPADDRESS_ZERO4 (&IpAddress)
         else
         if (IPADDRESS_IS_V6 (&PrimaryIpAddress))
            IPADDRESS_ZERO6 (&IpAddress)
         else
            ErrorExitVmsStatus (SS$_BUGCHECK, ErrorSanityCheck, FI_LI);
      }
      else
         IPADDRESS_COPY (&IpAddress, &svptr->ServerIpAddress)
      IpPort = svptr->ServerPort;

      /**********************************/
      /* we may need to try this twice */
      /**********************************/

      /*
         Once to try an bind to any non-primary or supplied IP address.
         If this fails then a second attempt against address INADDR_ANY.
      */

      svptr->ServerBindStatus = SS$_NORMAL;

      for (;;)
      {
         /* status explicitly set to zero (is checked for at end of loop!) */
         ServerChannel = status = 0;

         /********************/
         /* existing socket? */
         /********************/

         /* check if this instance has a channel to the socket */
         SocketDevNamePtr = InstanceSocket (&IpAddress, IpPort, NULL);
         if (SocketDevNamePtr && SocketDevNamePtr[0] == '_')
         {
            /*******/
            /* yes */
            /*******/

            /* step over the indicative leading underscore */
            SocketDevNamePtr++;
            /* find it by device name */
            for (tsvptr = ServiceListHead; tsvptr; tsvptr = tsvptr->NextPtr)
               if (strsame (SocketDevNamePtr, tsvptr->BgDevName, -1)) break;
            if (!tsvptr)
            {
               char  String [256];
               WriteFao (String, sizeof(String), NULL, "%&I,!UL !AZ",
                         &IpAddress, IpPort, SocketDevNamePtr);
               ErrorExitVmsStatus (SS$_BUGCHECK, String, FI_LI);
            }

            if (svptr->RequestScheme != tsvptr->RequestScheme)
            {
               WriteFaoStdout (
"-!AZ-W-SERVICE, IP address and port already in use for \
!&?SSL\rHTTP\r service\n",
                  Utility, svptr->RequestScheme == SCHEME_HTTP);
               break;
            }

            if (svptr->RequestScheme == SCHEME_HTTPS)
            {
               /* shared, initialize by "cloning" the original service */
               WriteFaoStdout (
"-!AZ-W-SERVICE, IP address and port shared with existing SSL service\n",
                  Utility);
               if (!SesolaInitService (svptr, tsvptr)) break;
            }

            svptr->ServerBindStatus = 0;
            IPADDRESS_COPY (&svptr->ServerIpAddress, &tsvptr->ServerIpAddress);
            svptr->ServerChannel = tsvptr->ServerChannel;
            strcpy (svptr->BgDevName, tsvptr->BgDevName);
            break;
         }

         /* if an SSL service then initialize, problem just continue */
         if (svptr->RequestScheme == SCHEME_HTTPS)
         {
            if (!svptr->SSLserverPtr)
               svptr->SSLserverPtr = VmGet (sizeof(SESOLA_CONTEXT));
            if (!SesolaInitService (svptr, NULL))
            {
               WriteFaoStdout ("-!AZ-W-SSL, service not configured\n", Utility);
               break;
            }
         }

         /* if HTTP-SSL proxy service then initialize, problem just continue */
         if (svptr->SSLclientEnabled)
            if (!SesolaInitClientService (svptr))
            {
               WriteFaoStdout ("-!AZ-W-SSL, client not configured\n", Utility);
               break;
            }

         /******************************************/
         /* create and/or assign channel to socket */
         /******************************************/

         if (!NetListenBytLmRequired) BytLmBefore = GetJpiBytLm ();

         if (SocketDevNamePtr)
            status = SS$_NORMAL;
         else
         {
            /*****************/
            /* create socket */
            /*****************/

            /* create it now then */
            status = sys$assign (&TcpIpDeviceDsc, &ServerChannel, 0, 0);
            if (VMSnok (status))
               ErrorExitVmsStatus (status, "sys$assign()", FI_LI);

            /* prepare to bind the server socket to the IP address and port */
            if (IPADDRESS_IS_V4 (&IpAddress))
            {
               SOCKADDRESS_ZERO4 (&svptr->ServerSocketName)
               sin4ptr = &svptr->ServerSocketName.sa.v4;
               sin4ptr->SIN$W_FAMILY = TCPIP$C_AF_INET;
               sin4ptr->SIN$W_PORT = htons(IpPort);
               IPADDRESS_SET4 (sin4ptr->SIN$L_ADDR, &IpAddress)

               il2ptr = &SocketNameItem;
               il2ptr->buf_len = sizeof(SOCKADDRIN);
               il2ptr->item = 0;
               il2ptr->buf_addr = sin4ptr;

               TcpSocketPtr = &TcpIpSocket4;
            }
            else
            if (IPADDRESS_IS_V6 (&IpAddress))
            {
               SOCKADDRESS_ZERO6 (&svptr->ServerSocketName)
               sin6ptr = &svptr->ServerSocketName.sa.v6;
               sin6ptr->SIN6$B_FAMILY = TCPIP$C_AF_INET6;
               sin6ptr->SIN6$W_PORT = htons(IpPort);
               IPADDRESS_SET6 (&sin6ptr->SIN6$R_ADDR_OVERLAY.SIN6$T_ADDR,
                               &IpAddress)

               il2ptr = &SocketNameItem;
               il2ptr->buf_len = sizeof(SOCKADDRIN6);
               il2ptr->item = 0;
               il2ptr->buf_addr = sin6ptr;

               TcpSocketPtr = &TcpIpSocket6;
            }
            else
               ErrorExitVmsStatus (SS$_BUGCHECK, ErrorSanityCheck, FI_LI);

            /* make the channel a TCP, connection-oriented socket */
            status = sys$qiow (EfnWait, ServerChannel, IO$_SETMODE,
                               &svptr->ServerIOsb, 0, 0, TcpSocketPtr, 0, 0, 0,
                               &TcpIpSocketReuseAddrOption, 0);
            if (VMSok (status)) status = svptr->ServerIOsb.Status;
            if (VMSnok (status))
               ErrorExitVmsStatus (status, "sys$qiow()", FI_LI);

            if (VMSok (status) && svptr->AdminService)
            {
               /* admin service chooses the first available of a range */
               while (IpPort < 65535)
               {
                  status = sys$qiow (EfnWait, ServerChannel, IO$_SETMODE,
                                     &svptr->ServerIOsb, 0, 0,
                                     0, 0, &SocketNameItem,
                                     svptr->ListenBacklog, 0, 0);
                  if (VMSok (status)) status = svptr->ServerIOsb.Status;
                  if (status != SS$_DUPLNAM) break;
                  IpPort++;
                  if (IPADDRESS_IS_V4 (&IpAddress))
                     sin4ptr->SIN$W_PORT = htons(IpPort);
                  else
                     sin6ptr->SIN6$W_PORT = htons(IpPort);
               }
            }
            else
            if (VMSok (status))
            {
               /* no existing device bound to this address and port */
               status = sys$qiow (EfnWait, ServerChannel, IO$_SETMODE,
                                  &svptr->ServerIOsb, 0, 0,
                                  0, 0, &SocketNameItem,
                                  svptr->ListenBacklog, 0, 0);
               if (VMSok (status)) status = svptr->ServerIOsb.Status;
            }
            if (VMSok (status))
               SocketDevNamePtr = NetGetBgDevice(ServerChannel, NULL, 0);
         }

         if (VMSok (status))
         {
            /*********************************************/
            /* assign channel to existing/created socket */
            /*********************************************/

            strcpy (svptr->BgDevName, SocketDevNamePtr);
            BgDevNameDsc.dsc$a_pointer = svptr->BgDevName;
            BgDevNameDsc.dsc$w_length = strlen(svptr->BgDevName);
            status = sys$assign (&BgDevNameDsc, &svptr->ServerChannel, 0, 0);
            if (VMSnok (status))
            {
               WriteFaoStdout (
"-!AZ-W-SERVICE, error assigning channel to !AZ \
(!&I!&?(INADDR_ANY)\r\r!UL)\n-!&M\n",
                  Utility, SocketDevNamePtr,
                  &IpAddress, IPADDRESS_IS_ANY(&IpAddress),
                  IpPort, status);
            }
         }
         else
         {
            WriteFaoStdout (
"-!AZ-W-SERVICE, error binding to !&I!&?(INADDR_ANY)\r\r:!UL\n-\!&M\n",
               Utility, &IpAddress, IPADDRESS_IS_ANY(&IpAddress),
               IpPort, status);
         }

         if (VMSok(status) || IPADDRESS_IS_ANY(&IpAddress)) break;

         /* this time try to bind to 'any' address it can! */
         if (ServerChannel) sys$dassgn (ServerChannel);
         svptr->ServerBindStatus = status;
         IPADDRESS_SET_ANY (&IpAddress)
         WriteFaoStdout ("-!AZ-W-SERVICE, try again using INADDR_ANY\n",
                         Utility);

         /*************/
         /* try again */
         /*************/
      }

      /* status explicitly set to zero, just continue with next service */
      if (!status) continue;

      IPADDRESS_COPY (&svptr->ServerIpAddress, &IpAddress)

      if (VMSok (status) && InstanceNodeConfig > 1)
      {
         /* make the socket shareable (seems to work only if done here) */
         status = sys$qiow (EfnWait, svptr->ServerChannel, IO$_SETMODE,
                            &svptr->ServerIOsb, 0, 0,
                            0, 0, 0, 0, &TcpIpSocketShareOption, 0);
         if (VMSok (status)) status = svptr->ServerIOsb.Status;
         if (VMSnok (status)) ErrorExitVmsStatus (status, "sys$qiow()", FI_LI);
      }

      if (VMSnok (status))
      {
         /***********/
         /* problem */
         /***********/

         if (ServerChannel) sys$dassgn (ServerChannel);
         svptr->ServerBindStatus = status;
         svptr->ServerChannel = svptr->AdminPort = 0;
         svptr->BgDevName[0] = '\0';
         continue;
      }

      /***********/
      /* success */
      /***********/

      if (WATCH_MODULE(WATCH_MOD_NET))
         WatchThis (NULL, FI_LI, WATCH_MOD_NET,
            "LISTEN !AZ !&I!&?(INADDR_ANY)\r\r,!UL (!UL)",
            svptr->BgDevName, &IpAddress, IPADDRESS_IS_ANY(&IpAddress),
            IpPort, svptr->ListenBacklog);

      if (svptr->AdminService)
      {
         /* set the IP address and port of this instance's admin service */
         InstanceSocketAdmin ((short)IpPort);
         svptr->AdminPort = IpPort;
      }
      else
      if (ServerChannel)
      {
         /* inform the instance socket lock and table of the new device name */
         InstanceSocket (NULL, 0, svptr->BgDevName);
      }
      if (ServerChannel) sys$dassgn (ServerChannel);

      if (!NetListenBytLmRequired)
      {
         BytLmAfter = GetJpiBytLm ();
         NetListenBytLmRequired = BytLmBefore - BytLmAfter;
      }

      if (IPADDRESS_IS_V6 (&IpAddress)) TcpIpv6Configured = true;

      /* queue the first accept */
      NetAccept (svptr);
   }

   /* finished with service creation */
   if (VMSnok (status = InstanceUnLock (INSTANCE_NODE_SOCKET)))
      ErrorExitVmsStatus (status, "InstanceUnLock()", FI_LI);

   /* should be unnecessary these days, allow it by defining this logical */
   if (getenv ("NET_MULTIHOMED_HOST"))
   {
      /* determine if we have a multi-homed host */
      for (svptr = ServiceListHead; svptr; svptr = svptr->NextPtr)
         for (tsvptr = ServiceListHead; tsvptr; tsvptr = tsvptr->NextPtr)
            if (tsvptr->ServerSocketName.sa.v4.SIN$L_ADDR &&
                tsvptr->ServerSocketName.sa.v4.SIN$L_ADDR !=
                svptr->ServerSocketName.sa.v4.SIN$L_ADDR)
            {
               NetMultiHomedHost = true;
               break;
            }

      if (WATCH_MODULE(WATCH_MOD_NET))
         WatchThis (NULL, FI_LI, WATCH_MOD_NET,
                    "NetMultiHomedHost: !&B", NetMultiHomedHost);
   }

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Just get the "BGnnn:" device name associated with the channel, returning it in
the storage supplied.  If the storage pointer is NULL internal, static storage
is used ... good for one call per whatever.  If an error occurs the message
string is returned instead.
*/

char* NetGetBgDevice
(
unsigned short Channel,
char *DevName,
int SizeOfDevName
)
{
   static char  StaticDevName [64];
   static unsigned short  Length;
   static VMS_ITEM_LIST3  DevNamItemList [] = 
   {
      { 0, DVI$_DEVNAM, 0, &Length },
      { 0, 0, 0, 0 }
   };

   int  status;
   IO_SB  IOsb;

   /*********/
   /* begin */
   /*********/

   if (WATCH_MODULE(WATCH_MOD_NET))
      WatchThis (NULL, FI_LI, WATCH_MOD_NET, "NetGetBgDevice()");

   if (!DevName)
   {
      DevName = StaticDevName;
      SizeOfDevName = sizeof(StaticDevName);
   }

   DevNamItemList[0].buf_addr = DevName;
   DevNamItemList[0].buf_len = SizeOfDevName-1;

   status = sys$getdviw (EfnWait, Channel, 0, &DevNamItemList, &IOsb, 0, 0, 0);
   if (VMSok (status)) status = IOsb.Status;
   if (VMSok (status))
      DevName[Length] = '\0';
   else
      WriteFao (DevName, SizeOfDevName, NULL, "%!&M", status);

   if (WATCH_MODULE(WATCH_MOD_NET))
      WatchThis (NULL, FI_LI, WATCH_MOD_NET, "!AZ", DevName);

   if (DevName[0] == '_') return (DevName+1);
   return (DevName);
}

/*****************************************************************************/
/*
Get the device reference count of the supplied channel.
*/

int NetGetRefCnt (unsigned short Channel)

{
   static int  DviRefCnt;
   static VMS_ITEM_LIST3  DevNamItemList [] = 
   {
      { sizeof(DviRefCnt), DVI$_REFCNT, &DviRefCnt, 0 },
      { 0, 0, 0, 0 }
   };

   int  status;
   IO_SB  IOsb;

   /*********/
   /* begin */
   /*********/

   if (WATCH_MODULE(WATCH_MOD_NET))
      WatchThis (NULL, FI_LI, WATCH_MOD_NET, "NetGetRefCnt()");

   status = sys$getdviw (EfnWait, Channel, 0, &DevNamItemList, &IOsb, 0, 0, 0);
   if (VMSok (status)) status = IOsb.Status;
   if (VMSok (status)) return (DviRefCnt);
   return (0);
}

/*****************************************************************************/
/*
Zero the per-service accounting counters.
*/ 

NetServiceZeroAccounting ()

{
   SERVICE_STRUCT  *svptr;

   /*********/
   /* begin */
   /*********/

   if (WATCH_MODULE(WATCH_MOD_NET))
      WatchThis (NULL, FI_LI, WATCH_MOD_NET, "NetServiceZeroAccounting()");

   for (svptr = ServiceListHead; svptr; svptr = svptr->NextPtr)
   {
      svptr->ConnectCount =
         svptr->ReadErrorCount =
         svptr->WriteErrorCount = 0;
      memset (&svptr->QuadBytesRawRx, 0, 8);
      memset (&svptr->QuadBytesRawTx, 0, 8);
   }
}

/*****************************************************************************/
/*
Return a pointer to the name of the next service.  Begin with 'ContextPtr' set
to zero, then call successively until a NULL is returned.
*/ 

char* NetServiceNextHostPort (unsigned long *ContextPtr)

{
   char  *cptr;
   SERVICE_STRUCT  *svptr;

   /*********/
   /* begin */
   /*********/

   if (WATCH_MODULE(WATCH_MOD_NET))
      WatchThis (NULL, FI_LI, WATCH_MOD_NET, "NetServiceNextHostPort()");

   if (!*ContextPtr)
      *ContextPtr = (unsigned long)(svptr = ServiceListHead);
   else
   {
      svptr = (SERVICE_STRUCT*)*ContextPtr;
      *ContextPtr = (unsigned long)(svptr = svptr->NextPtr);
   }
   if ((cptr = (char*)svptr)) cptr = svptr->ServerHostPort;

   if (WATCH_MODULE(WATCH_MOD_NET))
      WatchThis (NULL, FI_LI, WATCH_MOD_NET, "!AZ", cptr);

   return (cptr);
}

/*****************************************************************************/
/*
Output service statistics (called from AdminReportServerStats()).
*/ 

int NetServiceReportStats (REQUEST_STRUCT *rqptr)

{
   static char  ServicesFao [] =
"<P><TABLE CELLPADDING=3 CELLSPACING=0 BORDER=1>\n\
<TR><TH>!&?\rInstance \rServices</TH></TR>\n\
<TR><TD>\n\
<TABLE CELLPADDING=0 CELLSPACING=5 BORDER=0>\n\
<TR><TH></TH><TH></TH>\
<TH ALIGN=left>&nbsp;&nbsp;<U>IP</U>&nbsp;&nbsp;</TH>\
<TH ALIGN=right><U>Count</U></TH>\
<TH ALIGN=right>&nbsp;&nbsp;<U><NOBR><FONT SIZE=-1>bytes</FONT> Rx</U></TH>\
<TH ALIGN=right><U><FONT SIZE=-1>err</FONT></U></TH>\
<TH ALIGN=right>&nbsp;&nbsp;<U><NOBR><FONT SIZE=-1>bytes</FONT> Tx</U></TH>\
<TH ALIGN=right><U><FONT SIZE=-1>err</FONT></U></TH>\
<TH ALIGN=right>&nbsp;&nbsp;<U>Traffic</U></TH></TR>\n";

   static char  OneServiceFao [] =
"<TR><TH ALIGN=right>!UL.&nbsp;</TH>\
<TD ALIGN=left>!AZ//!AZ</TD>\
<TD ALIGN=left>&nbsp;&nbsp;!&?v4\rv6\r</TD>\
<TD ALIGN=right>!&,UL</TD>\
<TD ALIGN=right>&nbsp;&nbsp;!&,@SQ</TD>\
<TD ALIGN=right>&nbsp;!&,UL</TD>\
<TD ALIGN=right>&nbsp;&nbsp;!&,@SQ</TD>\
<TD ALIGN=right>&nbsp;!&,UL</TD>\
<TD ALIGN=right>&nbsp;&nbsp;!UL%</TD></TR>\n";

   static char  TotalFao [] =
"<TR><TH COLSPAN=3 ALIGN=right>total:</TH>\
<TD ALIGN=right>!&,UL</TD>\
<TD ALIGN=right>&nbsp;&nbsp;!&,@SQ</TD>\
<TD ALIGN=right>&nbsp;!&,UL</TD>\
<TD ALIGN=right>&nbsp;&nbsp;!&,@SQ</TD>\
<TD ALIGN=right>&nbsp;!&,UL</TD>\
</TR>\n\
<TR><TD COLSPAN=4><FONT SIZE=-1>\
<SUP>*</SUP><I>counts are per-startup only</I></FONT></TD></TR>\n\
</TABLE>\n\
</TD></TR>\n\
</TABLE>\n";

   static long  Addx2 = 2;

   int  status,
        PercentService,
        ServiceListCount,
        ServiceTotalCount;
   unsigned short  Length;
   unsigned long  *vecptr;
   unsigned long  NetReadErrorTotal,
                  NetWriteErrorTotal;
   unsigned long  FaoVector [32],
                  QuadBytesRawRx [2],
                  QuadBytesRawTx [2],
                  QuadBytesTotal [2];
   float  FloatBytes,
          FloatBytesTotal,
          FloatPercent;
   SERVICE_STRUCT  *svptr;

   /*********/
   /* begin */
   /*********/

   if (WATCH_MODULE(WATCH_MOD_NET))
      WatchThis (NULL, FI_LI, WATCH_MOD_NET, "NetServiceReportStats()");

   NetReadErrorTotal =
      NetWriteErrorTotal =
      ServiceTotalCount = 0;
   QuadBytesRawRx[0] = QuadBytesRawRx[1] =
      QuadBytesRawTx[0] = QuadBytesRawTx[1] =
      QuadBytesTotal[0] = QuadBytesTotal[1] = 0;

   /* accumulate the raw (network) bytes for the services */
   for (svptr = ServiceListHead; svptr; svptr = svptr->NextPtr)
   {
      ServiceTotalCount += svptr->ConnectCount;

      status = lib$addx (&svptr->QuadBytesRawRx, &QuadBytesRawRx,
                         &QuadBytesRawRx, &Addx2);

      status = lib$addx (&svptr->QuadBytesRawTx, &QuadBytesRawTx,
                         &QuadBytesRawTx, &Addx2);
   }

   status = lib$addx (&QuadBytesRawRx, &QuadBytesRawTx,
                      &QuadBytesTotal, &Addx2);

   FloatBytesTotal = (float)QuadBytesTotal[0] +
                     (float)QuadBytesTotal[1] * (float)0xffffffff;

   vecptr = FaoVector;
   *vecptr++ = (InstanceNodeConfig <= 1);
   status = NetWriteFaol (rqptr, ServicesFao, &FaoVector);
   if (VMSnok (status)) ErrorNoticed (status, "NetWriteFaol()", FI_LI);

   ServiceListCount = 1;
   for (svptr = ServiceListHead; svptr; svptr = svptr->NextPtr)
   {
      FloatBytes = (float)svptr->QuadBytesRawRx[0] +
                   ((float)svptr->QuadBytesRawRx[1] * (float)0xffffffff) +
                   (float)svptr->QuadBytesRawTx[0] +
                   ((float)svptr->QuadBytesRawTx[1] * (float)0xffffffff);

      if (FloatBytesTotal > 0.0)
      {
         PercentService = (int)(FloatPercent =
                                FloatBytes * 100.0 / FloatBytesTotal);
         if (FloatPercent - (float)PercentService >= 0.5) PercentService++;
      }
      else
         PercentService = 0;

      NetReadErrorTotal += svptr->ReadErrorCount;
      NetWriteErrorTotal += svptr->WriteErrorCount;

      vecptr = FaoVector;
      *vecptr++ = ServiceListCount++;
      *vecptr++ = svptr->RequestSchemeNamePtr;
      *vecptr++ = svptr->ServerHostPort;
      *vecptr++ = IPADDRESS_IS_V4(&svptr->ServerIpAddress);
      *vecptr++ = svptr->ConnectCount;
      *vecptr++ = &svptr->QuadBytesRawRx;
      *vecptr++ = svptr->ReadErrorCount;
      *vecptr++ = &svptr->QuadBytesRawTx;
      *vecptr++ = svptr->WriteErrorCount;
      *vecptr++ = PercentService;

      status = NetWriteFaol (rqptr, OneServiceFao, &FaoVector);
      if (VMSnok (status)) ErrorNoticed (status, "NetWriteFaol()", FI_LI);
   }

   vecptr = FaoVector;

   *vecptr++ = ServiceTotalCount;
   *vecptr++ = &QuadBytesRawRx;
   *vecptr++ = NetReadErrorTotal;
   *vecptr++ = &QuadBytesRawTx;
   *vecptr++ = NetWriteErrorTotal;

   status = NetWriteFaol (rqptr, TotalFao, &FaoVector);
   if (VMSnok (status)) ErrorNoticed (status, "NetWriteFaol()", FI_LI);

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
When the server is active this function gets called once a second by
HttpdTick().  If the sys$assign() in NetAccept() failed due to IO channels
being exhausted this function attempts to requeue that accept, until it
succeeds or the server-wide attempt times-out.
*/ 

BOOL NetAcceptSupervisor ()

{
   int  status;
   SERVICE_STRUCT  *svptr;

   /*********/
   /* begin */
   /*********/

   if (WATCH_MODULE(WATCH_MOD_NET) && WATCH_MODULE(WATCH_MOD__DETAIL))
      WatchThis (NULL, FI_LI, WATCH_MOD_NET, "NetAcceptSupervisor()");

   if (!NetAcceptNoIoChan) return (false);

   if (!NetAcceptNoIoChanCount)
      ErrorNoticed (SS$_NOIOCHAN, "NetAcceptSupervisor()", FI_LI);
   NetAcceptNoIoChan = false;
   for (svptr = ServiceListHead; svptr; svptr = svptr->NextPtr)
      NetAccept (svptr);
   if (NetAcceptNoIoChan)
   {
      if (NetAcceptNoIoChanCount++ < NET_ACCEPT_NOIOCHAN_MAX) return (true);
      ErrorExitVmsStatus (SS$_BUGCHECK, "NetAcceptSupervisor()", FI_LI);
   }
   ErrorNoticed (SS$_NORMAL, "NetAcceptSupervisor()", FI_LI);
   NetAcceptNoIoChanCount = 0;
   return (false);
}

/*****************************************************************************/
/*
Queue an accept() to the listening server socket.
*/ 

NetAccept (SERVICE_STRUCT *svptr)

{
   int  status,
        BytLmAfter,
        BytLmBefore;
   unsigned short  Channel;
   REQUEST_STRUCT  *rqptr;
   VMS_ITEM_LIST3  *il3ptr;

   /*********/
   /* begin */
   /*********/

   if (WATCH_MODULE(WATCH_MOD_NET))
      WatchThis (NULL, FI_LI, WATCH_MOD_NET,
                 "NetAccept() !UL", svptr->AcceptQueued);

   /* server channel has been shutdown, most probably restart/exit underway */
   if (!svptr->ServerChannel) return;

   /* a service sharing a previously configured socket address/port */
   if (!svptr->BgDevName[0]) return;

   /* only need the one queued at a time */
   if (svptr->AcceptQueued) return;

   if (!NetAcceptBytLmRequired) BytLmBefore = GetJpiBytLm ();

   /* assign a channel to the internet template device */
   status = sys$assign (&TcpIpDeviceDsc, &Channel, 0, 0);
   if (VMSnok (status))
   {
      if (status == SS$_NOIOCHAN)
      {
         /* shouldn't have exhausted these, but seem to have, retry shortly */
         NetAcceptNoIoChan = true;
         return;
      }
      /* some other (serious) error */
      ErrorExitVmsStatus (status, "sys$assign()", FI_LI);
   }

   /* allocate zeroed dynamic memory for the connection thread */
   rqptr = VmGetRequest ();
   rqptr->ServicePtr = svptr;

   rqptr->rqClient.Channel = Channel;

   if (IPADDRESS_IS_V4 (&svptr->ServerIpAddress))
   {
      SOCKADDRESS_ZERO4 (&rqptr->rqClient.SocketName)
      il3ptr = &rqptr->rqClient.SocketNameItem;
      il3ptr->buf_len = sizeof(SOCKADDRIN);
      il3ptr->item = 0;
      il3ptr->buf_addr = &rqptr->rqClient.SocketName.sa.v4;
      il3ptr->ret_len = &rqptr->rqClient.SocketNameLength;
   }
   else
   if (IPADDRESS_IS_V6 (&svptr->ServerIpAddress))
   {
      SOCKADDRESS_ZERO6 (&rqptr->rqClient.SocketName)
      il3ptr = &rqptr->rqClient.SocketNameItem;
      il3ptr->buf_len = sizeof(SOCKADDRIN6);
      il3ptr->item = 0;
      il3ptr->buf_addr = &rqptr->rqClient.SocketName.sa.v6;
      il3ptr->ret_len = &rqptr->rqClient.SocketNameLength;
   }
   else
      ErrorExitVmsStatus (SS$_BUGCHECK, ErrorSanityCheck, FI_LI);

   status = sys$qio (EfnNoWait, svptr->ServerChannel,
                     IO$_ACCESS | IO$M_ACCEPT,
                     &rqptr->rqClient.IOsb, &NetAcceptAst, rqptr,
                     0, 0, &rqptr->rqClient.SocketNameItem,
                     &rqptr->rqClient.Channel, &TcpIpFullDuplexCloseOption, 0);
   if (VMSnok (status))  ErrorExitVmsStatus (status, "sys$qio()", FI_LI);

   svptr->AcceptQueued++;

   if (!NetAcceptBytLmRequired)
   {
      BytLmAfter = GetJpiBytLm ();
      NetAcceptBytLmRequired = BytLmBefore - BytLmAfter;
   }

   if (WATCH_CATEGORY(WATCH_NETWORK))
      WatchThis (NULL, FI_LI, WATCH_NETWORK,
         "ACCEPT !AZ !&I!&?(INADDR_ANY)\r\r,!AZ",
         NetGetBgDevice(svptr->ServerChannel, NULL, 0),
         &svptr->ServerIpAddress,
         IPADDRESS_IS_ANY(&svptr->ServerIpAddress),
         svptr->ServerPortString);
}

/*****************************************************************************/
/*
A connection has been accept()ed on the specified server socket.
*/ 

NetAcceptAst (REQUEST_STRUCT *rqptr)

{
   int  status;
   SERVICE_STRUCT  *svptr;

   /*********/
   /* begin */
   /*********/

   if (WATCH_MODULE(WATCH_MOD_NET))
      WatchThis (NULL, FI_LI, WATCH_MOD_NET,
                 "NetAcceptAst() !&F !UL !&S !&X",
                 &NetAcceptAst, rqptr->ServicePtr->AcceptQueued, 
                 rqptr->rqClient.IOsb.Status, rqptr->rqClient.IOsb.Unused);

   svptr = rqptr->ServicePtr;

   if (svptr->AcceptQueued) svptr->AcceptQueued--;

   if (VMSnok (status = rqptr->rqClient.IOsb.Status)) 
   {
      if (ControlExitRequested)
      {
         /* server exiting or restarting, no new accepts, just return */
         return;
      }

      /* if connect dropped, forget it, ready for next connection */
      if (status == SS$_CANCEL ||
          status == SS$_ABORT ||
          status == SS$_CONNECFAIL ||
          status == SS$_LINKABORT ||
          status == SS$_REJECT ||
          status == SS$_TIMEOUT ||
          status == SS$_INSFMEM)
      {
         /* if server channel zero most probably restart/exit underway */
         if (!svptr->ServerChannel) return;

         if (status != SS$_CONNECFAIL)
            ErrorNoticed (status, "sys$qio()", FI_LI);

         /* deassign socket channel, deallocate request structure */
         sys$dassgn (rqptr->rqClient.Channel);
         VmFreeRequest (rqptr, FI_LI);

         /* queue up the next request acceptance */
         NetAccept (svptr);
         return;
      }

      /* most often network/system shutting down ... SS$_SHUT */
      ErrorExitVmsStatus (status, "accept()", FI_LI);
   }

#if NET_TEST

   NetTestRequest (rqptr->rqClient.Channel);
   NetAccept (svptr);
   return;

#endif /* NET_TEST */

   if (svptr->RequestScheme == SCHEME_HTTPS)
      InstanceGblSecIncrLong (&AccountingPtr->ConnectSslCount);

   if (IPADDRESS_IS_V4 (&svptr->ServerIpAddress))
   {
      InstanceGblSecIncrLong (&AccountingPtr->ConnectIpv4Count);
      rqptr->rqClient.IpPort =
         ntohs(rqptr->rqClient.SocketName.sa.v4.SIN$W_PORT);
      IPADDRESS_GET4 (&rqptr->rqClient.IpAddress,
                      rqptr->rqClient.SocketName.sa.v4.SIN$L_ADDR);
      strcpy (rqptr->rqClient.IpAddressString,
         TcpIpAddressToString (&rqptr->rqClient.IpAddress, 0));
   }
   else
   if (IPADDRESS_IS_V6 (&svptr->ServerIpAddress))
   {
      InstanceGblSecIncrLong (&AccountingPtr->ConnectIpv6Count);
      rqptr->rqClient.IpPort =
         ntohs(rqptr->rqClient.SocketName.sa.v6.SIN6$W_PORT);
      IPADDRESS_GET6 (&rqptr->rqClient.IpAddress,
         rqptr->rqClient.SocketName.sa.v6.SIN6$R_ADDR_OVERLAY.SIN6$T_ADDR)
      strcpy (rqptr->rqClient.IpAddressString,
         TcpIpAddressToString (&rqptr->rqClient.IpAddress, 0));
   }
   else
      ErrorExitVmsStatus (SS$_BUGCHECK, ErrorSanityCheck, FI_LI);

   if (Config.cfMisc.DnsLookupClient)
   {
      if (WATCH_CATEGORY(WATCH_NETWORK))
         WatchThis (NULL, FI_LI, WATCH_NETWORK,
            "RESOLVE !&I", &rqptr->rqClient.IpAddress);

      /* asynchronous DNS lookup */
      TcpIpAddressToName (&rqptr->rqClient.Lookup,
                          &rqptr->rqClient.IpAddress,
                          ConfigDnsLookupRetryCount,
                          &NetAcceptProcess, rqptr);
      /* meanwhile queue another accept using a new request structure */
      NetAccept (svptr);
   }
   else
   {
      /* not using client DNS lookup, carry on synchronously */
      NetAcceptProcess (rqptr);
   }
}

/*****************************************************************************/
/*
This function can be called either as an AST by TcpIpAdddressToName() if DNS
host name lookup is enabled, or directly from NetAccept() if it's disabled. 
Either way just continue to process the connection accept.
*/ 

NetAcceptProcess (REQUEST_STRUCT *rqptr)

{
   BOOL WatchThisOne;
   int  status,
        SocketNameLength,
        ServerSocketNameLength;
   unsigned short  PortNumber;              
   char  ServerIpAddressString [32];
   SERVICE_STRUCT  *svptr,
                   *tsvptr;
   IO_SB  IOsb;
   SOCKADDRESS  SocketName;
   VMS_ITEM_LIST3  SocketNameItem;
   VMS_ITEM_LIST3  *il3ptr;

   /*********/
   /* begin */
   /*********/

   if (WATCH_MODULE(WATCH_MOD_NET))
      WatchThis (NULL, FI_LI, WATCH_MOD_NET,
                 "NetAcceptProcess() !&F !&S !UL",
                 &NetAcceptProcess, rqptr->rqClient.IOsb.Status,
                 rqptr->rqClient.Lookup.HostNameLength);

   svptr = rqptr->ServicePtr;

   if (VMSnok (rqptr->rqClient.Lookup.LookupIOsb.Status))
   {
      /* lookup not done or failed, substitute the IP address for the name */
      strcpy (rqptr->rqClient.Lookup.HostName,
         rqptr->rqClient.IpAddressString);
      rqptr->rqClient.Lookup.HostNameLength =
         strlen(rqptr->rqClient.Lookup.HostName);
   }

   if (WATCH_MODULE(WATCH_MOD_NET))
      WatchDataFormatted ("!&Z !&Z\n", rqptr->rqClient.IpAddressString,
                                       rqptr->rqClient.Lookup.HostName);

   /* see note above, where this boolean is set true */
   if (NetMultiHomedHost)
   {
      /************************************/
      /* multiple services, get host/port */
      /************************************/

      il3ptr = &SocketNameItem;
      il3ptr->buf_len = sizeof(SOCKADDRIN);
      il3ptr->item = 0;
      il3ptr->buf_addr = &SocketName.sa.v4;
      il3ptr->ret_len = &SocketNameLength;

      status = sys$qiow (EfnWait, rqptr->rqClient.Channel, IO$_SENSEMODE,
                         &IOsb, 0, 0, 0, 0, &SocketNameItem, 0, 0, 0);

      if (WATCH_MODULE(WATCH_MOD_NET))
         WatchThis (NULL, FI_LI, WATCH_MOD_NET,
                    "$QIOW() !&S !&S !4&I,!UL",
                    status, IOsb.Status,
                    SocketName.sa.v4.SIN$L_ADDR,
                    ntohs(SocketName.sa.v4.SIN$W_PORT));

      if (VMSok (status)) status = IOsb.Status;
      if (VMSnok (status))
      {
         /* hmmm, forget it, ready for next connection */
         ErrorNoticed (status, "sys$qiow()", FI_LI);

         /* deassign socket channel, deallocate request structure */
         sys$dassgn (rqptr->rqClient.Channel);
         VmFreeRequest (rqptr, FI_LI);

         NetAccept (svptr);
         return;
      }

      PortNumber = ntohs(SocketName.sa.v4.SIN$W_PORT);
      for (tsvptr = ServiceListHead; tsvptr; tsvptr = tsvptr->NextPtr)
      {
         if (SocketName.sa.v4.SIN$L_ADDR ==
             IPADDRESS_ADR4(&tsvptr->ServerIpAddress) &&
             (PortNumber == tsvptr->ServerPort ||
              PortNumber == tsvptr->AdminPort)) break;
      }
      if (!tsvptr)
      {
         if (WATCH_MODULE(WATCH_MOD_NET))
            WatchThis (NULL, FI_LI, WATCH_MOD_NET, "UNKNOWN multi-homed host");
         InstanceGblSecIncrLong (&AccountingPtr->ConnectRejectedCount);
         InstanceGblSecIncrLong (&AccountingPtr->ResponseStatusCodeCountArray[4]);
         NetDirectResponse (rqptr, MSG_GENERAL_ACCESS_DENIED);
         /* ready for the next connection */
         NetAccept (svptr);
         return;
      }
      /* point instead to the now multi-homed service */
      svptr = tsvptr;
   }

   if (WATCH_MODULE(WATCH_MOD_NET))
      WatchThis (NULL, FI_LI, WATCH_MOD_NET, "!AZ !AZ:!UL",
                 svptr->ServerIpAddressString,
                 svptr->ServerHostName, svptr->ServerPort);

   if (Watch.Category && Watch.Category != WATCH_ONE_SHOT_CAT)
      WatchThisOne = WatchFilter (rqptr->rqClient.Lookup.HostName,
                                  rqptr->rqClient.IpAddressString,
                                  svptr->RequestSchemeNamePtr,
                                  svptr->ServerHostPort,
                                  NULL, NULL);
   else
   if (Watch.AllRequests)
      WatchThisOne = true;
   else
      WatchThisOne = false;

   if (InstanceConnectCurrent >= InstanceConcurrentMax)
   {
      /************/
      /* too busy */
      /************/

      if (WatchThisOne && WATCH_CATEGORY(WATCH_CONNECT))
         WatchThis (NULL, FI_LI, WATCH_CONNECT,
                    "ACCEPTED too-busy !AZ,!UL on !AZ//!AZ,!AZ !AZ",
                     rqptr->rqClient.Lookup.HostName, rqptr->rqClient.IpPort, 
                     svptr->RequestSchemeNamePtr,
                     svptr->ServerIpAddressString,
                     svptr->ServerPortString,
                     NetGetBgDevice(rqptr->rqClient.Channel, NULL, 0));

      svptr->ConnectCount++;
      InstanceGblSecIncrLong (&AccountingPtr->ConnectTooBusyCount);
      InstanceGblSecIncrLong (&AccountingPtr->ResponseStatusCodeCountArray[5]);
      NetDirectResponse (rqptr, MSG_GENERAL_TOO_BUSY);
      /* ready for the next connection */
      NetAccept (svptr);
      return;
   }

   if (!ConfigAcceptClientHostName (rqptr->rqClient.IpAddressString,
                                    rqptr->rqClient.Lookup.HostName))
   {
      /************/
      /* rejected */
      /************/

      if (WatchThisOne && WATCH_CATEGORY(WATCH_CONNECT))
         WatchThis (NULL, FI_LI, WATCH_CONNECT,
                    "ACCEPTED reject !AZ,!UL on !AZ//!AZ,!AZ !AZ",
                     rqptr->rqClient.Lookup.HostName, rqptr->rqClient.IpPort, 
                     svptr->RequestSchemeNamePtr,
                     svptr->ServerIpAddressString,
                     svptr->ServerPortString,
                     NetGetBgDevice(rqptr->rqClient.Channel, NULL, 0));

      svptr->ConnectCount++;
      InstanceGblSecIncrLong (&AccountingPtr->ConnectRejectedCount);
      InstanceGblSecIncrLong (&AccountingPtr->ResponseStatusCodeCountArray[4]);
      NetDirectResponse (rqptr, MSG_GENERAL_ACCESS_DENIED);
      /* ready for the next connection */
      NetAccept (svptr);
      return;
   }

   InstanceGblSecIncrLong (&AccountingPtr->ConnectAcceptedCount);

   /***************************/
   /* process HTTP connection */
   /***************************/

   /* we have a client! */
   InstanceConnectCurrent++;
   rqptr->ConnectNumber = ++ConnectCountTotal;

   InstanceMutexLock (INSTANCE_MUTEX_HTTPD);
   AccountingPtr->ConnectCurrent++;
   if (AccountingPtr->ConnectCurrent > AccountingPtr->ConnectPeak)
      AccountingPtr->ConnectPeak = AccountingPtr->ConnectCurrent;
   InstanceMutexUnLock (INSTANCE_MUTEX_HTTPD);

   if (WatchThisOne)
   {
      /* assign a unique number for the life of this request structure */
      rqptr->WatchItem = ++Watch.Count;

      if WATCH_CATEGORY(WATCH_CONNECT)
         WatchThis (rqptr, FI_LI, WATCH_CONNECT,
                    "ACCEPTED !AZ,!UL on !AZ//!AZ,!AZ !AZ",
                    rqptr->rqClient.Lookup.HostName, rqptr->rqClient.IpPort, 
                    svptr->RequestSchemeNamePtr,
                    svptr->ServerIpAddressString,
                    svptr->ServerPortString,
                    NetGetBgDevice(rqptr->rqClient.Channel, NULL, 0));
   }

   /* begin processing this request */
   RequestBegin (rqptr, true);

   /* ready for the next connection */
   NetAccept (svptr);
}

/*****************************************************************************/
/*
'HostNamePort' can contain a "host.name:port" or just a "host.name" if the port
is supplied via 'PortNumber'.  Using the supplied or parsed host name get the
lookup host name into 'HostNamePtr', the decimal-dot-notation IP address string
into 'IpAddressStringPtr' and the 32 bit IP address into 'IpAddressPtr', the IP
port number (particularly if parsed from 'HostNamePort' in to 'IpPortPtr'.  Any
of the '...Ptr' parameters can be NULL and won't have the respective
information returned.
*/ 

int NetHostNameLookup
(
META_CONFIG *mcptr,
char *HostNamePort,
int PortNumber,
char *HostNamePtr,
char *HostPortPtr,
char *IpAddressStringPtr,
IPADDRESS *IpAddressPtr,
int *IpPortPtr
)
{

   static $DESCRIPTOR (HostPortFaoDsc, "!AZ:!UL\0");
   static $DESCRIPTOR (StringDsc, "");

   BOOL  FullyQualified;
   int  idx,
        status;
   char  *cptr, *sptr, *zptr,
         *ResolvedNamePtr;
   char  HostNameScratch [128];
   IPADDRESS  SuppliedIpAddress;
   TCPIP_HOST_LOOKUP  HostLookup;

   /*********/
   /* begin */
   /*********/

   if (WATCH_MODULE(WATCH_MOD_NET))
      WatchThis (NULL, FI_LI, WATCH_MOD_NET,
                 "NetHostNameLookup() !AZ", HostNamePort);

   FullyQualified = false;
   IPADDRESS_ZERO4 (&SuppliedIpAddress)
   HostNameScratch[0] = '\0';

   zptr = (sptr = HostNameScratch) + sizeof(HostNameScratch)-1;
   for (cptr = HostNamePort;
        *cptr && *cptr != ':' && sptr < zptr;
        *sptr++ = *cptr++)
       if (*cptr == '.') FullyQualified = true;
   *sptr = '\0';
   if (*cptr == ':' && !PortNumber) PortNumber = atoi(cptr+1);

   /*************************/
   /* UCX resolve host name */
   /*************************/

   memset (&HostLookup, 0, sizeof(HostLookup));
   status = TcpIpNameToAddress (&HostLookup, HostNameScratch, 0, NULL, 0);
   if (VMSnok (status)) return (status);
   ResolvedNamePtr = HostLookup.HostName;

   /*****************************/
   /* return values as required */
   /*****************************/

   if (FullyQualified)
   {
      /* looks better if its all in lower case (sometimes upper) */
      for (cptr = HostNameScratch; *cptr; cptr++) *cptr = tolower(*cptr);
   }
   else
   {
      /* use the resolved name */
      zptr = (sptr = HostNameScratch) + 127;
      /* looks better if its all in lower case (sometimes upper) */
      for (cptr = ResolvedNamePtr;
           *cptr && sptr < zptr;
           *sptr++ = tolower(*cptr++));
      *sptr = '\0';
   }

   if (HostNamePtr)
   {
      strcpy (HostNamePtr, HostNameScratch);
      if (WATCH_MODULE(WATCH_MOD_NET))
         WatchThis (NULL, FI_LI, WATCH_MOD_NET, "!AZ", HostNamePtr);
   }

   if (HostPortPtr)
   {
      StringDsc.dsc$a_pointer = HostPortPtr;
      StringDsc.dsc$w_length = 128+16;
      sys$fao (&HostPortFaoDsc, 0, &StringDsc, HostNameScratch, PortNumber);
      if (WATCH_MODULE(WATCH_MOD_NET))
         WatchThis (NULL, FI_LI, WATCH_MOD_NET, "!AZ", HostPortPtr);
   }

   if (IpAddressStringPtr)
   {
      /* convert the binary address into a string */
      strcpy (IpAddressStringPtr,
              TcpIpAddressToString (&HostLookup.IpAddress, 0));
      if (WATCH_MODULE(WATCH_MOD_NET))
         WatchThis (NULL, FI_LI, WATCH_MOD_NET, "!AZ", IpAddressStringPtr);
   }

   if (IpAddressPtr) IPADDRESS_COPY (IpAddressPtr, &HostLookup.IpAddress)
   if (IpPortPtr) *IpPortPtr = PortNumber;

   return (status);
}

/****************************************************************************/
/*
Just close the socket, bang!
*/

int NetCloseSocket (REQUEST_STRUCT *rqptr)

{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (WATCHING(rqptr) && WATCH_MODULE(WATCH_MOD_NET))
      WatchThis (rqptr, FI_LI, WATCH_MOD_NET, "NetCloseSocket()");

   if (!rqptr->rqClient.Channel) return (SS$_NORMAL);

   status = sys$dassgn (rqptr->rqClient.Channel);
   rqptr->rqClient.Channel = 0;
   if (rqptr->rqNet.SesolaPtr)
      SesolaNetSocketHasBeenClosed (rqptr->rqNet.SesolaPtr);

   if (WATCHING(rqptr) && WATCH_CATEGORY(WATCH_CONNECT))
   {
      if (VMSok(status))
         WatchThis (rqptr, FI_LI, WATCH_CONNECT, "CLOSE !AZ,!UL",
                    rqptr->rqClient.Lookup.HostName, rqptr->rqClient.IpPort); 
      else
         WatchThis (rqptr, FI_LI, WATCH_CONNECT, "CLOSE !AZ,!UL !&S %!-!&M",
                    rqptr->rqClient.Lookup.HostName, rqptr->rqClient.IpPort, status);
   }

   return (status);
}

/****************************************************************************/
/*
Stop the server from receiving incoming requests.
*/

NetShutdownServerSocket ()

{
   int  status;
   SERVICE_STRUCT  *svptr;

   /*********/
   /* begin */
   /*********/

   if (WATCH_MODULE(WATCH_MOD_NET))
      WatchThis (NULL, FI_LI, WATCH_MOD_NET, "NetShutdownServerSocket()");

   for (svptr = ServiceListHead; svptr; svptr = svptr->NextPtr)
   {
      status = sys$dassgn (svptr->ServerChannel);
      svptr->ServerChannel = 0;
   }
}

/*****************************************************************************/
/*
Called from NetWrite() is a response header needs to be sent before any data.
Response header has now been sent, send the data using the buffered
information about it.
*/ 

NetResponseHeaderAst (REQUEST_STRUCT *rqptr)

{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (WATCHING(rqptr) && WATCH_MODULE(WATCH_MOD_NET))
      WatchThis (rqptr, FI_LI, WATCH_MOD_NET,
                 "NetResponseHeaderAst() !&F !&X !&X !UL",
                 &NetResponseHeaderAst,
                 rqptr->rqResponse.HeaderAstFunction,
                 rqptr->rqResponse.HeaderDataPtr,
                 rqptr->rqResponse.HeaderDataLength);

   NetWrite (rqptr,
             rqptr->rqResponse.HeaderAstFunction,
             rqptr->rqResponse.HeaderDataPtr,
             rqptr->rqResponse.HeaderDataLength);
}

/*****************************************************************************/
/*
Write 'DataLength' bytes located at 'DataPtr' to the client either using either
the "raw" network or via the Secure Sockets Layer.

If 'AstFunction' zero then use sys$qiow(), waiting for completion. If
an AST completion address is supplied then use sys$qio().  If empty data
buffer is supplied (zero length) then declare an AST to service any AST
routine supplied.  If none then just return.

For responses generated by the server the HTTP header is a separate structure
which can be sent separately. If the HTTP method is "HEAD" only allow bytes in
the header to be sent, absorb any other, explicitly calling the AST completion
routine as necessary (this, in particular, is for scripts that don't recognise
this method).

Explicitly declares any AST routine if an error occurs. The calling function
must not do any error recovery if an AST routine has been supplied but the
associated AST routine must!  If an AST was not supplied then the return
status can be checked.
*/ 

int NetWrite
(
REQUEST_STRUCT *rqptr,
REQUEST_AST AstFunction,
char *DataPtr,
int DataLength
)
{
   int  cnt, nlcnt, status,
        ResponseHeaderLength;
   char  *cptr, *sptr,
         *ResponseHeaderPtr;

   /*********/
   /* begin */
   /*********/

   if (WATCHING(rqptr) && WATCH_MODULE(WATCH_MOD_NET))
      WatchThis (rqptr, FI_LI, WATCH_MOD_NET,
                 "NetWrite() !&A !&X !UL !&X !UL !&B !UL",
                 AstFunction, DataPtr, DataLength,
                 rqptr->rqResponse.HeaderPtr,
                 rqptr->rqResponse.HeaderLength,
                 rqptr->rqResponse.HeaderSent,
                 rqptr->rqResponse.HeaderNewlineCount);

   status = SS$_NORMAL;

   /* initiate cache load if .. */
   if (rqptr->rqPathSet.CacheNet &&   /* the path indicates it */
       !rqptr->rqCache.LoadCheck &&   /* not been checked already */
       !rqptr->rqCache.EntryPtr)      /* output is not from the cache! */
   {
      /* request output to be cached */
      rqptr->rqCache.LoadFromNet =
         CacheLoadBegin (rqptr, rqptr->rqResponse.ContentLength,
                                rqptr->rqResponse.ContentTypePtr);
   } 

   if (!rqptr->rqResponse.HeaderSent)
   {
      /*******************************************/
      /* nothing of response sent to client yet! */
      /*******************************************/

      rqptr->rqResponse.HeaderSent = true;

      if (rqptr->rqResponse.HeaderPtr &&
          rqptr->rqPathSet.ResponseHeaderNone &&
          rqptr->rqResponse.HttpStatus / 100 == 2)
      {
         /*********************/
         /* header suppressed */
         /*********************/

         if (WATCHING(rqptr) && WATCH_CATEGORY(WATCH_RESPONSE_HEADER))
            WatchThis (rqptr, FI_LI, WATCH_RESPONSE_HEADER,
                       "HEADER !UL bytes (NONE)",
                       rqptr->rqResponse.HeaderLength);
      }
      else
      if (rqptr->rqResponse.HeaderPtr &&
          rqptr->rqHeader.HttpVersion != HTTP_VERSION_0_9)
      {
         /************************/
         /* send response header */
         /************************/

         if (!rqptr->rqResponse.HeaderLength)
            rqptr->rqResponse.HeaderLength =
               strlen(rqptr->rqResponse.HeaderPtr);

         /* only need header via the network if it's not already generated */
         if (rqptr->rqCache.LoadFromNet &&
             !rqptr->rqResponse.ContentTypePtr)
            CacheLoadData (rqptr, rqptr->rqResponse.HeaderPtr,
                                  rqptr->rqResponse.HeaderLength);

         ResponseHeaderLength = rqptr->rqResponse.HeaderLength;

         if (rqptr->rqPathSet.ResponseHeaderBegin &&
             rqptr->rqResponse.HttpStatus / 100 == 2)
         {
            cptr = (sptr = rqptr->rqResponse.HeaderPtr) + ResponseHeaderLength;
            if (cptr > sptr && cptr[-1] == '\n')
            {
               cptr--;
               if (cptr > sptr && cptr[-1] == '\r') cptr--;
            }
            ResponseHeaderLength = cptr - sptr;
            cptr = " (BEGIN)";
         }
         else
            cptr = "";

         if (WATCHING(rqptr) && WATCH_CATEGORY(WATCH_RESPONSE_HEADER) &&
             rqptr != Watch.RequestPtr)
         {
            WatchThis (rqptr, FI_LI, WATCH_RESPONSE_HEADER,
                       "HEADER !UL bytes!AZ", ResponseHeaderLength, cptr);
            WatchData (rqptr->rqResponse.HeaderPtr, ResponseHeaderLength);
         }

         rqptr->BytesTx += ResponseHeaderLength;

         if (AstFunction)
         {
            /************************/
            /* AST routine, no wait */
            /************************/

            if (DataLength)
            {
               rqptr->rqResponse.HeaderAstFunction = AstFunction;
               rqptr->rqResponse.HeaderDataPtr = DataPtr;
               rqptr->rqResponse.HeaderDataLength = DataLength;
               AstFunction = &NetResponseHeaderAst;
            }

            if (rqptr->rqNet.SesolaPtr)
               status = SesolaNetWrite (rqptr->rqNet.SesolaPtr, AstFunction,
                                        rqptr->rqResponse.HeaderPtr,
                                        ResponseHeaderLength);
            else
               status = NetWriteRaw (rqptr, AstFunction,
                                     rqptr->rqResponse.HeaderPtr,
                                     ResponseHeaderLength);

            /* return, the AST function will sys$qiow() the data */
            return (status);
         }
         else
         {
            /************************/
            /* no AST routine, wait */
            /************************/

            if (rqptr->rqNet.SesolaPtr)
               status = SesolaNetWrite (rqptr->rqNet.SesolaPtr, 0,
                                        rqptr->rqResponse.HeaderPtr,
                                        ResponseHeaderLength);
            else
               status = NetWriteRaw (rqptr, 0,
                                     rqptr->rqResponse.HeaderPtr,
                                     ResponseHeaderLength);

            /* continue on to sys$qiow() the data */
         }
      }
   }

   /********/
   /* data */
   /********/

   rqptr->BytesTx += DataLength;

   if (!DataLength)
   {
      /* without a real network write just fudge the status */
      rqptr->rqNet.WriteIOsb.Status = SS$_NORMAL;
      rqptr->rqNet.WriteIOsb.Count = 0;
      if (!AstFunction) return (SS$_NORMAL);
      SysDclAst (AstFunction, rqptr);
      return (SS$_NORMAL);
   }

   if (rqptr->rqHeader.Method == HTTP_METHOD_HEAD)
   {
      /*****************************/
      /* send only response header */
      /*****************************/

      if (!rqptr->rqResponse.HeaderPtr &&
          rqptr->rqResponse.HeaderNewlineCount <= 1 &&
          !rqptr->ProxyTaskPtr)
      {
         /**********************/
         /* header from script */
         /**********************/

         nlcnt = rqptr->rqResponse.HeaderNewlineCount;
         cptr = DataPtr;
         cnt = DataLength;
         while (cnt--)
         {
            if (*cptr == '\n' && ++nlcnt == 2)
            {
               /* two successive end-of-lines, therefore end of header */
               cptr++;
               break;
            }
            else
            if (*cptr != '\r' && *cptr != '\n')
               nlcnt = 0;
            cptr++;
         }

         if ((rqptr->rqResponse.HeaderNewlineCount = nlcnt) == 2)
         {
            /* finally found those two consecutive newlines! */
            rqptr->rqResponse.HeaderSent = true;
            if (DataLength = cptr - DataPtr)
            {
               /* adjust data length to include only the HTTP header */
               DataLength = cptr - DataPtr;
            }
            else
            {
               /* no data in HTTP header left at all */
               rqptr->rqNet.WriteIOsb.Status = STS$K_SUCCESS;
               rqptr->rqNet.WriteIOsb.Count = 0;
               if (AstFunction) SysDclAst (AstFunction, rqptr);
               return (SS$_NORMAL);
            }
         }
      }
      else
      {
         /* HTTP header has been completely sent, absorb anything else */
         rqptr->rqNet.WriteIOsb.Status = STS$K_SUCCESS;
         rqptr->rqNet.WriteIOsb.Count = 0;
         if (AstFunction) SysDclAst (AstFunction, rqptr);
         return (SS$_NORMAL);
      }

      if (WATCHING(rqptr) && WATCH_MODULE(WATCH_MOD_NET))
         WatchThis (rqptr, FI_LI, WATCH_MOD_NET,
                    "HEAD !&X !UL", DataPtr, DataLength);
   }

   if (rqptr->rqHeader.HttpVersion == HTTP_VERSION_0_9 &&
       !rqptr->rqResponse.HeaderPtr &&
       rqptr->rqResponse.HeaderNewlineCount <= 1 &&
       !rqptr->ProxyTaskPtr)
   {
      /*********************************/
      /* absorb any header from script */
      /*********************************/

      int  cnt, nlcnt;
      char  *cptr;

      nlcnt = rqptr->rqResponse.HeaderNewlineCount;
      cptr = DataPtr;
      cnt = DataLength;
      while (cnt--)
      {
         if (*cptr == '\n' && ++nlcnt == 2)
         {
            /* two successive end-of-lines, therefore end of header */
            cptr++;
            break;
         }
         else
         if (*cptr != '\r' && *cptr != '\n')
            nlcnt = 0;
         cptr++;
      }

      if ((rqptr->rqResponse.HeaderNewlineCount = nlcnt) == 2)
      {
         /* adjust data pointer and length to exclude header */
         rqptr->rqResponse.HeaderSent = true;
         DataLength = cptr - DataPtr;
         DataPtr = cptr;
      }
      else
      {
         /* no data in HTTP header left at all */
         rqptr->rqNet.WriteIOsb.Status = STS$K_SUCCESS;
         rqptr->rqNet.WriteIOsb.Count = 0;
         if (AstFunction) SysDclAst (AstFunction, rqptr);
         return (SS$_NORMAL);
      }

      /* HTTP header has been completely absorbed, send everything else */
      if (WATCHING(rqptr) && WATCH_MODULE(WATCH_MOD_NET))
         WatchThis (rqptr, FI_LI, WATCH_MOD_NET,
                    "HTTP/0.9 !&X !UL", DataPtr, DataLength);
   }

   /*************/
   /* send data */
   /*************/

   if (WATCHING(rqptr) && WATCH_CATEGORY(WATCH_RESPONSE_BODY))
   {
      if (!rqptr->rqResponse.HeaderPtr)
         WatchThis (rqptr, FI_LI, WATCH_RESPONSE_BODY,
                    "STREAM !UL bytes", DataLength);
      else
         WatchThis (rqptr, FI_LI, WATCH_RESPONSE_BODY,
                    "BODY !UL bytes", DataLength);
      WatchDataDump (DataPtr, DataLength);
   }

   if (rqptr->rqCache.LoadFromNet) CacheLoadData (rqptr, DataPtr, DataLength);

   if (rqptr->rqResponse.CharsetNcsCf)
   {
      status = ResponseCharsetConvert (rqptr, &DataPtr, &DataLength);
      if (VMSnok (status))
      {
         /* fudge the status */
         rqptr->rqNet.WriteIOsb.Status = status;
         rqptr->rqNet.WriteIOsb.Count = 0;
         if (!AstFunction) return (status);
         SysDclAst (AstFunction, rqptr);
         return (status);
      }
   }

   if (rqptr->rqNet.SesolaPtr)
      status = SesolaNetWrite (rqptr->rqNet.SesolaPtr, AstFunction,
                               DataPtr, DataLength);
   else
      status = NetWriteRaw (rqptr, AstFunction, DataPtr, DataLength);

   return (status);
}

/*****************************************************************************/
/*
Write data to the network. Explicitly declares any AST routine if an error
occurs. The calling function must not do any error recovery if an AST routine
has been supplied but the associated AST routine must! If an AST was not
supplied then the return status can be checked.  AST to NetWriteRaw() which
calls the supplied AST function. 
*/

int NetWriteRaw
(
REQUEST_STRUCT *rqptr,
REQUEST_AST AstFunction,
char *DataPtr,
int DataLength
)

{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (WATCHING(rqptr) && WATCH_MODULE(WATCH_MOD_NET))
      WatchThis (rqptr, FI_LI, WATCH_MOD_NET,
                 "NetWriteRaw() !&A !&X !UL",
                 AstFunction, DataPtr, DataLength);

   if (rqptr->rqNet.WriteRawAstFunction || !DataLength)
   {
      rqptr->rqNet.WriteIOsb.Status = SS$_BUGCHECK;
      rqptr->rqNet.WriteIOsb.Count = 0;
      if (AstFunction) SysDclAst (AstFunction, rqptr);
      return (rqptr->rqNet.WriteIOsb.Status);
   }

   rqptr->rqNet.WriteRawAstFunction = AstFunction;
   rqptr->rqNet.WriteRawDataPtr = DataPtr;
   rqptr->rqNet.WriteRawDataLength = DataLength;

   if (WATCHING(rqptr) && WATCH_CATEGORY(WATCH_NETWORK_OCTETS))
   {
      WatchThis (rqptr, FI_LI, WATCH_NETWORK, "WRITE !UL bytes", DataLength);
      WatchDataDump (DataPtr, DataLength);
   }

   if (AstFunction)
   {
      /*******************/
      /* non-blocking IO */
      /*******************/

      status = sys$qio (EfnNoWait, rqptr->rqClient.Channel,
                        IO$_WRITEVBLK, &rqptr->rqNet.WriteIOsb,
                        &NetWriteRawAst, rqptr,
                        DataPtr, DataLength, 0, 0, 0, 0);
   }
   else
   {
      /***************/
      /* blocking IO */
      /***************/

      status = sys$qiow (EfnWait, rqptr->rqClient.Channel,
                         IO$_WRITEVBLK, &rqptr->rqNet.WriteIOsb, 0, 0,
                         DataPtr, DataLength, 0, 0, 0, 0);
      if (VMSok (status)) status = rqptr->rqNet.WriteIOsb.Status;
   }

   /****************/
   /* check status */
   /****************/

   if (VMSok (status)) return (status);

   /* if resource wait enabled the only quota not waited for is ASTLM */
   if (status == SS$_EXQUOTA)
      ErrorExitVmsStatus (status, "sys$qio()", FI_LI);

   /* write failed, call AST explicitly, status in the IOsb */
   rqptr->rqNet.WriteIOsb.Status = status;
   rqptr->rqNet.WriteIOsb.Count = 0;
   if (NetWriteRawAst) SysDclAst (NetWriteRawAst, rqptr);
   return (status);
}

/*****************************************************************************/
/*
AST from NetWriteRaw().  Call the AST function.
*/ 

NetWriteRawAst (REQUEST_STRUCT *rqptr)

{
   int  status;
   REQUEST_AST  AstFunction;

   /*********/
   /* begin */
   /*********/

   if (WATCHING(rqptr) && WATCH_MODULE(WATCH_MOD_NET))
      WatchThis (rqptr, FI_LI, WATCH_MOD_NET,
                 "NetWriteRawAst() !&F !&S !UL", &NetWriteRawAst,
                 rqptr->rqNet.WriteIOsb.Status, rqptr->rqNet.WriteIOsb.Count);

   if (WATCHING(rqptr))
   {
      if (WATCH_CATEGORY(WATCH_NETWORK))
      {
         if (VMSok (rqptr->rqNet.WriteIOsb.Status))
            WatchThis (rqptr, FI_LI, WATCH_NETWORK, "WRITE !&S !UL bytes",
                       rqptr->rqNet.WriteIOsb.Status,
                       rqptr->rqNet.WriteIOsb.Count);
         else
            WatchThis (rqptr, FI_LI, WATCH_NETWORK, "WRITE !&S %!-!&M",
                       rqptr->rqNet.WriteIOsb.Status);
      }
      if WATCH_CATEGORY(WATCH_RESPONSE)
         if (VMSnok (rqptr->rqNet.WriteIOsb.Status))
            WatchThis (rqptr, FI_LI, WATCH_RESPONSE, "NETWORK !&S %!-!&M",
                       rqptr->rqNet.WriteIOsb.Status);
   }

   if (VMSok (rqptr->rqNet.WriteIOsb.Status))
      rqptr->BytesRawTx += rqptr->rqNet.WriteIOsb.Count;
   else
   {
      rqptr->rqNet.WriteErrorCount++;
      /* just note the first error status */
      if (!rqptr->rqNet.WriteErrorStatus)
         rqptr->rqNet.WriteErrorStatus = rqptr->rqNet.WriteIOsb.Status;
      if (rqptr->rqNet.WriteIOsb.Status != SS$_LINKDISCON &&
          rqptr->rqNet.WriteIOsb.Status != SS$_CONNECFAIL &&
          rqptr->rqNet.WriteIOsb.Status != SS$_ABORT &&
          rqptr->rqNet.WriteIOsb.Status != SS$_CANCEL &&
          rqptr->rqNet.WriteIOsb.Status != SS$_IVCHAN)
         ErrorNoticed (rqptr->rqNet.WriteIOsb.Status, "WriteIOsb", FI_LI);
   }
      
   rqptr->rqNet.WriteRawDataPtr = rqptr->rqNet.WriteRawDataLength = 0;
   if (!(AstFunction = rqptr->rqNet.WriteRawAstFunction)) return;
   rqptr->rqNet.WriteRawAstFunction = NULL;
   (*AstFunction)(rqptr);
}

/*****************************************************************************/
/*
Wrapper for NetReadRaw().
*/ 

int NetRead
(
REQUEST_STRUCT *rqptr,
REQUEST_AST AstFunction,
char *DataPtr,
int DataSize
)
{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (WATCHING(rqptr) && WATCH_MODULE(WATCH_MOD_NET))
      WatchThis (rqptr, FI_LI, WATCH_MOD_NET, "NetRead() !&A !&X !UL",
                 AstFunction, DataPtr, DataSize);

   if (rqptr->rqNet.SesolaPtr)
      status = SesolaNetRead (rqptr->rqNet.SesolaPtr, AstFunction,
                              DataPtr, DataSize);
   else
      status = NetReadRaw (rqptr, AstFunction, DataPtr, DataSize);
   return (status);
}

/*****************************************************************************/
/*
Queue up a read from the client over the network. If 'AstFunction' 
is zero then no I/O completion AST routine is called.  If it is non-zero then 
the function pointed to by the parameter is called when the network write 
completes.

Explicitly declares any AST routine if an error occurs. The calling function
must not do any error recovery if an AST routine has been supplied but the
associated AST routine must!  If an AST was not supplied then the return
status can be checked.
*/ 

int NetReadRaw
(
REQUEST_STRUCT *rqptr,
REQUEST_AST AstFunction,
char *DataPtr,
int DataSize
)
{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (WATCHING(rqptr) && WATCH_MODULE(WATCH_MOD_NET))
      WatchThis (rqptr, FI_LI, WATCH_MOD_NET,
                 "NetReadRaw() !&A !&X !UL",
                 AstFunction, DataPtr, DataSize);

   if (rqptr->rqNet.ReadRawAstFunction)
   {
      rqptr->rqNet.ReadIOsb.Status = SS$_BUGCHECK;
      rqptr->rqNet.ReadIOsb.Count = 0;
      if (AstFunction) SysDclAst (AstFunction, rqptr);
      return (rqptr->rqNet.ReadIOsb.Status);
   }
   rqptr->rqNet.ReadRawAstFunction = AstFunction;
   rqptr->rqNet.ReadRawDataPtr = DataPtr;
   rqptr->rqNet.ReadRawDataSize = DataSize;

   if (WATCHING(rqptr) && WATCH_CATEGORY(WATCH_NETWORK_OCTETS))
      WatchThis (rqptr, FI_LI, WATCH_NETWORK, "READ !UL bytes max", DataSize);

   if (AstFunction)
   {
      /*******************/
      /* non-blocking IO */
      /*******************/

      status = sys$qio (EfnNoWait, rqptr->rqClient.Channel,
                        IO$_READVBLK, &rqptr->rqNet.ReadIOsb,
                        &NetReadRawAst, rqptr,
                        DataPtr, DataSize, 0, 0, 0, 0);
   }
   else
   {
      /***************/
      /* blocking IO */
      /***************/

      status = sys$qiow (EfnWait, rqptr->rqClient.Channel,
                         IO$_READVBLK, &rqptr->rqNet.ReadIOsb, 0, 0,
                         DataPtr, DataSize, 0, 0, 0, 0);
      if (VMSok (status)) status = rqptr->rqNet.ReadIOsb.Status;
   }

   /****************/
   /* check status */
   /****************/

   /* if I/O successful */
   if (VMSok (status)) return (status);

   /* with resource wait enabled the only quota not waited for is ASTLM */
   if (status == SS$_EXQUOTA)
      ErrorExitVmsStatus (status, "sys$qio()", FI_LI);

   /* queuing of read failed, call AST explicitly, status in the IOsb */
   rqptr->rqNet.ReadIOsb.Status = status;
   rqptr->rqNet.ReadIOsb.Count = 0;
   if (NetReadRawAst) SysDclAst (NetReadRawAst, rqptr);
   return (status);
}

/*****************************************************************************/
/*
AST from NetReadRaw().  Call the AST function.
*/ 

NetReadRawAst (REQUEST_STRUCT *rqptr)

{
   int  status;
   REQUEST_AST  AstFunction;

   /*********/
   /* begin */
   /*********/

   if (WATCHING(rqptr) && WATCH_MODULE(WATCH_MOD_NET))
      WatchThis (rqptr, FI_LI, WATCH_MOD_NET,
                 "NetReadRawAst() !&F !&S !UL", NetReadRawAst,
                 rqptr->rqNet.ReadIOsb.Status, rqptr->rqNet.ReadIOsb.Count);

   if (WATCHING(rqptr))
   {
      if (WATCH_CATEGORY(WATCH_NETWORK) ||
          WATCH_CATEGORY(WATCH_NETWORK_OCTETS))
      {
         if (VMSok(rqptr->rqNet.ReadIOsb.Status))
         {
            WatchThis (rqptr, FI_LI, WATCH_NETWORK, "READ !&S !UL bytes",
                       rqptr->rqNet.ReadIOsb.Status,
                       rqptr->rqNet.ReadIOsb.Count);
 
           if WATCH_CATEGORY(WATCH_NETWORK_OCTETS)
               WatchDataDump (rqptr->rqNet.ReadRawDataPtr,
                              rqptr->rqNet.ReadIOsb.Count);
         }
         else
            WatchThis (rqptr, FI_LI, WATCH_NETWORK, "READ !&S %!-!&M",
                       rqptr->rqNet.ReadIOsb.Status);
      }
   }

   if (VMSok (rqptr->rqNet.ReadIOsb.Status))
   {
      rqptr->BytesRawRx += rqptr->rqNet.ReadIOsb.Count;
      /* zero bytes with a normal status (once seen with TGV-Multinet) */
      if (!rqptr->rqNet.ReadIOsb.Count)
         rqptr->rqNet.ReadIOsb.Status = SS$_ABORT;
   }
   else
   /* CONNECT method connection cancellation is normal behaviour */
   if (rqptr->rqHeader.Method != HTTP_METHOD_CONNECT ||
       rqptr->rqNet.ReadIOsb.Status != SS$_CANCEL)
   {
      rqptr->rqNet.ReadErrorCount++;
      /* just note the first error status */
      if (!rqptr->rqNet.ReadErrorStatus)
         rqptr->rqNet.ReadErrorStatus = rqptr->rqNet.ReadIOsb.Status;
      if (rqptr->rqNet.ReadIOsb.Status != SS$_LINKDISCON &&
          rqptr->rqNet.ReadIOsb.Status != SS$_CONNECFAIL &&
          rqptr->rqNet.ReadIOsb.Status != SS$_ABORT &&
          rqptr->rqNet.ReadIOsb.Status != SS$_CANCEL &&
          rqptr->rqNet.ReadIOsb.Status != SS$_TIMEOUT &&
          rqptr->rqNet.ReadIOsb.Status != SS$_IVCHAN)
         ErrorNoticed (rqptr->rqNet.ReadIOsb.Status, "ReadIOsb", FI_LI);
   }

   rqptr->rqNet.ReadRawDataPtr = rqptr->rqNet.ReadRawDataSize = 0;
   if (!(AstFunction = rqptr->rqNet.ReadRawAstFunction)) return;
   rqptr->rqNet.ReadRawAstFunction = NULL;
   (*AstFunction)(rqptr);
}

/*****************************************************************************/
/*
This function buffers output without actually sending it to the client until
ready, either when the first buffer fills or when all output is completely
buffered. It will store any amount of output in a linked list of separately
allocated buffers. This is useful when a function that must not be interrupted
can rapidly store all required output without being slowed by actually
transfering it on the network, then flush it all asynchronously (e.g. the
server administration reports, for instance CacheReport(), which are
blocking).

If the data pointer is not NULL and the data length is -1 then the data is
assumed to be a null-terminated string.

Providing an AST parameter and a data parameter implies that after the first
buffer fills (and usually overflows into a second) the current buffered output
should be written to the client.

Providing an AST parameter and setting the data parameter to NULL and
the data length to 0 indicates all the currently buffered contents should be
written to the client.

Providing an AST parameter and setting the data parameter to NULL and the data
length parameter to -1 indicates all the currently buffered contents should be
written to the client if more than buffer-full has been written, otherwise just
declare the AST. 

Providing all parameters as NULL and zero, (except 'rqptr') as appropriate,
results in the data buffer initialized (if it didn't exist) or reset (if it
did).
*/ 

NetWriteBuffered
(
REQUEST_STRUCT *rqptr,
REQUEST_AST AstFunction,
char *DataPtr,
int DataLength
)
{
   int  status;
   char  *BufferPtr;

   /*********/
   /* begin */
   /*********/

   if (WATCHING(rqptr) && WATCH_MODULE(WATCH_MOD_NET))
   {
      WatchThis (rqptr, FI_LI, WATCH_MOD_NET,
                 "NetWriteBuffered() !&A !&X !UL",
                 AstFunction, DataPtr, DataLength);
      WatchDataDump (DataPtr, DataLength);
   }

   /* if all parameters essentially empty then reset the buffers */
   if (!AstFunction && !DataPtr && !DataLength)
   {
      NetWriteBufferedInit (rqptr, true);
      return;
   }

   /* first call, initialize a buffer */
   if (!rqptr->rqOutput.BufferStructPtr)
      NetWriteBufferedInit (rqptr, true);

   if (DataPtr)
   {
      /**********************/
      /* buffer this output */
      /**********************/

      if (DataLength == -1) DataLength = strlen(DataPtr);

      if (rqptr->rqOutput.BufferEscapeHtml)
      {
         /* escape HTML-forbidden characters */
         NetBufferEscapeHtml (rqptr, DataPtr, DataLength);
      }
      else
      {
         while (DataLength)
         {
            if (DataLength <= rqptr->rqOutput.BufferRemaining)
            {
               /* enough space in this buffer */
               memcpy (rqptr->rqOutput.BufferCurrentPtr, DataPtr, DataLength);
               rqptr->rqOutput.BufferCount += DataLength;
               rqptr->rqOutput.BufferCurrentPtr += DataLength;
               rqptr->rqOutput.BufferRemaining -= DataLength;
               DataLength = 0;
            }
            else
            {
               /* fill up any that's left */
               memcpy (rqptr->rqOutput.BufferCurrentPtr,
                       DataPtr,
                       rqptr->rqOutput.BufferRemaining);
               DataPtr += rqptr->rqOutput.BufferRemaining;
               DataLength -= rqptr->rqOutput.BufferRemaining;
               rqptr->rqOutput.BufferCount += rqptr->rqOutput.BufferRemaining;
               /* need another buffer */
               NetWriteBufferedInit (rqptr, false);
            }
         }
      }
   }

   /* if an AST routine supplied then we can think about buffer flushing */
   if (!AstFunction) return;

   /* if the first buffer is full or if the flush is being forced */
   if ((!DataPtr && DataLength != -1) ||
       (rqptr->rqOutput.BufferCount > rqptr->rqOutput.BufferSize))
   {
      /***********************/
      /* flush the buffer(s) */
      /***********************/

      /* force all buffers to be flushed if 'DataPtr' is NULL */
      if (!DataPtr) rqptr->rqOutput.BufferFlush = true;
      /* buffer the AST function pointer */
      rqptr->rqOutput.BufferAstFunction = AstFunction;
      /* get the "write now" pointer to the first buffer in the list */
      rqptr->rqOutput.BufferNowStructPtr =
         LIST_GET_HEAD (&rqptr->rqOutput.BufferList);
      /* fudge this status for NetWriteBufferedNow() */
      rqptr->rqNet.WriteIOsb.Status = SS$_NORMAL;
      NetWriteBufferedNow (rqptr);
      return;
   }

   /************************/
   /* just declare the AST */
   /************************/

   /* fudge this status for the AST routine check */
   rqptr->rqNet.WriteIOsb.Status = SS$_NORMAL;
   rqptr->rqNet.WriteIOsb.Count = DataLength;

   SysDclAst (AstFunction, rqptr);
}

/*****************************************************************************/
/*
Copy the data into the buffer escaping HTML-forbidden characters.  If
'DataLength' is -1 then assumed to be a null-terminated string.
*/ 

NetBufferEscapeHtml
(
REQUEST_STRUCT *rqptr,
char *DataPtr,
int DataLength
)
{
   int  ch, dcnt, bcnt, brcnt;
   char  *cptr, *bptr, *dptr;

   /*********/
   /* begin */
   /*********/

   if (WATCHING(rqptr) && WATCH_MODULE(WATCH_MOD_NET))
      WatchThis (rqptr, FI_LI, WATCH_MOD_NET,
                 "NetBufferEscapeHtml() !&X !UL", DataPtr, DataLength);

   /* first call, initialize a buffer */
   if (!rqptr->rqOutput.BufferStructPtr)
      NetWriteBufferedInit (rqptr, true);

   dptr = DataPtr;
   if ((dcnt = DataLength) == -1) dcnt = strlen(dptr);
   bptr = rqptr->rqOutput.BufferCurrentPtr;
   bcnt = rqptr->rqOutput.BufferCount;
   brcnt = rqptr->rqOutput.BufferRemaining;

   while (dcnt)
   {
      if (!brcnt)
      {
         /* need another buffer */
         NetWriteBufferedInit (rqptr, false);
         bptr = rqptr->rqOutput.BufferCurrentPtr;
         brcnt = rqptr->rqOutput.BufferRemaining;
      }

      dcnt--;
      switch (ch = *dptr++)
      {
         case '<' :
            cptr = "&lt;";
            break;
         case '>' :
            cptr = "&gt;";
            break;
         case '&' :
            cptr = "&amp;";
            break;
         case '\"' :
            cptr = "&quot;";
            break;
         default :
            /* insert this character as-is */
            *bptr++ = ch;
            bcnt++; 
            brcnt--;
            continue;
      }

      /* add the escaped character */
      while (*cptr)
      {
         *bptr++ = *cptr++;
         bcnt++; 
         if (--brcnt) continue;
         /* need another buffer */
         NetWriteBufferedInit (rqptr, false);
         bptr = rqptr->rqOutput.BufferCurrentPtr;
         brcnt = rqptr->rqOutput.BufferRemaining;
      }
   }

   rqptr->rqOutput.BufferCurrentPtr = bptr;
   rqptr->rqOutput.BufferRemaining = brcnt;
   rqptr->rqOutput.BufferCount = bcnt;
}

/*****************************************************************************/
/*
This function processes the linked list of output buffers, writing them to the
client, and operates in two distinct modes. First, non-flush mode. In this
mode any full output buffers are written to the client. Any last,
partially-filled buffer is not, instead it is moved to the head of the list to
essentially become the first data in the output buffer to which more can then
be added. In this way when a buffer fills and overflows into a second buffer
the first is written but the second, scantily filled buffer is not. The second
mode is full-flush, where all buffers are written to the client. This is
forced by NetWriteBuffered() whenever an AST routine is supplied but no data
('DataPtr' is NULL).
*/ 

NetWriteBufferedNow (REQUEST_STRUCT *rqptr)

{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (WATCHING(rqptr) && WATCH_MODULE(WATCH_MOD_NET))
      WatchThis (rqptr, FI_LI, WATCH_MOD_NET,
                 "NetWriteBufferedNow() !&F !&S !&A !UL !UL !UL",
                 &NetWriteBufferedNow,
                 rqptr->rqNet.WriteIOsb.Status,
                 rqptr->rqOutput.BufferAstFunction,
                 rqptr->rqOutput.BufferRemaining,
                 rqptr->rqOutput.BufferCount,
                 rqptr->rqOutput.BufferFlush);

   if (VMSnok (rqptr->rqNet.WriteIOsb.Status))
   {
      /* network write has failed (as AST), bail out now */
      SysDclAst (rqptr->rqOutput.BufferAstFunction, rqptr);
      return;
   }

   if ((rqptr->rqOutput.BufferFlush &&
        rqptr->rqOutput.BufferCount > rqptr->rqOutput.BufferSize) ||
       rqptr->rqOutput.BufferCount > rqptr->rqOutput.BufferSize +
                                     rqptr->rqOutput.BufferSize)
   {
      /*
         If flushing all buffers and at least one full buffer in list, or
         if two or more full buffers in list.
      */
      NetWrite (rqptr, &NetWriteBufferedNow,
                LIST_GET_DATA (rqptr->rqOutput.BufferNowStructPtr),
                rqptr->rqOutput.BufferSize);
      rqptr->rqOutput.BufferCount -= rqptr->rqOutput.BufferSize;
      /* set the "write now" buffer pointer to the next buffer */
      rqptr->rqOutput.BufferNowStructPtr =
         LIST_GET_NEXT (rqptr->rqOutput.BufferNowStructPtr);
      return;
   }
   else
   if (rqptr->rqOutput.BufferCount > rqptr->rqOutput.BufferSize)
   {
      /* if one full buffer, then one partially filled, NOTE the AST! */
      NetWrite (rqptr, rqptr->rqOutput.BufferAstFunction,
                LIST_GET_DATA (rqptr->rqOutput.BufferNowStructPtr),
                rqptr->rqOutput.BufferSize);
      rqptr->rqOutput.BufferCount -= rqptr->rqOutput.BufferSize;
      /*
         Move currently-being-written-to buffer to the head of the list,
         effectively to the front of the buffer.
      */
      ListMoveHead (&rqptr->rqOutput.BufferList,
                    rqptr->rqOutput.BufferStructPtr);
      return;
   }
   else
   {
      /* if last buffer in linked list, write whatever remains in it */
      NetWrite (rqptr, rqptr->rqOutput.BufferAstFunction,
                LIST_GET_DATA (rqptr->rqOutput.BufferNowStructPtr),
                rqptr->rqOutput.BufferCount);
      /* reset the buffer(s) */
      NetWriteBufferedInit (rqptr, true);
      /* no more buffers therefore flush (if set) is complete */
      rqptr->rqOutput.BufferFlush = false;
      return;
   }
}

/*****************************************************************************/
/*
Get another output buffer.
*/ 

NetWriteBufferedInit
(
REQUEST_STRUCT *rqptr,
BOOL ResetBuffers
)
{
   /*********/
   /* begin */
   /*********/

   if (WATCHING(rqptr) && WATCH_MODULE(WATCH_MOD_NET))
      WatchThis (rqptr, FI_LI, WATCH_MOD_NET, "NetWriteBufferedInit() !UL !UL",
                 ResetBuffers, !LIST_IS_EMPTY(&rqptr->rqOutput.BufferList));

   if (ResetBuffers && !LIST_IS_EMPTY(&rqptr->rqOutput.BufferList))
   {
      /* reset a non-empty list to the first buffer */
      rqptr->rqOutput.BufferStructPtr =
         LIST_GET_HEAD (&rqptr->rqOutput.BufferList);
      rqptr->rqOutput.BufferPtr =
         LIST_GET_DATA (rqptr->rqOutput.BufferStructPtr);
      rqptr->rqOutput.BufferCurrentPtr = rqptr->rqOutput.BufferPtr;
      rqptr->rqOutput.BufferRemaining = rqptr->rqOutput.BufferSize;
      rqptr->rqOutput.BufferCount = 0;
      return;
   }

   if (rqptr->rqOutput.BufferStructPtr &&
       LIST_HAS_NEXT (rqptr->rqOutput.BufferStructPtr))
   {
      /* reuse a previously allocated buffer from the list */
      rqptr->rqOutput.BufferStructPtr =
         LIST_GET_NEXT (rqptr->rqOutput.BufferStructPtr);
      rqptr->rqOutput.BufferPtr =
         LIST_GET_DATA (rqptr->rqOutput.BufferStructPtr);
      rqptr->rqOutput.BufferCurrentPtr = rqptr->rqOutput.BufferPtr;
      rqptr->rqOutput.BufferRemaining = rqptr->rqOutput.BufferSize;
      return;
   }

   /* allocate the first or another buffer to the list */
   rqptr->rqOutput.BufferStructPtr =
      VmGetHeap (rqptr, sizeof(LIST_ENTRY) + OutputBufferSize + 3);
   ListAddTail (&rqptr->rqOutput.BufferList, rqptr->rqOutput.BufferStructPtr);
   rqptr->rqOutput.BufferPtr = LIST_GET_DATA (rqptr->rqOutput.BufferStructPtr);
   rqptr->rqOutput.BufferCurrentPtr = rqptr->rqOutput.BufferPtr;
   rqptr->rqOutput.BufferRemaining =
      rqptr->rqOutput.BufferSize = OutputBufferSize;
}

/*****************************************************************************/
/*
Get create an output structure with a specified size buffer.
*/ 

NetWriteBufferedSizeInit
(
REQUEST_STRUCT *rqptr,
int BufferSize
)
{
   /*********/
   /* begin */
   /*********/

   if (WATCHING(rqptr) && WATCH_MODULE(WATCH_MOD_NET))
      WatchThis (rqptr, FI_LI, WATCH_MOD_NET,
                 "NetWriteBufferedSizeInit() !UL", BufferSize);

   rqptr->rqOutput.BufferStructPtr =
      VmGetHeap (rqptr, sizeof(LIST_ENTRY) + BufferSize + 3);
   ListAddTail (&rqptr->rqOutput.BufferList, rqptr->rqOutput.BufferStructPtr);
   rqptr->rqOutput.BufferPtr = LIST_GET_DATA (rqptr->rqOutput.BufferStructPtr);
   rqptr->rqOutput.BufferCurrentPtr = rqptr->rqOutput.BufferPtr;
   rqptr->rqOutput.BufferRemaining = rqptr->rqOutput.BufferSize = BufferSize;
}

/*****************************************************************************/
/*
Respond direct to the client without beginning request processing.
Read something from the client, they seem to enjoy (need) that!
*/
 
NetDirectResponse
(
REQUEST_STRUCT *rqptr,
int Message
)
{
   static unsigned long  FiveSecondsDelta [2] = { -50000000, -1 };
   static $DESCRIPTOR (BufferDsc, "");

   static $DESCRIPTOR (TooBusyFaoDsc,
"HTTP/1.0 502 Too busy!!\r\n\
Content-Type: text/html!AZ!AZ\r\n\
\r\n\
<HTML>\n\
<HEAD>\n\
<TITLE>!AZ 502</TITLE>\n\
</HEAD>\n\
!AZ\n\
!AZ\n\
</BODY>\n\
</HTML>\n");

   static $DESCRIPTOR (ForbiddenFaoDsc,
"HTTP/1.0 403 Forbidden\r\n\
Content-Type: text/html!AZ!AZ\r\n\
\r\n\
<HTML>\n\
<HEAD>\n\
<TITLE>!AZ 403</TITLE>\n\
</HEAD>\n\
!AZ\n\
!AZ\n\
</BODY>\n\
</HTML>\n");

   int  status;
   short  Length;
   IO_SB  IOsb;

   /*********/
   /* begin */
   /*********/

   if (WATCH_MODULE(WATCH_MOD_NET))
      WatchThis (NULL, FI_LI, WATCH_MOD_NET,
                 "NetDirectResponse() !UL", Message);

   status = sys$setimr (0, &FiveSecondsDelta,
                        &NetDirectResponseTimeoutAst, rqptr, 0);
   if (VMSnok (status))
   {
      ErrorNoticed (status, "sys$setimr()", FI_LI);
      sys$dassgn (rqptr->rqClient.Channel);
      VmFreeRequest (rqptr, FI_LI);
      return;
   }

   rqptr->rqNet.ReadBufferPtr = VmGetHeap (rqptr, NetReadBufferSize);

   BufferDsc.dsc$a_pointer = rqptr->rqNet.ReadBufferPtr;
   BufferDsc.dsc$w_length = NetReadBufferSize;

   if (Message == MSG_GENERAL_ACCESS_DENIED)
      sys$fao (&ForbiddenFaoDsc, &Length, &BufferDsc,
               Config.cfContent.CharsetDefault[0] ? "; charset=" : "",
               Config.cfContent.CharsetDefault,
               MsgFor (rqptr, MSG_STATUS_ERROR),
               Config.cfServer.ReportBodyTag,
               MsgFor (rqptr, MSG_GENERAL_ACCESS_DENIED));
   else
   if (Message == MSG_GENERAL_TOO_BUSY)
      sys$fao (&TooBusyFaoDsc, &Length, &BufferDsc,
               Config.cfContent.CharsetDefault[0] ? "; charset=" : "",
               Config.cfContent.CharsetDefault,
               MsgFor (rqptr, MSG_STATUS_ERROR),
               Config.cfServer.ReportBodyTag,
               MsgFor (rqptr, MSG_GENERAL_TOO_BUSY));
   else
      ErrorExitVmsStatus (SS$_BUGCHECK, ErrorSanityCheck, FI_LI);

   status = sys$qiow (EfnWait, rqptr->rqClient.Channel,
                      IO$_WRITEVBLK | IO$M_NOWAIT,
                      &IOsb, 0, 0, rqptr->rqNet.ReadBufferPtr, Length,
                      0, 0, 0, 0);
   if (VMSok (status)) status = IOsb.Status;
   if (VMSok (status))
   {
      status = sys$qio (EfnNoWait, rqptr->rqClient.Channel,
                        IO$_READVBLK, &rqptr->rqNet.ReadIOsb,
                        &NetDirectResponseAst, rqptr,
                        rqptr->rqNet.ReadBufferPtr, NetReadBufferSize,
                        0, 0, 0, 0);
      if (VMSnok (status)) ErrorNoticed (status, "sys$qio()", FI_LI);
   }
   else
      ErrorNoticed (status, "sys$qiow()", FI_LI);

   if (VMSnok (status))
   {
      sys$dassgn (rqptr->rqClient.Channel);
      VmFreeHeap (rqptr, FI_LI);
      VmFreeRequest (rqptr, FI_LI);
   }
}

/*****************************************************************************/
/*
See commentary in NetDirectResponse().
*/ 

NetDirectResponseAst (REQUEST_STRUCT *rqptr)

{
   /*********/
   /* begin */
   /*********/

   if (WATCH_MODULE(WATCH_MOD_NET))
      WatchThis (NULL, FI_LI, WATCH_MOD_NET, "NetDirectResponseAst()");

   sys$cantim (rqptr, 0);
   sys$dassgn (rqptr->rqClient.Channel);
   VmFreeHeap (rqptr, FI_LI);
   VmFreeRequest (rqptr, FI_LI);
}

/*****************************************************************************/
/*
See commentary in NetDirectResponse().
*/ 

NetDirectResponseTimeoutAst (REQUEST_STRUCT *rqptr)

{
   /*********/
   /* begin */
   /*********/

   if (WATCH_MODULE(WATCH_MOD_NET))
      WatchThis (NULL, FI_LI, WATCH_MOD_NET, "NetDirectResponseTimeoutAst()");

   sys$cancel (rqptr->rqClient.Channel);
}

/*****************************************************************************/
/*
Needed a rig to test/experiment with multiple-instance network device (socket)
handling.
*/ 

#if NET_TEST

NetTestRequest (unsigned short Channel)

{
   int  status;
   struct NetTestStruct  *ntptr;

   /*********/
   /* begin */
   /*********/

   if (WATCH_MODULE(WATCH_MOD_NET))
      WatchThis (NULL, FI_LI, WATCH_MOD_NET, "NetTestRequest() !UL", Channel);

   ntptr = VmGet (sizeof(struct NetTestStruct));

   ntptr->Channel = Channel;

   status = sys$qio (EfnNoWait, ntptr->Channel, IO$_READVBLK, &ntptr->IOsb,
                     &NetTestReadAst, ntptr,
                     ntptr->Buffer, sizeof(ntptr->Buffer), 0, 0, 0, 0);
   if (VMSnok (status)) ErrorNoticed (status, "sys$qio()", FI_LI);
}

/*****************************************************************************/
/*
See comments in NetTestRequest().
*/ 

NetTestReadAst (struct NetTestStruct *ntptr)

{
   static char  ResponseHeader [] =
"HTTP/1.0 200 OK\r\n\
Server: WASD/Test\r\n\
Content-Type: text/plain\r\n\
\r\n";
   static char  ResponseBody [] =
"abcdefghijklnmopqrstuvwxyzABCDEFGHIJKLNMOPQRSTUVWXYZ0123456789\n";

   int  status;
   char  *cptr, *sptr, *zptr;

   /*********/
   /* begin */
   /*********/

   if (WATCH_MODULE(WATCH_MOD_NET))
      WatchThis (NULL, FI_LI, WATCH_MOD_NET,
                 "NetTestReadAst() !UL !&S !UL",
                 ntptr->Channel, ntptr->IOsb.Status, ntptr->IOsb.Count);

   if (VMSnok (ntptr->IOsb.Status))
   {
      ErrorNoticed (ntptr->IOsb.Status, "NetTestReadAst()", FI_LI);
      status = sys$dassgn (ntptr->Channel);
      if (VMSnok (status)) ErrorNoticed (status, "sys$dassgn()", FI_LI);
      VmFree (ntptr, FI_LI);
      return;
   }

   memcpy (ntptr->Buffer, ResponseHeader, sizeof(ResponseHeader));
   sptr = ntptr->Buffer + sizeof(ResponseHeader);
   zptr = ntptr->Buffer + sizeof(ntptr->Buffer);
   cptr = ResponseBody;
   while (sptr < zptr)
   {
      if (!*cptr) cptr = ResponseBody;
      *sptr++ = *cptr++;
   }

   status = sys$qio (EfnNoWait, ntptr->Channel, IO$_WRITEVBLK, &ntptr->IOsb,
                     &NetTestWriteAst, ntptr,
                     ntptr->Buffer, sizeof(ntptr->Buffer), 0, 0, 0, 0);
   if (VMSnok (status)) ErrorNoticed (status, "sys$qio()", FI_LI);
}

/*****************************************************************************/
/*
See comments in NetTestRequest().
*/ 

NetTestWriteAst (struct NetTestStruct *ntptr)

{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (WATCH_MODULE(WATCH_MOD_NET))
      WatchThis (NULL, FI_LI, WATCH_MOD_NET,
                 "NetTestWriteAst() !UL !&S",
                 ntptr->Channel, ntptr->IOsb.Status);

   status = sys$dassgn (ntptr->Channel);
   if (VMSnok (status)) ErrorNoticed (status, "sys$dassgn()", FI_LI);
   VmFree (ntptr, FI_LI);
}

#endif /* NET_TEST */

/****************************************************************************/

                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  