/*****************************************************************************/
/*
                                   TCPIP.h
*/
/*****************************************************************************/

#ifndef TCPIP_H_LOADED
#define TCPIP_H_LOADED 1

/********************/
/* TCP/IP constants */
/********************/

#define INET_PROTYP$C_STREAM 1
#define INETACP$C_TRANS 2
#define INETACP_FUNC$C_GETHOSTBYNAME 1
#define INETACP_FUNC$C_GETHOSTBYADDR 2
#define TCPIP$C_INADDR_ANY 0
#define TCPIP$C_AF_INET 2
#define TCPIP$C_AF_INET6 26
#define TCPIP$C_DSC_ALL 2
#define TCPIP$C_FULL_DUPLEX_CLOSE 8192
#define TCPIP$C_REUSEADDR 4
#define TCPIP$C_SHARE 4105
#define TCPIP$C_SOCK_NAME 4
#define TCPIP$C_SOCKOPT 1
#define TCPIP$C_TCP 6

/*********************/
/* TCP/IP structures */
/*********************/

#pragma __member_alignment __save
#pragma __nomember_alignment

#define SIN$C_LENGTH 16
#define SIN$K_LENGTH 16

typedef struct _SOCKADDRIN
{
   unsigned short int SIN$W_FAMILY;    /* address family */
   unsigned short int SIN$W_PORT;      /* 2 bytes specifying a port */
   unsigned int SIN$L_ADDR;            /* 4 bytes specifying an IP address */
   char SIN$T_ZERO [8];                /* 8 bytes */
}
SOCKADDRIN;

#define SIN6$K_LENGTH 28
#define SIN6$C_LENGTH 28

typedef struct _SOCKADDRIN6
{
   unsigned char SIN6$B_LEN;           /* length of this struct */
   unsigned char SIN6$B_FAMILY;        /* AF_INET6 */
   unsigned short int SIN6$W_PORT;     /* Transport layer port number */
   unsigned int SIN6$L_FLOWLABEL;      /* IPv6 flow information */
   union
   {
      char SIN6$T_ADDR [16];
      unsigned char SIN6$B_SA6_ADDR [16];
      unsigned short int SIN6$W_SA6_WADDR [8];
      unsigned int SIN6$L_SA6_LADDR [4];
      unsigned int SIN6$Q_SA6_QADDR [2] [2];
   }
   SIN6$R_ADDR_OVERLAY;
   unsigned int SIN6$L_SCOPE_ID;       /* set of interfaces for a scope */
}
SOCKADDRIN6;
 
#pragma __member_alignment __restore

/*****************/
/* module macros */
/*****************/

#define TCPIP_HOSTNAME_MAX 127

/* must be able to accomodate both IPv4 and Ipv6 address strings */
#define TCPIP_ADDRESS_STRING_MAX 39

/* default number of seconds each cache entry remains current */
#define TCPIP_HOST_CACHE_EXPIRE_SECONDS 600

/* default number of 2 second retries */
#define TCPIP_LOOKUP_RETRY_COUNT 5

/* number of entries by which host/address cache is expanded */
#define TCPIP_HOST_CACHE_CHUNK 128

/* absolute maximum number of entries for host/address cache */
#define TCPIP_HOST_CACHE_MAX 1024

/* set this IPADDRESS to have an IPv4 size */
#define IPADDRESS_V4(addr) (addr)->size=4

/* set this IPADDRESS to have an IPv6 size */
#define IPADDRESS_V6(addr) (addr)->size=16

/* to have an IPv4 size and zeroed address */
#define IPADDRESS_ZERO4(addr) \
{(addr)->size=4;memset((addr)->address,0,16);}

/* to have an IPv6 size and zeroed address */
#define IPADDRESS_ZERO6(addr) \
{(addr)->size=16;memset((addr)->address,0,16);}

/* set the address to contain 0x00 (IPADDR_ANY) */
#define IPADDRESS_SET_ANY(addr) \
memset((addr)->address,0x00,16);

/* set the address to contain 0xff (unusable) */
#define IPADDRESS_SET_UNUSABLE(addr) \
{((addr)->size==4?\
memset(&(addr)->address[0],0xff,4):\
memset(&(addr)->address[0],0xff,16));}

/* true if this IPADDRESS has an IPv4 size */
#define IPADDRESS_IS_V4(addr) (addr)->size==4

/* true if this IPADDRESS has an IPv6 size */
#define IPADDRESS_IS_V6(addr) (addr)->size==16

/* true if this address is INADDR_ANY (all zero) */
#define IPADDRESS_IS_ANY(addr) \
(!memcmp((addr)->address,TcpIpEmptyAddress.address,(addr)->size))

/* true if this address is all zero */
#define IPADDRESS_IS_RESET(addr) \
(!memcmp((addr)->address,TcpIpEmptyAddress.address,(addr)->size))

/* true if this address is not all zero */
#define IPADDRESS_IS_SET(addr) \
memcmp((addr)->address,TcpIpEmptyAddress.address,(addr)->size)

/* true if the address structures are exactly the same */
#define IPADDRESS_IS_SAME(addr1,addr2) \
(!memcmp((char*)addr1,(char*)addr2,sizeof(IPADDRESS)))

/* copy address 2 onto address 1 */
#define IPADDRESS_COPY(addr1,addr2) \
memcpy((char*)addr1,(char*)addr2,sizeof(*addr1));

/* get IPv4 integer into address setting IPv4 size */
#define IPADDRESS_GET4(addr,ip4) \
{(addr)->size=4;(addr)->address[0]=ip4;}

/* get IPv6 byte-array into address setting IPv6 size */
#define IPADDRESS_GET6(addr,ip6) \
{(addr)->size=16;memcpy((char*)(addr)->address,(char*)ip6,16);}

/* access an IPv4 integer address */
#define IPADDRESS_ADR4(addr) \
(addr)->address[0]

/* access an IPv6 address, as a pointer */
#define IPADDRESS_ADR6(addr) \
(addr)->address

/* access an IPv4 or IPv6 address, as a pointer */
#define IPADDRESS_ADR46(addr) \
(addr)->address

/* set an IPv4 integer address */
#define IPADDRESS_SET4(ip4,addr) \
ip4=(addr)->address[0];

/* set an IPv6 byte-array address */
#define IPADDRESS_SET6(ip6,addr) \
memcpy((char*)ip6,(char*)(addr)->address,16);

/* the int size of the IPADDRESS */
#define IPADDRESS_SIZE(addr) \
(addr)->size

/* set this SOCKADDRESS to have an IPv4 size */
#define SOCKADDRESS_V4(addr) (addr)->size=16

/* set this SOCKADDRESS to have an IPv6 size */
#define SOCKADDRESS_V6(addr) (addr)->size=28

/* to have an IPv4 size and zeroed structure */
#define SOCKADDRESS_ZERO4(addr) \
{(addr)->size=16;memset(&(addr)->sa.v4,0,16);}

/* to have an IPv6 size and zeroed address */
#define SOCKADDRESS_ZERO6(addr) \
{(addr)->size=28;memset(&(addr)->sa.v6,0,28);}

/* true if this SOCKADDRESS has an IPv4 size */
#define SOCKADDRESS_IS_V4(addr) (addr)->size==16

/* true if this SOCKADDRESS has an IPv6 size */
#define SOCKADDRESS_IS_V6(addr) (addr)->size==28

/* access an IPv4 integer address */
#define SOCKADDRESS_ADR4(addr) \
(addr)->sa.v4.SIN$L_ADDR

/* access an IPv6 byte-array address */
#define SOCKADDRESS_ADR6(addr) \
&(addr)->sa.v6.SIN6$R_ADDR_OVERLAY.SIN6$T_ADDR

/*********************/
/* module structures */
/*********************/

#pragma __member_alignment __save
#pragma __member_alignment

/*
   This structure will contain both IPv4 and IPv6 addresses.
   The 'size' element indicates the former with 4 and the latter with 16.
   The IPv4 address is stored in the
*/
typedef struct _IPADDRESS
{
   /* if the size is 4 it's IPv4, if the size is 16 it's IPv6 */
   int  size; 
   unsigned int address [4];
}
IPADDRESS;
 
typedef struct _SOCKADDRESS
{
   /* if the size is 16 it's IPv4, if the size is 28 it's IPv6 */
   int  size;
   union
   {
      struct _SOCKADDRIN  v4;
      struct _SOCKADDRIN6  v6;
   }
   sa;
}
SOCKADDRESS;
 
typedef struct _TCPIP_HOST_LOOKUP
{
   int  HostNameLength,
        RetryCount;
   unsigned short  HostLength,
                   LookupChannel;
   char  HostName [TCPIP_HOSTNAME_MAX+1];
   struct dsc$descriptor HostAddressDsc;
   struct dsc$descriptor HostNameDsc;
   IO_SB  LookupIOsb;
   void  *AstFunction;
   unsigned long  AstParam;
   IPADDRESS  IpAddress;
}
TCPIP_HOST_LOOKUP;

typedef struct _TCPIP_HOST_CACHE
{
   int  ExpiresTickSecond,
        HitCountAddressToName,
        HitCountNameToAddress,
        HostNameLength;
   char  HostName [TCPIP_HOSTNAME_MAX+1];
   IPADDRESS  IpAddress;
}
TCPIP_HOST_CACHE;

#pragma __nomember_alignment

typedef struct _TCP_SOCKET_ITEM
{
   unsigned short  Protocol;
   unsigned char  Type;
   unsigned char  Family;
}
TCP_SOCKET_ITEM;

#pragma __member_alignment __restore

/***********************/
/* function prototypes */
/***********************/

char* TcpIpAddressToString (int, int);
int TcpIpAddressToName (TCPIP_HOST_LOOKUP*, IPADDRESS*, int,
                        void*, unsigned long);
int TcpIpCacheAddressToName (TCPIP_HOST_LOOKUP*, IPADDRESS*);
int TcpIpCacheNameToAddress (TCPIP_HOST_LOOKUP*, char*, int);
int TcpIpCacheSetEntry (char*, int, IPADDRESS*);
TcpIpHostCacheInit ();
TcpIpHostCacheReport ();
int TcpIpNameToAddress (TCPIP_HOST_LOOKUP*, char*, int,
                        void*, unsigned long);
int TcpIpNetMask (REQUEST_STRUCT*, int, char**, IPADDRESS*);
TcpIpSetAgentInfo ();
int TcpIpStringToAddress (char*, IPADDRESS*);

#endif /* TCPIP_H_LOADED */

/*****************************************************************************/

