/*****************************************************************************/
/*
                                 CallMail.c

This module contain pretty much all of the yahMAIL functions used to interface
with VMS callable mail.


COPYRIGHT
---------
Copyright (C) 1999-2003 Mark G.Daniel
This program, comes with ABSOLUTELY NO WARRANTY.
This is free software, and you are welcome to redistribute it
under the conditions of the GNU GENERAL PUBLIC LICENSE, version 2.


VERSION HISTORY
---------------
21-APR-2003  MGD  SPAM filtering
28-FEB-2003  MGD  fromiso_string() 'VmsMailRfc822From', 'VmsMailRfc822ReplyTo'
11-FEB-2003  MGD  get (any) RFC822 "From:" and/or "Reply-To:" header into
                  'VmsMailRfc822From' and 'VmsMailRfc822ReplyTo'
11-SEP-2002  MGD  bugfix; terminate on ';' when parsing MIME 'charset='
23-MAR-2002  MGD  MIME quoted-printable,
                  provide RFC1522 encoded-words in message header,
                  bugfix; StripTransport() with multiple addresses
16-FEB-2002  MGD  StripTransport() from addresses,
                  bugfixes; small bunch (thanks to Carl Karcher of WISC)
12-FEB-2002  MGD  (potential) bugfix; null-terminate 'VmsMailFrom'
10-FEB-2002  MGD  increase MAIL$MESSAGE_GET() item MAIL$_MESSAGE_RECORD from
                  a documented 255 characters to an emperical 2048 for later
                  versions of VMS (though exactly which is a bit of a mystery)
19-FEB-2000  MGD  unbundled from YAHMAIL.C for v1.3
*/

/*****************************************************************************/

#ifndef __VAX
#   pragma nomember_alignment
#endif

/* standard C header files */
#include <ctype.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* VMS related header files */
#include <descrip.h>
#include <maildef.h>
#include <mailmsgdef.h>
#include <ssdef.h>
#include <stsdef.h>

/* application header file */
#include "yahmail.h"
#include "externs.h"

#define FI_LI __FILE__, __LINE__

/*****************************************************************************/
/*
Read the header information and body of the specified message.
*/

int MailReadMessage ()

{
   static unsigned long  MessageContext,
                         MessageSelected,
                         MessageRecordLength;

   static char  MessageRecord [2048+2];

   static struct VmsItemLongStruct   MessageBeginItem [] = {
      { 4, MAIL$_MESSAGE_FILE_CTX, &MailFileContext, 0 },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   static struct VmsItemLongStruct   MessageContinueItem [] = {
      { 0, MAIL$_MESSAGE_CONTINUE, 0, 0 },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   static struct VmsItemLongStruct   MessageIdItem [] = {
      { 4, MAIL$_MESSAGE_ID, &MailMessageId[0], 0 },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   static struct VmsItemLongStruct   MessageInfoItem [] = {
      { 255, MAIL$_MESSAGE_EXTID, &VmsMailExtId, &VmsMailExtIdLength },
      { 4, MAIL$_MESSAGE_CURRENT_ID, &VmsMailMessageId, 0 },
      { 4, MAIL$_MESSAGE_SIZE, &VmsMailMessageSize, 0 },
      { 2, MAIL$_MESSAGE_RETURN_FLAGS, &VmsMailMessageFlags, 0 },
      { 255, MAIL$_MESSAGE_CC, VmsMailCc, &VmsMailCcLength },
      { 255, MAIL$_MESSAGE_DATE, VmsMailDate, &VmsMailDateLength },
      { 255, MAIL$_MESSAGE_FROM, VmsMailFrom, &VmsMailFromLength },
      { 255, MAIL$_MESSAGE_SENDER, VmsMailSender, &VmsMailSenderLength },
      { 255, MAIL$_MESSAGE_SUBJECT, VmsMailSubject, &VmsMailSubjectLength },
      { 255, MAIL$_MESSAGE_TO, VmsMailTo, &VmsMailToLength },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   static struct VmsItemLongStruct   MessageNextItem [] = {
      { 0, MAIL$_MESSAGE_NEXT, 0, 0 },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   static struct VmsItemLongStruct   MessageRecordItem [] = {
      { sizeof(MessageRecord)-2, MAIL$_MESSAGE_RECORD,
        MessageRecord, &MessageRecordLength },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   static struct VmsItemLongStruct   MessagesSelectedItem [] = {
      { 4, MAIL$_MESSAGE_SELECTED, &VmsMailMessageSelectedCount, 0 },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   boolean  InRfc822From,
            InRfc822ReplyTo,
            Rfc822From,
            Rfc822MessageId,
            Rfc822Received,
            Rfc822ReplyTo,
            Rfc822ReturnPath;
   int  status,
        LineCount,
        ItemRetry;
   char  *cptr, *sptr, *zptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MailReadMessage()\n");

   MimeQuotedPrintable =
      InRfc822From = InRfc822ReplyTo = 
      Rfc822From = Rfc822MessageId = Rfc822Received =
      Rfc822ReplyTo = Rfc822ReturnPath = false;
   VmsMailRfc822From[0] = 
      VmsMailRfc822MimeCharset[0] = 
      VmsMailRfc822MimeContTransEnc[0] = 
      VmsMailRfc822MimeContentType[0] = '\0';
   VmsMailRfc822FromLength = 0;
   ItemRetry = MessageContext = VmsMailRfc822MimeVersion = 0;

   if (VmsMailTextPtr != NULL)
   {
      free (VmsMailTextPtr);
      VmsMailTextPtr = NULL;
   }
   VmsMailTextLength = VmsMailHeaderLength = VmsMailRfc822HeaderLength = 0;
   VmsMailTextPtr = calloc (1, MESSAGE_TEXT_CHUNK);
   VmsMailTextSize = MESSAGE_TEXT_CHUNK;

   status = mail$message_begin (&MessageContext,
                                &MessageBeginItem,
                                &NullItem);
   if (Debug) fprintf (stdout, "mail$message_begin() %%X%08.08X\n", status);
   if (VMSnok (status)) return (status);

   MailMessageSelect (&MessageContext);

   if (CurrentMessageCount != -1 &&
       CurrentMessageCount != VmsMailMessageSelectedCount)
   {
      /* just use this bogus status to indicate a message count disparity */
      status = SS$_POWERFAIL;
   }

   while (VMSok (status))
   {
      status = mail$message_get (&MessageContext,
                                 &MessageIdItem,
                                 &MessageInfoItem);
      if (VMSnok (status)) break;

      if (Debug)
         fprintf (stdout, "mail$message_get() %%X%08.08X %d\n",
                  status, VmsMailMessageId);
                  
      if (VMSnok (status)) break;

      VmsMailTo[VmsMailToLength] = '\0';
      VmsMailCc[VmsMailCcLength] = '\0';
      VmsMailDate[VmsMailDateLength] = '\0';
      if (isspace(VmsMailDate[0])) VmsMailDate[0] = '0';
      VmsMailExtId[VmsMailExtIdLength] = '\0';
      VmsMailFrom[VmsMailFromLength] = '\0';
      VmsMailSender[VmsMailSenderLength] = '\0';
      VmsMailSubject[VmsMailSubjectLength] = '\0';

#ifdef YAHMAIL_MUNPACK
      VmsMailToLength = fromiso_string (VmsMailTo);
      VmsMailCcLength = fromiso_string (VmsMailCc);
      VmsMailFromLength = fromiso_string (VmsMailFrom);
      VmsMailSenderLength = fromiso_string (VmsMailSender);
      VmsMailSubjectLength = fromiso_string (VmsMailSubject);
#endif /* YAHMAIL_MUNPACK */

      StripTransport (VmsMailCc, &VmsMailCcLength);
      StripTransport (VmsMailFrom, &VmsMailFromLength);
      StripTransport (VmsMailSender, &VmsMailSenderLength);
      StripTransport (VmsMailTo, &VmsMailToLength);

      if (Debug)
         fprintf (stdout, "|%s|\n|%s|\n|%s|\n|%s|\n|%s|\n|%s|\n|%s|\n",
                  VmsMailDate, VmsMailTo, VmsMailCc, VmsMailFrom,
                  VmsMailSender, VmsMailSubject, VmsMailExtId);

      LineCount = 0;

      for (;;)
      {
         status = mail$message_get (&MessageContext,
                                    &MessageContinueItem,
                                    &MessageRecordItem);
/**
         if (Debug)
            fprintf (stdout, "mail$message_get() %%X%08.08X %d bytes\n",
                     status, MessageRecordLength);
**/
         if (VMSnok (status))
         {
            if (status == MAIL$_INVITMLEN && !ItemRetry++)
            {
               /*
                  Nobody (on c.o.v. anyway) seems interested in providing
                  information on which VMS version the message_get item
                  increased from 255 to 2048 characters, so ... if we get
                  an invalid item length status just drop back from 2048
                  to 255.  A little inefficient under older VMS versions,
                  but it works on all!
               */  
               MessageRecordItem[0].buf_len = 255;
               continue;
            }
            break;
         }

         MessageRecord[MessageRecordLength++] = '\n';
         MessageRecord[MessageRecordLength] = '\0';
         /** if (Debug) fprintf (stdout, "|%s|\n", MessageRecord); **/

         if (VmsMailTextLength + MessageRecordLength > VmsMailTextSize)
         {
             VmsMailTextSize += MESSAGE_TEXT_CHUNK;
             VmsMailTextPtr = realloc (VmsMailTextPtr, VmsMailTextSize);
         }

         /* add in the date as if it's part of the VMS mail heading */
         if (!LineCount++)
            VmsMailTextLength +=
               sprintf (VmsMailTextPtr, "Date:\t%s\n", VmsMailDate);

         if (MessageRecordLength == 1 &&
             !VmsMailHeaderLength)
            VmsMailHeaderLength = VmsMailTextLength + 1;
         else
         if (MessageRecordLength == 1 &&
             VmsMailHeaderLength &&
             !VmsMailRfc822HeaderLength)
         {
            VmsMailRfc822HeaderLength = VmsMailTextLength -
                                        VmsMailHeaderLength + 1;
            if (Debug)
               fprintf (stdout, "rfc822: %d\n", VmsMailRfc822HeaderLength);
         }

         /* if haven't encountered the first empty line in the message body */
         if (VmsMailHeaderLength &&
             !VmsMailRfc822HeaderLength)
         {
            /* anything look a little RFC822? */
            if (strsame (MessageRecord, "Message-Id:", 11))
               Rfc822MessageId = true;
            else
            if (!Rfc822Received && strsame (MessageRecord, "Received:", 9))
               Rfc822Received = true;
            else
            if (!Rfc822ReturnPath &&
                strsame (MessageRecord, "Return-Path:", 12))
               Rfc822ReturnPath = true;
            else
            if (strsame (MessageRecord, "MIME-version:", 13))
               VmsMailRfc822MimeVersion = 1;
            else
            if (strsame (MessageRecord, "Content-Transfer-Encoding:", 26))
            {
               /* MIME content-transfer-encoding field */
               cptr = MessageRecord + 26;
               while (*cptr && isspace(*cptr)) cptr++;
               sptr = VmsMailRfc822MimeContTransEnc;
               while (*cptr && *cptr != ';' && !isspace(*cptr))
                  *sptr++ = *cptr++;
               *sptr = '\0';
               if (Debug)
                  fprintf (stdout, "rfc822 cont-trans-enc |%s|\n",
                           VmsMailRfc822MimeContTransEnc);
               if (strsame (VmsMailRfc822MimeContTransEnc,
                            "quoted-printable", 16))
                  MimeQuotedPrintable = true;
            }
            else
            if (strsame (MessageRecord, "Content-Type:", 13))
            {
               /* MIME content-type field */
               cptr = MessageRecord + 13;
               while (*cptr && isspace(*cptr)) cptr++;
               sptr = VmsMailRfc822MimeContentType;
               while (*cptr && *cptr != ';' && !isspace(*cptr))
                  *sptr++ = *cptr++;
               *sptr = '\0';
               if (Debug)
                  fprintf (stdout, "|%s|\n", VmsMailRfc822MimeContentType);

               if ((cptr = strstr (MessageRecord, "charset=")) != NULL ||
                   (cptr = strstr (MessageRecord, "CHARSET=")) != NULL)
               {
                  cptr += 8;
                  sptr = VmsMailRfc822MimeCharset;
                  while (*cptr && !isspace(*cptr) && *cptr != ';')
                     *sptr++ = *cptr++;
                  *sptr = '\0';
                  if (Debug)
                     fprintf (stdout, "rfc822 charset |%s|\n",
                              VmsMailRfc822MimeCharset);
               }
            }

            /* now RFC822 header lines can span multiple records */
            if (strsame (MessageRecord, "From:", 5))
            {
               InRfc822From = Rfc822From = true;
               cptr = MessageRecord + 5;
               while (*cptr && isspace(*cptr)) cptr++;
            }
            else
            if (InRfc822From && isspace(MessageRecord[0]))
            {
               cptr = MessageRecord;
               while (*cptr && isspace(*cptr)) cptr++;
            }
            else
            if (InRfc822From)
            {
               InRfc822From = false;
               sptr = VmsMailRfc822From + VmsMailRfc822FromLength;
               if (sptr > VmsMailRfc822From) sptr--;
               while (sptr > VmsMailRfc822From && isspace(*sptr)) sptr--;
               if (sptr > VmsMailRfc822From) sptr++;
               *sptr = '\0';
               VmsMailRfc822FromLength = sptr - VmsMailRfc822From;
               if (Debug)
                  fprintf (stdout, "rfc822 from |%s|\n", VmsMailRfc822From);
            }

            if (InRfc822From)
            {
               zptr = VmsMailRfc822From + SizeOfVmsMailRfc822From - 1;
               sptr = VmsMailRfc822From + VmsMailRfc822FromLength;
               while (*cptr && sptr < zptr)
               {
                  if (!isprint(*cptr))
                     *sptr++ = ' ';
                  else
                     *sptr++ = *cptr;
                  cptr++;
               }
               *sptr = '\0';
               VmsMailRfc822FromLength = sptr - VmsMailRfc822From;
            }

            /* so can 'reply-to's */
            if (!Rfc822ReplyTo && strsame (MessageRecord, "Reply-To:", 9))
            {
               InRfc822ReplyTo = Rfc822ReplyTo = true;
               cptr = MessageRecord + 9;
               while (*cptr && isspace(*cptr)) cptr++;
            }
            else
            if (InRfc822ReplyTo && isspace(MessageRecord[0]))
            {
               cptr = MessageRecord;
               while (*cptr && isspace(*cptr)) cptr++;
            }
            else
            if (InRfc822ReplyTo)
            {
               InRfc822ReplyTo = false;
               sptr = VmsMailRfc822ReplyTo + VmsMailRfc822ReplyToLength;
               if (sptr > VmsMailRfc822ReplyTo) sptr--;
               while (sptr > VmsMailRfc822ReplyTo && isspace(*sptr)) sptr--;
               if (sptr > VmsMailRfc822ReplyTo) sptr++;
               *sptr = '\0';
               VmsMailRfc822ReplyToLength = sptr - VmsMailRfc822ReplyTo;
               if (Debug)
                  fprintf (stdout, "rfc822 reply-to |%s|\n",
                           VmsMailRfc822ReplyTo);
            }

            if (InRfc822ReplyTo)
            {
               zptr = VmsMailRfc822ReplyTo + SizeOfVmsMailRfc822ReplyTo - 1;
               sptr = VmsMailRfc822ReplyTo + VmsMailRfc822ReplyToLength;
               while (*cptr && sptr < zptr)
               {
                  if (!isprint(*cptr))
                     *sptr++ = ' ';
                  else
                     *sptr++ = *cptr;
                  cptr++;
               }
               *sptr = '\0';
               VmsMailRfc822ReplyToLength = sptr - VmsMailRfc822ReplyTo;
            }
         }

         /* copy the terminating null as well! */
         memcpy (VmsMailTextPtr+VmsMailTextLength,
                 MessageRecord, MessageRecordLength+1);

         VmsMailTextLength += MessageRecordLength;
      }

      if (Debug)
         fprintf (stdout, "%d %d %d %d |%s|\n",
                  VmsMailTextSize, VmsMailTextLength, VmsMailHeaderLength,
                  VmsMailRfc822HeaderLength, "" /** VmsMailTextPtr **/);

      if (!VmsMailHeaderLength) VmsMailHeaderLength = VmsMailTextLength;
   }

   /* no more records is just fine! */
   if (status == MAIL$_NOMOREREC) status = SS$_NORMAL;

   /* if it doesn't look like an RFC822 headered body then forget it! */
   if (VmsMailRfc822HeaderLength &&
       !(Rfc822From ||
         Rfc822MessageId ||
         Rfc822Received ||
         Rfc822ReplyTo ||
         Rfc822ReturnPath ||
         VmsMailRfc822MimeVersion))
   {
      VmsMailRfc822HeaderLength = VmsMailRfc822MimeVersion = 0;
      VmsMailRfc822MimeCharset[0] =
         VmsMailRfc822MimeContTransEnc[0] =
         VmsMailRfc822MimeContentType[0] = '\0';
   }

#ifdef YAHMAIL_MUNPACK
   VmsMailRfc822FromLength = fromiso_string (VmsMailRfc822From);
   VmsMailRfc822ReplyToLength = fromiso_string (VmsMailRfc822ReplyTo);
#endif /* YAHMAIL_MUNPACK */

   mail$message_end (&MessageContext, &NullItem, &NullItem);

   return (status);
}

/*****************************************************************************/
/*
Generate a linked list containing details of all selected messages in the
selected folder.  Begin with the message ID calculated using 'BrowsePage' and
'BrowseWindow' into 'IdTo' (the least recent) and get up until 'IdFrom' (the
most recent).
*/

int GenerateMessageList ()

{
   static int  IdTo;
   static unsigned long  MessageContext,
                         MessageSelected;

   static struct VmsItemLongStruct   MessageBeginItem [] = {
      { 4, MAIL$_MESSAGE_FILE_CTX, &MailFileContext, 0 },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   static struct VmsItemLongStruct   MessageInfoItem [] = {
      { 4, MAIL$_MESSAGE_CURRENT_ID, &VmsMailMessageId, 0 },
      { 8, MAIL$_MESSAGE_BINARY_DATE, &VmsMailBinaryDate, 0 },
      { 255, MAIL$_MESSAGE_DATE, VmsMailDate, &VmsMailDateLength },
      { 255, MAIL$_MESSAGE_FROM, VmsMailFrom, &VmsMailFromLength },
      { 255, MAIL$_MESSAGE_SUBJECT, VmsMailSubject, &VmsMailSubjectLength },
      { 255, MAIL$_MESSAGE_TO, VmsMailTo, &VmsMailToLength },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   static struct VmsItemLongStruct   MessageNextItem [] = {
      { 0, MAIL$_MESSAGE_NEXT, 0, 0 },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   /* least recent item of window */
   static struct VmsItemLongStruct   MessageFirstItem [] = {
      { 4, MAIL$_MESSAGE_ID, &IdTo, 0 },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   static struct VmsItemLongStruct   MessagesSelectedItem [] = {
      { 4, MAIL$_MESSAGE_SELECTED, &VmsMailMessageSelectedCount, 0 },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   boolean  LooksLikeIt;
   int  status,
        CallocSize,
        IdFrom,
        IdPage,
        LineCount;
   char  *SpamReasonPtr;
   char  SpamReason [256];
   struct MessageListStruct  *mlptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "GenerateMessageList()\n");

   if (MessageListHeadPtr != NULL)
   {
      /* empty any current list of messages */
      struct MessageListStruct  *meptr;
      mlptr = MessageListHeadPtr;
      while (mlptr != NULL)
      {
         meptr = mlptr;
         mlptr = mlptr->NextPtr;
         free (meptr);
      }
      MessageListHeadPtr = MessageListTailPtr = NULL;
      MessageListCount = 0;
   }

   LooksLikeSpamCount = MessageContext = PublicConfigSubjCount = 0;

   status = mail$message_begin (&MessageContext,
                                &MessageBeginItem,
                                &NullItem);
   if (Debug) fprintf (stdout, "mail$message_begin() %%X%08.08X\n", status);
   if (VMSnok (status)) return (status);

   MailMessageSelect (&MessageContext);

   /* after it is established how many messages we have to work with */
   IdPage = 1;
   IdFrom = VmsMailMessageSelectedCount;
   IdTo = IdFrom - BrowseWindow + 1;
   if (IdTo <= 0) IdTo = 1;
   while (IdPage != BrowsePage)
   {
      IdFrom -= BrowseWindow;
      IdTo = IdFrom - BrowseWindow + 1;
      IdPage++;
      if (IdTo <= 0)
      {
         IdTo = 1;
         break;
      }
   }

   /* start with the least recent of the browse window */
   status = mail$message_get (&MessageContext,
                              &MessageFirstItem,
                              &MessageInfoItem);
   if (Debug)
      fprintf (stdout, "mail$message_get() %%X%08.08X %d\n",
               status, VmsMailMessageId);

   while (VMSok (status))
   {
      VmsMailTo[VmsMailToLength] = '\0';
      VmsMailDate[VmsMailDateLength] = '\0';
      if (isspace(VmsMailDate[0])) VmsMailDate[0] = '0';
      VmsMailFrom[VmsMailFromLength] = '\0';
      VmsMailSubject[VmsMailSubjectLength] = '\0';

#ifdef YAHMAIL_MUNPACK
      VmsMailToLength = fromiso_string (VmsMailTo);
      VmsMailFromLength = fromiso_string (VmsMailFrom);
      VmsMailSubjectLength = fromiso_string (VmsMailSubject);
#endif /* YAHMAIL_MUNPACK */

      StripTransport (VmsMailFrom, &VmsMailFromLength);
      StripTransport (VmsMailTo, &VmsMailToLength);

      if (Debug)
         fprintf (stdout, "|%s|\n|%s|\n|%s|\n|%s|\n",
                  VmsMailDate, VmsMailTo, VmsMailSender, VmsMailSubject);

      if (DoDisplaySpam || HideSpam >= 10)
         SpamReasonPtr = SpamReason;
      else
         SpamReasonPtr = NULL;

      if (HideSpam)
         LooksLikeIt = LooksLikeSpam (VmsMailTo, VmsMailFrom,
                                      VmsMailSender, VmsMailSubject,
                                      SpamReasonPtr);
      else
         LooksLikeIt = false;

      if (ConfigPublic &&
          strstr (VmsMailSubject, PublicConfigSubjPtr) != NULL)
      {
         /* ignore the YAHMAIL folder configuration message */
         PublicConfigSubjCount++;
      }
      else
      if (HideSpam && DoNotDisplaySpam && LooksLikeIt)
      {
         /* looks like a duck, walks like a duck, quacks like a duck */
         LooksLikeSpamCount++;
      }
      else
      if (!HideSpam ||
          (!DoDisplaySpam && !DoNotDisplaySpam) ||
          (DoDisplaySpam && LooksLikeIt) ||
          (DoNotDisplaySpam && !LooksLikeIt))
      {
         /* add to message list */
         if (LooksLikeIt) LooksLikeSpamCount++;
         CallocSize = sizeof(struct MessageListStruct) +
                      VmsMailDateLength + 1 +
                      VmsMailToLength + 1 +
                      VmsMailFromLength + 1 +
                      VmsMailSubjectLength + 1;
         if (SpamReasonPtr) CallocSize += strlen(SpamReasonPtr) + 1;
         CallocSize += 16;  /* elbow room */

         mlptr = calloc (1, CallocSize);
         if (mlptr == NULL) exit (vaxc$errno);
         mlptr->NextPtr = NULL;
         mlptr->MessageId = VmsMailMessageId;
         memcpy (&mlptr->BinaryDate,
                 &VmsMailBinaryDate,
                 sizeof(mlptr->BinaryDate));
         mlptr->DatePtr = (char*)&mlptr->Data;
         strcpy (mlptr->DatePtr, VmsMailDate);
         mlptr->ToPtr = mlptr->DatePtr + VmsMailDateLength + 1;
         strcpy (mlptr->ToPtr, VmsMailTo);
         mlptr->FromPtr = mlptr->ToPtr + VmsMailToLength + 1;
         strcpy (mlptr->FromPtr, VmsMailFrom);
         mlptr->SubjectPtr = mlptr->FromPtr + VmsMailFromLength + 1;
         strcpy (mlptr->SubjectPtr, VmsMailSubject);
         mlptr->SpamPtr = mlptr->SubjectPtr + VmsMailSubjectLength + 1;
         if (SpamReasonPtr) strcpy (mlptr->SpamPtr, SpamReasonPtr);
         if (MessageListHeadPtr == NULL)
         {
            MessageListHeadPtr = MessageListTailPtr = mlptr;
            mlptr->PrevPtr = mlptr->NextPtr = NULL;
         }
         else
         {
            mlptr->PrevPtr = MessageListTailPtr;
            mlptr->NextPtr = NULL;
            MessageListTailPtr->NextPtr = mlptr;
            MessageListTailPtr = mlptr;
         }
         MessageListCount++;
      }

      /* just did the most recent of the browse window */
      if (VmsMailMessageId == IdFrom) break;

      status = mail$message_get (&MessageContext,
                                 &MessageNextItem,
                                 &MessageInfoItem);
      if (Debug)
         fprintf (stdout, "mail$message_get() %%X%08.08X %d\n",
                  status, VmsMailMessageId);
   }

   if (VMSnok (status))
   {
      /* cannot get this to work for the life of me!!! */
      /** if (status == MAIL$_NOMOREMSG) status = SS$_NORMAL; */
      if (status - MAIL$_NOMOREMSG == 0) status = SS$_NORMAL;
   }

   mail$message_end (&MessageContext, &NullItem, &NullItem);

   return (status);
}

/*****************************************************************************/
/*
Read the body of the special YAHMAIL configuration message.  Selects messages
that contain the special configuration subject string and then reads the latest
(highest message ID number) into the configuration text.
*/

int MailReadPublicConfigMessage ()

{
   static unsigned long  MessageContext,
                         MessageRecordLength;

   static char  MessageRecord [255+2];

   static char  *ConfigTextPtr;

   static struct VmsItemLongStruct   MessageBeginItem [] = {
      { 4, MAIL$_MESSAGE_FILE_CTX, &MailFileContext, 0 },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   static struct VmsItemLongStruct   MessageContinueItem [] = {
      { 0, MAIL$_MESSAGE_CONTINUE, 0, 0 },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   static struct VmsItemLongStruct   MessageIdItem [] = {
      { 4, MAIL$_MESSAGE_ID, &VmsMailMessageSelectedCount, 0 },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   static struct VmsItemLongStruct  MessageSelectItem [] = {
      { 0, MAIL$_MESSAGE_FOLDER, MailFolderName, 0 },
      { 0, MAIL$_MESSAGE_SUBJ_SUBSTRING, 0, 0 },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   static struct VmsItemLongStruct   MessageNextItem [] = {
      { 0, MAIL$_MESSAGE_NEXT, 0, 0 },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   static struct VmsItemLongStruct   MessageRecordItem [] = {
      { 255, MAIL$_MESSAGE_RECORD, MessageRecord, &MessageRecordLength },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   static struct VmsItemLongStruct   MessageSelectedItem [] = {
      { 4, MAIL$_MESSAGE_SELECTED, &VmsMailMessageSelectedCount, 0 },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   int  cnt,
        status,
        ConfigTextLength,
        ConfigTextSize;
   char  *bptr, *cptr, *sptr, *tptr;

   /*********/
   /* begin */
   /*********/
                                 
   if (Debug) fprintf (stdout, "MailReadPublicConfigMessage()\n");

   PublicHeaderPtr = PublicFooterPtr = PublicTitlePtr = NULL;

   MessageContext = 0;

   status = mail$message_begin (&MessageContext,
                                &MessageBeginItem,
                                &NullItem);
   if (Debug) fprintf (stdout, "mail$message_begin() %%X%08.08X\n", status);
   if (VMSnok (status)) return (status);

   if (!strcmp (MailFolderName, lang_BtnToWasteBasket))
   {
      MessageSelectItem[0].buf_addr = VmsMailWasteBasketName;
      MessageSelectItem[0].buf_len = VmsMailWasteBasketNameLength;
      if (Debug) fprintf (stdout, "w/b |%s|\n", VmsMailWasteBasketName);
   }
   else
   {
      MessageSelectItem[0].buf_addr = MailFolderName;
      MessageSelectItem[0].buf_len = MailFolderNameLength;
   }

   MessageSelectItem[1].buf_len = PublicConfigSubjLength;
   MessageSelectItem[1].buf_addr = PublicConfigSubjPtr;

   status = mail$message_select (&MessageContext,
                                 &MessageSelectItem,
                                 &MessageSelectedItem);
   if (Debug)
      fprintf (stdout, "mail$message_select() %%X%08.08X %d\n",
               status, VmsMailMessageSelectedCount);

   while (VMSok (status))
   {
      /* always get the latest config message in the folder */
      status = mail$message_get (&MessageContext,
                                 &MessageIdItem,
                                 &NullItem);
      if (VMSnok (status)) break;

      if (Debug)
         fprintf (stdout, "mail$message_get() %%X%08.08X %d\n",
                  status, VmsMailMessageId);
                  
      if (VMSnok (status)) break;

      /************************/
      /* get the message text */
      /************************/

      if (ConfigTextPtr != NULL) free (ConfigTextPtr);
      ConfigTextLength = 0;
      ConfigTextPtr = calloc (1, CONFIG_TEXT_CHUNK);
      ConfigTextSize = CONFIG_TEXT_CHUNK;

      for (;;)
      {
         status = mail$message_get (&MessageContext,
                                    &MessageContinueItem,
                                    &MessageRecordItem);
         if (Debug)
            fprintf (stdout, "mail$message_get() %%X%08.08X %d bytes\n",
                     status, MessageRecordLength);
         if (VMSnok (status)) break;

         MessageRecord[MessageRecordLength++] = '\n';
         MessageRecord[MessageRecordLength] = '\0';
         if (Debug) fprintf (stdout, "|%s|\n", MessageRecord);

         if (ConfigTextLength+MessageRecordLength > ConfigTextSize)
         {
             ConfigTextSize += CONFIG_TEXT_CHUNK;
             ConfigTextPtr = realloc (ConfigTextPtr, ConfigTextSize);
         }

         /* copy the terminating null as well! */
         memcpy (ConfigTextPtr+ConfigTextLength,
                 MessageRecord, MessageRecordLength+1);

         ConfigTextLength += MessageRecordLength;
      }

      if (Debug)
         fprintf (stdout, "%d %d |%s|\n",
                  ConfigTextSize, ConfigTextLength, ConfigTextPtr);

      /*********************************/
      /* process configuration message */
      /*********************************/

      cptr = ConfigTextPtr;
      while (*cptr)
      {
         if (strsame (cptr, bptr = "[body]", cnt = 6) ||
             strsame (cptr, bptr = "[header]", cnt = 8) ||
             strsame (cptr, bptr = "[footer]", cnt = 8) ||
             strsame (cptr, bptr = "[title]", cnt = 7))
         {
            /* find the start of the next directive or end of text */
            sptr = (cptr += cnt);
            while (*cptr && *(unsigned short*)cptr != '\n[') cptr++;
            if (*cptr) *cptr++ = '\0';

            /* trim leading then trailing white space */
            while (*sptr && isspace(*sptr)) sptr++;
            tptr = sptr;
            while (*sptr) sptr++;
            if (sptr > tptr) sptr--;
            while (sptr > tptr && isspace(*sptr)) sptr--;
            if (sptr > tptr) *++sptr = '\0';

            /* set the appropriate pointer */
            if (strsame (bptr, "[body]", 6))
               ConfigBodyTagPtr = tptr;
            else
            if (strsame (bptr, "[header]", 8))
               PublicHeaderPtr = tptr;
            else
            if (strsame (bptr, "[footer]", 8))
               PublicFooterPtr = tptr;
            else
            if (strsame (bptr, "[title]", 7))
               PublicTitlePtr = tptr;
         }
         else
         {
            /* find the start of the next line */
            while (*cptr && *cptr != '\n') cptr++;
            if (*cptr) cptr++;
         }
      }

      if (Debug)
         fprintf (stdout, "|%s|\n|%s|\n|%s|\n|%s|\n",
                  ConfigBodyTagPtr, PublicTitlePtr,
                  PublicHeaderPtr, PublicFooterPtr);
   }

   /* no more records is just fine! */
   if (status == MAIL$_NOMOREREC) status = SS$_NORMAL;

   mail$message_end (&MessageContext, &NullItem, &NullItem);

   return (status);
}

/*****************************************************************************/
/*
Delete one or more mail messages.  Folder name is specified by 'MailFolderName'
and messages IDs in 'MailMessageId[0..n]'.
*/

int DeleteMessage ()

{
   static unsigned long  MessageContext,
                         MessageSelected;

   static struct VmsItemLongStruct   MessageBeginItem [] = {
      { 4, MAIL$_MESSAGE_FILE_CTX, &MailFileContext, 0 },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   static struct VmsItemLongStruct   MessageDeleteItem [] = {
      { 4, MAIL$_MESSAGE_ID, 0, 0 },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   int  status,
        MessageIdCount;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DeleteMessage()\n");

   MessageContext = 0;

   status = mail$message_begin (&MessageContext,
                                &MessageBeginItem,
                                &NullItem);
   if (Debug) fprintf (stdout, "mail$message_begin() %%X%08.08X\n", status);
   if (VMSnok (status)) return (status);

   MailMessageSelect (&MessageContext);

   if (CurrentMessageCount != -1 &&
       CurrentMessageCount != VmsMailMessageSelectedCount)
   {
      /* just use this bogus status to indicate a message count disparity */
      status = SS$_POWERFAIL;
   }

   MessageIdCount = 0;

   while (VMSok (status))
   {
      if (!MailMessageId[MessageIdCount]) break;
      if (Debug) fprintf (stdout, "id %d\n", MailMessageId[MessageIdCount]);
      MessageDeleteItem[0].buf_addr = &MailMessageId[MessageIdCount];
      MessageIdCount++;

      status = mail$message_delete (&MessageContext,
                                    &MessageDeleteItem,
                                    &NullItem);
      if (Debug)
         fprintf (stdout, "mail$message_delete() %%X%08.08X\n", status);

      if (VMSnok (status)) break;
   }

   mail$message_end (&MessageContext, &NullItem, &NullItem);

   /* prevent the message complaining you can't delete from the WASTEBASKET */
   if (status == MAIL$_DELWASTE) status = SS$_NORMAL;
   /* 'cause you can! */

   if (VMSok (status))
   {
      /* if the NEWMAIL folder was involved then adjust new message count */
      if (!strcmp (MailFileName, "MAIL") ||
          !strncmp (MailFileName, "MAIL.", 5))
      {
         if (!strcmp (MailFolderName, "NEWMAIL"))
            MailSetUserNewMessages (-MessageIdCount);
      }
   }

   return (status);
}

/*****************************************************************************/
/*
Purge the WASTEBASKET folder.
*/

int EmptyWasteBasket ()

{
   static struct VmsItemLongStruct   MailFileMessagesDeletedItem [] = {
      { 4, MAIL$_MAILFILE_MESSAGES_DELETED, &MailFileMessagesDeleted, 0 },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "EmptyWasteBasket()\n");

   MailFileMessagesDeleted = 0;

   status = mail$mailfile_purge_waste (&MailFileContext,
                                       &NullItem,
                                       &MailFileMessagesDeletedItem);
   if (Debug) fprintf (stdout, "mail$message_select() %%X%08.08X\n", status);

   return (status);
}

/*****************************************************************************/
/*
Copy or move (copy with delete from original) one or more mail messages. 
Source folder name is specified by  'MailFolderName', destination folder name
by 'CopyFolderName' and message IDs in 'MailMessageId[0..n]'.
*/

int CopyMoveMessage ()

{
   static unsigned long  MessageContext,
                         MessageSelected;

   static struct VmsItemLongStruct   MessageBeginItem [] = {
      { 4, MAIL$_MESSAGE_FILE_CTX, &MailFileContext, 0 },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   static struct VmsItemLongStruct   MessageCopyItem [] = {
      { 4, MAIL$_MESSAGE_ID, 0, 0 },
      { 0, MAIL$_MESSAGE_FOLDER, 0, 0 },
      { 0,0,0,0 }
   };

   static struct VmsItemLongStruct   MessageMoveItem [] = {
      { 4, MAIL$_MESSAGE_ID, 0, 0 },
      { 0, MAIL$_MESSAGE_FOLDER, 0, 0 },
      { 0, MAIL$_MESSAGE_DELETE, 0, 0 },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   int  status,
        MessageIdCount;
   struct VmsItemLongStruct   *ItemPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "CopyMoveMessage() |%s|->|%s|\n",
               MailFolderName, CopyFolderName);

   MessageContext = 0;

   status = mail$message_begin (&MessageContext,
                                &MessageBeginItem,
                                &NullItem);
   if (Debug) fprintf (stdout, "mail$message_begin() %%X%08.08X\n", status);
   if (VMSnok (status)) return (status);

   MailMessageSelect (&MessageContext);

   if (CurrentMessageCount != -1 &&
       CurrentMessageCount != VmsMailMessageSelectedCount)
   {
      /* just use this bogus status to indicate a message count disparity */
      status = SS$_POWERFAIL;
   }

   if (DoMessageCopy)
      ItemPtr = (struct VmsItemLongStruct*)&MessageCopyItem;
   else
      ItemPtr = (struct VmsItemLongStruct*)&MessageMoveItem;

   if (!strcmp (CopyFolderName, lang_BtnToWasteBasket))
   {
      ItemPtr[1].buf_addr = VmsMailWasteBasketName;
      ItemPtr[1].buf_len = VmsMailWasteBasketNameLength;
      if (Debug) fprintf (stdout, "w/b |%s|\n", VmsMailWasteBasketName);
   }
   else
   {
      ItemPtr[1].buf_addr = CopyFolderName;
      ItemPtr[1].buf_len = CopyFolderNameLength;
   }

   MessageIdCount = 0;

   while (VMSok (status))
   {
      if (!MailMessageId[MessageIdCount]) break;
      if (Debug) fprintf (stdout, "id %d\n", MailMessageId[MessageIdCount]);
      ItemPtr[0].buf_addr = &MailMessageId[MessageIdCount];
      MessageIdCount++;

      status = mail$message_copy (&MessageContext,
                                  ItemPtr,
                                  &NullItem);
      if (Debug) fprintf (stdout, "mail$message_copy() %%X%08.08X\n", status);

      if (VMSnok (status)) break;
   }

   mail$message_end (&MessageContext, &NullItem, &NullItem);

   if (VMSok (status))
   {
      /* if the NEWMAIL folder was involved then adjust new message count */
      if (!strcmp (MailFileName, "MAIL") ||
          !strncmp (MailFileName, "MAIL.", 5))
      {
         if (!strcmp (MailFolderName, "NEWMAIL"))
            MailSetUserNewMessages (-MessageIdCount);
         else
         if (!strcmp (CopyFolderName, "NEWMAIL"))
            MailSetUserNewMessages (MessageIdCount);
      }
   }

   return (status);
}

/*****************************************************************************/
/*
Generate a linked list of folder names by calling GenerateFolderListEntry() for
each folder in the current mail file.
*/

int GenerateFolderList ()

{
   static unsigned long  MessageContext;

   static struct VmsItemLongStruct   MessageBeginItem [] = {
      { 4, MAIL$_MESSAGE_FILE_CTX, &MailFileContext, 0 },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   static struct VmsItemLongStruct   MessageFolderRoutineItem [] = {
      { 4, MAIL$_MAILFILE_FOLDER_ROUTINE, (void*)GenerateFolderListEntry, 0 },
      { 4, MAIL$_MAILFILE_USER_DATA, &MessageContext, 0 },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "GenerateFolderList()\n");

   if (FolderListHeadPtr != NULL)
   {
      /* empty any current list of folders */
      struct FolderListStruct  *flptr, *feptr;
      flptr = FolderListHeadPtr;
      while (flptr != NULL)
      {
         feptr = flptr;
         flptr = flptr->NextPtr;
         free (feptr);
      }
      FolderListHeadPtr = FolderListTailPtr = NULL;
      FolderListCount = 0;
   }

   MessageContext = 0;

   status = mail$mailfile_info_file (&MailFileContext,
                                     &MessageFolderRoutineItem,
                                     &NullItem);
   if (Debug)
      fprintf (stdout, "mail$message_select() %%X%08.08X\n",
               status);

   mail$message_end (&MessageContext, &NullItem, &NullItem);

   return (status);
}

/*****************************************************************************/
/*
Called once from GenerateFolderList() for each folder name in the current mail
file.  Put the name into a linked list.
*/

int GenerateFolderListEntry
(
unsigned long UserData,
struct dsc$descriptor *FolderDscPtr
)
{
   int  status,
        CallocSize;
   struct FolderListStruct  *flptr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "GenerateFolderListEntry() %d\n",
               FolderDscPtr->dsc$w_length);

   if (!FolderDscPtr->dsc$w_length) return (SS$_NORMAL);

   strncpy (VmsMailFolder, FolderDscPtr->dsc$a_pointer,
                           FolderDscPtr->dsc$w_length);
   VmsMailFolder[FolderDscPtr->dsc$w_length] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", VmsMailFolder);

   CallocSize = sizeof(struct FolderListStruct) + 
                FolderDscPtr->dsc$w_length + 1;
   flptr = calloc (1, CallocSize);
   if (flptr == NULL) exit (vaxc$errno);
   flptr->NextPtr = NULL;
   flptr->FolderNamePtr = (char*)&flptr->Data;
   strcpy (flptr->FolderNamePtr, VmsMailFolder);
   if (FolderListHeadPtr == NULL)
      FolderListHeadPtr = FolderListTailPtr = flptr;
   else
   {
      FolderListTailPtr->NextPtr = flptr;
      FolderListTailPtr = flptr;
   }
   FolderListCount++;

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Open the Mail file in 'MailFileName'.  If this contains a ':' then consider it
a full VMS file specification (e.g. USER_DISK:[ME.MAIL]TEST.MAI), otherwise a
mail file based in the current Mail directory (e.g. MAIL).  The ".MAI"
extension defaults and so is not needed.
*/

int MailFileOpen ()

{
   static struct VmsItemLongStruct   MailFileNameItem [] = {
      { 0, MAIL$_MAILFILE_NAME, MailFileName, &MailFileNameLength },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   static struct VmsItemLongStruct   WasteBasketNameItem [] = {
      { 255, MAIL$_MAILFILE_WASTEBASKET,
        VmsMailWasteBasketName, &VmsMailWasteBasketNameLength },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   int  status;
   char  Scratch [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MailFileOpen()\n");

   UserContext = MailFileContext = 0;

   status = mail$mailfile_begin (&MailFileContext, &NullItem, &NullItem);
   if (Debug) fprintf (stdout, "mail$mailfile_begin() %%X%08.08X\n", status);
   if (VMSnok (status)) return (status);

   if (strchr (MailFileName, ':') == NULL)
   {
      strcpy (Scratch, VmsMailUserFullDirectory);
      strcat (Scratch, MailFileName);
      MailFileNameItem[0].buf_addr = Scratch;
      MailFileNameItem[0].buf_len = strlen(Scratch);
   }
   else
   {
      MailFileNameItem[0].buf_addr = MailFileName;
      MailFileNameItem[0].buf_len = MailFileNameLength;
   }
   if (Debug)
      fprintf (stdout, "MailFileNameItem[0].buf_addr |%s|\n",
               (char*)MailFileNameItem[0].buf_addr);

   status = mail$mailfile_open (&MailFileContext,
                                &MailFileNameItem,
                                &WasteBasketNameItem);
   if (Debug) fprintf (stdout, "mail$mailfile_open() %%X%08.08X\n", status);
                            
   if (VMSok (status))
   {
      VmsMailWasteBasketName[VmsMailWasteBasketNameLength] = '\0';
      if (Debug)
         fprintf (stdout, "VmsMailWasteBasketName |%s|\n",
                  VmsMailWasteBasketName);
   }

   return (status);
}

/*****************************************************************************/
/*
Close the currently open mail file.  If the request involved a message delete
the also purge the WASTEBASKET.
*/ 

MailFileClose ()

{
   static unsigned long  MailFileTotalReclaimBytes;

   static struct VmsItemLongStruct   MailFileFullCloseItem [] = {
      { 0, MAIL$_MAILFILE_FULL_CLOSE, 0, 0 },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   static struct VmsItemLongStruct   MailFileTotalReclaimItem [] = {
      { 4, MAIL$_MAILFILE_TOTAL_RECLAIM, &MailFileTotalReclaimBytes, 0 },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MailFileClose()\n");

   MailFileTotalReclaimBytes = 0;

   if (DoMessageDelete && VmsMailUserAutoPurge)
      status = mail$mailfile_close (&MailFileContext,
                                    &MailFileFullCloseItem,
                                    &MailFileTotalReclaimItem);
   else
      status = mail$mailfile_close (&MailFileContext,
                                    &NullItem,
                                    &NullItem);
   if (Debug) fprintf (stdout, "mail$mailfile_close() %%X%08.08X\n", status);
   if (VMSnok (status)) return (status);

   status = mail$mailfile_end (&MailFileContext, &NullItem, &NullItem);
   if (Debug) fprintf (stdout, "mail$mailfile_end() %%X%08.08X\n", status);
   return (status);
}

/*****************************************************************************/
/*
Select messages from the currently open Mail file, in the 'MailFolderName'
folder, based on the contents of 'MailSubstringCc', 'MailSubstringFrom',
'MailSubstringSubj', and 'MailSubstringTo'.
*/

int MailMessageSelect (unsigned long *MessageContextPtr)

{
   static struct VmsItemLongStruct   MessageSelectItem [7];

   static struct VmsItemLongStruct   MessageSelectedItem [] = {
      { 4, MAIL$_MESSAGE_SELECTED, &VmsMailMessageSelectedCount, 0 },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   int  idx,
        status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MailMessageSelect() |%s|\n", MailFolderName);

   memset (&MessageSelectItem[idx=0], 0, sizeof(MessageSelectItem));

   MessageSelectItem[idx].item = MAIL$_MESSAGE_FOLDER;

   if (!strcmp (MailFolderName, lang_BtnToWasteBasket))
   {
      MessageSelectItem[idx].buf_addr = VmsMailWasteBasketName;
      MessageSelectItem[idx].buf_len = VmsMailWasteBasketNameLength;
      if (Debug) fprintf (stdout, "w/b |%s|\n", VmsMailWasteBasketName);
   }
   else
   {
      MessageSelectItem[idx].buf_addr = MailFolderName;
      MessageSelectItem[idx].buf_len = MailFolderNameLength;
   }

   if (MailSubstringCcLength)
   {
      MessageSelectItem[++idx].item = MAIL$_MESSAGE_CC_SUBSTRING;
      MessageSelectItem[idx].buf_addr = MailSubstringCc;
      MessageSelectItem[idx].buf_len = MailSubstringCcLength;
   }

   if (MailSubstringFromLength)
   {
      MessageSelectItem[++idx].item = MAIL$_MESSAGE_FROM_SUBSTRING;
      MessageSelectItem[idx].buf_addr = MailSubstringFrom;
      MessageSelectItem[idx].buf_len = MailSubstringFromLength;
   }

   if (MailSubstringSubjLength)
   {
      MessageSelectItem[++idx].item = MAIL$_MESSAGE_SUBJ_SUBSTRING;
      MessageSelectItem[idx].buf_addr = MailSubstringSubj;
      MessageSelectItem[idx].buf_len = MailSubstringSubjLength;
   }

   if (MailSubstringToLength)
   {
      MessageSelectItem[++idx].item = MAIL$_MESSAGE_TO_SUBSTRING;
      MessageSelectItem[idx].buf_addr = MailSubstringTo;
      MessageSelectItem[idx].buf_len = MailSubstringToLength;
   }

   MessageSelectItem[++idx].item = MAIL$_NOSIGNAL;

   status = mail$message_select (MessageContextPtr,
                                 &MessageSelectItem,
                                 &MessageSelectedItem);
   if (Debug)
      fprintf (stdout, "mail$message_select() %%X%08.08X selected: %d\n",
               status, VmsMailMessageSelectedCount);

   return (status);
}

/*****************************************************************************/
/*
This function retieves user profile information for the user name specified by
the global storage 'MailUserName'.
*/

int MailGetUserInfo ()

{
   static struct VmsItemLongStruct   UserDetailsItem [] = {
      { 2, MAIL$_USER_NEW_MESSAGES, &VmsMailUserNewMessages, 0 },
      { 4, MAIL$_USER_AUTO_PURGE, &VmsMailUserAutoPurge, 0 },
      { 4, MAIL$_USER_CC_PROMPT, &VmsMailUserCcPrompt, 0 },
      { 4, MAIL$_USER_COPY_FORWARD, &VmsMailUserCopyForward, 0 },
      { 4, MAIL$_USER_COPY_REPLY, &VmsMailUserCopyReply, 0 },
      { 4, MAIL$_USER_COPY_SEND, &VmsMailUserCopySend, 0 },
      { 255, MAIL$_USER_FORM, VmsMailUserForm, &VmsMailUserFormLength },
      { 255, MAIL$_USER_EDITOR, VmsMailUserEditor, &VmsMailUserEditorLength },
      { 255, MAIL$_USER_FORWARDING, VmsMailUserForwarding, &VmsMailUserForwardingLength },
      { 255, MAIL$_USER_FULL_DIRECTORY, VmsMailUserFullDirectory, &VmsMailUserFullDirectoryLength },
      { 127, MAIL$_USER_PERSONAL_NAME, VmsMailUserPersonalName, &VmsMailUserPersonalNameLength },
      { 255, MAIL$_USER_QUEUE, VmsMailUserQueue, &VmsMailUserQueueLength },
      { 0,0,0,0 }
   };

   static struct VmsItemLongStruct   UserSigFileItem [] = {
      { 255, MAIL$_USER_SIGFILE, VmsMailUserSigFile, &VmsMailUserSigFileLength },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   static struct VmsItemLongStruct   UserNameItem [] = {
      { 0, MAIL$_USER_USERNAME, MailUserName, &MailUserNameLength },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MailGetUserInfo()\n");

   UserContext = 0;

   MailUserNameLength = strlen(MailUserName);
   UserNameItem[0].buf_len = MailUserNameLength;

   status = mail$user_begin (&UserContext, &NullItem, &NullItem);
   if (Debug) fprintf (stdout, "mail$user_begin() %%X%08.08X\n", status);

   /* status from mail$user_begin() */
   if (VMSnok (status)) return (status);

   status = mail$user_get_info (&UserContext,
                                &UserNameItem,
                                &UserDetailsItem);
   if (Debug) fprintf (stdout, "mail$user_get_info() %%X%08.08X\n", status);

   if (VMSok (status))
   {
      VmsMailUserEditor[VmsMailUserEditorLength] = '\0';
      VmsMailUserForwarding[VmsMailUserForwardingLength] = '\0';
      VmsMailUserForm[VmsMailUserFormLength] = '\0';
      VmsMailUserFullDirectory[VmsMailUserFullDirectoryLength] = '\0';
      VmsMailUserPersonalName[VmsMailUserPersonalNameLength] = '\0';
      VmsMailUserQueue[VmsMailUserQueueLength] = '\0';
      VmsMailUserAutoPurge = VmsMailUserAutoPurge & 1;
      VmsMailUserCcPrompt = VmsMailUserCcPrompt & 1;
      VmsMailUserCopyForward = VmsMailUserCopyForward & 1;
      VmsMailUserCopyReply = VmsMailUserCopyReply & 1;
      VmsMailUserCopySend = VmsMailUserCopySend & 1;
      if (Debug)
         fprintf (stdout,
"%d %d %d %d %d %d\n|%s|\n|%s|\n|%s|\n|%s|\n|%s|\n|%s|\n|%s|\n",
                 VmsMailUserNewMessages,
                 VmsMailUserAutoPurge,
                 VmsMailUserCcPrompt,
                 VmsMailUserCopyForward,
                 VmsMailUserCopyReply, 
                 VmsMailUserCopySend, 
                 VmsMailUserEditor,
                 VmsMailUserForm,
                 VmsMailUserForwarding,
                 VmsMailUserFullDirectory,
                 VmsMailUserPersonalName,
                 VmsMailUserSigFile,
                 VmsMailUserQueue);
   }

   VmsMailUserSigFile[0] = '\0';
   if (VmsVersion >= 70 &&
       VMSok (status))
   {
      status = mail$user_get_info (&UserContext,
                                   &UserNameItem,
                                   &UserSigFileItem);
      if (Debug) fprintf (stdout, "mail$user_get_info() %%X%08.08X\n", status);
      if (VMSok (status))
      {
         VmsMailUserSigFile[VmsMailUserSigFileLength] = '\0';
         if (Debug) fprintf (stdout, "|%s|\n", VmsMailUserSigFile);
      }
      status = SS$_NORMAL;
   }

   mail$user_end (&UserContext, &NullItem, &NullItem);

   return (status);
}

/*****************************************************************************/
/*
This function changes entries in the user profile specified by the global
storage 'MailUserName'.  It checks for the existance of CGI variables
representing each of the possible items to set.  If that CGI variables exists
it uses the information to set the item.  If it does not exist it makes no
change to that item.
*/

MailSetUserInfo ()

{
   static struct VmsItemLongStruct   UserSetItem [] = {
      { 0, 0, 0, 0 },
      { 0, MAIL$_USER_USERNAME, MailUserName, 0 },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   int  status;
   unsigned short  NewMessages;
   char  *cptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MailSetUserInfo()\n");

   UserContext = 0;

   status = mail$user_begin (&UserContext, &NullItem, &NullItem);
   if (Debug) fprintf (stdout, "mail$user_begin() %%X%08.08X\n", status);

   /* status from mail$user_begin() */
   if (VMSnok (status)) return (status);

   UserSetItem[1].buf_len = MailUserNameLength;

   if ((cptr = CgiLibVarNull ("WWW_FORM_UAP")) != NULL)
   {
      memset (&UserSetItem[0], 0, sizeof(UserSetItem[0]));
      if (cptr[0] == '1')
         UserSetItem[0].item = MAIL$_USER_SET_AUTO_PURGE;
      else
         UserSetItem[0].item = MAIL$_USER_SET_NO_AUTO_PURGE;
      status = mail$user_set_info (&UserContext,
                                   &UserSetItem,
                                   &NullItem);
      if (Debug) fprintf (stdout, "mail$user_set_info() %%X%08.08X\n", status);
      if (VMSnok (status))
         ResponseErrorMail (FI_LI, status, lang_ErrSetUserAutoPurge);
   }

   if (VMSok (status) &&
       (cptr = CgiLibVarNull ("WWW_FORM_UCP")) != NULL)
   {
      memset (&UserSetItem[0], 0, sizeof(UserSetItem[0]));
      if (cptr[0] == '1')
         UserSetItem[0].item = MAIL$_USER_SET_CC_PROMPT;
      else
         UserSetItem[0].item = MAIL$_USER_SET_NO_CC_PROMPT;
      status = mail$user_set_info (&UserContext,
                                   &UserSetItem,
                                   &NullItem);
      if (Debug) fprintf (stdout, "mail$user_set_info() %%X%08.08X\n", status);
      if (VMSnok (status))
         ResponseErrorMail (FI_LI, status, lang_ErrSetUserCcPrompt);
   }

   if (VMSok (status) &&
       (cptr = CgiLibVarNull ("WWW_FORM_UCF")) != NULL)
   {
      memset (&UserSetItem[0], 0, sizeof(UserSetItem[0]));
      if (cptr[0] == '1')
         UserSetItem[0].item = MAIL$_USER_SET_COPY_FORWARD;
      else
         UserSetItem[0].item = MAIL$_USER_SET_NO_COPY_FORWARD;
      status = mail$user_set_info (&UserContext,
                                   &UserSetItem,
                                   &NullItem);
      if (Debug) fprintf (stdout, "mail$user_set_info() %%X%08.08X\n", status);
      if (VMSnok (status))
         ResponseErrorMail (FI_LI, status, lang_ErrSetUserCopyForward);
   }

   if (VMSok (status) &&
       (cptr = CgiLibVarNull ("WWW_FORM_UCR")) != NULL)
   {
      memset (&UserSetItem[0], 0, sizeof(UserSetItem[0]));
      if (cptr[0] == '1')
         UserSetItem[0].item = MAIL$_USER_SET_COPY_REPLY;
      else
         UserSetItem[0].item = MAIL$_USER_SET_NO_COPY_REPLY;
      status = mail$user_set_info (&UserContext,
                                   &UserSetItem,
                                   &NullItem);
      if (Debug) fprintf (stdout, "mail$user_set_info() %%X%08.08X\n", status);
      if (VMSnok (status))
         ResponseErrorMail (FI_LI, status, lang_ErrSetUserCopyReply);
   }

   if (VMSok (status) &&
       (cptr = CgiLibVarNull ("WWW_FORM_UCS")) != NULL)
   {
      memset (&UserSetItem[0], 0, sizeof(UserSetItem[0]));
      if (cptr[0] == '1')
         UserSetItem[0].item = MAIL$_USER_SET_COPY_SEND;
      else
         UserSetItem[0].item = MAIL$_USER_SET_NO_COPY_SEND;
      status = mail$user_set_info (&UserContext,
                                   &UserSetItem,
                                   &NullItem);
      if (Debug) fprintf (stdout, "mail$user_set_info() %%X%08.08X\n", status);
      if (VMSnok (status))
         ResponseErrorMail (FI_LI, status, lang_ErrSetUserCopySend);
   }

   if (VMSok (status) &&
       (cptr = CgiLibVarNull ("WWW_FORM_UED")) != NULL)
   {
      memset (&UserSetItem[0], 0, sizeof(UserSetItem[0]));
      if (cptr[0])
      {
         UserSetItem[0].item = MAIL$_USER_SET_EDITOR;
         UserSetItem[0].buf_addr = cptr;
         UserSetItem[0].buf_len = strlen(cptr);
      }
      else
         UserSetItem[0].item = MAIL$_USER_SET_NO_EDITOR;
      status = mail$user_set_info (&UserContext,
                                   &UserSetItem,
                                   &NullItem);
      if (Debug) fprintf (stdout, "mail$user_set_info() %%X%08.08X\n", status);
      if (VMSnok (status))
         ResponseErrorMail (FI_LI, status, lang_ErrSetUserEditor);
   }

   if (VMSok (status) &&
       (cptr = CgiLibVarNull ("WWW_FORM_UFW")) != NULL)
   {
      memset (&UserSetItem[0], 0, sizeof(UserSetItem[0]));
      if (cptr[0])
      {
         UserSetItem[0].item = MAIL$_USER_SET_FORWARDING;
         UserSetItem[0].buf_addr = cptr;
         UserSetItem[0].buf_len = strlen(cptr);
      }
      else
         UserSetItem[0].item = MAIL$_USER_SET_NO_FORWARDING;
      status = mail$user_set_info (&UserContext,
                                   &UserSetItem,
                                   &NullItem);
      if (Debug) fprintf (stdout, "mail$user_set_info() %%X%08.08X\n", status);
      if (VMSnok (status))
         ResponseErrorMail (FI_LI, status, lang_ErrSetUserForwarding);
   }

   if (VMSok (status) &&
       (cptr = CgiLibVarNull ("WWW_FORM_UNM")) != NULL)
   {
      memset (&UserSetItem[0], 0, sizeof(UserSetItem[0]));
      if (isdigit(cptr[0]))
      {
         NewMessages = atoi(cptr);
         if (Debug) fprintf (stdout, "NewMessages: %d\n", NewMessages);

         UserSetItem[0].item = MAIL$_USER_SET_NEW_MESSAGES;
         UserSetItem[0].buf_addr = &NewMessages;
         UserSetItem[0].buf_len = sizeof(NewMessages);

         status = mail$user_set_info (&UserContext,
                                      &UserSetItem,
                                      &NullItem);
         if (Debug)
            fprintf (stdout, "mail$user_set_info() %%X%08.08X\n", status);
         if (VMSnok (status))
            ResponseErrorMail (FI_LI, status, lang_ErrSetUserNewMessages);
      }
   }

   if (VMSok (status) &&
       (cptr = CgiLibVarNull ("WWW_FORM_UPN")) != NULL)
   {
      memset (&UserSetItem[0], 0, sizeof(UserSetItem[0]));
      if (cptr[0])
      {
         UserSetItem[0].item = MAIL$_USER_SET_PERSONAL_NAME;
         UserSetItem[0].buf_addr = cptr;
         UserSetItem[0].buf_len = strlen(cptr);
      }
      else
         UserSetItem[0].item = MAIL$_USER_SET_NO_PERSONAL_NAME;
      status = mail$user_set_info (&UserContext,
                                   &UserSetItem,
                                   &NullItem);
      if (Debug) fprintf (stdout, "mail$user_set_info() %%X%08.08X\n", status);
      if (VMSnok (status))
         ResponseErrorMail (FI_LI, status, lang_ErrSetUserPersonalName);
   }

   if (VmsVersion >= 70 &&
       VMSok (status) &&
       (cptr = CgiLibVarNull ("WWW_FORM_USF")) != NULL)
   {
      memset (&UserSetItem[0], 0, sizeof(UserSetItem[0]));
      if (cptr[0])
      {
         UserSetItem[0].item = MAIL$_USER_SET_SIGFILE;
         UserSetItem[0].buf_addr = cptr;
         UserSetItem[0].buf_len = strlen(cptr);
      }
      else
         UserSetItem[0].item = MAIL$_USER_SET_NO_SIGFILE;
      status = mail$user_set_info (&UserContext,
                                   &UserSetItem,
                                   &NullItem);
      if (Debug) fprintf (stdout, "mail$user_set_info() %%X%08.08X\n", status);
      if (VMSnok (status))
         ResponseErrorMail (FI_LI, status, lang_ErrSetUserSigFile);
   }

   if (VMSok (status) &&
       (cptr = CgiLibVarNull ("WWW_FORM_UPQ")) != NULL)
   {
      memset (&UserSetItem[0], 0, sizeof(UserSetItem[0]));
      if (cptr[0])
      {
         UserSetItem[0].item = MAIL$_USER_SET_QUEUE;
         UserSetItem[0].buf_addr = cptr;      
         UserSetItem[0].buf_len = strlen(cptr);
      }
      else
         UserSetItem[0].item = MAIL$_USER_SET_NO_QUEUE;
      status = mail$user_set_info (&UserContext,
                                   &UserSetItem,
                                   &NullItem);
      if (Debug) fprintf (stdout, "mail$user_set_info() %%X%08.08X\n", status);
      if (VMSnok (status))
         ResponseErrorMail (FI_LI, status, lang_ErrSetUserQueue);
   }

   if (VMSok (status) &&
       (cptr = CgiLibVarNull ("WWW_FORM_UPF")) != NULL)
   {
      memset (&UserSetItem[0], 0, sizeof(UserSetItem[0]));
      if (cptr[0])
      {
         UserSetItem[0].item = MAIL$_USER_SET_FORM;
         UserSetItem[0].buf_addr = cptr;
         UserSetItem[0].buf_len = strlen(cptr);
      }
      else
         UserSetItem[0].item = MAIL$_USER_SET_NO_FORM;
      status = mail$user_set_info (&UserContext,
                                   &UserSetItem,
                                   &NullItem);
      if (Debug) fprintf (stdout, "mail$user_set_info() %%X%08.08X\n", status);
      if (VMSnok (status))
         ResponseErrorMail (FI_LI, status, lang_ErrSetUserForm);
   }

   mail$user_end (&UserContext, &NullItem, &NullItem);

   return (status);
}

/*****************************************************************************/
/*
If messages are moved to or from, or deleted from the NEWMAIL folder update the
count of new messages.
*/

MailSetUserNewMessages (int DeltaValue)

{
   static short  NewMessages;

   static struct VmsItemLongStruct   UserGetItem [] = {
      { 2, MAIL$_USER_NEW_MESSAGES, &NewMessages, 0 },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   static struct VmsItemLongStruct   UserSetItem [] = {
      { 2, MAIL$_USER_SET_NEW_MESSAGES, &NewMessages, 0 },
      { 0, MAIL$_USER_USERNAME, MailUserName, 0 },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   static struct VmsItemLongStruct   UserNameItem [] = {
      { 0, MAIL$_USER_USERNAME, MailUserName, &MailUserNameLength },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MailSetUserNewMessages()\n");

   UserContext = 0;

   status = mail$user_begin (&UserContext, &NullItem, &NullItem);
   if (Debug) fprintf (stdout, "mail$user_begin() %%X%08.08X\n", status);

   /* status from mail$user_begin() */
   if (VMSnok (status)) return (status);

   UserNameItem[0].buf_len = MailUserNameLength;

   status = mail$user_get_info (&UserContext,
                                &UserNameItem,
                                &UserGetItem);
   if (Debug) fprintf (stdout, "mail$user_get_info() %%X%08.08X\n", status);

   if (VMSok (status))
   {
      if (Debug)
         fprintf (stdout, "NewMessages: %d (%d) %d\n",
                  NewMessages, DeltaValue, NewMessages+DeltaValue);
      NewMessages += DeltaValue;
      if (NewMessages < 0) NewMessages = 0;

      UserSetItem[1].buf_len = MailUserNameLength;

      status = mail$user_set_info (&UserContext,
                                   &UserSetItem,
                                   &NullItem);
      if (Debug) fprintf (stdout, "mail$user_set_info() %%X%08.08X\n", status);
   }

   mail$user_end (&UserContext, &NullItem, &NullItem);

   return (status);
}

/*****************************************************************************/
/*
This function creates the user profile entry specified by the global storage
'MailUserName'.  As this is probably a "bogus" entry forwarding is set to some
"real" mail username. 
*/

MailCreateUserEntry ()

{
   static struct VmsItemLongStruct   UserSetItem [] = {
      { 0, MAIL$_USER_SET_FORWARDING, 0, 0 },
      { 0, MAIL$_USER_USERNAME, MailUserName, 0 },
      { 0, MAIL$_USER_CREATE_IF, 0, 0 },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   int  status;
   char  *cptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MailCreateUserEntry()\n");

   UserContext = 0;

   status = mail$user_begin (&UserContext, &NullItem, &NullItem);
   if (Debug) fprintf (stdout, "mail$user_begin() %%X%08.08X\n", status);

   /* status from mail$user_begin() */
   if (VMSnok (status)) return (status);

   UserSetItem[1].buf_len = MailUserNameLength;

   if ((cptr = CgiLibVarNull ("WWW_FORM_UFW")) != NULL)
   {
      if (cptr[0])
      {
         UserSetItem[0].buf_addr = cptr;
         UserSetItem[0].buf_len = strlen(cptr);

         status = mail$user_set_info (&UserContext,
                                      &UserSetItem,
                                      &NullItem);
         if (Debug)
            fprintf (stdout, "mail$user_set_info() %%X%08.08X\n", status);
         if (VMSnok (status))
            ResponseErrorMail (FI_LI, status, lang_ErrUserCreate);
      }
      else
      {
         CgiLibResponseError (FI_LI, 0, lang_ErrUserCreateForward);
         status = STS$K_ERROR;
      }
   }

   mail$user_end (&UserContext, &NullItem, &NullItem);

   return (status);
}

/*****************************************************************************/
/*
This function deletes the user profile entry specified by the global storage
'MailUserName'. 
*/

MailDeleteUserEntry ()

{
   static struct VmsItemLongStruct   UserDeleteItem [] = {
      { 0, MAIL$_USER_USERNAME, MailUserName, 0 },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MailDeleteUserEntry()\n");

   UserContext = 0;

   status = mail$user_begin (&UserContext, &NullItem, &NullItem);
   if (Debug) fprintf (stdout, "mail$user_begin() %%X%08.08X\n", status);

   /* status from mail$user_begin() */
   if (VMSnok (status)) return (status);

   UserDeleteItem[0].buf_len = MailUserNameLength;

   status = mail$user_delete_info (&UserContext,
                                   &UserDeleteItem,
                                   &NullItem);
   if (Debug) fprintf (stdout, "mail$user_delete_info() %%X%08.08X\n", status);
   if (VMSnok (status))
      ResponseErrorMail (FI_LI, status, lang_ErrUserDelete);

   mail$user_end (&UserContext, &NullItem, &NullItem);

   return (status);
}

/*****************************************************************************/
/*
Generate a linked list of user names from the VMS mail profile file.
*/

int GenerateUserList ()

{
   static unsigned long  UserForwardingLength,
                         UserReturnNameLength;
   static char  UserForwarding [256],
                UserReturnName [256];

   static struct VmsItemLongStruct   UserDetailsItem [] = {
      { 255, MAIL$_USER_FORWARDING, UserForwarding, &UserForwardingLength },
      { 255, MAIL$_USER_RETURN_USERNAME,
        UserReturnName, &UserReturnNameLength },
      { 0,0,0,0 }
   };

   static struct VmsItemLongStruct   UserFirstItem [] = {
      { 0, MAIL$_USER_FIRST, 0, 0 },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   static struct VmsItemLongStruct   UserNextItem [] = {
      { 0, MAIL$_USER_NEXT, 0, 0 },
      { 0, MAIL$_NOSIGNAL, 0, 0 },
      { 0,0,0,0 }
   };

   int  status,
        CallocSize;
   struct UserListStruct  *ulptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "GenerateUserList()\n");

   if (UserListHeadPtr != NULL)
   {
      /* empty any current list of users */
      struct UserListStruct  *ueptr;
      ulptr = UserListHeadPtr;
      while (ulptr != NULL)
      {
         ueptr = ulptr;
         ulptr = ulptr->NextPtr;
         free (ueptr);
      }
      UserListHeadPtr = UserListTailPtr = NULL;
      UserListCount = 0;
   }

   UserContext = 0;

   status = mail$user_begin (&UserContext, &NullItem, &NullItem);
   if (Debug) fprintf (stdout, "mail$user_begin() %%X%08.08X\n", status);

   /* status from mail$user_begin() */
   if (VMSnok (status)) return (status);

   status = mail$user_get_info (&UserContext,
                                &UserFirstItem,
                                &UserDetailsItem);
   if (Debug) fprintf (stdout, "mail$user_get_info() %%X%08.08X\n", status);

   while (VMSok (status))
   {
      UserReturnName[UserReturnNameLength] = '\0';
      if (Debug)
         fprintf (stdout, "%d |%s|\n",
                  UserReturnNameLength, UserReturnName);

      if (isprint(UserReturnName[0]) &&
          decc$match_wild (UserReturnName, MailUserName))
      {
         CallocSize = sizeof(struct UserListStruct) + 
                      UserReturnNameLength + 1;
         ulptr = calloc (1, CallocSize);
         if (ulptr == NULL) exit (vaxc$errno);
         ulptr->NextPtr = NULL;
         ulptr->UserNamePtr = (char*)&ulptr->Data;
         strcpy (ulptr->UserNamePtr, UserReturnName);
         if (UserListHeadPtr == NULL)
            UserListHeadPtr = UserListTailPtr = ulptr;
         else
         {
            UserListTailPtr->NextPtr = ulptr;
            UserListTailPtr = ulptr;
         }
         UserListCount++;
      }

      status = mail$user_get_info (&UserContext,
                                   &UserNextItem,
                                   &UserDetailsItem);
      if (Debug) fprintf (stdout, "mail$user_get_info() %%X%08.08X\n", status);
   }

   if (status == MAIL$_NOMOREREC) status = SS$_NORMAL;

   mail$user_end (&UserContext, &NullItem, &NullItem);

   return (status);
}

/*****************************************************************************/
/*
Filter some callable MAIL status to generate specific messages.
*/

ResponseErrorMail
(
char *SrcFileName,
int SrcLineNumber,
int StatusValue,
char *Text
)
{
   char  String [1024];

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "ResponseErrorMail() %%X%08.08X (%d)\n",
               StatusValue, StatusValue);

   /* cannot get this to work for the life of me!!! */
   /** if (StatusValue == MAIL$_NOMOREMSG) **/
   if (StatusValue - MAIL$_NOMOREMSG == 0)
      strcpy (String, lang_ErrMailMessageNotFound);
   else
   if (StatusValue == MAIL$_NOSUCHUSR)
   {
      if (PostMasterAuthenticated)
      {
         sprintf (String,
"%s \'%s\'.\n\
<P><FORM ACTION=\"%s\">\n\
<FONT SIZE=+1 COLOR=\"#ff0000\"><B><U>%s</U></B></FONT>\n\
<BR><B>%s:</B>&nbsp;&nbsp;<INPUT TYPE=text SIZE=60 NAME=ufw>\n\
<BR><INPUT TYPE=submit NAME=act VALUE=\"%s\">\n\
</FORM>\n",
            lang_ErrMailNoSuchUser, MailUserName,
            ActionPathInfo, lang_PostMaster, lang_ProForwarding,
            lang_BtnCreateUserProfile);
         CgiLibResponseError (SrcFileName, SrcLineNumber, 0, String); 
      }
      else
      {
         sprintf (String, "%s \'%s\'.", lang_ErrMailNoSuchUser, MailUserName);
         CgiLibResponseError (SrcFileName, SrcLineNumber, 0, String); 
      }
      return;
   }
   else
   if (StatusValue == MAIL$_NOTEXIST)
   {
      sprintf (String, "%s \'%s\'", lang_ErrMailNoSuchFolder, MailFolderName);
      CgiLibResponseError (SrcFileName, SrcLineNumber, 0, String); 

      if (!FolderListCount) return (SS$_NORMAL);

      fprintf (stdout, "<P><FORM ACTION=\"%s\">\n", ActionPathInfo);

      FolderListHtmlSelect ();

      fprintf (stdout,
"<INPUT TYPE=submit NAME=act VALUE=\"%s\">&nbsp;\n\
<INPUT TYPE=reset VALUE=\"%s\">\n\
</FORM>\n",
         lang_BtnOpen, lang_BtnReset);

      return;
   }
   else
   if (StatusValue == SS$_POWERFAIL)
   {
      /* this is a bogus status, to indicate a message disparity */
      CgiLibResponseError (SrcFileName, SrcLineNumber, 0,
                           lang_ErrCurrentMessageCount); 
      return;
   }
   else
   if (StatusValue == MAIL$_ILLPERNAM)
      strcpy (String, lang_ErrMailIllegalPersonal);
   else
   if (StatusValue == MAIL$_OPENIN)
      sprintf (String, "%s \'%s\'.", lang_ErrMailOpening, MailFileName);
   else
      String[0] = '\0';

   if (String[0])
      CgiLibResponseError (SrcFileName, SrcLineNumber, 0, String); 
   else
      CgiLibResponseError (SrcFileName, SrcLineNumber, StatusValue, Text);
   return;
}

/*****************************************************************************/
/*
String the MX%, SMTP% (etc.) and the PMDF idiosyncracy IN:: (as reported by
Carl Karcher) from mail addresses.
*/

StripTransport
(
char *String,
int *StringLengthPtr
)
{
   char  ch;
   char  *cptr, *sptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "StripTransport() |%s|", String);

   cptr = sptr = String;
   while (*cptr)
   {
      while (*cptr)
      {
         if (cptr[0] == '%' && (cptr[1] == '\"' || cptr[1] == '\'')) break;
         if (toupper(cptr[0]) == 'I' && toupper(cptr[1]) == 'N' &&
             cptr[2] == ':' && cptr[3] == ':' &&
             (cptr[4] == '\"' || cptr[4] == '\'')) break;
        *sptr++ = *cptr++;
      }
      if (!*cptr) break;
      if (*cptr == '%')
         cptr++;
      else
         cptr += 4;
      while (sptr > String && !isspace(*sptr) && *sptr != ',') sptr--;
      if (*sptr == ',') sptr++;
      ch = *cptr++;
      while (*cptr && *cptr != ch)
      {
         if (*cptr == '\\') *sptr++ = *cptr++;
         if (*cptr) *sptr++ = *cptr++;
      }
      if (*cptr) cptr++;
   }
   *sptr = '\0';
   *StringLengthPtr = sptr - String;

   if (Debug) fprintf (stdout, "|%s|\n", String);
}

/*****************************************************************************/

