/*****************************************************************************/
/*
                                 config.c

Loads global (YAHMAIL$CONFIG) and private config ([MAIL]YAHMAIL.CONF) files.


VERSION HISTORY 
---------------
30-JAN-2004  MGD  global configuration unbundled from YAHMAIL.C,
                  personal setup configuration
*/

/*****************************************************************************/

#ifndef __VAX
#   pragma nomember_alignment
#endif

/* standard C header files */
#include <ctype.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* VMS related header files */
#include <descrip.h>
#include <rmsdef.h>
#include <ssdef.h>
#include <stsdef.h>

/* application specific */
#include "yahmail.h"
#include "externs.h"

#define FI_LI __FILE__, __LINE__

/******************/
/* global storage */
/******************/

int  ConfigTextLength,
     PrivateTextLength;

char  *ConfigSpamPtr,
      *ConfigTextPtr,
      *PrivateTextPtr;

/*****************************************************************************/
/*
Read the contents of the configuration file into memory.  The process the text
from the file.  Keyword delimited elements are parsed from the file.  These are
null-terminated and used by being pointed at so the file's contents cannot be
disposed of before reprocessing a new request (for CGIplus).  The [private]
and/or [public] directives MUST follow any others.
*/

GlobalConfigFileLoad ()

{
   boolean  AuthenticatedNameWildcard,
            ConfigCheck;
   int  cnt,
        status,
        ConfigTextSize;
   char  *bptr, *cptr, *sptr, *tptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "GlobalConfigFileLoad()\n");

   if (getenv("YAHMAIL$CHECK") != NULL)
   {
      CgiLibResponseHeader (200, "text/html");
      fputs (lang_CurrentlyUnavailable, stdout);
      fputs ("<!--\n", stdout);
      ConfigCheck = true;
   }
   else
   if (getenv("YAHMAIL$DISABLE") != NULL)
   {
      CgiLibResponseHeader (200, "text/html");
      fputs (lang_CurrentlyUnavailable, stdout);
      exit (SS$_NORMAL);
   }
   else
      ConfigCheck = false;

   BrowseWindow = DEFAULT_BROWSE_WINDOW;
   ConfigBodyTagPtr = DEFAULT_BODY_TAG;
   MakeAnchors = DEFAULT_ANCHORS;
   PublicConfigSubjPtr = DEFAULT_PUBLIC_CONFIG_SUBJ;
   PublicConfigSubjLength = sizeof(DEFAULT_PUBLIC_CONFIG_SUBJ)-1;
   ConfigCreateFooterPtr = ConfigFolderHeaderPtr =
      ConfigFolderFooterPtr = ConfigAddrListFooterPtr =
      ConfigReadFooterPtr = ConfigAddrListPtr =
      ConfigVMSmailFooterPtr = PostMasterWarningPtr = "";
   ConfigCharsetPtr =  ConfigNewMailPtr = ConfigSmtpTransportPtr =
      ConfigPublicHeaderPtr = ConfigPublicFooterPtr = NULL;
   HtmlEscapedMailFolderName[0] =
      PublicFileName[0] = PublicFolderName[0] = PublicUserName[0] =
      MailFileName[0] = MailFolderName[0] = MailUserName[0] = '\0';
   ConfigPrivate = ConfigPublic = ConfigPublicBrowseFolders =
      PostMasterAuthenticated = false;

   /*************************/
   /* read file into memory */
   /*************************/

   if (ConfigTextPtr != NULL) free (ConfigTextPtr);

   status = ReadFileIntoMemory (CliConfigFilePtr,
                                &ConfigTextPtr, &ConfigTextLength);
   if (VMSnok (status))
   {
      if (ConfigCheck)
      {
         fprintf (stdout, "error opening %s: %%X%08.08X\n-->\n",
                  CliConfigFilePtr, status);
         exit (SS$_NORMAL);
      }
      return;
   }

   /********************/
   /* process contents */
   /********************/

   cptr = ConfigTextPtr;
   while (*cptr)
   {
      if (strsame (cptr, "[public]", 8) ||
          strsame (cptr, "[private]", 9))
         break;

      if (strsame (cptr, bptr = "[anchor]", cnt = 8) ||
          strsame (cptr, bptr = "[createfooter]", cnt = 14) ||
          strsame (cptr, bptr = "[body]", cnt = 6) ||
          strsame (cptr, bptr = "[charset]", cnt = 9) ||
          strsame (cptr, bptr = "[direct]", cnt = 8) ||
          strsame (cptr, bptr = "[encoding]", cnt = 10) ||
          strsame (cptr, bptr = "[folderfooter]", cnt = 14) ||
          strsame (cptr, bptr = "[folderheader]", cnt = 14) ||
          /* backward compatibilitiy with [folderfooter] (v1.3) */
          strsame (cptr, bptr = "[footer]", cnt = 8) ||
          /* backward compatibilitiy with [folderheader] (v1.3) */
          strsame (cptr, bptr = "[header]", cnt = 8) ||
          strsame (cptr, bptr = "[list]", cnt = 6) ||
          strsame (cptr, bptr = "[listfooter]", cnt = 12) ||
          strsame (cptr, bptr = "[vmsmailfooter]", cnt = 15) ||
          strsame (cptr, bptr = "[localtosmtp]", cnt = 13) ||
          strsame (cptr, bptr = "[newmail]", cnt = 9) ||
          strsame (cptr, bptr = "[postmaster]", cnt = 12) ||
          strsame (cptr, bptr = "[publicfooter]", cnt = 14) ||
          strsame (cptr, bptr = "[publicheader]", cnt = 14) ||
          strsame (cptr, bptr = "[readfooter]", cnt = 12) ||
          strsame (cptr, bptr = "[replyuseRFC822]", cnt = 16) ||
          strsame (cptr, bptr = "[smtp]", cnt = 6) ||
          strsame (cptr, bptr = "[SPAM]", cnt = 6) ||
          strsame (cptr, bptr = "[window]", cnt = 8) ||
          strsame (cptr, bptr = "[!yahmail!]", cnt = 11))
      {    
         /* find the start of the next directive or end of text */
         sptr = (cptr += cnt);
         while (*cptr && *(unsigned short*)cptr != '\n[') cptr++;
         if (*cptr) *cptr++ = '\0';

         /* trim leading then trailing white space */
         while (*sptr && isspace(*sptr)) sptr++;
         tptr = sptr;
         while (*sptr) sptr++;
         if (sptr > tptr) sptr--;
         while (sptr > tptr && isspace(*sptr)) sptr--;
         if (sptr > tptr) *++sptr = '\0';         

         /* set the appropriate pointer */
         if (strsame (bptr, "[anchor]", 8))
            MakeAnchors = atoi(tptr);
         else
         if (strsame (bptr, "[body]", 6))
            ConfigBodyTagPtr = tptr;
         else
         if (strsame (bptr, "[charset]", 9))
            ConfigCharsetPtr = tptr;
         else
         if (strsame (bptr, "[createfooter]", 14))
            ConfigCreateFooterPtr = tptr;
         else
         if (strsame (bptr, "[direct]", 8))
         {
            if (tolower(*tptr) == 'e')
            {
               /* "enabled" */
               ConfigDirectActionUI = true ;
            }
            else
               ConfigDirectActionUI = false ;
         }
         else
         if (strsame (bptr, "[encoding]", 10))
            ConfigEncodingPtr = tptr;
         else
         if (strsame (bptr, "[folderfooter]", 14))
            ConfigFolderFooterPtr = tptr;
         else
         if (strsame (bptr, "[folderheader]", 14))
            ConfigFolderHeaderPtr = tptr;
         else
         /* backward compatibilitiy with [folderfooter] (v1.3) */
         if (strsame (bptr, "[footer]", 8))
            ConfigFolderFooterPtr = tptr;
         else
         /* backward compatibilitiy with [folderheader] (v1.3) */
         if (strsame (bptr, "[header]", 8))
            ConfigFolderHeaderPtr = tptr;
         else
         if (strsame (bptr, "[list]", 6))
            ConfigAddrListPtr = tptr;
         else
         if (strsame (bptr, "[listfooter]", 12))
            ConfigAddrListFooterPtr = tptr;
         else
         if (strsame (bptr, "[vmsmailfooter]", 12))
            ConfigVMSmailFooterPtr = tptr;
         else
         if (strsame (bptr, "[localtosmtp]", 13))
            ConfigLocalToSmtpPtr = tptr;
         else
         if (strsame (bptr, "[newmail]", 9))
            ConfigNewMailPtr = tptr;
         else
         if (strsame (bptr, "[postmaster]", 12))
            PostMasterWarningPtr = tptr;
         else
         if (strsame (bptr, "[publicfooter]", 14))
            ConfigPublicFooterPtr = tptr;
         else
         if (strsame (bptr, "[publicheader]", 14))
            ConfigPublicHeaderPtr = tptr;
         else
         if (strsame (bptr, "[readfooter]", 12))
            ConfigReadFooterPtr = tptr;
         else
         if (strsame (bptr, "[replyuseRFC822]", 16))
         {
            if (tolower(*tptr) == 'e')
            {
               /* "enabled" */
               ConfigReplyUseRfc822 = true;
               ConfigReplyParenRfc822 = ConfigReplyQuoteRfc822 = false;
            }
            else
            if (tolower(*tptr) == 'p')
            {
               /* "parentheses" */
               ConfigReplyUseRfc822 = ConfigReplyParenRfc822 = true;
               ConfigReplyQuoteRfc822 = false;
            }
            else
            if (tolower(*tptr) == 'q')
            {
               /* "quote" */
               ConfigReplyUseRfc822 = ConfigReplyQuoteRfc822 = true;
               ConfigReplyParenRfc822 = false;
            }
            else
               ConfigReplyUseRfc822 =
                  ConfigReplyParenRfc822 = 
                  ConfigReplyQuoteRfc822 = false;
         }
         else
         if (strsame (bptr, "[smtp]", 6))
            ConfigSmtpTransportPtr = tptr;
         else
         if (strsame (bptr, "[SPAM]", 6))
            ConfigSpamPtr = tptr;
         else
         if (strsame (bptr, "[window]", 8))
            BrowseWindow = atoi(tptr);
         else
         if (strsame (bptr, "[!yahmail!]", 11))
         {
            PublicConfigSubjPtr = tptr;
            PublicConfigSubjLength = strlen(tptr);
         }
      }
      else
      {
         /* find the start of the next line */
         while (*cptr && *cptr != '\n') cptr++;
         if (*cptr) cptr++;
      }
   }

   if (!ConfigEncodingPtr) ConfigEncodingPtr = CliEncodingPtr;
   if (!ConfigEncodingPtr) ConfigEncodingPtr = "";

   if (Debug)
      fprintf (stdout, "|%s|\n|%s|\n|%s|\n|%s|\n|%s|\n|%s|\n|%s|\n|%s|\n",
               ConfigBodyTagPtr, ConfigFolderHeaderPtr, ConfigFolderFooterPtr,
               ConfigPublicHeaderPtr, ConfigPublicFooterPtr,
               ConfigAddrListPtr, PostMasterWarningPtr, PublicConfigSubjPtr);

   /******************************/
   /* encountered access control */
   /******************************/

   if (CgiRemoteUserPtr[0])
   {
      /**********************/
      /* authenticated user */
      /**********************/

#if !YAHMAIL_PRIVATE
      fprintf (stdout, "%%%s-E-BUILD, build does not support [private]\n",
               Utility);
      exit (STS$K_ERROR | STS$M_INHIB_MSG);
#endif

      /* find the appropriate section */
      while (!strsame (cptr, "[private]", 9))
      {
         /** if (Debug) fprintf (stdout, "cptr |%s|\n", cptr); **/
         while (*cptr && *(unsigned short*)cptr != '\n[') cptr++;
         if (!*cptr) break;
         cptr++;
      }
      /** if (Debug) fprintf (stdout, "cptr |%s|\n", cptr); **/

      if (ConfigCheck)
         fprintf (stdout, "[private] %s%s%s\n",
                  CgiRemoteUserPtr,
                  CgiAuthRealmPtr[0] ? "\\" : "\\?", CgiAuthRealmPtr);

      /* authenticated-name\realm\equivalent-VMS-username */

      while (*cptr)
      {
         /* find start of next line */
         while (*cptr && *cptr != '\n') cptr++;
         if (!*cptr) break;
         cptr++;
         /** if (Debug) fprintf (stdout, "|%s|\n", cptr); **/

         if (ConfigCheck)
         {
            for (sptr = cptr; *sptr && *sptr != '\n'; sptr++);
            if (*cptr != '\n')
               fprintf (stdout, "|%*.*s|\n", sptr-cptr, sptr-cptr, cptr);
         }

         while (*cptr && isspace(*cptr)) cptr++;
         /* if comment line */
         if (*cptr == '#' || *cptr == '!') continue;
         /* if end of section break */
         if (*cptr == '[') break;

         /* authenticated name */
         sptr = cptr;
         while (*cptr && !isspace(*cptr) && *cptr != '\\') cptr++;
         if (*cptr) *cptr++ = '\0';
         if (*sptr != '*' && !strsame (sptr, CgiRemoteUserPtr, -1))
         {
            if (ConfigCheck) fprintf (stdout, "reject: authenticated name\n");
            continue;
         }
         if (*sptr == '*')
            AuthenticatedNameWildcard = true;
         else
            AuthenticatedNameWildcard = false;

         /* authentication realm */
         sptr = cptr;
         while (*cptr && !isspace(*cptr) && *cptr != '\\') cptr++;
         if (*cptr) *cptr++ = '\0';
         if (*sptr != '*' && !strsame (sptr, CgiAuthRealmPtr, -1))
         {
            if (ConfigCheck) fprintf (stdout, "reject: authentication realm\n");
            continue;
         }

         /* get the resultant VMS user name */
         if (*cptr == '*')
         {
            strzcpy (MailUserName, CgiRemoteUserPtr, SizeOfMailUserName);
            for (sptr = MailUserName; *sptr; sptr++) *sptr = toupper(*sptr);
         }
         else
         {
            sptr = MailUserName;
            while (*cptr && !isspace(*cptr)) *sptr++ = toupper(*cptr++);
            *sptr = '\0';
         }
         if (Debug) fprintf (stdout, "MailUserName |%s|\n", MailUserName);
         if (!isalnum(MailUserName[0]))
         {
            if (ConfigCheck) fprintf (stdout, "reject: VMS username\n");
            continue;
         }

#if YAHMAIL_POSTMASTER
         if (strsame (MailUserName, "POSTMASTER", -1))
         {
            /* shouldn't/mistake map all users to be POSTMASTERs! */
            if (AuthenticatedNameWildcard)
            {
               if (ConfigCheck)
                  fprintf (stdout, "reject: wildcard POSTMASTER\n");
               continue;
            }

            if (PostMasterAuthenticated = CliPostMasterEnabled)
            {
               strzcpy (MailUserName, CgiRemoteUserPtr, SizeOfMailUserName);
               for (sptr = MailUserName; *sptr; sptr++) *sptr = toupper(*sptr);
               if (Debug) fprintf (stdout, "|%s|\n", MailUserName);
               PostMasterWarningPtr = lang_PostMasterWarning;
            }
            else
            {
               if (ConfigCheck)
                  fprintf (stdout, "reject: POSTMASTER not enabled\n");
               continue;
            }
         }
#endif /* YAHMAIL_POSTMASTER */

         /* access is OK */
         ConfigPrivate = true;
         if (ConfigCheck)
         {
            fprintf (stdout, "allow: %s%s\n-->\n",
                MailUserName, PostMasterAuthenticated ? " as POSTMASTER" : "");
            exit (SS$_NORMAL);
         }
         return;
      }
   }
   else
   {
      /**********/
      /* public */
      /**********/

      /* find the appropriate section */
      while (!strsame (cptr, "[public]", 8))
      {
         /** if (Debug) fprintf (stdout, "cptr |%s|\n", cptr); **/
         while (*cptr && *(unsigned short*)cptr != '\n[') cptr++;
         if (!*cptr) break;
         cptr++;
      }
      /** if (Debug) fprintf (stdout, "cptr |%s|\n", cptr); **/

      if (ConfigCheck) fprintf (stdout, "[public] %s\n", PathPart[0]);

      /* first-element-of-path\VMS-file-path\optional-folder-name */

      if (Debug) fprintf (stdout, "PathPart[0] |%s|\n", PathPart[0]);

      while (*cptr)
      {
         /* find start of next line */
         while (*cptr && *cptr != '\n') cptr++;
         if (!*cptr) break;
         cptr++;
         /** if (Debug) fprintf (stdout, "|%s|\n", cptr); **/

         if (ConfigCheck)
         {
            for (sptr = cptr; *sptr && *sptr != '\n'; sptr++);
            if (*cptr != '\n')
               fprintf (stdout, "|%*.*s|\n", sptr-cptr, sptr-cptr, cptr);
         }

         while (*cptr && isspace(*cptr)) cptr++;
         /* if comment line */
         if (*cptr == '#' || *cptr == '!') continue;
         /* if end of section break */
         if (*cptr == '[') break;

         /* first element of path */
         sptr = cptr;
         while (*cptr && !isspace(*cptr) && *cptr != '\\') cptr++;
         if (*cptr) *cptr++ = '\0';
         if (*sptr != '*' && !strsame (sptr, PathPart[0], -1))
         {
            if (ConfigCheck) fprintf (stdout, "reject: path\n");
            continue;
         }
         strzcpy (PublicUserName, PathPart[0], SizeOfPublicUserName);

         /* mail file name */
         sptr = PublicFileName;
         while (*cptr && !isspace(*cptr) && *cptr != '\\') *sptr++ = *cptr++;
         *sptr = '\0';
         if (*cptr) *cptr++ = '\0';
         if (Debug) fprintf (stdout, "PublicFileName |%s|\n", PublicFileName);
         /* a configuration file name is mandatory */
         if (!PublicFileName[0])
         {
            if (ConfigCheck) fprintf (stdout, "reject: MAIL file name\n");
            continue;
         }

         /* restrict folder to this specified name */
         sptr = PublicFolderName;
         while (*cptr && !isspace(*cptr) && *cptr != '\\') *sptr++ = *cptr++;
         *sptr = '\0';
         if (*cptr) *cptr++ = '\0';
         if (Debug)
            fprintf (stdout, "PublicFolderName |%s|\n", PublicFolderName);

         /* if no specific folder is configured then allowed to browse */
         if (PublicFolderName[0] == '*') PublicFolderName[0] = '\0';
         if (!PublicFolderName[0]) ConfigPublicBrowseFolders = true;

         /* access is OK */
         ConfigPublic = true;

         if (ConfigCheck)
         {
            fprintf (stdout, "allow: %s %s\n-->\n",
                     PublicFileName, PublicFolderName);
            exit (SS$_NORMAL);
         }
         return;
      }
   }

   if (ConfigCheck)
   {
      fprintf (stdout, "end of configuration!\n-->\n");
      exit (SS$_NORMAL);
   }
}

/*****************************************************************************/
/*
Get and process CGI variables controlling private browse setup.  This parallels
the setup done by PersonalSetupFileLoad().
*/

PersonalSetupFromCgi ()

{
   static char  TextAreaRows [16];

   char  *cptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "PersonalSetupFromCgi()\n");

   if (!(PersonalSetupShow || PersonalSetupHide))
      PersonalSetupShow = atoi(CgiLibVar ("WWW_FORM_SPS"));

   /* standard "mail" folder buttons (default to bottom) */
   cptr = CgiLibVar ("WWW_FORM_BWM");
   if (cptr[0] != '0' && cptr[0] != '1' && cptr[0] != '2') cptr = "2";
   CgiFormBwmPtr = cptr;

   /* "user" (all) folder buttons (default to none) */
   cptr = CgiLibVar ("WWW_FORM_BWU");
   if (cptr[0] != '0' && cptr[0] != '1' && cptr[0] != '2') cptr = "0";
   CgiFormBwuPtr = cptr;

   /* folder selector and open/move/copy buttons (default to bottom) */
   cptr = CgiLibVar ("WWW_FORM_BWF");
   if (cptr[0] != '0' && cptr[0] != '1' && cptr[0] != '2') cptr = "2";
   CgiFormBwfPtr = cptr;

   /* create-send button (default to bottom) */
   cptr = CgiLibVar ("WWW_FORM_BWC");
   if (cptr[0] != '0' && cptr[0] != '1' && cptr[0] != '2') cptr = "2";
   CgiFormBwcPtr = cptr;

   /* "pick" fields (default to none) */
   cptr = CgiLibVar ("WWW_FORM_BWP");
   if (cptr[0] != '0' && cptr[0] != '1' && cptr[0] != '2') cptr = "0";
   CgiFormBwpPtr = cptr;

   /* create-send <TEXTAREA> rows (with default) */
   CgiFormTarPtr = CgiLibVar ("WWW_FORM_TAR");
   if (!CgiFormTarPtr[0])
      sprintf (CgiFormTarPtr = TextAreaRows, "%d", CREATESEND_TEXTAREA_ROWS);
   if (Debug) fprintf (stdout, "CgiFormTarPtr |%s|\n", CgiFormTarPtr);

   /* header and body encoding preferences */
   CgiFormEncPtr = CgiLibVar ("WWW_FORM_ENC");

   /* reply-to preference */
   CgiFormRtoPtr = CgiLibVar ("WWW_FORM_RTO");

   /* folder browse window size */
   cptr = CgiLibVar ("WWW_FORM_WINDOW");
   if (!cptr[0]) cptr = CgiLibVar ("WWW_FORM_WIN");
   if (isdigit(*cptr)) BrowseWindow = atoi(cptr);
   if (BrowseWindow <= 0 || BrowseWindow > 999)
      BrowseWindow = DEFAULT_BROWSE_WINDOW;

   /* wrap long lines */
   LongLineWrap = 0;
   cptr = CgiLibVar("WWW_FORM_LLW");
   if (cptr[0]) LongLineWrap = atoi(cptr);
   /* let's keep it within the bounds of reason! */
   if (LongLineWrap && (LongLineWrap < 48 || LongLineWrap > 255))
      LongLineWrap = 72;

   /* check NEWMAIL minutes */
   cptr = CgiLibVar ("WWW_FORM_NEWMAIL");
   if (!cptr[0]) cptr = CgiLibVar ("WWW_FORM_CNM");
   if (isdigit(*cptr)) CheckNewMailMinutes = atoi(cptr);

   /* direct action user interface */
   cptr = CgiLibVar ("WWW_FORM_DUI");
   if (cptr[0] != '0' && cptr[0] != '1') cptr = "";
   CgiFormDuiPtr = cptr;
   if (cptr[0] == '1')
      ConfigDirectActionUI = true;
   else
   if (cptr[0] == '0')
      ConfigDirectActionUI = false;
   /* otherwise don't change the default */

   /* SPAM filtering */
   cptr = CgiLibVar ("WWW_FORM_SPM");
   /* when set greater than 1 the folder browse displays SPAM reasons */
   HideSpam = atoi(cptr);
}

/*****************************************************************************/
/*
Read the contents of the private configuration file into memory.  Process the
directives setting the various setup booleans and other setup data.  This
parallels the setup done by PersonalSetupFromCgi().
*/

PersonalSetupFileLoad ()

{
   static char  TextAreaRows [16];

   int  status;
   char  *cptr, *sptr, *zptr;
   char  FileName [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "PersonalSetupFileLoad()\n");

   /* initialize (most important considering the pointers) */
   CgiFormBwmPtr = "2";
   CgiFormBwuPtr = "0";
   CgiFormBwfPtr = "2";
   CgiFormBwcPtr = "2";
   CgiFormBwpPtr = "0";
   sprintf (CgiFormTarPtr = TextAreaRows, "%d", CREATESEND_TEXTAREA_ROWS);
   BrowseWindow = DEFAULT_BROWSE_WINDOW;
   LongLineWrap = 0;
   CheckNewMailMinutes = 0;
   CgiFormEncPtr = "";
   CgiFormRtoPtr = "";
   CgiFormDuiPtr = "";
   HideSpam = 0;

   if (PrivateTextPtr != NULL) free (PrivateTextPtr);

   sprintf (FileName, "%s%s%s",
            VmsMailUserFullDirectory,
            PERSONAL_CONFIG_FILE_NAME, PERSONAL_CONFIG_FILE_TYPE);

   status = ReadFileIntoMemory (FileName,
                                &PrivateTextPtr, &PrivateTextLength);
   if (VMSnok (status)) return;

   /* plus the terminating null */
   cptr = calloc (1, PrivateTextLength+1);
   memcpy (cptr, PrivateTextPtr, PrivateTextLength+1);

   while (*cptr)
   {
      while (*cptr && isspace(*cptr)) cptr++;
      if (*cptr != '[')
      {
         while (*cptr && *cptr != '\n') cptr++;
         while (*cptr == '\n') cptr++;
         continue;
      }

      sptr = cptr++;
      while (*cptr && *cptr != ']' && *cptr != '\n') cptr++;
      if (*cptr == ']') cptr++;
      while (*cptr && isspace(*cptr) && *cptr != '\n') cptr++;
      for (zptr = cptr; *zptr && *zptr != '\n'; zptr++);
      while (*zptr == '\n') zptr++;

      if (strsame (sptr, "[MailFolderButtons]", 19))
      {
         /* standard "mail" folder buttons (default to bottom) */
         if (tolower(*cptr) == 'n')
            CgiFormBwmPtr = "0";
         else
         if (tolower(*cptr) == 't')
            CgiFormBwmPtr = "1";
         else
         if (tolower(*cptr) == 'b')
            CgiFormBwmPtr = "2";
      }
      else
      if (strsame (sptr, "[UserFolderButtons]", 19))
      {
         /* "user" (all) folder buttons (default to none) */
         if (tolower(*cptr) == 'n')
            CgiFormBwuPtr = "0";
         else
         if (tolower(*cptr) == 't')
            CgiFormBwuPtr = "1";
         else
         if (tolower(*cptr) == 'b')
            CgiFormBwuPtr = "2";
      }
      else
      if (strsame (sptr, "[FolderButtons]", 15))
      {
         /* folder selector and open/move/copy buttons (default to bottom) */
         if (tolower(*cptr) == 't')
            CgiFormBwfPtr = "1";
         else
         if (tolower(*cptr) == 'b')
            CgiFormBwfPtr = "2";
      }
      else
      if (strsame (sptr, "[CreateSendButton]", 18))
      {
         /* create-send button (default to bottom) */
         if (tolower(*cptr) == 't')
            CgiFormBwcPtr = "1";
         else
         if (tolower(*cptr) == 'b')
            CgiFormBwcPtr = "2";
      }
      else
      if (strsame (sptr, "[PickFields]", 12))
      {
         /* "pick" fields (default to none) */
         if (tolower(*cptr) == 'n')
            CgiFormBwpPtr = "0";
         else
         if (tolower(*cptr) == 't')
            CgiFormBwpPtr = "1";
         else
         if (tolower(*cptr) == 'b')
            CgiFormBwpPtr = "2";
      }
      else
      if (strsame (sptr, "[TextAreaRows]", 14))
      {
         /* create-send <TEXTAREA> rows (with default) */
         if (isdigit(*cptr))
         {
            CgiFormTarPtr = cptr;
            while (isdigit(*cptr)) cptr++;
            if (*cptr) *cptr++ = '\0';
         }
         else
            sprintf (CgiFormTarPtr = TextAreaRows, "%d",
                     CREATESEND_TEXTAREA_ROWS);
      }
      else
      if (strsame (sptr, "[BrowseWindow]", 14))
      {
         /* number of messages displayed on each browse page */
         if (isdigit(*cptr)) BrowseWindow = atoi(cptr);
         if (BrowseWindow <= 0 || BrowseWindow > 999)
            BrowseWindow = DEFAULT_BROWSE_WINDOW;
      }
      else
      if (strsame (sptr, "[LongLineWrap]", 14))
      {
         /* header and body encoding preferences */
         if (isdigit(*cptr)) LongLineWrap = atoi(cptr);
      }
      else
      if (strsame (sptr, "[CheckNEWMAIL]", 14))
      {
         /* check NEWMAIL folder minutes */
         if (isdigit(*cptr)) CheckNewMailMinutes = atoi(cptr);
      }
      else
      if (strsame (sptr, "[BodyEncoding]", 14))
      {
         /* header and body encoding preferences */
         CgiFormEncPtr = cptr;
         while (*cptr && !isspace(*cptr)) cptr++;
         if (*cptr) *cptr++ = '\0';
      }
      else
      if (strsame (sptr, "[ReplyTo]", 9))
      {
         /* header and body encoding preferences */
         CgiFormRtoPtr = cptr;
         while (*cptr && !isspace(*cptr)) cptr++;
         if (*cptr) *cptr++ = '\0';
      }
      else
      if (strsame (sptr, "[DirectAction]", 14))
      {
         /* direct action user interface */
         if (tolower(*cptr) == 'e')
         {
            CgiFormDuiPtr = "1";
            ConfigDirectActionUI = true;
         }
         else
         if (tolower(*cptr) == 'd')
         {
            CgiFormDuiPtr = "0";
            ConfigDirectActionUI = false;
         }
      }
      else
      if (strsame (sptr, "[HideSPAM]", 10))
      {
         /* filter known addresses out from SPAM */
         if (tolower(*cptr) == 'e')
            HideSpam = 1;
         else
         if (isdigit(*cptr))
            HideSpam = atoi(cptr);
         if (HideSpam) DoNotDisplaySpam = true;
      }

      cptr = zptr;
   }
}

/*****************************************************************************/
/*
Update the private configuration file by creating a new version, writing the
date/time and current settings, then reading all the existing version's lines
into the new version excluding those beginning with '[' (i.e. the settings).
*/

PersonalSetupFileSave ()
       
{
   static $DESCRIPTOR (DateTimeFaoDsc, "!20%D\0");

   int  status;
   char  DateTimeString [32],
         FileName [256],
         Line [256],
         NewLine [256],
         TempFileName [256];
   char  *cptr, *sptr,
         SpamUserPtr;
   FILE  *FilePtr,
         *NewFilePtr;
   $DESCRIPTOR (DateTimeStringDsc, DateTimeString);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "PersonalSetupFileSave()\n");

   sprintf (FileName, "%s%s%s",
            VmsMailUserFullDirectory,
            PERSONAL_CONFIG_FILE_NAME, PERSONAL_CONFIG_FILE_TYPE);

   sprintf (TempFileName, "%s%s%s",
            VmsMailUserFullDirectory,
            PERSONAL_CONFIG_FILE_NAME, PERSONAL_CONFIG_FILE_TEMP);

   FilePtr = fopen (FileName, "r+", "shr=nil");
   if (!FilePtr)
   {
      PersonalSetupFileCreate ();
      FilePtr = fopen (FileName, "r+", "shr=nil");
   }
   if (!FilePtr)
   {
      status = vaxc$errno;
      CgiLibResponseError (FI_LI, status,
"%s\n\
<P><FORM>\n\
<INPUT TYPE=button VALUE=\"%s\" onClick=\"history.go(-1)\">\n\
</FORM>",
            FileName, lang_BtnGoBack);
      return;
   }

   NewFilePtr = fopen (TempFileName, "w", "shr=nil");
   if (!NewFilePtr)
   {
      status = vaxc$errno;
      CgiLibResponseError (FI_LI, status,
"%s\n\
<P><FORM>\n\
<INPUT TYPE=button VALUE=\"%s\" onClick=\"history.go(-1)\">\n\
</FORM>",
         FileName, lang_BtnGoBack);
      fclose (FilePtr);
      return;
   }

   PersonalSetupFromCgi ();

   sys$fao (&DateTimeFaoDsc, 0, &DateTimeStringDsc, 0);

   fprintf (NewFilePtr,
"# %s\n\
[FolderButtons] %s%s\n\
[MailFolderButtons] %s%s%s\n\
[UserFolderButtons] %s%s%s\n\
[CreateSendButton] %s%s\n\
[PickFields] %s%s%s\n\
[BrowseWindow] %d\n\
[LongLineWrap] %d\n\
[CheckNEWMAIL] %d\n\
[TextAreaRows] %s\n\
[BodyEncoding] %s\n\
[ReplyTo] %s\n\
[DirectAction] %s%s%s\n",
            DateTimeString,
            CgiFormBwfPtr && *CgiFormBwfPtr == '1' ? "top" : "",
            CgiFormBwfPtr && *CgiFormBwfPtr == '2' ? "bottom" : "",
            CgiFormBwmPtr && *CgiFormBwmPtr == '0' ? "none" : "",
            CgiFormBwmPtr && *CgiFormBwmPtr == '1' ? "top" : "",
            CgiFormBwmPtr && *CgiFormBwmPtr == '2' ? "bottom" : "",
            CgiFormBwuPtr && *CgiFormBwuPtr == '0' ? "none" : "",
            CgiFormBwuPtr && *CgiFormBwuPtr == '1' ? "top" : "",
            CgiFormBwuPtr && *CgiFormBwuPtr == '2' ? "bottom" : "",
            CgiFormBwcPtr && *CgiFormBwcPtr == '1' ? "top" : "",
            CgiFormBwcPtr && *CgiFormBwcPtr == '2' ? "bottom" : "",
            CgiFormBwpPtr && *CgiFormBwpPtr == '0' ? "none" : "",
            CgiFormBwpPtr && *CgiFormBwpPtr == '1' ? "top" : "",
            CgiFormBwpPtr && *CgiFormBwpPtr == '2' ? "bottom" : "",
            BrowseWindow,
            LongLineWrap,
            CheckNewMailMinutes,
            CgiFormTarPtr ? CgiFormTarPtr : "",
            CgiFormEncPtr ? CgiFormEncPtr : "",
            CgiFormRtoPtr ? CgiFormRtoPtr : "",
            CgiFormDuiPtr && *CgiFormDuiPtr == '1' ? "enabled" : "",
            CgiFormDuiPtr && *CgiFormDuiPtr == '0' ? "disabled" : "",
            CgiFormDuiPtr && !*CgiFormDuiPtr ? "(default)" : "");

   if (HideSpam > 1)
      fprintf (NewFilePtr, "[HideSPAM] %d\n", HideSpam);
   else
      fprintf (NewFilePtr, "[HideSPAM] %s\n",
               HideSpam ? "enabled" : "disabled");

   while (fgets (Line, sizeof(Line), FilePtr))
   {
      for (cptr = Line; *cptr && isspace(*cptr); cptr++);
      if (!*cptr || *cptr == '#' || *cptr == '[') continue;
      if (fputs (Line, NewFilePtr) == EOF)
      {
         status = vaxc$errno;
         CgiLibResponseError (FI_LI, status,
"%s\n\
<P><FORM>\n\
<INPUT TYPE=button VALUE=\"%s\" onClick=\"history.go(-1)\">\n\
</FORM>",
            FileName, lang_BtnGoBack);

         fclose (FilePtr);
         fclose (NewFilePtr);
         remove (TempFileName);

         return;
      }
   }

   fclose (FilePtr);
   fclose (NewFilePtr);

   rename (TempFileName, FileName);

   /* purge the versions back to a maximum of 3 */
   strcat (FileName, ";-3");
   while (!remove (FileName));

   if (ConfigDirectActionUI)
      returnFromAction(-1);
   else
      CgiLibResponseSuccess (
"%s\n\
<P><FORM>\n\
<INPUT TYPE=button VALUE=\"%s\" onClick=\"history.go(-1)\">&nbsp;\n\
</FORM>",
         lang_PersonalSetupUpdated, lang_BtnGoBack);
}

/*****************************************************************************/
/*
Create the file, add a date/time.  This is called not only by the private
configuration functions but also by the SPAM list update functions.
*/

int PersonalSetupFileCreate ()
{
   static $DESCRIPTOR (DateTimeFaoDsc, "!20%D\0");

   char  DateTimeString [32],
         FileName [256],
         TempFileName [256];
   $DESCRIPTOR (DateTimeStringDsc, DateTimeString);
   FILE  *FilePtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "PersonalSetupFileCreate()\n");

   sprintf (FileName, "%s%s%s",
            VmsMailUserFullDirectory,
            PERSONAL_CONFIG_FILE_NAME, PERSONAL_CONFIG_FILE_TYPE);

   sprintf (TempFileName, "%s%s%s",
            VmsMailUserFullDirectory,
            PERSONAL_CONFIG_FILE_NAME, PERSONAL_CONFIG_FILE_TEMP);

   FilePtr = fopen (TempFileName, "w", "shr=nil", "ctx=rec", "rfm=stmlf");
   if (!FilePtr) return (vaxc$errno);
   sys$fao (&DateTimeFaoDsc, 0, &DateTimeStringDsc, 0);
   fprintf (FilePtr, "# %s\n", DateTimeString);
   fclose (FilePtr);

   rename (TempFileName, FileName);

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Display the contents of a private config file in a <TEXTAREA></TEXTAREA> tag. 
*/

PersonalSetupFileEdit ()
       
{
   int  status,
        TextLength;
   char  FileName [256];
   char  *cptr, *sptr,
         *EscapedTextPtr,
         *TextPtr;
   FILE  *FilePtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "PersonalSetupFileEdit()\n");

   sprintf (FileName, "%s%s%s",
            VmsMailUserFullDirectory,
            PERSONAL_CONFIG_FILE_NAME, PERSONAL_CONFIG_FILE_TYPE);

   status = ReadFileIntoMemory (FileName, &TextPtr, &TextLength);
   if (status == RMS$_FNF)
   {
      PersonalSetupFileCreate ();
      status = ReadFileIntoMemory (FileName, &TextPtr, &TextLength);
   }
   if (VMSnok (status))
   {
      CgiLibResponseError (FI_LI, status,
"%s\n\
<P><FORM>\n\
<INPUT TYPE=button VALUE=\"%s\" onClick=\"window.close()\">\n\
</FORM>",
         FileName, lang_BtnClose);
      return;
   }

   EscapedTextPtr = (char*)CgiLibHtmlEscape (TextPtr, -1, NULL, -1);
   free (TextPtr);

   CgiLibResponseHeader (200, "text/html");
   fprintf (stdout,
"<HTML>\n\
<HEAD>\n\
<META NAME=\"generator\" CONTENT=\"%s\">\n\
<META NAME=\"environment\" CONTENT=\"%s\">\n\
<META NAME=\"language\" CONTENT=\"%s %s\">\n\
<TITLE>yahMAIL ... %s</TITLE>\n\
</HEAD>\n\
<BODY %s>\n\
%s\
<B><FONT SIZE=+1><U>%s</U></FONT></B>\n\
<BR>&nbsp;<FONT SIZE=-1>yahMAIL<SUP>*</SUP></FONT>\n\
<FORM ACTION=\"%s\" METHOD=POST>\n\
<TABLE CELLPADDING=0 CELLSPACING=0 BORDER=0>\n\
<TR><TD>\n\
<TEXTAREA NAME=txt ROWS=16 COLS=48 WRAP=off>\n\
%s\
</TEXTAREA>\n\
</TD></TR>\n\
</TABLE>\n",
      SoftwareID, CgiEnvironmentPtr, lang_Language, lang_LanguageAuthor,
      lang_PersonalSetupFile, ConfigBodyTagPtr,
      PostMasterWarningPtr, lang_PersonalSetupFile,
      ActionPathInfo,
      EscapedTextPtr);

   fprintf (stdout,
"<P><INPUT TYPE=hidden NAME=wha VALUE=\"pcu\">\n\
<INPUT TYPE=submit NAME=act VALUE=\"%s\">&nbsp;\n\
<INPUT TYPE=reset VALUE=\"%s\">\
<I> ... %s</I>&nbsp;\n\
<P><INPUT TYPE=button VALUE=\"%s\" onClick=\"window.close()\">\n\
</FORM>\n\
</BODY>\n\
</HTML>\n",
      lang_BtnUpdate, lang_BtnReset, lang_ResetCaution,
      lang_BtnClose);

   free (EscapedTextPtr);
}

/*****************************************************************************/
/*
Write the contents of the private config sent from PersonalSetupEdit() into
the appropriate file in the user's mail directory.
*/

PersonalSetupFileUpdate ()
       
{
   int  status,
        PostBufferCount;
   char  FileName [256],
         TempFileName [256];
   char  *cptr, *sptr,
         *TextPtr,
         *PostBufferPtr;
   FILE  *FilePtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "PersonalSetupFileUpdate()\n");

   CgiLibReadRequestBody (&PostBufferPtr, &PostBufferCount);
   CgiLibFormRequestBody (PostBufferPtr, PostBufferCount);
   free (PostBufferPtr);
   if (Debug)
      while ((cptr = CgiLibVarNull ("*")) != NULL)
         fprintf (stdout, "|%s|\n", cptr);

   TextPtr = CgiLibVar ("WWW_FORM_TXT");
   /* eliminate <CR><LF>s that originate from DOS/Windows platforms */
   cptr = sptr = TextPtr;
   while (*cptr)
   {
      if (*(unsigned short*)cptr == '\r\n')
         *sptr++ = *++cptr;
      else
         *sptr++ = *cptr;
      cptr++;
   }
   *sptr = '\0';

   sprintf (FileName, "%s%s%s",
            VmsMailUserFullDirectory,
            PERSONAL_CONFIG_FILE_NAME, PERSONAL_CONFIG_FILE_TYPE);

   sprintf (TempFileName, "%s%s%s",
            VmsMailUserFullDirectory,
            PERSONAL_CONFIG_FILE_NAME, PERSONAL_CONFIG_FILE_TEMP);

   FilePtr = fopen (TempFileName, "w", "shr=nil", "ctx=rec", "rfm=stmlf");
   if (!FilePtr)
   {
      status = vaxc$errno;
      CgiLibResponseError (FI_LI, status,
"%s\n\
<P><FORM>\n\
<INPUT TYPE=button VALUE=\"%s\" onClick=\"history.go(-1)\">&nbsp;\n\
<INPUT TYPE=button VALUE=\"%s\" onClick=\"window.close()\">\n\
</FORM>",
         FileName, lang_BtnGoBack, lang_BtnClose);
      return;
   }

   if (fputs (TextPtr, FilePtr) == EOF)
   {
      status = vaxc$errno;
      CgiLibResponseError (FI_LI, status,
"%s\n\
<P><FORM>\n\
<INPUT TYPE=button VALUE=\"%s\" onClick=\"history.go(-1)\">&nbsp;\n\
<INPUT TYPE=button VALUE=\"%s\" onClick=\"window.close()\">\n\
</FORM>",
         FileName, lang_BtnGoBack, lang_BtnClose);

      fclose (FilePtr);
      remove (TempFileName);
      return;
   }

   fclose (FilePtr);

   rename (TempFileName, FileName);

   /* purge the versions back to a maximum of 3 */
   strcat (FileName, ";-3");
   while (!remove (FileName));

   CgiLibResponseSuccess (
"%s\n\
<P><FORM>\n\
<INPUT TYPE=button VALUE=\"%s\" onClick=\"history.go(-1)\">&nbsp;\n\
<INPUT TYPE=button VALUE=\"%s\" onClick=\"window.close()\">\n\
</FORM>",
      lang_PersonalSetupUpdated, lang_BtnGoBack, lang_BtnClose);
}

/*****************************************************************************/
