/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%        AAA   TTTTT  TTTTT  RRRR   IIIII  BBBB   U   U  TTTTT  EEEEE         %
%       A   A    T      T    R   R    I    B   B  U   U    T    E             %
%       AAAAA    T      T    RRRR     I    BBBB   U   U    T    EEE           %
%       A   A    T      T    R R      I    B   B  U   U    T    E             %
%       A   A    T      T    R  R   IIIII  BBBB    UUU     T    EEEEE         %
%                                                                             %
%                                                                             %
%              Methods to Get/Set/Destroy Image Text Attributes               %
%                                                                             %
%                             Software Design                                 %
%                               John Cristy                                   %
%                              February 2000                                  %
%                                                                             %
%                                                                             %
%  Copyright 1999-2004 ImageMagick Studio LLC, a non-profit organization      %
%  dedicated to making software imaging solutions freely available.           %
%                                                                             %
%  You may not use this file except in compliance with the License.  You may  %
%  obtain a copy of the License at                                            %
%                                                                             %
%    http://www.imagemagick.org/www/Copyright.html                            %
%                                                                             %
%  Unless required by applicable law or agreed to in writing, software        %
%  distributed under the License is distributed on an "AS IS" BASIS,          %
%  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.   %
%  See the License for the specific language governing permissions and        %
%  limitations under the License.                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  The Attributes methods gets, sets, or destroys attributes associated
%  with a particular image (e.g. comments, copyright, author, etc).
%
%
*/

/*
  Include declarations.
*/
#include "magick/studio.h"
#include "magick/attribute.h"
#include "magick/blob.h"
#include "magick/draw.h"
#include "magick/exception_private.h"
#include "magick/list.h"
#include "magick/memory_.h"
#include "magick/profile.h"
#include "magick/string_.h"
#include "magick/utility.h"

/*
  Forward declarations.
*/
static char
  *TracePSClippath(unsigned char *,size_t,const unsigned long,
    const unsigned long),
  *TraceSVGClippath(unsigned char *,size_t,const unsigned long,
    const unsigned long);

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   D e s t r o y I m a g e A t t r i b u t e s                               %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  DestroyImageAttributes() deallocates memory associated with the image
%  attribute list.
%
%  The format of the DestroyImageAttributes method is:
%
%      DestroyImageAttributes(Image *image)
%
%  A description of each parameter follows:
%
%    o image: The image.
%
%
*/
MagickExport void DestroyImageAttributes(Image *image)
{
  ImageAttribute
    *attribute;

  register ImageAttribute
    *p;

  assert(image != (Image *) NULL);
  assert(image->signature == MagickSignature);
  if (image->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),image->filename);
  for (p=image->attributes; p != (ImageAttribute *) NULL; )
  {
    attribute=p;
    p=p->next;
    if (attribute->key != (char *) NULL)
      attribute->key=(char *) RelinquishMagickMemory(attribute->key);
    if (attribute->value != (char *) NULL)
      attribute->value=(char *) RelinquishMagickMemory(attribute->value);
    attribute=(ImageAttribute *) RelinquishMagickMemory(attribute);
  }
  image->attributes=(ImageAttribute *) NULL;
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   G e t I m a g e A t t r i b u t e                                         %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  GetImageAttribute() searches the list of image attributes and returns
%  a pointer to the attribute if it exists otherwise NULL.
%
%  The format of the GetImageAttribute method is:
%
%      const ImageAttribute *GetImageAttribute(const Image *image,
%        const char *key)
%
%  A description of each parameter follows:
%
%    o image: The image.
%
%    o key:  These character strings are the name of an image attribute to
%      return.
%
%
*/

static MagickBooleanType GenerateIPTCAttribute(Image *image,const char *key)
{
  char
    *attribute;

  const StringInfo
    *profile;

  int
    count,
    dataset,
    record;

  register long
    i;

  size_t
    length;

  profile=GetImageProfile(image,"8bim");
  if (profile == (StringInfo *) NULL)
    return(MagickFalse);
  count=sscanf(key,"IPTC:%d:%d",&dataset,&record);
  if (count != 2)
    return(MagickFalse);
  for (i=0; i < (long) profile->length; i++)
  {
    if ((int) profile->datum[i] != 0x1c)
      continue;
    if ((int) profile->datum[i+1] != dataset)
      continue;
    if ((int) profile->datum[i+2] != record)
      continue;
    length=(size_t) (profile->datum[i+3] << 8);
    length|=profile->datum[i+4];
    attribute=(char *) AcquireMagickMemory(length+MaxTextExtent);
    if (attribute == (char *) NULL)
      continue;
    (void) CopyMagickString(attribute,(char *) profile->datum+i+5,length+1);
    (void) SetImageAttribute(image,key,(const char *) attribute);
    attribute=(char *) RelinquishMagickMemory(attribute);
    break;
  }
  return((MagickBooleanType) (i < (long) profile->length));
}

static unsigned char ReadByte(unsigned char **p,size_t *length)
{
  unsigned char
    c;

  if (*length < 1)
    return((unsigned char) 0xff);
  c=(*(*p)++);
  (*length)--;
  return(c);
}

static long ReadMSBLong(unsigned char **p,size_t *length)
{
  int
    c;

  long
    value;

  register long
    i;

  unsigned char
    buffer[4];

  if (*length < 4)
    return(-1);
  for (i=0; i < 4; i++)
  {
    c=(int) (*(*p)++);
    (*length)--;
    buffer[i]=(unsigned char) c;
  }
  value=(long) (buffer[0] << 24);
  value|=buffer[1] << 16;
  value|=buffer[2] << 8;
  value|=buffer[3];
  return(value);
}

static long ReadMSBShort(unsigned char **p,size_t *length)
{
  int
    c;

  long
    value;

  register long
    i;

  unsigned char
    buffer[2];

  if (*length < 2)
    return(-1);
  for (i=0; i < 2; i++)
  {
    c=(int) (*(*p)++);
    (*length)--;
    buffer[i]=(unsigned char) c;
  }
  value=(long) (buffer[0] << 8);
  value|=buffer[1];
  return(value);
}

static MagickBooleanType Generate8BIMAttribute(Image *image,const char *key)
{
  char
    *attribute,
    format[MaxTextExtent],
    name[MaxTextExtent],
    *resource;

  const StringInfo
    *profile;

  long
    id,
    start,
    stop,
    sub_number;

  MagickBooleanType
    status;

  register long
    i;

  ssize_t
    count;

  size_t
    length;

  unsigned char
    *info;

  /*
    There's no newlines in path names, so it's safe as terminator.
  */
  profile=GetImageProfile(image,"iptc");
  if (profile == (StringInfo *) NULL)
    return(MagickFalse);
  count=(ssize_t) sscanf(key,"8BIM:%ld,%ld:%[^\n]\n%[^\n]",&start,&stop,name,
    format);
  if ((count != 2) && (count != 3) && (count != 4))
    return(MagickFalse);
  if (count < 4)
    (void) strcpy(format,"SVG");
  if (count < 3)
    *name='\0';
  sub_number=1;
  if (*name == '#')
    sub_number=atol(&name[1]);
  sub_number=Max(sub_number,1);
  resource=(char *) NULL;
  status=MagickFalse;
  length=profile->length;
  info=(unsigned char *) profile->datum;
  while ((length > 0) && (status == MagickFalse))
  {
    if (ReadByte(&info,&length) != (unsigned char) '8')
      continue;
    if (ReadByte(&info,&length) != (unsigned char) 'B')
      continue;
    if (ReadByte(&info,&length) != (unsigned char) 'I')
      continue;
    if (ReadByte(&info,&length) != (unsigned char) 'M')
      continue;
    id=ReadMSBShort(&info,&length);
    if (id < start)
      continue;
    if (id > stop)
      continue;
    if (resource != (char *) NULL)
      resource=(char *) RelinquishMagickMemory(resource);
    count=(ssize_t) ReadByte(&info,&length);
    if ((count != 0) && ((size_t) count <= length))
      {
        resource=(char *) AcquireMagickMemory((size_t) count+MaxTextExtent);
        if (resource != (char *) NULL)
          {
            for (i=0; i < (long) count; i++)
              resource[i]=(char) ReadByte(&info,&length);
            resource[count]='\0';
          }
      }
    if ((count & 0x01) == 0)
      (void) ReadByte(&info,&length);
    count=(ssize_t) ReadMSBLong(&info,&length);
    if ((*name != '\0') && (*name != '#'))
      if ((resource == (char *) NULL) || (LocaleCompare(name,resource) != 0))
        {
          /*
            No name match, scroll forward and try next.
          */
          info+=count;
          length-=count;
          continue;
        }
    if ((*name == '#') && (sub_number != 1))
      {
        /*
          No numbered match, scroll forward and try next.
        */
        sub_number--;
        info+=count;
        length-=count;
        continue;
      }
    /*
      We have the resource of interest.
    */
    attribute=(char *) AcquireMagickMemory((size_t) count+MaxTextExtent);
    if (attribute != (char *) NULL)
      {
        (void) CopyMagickMemory(attribute,(char *) info,(size_t) count);
        attribute[count]='\0';
        info+=count;
        length-=count;
        if ((id <= 1999) || (id >= 2999))
          (void) SetImageAttribute(image,key,(const char *) attribute);
        else
          {
            char
              *path;

            if (LocaleCompare("svg",format) == 0)
              path=TraceSVGClippath((unsigned char *) attribute,(size_t) count,
                image->columns,image->rows);
            else
              path=TracePSClippath((unsigned char *) attribute,(size_t) count,
                image->columns,image->rows);
            (void) SetImageAttribute(image,key,(const char *) path);
            path=(char *) RelinquishMagickMemory(path);
          }
        attribute=(char *) RelinquishMagickMemory(attribute);
        status=MagickTrue;
      }
  }
  if (resource != (char *) NULL)
    resource=(char *) RelinquishMagickMemory(resource);
  return(status);
}

#define DE_STACK_SIZE  16
#define EXIF_DELIMITER  "\n"
#define EXIF_NUM_FORMATS  12
#define EXIF_FMT_BYTE  1
#define EXIF_FMT_STRING  2
#define EXIF_FMT_USHORT  3
#define EXIF_FMT_ULONG  4
#define EXIF_FMT_URATIONAL  5
#define EXIF_FMT_SBYTE  6
#define EXIF_FMT_UNDEFINED  7
#define EXIF_FMT_SSHORT  8
#define EXIF_FMT_SLONG  9
#define EXIF_FMT_SRATIONAL  10
#define EXIF_FMT_SINGLE  11
#define EXIF_FMT_DOUBLE  12
#define TAG_EXIF_OFFSET  0x8769
#define TAG_INTEROP_OFFSET  0xa005

typedef struct _TagInfo
{
  unsigned short
    tag;

  char
    *description;
} TagInfo;

static TagInfo
  tag_table[] =
  {
    {  0x100, (char *) "ImageWidth"},
    {  0x101, (char *) "ImageLength"},
    {  0x102, (char *) "BitsPerSample"},
    {  0x103, (char *) "Compression"},
    {  0x106, (char *) "PhotometricInterpretation"},
    {  0x10a, (char *) "FillOrder"},
    {  0x10d, (char *) "DocumentName"},
    {  0x10e, (char *) "ImageDescription"},
    {  0x10f, (char *) "Make"},
    {  0x110, (char *) "Model"},
    {  0x111, (char *) "StripOffsets"},
    {  0x112, (char *) "Orientation"},
    {  0x115, (char *) "SamplesPerPixel"},
    {  0x116, (char *) "RowsPerStrip"},
    {  0x117, (char *) "StripByteCounts"},
    {  0x11a, (char *) "XResolution"},
    {  0x11b, (char *) "YResolution"},
    {  0x11c, (char *) "PlanarConfiguration"},
    {  0x128, (char *) "ResolutionUnit"},
    {  0x12d, (char *) "TransferFunction"},
    {  0x131, (char *) "Software"},
    {  0x132, (char *) "DateTime"},
    {  0x13b, (char *) "Artist"},
    {  0x13e, (char *) "WhitePoint"},
    {  0x13f, (char *) "PrimaryChromaticities"},
    {  0x156, (char *) "TransferRange"},
    {  0x200, (char *) "JPEGProc"},
    {  0x201, (char *) "JPEGInterchangeFormat"},
    {  0x202, (char *) "JPEGInterchangeFormatLength"},
    {  0x211, (char *) "YCbCrCoefficients"},
    {  0x212, (char *) "YCbCrSubSampling"},
    {  0x213, (char *) "YCbCrPositioning"},
    {  0x214, (char *) "ReferenceBlackWhite"},
    {  0x1000, (char *) "RelatedImageFileFormat"},
    {  0x1001, (char *) "RelatedImageLength"},
    {  0x1002, (char *) "RelatedImageWidth"},
    {  0x828d, (char *) "CFARepeatPatternDim"},
    {  0x828e, (char *) "CFAPattern"},
    {  0x828f, (char *) "BatteryLevel"},
    {  0x8298, (char *) "Copyright"},
    {  0x829a, (char *) "ExposureTime"},
    {  0x829d, (char *) "FNumber"},
    {  0x83Bb, (char *) "IPTC/NAA"},
    {  0x8769, (char *) "ExifOffset"},
    {  0x8773, (char *) "InterColorProfile"},
    {  0x8822, (char *) "ExposureProgram"},
    {  0x8824, (char *) "SpectralSensitivity"},
    {  0x8825, (char *) "GPSInfo"},
    {  0x8827, (char *) "ISOSpeedRatings"},
    {  0x8828, (char *) "OECF"},
    {  0x8829, (char *) "Interlace"},
    {  0x882a, (char *) "TimeZoneOffset"},
    {  0x882b, (char *) "SelfTimerMode"},
    {  0x9000, (char *) "ExifVersion"},
    {  0x9003, (char *) "DateTimeOriginal"},
    {  0x9004, (char *) "DateTimeDigitized"},
    {  0x9101, (char *) "ComponentsConfiguration"},
    {  0x9102, (char *) "CompressedBitsPerPixel"},
    {  0x9201, (char *) "ShutterSpeedValue"},
    {  0x9202, (char *) "ApertureValue"},
    {  0x9203, (char *) "BrightnessValue"},
    {  0x9204, (char *) "ExposureBiasValue"},
    {  0x9205, (char *) "MaxApertureValue"},
    {  0x9206, (char *) "SubjectDistance"},
    {  0x9207, (char *) "MeteringMode"},
    {  0x9208, (char *) "LightSrc"},
    {  0x9209, (char *) "Flash"},
    {  0x920a, (char *) "FocalLength"},
    {  0x920b, (char *) "FlashEnergy"},
    {  0x920c, (char *) "SpatialFrequencyResponse"},
    {  0x920d, (char *) "Noise"},
    {  0x9211, (char *) "ImageNumber"},
    {  0x9212, (char *) "SecurityClassification"},
    {  0x9213, (char *) "ImageHistory"},
    {  0x9214, (char *) "SubjectLocation"},
    {  0x9215, (char *) "ExposureIndex"},
    {  0x9216, (char *) "TIFF/EPStandardID"},
    {  0x927c, (char *) "MakerNote"},
    {  0x9286, (char *) "UserComment"},
    {  0x9290, (char *) "SubSecTime"},
    {  0x9291, (char *) "SubSecTimeOriginal"},
    {  0x9292, (char *) "SubSecTimeDigitized"},
    {  0xA000, (char *) "FlashPixVersion"},
    {  0xA001, (char *) "ColorSpace"},
    {  0xA002, (char *) "ExifImageWidth"},
    {  0xA003, (char *) "ExifImageLength"},
    {  0xA005, (char *) "InteroperabilityOffset"},
    {  0xA20b, (char *) "FlashEnergy"},
    {  0xA20c, (char *) "SpatialFrequencyResponse"},
    {  0xA20e, (char *) "FocalPlaneXResolution"},
    {  0xA20f, (char *) "FocalPlaneYResolution"},
    {  0xA210, (char *) "FocalPlaneResolutionUnit"},
    {  0xA214, (char *) "SubjectLocation"},
    {  0xA215, (char *) "ExposureIndex"},
    {  0xA217, (char *) "SensingMethod"},
    {  0xA300, (char *) "FileSource"},
    {  0xA301, (char *) "SceneType"},
    {  0xA302, (char *) "CFAPattern"},
    {  0x0000, (char *) NULL}
  };

static int
  format_bytes[] = {0, 1, 1, 2, 4, 8, 1, 1, 2, 4, 8, 4, 8};

static short ReadInt16(unsigned int msb_order,void *buffer)
{
  short
    value;

  if (msb_order != MagickFalse)
    {
      value=(short) ((((unsigned char *) buffer)[0] << 8) |
        ((unsigned char *) buffer)[1]);
      return(value);
    }
  value=(short) ((((unsigned char *) buffer)[1] << 8) |
    ((unsigned char *) buffer)[0]);
  return(value);
}

static long ReadInt32(unsigned int msb_order,void *buffer)
{
  long
    value;

  if (msb_order != MagickFalse)
    {
      value=(long) ((((unsigned char *) buffer)[0] << 24) |
        (((unsigned char *) buffer)[1] << 16) |
        (((unsigned char *) buffer)[2] << 8) | (((unsigned char *) buffer)[3]));
      return(value);
    }
  value=(long) ((((unsigned char *) buffer)[3] << 24) |
    (((unsigned char *) buffer)[2] << 16) |
    (((unsigned char *) buffer)[1] << 8 ) |
    (((unsigned char *) buffer)[0]));
  return(value);
}

static unsigned short ReadUint16(unsigned int msb_order,void *buffer)
{
  unsigned short
    value;

  if (msb_order != MagickFalse)
    {
      value=(unsigned short) ((((unsigned char *) buffer)[0] << 8) |
        ((unsigned char *) buffer)[1]);
      return(value);
    }
  value=(unsigned short) ((((unsigned char *) buffer)[1] << 8) |
    ((unsigned char *) buffer)[0]);
  return(value);
}

static unsigned long ReadUint32(unsigned int msb_order,void *buffer)
{
  return((unsigned long) ReadInt32(msb_order,buffer) & 0xffffffff);
}

static int GenerateEXIFAttribute(Image *image,const char *specification)
{
  char
    *key,
    *value,
    *final;

  const StringInfo
    *profile;

  int
    all,
    id,
    level;

  register long
    i;

  size_t
    length;

  unsigned long
    offset;

  unsigned char
    *tiffp,
    *ifdstack[DE_STACK_SIZE],
    *ifdp,
    *info;

  unsigned int
    de,
    destack[DE_STACK_SIZE],
    msb_order,
    nde;

  unsigned long
    tag;

  /*
    If EXIF data exists, then try to parse the request for a tag.
  */
  profile=GetImageProfile(image,"exif");
  if (profile == (StringInfo *) NULL)
    return(MagickFalse);
  value=(char *) NULL;
  final=AcquireString("");
  key=(char *) &specification[5];
  if ((key == (char *) NULL) || (*key == '\0'))
    return(MagickFalse);
  while (isspace((int) ((unsigned char) *key)) != 0)
    key++;
  all=0;
  tag=(~0UL);
  switch (*key)
  {
    /*
      Caller has asked for all the tags in the EXIF data.
    */
    case '*':
    {
      tag=0;
      all=1; /* return the data in description=value format */
      break;
    }
    case '!':
    {
      tag=0;
      all=2; /* return the data in tageid=value format */
      break;
    }
    /*
      Check for a hex based tag specification first.
    */
    case '#':
    {
      char
        c;

      unsigned long
        n;

      tag=0;
      key++;
      n=(unsigned long) strlen(key);
      if (n != 4)
        return(MagickFalse);
      else
        {
          /*
            Parse tag specification as a hex number.
          */
          n/=4;
          do
          {
            for (i=(long) n-1; i >= 0; i--)
            {
              c=(*key++);
              tag<<=4;
              if ((c >= '0') && (c <= '9'))
                tag|=(int) (c-'0');
              else
                if ((c >= 'A') && (c <= 'F'))
                  tag|=(int) (c-('A'-(char) 10));
                else
                  if ((c >= 'a') && (c <= 'f'))
                    tag|=(int) (c-('a'-(char) 10));
                  else
                    return(MagickFalse);
            }
          } while (*key != '\0');
        }
      break;
    }
    default:
    {
      /*
        Try to match the text with a tag name instead.
      */
      for (i=0; ; i++)
      {
        if (tag_table[i].tag == 0)
          break;
        if (LocaleCompare(tag_table[i].description,key) == 0)
          {
            tag=(unsigned long) tag_table[i].tag;
            break;
          }
      }
      break;
    }
  }
  if (tag == (~0UL))
    return(MagickFalse);
  length=profile->length;
  info=(unsigned char *) profile->datum;
  while (length != 0)
  {
    if ((int) ReadByte(&info,&length) != 0x45)
      continue;
    if ((int) ReadByte(&info,&length) != 0x78)
      continue;
    if ((int) ReadByte(&info,&length) != 0x69)
      continue;
    if ((int) ReadByte(&info,&length) != 0x66)
      continue;
    if ((int) ReadByte(&info,&length) != 0x00)
      continue;
    if ((int) ReadByte(&info,&length) != 0x00)
      continue;
    break;
  }
  if (length < 16)
    return(MagickFalse);
  tiffp=info;
  id=(int) ReadUint16(0,tiffp);
  msb_order=0;
  if (id == 0x4949) /* LSB */
    msb_order=0;
  else
    if (id == 0x4D4D) /* MSB */
      msb_order=1;
    else
      return(MagickFalse);
  if (ReadUint16(msb_order,tiffp+2) != 0x002a)
    return(MagickFalse);
  /*
    This is the offset to the first IFD.
  */
  offset=ReadUint32(msb_order,tiffp+4);
  if ((size_t) offset >= length)
    return(MagickFalse);
  /*
    Set the pointer to the first IFD and follow it were it leads.
  */
  ifdp=tiffp+offset;
  level=0;
  de=0;
  do
  {
    /*
      If there is anything on the stack then pop it off.
    */
    if (level > 0)
      {
        level--;
        ifdp=ifdstack[level];
        de=destack[level];
      }
    /*
      Determine how many entries there are in the current IFD.
    */
    nde=ReadUint16(msb_order,ifdp);
    for (; de < nde; de++)
    {
      long
        n,
        t,
        f,
        c;

      char
        *pde,
        *pval;

      pde=(char *) (ifdp+2+(12*de));
      t=(long) ReadUint16(msb_order,pde); /* get tag value */
      f=(long) ReadUint16(msb_order,pde+2); /* get the format */
      if ((f-1) >= EXIF_NUM_FORMATS)
        break;
      c=(long) ReadUint32(msb_order,pde+4); /* get number of components */
      n=c*format_bytes[f];
      if (n <= 4)
        pval=pde+8;
      else
        {
          unsigned long
            oval;

          /*
            The directory entry contains an offset.
          */
          oval=ReadUint32(msb_order,pde+8);
          if ((size_t) (oval+n) > length)
            continue;
          pval=(char *)(tiffp+oval);
        }
      if ((all != 0)  || (tag == (unsigned long) t))
        {
          char
            buffer[MaxTextExtent];

          switch (f)
          {
            case EXIF_FMT_SBYTE:
            {
              (void) FormatMagickString(buffer,MaxTextExtent,"%ld",
                (long) (*(char *) pval));
              value=AcquireString(buffer);
              break;
            }
            case EXIF_FMT_BYTE:
            {
              (void) FormatMagickString(buffer,MaxTextExtent,"%ld",
                (long) (*(unsigned char *) pval));
              value=AcquireString(buffer);
              break;
            }
            case EXIF_FMT_SSHORT:
            {
              (void) FormatMagickString(buffer,MaxTextExtent,"%hd",
                ReadUint16(msb_order,pval));
              value=AcquireString(buffer);
              break;
            }
            case EXIF_FMT_USHORT:
            {
              (void) FormatMagickString(buffer,MaxTextExtent,"%hu",
                ReadInt16(msb_order,pval));
              value=AcquireString(buffer);
              break;
            }
            case EXIF_FMT_ULONG:
            {
              (void) FormatMagickString(buffer,MaxTextExtent,"%lu",
                ReadUint32(msb_order,pval));
              value=AcquireString(buffer);
              break;
            }
            case EXIF_FMT_SLONG:
            {
              (void) FormatMagickString(buffer,MaxTextExtent,"%ld",
                ReadInt32(msb_order,pval));
              value=AcquireString(buffer);
              break;
            }
            case EXIF_FMT_URATIONAL:
            {
              (void) FormatMagickString(buffer,MaxTextExtent,"%ld/%ld",
                ReadUint32(msb_order,pval),
                ReadUint32(msb_order,4+(char *) pval));
              value=AcquireString(buffer);
              break;
            }
            case EXIF_FMT_SRATIONAL:
            {
              (void) FormatMagickString(buffer,MaxTextExtent,"%ld/%ld",
                ReadInt32(msb_order,pval),
                ReadInt32(msb_order,4+(char *) pval));
              value=AcquireString(buffer);
              break;
            }
            case EXIF_FMT_SINGLE:
            {
              (void) FormatMagickString(buffer,MaxTextExtent,"%f",
                (double) *(float *) pval);
              value=AcquireString(buffer);
              break;
            }
            case EXIF_FMT_DOUBLE:
            {
              (void) FormatMagickString(buffer,MaxTextExtent,"%f",
                *(double *) pval);
              value=AcquireString(buffer);
              break;
            }
            default:
            case EXIF_FMT_UNDEFINED:
            case EXIF_FMT_STRING:
            {
              value=(char *) AcquireMagickMemory((size_t) n+1);
              if (value != (char *) NULL)
                {
                  long
                    a;

                  for (a=0; a < n; a++)
                  {
                    value[a]='.';
                    if (isprint((int) ((unsigned char) pval[a])) != 0)
                      value[a]=pval[a];
                  }
                  value[a]='\0';
                  break;
                }
              break;
            }
          }
          if (value != (char *) NULL)
            {
              int
                i;

              char
                *description;

              if (strlen(final) != 0)
                (void) ConcatenateString(&final,EXIF_DELIMITER);
              description=(char *) NULL;
              switch (all)
              {
                case 1:
                {
                  description=(char *) "unknown";
                  for (i=0; ; i++)
                  {
                    if (tag_table[i].tag == 0)
                      break;
                    if ((long) tag_table[i].tag == t)
                      {
                        description=tag_table[i].description;
                        break;
                      }
                  }
                  (void) FormatMagickString(buffer,MaxTextExtent,"%s=",
                    description);
                  (void) ConcatenateString(&final,buffer);
                  break;
                }
                case 2:
                {
                  (void) FormatMagickString(buffer,MaxTextExtent,"#%04lx=",t);
                  (void) ConcatenateString(&final,buffer);
                  break;
                }
              }
              (void) ConcatenateString(&final,value);
              value=(char *) RelinquishMagickMemory(value);
            }
        }
        if ((t == TAG_EXIF_OFFSET) || (t == TAG_INTEROP_OFFSET))
          {
            size_t
              offset;

            offset=(size_t) ReadUint32(msb_order,pval);
            if ((offset < length) && (level < (DE_STACK_SIZE-2)))
              {
                /*
                  Push our current directory state onto the stack.
                */
                ifdstack[level]=ifdp;
                de++; /* bump to the next entry */
                destack[level]=de;
                level++;
                /*
                  Push new state onto of stack to cause a jump.
                */
                ifdstack[level]=tiffp+offset;
                destack[level]=0;
                level++;
              }
            break; /* break out of the for loop */
          }
    }
  } while (level > 0);
  if (strlen(final) == 0)
    (void) ConcatenateString(&final,"unknown");
  (void) SetImageAttribute(image,specification,(const char *) final);
  final=(char *) RelinquishMagickMemory(final);
  return(MagickTrue);
}

MagickExport const ImageAttribute *GetImageAttribute(const Image *image,
  const char *key)
{
  register ImageAttribute
    *p;

  assert(image != (Image *) NULL);
  assert(image->signature == MagickSignature);
  if (image->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),image->filename);
  if (key == (char *) NULL)
    return(image->attributes);
  for (p=image->attributes; p != (ImageAttribute *) NULL; p=p->next)
    if (LocaleCompare(key,p->key) == 0)
      return(p);
  if (LocaleNCompare("iptc:",key,5) == 0)
    {
      if (GenerateIPTCAttribute((Image *) image,key) == MagickTrue)
        return(GetImageAttribute(image,key));
    }
  if (LocaleNCompare("8bim:",key,5) == 0)
    {
      if (Generate8BIMAttribute((Image *) image,key) == MagickTrue)
        return(GetImageAttribute(image,key));
    }
  if (LocaleNCompare("exif:",key,5) == 0)
    {
      if (GenerateEXIFAttribute((Image *) image,key) == MagickTrue)
        return(GetImageAttribute(image,key));
    }
  return(p);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   G e t I m a g e C l i p p i n g P a t h A t t r i b u t e                 %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  GetImageClippingPathAttribute() searches the list of image attributes and
%  returns a pointer to a clipping path if it exists otherwise NULL.
%
%  The format of the GetImageClippingPathAttribute method is:
%
%      const ImageAttribute *GetImageClippingPathAttribute(const Image *image)
%
%  A description of each parameter follows:
%
%    o attribute:  Method GetImageClippingPathAttribute returns the clipping
%      path if it exists otherwise NULL.
%
%    o image: The image.
%
%
*/
MagickExport const ImageAttribute *GetImageClippingPathAttribute(
  const Image *image)
{
  return(GetImageAttribute(image,"8BIM:1999,2998"));
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
+   G e t I m a g e I n f o A t t r i b u t e                                 %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  GetImageInfoAttribute() returns a "fake" attribute based on data in the
%  image info or image structures.
%
%  The format of the GetImageInfoAttribute method is:
%
%      const ImageAttribute *GetImageAttribute(const ImageInfo *image_info,
%        const Image *image,const char *key)
%
%  A description of each parameter follows:
%
%    o attribute:  Method GetImageInfoAttribute returns the attribute if it
%      exists otherwise NULL.
%
%    o image_info: The imageInfo.
%
%    o image: The image.
%
%    o key:  These character strings are the name of an image attribute to
%      return.
%
*/
MagickExport const ImageAttribute *GetImageInfoAttribute(
  const ImageInfo *image_info,const Image *image,const char *key)
{
  char
    attribute[MaxTextExtent],
    filename[MaxTextExtent];

  attribute[0]='\0';
  switch (*(key))
  {
    case 'b':
    {
      if (LocaleNCompare("base",key,2) == 0)
        {
          GetPathComponent(image->magick_filename,BasePath,filename);
          (void) CopyMagickString(attribute,filename,MaxTextExtent);
          break;
        }
      break;
    }
    case 'd':
    {
      if (LocaleNCompare("depth",key,2) == 0)
        {
          (void) FormatMagickString(attribute,MaxTextExtent,"%lu",image->depth);
          break;
        }
      if (LocaleNCompare("directory",key,2) == 0)
        {
          GetPathComponent(image->magick_filename,HeadPath,filename);
          (void) CopyMagickString(attribute,filename,MaxTextExtent);
          break;
        }
      break;
    }
    case 'e':
    {
      if (LocaleNCompare("extension",key,2) == 0)
        {
          GetPathComponent(image->magick_filename,ExtensionPath,filename);
          (void) CopyMagickString(attribute,filename,MaxTextExtent);
          break;
        }
      break;
    }
    case 'g':
    {
      if (LocaleNCompare("group",key,2) == 0)
        {
          (void) FormatMagickString(attribute,MaxTextExtent,"0x%lx",
            image_info->group);
          break;
        }
      break;
    }
    case 'h':
    {
      if (LocaleNCompare("height",key,2) == 0)
        {
          (void) FormatMagickString(attribute,MaxTextExtent,"%lu",
            image->magick_rows != 0 ? image->magick_rows : 256UL);
          break;
        }
      break;
    }
    case 'i':
    {
      if (LocaleNCompare("input",key,2) == 0)
        {
          (void) CopyMagickString(attribute,image->filename,MaxTextExtent);
          break;
        }
      break;
    }
    case 'm':
    {
      if (LocaleNCompare("magick",key,2) == 0)
        {
          (void) CopyMagickString(attribute,image->magick,MaxTextExtent);
          break;
        }
      break;
    }
    case 'n':
    {
      if (LocaleNCompare("name",key,2) == 0)
        {
          (void) CopyMagickString(attribute,filename,MaxTextExtent);
          break;
        }
     break;
    }
    case 's':
    {
      if (LocaleNCompare("size",key,2) == 0)
        {
          char
            format[MaxTextExtent];

          FormatSize(GetBlobSize(image),format);
          (void) FormatMagickString(attribute,MaxTextExtent,"%s",format);
          break;
        }
      if (LocaleNCompare("scene",key,2) == 0)
        {
          (void) FormatMagickString(attribute,MaxTextExtent,"%lu",image->scene);
          if (image_info->number_scenes != 0)
            (void) FormatMagickString(attribute,MaxTextExtent,"%lu",
              image_info->scene);
          break;
        }
      if (LocaleNCompare("scenes",key,6) == 0)
        {
          (void) FormatMagickString(attribute,MaxTextExtent,"%lu",
            (unsigned long) GetImageListLength(image));
          break;
        }
       break;
    }
    case 'o':
    {
      if (LocaleNCompare("output",key,2) == 0)
        {
          (void) CopyMagickString(attribute,image_info->filename,MaxTextExtent);
          break;
        }
     break;
    }
    case 'p':
    {
      if (LocaleNCompare("page",key,2) == 0)
        {
          register const Image
            *p;

          unsigned long
            page;

          p=image;
          for (page=1; p->previous != (Image *) NULL; page++)
            p=p->previous;
          (void) FormatMagickString(attribute,MaxTextExtent,"%lu",page);
          break;
        }
      break;
    }
    case 'u':
    {
      if (LocaleNCompare("unique",key,2) == 0)
        {
          (void) CopyMagickString(filename,image_info->unique,MaxTextExtent);
          (void) CopyMagickString(attribute,filename,MaxTextExtent);
          break;
        }
      break;
    }
    case 'w':
    {
      if (LocaleNCompare("width",key,2) == 0)
        {
          (void) FormatMagickString(attribute,MaxTextExtent,"%lu",
            image->magick_columns != 0 ? image->magick_columns : 256UL);
          break;
        }
      break;
    }
    case 'x':
    {
      if (LocaleNCompare("xresolution",key,2) == 0)
        {
          (void) FormatMagickString(attribute,MaxTextExtent,"%g",
            image->x_resolution);
          break;
        }
      break;
    }
    case 'y':
    {
      if (LocaleNCompare("yresolution",key,2) == 0)
        {
          (void) FormatMagickString(attribute,MaxTextExtent,"%g",
            image->y_resolution);
          break;
        }
      break;
    }
    case 'z':
    {
      if (LocaleNCompare("zero",key,2) == 0)
        {
          (void) CopyMagickString(filename,image_info->zero,MaxTextExtent);
          (void) CopyMagickString(attribute,filename,MaxTextExtent);
          break;
        }
      break;
    }
  }
  if (strlen(image->magick_filename) != 0)
    return(GetImageAttribute(image,key));
  return((ImageAttribute *) NULL);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   S e t I m a g e A t t r i b u t e                                         %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  SetImageAttribute() searches the list of image attributes and replaces the
%  attribute value.  If it is not found in the list, the attribute name
%  and value is added to the list.   If the attribute exists in the list,
%  the value is concatenated to the attribute.  SetImageAttribute returns
%  MagickTrue if the attribute is successfully concatenated or added to the
%  list, otherwise MagickFalse.  If the value is NULL, the matching key is
%  deleted from the list.
%
%  The format of the SetImageAttribute method is:
%
%       MagickBooleanType SetImageAttribute(Image *image,const char *key,
%        const char *value)
%
%  A description of each parameter follows:
%
%    o image: The image.
%
%    o key,value:  These character strings are the name and value of an image
%      attribute to replace or add to the list.
%
%
*/
MagickExport  MagickBooleanType SetImageAttribute(Image *image,const char *key,
  const char *value)
{
  ImageAttribute
    *attribute;

  register const char
    *q;

  register ImageAttribute
    *p;

  /*
    Initialize new attribute.
  */
  assert(image != (Image *) NULL);
  assert(image->signature == MagickSignature);
  if (image->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),image->filename);
  if ((key == (const char *) NULL) || (*key == '\0'))
    return(MagickFalse);
  if (value == (const char *) NULL)
    {
      /*
        Delete attribute from the image attributes list.
      */
      for (p=image->attributes; p != (ImageAttribute *) NULL; p=p->next)
        if (LocaleCompare(key,p->key) == 0)
          break;
      if (p == (ImageAttribute *) NULL)
        return(MagickFalse);
      if (p->key != (char *) NULL)
        p->key=(char *) RelinquishMagickMemory(p->key);
      if (p->value != (char *) NULL)
        p->value=(char *) RelinquishMagickMemory(p->value);
      if (p->previous != (ImageAttribute *) NULL)
        p->previous->next=p->next;
      else
        {
          image->attributes=p->next;
          if (p->next != (ImageAttribute *) NULL)
            p->next->previous=(ImageAttribute *) NULL;
        }
      if (p->next != (ImageAttribute *) NULL)
        p->next->previous=p->previous;
      attribute=p;
      attribute=(ImageAttribute *) RelinquishMagickMemory(attribute);
      return(MagickTrue);
    }
  if (*value == '\0')
    return(MagickFalse);
  attribute=(ImageAttribute *) AcquireMagickMemory(sizeof(*attribute));
  if (attribute == (ImageAttribute *) NULL)
    return(MagickFalse);
  attribute->key=AcquireString(key);
  for (q=value; *q != '\0'; q++)
    if (((int) ((unsigned char) *q) < 32) &&
        (isspace((int) ((unsigned char) *q)) == 0))
      break;
  if (*q != '\0')
    attribute->value=AcquireString(value);
  else
    attribute->value=TranslateText((ImageInfo *) NULL,image,value);
  attribute->compression=MagickFalse;
  attribute->previous=(ImageAttribute *) NULL;
  attribute->next=(ImageAttribute *) NULL;
  if (image->attributes == (ImageAttribute *) NULL)
    {
      image->attributes=attribute;
      return(MagickTrue);
    }
  for (p=image->attributes; p != (ImageAttribute *) NULL; p=p->next)
  {
    if (LocaleCompare(attribute->key,p->key) == 0)
      {
        (void) ConcatenateString(&p->value,attribute->value);
        attribute->value=(char *) RelinquishMagickMemory(attribute->value);
        attribute->key=(char *) RelinquishMagickMemory(attribute->key);
        attribute=(ImageAttribute *) RelinquishMagickMemory(attribute);
        return(MagickTrue);
      }
    if (p->next == (ImageAttribute *) NULL)
      break;
  }
  /*
    Place new attribute at the end of the attribute list.
  */
  attribute->previous=p;
  p->next=attribute;
  return(MagickTrue);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   T r a c e P S C l i p p a t h                                             %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  TracePSClipPath() traces a clip path and returns it as Postscript.
%
%  The format of the TracePSClipPath method is:
%
%      char *TracePSClipPath(unsigned char *blob,size_t length,
%        const unsigned long columns,const unsigned long rows)
%
%  A description of each parameter follows:
%
%    o blob: The blob.
%
%    o length: The length of the blob.
%
%    o columns: The image width.
%
%    o rows: The image height.
%
%
*/
static char *TracePSClippath(unsigned char *blob,size_t length,
  const unsigned long magick_unused(columns),
  const unsigned long magick_unused(rows))
{
  char
    *path,
    *message;

  long
    knot_count,
    selector,
    y;

  MagickBooleanType
    in_subpath;

  PointInfo
    first[3],
    last[3],
    point[3];

  register long
    i,
    x;

  path=AcquireString((char *) NULL);
  if (path == (char *) NULL)
    return((char *) NULL);
  message=AcquireString((char *) NULL);
  (void) FormatMagickString(message,MaxTextExtent,"/ClipImage\n");
  (void) ConcatenateString(&path,message);
  (void) FormatMagickString(message,MaxTextExtent,"{\n");
  (void) ConcatenateString(&path,message);
  (void) FormatMagickString(message,MaxTextExtent,"  /c {curveto} bind def\n");
  (void) ConcatenateString(&path,message);
  (void) FormatMagickString(message,MaxTextExtent,"  /l {lineto} bind def\n");
  (void) ConcatenateString(&path,message);
  (void) FormatMagickString(message,MaxTextExtent,"  /m {moveto} bind def\n");
  (void) ConcatenateString(&path,message);
  (void) FormatMagickString(message,MaxTextExtent,
    "  /v {currentpoint 6 2 roll curveto} bind def\n");
  (void) ConcatenateString(&path,message);
  (void) FormatMagickString(message,MaxTextExtent,
    "  /y {2 copy curveto} bind def\n");
  (void) ConcatenateString(&path,message);
  (void) FormatMagickString(message,MaxTextExtent,
    "  /z {closepath} bind def\n");
  (void) ConcatenateString(&path,message);
  (void) FormatMagickString(message,MaxTextExtent,"  newpath\n");
  (void) ConcatenateString(&path,message);
  /*
    The clipping path format is defined in "Adobe Photoshop File
    Formats Specification" version 6.0 downloadable from adobe.com.
  */
  knot_count=0;
  in_subpath=MagickFalse;
  while (length > 0)
  {
    selector=ReadMSBShort(&blob,&length);
    switch (selector)
    {
      case 0:
      case 3:
      {
        if (knot_count != 0)
          {
            blob+=24;
            length-=24;
            break;
          }
        /*
          Expected subpath length record.
        */
        knot_count=ReadMSBShort(&blob,&length);
        blob+=22;
        length-=22;
        break;
      }
      case 1:
      case 2:
      case 4:
      case 5:
      {
        if (knot_count == 0)
          {
            /*
              Unexpected subpath knot
            */
            blob+=24;
            length-=24;
            break;
          }
        /*
          Add sub-path knot
        */
        for (i=0; i < 3; i++)
        {
          y=ReadMSBLong(&blob,&length);
          x=ReadMSBLong(&blob,&length);
          point[i].x=(double) x/4096/4096;
          point[i].y=1.0-(double) y/4096/4096;
        }
        if (in_subpath == MagickFalse)
          {
            (void) FormatMagickString(message,MaxTextExtent,"  %g %g m\n",
              point[1].x,point[1].y);
            for (i=0; i < 3; i++)
            {
              first[i]=point[i];
              last[i]=point[i];
            }
          }
        else
          {
            /*
              Handle special cases when Bezier curves are used to describe
              corners and straight lines.
            */
            if ((last[1].x == last[2].x) && (last[1].y == last[2].y) &&
                (point[0].x == point[1].x) && (point[0].y == point[1].y))
              (void) FormatMagickString(message,MaxTextExtent,"  %g %g l\n",
                point[1].x,point[1].y);
            else
              if ((last[1].x == last[2].x) && (last[1].y == last[2].y))
                (void) FormatMagickString(message,MaxTextExtent,
                  "  %g %g %g %g v\n",point[0].x,point[0].y,point[1].x,
                  point[1].y);
              else
                if ((point[0].x == point[1].x) && (point[0].y == point[1].y))
                  (void) FormatMagickString(message,MaxTextExtent,
                    "  %g %g %g %g y\n",last[2].x,last[2].y,point[1].x,
                    point[1].y);
                else
                  (void) FormatMagickString(message,MaxTextExtent,
                    "  %g %g %g %g %g %g c\n",last[2].x,last[2].y,point[0].x,
                    point[0].y,point[1].x,point[1].y);
            for (i=0; i < 3; i++)
              last[i]=point[i];
          }
        (void) ConcatenateString(&path,message);
        in_subpath=MagickTrue;
        knot_count--;
        /*
          Close the subpath if there are no more knots.
        */
        if (knot_count == 0)
          {
            /*
              Same special handling as above except we compare to the
              first point in the path and close the path.
            */
            if ((last[1].x == last[2].x) && (last[1].y == last[2].y) &&
                (first[0].x == first[1].x) && (first[0].y == first[1].y))
              (void) FormatMagickString(message,MaxTextExtent,"  %g %g l z\n",
                first[1].x,first[1].y);
            else
              if ((last[1].x == last[2].x) && (last[1].y == last[2].y))
                (void) FormatMagickString(message,MaxTextExtent,
                  "  %g %g %g %g v z\n",first[0].x,first[0].y,first[1].x,
                  first[1].y);
              else
                if ((first[0].x == first[1].x) && (first[0].y == first[1].y))
                  (void) FormatMagickString(message,MaxTextExtent,
                    "  %g %g %g %g y z\n",last[2].x,last[2].y,first[1].x,
                    first[1].y);
                else
                  (void) FormatMagickString(message,MaxTextExtent,
                    "  %g %g %g %g %g %g c z\n",last[2].x,last[2].y,first[0].x,
                    first[0].y,first[1].x,first[1].y);
            (void) ConcatenateString(&path,message);
            in_subpath=MagickFalse;
          }
        break;
      }
      case 6:
      case 7:
      case 8:
      default:
      {
        blob+=24;
        length-=24;
        break;
      }
    }
  }
  /*
    Returns an empty PS path if the path has no knots.
  */
  (void) FormatMagickString(message,MaxTextExtent,"  eoclip\n");
  (void) ConcatenateString(&path,message);
  (void) FormatMagickString(message,MaxTextExtent,"} bind def");
  (void) ConcatenateString(&path,message);
  message=(char *) RelinquishMagickMemory(message);
  return(path);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   T r a c e S V G C l i p p a t h                                           %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  TraceSVGClipPath() traces a clip path and returns it as SVG.
%
%  The format of the TraceSVGClipPath method is:
%
%      char *TraceSVGClipPath(unsigned char *blob,size_t length,
%        const unsigned long columns,const unsigned long rows)
%
%  A description of each parameter follows:
%
%    o blob: The blob.
%
%    o length: The length of the blob.
%
%    o columns: The image width.
%
%    o rows: The image height.
%
%
*/
static char *TraceSVGClippath(unsigned char *blob,size_t length,
  const unsigned long columns,const unsigned long rows)
{
  char
    *path,
    *message;

  long
    knot_count,
    selector,
    x,
    y;

  MagickBooleanType
    in_subpath;

  PointInfo
    first[3],
    last[3],
    point[3];

  register long
    i;

  path=AcquireString((char *) NULL);
  if (path == (char *) NULL)
    return((char *) NULL);
  message=AcquireString((char *) NULL);
  (void) FormatMagickString(message,MaxTextExtent,
    "<?xml version=\"1.0\" encoding=\"iso-8859-1\"?>\n");
  (void) ConcatenateString(&path,message);
  (void) FormatMagickString(message,MaxTextExtent,
    "<svg width=\"%lu\" height=\"%lu\">\n",columns,rows);
  (void) ConcatenateString(&path,message);
  (void) FormatMagickString(message,MaxTextExtent,"<g>\n");
  (void) ConcatenateString(&path,message);
  (void) FormatMagickString(message,MaxTextExtent,
    "<path style=\"fill:#00000000;stroke:#00000000;");
  (void) ConcatenateString(&path,message);
  (void) FormatMagickString(message,MaxTextExtent,
    "stroke-width:0;stroke-antialiasing:false\" d=\"\n");
  (void) ConcatenateString(&path,message);
  knot_count=0;
  in_subpath=MagickFalse;
  while (length != 0)
  {
    selector=ReadMSBShort(&blob,&length);
    switch (selector)
    {
      case 0:
      case 3:
      {
        if (knot_count != 0)
          {
            blob+=24;
            length-=24;
            break;
          }
        /*
          Expected subpath length record.
        */
        knot_count=ReadMSBShort(&blob,&length);
        blob+=22;
        length-=22;
        break;
      }
      case 1:
      case 2:
      case 4:
      case 5:
      {
        if (knot_count == 0)
          {
            /*
              Unexpected subpath knot.
            */
            blob+=24;
            length-=24;
          }
        else
          {
            /*
              Add sub-path knot
            */
            for (i=0; i < 3; i++)
            {
              y=ReadMSBLong(&blob,&length);
              x=ReadMSBLong(&blob,&length);
              point[i].x=(double) x*columns/4096/4096;
              point[i].y=(double) y*rows/4096/4096;
            }
            if (in_subpath == MagickFalse)
              {
                FormatMagickString(message,MaxTextExtent,"M %g,%g\n",
                  point[1].x,point[1].y);
                for (i=0; i < 3; i++)
                {
                  first[i]=point[i];
                  last[i]=point[i];
                }
              }
            else
              {
                if ((last[1].x == last[2].x) && (last[1].y == last[2].y) &&
                    (point[0].x == point[1].x) && (point[0].y == point[1].y))
                  FormatMagickString(message,MaxTextExtent,"L %g,%g\n",
                    point[1].x,point[1].y);
                else
                  FormatMagickString(message,MaxTextExtent,
                    "C %g,%g %g,%g %g,%g\n",last[2].x,last[2].y,
                    point[0].x,point[0].y,point[1].x,point[1].y);
                for (i=0; i < 3; i++)
                  last[i]=point[i];
              }
            (void) ConcatenateString(&path,message);
            in_subpath=MagickTrue;
            knot_count--;
            /*
              Close the subpath if there are no more knots.
            */
            if (knot_count == 0)
              {
                if ((last[1].x == last[2].x) && (last[1].y == last[2].y) &&
                    (first[0].x == first[1].x) && (first[0].y == first[1].y))
                  FormatMagickString(message,MaxTextExtent,"L %g,%g Z\n",
                    first[1].x,first[1].y);
                else
                  {
                    FormatMagickString(message,MaxTextExtent,
                      "C %g,%g %g,%g %g,%g Z\n",last[2].x,last[2].y,
                      first[0].x,first[0].y,first[1].x,first[1].y);
                    (void) ConcatenateString(&path,message);
                  }
                in_subpath=MagickFalse;
              }
          }
          break;
      }
      case 6:
      case 7:
      case 8:
      default:
      {
        blob+=24;
        length-=24;
        break;
      }
    }
  }
  /*
    Return an empty SVG image if the path does not have knots.
  */
  (void) FormatMagickString(message,MaxTextExtent,"\"/>\n");
  (void) ConcatenateString(&path,message);
  (void) FormatMagickString(message,MaxTextExtent,"</g>\n");
  (void) ConcatenateString(&path,message);
  (void) FormatMagickString(message,MaxTextExtent,"</svg>\n");
  (void) ConcatenateString(&path,message);
  message=(char *) RelinquishMagickMemory(message);
  return(path);
}
