/*
	PWAIT$SDA	SDA extention to look at a waiting process

 COPYRIGHT NOTICE

 This software is COPYRIGHT (c) 2004, Ian Miller. ALL RIGHTS RESERVED.

 Released under licence described in aaareadme.txt 

 DISCLAIMER

THIS PACKAGE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/
#define VERSION "V0.A"
#pragma module PWAIT$SDA VERSION

#define __NEW_STARLET 1
#include <arch_defs.h>
#include <cpudef.h>
#include <descrip.h>
#include <dmpdef.h>
#include <dyndef.h>
#include <gen64def.h>
#include <ints.h>
#include <lib$routines.h>
#include <libdtdef.h>
#include <lnmdef.h>
#include <lnmstrdef.h>
#include <statedef.h>
#include <pcbdef.h>
#include <ucbdef.h>
#include <phddef.h>
#include <procstate.h>
#include <rmsdef.h>
#include <ccbdef.h>
#include <jibdef.h>
#include <irpdef.h>
#include <cebdef.h>
#include <dscdef.h>
#include <psldef.h>
#include <rsndef.h>
#include <dcdef.h>
#include <statedef.h>
#include <sda_routines.h>
#include <ssdef.h>
#include <starlet.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stsdef.h>
#include <tqedef.h>

/*  Global variables	*/

int	sda$extend_version = SDA_FLAGS$K_VERSION;

/* local variables	*/
static SDA_FLAGS flags;
static char *statenames[] = 
	{"NULL","COLPG","MWAIT","CEF","PFW","LEF","LEFO","HIB","HIBO",
	 "SUSP","SUSPO","FPQ","COM","COMO","CUR"};
static char *rsname[RSN$_MAX] = 
	{"RW???","RWAST","RWMBX","RWNPG","RWPFF","RWPAG","RWBRK","RWIMG",
	 "RWQUO","RWLCK","RWSWP","RWMPE","RWMPB","RWSCS","RWCLU","RWCAP",
	 "RWCSV","SNPFU","PSXFRK","INNER","KTEXH"};
static char *rsndesc[RSN$_MAX] = 
	{"unknown resource","AST, I/O quota or interlock","Mailbox space",
	"non-paged dynamic memory","page file space","paged dynamic memory",
	"terminal broadcast","image activation interlock",
	"job pooled quota","lock ID","swap file space",
	"modified page list empty","modified page writer busy",
	"System Communication Services","Cluster State Transition",
	"CPU Capability","Cluster Server","Snapshot","Posix Fork Wait",
	"Inner mode access for Kthreads","Exit Handler for Kthread"
	};
static char * ast_modes[] = {"(none)","K","E","KE","S","KS","ES","KES","U","KU","EU","KEU","SU","KSU","ESU","KESU"};

static char *mode[4] = {"U","S","E","K"};
static char *rmode[4] = {"K","E","S","U"};

static char *mutex_names[][3] = 
{{"LNM$AQ_MUTEX","Shared logical name","Q"},
 {"IOC$GQ_MUTEX","I/O Database","Q"},
 {"EXE$GL_CEBMTX","Common event flag block list",NULL},
 {"EXE$GL_PGDYNMTX","Paged pool",NULL},
 {"EXE$GL_GSDMTX","Global Section Descriptor list",NULL},
 {"EXE$GL_ENQMTX","ENQ ?",NULL},
 {"EXE$GL_ACLMTX","ACL ?",NULL},
 {"CIA$GL_MUTEX","System intrusion database",NULL},
 {"EXE$GQ_BASIMGMTX","executive image data structures","Q"},
 {"QMAN$GL_MUTEX","QMAN ?",NULL},
 {"SMP$GL_CPU_MUTEX","CPU database",NULL}
};

#define MAX_MUTEX_NAMES (sizeof(mutex_names)/sizeof(mutex_names[0]))

/* local routines	*/
static void analyze_efw(KTB *ktb,PCB *pcb, PHD *phd);
static void analyze_hib(KTB *ktb,PCB *pcb, PHD *phd);
static void display_ceb(int efc,int efcp);
static void analyze_jibwait(KTB *ktb, PCB *pcb, PHD *phd);
static void analyze_rwast(PCB *curpcb, PCB *pcb, PHD *phd);
static void analyze_rwmbx(KTB *ktb, PHD *phd);
static void analyze_mutex_wait(KTB *ktb, PHD *phd);
static void display_addr(char *name, unsigned long addr);
static void display_qaddr(char *name, uint64 addr);
static void get_procstate(PHD *phd, PROCSTATE *procstate);
unsigned long get_channels(CCB **ctpp, unsigned long *maxchan);
static int display_mb_ucb(struct _ucb *ucbaddr);
static void display_busy_channels(CCB *ccb_table,unsigned long maxchan);
static void display_irpq(unsigned long count, 	IRP *irpqhdr, IRP *irpptr);
static unsigned long display_irp(IRP *irpq, IRP *irp);
static void display_timerq(KTB *ktb);
static void display_time(char *name, unsigned long time);
static void display_io_func(unsigned int func);
static void display_io_sts(unsigned int sts);

/*

	This is the main entry point in SDA extension routine called from 
	within SDA.

	sda$extend	transfer_table, cmd_line, sda_flags

	transfer_table	- pointer to the routine transfer table
	cmd_line	- address of descriptor of the command line passed
	sda_flags	- flags 
*/
void sda$extend (int *transfer_table, 
		 struct dsc$descriptor_s *cmd_line, 
		 SDA_FLAGS sda_flags)
{
	int	ccode, i, t;
	PCB	*curpcb, pcb;
	PHD	*curphd, phd;
	KTB	*tktb, ktb, *ktbvec[PCB$K_MAX_KT_COUNT];
	static int sysdef_flag;
	unsigned short rsn;
	char spcmd[300];

	sda$print("PWAIT !AZ (c) 2004, Ian Miller (miller@encompasserve.org) built on VMS !AZ",
		&VERSION,__VMS_VERSION);

	/* Initialize the table and establish the condition handler */
	sda$vector_table = transfer_table;
	lib$establish(sda$cond_handler);
	flags = sda_flags;

#if DEBUG
	lib$signal(SS$_DEBUG);
#endif
	if (sysdef_flag == FALSE)
	{
	  	ccode = sda$read_symfile("SDA$READ_DIR:SYSDEF", SDA_OPT$M_READ_NOLOG );
	  	if (!(ccode & 1))
	    	{
	    		sda$print("SDA$READ_SYMFILE failed, ccode = !XL", ccode);
	    		return;
		}
	  	sysdef_flag = TRUE;
	}

	/* if a command line was given then give it to SET PROCESS */
	if (cmd_line->dsc$w_length != 0)
	{
		sprintf(spcmd,"SET PROCESS %-*.*s",
			cmd_line->dsc$w_length,cmd_line->dsc$w_length,
			cmd_line->dsc$a_pointer);
		sda$parse_command(spcmd);
	}	  

	/* get current process pcb	*/
	sda$get_current_pcb (&curpcb);			/* get addr	*/
	ccode = sda$getmem(curpcb, &pcb, PCB$K_LENGTH);	/* get pcb	*/
	if (!(ccode & 1))
	{
		sda$print("Error !XL getting PCB",ccode);
		return;
	}

	/* sanity check - we could be looking at a crash dump with corruption*/
	if (pcb.pcb$b_type != DYN$C_PCB)
	{
		sda$print("PCB has invalid type !UB",pcb.pcb$b_type);
		return;
	}

	/* sometimes the process is COM - try again if it is */
	for (i=0; ((sda_flags.sda_flags$v_current) && (pcb.pcb$l_state == SCH$C_COM) && (i<5)); i++)
	{
		sleep(5);
		sda$get_current_pcb (&curpcb);		/* get addr	*/
		sda$reqmem(curpcb, &pcb, PCB$K_LENGTH);	/* get pcb	*/
	}
	if (i>0)
		sda$print("!UL retries to get PCB",i);

	sda$print("Process PID !XL name !AC No. Threads !UL",
                pcb.pcb$l_epid,pcb.pcb$t_lname, pcb.pcb$l_kt_count);
	sda$format_heading("Process PID !XL name !AC No. Threads !UL",
                pcb.pcb$l_epid,pcb.pcb$t_lname, pcb.pcb$l_kt_count);

	/* get the addresses of the kernel thread blocks */
	ktbvec[0] = pcb.pcb$l_initial_ktb;	/* there is always 1 */
	if (pcb.pcb$l_kt_count > 1)
	{
		ccode = sda$getmem(pcb.pcb$l_ktbvec,ktbvec,
			pcb.pcb$l_kt_count*sizeof(ktbvec[0]));
		if (!(ccode & 1))
		{
			sda$print("Error !XL getting KTBVEC",ccode);
			return;
		}
	}
	for (t=0,tktb = ktbvec[t]; t < pcb.pcb$l_kt_count; ++t) 
	{
	ccode = sda$getmem(tktb,&ktb,sizeof(ktb));
	if (!(ccode & 1))
	{
		sda$print("Error !XL getting KTB",ccode);
		return;
	}

	sda$print("Thread !UL: state !AZ AST pending !AZ active !AZ blocked !AZ",
		t+1,statenames[ktb.ktb$l_state],
		ast_modes[ktb.ktb$l_ast_pending],
		ast_modes[ktb.ktb$l_astact],
		ast_modes[ktb.ktb$l_ast_blocked]);

	display_time("Process has been waiting for",ktb.ktb$l_waitime);
	if (ktb.ktb$l_sts & PCB$M_SSRWAIT)
		sda$print("Process thread resource wait is DISABLED");
	else
		sda$print("Process thread resource wait is ENABLED");

	if (ktb.ktb$l_sts & PCB$M_DELPEN)
		sda$print("Process is marked for deletion");

	sda$format_heading("Process PID !XL name !AC Thread !UL",
                ktb.ktb$l_epid,pcb.pcb$t_lname, t+1);

	/* and PHD			*/
	if (ktb.ktb$l_sts & PCB$M_PHDRES)
	{
		ccode = sda$getmem((VOID_PQ)&tktb->ktb$l_phd,&curphd,4); /* get addr */
		if (!(ccode & 1))
		{
			sda$print("Error !XL getting PHD address",ccode);
			return;
		}
		ccode = sda$getmem(curphd, &phd, PHD$K_LENGTH); /* get phd  */
		if (!(ccode & 1))
		{
			sda$print("Error !XL getting PHD",ccode);
			return;
		}
	}


	switch(ktb.ktb$l_state)
	{
	case SCH$C_MWAIT:
		/* misc wait state or mutex wait	*/
		if (ktb.ktb$l_efwm == (int)ktb.ktb$l_jib)
		{
			analyze_jibwait(&ktb,&pcb,&phd);
		}
		else if ((rsn=ktb.ktb$l_efwm&0xFFFF) < RSN$_MAX)
		{
			/* resource wait	*/
			sda$print("!AZ - waiting for !AZ",
				rsname[rsn],
				rsndesc[rsn]);
			switch(rsn)
			{
			case RSN$_ASTWAIT:
				/* RWAST 	*/
				analyze_rwast(curpcb,&pcb,&phd);
				break;
			case RSN$_MAILBOX:
				/* RWMBX	*/
				analyze_rwmbx(&ktb,&phd);
				break;
			case RSN$_NPDYNMEM:
			case RSN$_PGDYNMEM:
				/* RWNPG/RWPAG	*/
				sda$parse_command("SHOW MEMORY/POOL");
				break;
			case RSN$_PGFILE:
			case RSN$_SWPFILE:
			case RSN$_MPLEMPTY:
			case RSN$_MPWBUSY:
				sda$parse_command("SHOW MEMORY/FILE");
				break;
			}
		}
		else
		{
			/* mutex wait	       		*/
			analyze_mutex_wait(&ktb,&phd);
		}
		break;
	case SCH$C_CEF:
	case SCH$C_LEF:
	case SCH$C_LEFO:
		analyze_efw(&ktb,&pcb,&phd);
		break;
	case SCH$C_HIB:
	case SCH$C_HIBO:
		analyze_hib(&ktb,&pcb,&phd);
		break;
	default:
		/* not a wait state we are interested in */	
	}

	}
	return;
}

/*
	analyze_efw - display info about a process in LEF or CEF
*/
static void
analyze_efw(KTB *ktb, PCB *pcb, PHD *phd)
{
	unsigned long ccode, maxchan;
	int 	i, pos, efn;
	unsigned char size;
	char	outbuf[200], *p;
	CCB *ccb_table;

	sda$print("EFWM !XL WEFC !UL Local Event flags !XL !XL",
		ktb->ktb$l_efwm,ktb->ktb$l_wefc,
		phd->phd$l_lefc_1, phd->phd$l_lefc_0);

	if (ktb->ktb$l_wefc < 4)
	{
	/* find out which event flag(s) process is waiting for */
	p = outbuf;
	pos = 0;
	size = 32;
	ccode = 1;
	while ((pos < size) && (ccode &1))
	{
		ccode = LIB$FFC(&pos,&size, &ktb->ktb$l_efwm, &efn);
		if ((ccode & 1) && (efn < size))
		{
			if (pos == 0)   /* if first EFN		*/
			{
				sprintf(p,"%d",efn+ktb->ktb$l_wefc*32);
			}
			else 		/* if not first EFN 	*/
			{
				if (ktb->ktb$l_sts & PCB$M_WALL)
				{
					sprintf(p," AND %d",
						efn+ktb->ktb$l_wefc*32);
				}
				else
				{
					sprintf(p," OR %d",
						efn+ktb->ktb$l_wefc*32);
				}
			} 
			p += strlen(p);
		}
		pos = efn + 1; 	/* resume search 1 bit after */
	} 
	*p = '\0';
	sda$print("waiting for event flag !AZ",outbuf);
	}
	if (ktb->ktb$l_wefc == 2)
	{
		/* display details of Common event block 2 */
		display_ceb(ktb->ktb$l_wefc,pcb->pcb$l_efc2p);
	}
	else if (ktb->ktb$l_wefc == 3)
	{
		/* display details of Common event block 3 */
		display_ceb(ktb->ktb$l_wefc,pcb->pcb$l_efc3p);
	}
	/* now should try and work out what could set the EF */

// could be timer or I/O or something else 	*/
// could look at active timers to see if they are due to set this EF
// could look at busy channels to see if I/O completion are due to set EF

	/* get the channel table	*/
	ccode = get_channels(&ccb_table,&maxchan);
	if (ccode & 1)
	{
		display_busy_channels(ccb_table,maxchan);
	}
	display_timerq(ktb);
}

/*
	display_ceb - Display the specified common event block

*/
static void 
display_ceb(int efc, int efcp)
{
	unsigned long ccode;
	CEB *cebptr, ceb;

	if (efcp == 0)
		return; /*  nothing to do */

	ccode = sda$getmem((VOID_PQ)&efcp,&cebptr,4); /* get addr */
	if (!(ccode & 1))
	{
		sda$print("Error !XL getting CCB addr !XL",ccode,efcp);
		return;
	}
	ccode = sda$getmem(cebptr, &ceb, CEB$C_LENGTH);		   
	if (!(ccode & 1))
	{
		sda$print("Error !XL getting CCB !XL",ccode,cebptr);
		return;
	}

	if (ceb.ceb$b_type != DYN$C_CEB)
	{
		sda$print("cluster !UL invalid CEB block type !UB",efc,ceb.ceb$b_type);
		return;
	}
	sda$print("EFC !UL !AZ Common Event Flag Cluster !AC Creator PID %XL UIC [!OW,!OW]",
		efc,(ceb.ceb$v_perm)?"Permanent":"Temporary",
		ceb.ceb$t_efcnam,ceb.ceb$l_pid,ceb.ceb$w_grp,ceb.ceb$l_uic & 0xFFFF);
	sda$print("	flags !XL",ceb.ceb$l_efc);
}

/*
	analyze_hib - display info about a process in HIB
*/
static void
analyze_hib(KTB *ktb, PCB *pcb, PHD *phd)
{
	unsigned long ccode, maxchan;
	CCB *ccb_table;

	/* get the channel table	*/
	ccode = get_channels(&ccb_table,&maxchan);
	if (ccode & 1)
	{
		display_busy_channels(ccb_table,maxchan);
	}
	display_timerq(ktb);
}

/*
	analyze_jibwait - display info about a process waiting for a pooled quota
*/
static void 
analyze_jibwait(KTB *ktb, PCB *pcb, PHD *phd)
{
	unsigned long ccode;
	JIB *curjib, jib;
	CCB *ccb_table;
	unsigned long maxchan;

	/* pooled quota wait		*/

	/* get jib        		*/
	ccode = sda$getmem(ktb->ktb$l_jib, &jib, JIB$K_LENGTH);		 
	if (!(ccode & 1))
	{
		sda$print("Error !XL getting JIB",ccode);
		return;
	}
	if (jib.jib$b_type != DYN$C_JIB)
	{
		sda$print("JIB has invalid type !UB",jib.jib$b_type);
		return;
	}

	/* check jib flags to see which	*/
	if (jib.jib$l_flags & JIB$M_BYTCNT_WAITERS)
	{
		/* job buffered I/O count exhausted */
		sda$print("waiting for buffered I/O byte count - Available !UL Quota !UL",
			jib.jib$l_bytcnt, jib.jib$l_bytlm);
	}
	if (jib.jib$l_flags & JIB$M_TQCNT_WAITERS)
	{
		/* job timer quota exhausted	*/
		sda$print("waiting for timer quota (TQ count !UL TQ Limit !UL)",
			jib.jib$l_tqcnt,jib.jib$l_tqlm);
	}
	if (jib.jib$l_filcnt == 0)
	{
		sda$print("waiting for open file quota (FILLM !UL)",
			jib.jib$l_fillm);
	}
	if (jib.jib$l_pgflcnt == 0)
	{
		sda$print("waiting for page file quota (PGFLQOTA !UL)",
			jib.jib$l_pgflquota);
	}
	/* get the channel table	*/
	ccode = get_channels(&ccb_table,&maxchan);
	if (ccode & 1)
	{
		display_busy_channels(ccb_table,maxchan);
	}
	display_timerq(ktb);
}

/* 
	analyze_rwast 
*/ 
static void  
analyze_rwast(PCB *curpcb, PCB *pcb, PHD *phd)
{ 
	int c;
	CCB *ccb_table;
	PROCSTATE pstate;
	unsigned long ccode, maxchan;
	char outbuf[300];

	/* check quotas	*/ 
	if (pcb->pcb$l_biocnt == 0) 
	{ 
		sda$print("process has exhausted buffered I/O quota (count !UL limit !UL)", 
			pcb->pcb$l_biocnt,pcb->pcb$l_biolm); 
	} 
	else if ((c = pcb->pcb$l_biocnt - pcb->pcb$l_biolm) > 0)
	{ 
		sda$print("process has !UL outstanding buffered I/O operations", 
			c); 
	} 
	if (pcb->pcb$l_diocnt == 0) 
	{ 
		sda$print("process has exhausted direct I/O quota (count !UL limit !UL)", 
			pcb->pcb$l_diocnt, pcb->pcb$l_diolm); 
	} 
	else if ((c = pcb->pcb$l_diocnt - pcb->pcb$l_diolm) > 0) 
	{ 
		sda$print("process has !UL outstanding direct I/O operations", 
			c); 
	} 
	if (pcb->pcb$l_prccnt != 0) 
	{ 
		sda$print("process has !UL subprocess!1%C!%Ees!%F", 
			pcb->pcb$l_prccnt); 
	} 
	if (pcb->pcb$l_dpc != 0) 
	{ 
		sda$print("process has !UL outstanding XQP events!%S", 
			pcb->pcb$l_dpc); 
	}
	if (pcb->pcb$l_sts & PCB$M_PHDRES)
	{
		/* check for outstanding ASTs	*/
		c = phd->phd$l_astlm - pcb->pcb$l_astcnt;
		if (c > 0)
		{
			sda$print("process has !UL outstanding AST!%S (ASTCNT !UL ASTLM !UL)",
				c,pcb->pcb$l_astcnt,phd->phd$l_astlm);
		}		
#if __alpha
		/* display process waiting pc			*/
		get_procstate(phd,&pstate);
		display_qaddr("process waiting pc",pstate.pstate$q_pc);
#endif
	}
	/* get the channel table	*/
	ccode = get_channels(&ccb_table,&maxchan);
	if (ccode & 1)
	{
		display_busy_channels(ccb_table,maxchan);
	}
}

/*
	analyze_rwmbx
*/
static void 
analyze_rwmbx(KTB *ktb, PHD *phd)
{
	PROCSTATE pstate;
	CCB *ccb_table;
	unsigned long ccode, maxchan, i, j, chan, nreg;
	uint64 *rptr;

	if (ktb->ktb$l_sts & PCB$M_SSRWAIT)
		sda$print("Process thread resource wait is DISABLED");
	else
		sda$print("Process thread resource wait is ENABLED");

	if (ktb->ktb$l_sts & PCB$M_DELPEN)
		sda$print("Process is marked for deletion");

	/* try and work out which mailbox	*/
	/* on a vax then in the PHD r3=IRP, R5=mailbox UCB	*/
	/* on an alpha then R? in the PHD = mailbox channel	*/
	/* on an itanium ?					*/

#if __alpha
	/* get the saved regs 		*/
	get_procstate(phd,&pstate);	
	
	/* get the channel table	*/
	ccode = get_channels(&ccb_table,&maxchan);
	if (!(ccode & 1))
		return;

/*	
	for each valid channel (amod !=0, ucb != 0) compare channel number
	against the registers. 
	if a match is found then get the UCB using address in ccb
	if its a valid ucb (check btype) and its a mailbox
	display the device name, iniquo, bufquo
*/

	nreg = sizeof(pstate)/sizeof(uint64); /* number of saved regs */	
	for (i=0; i < maxchan; i++)
	{
		/* if ccb valid			*/
		if ((ccb_table[i].ccb$b_amod ==0)
		||  (ccb_table[i].ccb$l_ucb == NULL))
			continue;	/* skip unused ones */
		if (ccb_table[i].ccb$l_wind != NULL)
			continue;	/* skip if file structured device */

		/* get channel			*/
		chan = ccb_table[i].ccb$l_chan;

		/* check agains saved regs	*/
		for (j=0, rptr = &pstate.pstate$q_r8; j < nreg; j++, rptr++)
		{
			if (*rptr == chan)
			{
				/* a match !	*/
				if (display_mb_ucb(ccb_table[i].ccb$l_ucb))
				{
					/* quit on successful display*/
					i=maxchan;
					break;	
				}
			}
		}
	}

	sda$deallocate((void *)ccb_table,maxchan*CCB$K_LENGTH);
#endif
#if __ia64
	/* get the channel table	*/
	ccode = get_channels(&ccb_table,&maxchan);
	if (ccode & 1)
	{
		display_busy_channels(ccb_table,maxchan);
		sda$deallocate((void *)ccb_table,maxchan*CCB$K_LENGTH);
	}
#endif
}

/*
	display_mb_ucb - display a UCB if its a mailbox

	returns TRUE if ok, FALSE if not
*/
static int
display_mb_ucb(struct _ucb *ucbaddr)
{
	static MB_UCB *ucb;
	unsigned long ccode;
	static LNMB	*lnmb;
	static int	lnmb_size;
	int new_lnmb_size;

	if (ucb == NULL) 
		sda$allocate(UCB$K_MB_UCBLENGTH, (void *)&ucb);

	if (lnmb == NULL)
	{
		sda$allocate(sizeof (LNMB),(void *)&lnmb);
		lnmb_size = sizeof (LNMB);
	}

	ccode = sda$trymem(ucbaddr,ucb,UCB$K_MB_UCBLENGTH);
	if (!(ccode & 1))
		return FALSE;

	/* if is not a mailbox quit	*/
	if (ucb->ucb$r_ucb.ucb$b_devclass != DC$_MAILBOX)
		return FALSE;

	/*
		If the mailbox has a logical name associated with it, read in
		the logical name block (LNMB). Since we don't know the exact
		size of it then read the fixed header which contains
		the size, followed by another read of the whole LNMB into local
		memory.
	*/
	if (ucb->ucb$l_mb_logadr != NULL)
	{
		ccode = sda$trymem(ucb->ucb$l_mb_logadr,lnmb,sizeof(LNMB));
		if (!(ccode & 1))
			return FALSE;
		new_lnmb_size = lnmb->lnmb$w_size;
		/* make sure we have a big enough LNMB	*/
	      	if (new_lnmb_size > lnmb_size)
		{
			sda$deallocate((void*)lnmb,lnmb_size);
			sda$allocate(new_lnmb_size,(void *)&lnmb);
			lnmb_size = new_lnmb_size;
		}
	      	ccode  = sda$trymem(ucb->ucb$l_mb_logadr,lnmb,new_lnmb_size);
		if (!(ccode & 1))
			return FALSE;
	}
	else
		lnmb->lnmb$l_namelen = 0;
#ifdef UCB$L_MB_INIQUO
	sda$print("MBA!UW !AF msgcnt = !UW, quota !UL, remaining = !UL, rdr count !UL, wtr count !UL",
			ucb->ucb$r_ucb.ucb$w_unit,
			lnmb->lnmb$l_namelen, 
			&lnmb->lnmb$t_name,
			ucb->ucb$r_ucb.ucb$w_msgcnt,
			ucb->ucb$l_mb_iniquo, 
			ucb->ucb$l_mb_bufquo,
			ucb->ucb$l_mb_readerrefc,
			ucb->ucb$l_mb_writerrefc);
#else
	sda$print("MBA!UW !AF msgcnt = !UW, quota !UW, remaining = !UW, rdr count !UL, wtr count !UL",
			ucb->ucb$r_ucb.ucb$w_unit,
			lnmb->lnmb$l_namelen, 
			&lnmb->lnmb$t_name,
			ucb->ucb$r_ucb.ucb$w_msgcnt,
			ucb->ucb$r_ucb.ucb$w_iniquo, 
			ucb->ucb$r_ucb.ucb$w_bufquo,
			ucb->ucb$l_mb_readerrefc,
			ucb->ucb$l_mb_writerrefc);
#endif
	if (ucb->ucb$r_ucb.ucb$l_refc == 1)
		sda$print("no other process is accessing this mailbox");
	return TRUE;	/* report the good news	*/
}

/*
	analyze_mutex_wait
*/
static void 
analyze_mutex_wait(KTB *ktb, PHD *phd)
{
	GENERIC_64 addr;
	int i, found = FALSE;
	unsigned long ccode;
	short lmutex[2];
	long qmutex[2];

	display_addr("	waiting for mutex",ktb->ktb$l_efwm);

/* 	use table of known mutex names, get value of each one and compare
	with mutex that process is waiting for. If known muxtex then say so
*/
	for (i=0;i < 0; i++)
	{
		ccode = sda$symbol_value(mutex_names[i][0],(uint64 *)&addr);
		if (ccode & 1)
		{
			if (addr.gen64$l_longword[0] == ktb->ktb$l_efwm)
			{
				found = TRUE;
				break;
			}
		}		
        }
	if (found)
	{
		if (addr.gen64$l_longword[0] & 0x80000000)
			addr.gen64$l_longword[1] = 0xFFFFFFFF;
		else
			addr.gen64$l_longword[1] = 0;
		if (mutex_names[i][2] == NULL)
		{
			/* longword mutex	*/
			ccode = sda$trymem((VOID_PQ *)addr.gen64$q_quadword, 
				lmutex, 
				sizeof(lmutex));
			if (ccode & 1)
			{
				sda$print("	!AZ !AZ count=!ZW !AZ",
					mutex_names[i][0],
					mutex_names[i][1],
					lmutex[0],
					(lmutex[1]&1)?"writer":"nowriter");
			}
			else
			{
				sda$print("	!AZ !AZ",mutex_names[i][0],
					mutex_names[i][1]);
			}
		}
		else
		{
			/* quadword mutex	*/
			ccode = sda$trymem((VOID_PQ *)addr.gen64$q_quadword, 
				qmutex, 
				sizeof(qmutex));
			if (ccode & 1)
			{
				sda$print("	!AZ !AZ count=!ZL !AZ",
					mutex_names[i][0],
					mutex_names[i][1],
					qmutex[0],
					(qmutex[1]&1)?"writer":"nowriter");
			}
			else
			{
				sda$print("	!AZ !AZ",mutex_names[i][0],
					mutex_names[i][1]);
			}
		}
	}
}

/*
	display_addr - display an address with a message and its symbol name if available
*/
static void
display_addr(char *name, unsigned long addr)
{
	unsigned long ccode;
	char outbuf[300];
	GENERIC_64 value;

	/* copy and sign extend addr 	*/
	value.gen64$l_longword[0] = addr;
	if (value.gen64$l_longword[0] & 0x80000000)
		value.gen64$l_longword[1] = 0xFFFFFFFF;
	else
		value.gen64$l_longword[1] = 0;

	ccode = sda$symbolize(value.gen64$q_quadword, outbuf, sizeof(outbuf));
	if (ccode & 1)
	{
		sda$print("!AZ !XL !AZ",name,addr,outbuf);
	}
	else
	{
		sda$print("!AZ !XL", name,addr);
	}	
}

/*
	display_qaddr - display an address with a message and its symbol name if available
*/
static void
display_qaddr(char *name, uint64 addr)
{
	unsigned long ccode;
	char outbuf[300];
	GENERIC_64 value;

	value.gen64$q_quadword = addr;
	ccode = sda$symbolize(addr, outbuf, sizeof(outbuf));
	if (ccode & 1)
	{
		sda$print("!AZ !XL.!XL !AZ",name,
			value.gen64$l_longword[1],value.gen64$l_longword[0],
			outbuf);
	}
	else
	{
		sda$print("!AZ !XL.!XL", name,
			value.gen64$l_longword[1],value.gen64$l_longword[0]);
	}	
}

/*
	get_registers - get the process saved regs
*/
static void
get_procstate(PHD *phd, PROCSTATE *procstate)
{
/*
	for alpha some registers are saved on the Kernal stack
	and others are in the PHD (floating point, stack pointers)
*/

	/* get the saved process state from the kernal stack */
	sda$reqmem((VOID_PQ)phd->phd$q_ksp,procstate,PSTATE$K_LENGTH);
}

/*
	get_channels - copy the channel table

	ctpp is the address of a ptr to the allocated memory containing
		a copy of the channel table.
	maxchan is a ptr to a longword for the max channel number in use.
*/
unsigned long
get_channels(CCB **ctpp, unsigned long *maxchan)
{
	CCB *ctp;
	unsigned long ccode, chindx;
	VOID_PQ	ctl$ga_ccb_table, ctl$gl_chindx;
	GENERIC_64 ccb_table;

	/* address of start of channel table	*/
	ccode = sda$symbol_value("CTL$GA_CCB_TABLE",(uint64 *)&ctl$ga_ccb_table);
	if (!(ccode & 1))
	{
		sda$print("can't get channel table start address");
		return ccode;
	}

	ccode = sda$getmem(ctl$ga_ccb_table,&ccb_table.gen64$l_longword[0],4);
	if (!(ccode & 1))
	{
		sda$print("can't get channel table start address");
		return ccode;
	}
	/* validate address	*/
	if ((ccb_table.gen64$l_longword[0] == 0)
	||  (ccb_table.gen64$l_longword[0] >= 0x80000000))
	{
		sda$print("can't get channel table");
		return 0;
	}

	/* get channel high water mark	*/
	ccode = sda$symbol_value("CTL$GL_CHINDX",(uint64 *)&ctl$gl_chindx);
	if (!(ccode & 1))
	{
		sda$print("can't get channel table size");
		return ccode;
	}
	ccode = sda$getmem(ctl$gl_chindx,&chindx,4);
	if (!(ccode & 1))
	{
		sda$print("can't get channel table size");
		return ccode;
	}
	/* allocate memory for channel table	*/
	sda$allocate(chindx*CCB$K_LENGTH,(void *)&ctp);

	/* copy table				*/
	ccode = sda$getmem((VOID_PQ)ccb_table.gen64$q_quadword,ctp,chindx*CCB$K_LENGTH);
	if (ccode & 1)
	{
		*ctpp = ctp;
		*maxchan = chindx;
	}
	else
	{
		sda$deallocate(ctp,chindx*CCB$K_LENGTH);
		*ctpp = NULL;
		*maxchan = 0;
	}

	return ccode;
}

/*
	display_busy_channels

	INPUT
*/
static void
display_busy_channels(CCB *ccb_table,unsigned long maxchan)
{
	int i,c,u;
	char device_name[32], outbuf[300];
	unsigned long ccode, irpoff;
	UCB *ucbptr, ucb;
	IRP *irpq, irp, *irpqhdr;

	for (i=0,u=0,c=0; i < maxchan; i++)
	{
		/* if ccb valid			*/
		if ((ccb_table[i].ccb$b_amod ==0)
		||  (ccb_table[i].ccb$l_ucb == NULL))
			continue;	/* skip	*/
		u++;	/* one more inuse */
		if (ccb_table[i].ccb$l_ioc == 0) 
			continue;	/* skip	idle 	*/
		c++;		/* one more busy	*/
		if (c==1)	/* if first	*/
		{
			/* output header */
			sda$print("!/Process busy channels!/_____________________");
			sda$print("Index  Chan     Mode I/O cnt WIND     Device");
			sda$format_heading("Index  Chan     Mode I/O cnt WIND     Device");
		}
		/* get device name	*/
		ccode = sda$get_device_name(ccb_table[i].ccb$l_ucb,
			device_name, sizeof(device_name));
		if (!(ccode & 1))
			device_name[0] = '\0';		/* no name	*/

// should get the WCB and display the FID (or name using LIB$FID_TO_NAME?)
// if wind < 0x80000000 then its a process section index
// but SDA SHOW CHANNEL command does this so is it worth the trouble

		sprintf(outbuf,"%-6d %08x    %s %7d %08x %s",
			i,ccb_table[i].ccb$l_chan,
			mode[ccb_table[i].ccb$b_amod & 3],
			ccb_table[i].ccb$l_ioc,
			ccb_table[i].ccb$l_wind,
			device_name);
		sda$print(outbuf);

		ucbptr = ccb_table[i].ccb$l_ucb;
		ccode = sda$getmem((VOID_PQ)ucbptr, &ucb, sizeof(UCB));
		if (!(ccode & 1))
		{
			sda$print("error getting UCB");
			continue;
		}
		if (ucb.ucb$l_irp)
		{
			sda$print("!/    Current IRP for this chan is @ !XL!/    _______________________________________",
				ucb.ucb$l_irp);
			display_irp(ucb.ucb$l_irp, &irp);
		}
		if (ucb.ucb$l_qlen != 0)
		{
			/* determine address of irpq */
			irpoff = (char *)&ucb.ucb$l_ioqfl - (char *)&ucb;
			irpqhdr = (IRP *)((char *)ucbptr + irpoff);
			sda$print("!/    I/O Queue for this chan!/    _______________________");
			display_irpq(ucb.ucb$l_qlen,irpqhdr,ucb.ucb$l_ioqfl);
		}
		if (ucb.ucb$b_devclass == DC$_MAILBOX)
			display_mb_ucb(ucbptr);

// here could look at other queues in UCB depending on class of device
// may have to get extended UCB. For mailbox devices would have to scan
// message queue and get irp address from message buffer header

	}
	sda$print("process has !UL channel!%S !UL of which are busy",u,c);
}

/*
	display_irpq - display a queue of IRP
*/
static void
display_irpq(unsigned long count, IRP *irpq, IRP *irpptr)
{
	unsigned long ccode;
	int j, maxirp;
	IRP irp;

	/* display up to 5	*/
	if (count > 5)
	{
		maxirp = 5;
		sda$print("displaying first 5 of !UL",count);
	}
	else
	{
		maxirp = count;
	}

	for (j=0; (j < maxirp); j++)
	{
		if (irpptr == irpq)
			break;
		ccode = display_irp(irpptr,&irp);
		irpptr = irp.irp$l_ioqfl;
	}
}

/*
	display_irp	- display one irp

	INPUT
		irpptr - address of the irp
		irp	- address of a IRP structure to use
*/
static unsigned long
display_irp(IRP *irpptr, IRP *irp)
{
	unsigned long ccode;

	ccode = sda$getmem((VOID_PQ)irpptr, irp, sizeof(IRP));
	if (!(ccode & 1))
	{
		sda$print("**** error getting IRP");
		return ccode;
	}
	/* sanity check structure type	*/
	if (irp->irp$b_type != DYN$C_IRP)
	{
		return 0;	/* its not a IRP */
	}
	sda$print("    IRP @!XL: tPID !XL chan !XL mode !AZ EFN !UB FUNC !XL",
		irpptr,irp->irp$l_thread_pid,irp->irp$l_chan,
		rmode[irp->irp$v_mode],
		irp->irp$b_efn,
		irp->irp$l_func);
	display_io_func(irp->irp$l_func);
	display_io_sts(irp->irp$l_sts);
	sda$print("        P1-6 !XL !XL !XL !XL !XL !XL",
		irp->irp$l_qio_p1,irp->irp$l_qio_p2,
		irp->irp$l_qio_p3,irp->irp$l_qio_p4,
		irp->irp$l_qio_p5,irp->irp$l_qio_p6);

	display_qaddr("        AST",(uint64)irp->irp$pq_acb64_ast);
	display_qaddr("        ASTPRM",irp->irp$q_acb64_astprm);
	return 1;
}

/*
	display the name of the io function
*/
static void 
display_io_func(unsigned int func)
{
	unsigned long fcode;

	fcode = func & IRP$M_FCODE;
}

/*
	display the translation of the io request status
*/
static void 
display_io_sts(unsigned int sts)
{
	static readonly struct
	{
		unsigned int value;
		char *name;
	} stsnames[] =
	{{IRP$M_BUFIO,"bufio"},
	 {IRP$M_FUNC,"read-func"},
	 {IRP$M_PAGIO,"pagio"},
	 {IRP$M_COMPLX,"complx"},
	 {IRP$M_VIRTUAL,"virtual"},
	 {IRP$M_CHAINED,"chained"},
	 {IRP$M_SWAPIO,"swapio"},
	 {IRP$M_DIAGBUF,"diagbuf"},
	 {IRP$M_PHYSIO,"physio"},
	 {IRP$M_TERMIO,"termio"},
	 {IRP$M_MBXIO,"mbxio"},
	 {IRP$M_EXTEND,"extend"},
	 {IRP$M_FILACP,"filacp"},
	 {IRP$M_MVIRP,"mvirp"},
	 {IRP$M_SRVIO,"srvio"},
	 {IRP$M_CCB_LOOKED_UP,"ccb_looked_up"},
	 {IRP$M_CACHE_PAGIO,"cache_pagio"},
	 {IRP$M_BUFOBJ,"bufobj"},
	 {IRP$M_TRUSTED,"trusted"},
	 {IRP$M_FASTIO_DONE,"fastio_done"},
	 {IRP$M_FASTIO,"fastio"},
	 {IRP$M_FAST_FINISH,"fast_finish"},
	 {IRP$M_DOPMS,"dopms"},
	 {IRP$M_HIFORK,"hifork"},
	 {IRP$M_SRV_ABORT,"srv_abort"},
	 {IRP$M_LOCK_RELEASEABLE,"lock_releasable"},
	 {IRP$M_DID_FAST_FDT,"did_fast_fdt"},
	 {IRP$M_SYNCSTS,"syncsts"},
	 {IRP$M_FINIPL8,"finipl8"},
	 {IRP$M_FILE_FLUSH,"file_flush"},
	 {IRP$M_BARRIER,"barrier"}
	};
#define MAX_STSNAMES (sizeof(stsnames)/sizeof(stsnames[0]))
	char line[133], *p;
	int i, len, namelen;

	/* setup to make a line of names of the status flags that are set */
	line[0] = '\t';
	p = &line[1];
	len = sizeof(line) - 2;

	/* scan the table adding names for flags that are set 	*/
	for (i=0; i < MAX_STSNAMES; i++)
	{
		/* if the flag is set */
		if (sts & stsnames[i].value)
		{
			namelen = strlen(stsnames[i].name);
			if (namelen < len) 	/* if there is enough room */
			{
				strcpy(p, stsnames[i].name);
				p += namelen;
				*p++ = ' ';
				len -= namelen+1;
			}
			else
			{
				break;		/* line is full */
			}
		}
	}	
	*p = '\0';	/* terminate the line	*/
	if (line[1] != '\0')	/* if there is something to output */
	{
		sda$print(line);
	}
}

/*
	display_timerq
*/
static void
display_timerq(KTB *ktb)
{
	VOID_PQ exe$gl_tqfl;
	TQE *tqfl, *tqeptr, tqe;
	unsigned long ccode, count;
		
	if (strcmp(__VMS_VERSION+1,"7.3-1  ") >= 0) /* V7.3-1 or later */
	{
		sda$parse_command("SHOW PROCESS/TQE=ALL");
	}
	else
	{
		/* if working on live system and have therefore time or synchronistion issues */
		if (flags.sda_flags$v_current)
		{
			sda$print("**** Can't display TQEs for live system");
			return;
		}
		/* walk the TQE list displaying entries for this process */
		ccode = sda$symbol_value("EXE$GL_TQFL",(uint64 *)&exe$gl_tqfl);
		if (!(ccode & 1))
		{
			sda$print("**** can't get address of tqe list");
			return;
		}
		ccode = sda$getmem(exe$gl_tqfl,&tqfl,4);
		if (!(ccode & 1))
		{
			sda$print("**** can't get address of tqe list");
			return;
		}
		for (count=0,tqeptr=tqfl; tqeptr != tqfl; tqeptr=tqe.tqe$l_tqfl)
		{
			ccode = sda$getmem(tqeptr,&tqe,sizeof(tqe));
			if (!(ccode & 1))
				break;	/* give up on error 	*/
			/* validate TQE block type */
			if (tqe.tqe$b_type != DYN$C_TQE)
			{
				sda$print("unknown TQE type !XB in TQE @ !XL",
					tqe.tqe$b_type,tqeptr);
				break;
			}
			if (tqe.tqe$v_tqtype & 1)
				continue;	/* skip subroutine entries */
			if (tqe.tqe$l_pid != ktb->ktb$l_epid)
				continue;	/* skip entries for other processes */
			if (count == 0)
			{
				sda$print("TQE      Type Mode EFN ASTPRM AST");
			}
			count++;
			sda$print("!X  !XB !AZ !3UB !XL !XL",
				tqeptr, tqe.tqe$b_rqtype, 
				rmode[tqe.tqe$l_rmod&3],
				tqe.tqe$l_efn,
				tqe.tqe$l_astprm,
				tqe.tqe$l_ast);
		}
	}
}

/*
	display_time
*/
static void 
display_time(char *name, unsigned long time)
{
	VOID_PQ exe$gl_abstim_tics;
	unsigned long ccode, abstim_tics;
	float t;
	GENERIC_64 qt;
	static unsigned int m=100000, op = LIB$K_DELTA_SECONDS_F, cvtflag=1;
	unsigned short timlen;
	static char timbuf[64];
	$DESCRIPTOR(timbuf_dsc,timbuf);

	ccode = sda$symbol_value("EXE$GL_ABSTIM_TICS",
		(uint64 *)&exe$gl_abstim_tics);
	if (!(ccode & 1))
	{
		sda$print("error !XL getting abstim_tics",ccode);
		return;
	}
	ccode = sda$getmem(exe$gl_abstim_tics,&abstim_tics,sizeof(abstim_tics));
	t = (abstim_tics - time)/100.0; /* number of seconds */
	ccode = LIB$CVTF_TO_INTERNAL_TIME(&op, &t, &qt.gen64$q_quadword);
	if (!(ccode & 1))
	{
		sda$print("error !XL converting time 0x!XL",ccode,time);
		return;
	}
	ccode = sys$asctim(&timlen,&timbuf_dsc,&qt,1);
	if (!(ccode & 1))
	{
		sda$print("error !XL converting time",ccode);
		return;
	}
	timbuf[timlen] = '\0';
	sda$print("!AZ !AZ",name,timbuf);	
}
