/* 
   Unix SMB/Netbios implementation.
   Version 2.0.
   Copyright (C) Andrew Tridgell 1994-1999

   This file is part of the port to OpenVMS
   Stream read and open routines.
   Copyright (C) Eckart Meyer 1996-1999
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/*	stm_read.c
 *	V1.1			28-Feb-1999
 *
 *	Copyright E. Meyer <meyer@ifn.ing.tu-bs.de>
 *+
 * This module contains replacement routines for OpenVMS file handling
 * routines
 *
 *	open, close, read, lseek
 *
 * to read record oriented (text) files as if they were stream files
 * (CR and LF as record terminators), maintaining the stream position.
 *
 * The record terminators are appended to each record read from VMS.
 * 'lseek' returns the stream position (counting all record bytes plus
 * the terminator bytes). Thus, seeking means reading the file and
 * performance may be bad. Seeking backwards even means rewinding and
 * reading from the beginning.
 *-
 */

/*
 * in case the original routines have been previously redirected,
 * we call those by defining names preceded by an underscore.
 * This is usefull if e.g. read and close are redirected to a socket
 * library like SOCKETSHR.
 */

#include "includes.h"
#include <descrip.h>
#include <fab.h>
#include <rab.h>
#include <rmsdef.h>
#include <prvdef.h>

#ifdef _open
#define open _open
#endif
#ifdef _read
#define read _read
#endif
#ifdef _close
#define close _close
#endif
#ifdef _lseek
#define lseek _lseek
#endif
#ifdef open
#undef open
#endif
#ifdef close
#undef close
#endif
#ifdef read
#undef read
#endif
#ifdef lseek
#undef lseek
#endif
#ifdef write
#undef write
#endif
#undef strcpy

#define const
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <stat.h>
#if __DECC_VER < 50200000  ||  __VMS_VER < 70000000
#include <unixio.h>
#endif
#include <string.h>
#include <fcntl.h>
#include <errno.h>

#ifndef TRUE
#define TRUE 1
#endif

#ifndef FALSE
#define FALSE 0
#endif

/*
 * Only for "Carriage return" and no stream.
 * Assume that STM, STMLF and STMCR are the 3 last record format
 */
#define IS_TEXTFILE (((st.st_fab_rat != 0) && (st.st_fab_rfm < FAB$C_STM)) ? -1 : 1)

/*
 * if read() should fill up with 0 until the file size returned by stat()
 *  has been returned, define FILL_WITH_NULL. Or use FILL_WITH_CTRLZ
 */
/* #define FILL_WITH_NULL */
/* #define FILL_WITH_CTRLZ */

int vms_stat(const char *__path, struct stat *__buf); /* SAMBA */

/*
 * Foward declarations
 */
int stm_write (int __fd, void * __buf, unsigned int __len);
int stm_read(int __fd, void * __buf, unsigned int __maxlen);
long stm_lseek(int __fd, long __offset, int __origin);
int stm_open(char * __fname, int __flags, int __mode);
int stm_close(int __fd);
int sent_size;

/*
 * Define our internal file descriptor table.
 */
struct fd_entry {
	int fd;				/* the file descriptor */
	int new_fd;
	long stream_pos;		/* bytes read (including rec. term.) */
	long rec_pos;			/* record position returned by lseek */
	long rec_offset;		/* offset into next record */
	char pending;			/* number of bytes pending */
	long size;			/* file size from stat() */
	BOOL record_mode;		/* record mode flag */
	char rfm;
	BOOL open_for_write;
	BOOL written;
	BOOL var_to_stm;
	unsigned int mtime;
	unsigned int ctime;
	char fname[1024];
	int flags;
	int mode;
	struct FAB *fab;
	struct RAB *rab;
};

/*
 * Define the number of fd entries allocated. If all entries are used
 * the table must be enlarged by another chunk of entries.
 */
#define FD_CHUNK 10
static struct fd_entry *fds = NULL;
static int fd_tabsize = 0;

/*
 * Define the size of the read buffer for dummy reads in lseek.
 * This buffer is allocated dynamically for each lseek.
 */
#define SEEK_BUFFER 32768



/* Given a file descriptor, return the fd table entry if any */

static void acc_callback (struct fd_entry *pf, struct FAB *fab, struct RAB *rab)
{
	pf->fab = fab;
	pf->rab = rab;
}

static struct fd_entry *find_fd_entry(fd)
{
	struct fd_entry *pf;
	int i;

	if (fds == NULL) return(NULL);
	for(i = 0, pf = fds; i < fd_tabsize; i++, pf++) {
		if (pf->fd == fd) return(pf);
	}
	return(NULL);
}


/* allocate a new chunk of fd entries. */

static int allocate_fd_entries(int num)
{
	struct fd_entry *pf;
	int i;
/*
 * Allocate or reallocate (enlarge) the table.
 */	
	i = (fd_tabsize + num) * sizeof(struct fd_entry);
	if (fds == NULL)
		pf = (struct fd_entry *)malloc(i);
	else
		pf = (struct fd_entry *)realloc(fds,i);
	if (pf == NULL) {
		DEBUG (0,("stm_open: can't allocate memory for fd table\n"));
		errno = ENOMEM;
		return(-1);
	}
/*
 * Save new pointer. Set new entries to 'unused'.
 */
	fds = pf;
	for(i = fd_tabsize; i < fd_tabsize + num; i++) {
		fds[i].fd = -1;
	}
	fd_tabsize = fd_tabsize + num;
	return(0);
}


static void var_to_stm (char *filename)
{
	char cmd[256];
	int sysprv_prv[2] = {PRV$M_SYSPRV, 0};
	int old_prv[2];
	int dcmd[2];
	$DESCRIPTOR (null,"NL:");
	int spawn_sts = 0;

	sprintf (cmd,"$ set file/attr=(rfm:stm,rat:cr) %s", filename);
	sys$setprv (
              	  1, 
              	  sysprv_prv, 
              	  0, 
              	  old_prv);
	dcmd[0] = strlen(cmd);
	dcmd[1] = (int) cmd;
	lib$spawn (
            	dcmd, 
            	0, 
            	&null, 
            	0, 
            	0, 
            	0, 
            	&spawn_sts);

	sys$setprv (
              	  0, 
              	  sysprv_prv, 
              	  0, 
              	  0);
	sys$setprv (
              	  1, 
              	  old_prv, 
              	  0, 
              	  0);
}

/*
 * open
 *
 * Check the access type and file type.
 * If a file of type 'variable record' and 'carriage control' we
 * assume a text file and enable record to stream conversion.
 */

int stm_open(char *fname, int flags, int mode)
{
	long sts;
	struct stat st;
	struct fd_entry *pf;
	int fd;
	int is_text;
	char fname1[256];

	vms_invalidate_stat_cache(fname);
	sts = stat(fname,&st);
	if (sts == 0)
	{
	   is_text = IS_TEXTFILE;
	}
	else
	{
	   is_text = 1;
	   st.st_mtime = time(0);
	   st.st_ctime = st.st_mtime;
	   st.st_fab_rfm = FAB$C_STM;
	}

	pf = find_fd_entry(-1);		/* find empty entry */
	if (pf == NULL) {
		sts = allocate_fd_entries(FD_CHUNK);
		if (sts < 0) return(sts);
		pf = find_fd_entry(-1);		/* find empty entry */
	}
	pf->stream_pos = 0;
	pf->rec_pos = 0;
	pf->rec_offset = 0;
	pf->pending = FALSE;
	pf->open_for_write = FALSE;
	pf->written = FALSE;
	pf->record_mode = FALSE;
	pf->var_to_stm = FALSE;
	pf->rfm = st.st_fab_rfm;

	strcpy(fname1,vms_encode_filespec(fname,FALSE));
        DEBUG(2,("Open file \"%s\"  ->  \"%s\"\n",fname,fname1));
	if (flags & 0x40000000) {
		DEBUG(2,("Open file %s in binary mode\n",fname1));
		flags &= ~0x40000000;
		fd = open(fname1,flags,mode,
			  "rfm=fix","mrs=512","ctx=bin","shr=get,put");
	}
	else
	{
		if ( is_text != 1 ) 
		{
			sent_size = 0;
			DEBUG(2,("Open file %s in record mode\n",fname1));
			fd = open(fname1,flags,mode,"ctx=rec",
					/* "ctx=nocvt",*/ "shr=get,put",
					"acc",acc_callback,pf);
			pf->record_mode = TRUE;
		}
		else
		{
			DEBUG(2,("Open file %s in stream mode\n",fname1));
			fd = open(fname1,flags,mode,
				  "rfm=stm","ctx=stm","ctx=bin","shr=get,put");
		}

		if ((flags & O_WRONLY) != 0 || 
		    (flags & O_RDWR) != 0 ||
		    (flags & O_CREAT) != 0 )
		{
			pf->open_for_write = TRUE;
			pf->mtime = st.st_mtime;
			pf->ctime = st.st_ctime;
			if ( is_text != 1 ) {
				pf->var_to_stm = TRUE;
				getname (fd,pf->fname);
				pf->flags = flags;
				pf->mode = mode;
			}
		}
		
	}

	pf->fd = fd;
	pf->new_fd = fd;
	return(fd);
}


/*
 * close
 *
 * Just do it. And mark a possible fd table entry unused.
 */

int stm_close(int fd)
{
	struct fd_entry *pf;
	int sts;
	char filename[2048];

	if (fd < 0) {
		errno = EBADF;
		return(-1);
	}
	pf = find_fd_entry(fd);
	if (pf != NULL)
	{
		fd = pf->new_fd;
		if (pf->open_for_write)
		{
			if (pf->written)
			{
				if (pf->var_to_stm)
				{
					getname (fd, filename, 1);
					sts = close(fd);
					var_to_stm(filename);
				}
				else
					sts = close(fd);
			}
			else
			{
				struct utimbuf times;
				times.actime = pf->ctime;
				times.modtime = pf->mtime;
	
				getname (fd, filename, 1);
				sts = close(fd);
				vms_utime (filename, &times);
			}
							
		}
		else
			sts = close(fd);
		pf->fd = -1;
	}
	else
		sts = close(fd);

	return(sts);
}


int stm_write (int fd, void *buf, unsigned int len)
{
    struct fd_entry *pf;
    int written_size;
    int error;
    char *record;
    char *tmp;
    int record_size;

    pf = find_fd_entry(fd);
    if (pf != NULL)
    {
	if (pf->record_mode)
	{
	    close(fd);
	    DEBUG(3,("ReOpen file %s in stream mode\n",pf->fname));
   	    pf->new_fd = open(pf->fname,O_RDWR,pf->mode,
		      "rfm=stm","ctx=stm","shr=get,put");
	    if (pf->new_fd <= 0)
	    {
		DEBUG(0,("ReOpen %s in stream mode failed : %s\n",
				pf->fname, strerror(EVMSERR,vaxc$errno)));
		return -1;
	    }
	    pf->record_mode = FALSE;
	}	    
	pf->written = TRUE;
	fd = pf->new_fd;
    }
    written_size = write (fd, buf, len);
    return written_size;
}

/*
 * read
 *
 * Read the records from file and append the record termination bytes.
 * Maintain the stream position (pure bytes of the record plus termination
 * bytes).
 */

static char onerec[8192];
static int onerec_len;

int stm_read(int fd, void *buf, unsigned int maxlen)
{
	struct fd_entry *pf;
	int total_ret = 0;
	int ret = 0;
	int i;
	int pos;
	char freespace = TRUE;
	unsigned int ini_maxlen = maxlen;

	memset(buf,0,maxlen);
	DEBUG (2, ("stm_read: request to read %d bytes\n",maxlen));
	if (fd < 0) {
		errno = EBADF;
		return(-1);
	}
	if (buf == NULL) return(0);
	if (maxlen == 0) return(0);
	pf = find_fd_entry(fd);
	if (pf == NULL || (!pf->record_mode)) return(read(fd, buf, maxlen));

	DEBUG (2, ("stm_read: request to read %d bytes, stream_pos = %d\n",maxlen,pf->stream_pos));
/*
 * If there is a pending record which did not fit in
 * the buffer of a previous read, try to return it now.
 */
	if (pf->pending) 
	{
		if (onerec_len > maxlen)
		{
		        memcpy(buf,onerec,maxlen);
			memcpy(onerec,onerec+maxlen,onerec_len);
			onerec_len = onerec_len - maxlen;
			total_ret += maxlen;
			pf->pending = TRUE;
			freespace = FALSE;	
	        }
		else
		{
			memcpy (buf,onerec,onerec_len);
		        buf += onerec_len;
		        maxlen -= onerec_len;
			total_ret += onerec_len;
			pf->pending = FALSE;
		}
	}
/*
 * Now read records.
 * Each read does not read beyond the record boundary. If the last byte
 * of a record is read, the record position returned by lseek is updated.
 * It is not changed otherwise. Thus we maintain the record byte offset
 * ourself.
 */

	while (freespace)
	{
	    pf->rab->rab$l_ubf = onerec;
	    pf->rab->rab$w_usz = sizeof(onerec) - 2;
	    ret = sys$get(pf->rab);
	    if (ret == RMS$_EOF)
	    {
		ret = 0;	/* end of file */
		break;
	    }
	    if ((ret & 1) == 0)	/* error */
	    {
	 	errno = EVMSERR;
		vaxc$errno = ret;
		return(-1);
	    }
	    onerec_len = pf->rab->rab$w_rsz;
	    onerec[onerec_len++] = 13;
	    onerec[onerec_len++] = 10;
	    onerec[onerec_len] = 0;

/*
	    if (ini_maxlen != 62000)
		DEBUG (2, ("stm_read: %d : %s\n",onerec_len,onerec));
*/

	    if (onerec_len > maxlen)
	    {
	        memcpy(buf,onerec,maxlen);
		memcpy(onerec,onerec+maxlen,onerec_len);
		onerec_len = onerec_len - maxlen;
		total_ret += maxlen;
		pf->pending = TRUE;
		freespace = FALSE;	
	    }
	    else
	    {
		memcpy (buf, onerec, onerec_len);
	        buf += onerec_len;
	        maxlen -= onerec_len;
		total_ret += onerec_len;
	    }	    
	}
/*
 * Record the 'stream position' (bytes from the beginning of the file
 * including the record terinator bytes).
 */
	pf->stream_pos += total_ret;
	DEBUG (2, ("stm_read: %d bytes returned, total= %d, stream_pos = %d\n",
		total_ret, sent_size, pf->stream_pos));
	sent_size += total_ret;
	return(total_ret);
}


/*
 * lseek
 *
 * The offset is given as stream bytes. Seeking forward means reading
 * the file to maintain the stream position. Seeking backward means
 * rewinding the file and start reading from the beginning.
 * Thus, the lseek may be very slow.
 */

long stm_lseek(int fd, off_t offset, int origin)
{
	struct fd_entry *pf;
	int pos;
	int ret;
	char *buf = NULL;
	int i;

	DEBUG(2,("vms_lseek: offset = %d, origin = %d\n",offset,origin));
	if (fd < 0) {
		errno = EBADF;
		return(-1);
	}
	pf = find_fd_entry(fd);
	if (pf == NULL || (!pf->record_mode)) 
		return(lseek(fd, offset, origin));

	DEBUG(2,("vms_lseek: offset = %d, origin = %d, stream_pos = %d\n",
		offset,origin,pf->stream_pos));
/*
 * We can't seek backwards.
 * So we get the stream position of 'origin', add this to 'offset' and
 * use absolute seek from the beginning of the file.
 */
	if (offset < 0) {
		if (origin == SEEK_CUR) {
			offset += pf->stream_pos;
			origin = SEEK_SET;
		}
		else if (origin == SEEK_END) {
			pos = stm_lseek(fd, 0, SEEK_END); /* recursivly */
			offset += pos;
			origin = SEEK_SET;
		}
	}
/*
 * If the user just ask for the current position, why not simply return it...
 */
	else if (offset == 0) {
		if (origin == SEEK_CUR) {
			return(pf->stream_pos);
		}
	}
/*
 * If to seek before the beginning of the file, return error.
 */
	if (offset < 0) {
		errno = EINVAL;
		return(-1);
	}
/*
 * Seeking beyond the end of file is not permitted since we only support
 * reading files. convert to offset 0.
 */
	if (origin == SEEK_END) {
		if (offset > 0) offset = 0;
	}
/*
 * If now to seek from the beginning (SEEK_SET),
 * check if we are already positioned. Otherwise rewind the file.
 * If the user asks for the beginning, stop here.
 */
	if (origin == SEEK_SET) {
		if (offset == pf->stream_pos) return(offset); /* ready!! */
		if (offset < pf->stream_pos) {
/*
			i = lseek(fd, 0, SEEK_SET);
*/
			sys$rewind(pf->rab);			
			pf->stream_pos = 0;
			pf->rec_pos = 0;
			pf->rec_offset = 0;
			pf->pending = 0;
			if (offset == 0) return(0);
		}
		else {
			offset -= pf->stream_pos;
			origin = SEEK_CUR;
		}
	}
/*
 * Looking forward means reading the file. Allocate a buffer.
 */
	buf = (char *)malloc(SEEK_BUFFER);
	if (buf == 0) {
		DEBUG (0,("stm_lseek: can't allocate memory for buffer\n"));
		errno = ENOMEM;
		return(-1);
	}
/*
 * The following is possible now:
 *
 * - SEEK_CUR or SEEK_SET with positive (non 0) offset: read 'offset' bytes.
 * - SEEK_END with offset 0: read until end of file.
 */
	while (offset > 0 || origin == SEEK_END) {
		if (origin == SEEK_END) offset = SEEK_BUFFER;
		i = (offset > SEEK_BUFFER) ? SEEK_BUFFER : offset;
		ret = stm_read(fd, buf, i);
		if (ret == 0) break;	/* end of file */
		if (ret < 0) {
			errno = EINVAL;
			return(-1);	/* error */
		}
		offset -= ret;
	}

	if (buf != NULL) free(buf);
	return(pf->stream_pos);
}
