/*
 * Special VMS version of TDB persistent storage functions.  This version
 * uses an indexed file with the hash value as its key.
 *
 * Revised: 11-MAR-2003		Use callable FDL to create indexed TDB file
 *				if missing or existing has wrong record type.
 * Revised: 25-JUN-2003		Fix bug in find_int_data().
 */
#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <starlet.h>
#include <ssdef.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <unixio.h>		/* getname */
#include <unixlib.h>
#include <fdldef.h>			/* File description language */
#include <lib$routines.h>		/* lib$ routines (find_image_symbol) */
#include "tdb.h"
/* #include "spinlock.h" */

static 	u32 null_hash = 0;
static u32 hash;
static char real_fetch = FALSE;

/* NB assumes there is a local variable called "tdb" that is the
 * current context, also takes doubly-parenthesized print-style
 * argument. */
#define TDB_LOG(x) (tdb->log_fn?((tdb->log_fn x),0) : 0)

/* lock offsets */
#define GLOBAL_LOCK 0
#define ACTIVE_LOCK 4

TDB_DATA tdb_null;

/* all contexts, to ensure no double-opens (fcntl locks don't nest!) */
static TDB_CONTEXT *tdbs = NULL;

/*
 * Function to duplicate a TDB_DATA structure and it's contents.
 */
TDB_DATA tdb_dupe(TDB_DATA src )
{
    TDB_DATA dst;

    dst.dsize = src.dsize;
    if ( dst.dsize > 0 ) dst.dptr = malloc ( dst.dsize + 1 );
    else dst.dptr = 0;
/* printf ( "duping tdb(%d,%x) => %x\n", src.dsize, src.dptr, dst.dptr ); */
    if ( (dst.dsize > 0) && dst.dptr && src.dptr )
	memcpy ( dst.dptr, src.dptr, dst.dsize );

    return dst;
}
#ifdef DEBUG_DISP
static void disp_key ( TDB_DATA key, const char *prefix, TDB_CONTEXT *tdb )
{
    int i, j, length, smb_panic();
    char disp[412];

    length = key.dsize < 180 ? key.dsize : 179;
    if ( key.dptr ) {
    for ( i = j = 0; i < length; i++ ) {
	sprintf ( &disp[j], "%02x", key.dptr[i] ); j+= 2;
	if ( (i&3) == 3 ) if ( (i+1) < length) disp[j++] = '-';
    }
    } else {
	strcpy ( disp, "<NULL>" );
    }
    disp[j] = '\0';
    printf ( "%s key: size=%d, data= '%s'\n", prefix, key.dsize, disp );
    printf ( "   tdb address: %x, rab id: %d\n", tdb, tdb->rab.rab$b_bid );
    if ( !tdb->rab.rab$b_bid && !tdb->hash_mask ) 
	smb_panic ( "Invalid RAB block in non-internal TDB" );
}
#else
int smb_panic();
#define disp_key(x,y,z) if ( !z->rab.rab$b_bid && !z->hash_mask ) \
  smb_panic ( "Invalid RAB block in non-internal TDB\n" );
#endif

/***************************************************************
 Allow a caller to set a "alarm" flag that tdb can check to abort
 a blocking lock on SIGALRM.
 ***************************************************************/

static sig_atomic_t *palarm_fired;

void tdb_set_lock_alarm(sig_atomic_t *palarm)
{
    palarm_fired = palarm;
}

 /* a byte range locking function - return 1 on success
   this functions locks/unlocks 1 byte at the specified offset.

   On error, errno is also set so that errors are passed back properly
   through tdb_open(). */

#include <lckdef.h>
#include <descrip.h>
#include <prvdef.h>
#include <lkidef.h>
static struct smblock {
    struct smblock *next;
    int sts;
    int val;
    char resnam[40];
} *smblocks;

static int global_access_lock (TDB_CONTEXT *tdb, int rw_type, int *result )
{
    int status, length, flags, mode;
    char resnam[40], *tmp;
    struct smblock *lock, *prev_lock, provisional_lock;
    struct { int l; char *s; } resnam_dx;
    /*
     * Construct lock resource name, max allowed is 31 characters.
     * We really should be using a NAM block with the open and contructing
     * a unique name from the device and file id.
     */
    tmp = strrchr (tdb->name,'/');
    if ( !tmp ) tmp = tdb->name;		/* no '/' in name */
    else tmp++;					/* skip the / */
    length = strlen ( tmp );
    if ( length > 24 ) length = 24;
    strncpy ( resnam, "SMBTDB_", 7 );
    strncpy ( &resnam[7], tmp, length );
    resnam[length+7] = '\0';
    /*
     * Search lock list to see if a lock with this resource name already
     * acquired.  Keep track of previous entry in list for easy deletion.
     */
    prev_lock = 0;
    for ( lock = smblocks; lock; lock = lock->next ) {
	if ( strcasecmp(resnam,lock->resnam) == 0 ) break;
	prev_lock = lock;
    }
    /*
     * Set up flags for ENQ operation.
     */
    if ( lock ) {
	flags = LCK$M_CONVERT;		/* lock exists */
    } else {
	/*
	 * Lock not in list, use a temporary structure.
	 */
	flags = 0;
	lock = &provisional_lock;	/* automatic storage */

	memcpy ( lock->resnam, resnam, 32 );
	lock->next = 0;

    }
    /*
     * Take initial action based on lock operation requested.
     */
    switch ( rw_type ) {
    case F_UNLCK:
	if ( lock ) {
	    /*
	     * Lock present, dequeue and remove from lock list.
	     */
	    status = SYS$DEQ ( lock->val, 0, 0, 0 );
	    if ( prev_lock ) prev_lock->next = lock->next;
	    else smblocks = lock->next;		/* was first in list */

	    free ( lock );
	    lock = 0;
	}
	if ( result ) *result = F_UNLCK;
	break;

    case F_RDLCK:
	/*
 	 * Get a PR (protected read) mode lock or convert existing one.
	 */
 	resnam_dx.l = strlen ( lock->resnam );
	resnam_dx.s = lock->resnam;

	status = sys$enqw ( 0, LCK$K_PRMODE, &lock->sts, 
                     LCK$M_SYSTEM | flags,
                     &resnam_dx, 
                     0, 0, 0, 0, 0, 0, 0, 0);
	if ( (status&1) == 1 ) status = lock->sts;
	if ( result ) *result = F_RDLCK;

	break;

    case F_WRLCK:
	/*
 	 * Get a CR (concurrent read) mode lock or convert existing one.
	 */
 	resnam_dx.l = strlen ( lock->resnam );
	resnam_dx.s = lock->resnam;

	status = sys$enqw ( 0, LCK$K_CRMODE, &lock->sts, 
                     LCK$M_SYSTEM | flags,
                     &resnam_dx, 
                     0, 0, 0, 0, 0, 0, 0, 0);
	if ( (status&1) == 1 ) status = lock->sts;

	break;

    default:
	/*
	 * Bad rw_type.
	 */
	status = SS$_BADPARAM;
	break;
    }
    /*
     * Set error number and abort if lock operation failed.
     */
    if ( (status&1) == 0 ) {
	errno = EVMSERR;
	vaxc$errno = status;
	if ( result ) *result = F_UNLCK;
	return 0;
    }
    /*
     * Make a provisional lock permanent, copy to new structure and add to list.
     */
    if ( lock == &provisional_lock ) {
	lock = malloc ( sizeof(struct smblock) );
	if ( !lock ) {
	    return 0;
	}
	*lock = provisional_lock;

	lock->next = smblocks;
	smblocks = lock;
    }
    /*
     * Take additional action if rw_type is W_WRLCK.  Attempt to convert
     * the CR mode lock to PW (protected write) mode with the NOQUEUE option.
     * Other processes with a CR mode lock will not block the conversion but 
     * any holding a PR mode lock will cause the request to fail (leaving us 
     * with a CR mode lock).
     */
    if ( rw_type == F_WRLCK ) {

	if ( result ) *result = F_WRLCK;

	status = sys$enqw ( 0, LCK$K_EXMODE, &lock->sts,
	    LCK$M_SYSTEM | LCK$M_CONVERT | LCK$M_NOQUEUE, 
	    0, 0, 0, 0, 0, 0, 0, 0, 0 );

	if ( status == SS$_NOTQUEUED ) {
	    /*
	     * Another guy got a PW lock on the resource before we did. Upgrade
	     * our CR lock to PR mode, which forces us to wait for the other
	     * guy to complete initialization.
	     */
	    return global_access_lock ( tdb, F_RDLCK, result );

	} else if ( (status&1) == 1 ) {
	    /*
	     * Lock didn't block, verify that we got the lock.  If we did,
	     * leave result at F_WRLCK to tell caller we have the PW mode lock.
	     */
	    if ( (lock->sts & 1) == 0 ) {
		errno = EVMSERR;
		vaxc$errno = lock->sts;
	 	if ( result ) *result = F_UNLCK;
	       return 0;
	    }
	} else {
	    /*
	     * Error queuing lock.
	     */
	    errno = EVMSERR;
	    vaxc$errno = status;
	    if ( result ) *result = F_RDLCK;
	    return 0;
	}
    }
    return 1;
}

/* This is based on the hash algorithm from gdbm */
static u32 tdb_hash(TDB_DATA key)
{
	u32 value;	/* Used to compute the hash value.  */
	u32   i;	/* Used to cycle through random values. */

	/* Set the initial value from the key size. */
	for (value = 0x238F13AF * key.dsize, i=0; i < key.dsize; i++)
		value = (value + (key.dptr[i] << (i*5 % 24)));

	return (1103515243 * value + 12345);  
}


enum TDB_ERROR tdb_error(TDB_CONTEXT *tdb)
{
	return tdb->ecode;
}

static struct tdb_errname {
	enum TDB_ERROR ecode; const char *estring;
} emap[] = { {TDB_SUCCESS, "Success"},
	     {TDB_ERR_CORRUPT, "Corrupt database"},
	     {TDB_ERR_IO, "IO Error"},
	     {TDB_ERR_LOCK, "Locking error"},
	     {TDB_ERR_OOM, "Out of memory"},
	     {TDB_ERR_EXISTS, "Record exists"},
	     {TDB_ERR_NOLOCK, "Lock exists on other keys"},
	     {TDB_ERR_NOEXIST, "Record does not exist"} };

/* Error string for the last tdb error */
const char *tdb_errorstr(TDB_CONTEXT *tdb)
{
	u32 i;
	for (i = 0; i < sizeof(emap) / sizeof(struct tdb_errname); i++)
		if (tdb->ecode == emap[i].ecode)
			return emap[i].estring;
	return "Invalid error code";
}

/*
 * The FDL for creating the index file.
 */
static $DESCRIPTOR(tdb_fdl_dx, "\
  FILE;	GLOBAL_BUFFER_COUNT 6; ORGANIZATION indexed;\
    PROTECTION (system:RWD, owner:RWD, group:RW, world:RW);\
  RECORD; CARRIAGE_CONTROL carriage_return; FORMAT variable; SIZE 0;\
  AREA 0; ALLOCATION 999; BEST_TRY_CONTIGUOUS yes; BUCKET_SIZE 12;\
    EXTENSION 1296;\
  AREA 1; ALLOCATION 24; BEST_TRY_CONTIGUOUS yes; BUCKET_SIZE 12;\
    EXTENSION 12;\
  KEY 0; CHANGES no; DATA_AREA 0; DATA_FILL 100; DATA_KEY_COMPRESSION no;\
    DATA_RECORD_COMPRESSION yes; DUPLICATES yes; INDEX_AREA 1;\
    INDEX_COMPRESSION no; INDEX_FILL 100; LEVEL1_INDEX_AREA 1; PROLOG 3;\
    SEG0_LENGTH 4; SEG0_POSITION 0; TYPE bin4;\
");
static char to_vms_filename[1024];
/*
 * Action routines for decc$to_vms that verify the existence of target
 * file, saving its VMS-style name in global variable to_vms_filename[];
 */
static int verify_existing ( char *vms_name, int is_directory )
{
    /*
     * See if file exists and is a keyed file.  if so, set to_vms_filename.
     * Note that stat call sets errno on failure for reporting purposes.
     */
    struct stat info;
    int rfm, length;

    to_vms_filename[0] = '\0';

    if ( stat ( vms_name, &info )== 0 ) {
	/*
	 * Should test for file organization, for now use record format.
	 * Files that we didn't create are most likely stream files from
	 * the standard memory-based TDB functions.
	 */
	rfm = info.st_fab_rfm;
	if ( rfm != FAB$C_VAR ) return 0;
	/*
	 * File is good, save its name for open later with SYS$OPEN().
	 */
	length = strlen ( vms_name );
	if ( length >= sizeof ( to_vms_filename ) ) return 0;
	strcpy ( to_vms_filename, vms_name );

	return 1;
    }
    return 0;
}

static int verify_existing_or_create ( char *vms_name, int is_directory )
{
    int status, flags, length;
    static $DESCRIPTOR(filename,"");
    static $DESCRIPTOR(fdlshr,"FDLSHR");
    static $DESCRIPTOR(fdlcreate,"FDL$CREATE");
    static images_loaded = 0;
    static int (*fdl_create)();
    /*
     * First to verify part, we are done if file exists.  Otherwise
     * delete existing bad-format file.
     */
    status = verify_existing ( vms_name, is_directory );
    if ( status != 0 ) return status;
    delete ( vms_name );
    /*
     * Load FDL code dynamically since we expect to rarely need it.
     */
    if ( !images_loaded ) {
	status = lib$find_image_symbol ( &fdlshr, &fdlcreate, &fdl_create );
    	if ( (status&1) == 0 ) {
	    errno = EVMSERR;
	    vaxc$errno = status;
	    return 0;
	}
	images_loaded = 1;
    }
    /*
     * Set descriptors to point to passed strings and call FDL$CREATE.
     */
    filename.dsc$w_length = strlen ( vms_name );
    filename.dsc$a_pointer = vms_name;

    flags = FDL$M_FDL_STRING | FDL$M_SIGNAL;

    status = fdl_create ( &tdb_fdl_dx, &filename, 0, 0, 0, &flags );
    /*
     * Synthesize C errno behaviour on error, save creatd name on success.
     */
    if ( (status&1) == 0 ) {
	errno = EVMSERR;
	vaxc$errno = status;
    } else {
	length = strlen ( vms_name );
	if ( length >= sizeof ( to_vms_filename ) ) return 0;
	strcpy ( to_vms_filename, vms_name );
    }
    return (status&1);
}

/*
 * Internal find return hashs table index that was searched.  If not found
 * idat receives null pointer.
 */
static int find_int_data ( TDB_CONTEXT *tdb,
	TDB_DATA key, struct tdb_int_data **idat )
{
    int ndx;
    struct tdb_int_data *cur;

    ndx = tdb_hash ( key ) & tdb->hash_mask;
    if ( !tdb->hash_table[ndx] ) { *idat = 0; return ndx; }
    for ( cur = tdb->hash_table[ndx]; cur; cur = cur->next_hash ) {
	if ( key.dsize != cur->key_size ) continue;
	if ( memcmp ( key.dptr, cur->key, key.dsize ) == 0 ) {
	    *idat = cur;
	    return ndx;
	}
    }
    *idat = 0;
    return ndx;			/* not found */
}

static TDB_DATA fetch (TDB_CONTEXT *tdb, TDB_DATA key, int keep_lock)
{
    char found = FALSE;
    int sts, *kv;

    hash = tdb_hash(key);
    /* printf ( "   hash: %x, last_hash: %x chg: %d\n", hash, tdb->last_hash,
	tdb->changed ); */

    if (!tdb->changed && tdb->last_hash == hash)
    {
	if ( (key.dsize == tdb->tdb_dkey.dsize) &&
	    (memcmp ( key.dptr, tdb->tdb_dkey.dptr, key.dsize ) == 0) ) {
	    return tdb_dupe(tdb->tdb_dbuf);
	}
    }
    tdb->last_hash = hash;

    if ( tdb->hash_mask ) {
	/*  
	 * Internal database
	 */
	int ndx;
	struct tdb_int_data *cur;

	ndx = find_int_data ( tdb, key, &cur );
	if ( cur ) {
	    tdb->tdb_dkey.dsize = cur->key_size;
	    tdb->tdb_dkey.dptr = cur->key;

	    tdb->tdb_dbuf.dsize = cur->data.dsize;
	    tdb->tdb_dbuf.dptr = cur->data.dptr;
	    tdb->changed = FALSE;
	    return tdb->tdb_dbuf;
	} else {
	    /*
	     * Not found.
	     */
	    tdb->changed = FALSE;
	    return tdb_null;
	}
    };
    /*
     * Read keyed entry in file.
     */
    tdb->rab.rab$l_kbf = (char *) &hash;
    tdb->rab.rab$b_krf = 0;
    tdb->rab.rab$b_ksz = sizeof(hash);
    tdb->rab.rab$l_ubf = (char *) &tdb->tdb_record;
    tdb->rab.rab$w_usz = sizeof(tdb->tdb_record);
    tdb->rab.rab$w_rsz = 0;
    tdb->rab.rab$v_kge = 0;
    tdb->rab.rab$b_rac = RAB$C_KEY;
    sts = sys$get(&tdb->rab);
    /* printf ( "   status of initial get: %d, rab %x hdr; %x %d\n", sts,
	&tdb->rab, tdb->rab.rab$b_bid, tdb->rab.rab$b_bln ); */

    while ((sts & 1) == 1)
    {
        if (hash == tdb->tdb_record.hash_key)
        {
            if (tdb->tdb_record.key_size == key.dsize)
            {
                if (memcmp(tdb->tdb_record.value, key.dptr, key.dsize) == 0)
                {
                    found = TRUE;
                }
            }
            if (!found)
            {
                tdb->rab.rab$b_rac = RAB$C_SEQ;
                sts = sys$get(&tdb->rab);
    		/* printf ( "   status of secondary get: %d\n", sts ); */
            }
            else
                sts = 0;
        }
        else
            sts = 0;
    }

    if (found)
    {
	if (keep_lock == 0)
	    sys$free(&tdb->rab);
	
	tdb->tdb_dkey.dsize = tdb->tdb_record.key_size;
	tdb->tdb_dkey.dptr = tdb->tdb_record.value;

	tdb->tdb_dbuf.dsize = tdb->tdb_record.data_size;
	tdb->tdb_dbuf.dptr = &tdb->tdb_record.value[key.dsize];
	tdb->changed = FALSE;
	return tdb->tdb_dbuf;
     }
     else
     {
	tdb->changed = FALSE;
	return tdb_null;
    }
}


/* find an entry in the database given a key */
TDB_DATA tdb_fetch(TDB_CONTEXT *tdb, TDB_DATA key)
{
    int sts;

    real_fetch = TRUE;
    disp_key ( key, "Fetchin, ", tdb );
    tdb->tdb_dbuf = fetch (tdb, key, 0);
    return tdb_dupe(tdb->tdb_dbuf);
}

/* traverse the entire database - calling fn(tdb, key, data) on each element.
   return -1 on error or the record count traversed
   if fn is NULL then it is not called
   a non-zero return value from fn() indicates that the traversal should stop
  */
int tdb_traverse(TDB_CONTEXT *tdb, tdb_traverse_func fn, void *state)
{
	TDB_DATA key, dbuf;
	int count = 0;
	int sts;

	if ( tdb->hash_mask ) {
	    /*
	     * Using internal database, traverse hash table.
	     */
	    int i, limit;
	    struct tdb_int_data *cur, *next;
	    limit = tdb->hash_mask;
	    for ( i = 0; i <= limit; i++ ) {
		for ( cur = tdb->hash_table[i]; cur; cur = next ) {
		    next = cur->next_hash;
		    count++;
		    if (fn && fn(tdb, key, dbuf, state)) {
			/* They want us to terminate traversal */
			return count;
		    }
		}
	    }
	    return count;
	}
	tdb->rab.rab$b_rac = RAB$C_KEY;
	tdb->rab.rab$v_kge = 1;
	tdb->rab.rab$l_kbf = (char *) &null_hash;
	tdb->rab.rab$w_usz = sizeof(tdb->tdb_record);
	tdb->rab.rab$w_rsz = 1000;

	sts = sys$get(&tdb->rab);	
	while ((sts & 1) == 1)
	{
	    count++;
	    key.dsize = tdb->tdb_record.key_size;
	    key.dptr = &tdb->tdb_record.value[0];
	    dbuf.dsize = tdb->tdb_record.data_size;
	    dbuf.dptr = &tdb->tdb_record.value[key.dsize];
	    if (fn && fn(tdb, key, dbuf, state)) {
		/* They want us to terminate traversal */
		sys$free (&tdb->rab);
		return count;
	    }
	    tdb->rab.rab$w_usz = sizeof(tdb->tdb_record);
	    tdb->rab.rab$w_rsz = 1000;
	    tdb->rab.rab$b_rac = RAB$C_SEQ;
	    sts = sys$get(&tdb->rab);
	}

	return count;
}

/* find the first entry in the database and return its key */
TDB_DATA tdb_firstkey(TDB_CONTEXT *tdb)
{
    int sts;

    tdb->changed = TRUE;
    if ( tdb->hash_mask ) {	/* internal usage */
	int i;
	for ( i = 0; i <= tdb->hash_mask; i++ ) {
	    if ( tdb->hash_table[i] ) {
		tdb->tdb_dbuf.dsize = tdb->hash_table[i]->key_size;
		tdb->tdb_dbuf.dptr = tdb->hash_table[i]->key;
		tdb->hash_position = i;
		tdb->cur_int_data =tdb->hash_table[i];
		return tdb_dupe(tdb->tdb_dbuf);
	    }
	}
	tdb->hash_position = -1;
	tdb->cur_int_data = 0;
	return tdb_null;	/* table empty */
    }
    tdb->rab.rab$b_rac = RAB$C_KEY;
    tdb->rab.rab$v_kge = 1;
    tdb->rab.rab$l_kbf = (char *) &null_hash;
    sts = sys$get(&tdb->rab);
    if ((sts & 1) == 1)
    {
	sys$free(&tdb->rab);
	tdb->changed = TRUE;
	tdb->tdb_dbuf.dsize = tdb->tdb_record.key_size;
	tdb->tdb_dbuf.dptr = tdb->tdb_record.value;
	return tdb_dupe(tdb->tdb_dbuf);
    }
    else
	return tdb_null;
}

/* find the next entry in the database, returning its key */
TDB_DATA tdb_nextkey(TDB_CONTEXT *tdb, TDB_DATA oldkey)
{
    int sts;

    if ( tdb->hash_mask ) {		/* internal use */
	int i;
	i = tdb->hash_position;
	if ( tdb->cur_int_data ) 
	    tdb->cur_int_data = tdb->cur_int_data->next_hash;
	if ( !tdb->cur_int_data ) {
	    /* current hash chain exhuasted, find next chain */
	    for ( i = tdb->hash_position+1; i <= tdb->hash_mask; i++ ) {
		if ( tdb->hash_table[i] ) {
		    tdb->cur_int_data = tdb->hash_table[i];
		    break;
		}
		tdb->hash_position = i;		/* prepare for next call */
	    }
	}
	if ( tdb->cur_int_data ) {
	    tdb->changed = TRUE;
	    tdb->tdb_dbuf.dsize = tdb->cur_int_data->key_size;
	    tdb->tdb_dbuf.dptr = tdb->cur_int_data->key;
	    return tdb_dupe(tdb->tdb_dbuf);
	} else {
	    return tdb_null;
	}
    }

    tdb->rab.rab$b_rac = RAB$C_SEQ;
    tdb->rab.rab$l_ubf = (char *) &tdb->tdb_record;
    tdb->rab.rab$w_usz = sizeof(tdb->tdb_record);
    sts = sys$get(&tdb->rab);
    if ((sts & 1) == 1)
    {
	sys$free(&tdb->rab);
	tdb->changed = TRUE;
	tdb->tdb_dbuf.dsize = tdb->tdb_record.key_size;
	tdb->tdb_dbuf.dptr = tdb->tdb_record.value;
	return tdb_dupe(tdb->tdb_dbuf);
    }
    else
	return tdb_null;
}

/* delete an entry in the database given a key */
int tdb_delete(TDB_CONTEXT *tdb, TDB_DATA key)
{
    TDB_DATA data;
    int sts;

    tdb->changed = TRUE;
    disp_key ( key, "Deleting, ", tdb );
    if ( tdb->hash_mask ) {			/* internal delete */
	int ndx;
	struct tdb_int_data *cur, *prev;

	ndx = find_int_data ( tdb, key, &cur );
	if ( cur ) {
	    sts = 0;
	    if ( cur == tdb->hash_table[ndx] ) {
		/* easy case, we are at front of list. */
		tdb->hash_table[ndx] = cur->next_hash;
	    } else {
		for ( prev = tdb->hash_table[ndx]; 
			prev->next_hash != cur; prev = prev->next_hash ) {
		    if ( !prev->next_hash ) break; 	/* shouldn't happen */
		}
		prev->next_hash = cur->next_hash;
	    }
	    cur->next_hash = 0;
	    free ( cur );
	} else sts = -1;
	return sts;
    }

    real_fetch = FALSE;
    data = fetch (tdb, key, 1);
    if (data.dptr != NULL)
    {
	sts = sys$delete(&tdb->rab);
	if ((sts & 1) != 1)
                TDB_LOG((tdb, 0, "$delete failed (%s)\n", strerror(EVMSERR,sts)));
    }
    tdb->changed = TRUE;
    return 0;
}

/* store an element in the database, replacing any existing element
   with the same key 

   return 0 on success, -1 on failure
*/
int tdb_store(TDB_CONTEXT *tdb, TDB_DATA key, TDB_DATA dbuf, int flag)
{
    int sts, sd_dsize;
    char tobeupdated = FALSE;
    char save_data[sizeof(tdb->tdb_record.value)];

    tdb->changed = TRUE;
    disp_key ( key, "Storing, ", tdb );
    if ( tdb->hash_mask ) {
	int ndx, new_size, old_size;
	struct tdb_int_data *cur, *prev;

	new_size = key.dsize + dbuf.dsize + sizeof(struct tdb_int_data) + 8;
	ndx = find_int_data ( tdb, key, &cur );
	if ( cur ) {
	    /* Replace existing key */
	    old_size = cur->key_size + cur->data.dsize +
		sizeof(struct tdb_int_data) + 8;
	    if ( old_size >= new_size ) {
		cur->data.dsize = dbuf.dsize;
		memcpy ( cur->data.dptr, dbuf.dptr, dbuf.dsize );
		return 0;
	    } else if ( cur == tdb->hash_table[ndx] ) {
		/* Delete and let code below re-add */
		tdb->hash_table[ndx] = cur->next_hash;
	    } else {
		for (prev=tdb->hash_table[ndx]; prev; prev=prev->next_hash) {
		    if ( prev->next_hash == cur ) {
			prev->next_hash = cur->next_hash;
			break;
		    }
		}
	    }
	    cur->next_hash = 0;
	    free ( cur );
	}
	/* Key not found or data size too small, add new. */
	cur = malloc ( new_size );
	if ( !cur ) return -1;
	cur->key_size = key.dsize;
	memcpy ( cur->key, key.dptr, key.dsize );
	cur->data.dsize = dbuf.dsize;
	cur->data.dptr = &cur->key[cur->key_size];
	memcpy ( cur->data.dptr, dbuf.dptr, dbuf.dsize );

	cur->next_hash = tdb->hash_table[ndx];
	tdb->hash_table[ndx] = cur;

	return 0;
    }
    /*
     * Copy user's data buffer to local storage, don't know why (avoid
     * overlap issues with tdb_record.value?).
     */
    sd_dsize = dbuf.dsize;	/* size of saved data */
    if ( (dbuf.dsize + key.dsize) > sizeof(tdb->tdb_record.value) ) {
	fprintf(stderr,"Record for file '%s' is too big (%d+%d)\n",
		tdb->fab.fab$l_fna, key.dsize, dbuf.dsize );
	sd_dsize = sizeof(tdb->tdb_record.value) - key.dsize;
    }
    memcpy(save_data,dbuf.dptr,sd_dsize);

    real_fetch = FALSE;
    tdb->tdb_dbuf = fetch (tdb,key,1);

    if (tdb->tdb_dbuf.dptr != NULL)
	tobeupdated = TRUE;

    tdb->tdb_record.hash_key = tdb_hash(key);
    if (!tobeupdated)
    {
	if (flag == TDB_MODIFY) {
	    tdb->ecode = TDB_ERR_NOEXIST;
	    sys$free (&tdb->rab);
	    return -1;
	}
	tdb->rab.rab$b_rac = RAB$C_KEY;
	tdb->tdb_record.key_size = key.dsize;
	tdb->tdb_record.data_size = dbuf.dsize;
	memcpy(&tdb->tdb_record.value[0], key.dptr, key.dsize);
	memcpy(&tdb->tdb_record.value[key.dsize],dbuf.dptr,dbuf.dsize);
	tdb->rab.rab$w_usz = key.dsize + dbuf.dsize + 12;
	tdb->rab.rab$w_rsz = key.dsize + dbuf.dsize + 12;
	sts = sys$put(&tdb->rab);
	if ((sts & 1) != 1)
                TDB_LOG((tdb, 0, "$put failed (%s)\n", strerror(EVMSERR,sts)));
    }
    else
    {
	if (flag == TDB_INSERT) {
	    tdb->ecode = TDB_ERR_EXISTS;
	    sys$free (&tdb->rab);
	    return -1;
	}
	tdb->rab.rab$b_rac = RAB$C_KEY;
	tdb->tdb_record.key_size = key.dsize;
	tdb->tdb_record.data_size = dbuf.dsize;
	memcpy(&tdb->tdb_record.value[0], key.dptr, key.dsize);
	memcpy(&tdb->tdb_record.value[key.dsize],save_data,sd_dsize);
	tdb->rab.rab$w_usz = key.dsize + dbuf.dsize + 12;
	tdb->rab.rab$w_rsz = key.dsize + dbuf.dsize + 12;
	sts = sys$update (&tdb->rab);
	if ((sts & 1) != 1)
                TDB_LOG((tdb, 0, "$update failed (%s)\n", strerror(EVMSERR,sts)));
    }
    if ((sts&1) != 1)
    {
	TDB_LOG((tdb, 0, "tdb_store: can't store : %s\n",strerror(EVMSERR,sts)));
	return -1;	
    }
    tdb->changed = TRUE;
    tdb->ecode = TDB_SUCCESS;
    return 0;
}

/* open the database, creating it if necessary 

   The open_flags and mode are passed straight to the open call on the
   database file. A flags value of O_WRONLY is invalid. The hash size
   is advisory, use zero for a default value.

   Return is NULL on error, in which case errno is also set.  Don't 
   try to call tdb_error or tdb_errname, just do strerror(errno).

   @param name may be NULL for internal databases. */

TDB_CONTEXT *tdb_open(const char *name, int hash_size, int tdb_flags,
		      int open_flags, mode_t mode)
{
	return tdb_open_ex(name, hash_size, tdb_flags, open_flags, mode, NULL);
}

TDB_CONTEXT *tdb_open_ex (const char *name, int hash_size, int tdb_flags,
			 int open_flags, mode_t mode, 
			 tdb_log_func log_fn)
{
	TDB_CONTEXT *tdb;
	int fd, count;
	char filename[1024];
	int sts, ht_size;
        int locked;

	ht_size = 0;
	if ( tdb_flags&TDB_INTERNAL ) {
	    for ( ht_size = 128; ht_size < hash_size; ht_size *= 2 );
	    ht_size = ht_size - 1;	/* TDB_CONTEXT already has 1st in it */
	    tdb_flags |= ( TDB_NOLOCK | TDB_NOMMAP );
	    tdb_flags &= (~TDB_CLEAR_IF_FIRST);
	}
	tdb = calloc ( 1, sizeof(TDB_CONTEXT) + 
		ht_size*sizeof(struct tdb_int_data *) );
	if ( !tdb ) {
		/* Can't log this */
		errno = ENOMEM;
		goto fail;
	}
	tdb->log_fn = log_fn;
	tdb->changed = TRUE;
	tdb->last_hash = 0;
	tdb->flags = tdb_flags;
	tdb->hash_mask = ht_size;		/* size of hash_table-1 */

	if ( !name ) name = "/NULL";
	if (!(tdb->name = (char *)strdup(name))) {
		errno = ENOMEM;
		goto fail;
	}
	if ((open_flags & O_ACCMODE) == O_WRONLY) {
		TDB_LOG((tdb, 0, "tdb_open: can't open tdb %s write-only\n",
			 name));
		errno = EINVAL;
		goto fail;
	}
	if ( tdb->hash_mask == 0 ) {
	    count = decc$to_vms ( name, (open_flags&O_CREAT) ? 
	   	verify_existing_or_create : verify_existing,
	   	0, 0 );		/* don't expand wildcard, disallow directory */

	    if ( to_vms_filename[0] == '\0' ) {
		TDB_LOG((tdb, 0, "tdb_open: could not open file %s: %s\n",
			 name, strerror(EVMSERR,vaxc$errno)));
		goto fail;
	    }

	    tdb->fab = cc$rms_fab;
	    tdb->fab.fab$b_fns = strlen(to_vms_filename);
	    tdb->fab.fab$l_fna = to_vms_filename;
	    tdb->fab.fab$b_shr = 
		FAB$M_SHRPUT + FAB$M_SHRGET + FAB$M_SHRDEL + FAB$M_SHRUPD;
	    if ((open_flags & O_ACCMODE) == O_RDONLY) {
		tdb->read_only = 1;
		/* read only databases don't do locking or clear if first */
		tdb->flags |= TDB_NOLOCK;
		tdb->flags &= ~TDB_CLEAR_IF_FIRST;
		tdb->fab.fab$b_fac = FAB$M_GET;
	    }
	    else
	    {
		tdb->fab.fab$b_fac = FAB$M_GET + FAB$M_PUT + FAB$M_UPD + FAB$M_DEL;
	    }
	    sts = sys$open (&tdb->fab);
            if ((sts&1) != 1)
	    {
		TDB_LOG((tdb, 0, "tdb_open: can't open tdb %s : %s\n",
			 name,
			 strerror(EVMSERR,sts)));
		errno = EINVAL;
		goto fail;
	    }

	    tdb->rab = cc$rms_rab;
	    tdb->rab.rab$l_rop = RAB$M_WAT + RAB$M_TMO;
	    tdb->rab.rab$b_tmo = 2;
	    tdb->rab.rab$b_rac = RAB$C_KEY;
	    tdb->rab.rab$l_ubf = (char *) &tdb->tdb_record;
	    tdb->rab.rab$w_usz = sizeof(tdb->tdb_record);
	    tdb->rab.rab$l_rbf = (char *) &tdb->tdb_record;
	    tdb->rab.rab$b_ksz = 4;
	    tdb->rab.rab$l_fab = &tdb->fab;
	    sts = sys$connect (&tdb->rab);
            if ((sts&1) != 1)
	    {
		TDB_LOG((tdb, 0, "tdb_open: can't connect tdb %s : %s\n",
			 name,
			 strerror(EVMSERR,sts)));
		errno = EINVAL;
		goto fail;
	    }
        } else { 	/* internal */
	    /* printf ( "Created internal TDB, invalid rab %x\n", &tdb->rab);*/
	}
	tdb->flags = tdb_flags;
	tdb->open_flags = open_flags;

	/* ensure there is only one process initialising at once */
	/*
	 * Get a read lock the file or attempt to be the first writer if
         * TDB_CLEAR_IF_FIRST is set.  The locked argument returns
	 * F_WRLCK is we are the first writer or F_RDLCK if not.
	 */
	if ( 0 == global_access_lock ( tdb,
	    (tdb->flags&TDB_CLEAR_IF_FIRST) ? F_WRLCK : F_RDLCK, &locked ) ) {
		TDB_LOG((tdb, 0, "tdb_open: failed to get global lock on %s: %s\n",
			 name, strerror(errno)));
		goto fail;	/* errno set by global_access_lock */
	}
	if ( locked == F_WRLCK ) {
	    /*
	     * We are the first writer, clear the database.
	     */
		open_flags |= O_CREAT;
		tdb->rab.rab$b_rac = RAB$C_SEQ;
		tdb->rab.rab$w_usz = sizeof(tdb->tdb_record);
		tdb->rab.rab$w_rsz = 1000;
		sts = 1;
		while ((sts & 1) == 1)
		{
		    sts = sys$get(&tdb->rab);
		    if ((sts & 1) == 1)
			sys$delete (&tdb->rab);
		}
	    /*
	     * Lower lock to a read lock so others can proceed.
	     */
	    global_access_lock ( tdb, F_RDLCK, 0 );
	}

	return tdb;

fail:
	if ( tdb->hash_mask == 0 ) {
        global_access_lock ( tdb, F_UNLCK, 0 );
	sys$disconnect (&tdb->rab);
	sys$close(&tdb->fab);
	}
	free(tdb);
	return NULL;
	
}

/* close a database */
int tdb_close(TDB_CONTEXT *tdb)
{
    if ( tdb->hash_mask ) {
	struct tdb_int_data *cur, *next;
	int i;
	for ( i = tdb->hash_mask; i >= 0; --i ) {
	    for ( cur = tdb->hash_table[i]; cur; cur = next ) {
		next = cur->next_hash;		/* save for end of loop */
		cur->next_hash = 0;
		free ( cur );
	    }
	}
    } else {
	global_access_lock (tdb, F_UNLCK, 0 );
        sys$disconnect(&tdb->rab);
        sys$close(&tdb->fab);
    }
    free(tdb);
    return 0;
}

/* Unlock the keys previously locked by tdb_lockkeys() */
void tdb_unlockkeys(TDB_CONTEXT *tdb)
{
}

/* lock/unlock one hash chain. This is meant to be used to reduce
   contention - it cannot guarantee how many records will be locked */
int tdb_chainlock(TDB_CONTEXT *tdb, TDB_DATA key)
{
   return 0;
}

int tdb_chainunlock(TDB_CONTEXT *tdb, TDB_DATA key)
{
   return 0;
}


/* register a loging function */
void tdb_logging_function(TDB_CONTEXT *tdb, void (*fn)(TDB_CONTEXT *, int , const char *, ...))
{
	tdb->log_fn = fn;
}


/* reopen a tdb - this is used after a fork to ensure that we have an independent
   seek pointer from our parent and to re-establish locks */
int tdb_reopen(TDB_CONTEXT *tdb)
{
    return 0;
}

/* reopen all tdb's */
int tdb_reopen_all(void)
{
    return 0;
}

/* check if an entry in the database exists 

   note that 1 is returned if the key is found and 0 is returned if not found
   this doesn't match the conventions in the rest of this module, but is
   compatible with gdbm
*/
int tdb_exists(TDB_CONTEXT *tdb, TDB_DATA key)
{

    tdb->tdb_dbuf = fetch (tdb, key, 0);
    if (tdb->tdb_dbuf.dptr == NULL)
	return 0;
    else
	return 1;
}
