/* 
   Unix SMB/Netbios implementation.
   Version 1.9.
   run a command as a specified user
   Copyright (C) Andrew Tridgell 1992-1998
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "includes.h"
#ifdef VMS
#include <jpidef.h>
#endif

extern struct current_user current_user;

/****************************************************************************
This is a utility function of smbrun().
****************************************************************************/

static int setup_out_fd(void)
{  
	int fd;
	pstring path;

	slprintf(path, sizeof(path)-1, "%s/smb.XXXXXX", tmpdir());

	/* now create the file */
	fd = smb_mkstemp(path);

	if (fd == -1) {
		DEBUG(0,("setup_out_fd: Failed to create file %s. (%s)\n",
			path, strerror(errno) ));
		return -1;
	}

	DEBUG(10,("setup_out_fd: Created tmp file %s\n", path ));

	/* Ensure file only kept around by open fd. */
	unlink(path);
	return fd;
}

/****************************************************************************
run a command being careful about uid/gid handling and putting the output in
outfd (or discard it if outfd is NULL).
****************************************************************************/

int smbrun(char *cmd, int *outfd)
{
	pid_t pid;
	uid_t uid = current_user.uid;
	gid_t gid = current_user.gid;
	
#ifdef VMS
    {
    	int ret;
	pstring syscmd;  
	char *path = SMBRUN;
	char filename[1024];
	int pid;
	int dcmd[2];
        int dfile[2];
	int prcret;
#include <prvdef.h>
	int sysprv_prv[2] = {PRV$M_SYSPRV, 0};
	int old_prv[2];
	
	if (!file_exist(path,NULL)) {
		DEBUG(0,("SMBRUN ERROR: Can't find %s. Installation problem?\n",path));
		return(1);
	}

	slprintf(syscmd,sizeof(syscmd)-1,"@%s %d %d \"%s\"",
		 "SAMBA_ROOT:[BIN]SMBRUN", (int)uid, (int)gid, cmd);
	
	DEBUG(5,("smbrun - running %s ",syscmd));
	dcmd[0] = strlen(syscmd);
	dcmd[1] = (int)syscmd;
	lib$getjpi (
            &JPI$_PID, 
            0, 
            0, 
            &pid);
	sprintf (filename,"DCL$OUTPUT_%08x.LOG",pid);
	dfile[0] = strlen(filename);
	dfile[1] = (int)filename;

	ret = sys$setprv (
              	  1, 
              	  sysprv_prv, 
              	  0, 
              	  old_prv);

	ret = lib$spawn (
              	  dcmd, 
              	  0, 
              	  dfile, 
              	  0, 
              	  0, 
              	  0, 
              	  &prcret); 

	if ((ret & 1) == 1)
	    ret = prcret;
	DEBUG(5,("gave %d\n",ret));
	if ((ret & 1) == 1)
	{
	    ret = 0;
	    unlink(filename);
	}
	else
	{
	    ret = -1;
	    rename (filename,".ERR");
	}
	sys$setprv (
              	  0, 
              	  sysprv_prv, 
              	  0, 
              	  0);
	sys$setprv (
              	  1, 
              	  old_prv, 
              	  0, 
              	  0);
	return (ret);
    }
#else
	/* in this method we will exec /bin/sh with the correct
	   arguments, after first setting stdout to point at the file */

	/*
	 * We need to temporarily stop CatchChild from eating
	 * SIGCLD signals as it also eats the exit status code. JRA.
	 */

	/*
	 * Lose any kernel oplock capabilities we may have.
	 */
	oplock_set_capability(False, False);

	/* point our stdout at the file we want output to go into */

	if (outfd && ((*outfd = setup_out_fd()) == -1)) {
		return -1;
	}

	CatchChildLeaveStatus();
                                   	
	if ((pid=sys_fork()) < 0) {
		DEBUG(0,("smbrun: fork failed with error %s\n", strerror(errno) ));
		CatchChild(); 
		if (outfd) {
			close(*outfd);
			*outfd = -1;
		}
		return errno;
    }

	if (pid) {
		/*
		 * Parent.
		 */
		int status=0;
		pid_t wpid;

		
		/* the parent just waits for the child to exit */
		while((wpid = sys_waitpid(pid,&status,0)) < 0) {
			if(errno == EINTR) {
				errno = 0;
				continue;
			}
			break;
		}

		CatchChild(); 

		if (wpid != pid) {
			DEBUG(2,("waitpid(%d) : %s\n",(int)pid,strerror(errno)));
			if (outfd) {
				close(*outfd);
				*outfd = -1;
			}
			return -1;
		}

		/* Reset the seek pointer. */
		if (outfd) {
			sys_lseek(*outfd, 0, SEEK_SET);
		}

#if defined(WIFEXITED) && defined(WEXITSTATUS)
		if (WIFEXITED(status)) {
			return WEXITSTATUS(status);
		}
#endif

		return status;
	}
	
	CatchChild(); 
	
	/* we are in the child. we exec /bin/sh to do the work for us. we
	   don't directly exec the command we want because it may be a
	   pipeline or anything else the config file specifies */
	
	/* point our stdout at the file we want output to go into */
	if (outfd) {
		close(1);
		if (dup2(*outfd,1) != 1) {
			DEBUG(2,("Failed to create stdout file descriptor\n"));
			close(*outfd);
			exit(80);
		}
	}

	/* now completely lose our privileges. This is a fairly paranoid
	   way of doing it, but it does work on all systems that I know of */

	become_user_permanently(uid, gid);

	if (getuid() != uid || geteuid() != uid ||
	    getgid() != gid || getegid() != gid) {
		/* we failed to lose our privileges - do not execute
                   the command */
		exit(81); /* we can't print stuff at this stage,
			     instead use exit codes for debugging */
	}
	
#ifndef __INSURE__
	/* close all other file descriptors, leaving only 0, 1 and 2. 0 and
	   2 point to /dev/null from the startup code */
	{
	int fd;
	for (fd=3;fd<256;fd++) close(fd);
	}
#endif

	execl("/bin/sh","sh","-c",cmd,NULL);  
	
	/* not reached */
	exit(82);
#endif
	return 1;
}
