/* 
   Unix SMB/Netbios implementation.
   Version 2.0.
   Copyright (C) Andrew Tridgell 1994-1999

   This file is part of the port to OpenVMS
   UNIX <-> VMS file specification conversion routines.
   Copyright (C) Eckart Meyer 1996-1999
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "includes.h"
extern int DEBUGLEVEL;

#include <stdio.h>
#include <string.h>
#include <unixlib.h>
#include <stdlib.h>
#include <ctype.h>

#include <descrip.h>
#include <starlet.h>
#include <dvidef.h>
#include <syidef.h>

#ifndef DVI$C_ACP_F11V5
#define DVI$C_ACP_F11V5 11	/* not defined before VMS 7.2 */
#endif

#ifdef strcpy
#undef strcpy
#endif

/* we must use the original getcwd since we want the VMS style path */
#ifdef getcwd
#undef getcwd
char *getcwd(char *buffer, unsigned int size, ...);
#endif

struct itm_list_3 {
	unsigned short len;
	unsigned short code;
	void *bufadr;
	short int *retadr;
};

#define ENCODE_CHECK 0
#define ENCODE_ALWAYS 1
#define ENCODE_NEVER 2

#undef strcat

static int case_encode = -1;
static int use_encode = -1;
globalref char dir_changed;
globalref char wrkdir[1024];

/****************************************************************************
 pw6_encode - encode filespec into pathworks compatible encoded name
 ***************************************************************************/
/*
 * Make a suitable VMS file name from a DOS long filename (in UNIX syntax).
 * Characters not possible for a VMS filename are replaced by '__NN' where
 * 'NN' is the two-digit hex ASCII code. Switch between case by '__$', default
 * is lowercase.
 * Dots are only allowed in filename, never in directory names and only if
 * dirflag is false.
 * If more than one dot is present, the last one is used for VMS (if at all).
 * Filenames without a dot are converted to 'FILE.', in filenames that end
 * in a dot therefore the last dot have to be encoded.
 * Dots at the beginning will will be left untouched.
 * On consecutive underscores ('__'), the first one is replaced by '__NN'.
 */
char encoded_file[2048];
static char *workfile = NULL;
static char *pw6_encode(const char *fname, int dirflag)
{
	char escbuf[5];
	char *s;
	char *p,*q;
	int dotcnt = 0;
	int ulcnt = 0;
	int ucase = 0;	/* 1: uppercase, -1: lowercase, 0: don't know */
	int i;
	int encoded_len;
	char *lastslash, *lastdot;


	encoded_file[0] = 0;
        if (*fname == 0) return "";
	if (dirflag) dotcnt++;	/* no dots allowed at all */
	i = strlen(fname);
	if (strcmp(fname,".") == 0  ||  strcmp(fname,"..") == 0) {
		StrCpy(encoded_file,fname);	/* leave untouched */
		return(encoded_file);
	}
	s = (char *)fname;
	if (strncmp(fname,"./",2) == 0) s += 2;	/* don't convert these */
	if (strncmp(fname,"../",3) == 0) s += 3;
	p = (char *)fname + i - 1;	/* point to last character */
	q = encoded_file + (5 * i); /* last byte in buffer */
	*q = '\0';
	if (*p == '.') {		/* last character is a dot */
		q -= 4;
		sprintf(escbuf,"__%02X",*p & 0xFF); 
		strcpy(q, escbuf);
		--p;
	}
/*
 * Scan path name from tail backwards to handle multiple dots correctly.
 */
	for (; p >= s; --p) {
		if ((*p < 'a' || *p > 'z')
		&& (*p < 'A' || *p > 'Z')
		&& (*p < '0' || *p > '9')
		&& *p != '$' && *p != '-' && *p != '/'
		&& *p != '*' && *p != '?' && *p != '%'
/*  #define DBL_UL  */
#if DBL_UL
		&& *p != '_'
#else
		&& (*p != '_' || ulcnt >= 1)
#endif
		&& (*p != '.' || dotcnt >= 1)) {
			q -= 4;
			if (*p == '_') {
				sprintf(escbuf,"__%02X",*p & 0xFF);
				memcpy(q, escbuf,4);
			}
			else {
				ulcnt = 0;
				sprintf(escbuf,"__%02X",*p & 0xFF);
				memcpy(q, escbuf, 4);
			}
		}
		else {
			if ( case_encode  &&  ucase
			&& (islower(*p) && ucase > 0
			  ||  isupper(*p) && ucase < 0) ) {
				q -= 3;
				memcpy(q,"__$",3);
			}
			if (islower(*p)) ucase = -1;
			if (isupper(*p)) ucase = +1;
			if (*p == '.') dotcnt++;
			if (*p == '/') {
				dotcnt++; /* no dots in directory */
				if (case_encode  &&  ucase > 0) {
					q -= 3;
					memcpy(q,"__$",3);
					ucase = -1;
				}
			}
			if (*p == '_') {
				ulcnt++;
			}
			else {
				ulcnt = 0;
			}
			*--q = *p;
		}
	}
	if (case_encode  &&  ucase > 0) {
		q -= 3;
		memcpy(q,"__$",3);
	}
	for (; p >= fname; --p) {
		*--q = *p;	/* copy possible ../ or ./ */
	}

	strcpy (encoded_file,q);
	encoded_len = strlen(encoded_file);

	lastslash = strrchr (encoded_file, '/');
	if (lastslash == NULL)
	    lastslash = encoded_file;
	else
	    lastslash++;
	lastdot = strchr (lastslash, '.');
	if (lastdot == 0)
	    return(encoded_file);
	*lastdot = 0;
	if (strlen(lastslash) < 40)
	{
	    *lastdot ='.';
	    return(encoded_file);
	}

	workfile = realloc(workfile, encoded_len + 20);
	strcpy (workfile,encoded_file);
	strcat (workfile,"__2E");
	strcat (workfile,lastdot+1);
	lastslash = strrchr (workfile,'/');
	if (lastslash == 0)
		lastslash = workfile;
	else
		lastslash++;
	strcpy (lastslash+40, lastslash+39);
	*(lastslash+39) = '.';
	strcpy (encoded_file, workfile);

	return(encoded_file);
}


/****************************************************************************
 pw6_decode - decode a pathworks compatible encoded filespec
 ***************************************************************************/
static char *decoded_file = NULL;
static char *pw6_decode(const char *fname)
{
	char escbuf[5];
	char *p,*q;
	int ucase = -1;	/* 1: uppercase, -1: lowercase, 0: don't know */
	int i;
	char *lastslash, *lastdot;

	workfile = realloc(workfile,strlen(fname) + 1);
	strcpy (workfile, fname);
	fname = workfile;
	lastslash = strrchr (fname, '/');
	if (lastslash == NULL)
	    lastslash = fname;
	else
	    lastslash++;
	lastdot = strchr (lastslash, '.');
	if (lastdot != NULL && strlen(fname) != 40)
	{
	    *lastdot = 0;
	    if (strlen(lastslash) == 39)
	    {
		char *pt;
		strcpy (lastdot, lastdot+1);
		pt = lastslash + strlen(lastslash) - 4;
		while ( (strncmp(pt,"__2e",4) != 0) && pt > lastslash)
			pt--;
		*pt ='.';
		strcpy (pt+1,pt+4);
	    }
	    else
		*lastdot = '.';
	}

	decoded_file = realloc(decoded_file,strlen(fname)+1);
	for (p=(char *)fname, q=decoded_file; *p != '\0'; p++) {
		i = sscanf(p,"__%02X",q); /* check for encode sequence */
		if (i > 0) {
			sprintf(escbuf,"__%02X",*q & 0xFF);	/* convert back */
			if (strncasecmp(escbuf,p,4) != 0) i = 0;  /* no valid code */
		}
		if (i > 0) {
			p += 3;
			q++;
		}
		else if (case_encode) {	/* check for case toggle */
			if (strncmp(p,"__$",3) == 0) {
				i = 1;
				ucase = -ucase;
				p += 2;
			}
		}
		if (i == 0) {	/* no special case */
			if (case_encode  &&  ucase > 0) {
				*q++ = toupper(*p);
			}
			else {
				*q++ = tolower(*p);
			}
		}
	}
	*q = '\0';
	if (*(q-1) == '.')
		*(q-1) = '\0';
	return(decoded_file);
}


/****************************************************************************
 encode_init - one time init
 ***************************************************************************/
static void encode_init()
{
	char *p;
	int k = -1;
	struct itm_list_3 itmlist[2];
	long iosb[2];
	long sts;
	char version[9];
	short int retlen;


	if (use_encode < 0) {		/* executed only once */
#ifdef SAMBA
/*
 * Init Samba's own character set default table.
 */
		charset_initialise();
#endif
/*
 * Check parameter
 */
#ifdef __alpha
		use_encode = ENCODE_CHECK;
		p = getenv("SAMBA_FILESPEC_ENCODE");
		if (p) {
			k = atoi(p);
			if (k == ENCODE_CHECK || toupper(*p) == 'C')
				use_encode = ENCODE_CHECK;
			if (k == ENCODE_ALWAYS || toupper(*p) == 'A')
				use_encode = ENCODE_ALWAYS;
			if (k == ENCODE_NEVER || toupper(*p) == 'N')
				use_encode = ENCODE_NEVER;
		}
/*
 * Check if our system is V7.2 or later. Previous systems
 * do not support EFS.
 */
		itmlist[0].code = SYI$_VERSION;
		itmlist[0].len = 8;
		itmlist[0].bufadr = version;
		itmlist[0].retadr = &retlen;
		itmlist[1].code = 0;
		itmlist[1].len = 0;
		sts = sys$getsyiw(0,0,0,itmlist,iosb,0,0);
		version[4] = '\0';
		if ((!(sts&1)) || (!(iosb[0]&1)) || strcmp(version,"V7.2") < 0 ) {
			use_encode = ENCODE_ALWAYS;	/* EFS not supported */
		}
#else
		use_encode = ENCODE_ALWAYS;	/* EFS not supported on VAX */
#endif
	}

	if (case_encode < 0) {
		case_encode = FALSE;	/* defaults to no case encoding */
		p = getenv("SAMBA_CASE_ENCODE");
		if (p) {
			case_encode = 1;
		}
	}
}
/****************************************************************************
 is_efs - check if filename refers to an extended file system volume (ODS5)
 valid for Alpha systems only.
 ***************************************************************************/
static BOOL is_efs(const char *vmsdevice)
{
	struct dsc$descriptor_s dsc_device;
	struct itm_list_3 itmlist[2];
	long iosb[2];
	long sts;
	long acptype;
	short int retlen;

	dsc_device.dsc$a_pointer = (char *)vmsdevice;
	dsc_device.dsc$w_length = strlen(vmsdevice);
	itmlist[0].code = DVI$_ACPTYPE;
	itmlist[0].len = 4;
	itmlist[0].bufadr = &acptype;
	itmlist[0].retadr = &retlen;
	itmlist[1].code = 0;
	itmlist[1].len = 0;
	sts = sys$getdviw(0,0,&dsc_device,itmlist,iosb,0,0,0);
	if ((!(sts&1)) || (!(iosb[0]&1))) return(FALSE);
	return( (acptype == DVI$C_ACP_F11V5) );
}

/****************************************************************************
 check_encode - check if we should encode/decode a filename
 ***************************************************************************/
static BOOL check_encode(const char *unixfile)
{
	char *p;
	char *q;

	encode_init();

	if (use_encode == ENCODE_ALWAYS) return(TRUE);
	if (use_encode == ENCODE_NEVER) return(FALSE);

	p = (char *)unixfile;
/*
 * If unixfile starts with "/", use first path element as device name.
 * Otherwise get current working directory for encode checking.
 */
	if (*p == '/') {
		q = strchr(p+1,'/');
		if (!q) q = p + strlen(p);
		decoded_file = realloc(decoded_file,q - p + 1);
		strncpy(decoded_file,p+1,q - p - 1);
		decoded_file[q - p - 1] = '\0';
		p = decoded_file;
	}
	else {
		if (dir_changed)
		{
		    getcwd(wrkdir,1024,0); /* original getcwd!! */
		    dir_changed = FALSE;
		}
		p = wrkdir;
	}
	return(!is_efs(p));
}

static char not_changed[1024];

/****************************************************************************
 vms_encode_filespec - encode unsuitable characters if not EFS
 always prepend './' if filespec does not start with '/' or '.'.
 ***************************************************************************/
char *vms_encode_filespec(const char *unixfile, int dirflag)
{
	char *p;
	char *q;
	char *r;
	int n;

	if (check_encode(unixfile)) {
		p = pw6_encode(unixfile, dirflag);
	}
	else {
		int i = strlen(unixfile);

		strcpy(not_changed,unixfile);
		if (not_changed[i-1] == '.')
			not_changed[i-1] = 0;
		p = (char *)not_changed;
	}
        if (encoded_file == NULL) return p;
	if (*p != '/' && *p != '.') {
		n = strlen(p);
		r = encoded_file + n + 2;
		*r = '\0';
		q = p + n;
		while (r > encoded_file + 2) *--r = *--q;
		*--r = '/';
		*--r = '.';
		p = r;
	}
	return(p);
}

/****************************************************************************
 vms_decode_filespec - if not EFS
 ***************************************************************************/
char *vms_decode_filespec(const char *vmsfile)
{
	if (check_encode(vmsfile)) {
		return(pw6_decode(vmsfile));
	}
	else {
		int i = strlen(vmsfile);
		
		strcpy(not_changed,vmsfile);
		if (not_changed[i-1] == '.')
			not_changed[i-1] = 0;
		return((char *)not_changed);
	}
}


static char *vms_file = NULL;
static int vms_file_type;

static int vms_action_routine(char *s, int type)
{
	vms_file_type = type;
	vms_file = realloc(vms_file,2*strlen(s)); /* much larger buffer */
	if (!vms_file) return(0);
	StrCpy(vms_file,s);
	return(1);
}

char *vms_cvt_unixfile(char *unixfile, int dirflag)
{
	if (unixfile == NULL) return(NULL);
	vms_file_type = 0;
	decc$to_vms(unixfile, vms_action_routine, 0, dirflag);
	if (vms_file_type == 0) {
		vms_file = realloc(vms_file,strlen(unixfile+1));
		StrCpy(vms_file, unixfile);
	}
	return(vms_file);
}

static char *device_file = NULL;

static char *handle_device_only(char *unixfile)
{
	char *s;
	char *p;
	char *q;
/*
 * File buffer should be about 2 times larger than the filespec
 */
	pstrcat(unixfile,"/000000");
	s = vms_cvt_unixfile(unixfile, 2);
/*
 * (above:) flag = 2: translates the device logical (if not phyical device):
 *
 * physical device:	s = ddnn:[000000]		contains no dots.
 * concealed device:	s = ddnn:[dir1.dir2.][000000]
 * logical to dir:	s = ddnn:[dir1.dir2.000000]
 *
 * For physical device construct:
 *
 *	ddnn:[000000]000000.dir
 *
 * For logicals (dots in name) construct:
 *
 *	ddnn:[dir1.dir2.-]dir2.dir
 *                ^    ^
 *                p    q
 */
	if ( (p=q=strrchr(s,'.')) ) {
		*q = '\0';	/* cut string at last dot */
		while (*--p != '.'  &&  *p != '[');	/* find previous */
		device_file = realloc(device_file,strlen(s) + strlen(p) + 20);
		StrCpy(device_file,s);		/* ddnn:[dir1.dir2 */
		pstrcat(device_file,".-]");	/* ddnn:[dir1.dir2.-] */
		safe_strcat(device_file,p+1,strlen(device_file));
						/* ddnn:[dir1.dir2.-]dir2 */
		pstrcat(device_file,".DIR");	/* ddnn:[dir1.dir2.-]dir2.dir */
	}
	else {
		device_file = realloc(device_file,strlen(s) + 20);
		StrCpy(device_file,s);
		pstrcat(device_file,"000000.DIR");
	}
	DEBUG(3,("VMS \"Device-\" File: %s\n",device_file));
	return(device_file);
}

/****************************************************************************
 vms_get_filespec - convert a UNIX filespec into a VMS filespec
 directories are represented by their VMS .DIR file.
 ***************************************************************************/
char *vms_get_filespec(const char *unixfile)
{
	char *s;
	char *p;
	char *q;
	char cwd[1024];
/*
 * First translate into what VMS can use: Uppercase, "$", "_" and "-" only.
 */
	s = vms_encode_filespec(unixfile, FALSE);
/*
 * Check on . and ..
 */
	if (strcmp(s,".") == 0) {
		if (dir_changed)
		{
			getcwd(wrkdir,1024,0);
			dir_changed = FALSE;
		}
		s = wrkdir;
	}
	if (strcmp(s,"..") == 0) {
		if (dir_changed)
		{
			getcwd(wrkdir,1024,0);
			dir_changed = FALSE;
		}
		strcpy(cwd,wrkdir);
		p = strrchr(cwd,'/');
		if (p) *p = '\0';
		s = cwd;
	}
/*
 * Check if we start with a "/", meaning that the first part of the path
 * is a device (incl. concealed device).
 */
	if (*s == '/') {
		if ( !(p = strchr(s+1,'/')) ) {
			return(handle_device_only(s));
		}
		else if (strstr(s,"000000") == 0) {
			q = s + strlen(s);
			while (q > p) {		/* shift 7 characters */
				*(q+7) = *q;
				--q;
			}
			memcpy(p+1,"000000/",7); /* insert */
		}
	}
/*
 * Check type
 */
	DEBUG(3,("Convert to VMS: %s -> %s\n",unixfile,s));
	p = vms_cvt_unixfile(s, 0);
	if (vms_file_type == 2) {
		p = vms_cvt_unixfile(s, 1);
		pstrcat(p,".DIR");
	}
/*
 * If the filename does not contain a dot, append one to prevent
 * VMS from using a default extension (.TXT).
 */
	q = strrchr(p, ']');		/* isolate file name from path */
	if (!q) q = p;
	q = strchr(q, '.');		/* any dots in file name ? */
	if (!q) pstrcat(p, ".");	/* no, append one */ 

	DEBUG(3,("VMS File: %s\n",p));
	return(p);
}

/****************************************************************************
 vms_get_filedir - convert a UNIX path into a VMS directory
 	/ddnn/name -> DDNN:[NAME]
 ***************************************************************************/
char *vms_get_filedir(const char *unixfile)
{
	char *p;
/*
 * First translate into what VMS can use: Uppercase, "$", "_" and "-" only.
 */
	p = vms_encode_filespec(unixfile, TRUE);
	DEBUG(3,("Convert to VMS: %s -> %s\n",unixfile,p));
	p = vms_cvt_unixfile(p, 2);
	DEBUG(3,("VMS Directory: %s\n",p));
	return(p);
}

/****************************************************************************
 vms_get_unix_path
 convert a VMS directory into a UNIX path
 ***************************************************************************/
char *vms_get_unix_path(const char *vmsfile)
{
	char *s;
	char *p;
/*
 * Convert VMS syntax to UNIX path.
 * .DIR and trailing dots are removed.
 */
	vms_file = realloc(vms_file,strlen(vmsfile)+1);
	StrCpy(vms_file,vmsfile);	/* make a copy */
	s = vms_file;
	p = s + strlen(s) - 1;
	if (*p == '.') *p = '\0';	/* remove trailing dot */
	s = decc$translate_vms(vms_file);
	if (!s) {
		s = vms_file;	/* use copy */
	}
	DEBUG(3,("Convert to UNIX: %s -> %s\n",vmsfile,s));
	p = strrchr(s,'.');		/* find extension */
	if (p  &&  strcasecmp(p,".DIR") == 0) *p = '\0'; /* remove .DIR */
/*
 * Decode path
 */
	s = vms_decode_filespec(s);
	DEBUG(3,("UNIX path: %s\n",s));
	return(s);
}
