/* 

   Unix SMB/Netbios implementation.

   Version 3.0

   Samba readline wrapper implementation

   Copyright (C) Simo Sorce 2001

   Copyright (C) Andrew Tridgell 2001

   

   This program is free software; you can redistribute it and/or modify

   it under the terms of the GNU General Public License as published by

   the Free Software Foundation; either version 2 of the License, or

   (at your option) any later version.

   

   This program is distributed in the hope that it will be useful,

   but WITHOUT ANY WARRANTY; without even the implied warranty of

   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the

   GNU General Public License for more details.

   

   You should have received a copy of the GNU General Public License

   along with this program; if not, write to the Free Software

   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

/*

 * Make special VMS-specific version of the smb_readline() and cmd_history()

 * functions found in [-.lib]readline.c

 */

#include <smg$routines.h>		/* VMS RTL screen management */

#include <smgdef.h>

#include <smgtrmptr.h>

#include <starlet.h>			/* VMS system service prototypes */

#include <descrip.h>			/* VMS string descriptors */

#include <ssdef.h>			/* VMS system service cond. codes */

#include <dcdef.h>			/* VMS device class codes */

#include <ttdef.h>			/* VMS terminal driver codes */



#include <ctype.h>



#include "includes.h"





static unsigned long keyboard, smg_state = 0;

static unsigned long tdata_handle = 0;

static char cursor_up[64];

static struct smg$r_attribute_info_block kb_info;

/*

 * Test if all characters in line are blank.

 */

static int line_is_blank ( const char *line )

{

    int i, length;

    length = strlen ( line );

    for ( i = 0; i < length; i++ ) if ( !isspace(line[i]) ) return 0;

    return 1;

}



/****************************************************************************

 Display the prompt and wait for input. Call callback() regularly

****************************************************************************/



char *smb_readline(char *prompt, void (*callback)(void), 

		   char **(completion_fn)(const char *text, int start, int end))

{

	char *ret;

	int fd = fileno(stdin);

    int status, length, count, i;

    static pstring line, ini_line;

    static $DESCRIPTOR(line_dx,line);

    static $DESCRIPTOR(ini_dx, ini_line);

    static $DESCRIPTOR(prompt_dx,"");

    static $DESCRIPTOR(input_dev, "SYS$INPUT");

    /*

     * Initialize SMG if first call.

     */

    if ( smg_state == 0 ) {

	int code, dev_class, tt_type, cu_size, cu_length;

	/*

	 * Create a virtual keyboard, freak out if init fails.

	 */

 	smg_state = 1;

	status = smg$create_virtual_keyboard ( &keyboard, 0, 0, 0, 0 );

	if ( (status&1) == 0 ) {

	    DEBUG(0,("smb_readline: Failed to create virtual keyboard\n" ));

	    exit ( status );

	}

	/*

	 * See if keyboard is a terminal and initialize term-table

	 * info so we can cheat.

	 */

	length = sizeof(kb_info);

	status = SMG$GET_KEYBOARD_ATTRIBUTES ( &keyboard, &kb_info, &length );

	if ( (status&1) && (kb_info.smg$b_dev_class == DC$_TERM) ) {



	    tt_type = kb_info.smg$b_dev_type;

	    status = SMG$INIT_TERM_TABLE_BY_TYPE (&tt_type, &tdata_handle, 0);

	    if ( (status&1) == 1 ) {

		code = SMG$K_CURSOR_UP;

		cu_size = sizeof(cursor_up) - 1;

		status = SMG$GET_TERM_DATA ( &tdata_handle, &code,

			&cu_size, &cu_length, cursor_up, 0 );

		if ( (status&1) == 0 ) cu_length = 0;

		cursor_up[cu_length] = 0;

		smg_state = 2;

	    }

	}

    }

    /*

     * Set up for read.

     */

    prompt_dx.dsc$a_pointer = prompt;

    prompt_dx.dsc$w_length = strlen ( prompt );

    line_dx.dsc$w_length = sizeof(line) - 2;

    length = 0;

    /*

     * Read line using SMG so we get command recall buffer.

     */

    if ( callback ) {

	/*

	 * Perform read, but periodically call user's function.

	 */

	int timeout_sec = 5;

	ini_dx.dsc$w_length = 0;



	for ( status = SS$_TIMEOUT; status == SS$_TIMEOUT; ) {

	    /*

	     * Get next line.

	     */

	    status = smg$read_composed_line ( &keyboard, 0, &line_dx,

		&prompt_dx, &length, 0, 0, &ini_dx, &timeout_sec, 0, 0, 0 );

	    

	    if ( status == SS$_TIMEOUT ) {

		/*

		 * More that five seconds elapsed without a character

		 * entered.  Call the user's work function and redo.

	         */

		callback();

		memcpy ( ini_line, line, length );

		ini_dx.dsc$w_length = length;

		/*

		 * Back up the cursor one or more lines to counter the newlines

	 	 * sent by a previous read_composed line completing.

	 	 */

		if ( smg_state == 2 ) {

		    printf ( "%s", cursor_up );

		    count = ((prompt_dx.dsc$w_length+ini_dx.dsc$w_length) / 

			kb_info.smg$w_num_columns);

		    for (i=0; i<count; i++ ) printf ( "%s", cursor_up );

		}

	    }

	} 

	if ( (status&1) == 0 ) return 0;



    } else {

	/*

	 * Simple call.

         */

	status = smg$read_composed_line ( &keyboard, 0, &line_dx,

		&prompt_dx, &length, 0, 0, 0, 0, 0, 0, 0 );

	if ( (status&1) == 0 ) return 0;

    }

    line[length] = 0;			/* terminate line */

    /*

     * see if line is all whitespace and remove from recall buffer.

     */

    if ( line_is_blank ( line ) ) {

	int flags;

	flags = 0; /* SMG$M_KEEP_CONTENTS; */

	SMG$REPLACE_INPUT_LINE ( &keyboard, 0, 0, &flags );

    }

    return line;

}



/****************************************************************************

history

****************************************************************************/

void cmd_history(void)

{

    int lnum, i, status, length;

    static pstring line;

    static $DESCRIPTOR(line_dx,line);



    for ( lnum = 1; lnum <= 200; lnum++ ) {

	length = 0;

	status = smg$return_input_line ( &keyboard, &line_dx,  0, &lnum,

		&length );

	if ( (status&1) == 0 ) break;

        

	line[length] = '\0';

	if ( !line_is_blank(line) ) printf ( "%4d: %s\n", lnum, line );

    }

}

