/* 
   Unix SMB/Netbios implementation.
   Version 2.0.
   Copyright (C) Andrew Tridgell 1994-1999

   This file is part of the port to OpenVMS
   General VMS support routines.
   Copyright (C) Eckart Meyer 1996-1999
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/*	vms_support.c
 *	V2.0			28-Feb-1999	IfN/Mey
 *
 *	Copyright E. Meyer <meyer@ifn.ing.tu-bs.de>
 *      With addition from J.Y. Collot
 *+
 * Samba VMS support.
 * Misc. routines not present on VMS.
 *
 * NOTE: includes.h *IS* included here.
 *-
 */
#include "includes.h"

#include <starlet.h>
#include <lib$routines.h>
#include <descrip.h>
#include <dvidef.h>
#include <quidef.h>
#include <jpidef.h>
#include <uaidef.h>
#include <ssdef.h>
#include <iodef.h>
#include <ttdef.h>
#include <descrip.h>
#include <prvdef.h>
#include <fab.h>
#include <time.h>

#define DIR_CACHE_TIMEOUT 30

#undef strcpy
#undef strcat

static unsigned int opendir_calls;
static unsigned int opendir_real;

globaldef char exact_size;
globaldef char dir_changed = TRUE;

#define TYPE_STAT 1
#define TYPE_DIR 2

globaldef char wrkdir[1024];
globaldef char do_encode = TRUE;

int vms_setuid_1(uid_t uid, char **new_username, uid_t *new_uid);
int vms_setuid_2();
long sys_set_username(void *);		/* These are the ... */
long sys_set_uic(unsigned int *);	/* ...kernel hacking routines */

struct id {
	unsigned long uic;
	unsigned long attr;
};

struct uai {
	char name[13];		/* must be at first position */
	unsigned long uic;
	unsigned long priv[2];
	struct id *id_list;
	char dir[256];
	char password[9];
	struct uai *next;
};

struct itemdesc {
  short int buffer_length;
  short int item_code;
  char *buffer;
  short int *ret_length;
  };

struct itm_list_3 {
	unsigned short len;
	unsigned short code;
	void *bufadr;
	short int *retadr;
};

/* SAMBA debugging */
extern int DEBUG_LEVEL;

static struct uai *cuai = NULL;
static char result[64];
static char pwd[256];
static $DESCRIPTOR(termdesc, "SYS$INPUT:");
static short termchan = -1;
static struct passwd password_struct;
static char login_name[64];
static struct group user_grp;
static char gr_name[32];
static struct uai current_uai = {""};
static struct uai *pnew_uai;


#ifdef getpwnam
#undef getpwnam
struct passwd * getpwnam (const char * __name);
#endif

static char pw_dir[256] = "";
static struct passwd password_struct = {NULL,0,0,pw_dir,NULL};

/****************************************************************************
 get_user_identifiers
 ***************************************************************************/
static struct id *get_user_identifiers(unsigned long uic)
{
	unsigned long sts;
	unsigned long ctx;
	struct id *id_list;
	struct id holder;
	struct id id;
	int count;

	holder.uic = uic;
	holder.attr = 0;

	id_list = (struct id *)malloc(sizeof(struct id));
	if (id_list == NULL) return(NULL);

	count = 0;
	ctx = 0;

	while ( (sts=sys$find_held(&holder, &id.uic, &id.attr, &ctx))&1 ) {
		id_list = (struct id *)realloc(id_list,
					sizeof(struct id) * (count+2) );
		if (id_list == NULL) return(NULL);
		id_list[count].uic = id.uic;
		id_list[count].attr = id.attr;
		DEBUG(6,("  identifier %d = %08X\n",count+1,id.uic));
		count++;
	}

	id_list[count].uic = -1;
	DEBUG(6,("  uic %08X holds %d identifiers\n",uic,count));
	return(id_list);
}

/****************************************************************************
 get_uai - get UAI info from VMS
 ***************************************************************************/
static struct uai *get_uai(char *username)
{
	short int iosb[4], lgth[6];
	long int st;
	struct {
		struct itemdesc id[6];
		int eol;
	} itmlst;
	struct {
		int size;
		char *ptr;
	} namedesc;
	char defdir[64];
	char defdev[64];
	char usrnam[14];
	char *p;

	DEBUG (3,("get_uai: for username = \"%s\"\n", username));

	if (!cuai) {	/* first time init */
		cuai = (struct uai *)malloc(sizeof(struct uai));
	}
	else {		/* return memory used by previous contents */
		if (cuai->id_list) free(cuai->id_list);
	}
	StrCpy(cuai->name, username);	/* Preset */

	itmlst.id[0].item_code = UAI$_PWD;
	itmlst.id[0].buffer_length = 8;
	itmlst.id[0].buffer = cuai->password;
	itmlst.id[0].ret_length = &lgth[0];

	itmlst.id[1].item_code = UAI$_DEFDIR;
	itmlst.id[1].buffer_length = 64;
	itmlst.id[1].buffer = defdir;
	itmlst.id[1].ret_length = &lgth[1];

	itmlst.id[2].item_code = UAI$_DEFDEV;
	itmlst.id[2].buffer_length = 64;
	itmlst.id[2].buffer = defdev;
	itmlst.id[2].ret_length = &lgth[2];

	itmlst.id[3].item_code = UAI$_UIC;
	itmlst.id[3].buffer_length = 4;
	itmlst.id[3].buffer = (char *) &cuai->uic;
	itmlst.id[3].ret_length = &lgth[3];

	itmlst.id[4].item_code = UAI$_PRIV;
	itmlst.id[4].buffer_length = 8;
	itmlst.id[4].buffer = (char *) cuai->priv;
	itmlst.id[4].ret_length = &lgth[4];

	itmlst.id[5].item_code = UAI$_USERNAME;
	itmlst.id[5].buffer_length = 12;
	itmlst.id[5].buffer = cuai->name;
	itmlst.id[5].ret_length = &lgth[5];

	itmlst.eol = 0;

	namedesc.size = strlen(username);
	namedesc.ptr = username;
	st = sys$getuai(0, 0, &namedesc, &itmlst, 0, 0, 0);
	if ((st != SS$_NORMAL) && !(st & 0x01)) {
		DEBUG (3,("get_uai: sys$getuai error: %08X %d\n", st, st));
		return(NULL);
	}

	for (p = cuai->name + lgth[5] - 1; p >= cuai->name && *p == ' ';--p) *p = '\0';
	cuai->password[8] = '\0';
	defdir[defdir[0]+1] = '\0';
	defdev[defdev[0]+1] = '\0';
	pstrcat(defdev + 1, defdir +1 );	/* append dir to device */
	p = vms_get_unix_path(defdev+1);
	StrCpy(cuai->dir, p);
	DEBUG (5,("get_uai: login dir = %s\n", cuai->dir));

	cuai->id_list = get_user_identifiers(cuai->uic);
#if 0
	if (uai_head == NULL) {
		cuai->next = NULL;
		uai_head = cuai;
	}
	else {
		cuai->next = uai_head;
		uai_head = cuai;
	}
#endif

DEBUG    (3,("get_uai: $getuai returns UIC = %08X\n", cuai->uic));
DEBUGADD (3,("         priv = %08X %08X\n", cuai->priv[0],cuai->priv[1]));
	return(cuai);
}

/****************************************************************************
 copy_pw - copy from DEC-C passwd to our extended passwd struct
 ***************************************************************************/
static copy_pw(struct passwd * my_pw, struct passwd * vms_pw)
{
	my_pw->pw_name = vms_pw->pw_name;
	my_pw->pw_uid = vms_pw->pw_uid;
	my_pw->pw_gid = vms_pw->pw_gid;
	my_pw->pw_dir = vms_pw->pw_dir;
	my_pw->pw_shell = vms_pw->pw_shell;
	my_pw->pw_passwd = NULL;
	my_pw->pw_comment = NULL;
	my_pw->pw_gecos = NULL;
/*
 * Samba needs SYSTEM to be [0,0]
 */
	if (strcasecmp(vms_pw->pw_name,"SYSTEM") == 0) {
		my_pw->pw_uid = 0;
		my_pw->pw_gid = 0;
	}
}

/****************************************************************************
 vms_set_process_name
 ***************************************************************************/
void vms_set_process_name(char *name)
{
	int lgth;
	char buf[16];	/* max 15 characters */
	struct dsc$descriptor prcnam;

	lgth = strlen(name);
	if (lgth > 15) lgth = 15;
	strncpy(buf,name,lgth);
	prcnam.dsc$a_pointer = buf;
	prcnam.dsc$w_length = lgth;
	sys$setprn(&prcnam);
}

/****************************************************************************
 lp_share_modes - HACK to switch off share modes
 ***************************************************************************/
BOOL lp_share_modes(i)
{
	return(FALSE);
}

/****************************************************************************
 statfs - return mounted file system info
 ***************************************************************************/
int vms_statfs(const char *path, struct statfs *buf)
{
	unsigned long iosb[2];
	struct itm_list_3 dvi_itm[4];
	unsigned long sts;
	char *name;
	struct dsc$descriptor dev;

	name = vms_get_filespec(path);
	DEBUG (3,("vms_statfs: path = \"%s\", vms_file = \"%s\"\n",path,name));

	dvi_itm[0].code = DVI$_MAXBLOCK;
	dvi_itm[0].len = 4;
	dvi_itm[0].bufadr = &buf->f_blocks;	/* total blocks */
	dvi_itm[0].retadr = NULL;

	dvi_itm[1].code = DVI$_FREEBLOCKS;
	dvi_itm[1].len = 4;
	dvi_itm[1].bufadr = &buf->f_bfree;	/* free blocks */
	dvi_itm[1].retadr = NULL;

	dvi_itm[2].code = DVI$_MAXFILES;
	dvi_itm[2].len = 4;
	dvi_itm[2].bufadr = &buf->f_files;	/* total files in file system */
	dvi_itm[2].retadr = NULL;

	dvi_itm[3].code = 0;
	dvi_itm[3].len = 0;

	dev.dsc$a_pointer = name;
	dev.dsc$w_length = strlen(dev.dsc$a_pointer);
	dev.dsc$b_class = DSC$K_CLASS_S;
	dev.dsc$b_dtype = DSC$K_DTYPE_T;

	sts = sys$getdviw(0,0,&dev,&dvi_itm,iosb,0,0,0);
	if ( (!(sts&1)) || (!(iosb[0]&1)) ) {
		DEBUG (0,("vms_statfs: $GETDVI ERROR for %s: sts = %08X, iosb = %08X\n",
			name,sts,iosb[0]));
		return(-1);
	}
	
	buf->f_type = 0;
	buf->f_bsize = 512;		/* unsupported */
	buf->f_bavail = buf->f_bfree;	/* avaiable blocks to non-super-user */
	buf->f_ffree = -1;		/* free files, unsupported */
	DEBUG (4,("   f_blocks = %d\n",buf->f_blocks));
	DEBUG (4,("   f_bfree = %d\n",buf->f_bfree));
	DEBUG (4,("   f_files = %d\n",buf->f_files));
	return(0);
}

/****************************************************************************
 getpwnam - return user data by name
 ***************************************************************************/
struct passwd *vms_getpwnam(const char *name)
{
	struct passwd *pw;
	struct uai *puai;
	char *p;

	p = (char *)name;
DEBUG (3,("getpwnam: name = \"%s\"\n",name));

	if (strcasecmp(name,"root") == 0) {
		p = "SYSTEM";
	}
	if (password_struct.pw_name == NULL  
	|| strcasecmp(name,password_struct.pw_name) != 0) {
		pw = getpwnam(p);
		if (!pw) return(NULL);
		copy_pw(&password_struct, pw);
		puai = get_uai(pw->pw_name);
		password_struct.pw_passwd = puai->password;
		StrCpy(pw_dir, puai->dir);
		password_struct.pw_dir = pw_dir;
	}
	return(&password_struct);
}

/****************************************************************************
 dummy_true - routine always returns TRUE
 ***************************************************************************/
BOOL dummy_true(void)
{
	return(TRUE);
}

#ifdef getuid
#undef getuid
#define getuid __unix_getuid
uid_t getuid();
#endif
/****************************************************************************
 getuid - get uic from VMS
 ***************************************************************************/
uid_t vms_getuid()
{
	int uid[2] = {0, 0};
	$DESCRIPTOR (root_ident,"SAMBA_ROOT");
	unsigned int ident = -1;
	unsigned int held;
	unsigned int context = 0;
	int status = 1;
	char isroot = FALSE;

	uid[0] = getuid();
	sys$asctoid(&root_ident,&ident,0);

	while ((status & 1) == 1)
	{
	   status = sys$find_held(uid, &held, 0, &context);
	   if (held == ident)
		isroot = TRUE;
	}
	if (isroot)
		return(0);
	else
		return(uid[0]);
}

#ifdef getgid
#undef getgid
gid_t getgid();
#endif
/****************************************************************************
 getgid - get uic group part from VMS
 ***************************************************************************/
gid_t vms_getgid()
{
	gid_t gid;
	gid = getgid();
	if (gid == 1) gid =0;
	return(gid);
}

off_t vms_get_filesize (struct stat *st_buf, char *fname)
{
/*
    FILE *fp;
    char *ptr;
*/
    int fd;
    char line[8192];
    unsigned int size = 0;
    int linsiz;
    static char called=0;

    if (called == 1)
	return 0;

    called = 1;
        
    if (st_buf->st_fab_rfm == FAB$C_FIX)
    {
	unsigned int nbrec;
	unsigned int new_size;
	if ((st_buf->st_fab_mrs % 2) == 1)
		nbrec = st_buf->st_size / (st_buf->st_fab_mrs + 1);
	else
		nbrec = st_buf->st_size / st_buf->st_fab_mrs;
	new_size = nbrec*(st_buf->st_fab_mrs + 2);
	DEBUG(3,("VMS file size %s, rfm = %d, size = %d\n", fname, 
		st_buf->st_fab_rfm, new_size));
        called = 0;
	return (new_size);
    }
    else
    {
	fd = stm_open(fname,0,0);
	while (	(linsiz = stm_read(fd,line,8192)) > 0)
	     size += linsiz;
	stm_close(fd);
	DEBUG(3,("VMS file size %s, rfm = %d, size = %d\n", fname, 
		st_buf->st_fab_rfm, size));
        called = 0;
	return size;
    }
}
#ifdef fopen
#undef fopen
FILE *fopen (const char *__filename, const char *__mode, ...);
#endif
/****************************************************************************
 fopen
 ***************************************************************************/
FILE *vms_fopen (const char *filename, const char *mode, ...)
{
	char *p;
	FILE *fp;
	int dbg_save = DEBUG_LEVEL;
/*
 * We've to turn off debug output since the debug routine could try to
 * open the log via fopen... You've got it....
 */
	DEBUG_LEVEL = 0;
	p = vms_encode_filespec(filename,FALSE);
	fp = fopen(p,mode,"shr=get,put");
	DEBUG_LEVEL = dbg_save;
	return(fp);
}

/****************************************************************************
 do_file_lock
 This routine is a dummy since on VMS each opened file implicity locked.
 On the other hand, we do not habe the locking functions of fcntl()
 ***************************************************************************/
BOOL do_file_lock(int fd, int waitsecs, int type)
{
	return(TRUE);
}

struct group *vms_getgrent()
{
   return(NULL);
}

void vms_endgrent()
{
}

void vms_setgrent()
{
}

/****************************************************************************
 getgroups - VMS users belong to one group only

 The value is taken from the last getpw* call.
 ***************************************************************************/
int vms_getgroups(int gidsetsize, gid_t grouplist[])
{
	if (gidsetsize > 0) {
		grouplist[0] = password_struct.pw_gid;
	}
	return(1);
}

/****************************************************************************
 getprgid - return group data by gid
 ***************************************************************************/
struct group *vms_getgrgid(gid_t gid)
{
	struct dsc$descriptor_s dsc_name;
	unsigned short namlen;
	unsigned long sts;
	unsigned long id;

	dsc_name.dsc$a_pointer = gr_name;
	dsc_name.dsc$w_length = sizeof(gr_name) - 1;
	dsc_name.dsc$b_class = DSC$K_CLASS_S;
	dsc_name.dsc$b_dtype = DSC$K_DTYPE_T;
	id = ((gid&0xFFFF) << 16) | 0xFFFF;
	sts = sys$idtoasc(id,&namlen,&dsc_name,0,0,0);
	if (!(sts&1)) return(NULL);
	gr_name[namlen] = '\0';
	user_grp.gr_name = gr_name;
	user_grp.gr_passwd = NULL;	/* not supported */
	user_grp.gr_gid = gid;
	user_grp.gr_mem = NULL;		/* not supported */
	return(&user_grp);
}

/****************************************************************************
 getgrnam - return user data by name
 ***************************************************************************/
struct group *vms_getgrnam(const char *name)
{
	struct dsc$descriptor_s dsc_name;
	unsigned long sts;
	unsigned long id;

	dsc_name.dsc$a_pointer = (char *)name;
	dsc_name.dsc$w_length = strlen(name);
	dsc_name.dsc$b_class = DSC$K_CLASS_S;
	dsc_name.dsc$b_dtype = DSC$K_DTYPE_T;
	sts = sys$asctoid(&dsc_name,&id,0);
	if (!(sts&1)) return(NULL);
	id = (id >> 16) & 0xFFFF;
	user_grp.gr_name = (char *)name;
	user_grp.gr_passwd = NULL;	/* not supported */
	user_grp.gr_gid = id;
	user_grp.gr_mem = NULL;		/* not supported */
	return(&user_grp);
}

/****************************************************************************
 lp_locking - locking cannot be used.
 ***************************************************************************/
BOOL lp_locking(int i)
{
	return(FALSE);
}

/****************************************************************************
 fcntl_lock
 ***************************************************************************/
BOOL fcntl_lock(int fd, int op, SMB_OFF_T offset, SMB_OFF_T count, int type)
{
	return(TRUE);
}

/****************************************************************************
 crypt

 Special version. We need to know the username here since the salt value
 is not taken from the given argument, but from the UAI. Since a user
 name is provided by the caller we use the user last seen by the getpw*
 routines. Seems to work for samba...
 ***************************************************************************/
char *crypt(const char *key, const char *salt)
{
	short int st, iosb[4], lgth[3];
	struct {
		struct itemdesc id[3];
		int eol;
	} itmlst;
	struct {
		int size;
		char *ptr;
	} fullnamedesc, namedesc, passwdesc;
	long encrypt, ssalt;
	char username[64];
	char *p;
	char *q;
	char *login_name;

	encrypt = 0;
	ssalt = 0;

	itmlst.id[0].item_code = UAI$_SALT;
	itmlst.id[0].buffer_length = 2;
	itmlst.id[0].buffer = (char *) &ssalt;
	itmlst.id[0].ret_length = &lgth[0];

	itmlst.id[1].item_code = UAI$_ENCRYPT;
	itmlst.id[1].buffer_length = 1;
	itmlst.id[1].buffer = (char *) &encrypt;
	itmlst.id[1].ret_length = &lgth[1];

	itmlst.id[2].item_code = UAI$_USERNAME;
	itmlst.id[2].buffer_length = 64;
	itmlst.id[2].buffer = username;
	itmlst.id[2].ret_length = &lgth[2];

	itmlst.eol = 0;

	login_name = cuai->name;
	namedesc.size = strlen(login_name);
	namedesc.ptr = login_name;
	DEBUG (3,  ("crypt: username = \"%s\"\n",login_name));
	if ((st=sys$getuai(0,0,&namedesc,&itmlst,0,0,0)) != SS$_NORMAL) {
		DEBUG (0,("sys$getuai error %08X %d\n", st, st));
		result[0] = '\0';
		return(result);
	}
	passwdesc.ptr = malloc(strlen(key)+1);
	p = key;
	q = passwdesc.ptr;
#ifdef toupper
#undef toupper
#endif
	for(;*p != 0;p++) 
	    *q++ = toupper(*p);
	*q = '\0';
	passwdesc.size = strlen(key);
	DEBUG (100,("crypt: key = \"%s\"\n",passwdesc.ptr));
	fullnamedesc.size = strlen(login_name);
	fullnamedesc.ptr = login_name;
	st = sys$hash_password (
             	 &passwdesc, 
             	 encrypt, 
             	 ssalt, 
             	 &fullnamedesc, 
             	 &result);
	free(passwdesc.ptr);
	result[8] = '\0';
	DEBUG    (3,("crypt: result=%08X\n",*((int *)result)));
	DEBUGADD (3,("crypt: passwd=%08X\n",*((int *)cuai->password))); /**/
	return(result);
}

/****************************************************************************
 getpass - read with prompt and don't echo
 ***************************************************************************/
char *getpass(char *prompt)
{
	int st;
	short int iosb[4];

	printf("\n");
	if (termchan == -1) {
		sys$assign(&termdesc,&termchan,0,0);
	}
	st = sys$qiow (
             	 0, 
             	 termchan, 
             	 IO$_READPROMPT | IO$M_NOECHO, 
             	 iosb, 
             	 0, 
             	 0, 
             	 pwd, 
             	 255, 
             	 0, 
             	 0, 
             	 prompt, 
             	 strlen(prompt));
	pwd[iosb[1]] = 0;
	return(pwd);
}

int vms_lstat(const char *file_spec, struct stat *st_buf)
{
	return(vms_stat(file_spec,st_buf));
}

/****************************************************************************
 setuid - set user
 ***************************************************************************/
int vms_setuid(uid_t uid)
{
	int st;
	uid_t new_uid;
	char *new_username;
	struct {
		int size;
		char *ptr;
	} userdesc;

	st = vms_setuid_1(uid, &new_username, &new_uid);
	if (st <= 0) return(st);
/*
 *	- set username/uic - this is kernel mode hacking!
 */
	userdesc.size = strlen(new_username);
	userdesc.ptr  = new_username;
	st = sys_set_username(&userdesc);
	if (st != SS$_NORMAL) return(-1);
	st = sys_set_uic(&new_uid); /* do this after $GRANTID */
	if (st != SS$_NORMAL) return(-1);

	return(vms_setuid_2());
}

/****************************************************************************
 seteuid - dummy
 ***************************************************************************/
int vms_seteuid(gid_t uid)
{
	return(vms_setuid(uid));
}

/****************************************************************************
 grant_identifiers
 ***************************************************************************/
static void grant_identifiers(struct id *id_list)
{
	unsigned long sts;

	if (id_list == NULL) return;
	while(id_list->uic != -1) {
		DEBUG(6,("  grant identifier %08X\n",id_list->uic));
		sts = sys$grantid(0,0,id_list,0,0);
		id_list++;
	}
}

/****************************************************************************
 revoke_identifiers
 ***************************************************************************/
static void revoke_identifiers(struct id *id_list)
{
	unsigned long sts;

	if (id_list == NULL) return;
	while(id_list->uic != -1) {
		DEBUG(6,("  revoke identifier %08X\n",id_list->uic));
		sts = sys$revokid(0,0,id_list,0,0);
		id_list++;
	}
}



static void copy_uai(struct uai *dst, struct uai *src)
{
	int count;
	struct id *id_list;

	if (src == NULL  ||  dst == NULL) return;
	memcpy(dst,src,sizeof(struct uai));

	id_list = (struct id *)malloc(sizeof(struct id));
	if (id_list == NULL) return;

	count = 0;

	while ( src->id_list[count].uic != -1 ) {
		id_list = (struct id *)realloc(id_list,
					sizeof(struct id) * (count+2) );
		if (id_list == NULL) break;
		id_list[count].uic = src->id_list[count].uic;
		id_list[count].attr = src->id_list[count].attr;
		DEBUGADD(6,("copy_uai: identifier %d = %08X\n",count+1,id_list[count].uic));
		count++;
	}

	id_list[count].uic = -1;
	DEBUGADD(6,("   copied %d identifiers\n",count));
	dst->id_list = id_list;
}

#ifdef getpwuid
#undef getpwuid
struct passwd * getpwuid (uid_t __uid);
#endif
/****************************************************************************
 getpwuid - return user data by uid
 ***************************************************************************/
struct passwd *vms_getpwuid(uid_t uid)
{
	struct passwd *pw;
	struct uai *puai;
/*
 * Samba needs SYSTEM to be [0,0]
 */
	if (uid == 0) uid = 0x00010004L;	/* SYSTEM is [1,4] */
DEBUG (3,("getpwuid: uid = [%08X]\n",uid));
	if (password_struct.pw_name == NULL  
	||  uid != password_struct.pw_uid) {
		pw = getpwuid(uid);
		if (!pw) return(NULL);
		copy_pw(&password_struct, pw);
		puai = get_uai(pw->pw_name);
		password_struct.pw_passwd = puai->password;
		StrCpy(pw_dir, puai->dir);
		password_struct.pw_dir = pw_dir;
	}
	return(&password_struct);
}

/****************************************************************************
 vms_setuid_1 - called by setuid - lookup old and new user
 ***************************************************************************/
int vms_setuid_1(uid_t uid, char **new_username, uid_t *new_uid)
{
	struct passwd *pw;
	int st;
	struct dsc {
		int size;
		char *ptr;
	};
	long sys_set_username(void *);		/* These are the ... */
	long sys_set_uic(unsigned int *);	/* ...kernel hacking routines */
	unsigned long tmpprv[2];

	DEBUG (3,("vms_setuid: uid = %08X\n",uid));
/*
 * Current UAI
 */
	if (*current_uai.name == '\0') {
		struct passwd *pw = vms_getpwuid(vms_getuid());
		if (!cuai) return(-1);	/* vms_getpwuid implicitly set cuai */
		copy_uai(&current_uai,cuai);
	}
/*
 * New UAI
 */
	if (uid == 0) {
		*new_uid = 0x00010004;
	}
	else {
		*new_uid = uid;
	}
/*
 * If same than current, nothing to do.
 */
	if (*new_uid == current_uai.uic) return(0);

/*
 * Set new use user name, UIC and privileges.
 *
 *	- clear all privs and set CMKRNL for setting username/uic.
 */

	tmpprv[0] = 0xFFFFFFFF;
	tmpprv[1] = 0xFFFFFFFF;
	st = sys$setprv(1,tmpprv,1,0);
	if (st != SS$_NORMAL) return(-1);

	if (uid == 0) {
		pnew_uai = get_uai("SYSTEM");
	}
	else {
		pw = getpwuid(*new_uid);
		if (!pw) return(-1);
		pnew_uai = get_uai(pw->pw_name);
	}
	if (!pnew_uai) return(-1);
	DEBUG (3,("vms_setuid: uid = %08X, username = \"%s\", uic = %08X\n",
		*new_uid,pnew_uai->name,pnew_uai->uic));
/*
 *	- revoke current user's identifiers and setup new
 */
	revoke_identifiers(current_uai.id_list);
	grant_identifiers(pnew_uai->id_list);

/*
 * Return new name and uid
 */
	*new_username = pnew_uai->name;
	return(1);
}


int vms_setuid_2()
{
	int st;
	unsigned long tmpprv[2] = {-1,-1};
/*
 * New UAI is now current - free old resource!
 */
	if (current_uai.id_list) free(current_uai.id_list);
	copy_uai(&current_uai,pnew_uai);
/*
 *	- clear privileges required for hacking...
 */
	st = sys$setprv(0,tmpprv,1,0);	/* reset those privs */
	if (st != SS$_NORMAL) return(-1);
/*
 *	- finally set new users privs
 */
/*
 * We always need SYSLCK and CMKRNL
 */
	pnew_uai->priv[0] = pnew_uai->priv[0] | (PRV$M_SYSLCK + PRV$M_CMKRNL);
	st = sys$setprv(1,pnew_uai->priv,1,0);	/* set new user's privs */
	if (st != SS$_NORMAL) return(-1);

	return(0);
}

#ifdef mkdir
#undef mkdir
#endif
int vms_mkdir(const char *name, mode_t mode)
{
    char *vmsdir = vms_encode_filespec(name,TRUE);
    return (mkdir (vmsdir, mode));
}

/****************************************************************************
 rmdir - Get the VMS directory filespec and delete the <path>.DIR;1 file.
 ***************************************************************************/
int vms_rmdir (char *path)
{
	char *local_path;
	int sts;

	local_path = malloc (strlen(path) + 5);
	strcpy (local_path, path);
	strcat (local_path,".dir");
	chmod (local_path,0777);	
	sts = vms_delete(local_path);
	free (local_path);
	return sts;
}

static char *cur_dir_ptr;
static char *dir_buf;
#undef opendir
#undef readdir
#undef closedir

/****************************************************************************
 opendir
 ***************************************************************************/
DIR *vms_opendir (const char *dirname)
{
    char *dirnm;
    char *tmp;
    DIR *dir_context;
    struct dirent *d;
    char *tmp1;
    int allocated = 0;
    int used = 0;
    int size;
    char *p;
    int n;
    int sts;
    struct stat dir_stat;
    char wrkfil[2048];
    char copyfil[2048];

    opendir_calls++;

    dirnm = vms_encode_filespec(dirname,TRUE);

    if (*dirnm != '/')
    {
	if (dir_changed)
	{
	    getcwd (wrkdir, 255, 0);
	    dir_changed = FALSE;
	}
	strcpy (wrkfil,wrkdir);
	strcat (wrkfil,"/");
	strcat (wrkfil,dirnm);
    }
    else
    {
	strcpy (wrkfil,dirnm);
    }
    tmp = strstr(wrkfil,"//");
    if (tmp != NULL)
	strcpy (tmp,tmp+1);
    tmp = strstr(wrkfil,"/./");
    if (tmp != NULL)
	strcpy (tmp,tmp+2);
    tmp = strstr(wrkfil,"/.");
    if (tmp != 0)
	*tmp = 0;
    n = strlen(wrkfil);
    if (wrkfil[n-1] == '/')
	wrkfil[n-1] = 0;

    cur_dir_ptr = vms_check_cache(wrkfil,TYPE_DIR);
    if (cur_dir_ptr != NULL)
    {
	return ((DIR *) &dir_context);	
    }

    opendir_real++;

    dir_context = opendir(wrkfil);    
    if (dir_context == NULL)
	return NULL;

skip_this_file:
    while ((d = readdir(dir_context)) != NULL)
    {
	/* We ignore files name .; */
	if (strcmp(d->d_name,".") == 0)
	    goto skip_this_file;

	p = vms_decode_filespec(d->d_name);
	size = strlen(p);
	if ((used + size + 2) > allocated)
	{
	    if (allocated == 0)
	    {
		dir_buf = malloc(1024);
		allocated = 1024;
	    }
	    else
	    {
		dir_buf = realloc (dir_buf, allocated + 1024);
		allocated = allocated + 1024;
	    }
	}
	strcpy (dir_buf+used, p);
	used = used + size + 1;
    }
    if (allocated == 0)
    {
	dir_buf = malloc(1);
	*dir_buf = 0;
    }
    else
    {
	*(dir_buf+used+1) = 0;
	*(dir_buf+used+2) = 0;
    }
   strcpy(copyfil, wrkfil);
   sts = stat(wrkfil, &dir_stat);
   if (sts == 0)
   {
	vms_insert_cache(copyfil,dir_buf,TYPE_DIR,dir_stat.st_ino); 
   }
   cur_dir_ptr = dir_buf;

   closedir(dir_context);
   return (dir_context);	
}


/****************************************************************************
 readdir
 ***************************************************************************/
struct dirent *vms_readdir (DIR * dirp)
{
    static struct dirent *d;
    
    if (d == NULL)
	d = malloc(sizeof(struct dirent));

    if (*cur_dir_ptr != 0)
    {
	pstrcpy(d->d_name, cur_dir_ptr);
	cur_dir_ptr = cur_dir_ptr + strlen(cur_dir_ptr) + 1;
	return(d);    
    }
    else
    {
	return(NULL);
    }
}

int *vms_closedir (DIR * dirp)
{
    return 0;
}

#ifdef delete
#undef delete
#endif
int vms_delete (const char *path)
{
    int sts;
    char curdir[512];
    char *tmp;
    char *delfil = vms_encode_filespec(path,FALSE);

    sts = delete(delfil);
    strcpy (curdir,path);
    tmp = strrchr(curdir,'/');
    if (tmp == 0)
	vms_opendir("./");
    else
    {
	*tmp = 0;
	vms_opendir(curdir);
    }
    
    return (sts);
}
#ifdef rename
#undef rename
int rename (const char *__old, const char *__new);
#endif
/****************************************************************************
 rename - Rename files. Take care of VMS directory files.
 ***************************************************************************/
int vms_rename(const char *oldname, const char *newname)
{
	char *buf1;
	char *buf2;
	int sts;
	char curdir[512];
	char *tmp;

	buf2 = vms_encode_filespec(oldname,FALSE);
	buf1 = malloc(strlen(buf2)+10);
	StrCpy(buf1,buf2);
	buf2 = vms_encode_filespec(newname,FALSE);
	if (strchr (buf1+1,'.') == 0)
		strcat (buf1,".");
	if (strchr (buf2+1,'.') == 0)
		strcat (buf2,".");
	DEBUG(3,("vms_rename: VMS: %s -> %s\n",buf1, buf2));
	sts = rename(buf1, buf2);
	if (sts != 0)
	{
	   /* it could be directories... */
		strcat (buf1,"dir");
		strcat (buf2,"dir");
		sts = rename(buf1, buf2);
	}
	free(buf1);
	strcpy (curdir,oldname);
	tmp = strrchr(curdir,'/');
	if (tmp == 0)
	    vms_opendir("./");
	else
	{
	    *tmp = 0;
	    if (strcmp(curdir,".") == 0)
		vms_opendir("./");
	    else
		vms_opendir(curdir);
	}
	return(sts);
}

#undef chdir
int vms_chdir(char *dir_spec)
{
	dir_changed = TRUE;
	return(chdir(dir_spec));
} 

int vms_fstat(int fd, struct stat *st_buf)
{
	char filename[1024];
	char *tmp;
        char *p;
	int sts;

	memset(filename,0,sizeof(filename));
	memset(st_buf,0,sizeof(struct stat));
	tmp = getname(fd,filename,0);
	if (tmp == (char *) -1)
		return(-1);
	tmp = strrchr(filename,'.');
	if (tmp != 0) *tmp = 0;
	do_encode = FALSE;
	sts = vms_stat(filename,st_buf);
	do_encode = TRUE;
	return(sts);
}

int vms_fchown(int fd, uid_t uid, gid_t gid)
{
    char vmsname[1024];
    getname (fd, vmsname,0);
    return (chown (vmsname, uid, gid));
}

int vms_fchmod (int fd, mode_t mode)
{
    char vmsname[1024];
    getname (fd, vmsname, 0);
    return (chmod (vmsname,mode));
}

