/*
	Copyright (C) 2004 Jonas Lindholm

	This software was developed by Jonas Lindholm, jlhm@usa.net

	History

	V1.0		Jonas Lindholm	2004-05-14

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <string.h>

#include <ldap.h>

#include <starlet.h>
#include <ssdef.h>
#include <stsdef.h>
#include <psldef.h>
#include <jpidef.h>
#include <descrip.h>

#include <pthread.h>

#include "lgildap.h"

#include "server.h"


/*
	Description:

	This routine try to establish a session with the LDAP server
	using the identity and credential supplied by the user.

	If the job type is network and the password check fail the assword
	is converted to lower case and the check is done again but only if
	the logical name LGI_LDAP_PWD_NETWORK_IGNORE_CASE is defined.

	If logical name LGI_LDAP_PWD_IGNORE_CASE is defined the password
	is checked with both upper and lower case in addition to the mixture
	of characters the user enter at the password prompt.

	INPUT:		session handler
			LGI message

	OUTPUT:		SS$_NORMAL if the credential was ok
			LLSRV_INVCRED if the credential was incorrect

	IMPLICIT:	The identity must be valid, flag identityok must be set
*/

int thread_cmd_authenticate(LGI_SESS_T *sess, LIM_ENTRY_T *le) {
	int		stat;

        le->lim.rplyid = le->lim.reqid;
        le->lim.reqid++;
	le->lim.status = LLSRV_INVCRED;

	if (sess->dn != NULL && strlen(sess->dn) > 0 && sess->flags.identityok) {
		int		cnt = 0;
		int		igncase;
		char		dummy[256];
		char		*pwd;

		LDAP_LOCK;

		igncase = ($VMS_STATUS_SUCCESS(do_trnlnm("LGI_LDAP_PWD_IGNORE_CASE", dummy, sizeof(dummy) - 1, PSL$C_EXEC)) ||
			   (sess->jobtype == JPI$K_NETWORK &&
			    $VMS_STATUS_SUCCESS(do_trnlnm("LGI_LDAP_PWD_NETWORK_IGNORE_CASE", dummy, sizeof(dummy) - 1, PSL$C_EXEC))));

		for(; cnt < (igncase ? 3 : 1); cnt++) {

			switch (cnt) {

				case 0: break;  /* try with mixture of upper and lower case */
				case 1: for(pwd = le->lim.request.authenticate.credential; *pwd ; *pwd++ = tolower(*pwd)); break; /* lower case */
				case 2: for(pwd = le->lim.request.authenticate.credential; *pwd ; *pwd++ = toupper(*pwd)); break; /* upper case */

			}

			pwd = le->lim.request.authenticate.credential;

			stat = ldap_simple_bind_s(sess->ldap, sess->dn, pwd);

			if (stat == 0) { /* password is ok */

				LDAP_UNLOCK;

				synch_password(sess, pwd); /* update hash in SYSUAF.DAT if required */

				thread_cmd_password(sess, le); /* store valid password for the process */

				memset(pwd, 0, sizeof(le->lim.request.authenticate.credential)); /* password not needed any more */

				le->lim.status = SS$_NORMAL;

				sess->flags.credentialok = 1;

				return SS$_NORMAL;

			}
		}

		memset(pwd, 0, sizeof(le->lim.request.authenticate.credential));

		LDAP_UNLOCK;
        }

	return SS$_ABORT;
}
