/*
	Copyright (C) 2004 Jonas Lindholm

	This software was developed by Jonas Lindholm, jlhm@usa.net

	History

	V1.0		Jonas Lindholm	2004-05-14

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <string.h>

#include <ldap.h>

#include <starlet.h>
#include <ssdef.h>
#include <stsdef.h>
#include <psldef.h>
#include <descrip.h>

#include <lib$routines.h>

#include <pthread.h>

#include "lgildap.h"

#include "server.h"


/*
	Description:

	This routine is invoked when the client request the supplied identity
	to be verified.
	The verification is done using the base DN defined with logical name
	LGI_LDAP_BASEDN as base. The LDAP_SCOPE_SUBTREE is used to search any children
	under the base DN for a match.
	The logical name LGI_LDAP_FILTER can be defined to filter the matches.
	Default filer is "(&(objectclass=posixAccount)(uid=%s))". This filter
	will match entries that has an uid that match the supplied identity (username).

	Should several entries match for the same identity the first one is used.

	If several uid's are returned the first one that match an already existing
	SYSUAF entry is used. Should there be no matching SYSUAF entry the first
	returned uid is used to create a user in SYSUAF but only if logical name
	LGI_LDAP_ADD_USER is defined.

	If the specified identity result in an uid that correspond to a SYSUAF entry
	the validity of the identity is granted. Otherwise the identity is denied.

	INPUT:		session handler
			LGI message

	OUTPUT:		If the identity is valid SS$_NORMAL
			If the identity is invalid SS$_ABORT. The message status
			will be LLSRV_NOSUCHUSER.
*/

int thread_cmd_identity(LGI_SESS_T *sess, LIM_ENTRY_T *le) {
	int		stat;
	char		basedn[256];

	sess->jobtype = le->lim.request.identity.jobtype;

        le->lim.rplyid = le->lim.reqid;
        le->lim.reqid++;
	le->lim.status = LLSRV_NOSUCHUSER;

	if (!$VMS_STATUS_SUCCESS(do_trnlnm("LGI_LDAP_BASEDN", basedn, sizeof(basedn) - 1, PSL$C_EXEC))) {

		lib$stop(LLSRV_LDAPBASEDNMISS, 0);

	}

	if (sess->ldap != NULL) {
		int			msgid;
		char			search[255 + 32 * 2 + 1] = "";
		char                    *attrs[] = { "uid", NULL };

		get_filter(search, sizeof(search) - 1, le->lim.request.identity.identity);

		LDAP_LOCK;

	        if ((msgid = ldap_search(sess->ldap, basedn, LDAP_SCOPE_SUBTREE, search, attrs, 0)) > 0) {
			struct timeval		to = { 0, 0 };
			LDAPMessage		*res = NULL;
			LDAPMessage             *e = NULL;

			while ((stat = ldap_result(sess->ldap, msgid, 1, &to, &res)) == 0) {
				struct timespec		wait = { 0, 10000 };

				LDAP_UNLOCK;
				pthread_delay_np(&wait);
				LDAP_LOCK;

			}

			if (stat > 0) {
				int		cnt = 0;

				for (; cnt < 2; cnt++) { /* 0 = verify without creating, 1 = create if missing */

					for ((e = ldap_first_entry(sess->ldap, res)); e != NULL; e = ldap_next_entry(sess->ldap, e)) {

						if ((sess->dn = ldap_get_dn(sess->ldap, e)) != NULL) {
							char			**uids;

							if ((uids = ldap_get_values(sess->ldap, e, "uid")) != NULL) {
								int		idx = 0;

								while (uids[idx] != NULL) {

									LDAP_UNLOCK;

									if ($VMS_STATUS_SUCCESS(verify_username(sess, uids[idx], cnt))) {

										strncpy(sess->identity, uids[idx], sizeof(sess->identity));
										sess->identity[strlen(uids[idx])] = 0;

										strcpy(le->lim.reply.identity.identity, sess->identity);

										le->lim.status = SS$_NORMAL;

										sess->flags.identityok = 1;

										LDAP_LOCK;

										ldap_value_free(uids);
                         							ldap_msgfree(res);

										LDAP_UNLOCK;

										return SS$_NORMAL;
									}

									LDAP_LOCK;

									idx++;
								}

								ldap_value_free(uids);
							}

							ldap_memfree(sess->dn);
							sess->dn = NULL;
						}
					}
				}

				ldap_msgfree(res);

				LDAP_UNLOCK;

       				return SS$_ABORT;
			}
		}

		ldap_unbind(sess->ldap);
		sess->ldap = NULL;
	}

	LDAP_UNLOCK;

	return SS$_ABORT;
}
