/*
        Copyright (C) 2004 Jonas Lindholm

        This software was developed by Jonas Lindholm, jlhm@usa.net

        History

        V1.0            Jonas Lindholm  2004-05-14

        This program is free software; you can redistribute it and/or modify
        it under the terms of the GNU General Public License as published by
        the Free Software Foundation; either version 2 of the License, or
        (at your option) any later version.

        This program is distributed in the hope that it will be useful,
        but WITHOUT ANY WARRANTY; without even the implied warranty of
        MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
        GNU General Public License for more details.

        You should have received a copy of the GNU General Public License
        along with this program; if not, write to the Free Software
        Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
 
#include <stdio.h>
#include <string.h>

#include <pthread.h>

#include <starlet.h>
#include <descrip.h>
#include <ssdef.h>
#include <stsdef.h>
#include <jpidef.h>

#include <lib$routines.h>
#include <libdef.h>

#include "ldap.h"

#include "lgildap.h"
#include "server.h"



void *session(void *s);


int unsigned		newsessqh[4] = { 0, 0, 0, 0 };
int unsigned		delsessqh[4] = { 0, 0, 0, 0 };
int unsigned		msgsessqh[4] = { 0, 0, 0, 0 };
int unsigned		waitsessqh[4] = { 0, 0, 0, 0 };

int			use_ssl = 0;
int			sesscnt = 0;

pthread_cond_t		sesscnd;
pthread_mutex_t		sessmtx;
pthread_mutex_t		sesscntmtx;


/*
	Description:

	This routine is invoked when the session thread must finish.
	This occure when the ICC connection is disconnected.

	If there are any waiting sessions to be run the next one is started.

	INPUT:		session handler

	OUTPUT:		none
*/

void thread_finish(LGI_SESS_T *sess) {
	int			stat;
	LIM_ENTRY_T		*le;

	stat = sys$icc_disconnectw(sess->conn, &sess->icciosb, 0, 0, 0, 0);

	if (sess->ldap != NULL) {

		LDAP_LOCK;

		if (sess->dn != NULL) { ldap_memfree(sess->dn); }

		ldap_unbind_s(sess->ldap);

		LDAP_UNLOCK;
	}

	pthread_mutex_destroy(&sess->mtxle);
	pthread_cond_destroy(&sess->cndle);
	pthread_mutex_destroy(&sess->mtxw);
	pthread_cond_destroy(&sess->cndw);

	pthread_detach(sess->thr);

	memset(sess, 0, sizeof(LGI_SESS_T)); /* clear sensitive data */

	insert_qt(&freesessqh, sess);

	pthread_mutex_lock(&sesscntmtx);

	sesscnt--;

	pthread_mutex_unlock(&sesscntmtx);

	if ((sess = remove_qh(&waitsessqh)) != NULL) {

		stat = pthread_create(&sess->thr, NULL, &session, sess);

	}

	pthread_exit(NULL);
}


/*
	Description:

	This routine is invoked when the client sends the init command.
	The only purpose is to establish an LDAP session to the LDAP server.

	INPUT:		session handler
			LGI message

	OUTPUT:		updated LGI status.
*/

void thread_cmd_init(LGI_SESS_T *sess, LIM_ENTRY_T *le) {
	int			stat = SS$_NORMAL;

	le->lim.rplyid = le->lim.reqid;
	le->lim.reqid++;
	le->lim.status = LLSRV_INVLOGIN;

	if ((sess->ldap = open_ldap()) != NULL) {

		le->lim.status = SS$_NORMAL;

	}
}


/*
	Description:

	This routine is invoked as a thread.
	It's purpose is to read ICC messages from the client and do commands
	requested by the client.
	If the client request an unknown command the thread disconnect the session
	and terminate.

	INPUT:		session handler

	OUTPUT:		none
*/

void *session(void *s) {
	int			stat;
	LGI_SESS_T		*sess = s;

	sess->flags.running = 1;

	while (1) {
                LIM_ENTRY_T		le;

		do {

			stat = sys$icc_receive(sess->conn, &le.iosb, 0, 0, &le.lim, sizeof(le.lim));

			if (stat != SS$_NORMAL) {

				pthread_cond_wait(&sess->cndle, &sess->mtxle);

			}

		} while (!sess->flags.finish && stat != SS$_NORMAL);

		if (sess->flags.finish || !$VMS_STATUS_SUCCESS(le.iosb[0])) { /* we must finish this thread */

			thread_finish(sess);

		}

		switch (le.lim.cmd) {

			case IPC_CMD_INIT: {

				thread_cmd_init(sess, &le);
				break;

			}

			case IPC_CMD_IDENTITY: {

				thread_cmd_identity(sess, &le);
				break;

			}

			case IPC_CMD_AUTHENTICATE: {

				thread_cmd_authenticate(sess, &le);
				break;

			}

			case IPC_CMD_PASSWORD: {

				thread_cmd_password(sess, &le);
				break;

			}

			default: {

				memset(&le, 0, sizeof(le)); /* must clear sensitive data */
				thread_finish(sess);

			}
		}

		sess->icciosb[1] = le.iosb[2];

		stat = sys$icc_replyw(sess->conn, &sess->icciosb, 0, 0, &le.lim, sizeof(le.lim));

		memset(&le, 0, sizeof(le)); /* must clear sensitive data */

		if (!$VMS_STATUS_SUCCESS(stat)) {

			thread_finish(sess);

		}
	}

	return NULL;
}


/*
	Description:

	This routine is invoked for every new ICC connection.

	INPUT:		session handler

	OUTPUT:		none
*/

void new_sess(LGI_SESS_T *sess) {
	int		stat;

	sess->flags.accepted = 1;

	if ((stat = sys$icc_accept(sess->conn, 0, 0, sess, 0)) == SS$_NORMAL) {

		stat = pthread_cond_init(&sess->cndw, NULL);
		stat = pthread_mutex_init(&sess->mtxw, NULL);
		stat = pthread_mutex_init(&sess->mtxle, NULL);
		stat = pthread_cond_init(&sess->cndle, NULL);

		pthread_mutex_lock(&sesscntmtx);

		if (sesscnt++ < MAX_THREADS) { 

			stat = pthread_create(&sess->thr, NULL, &session, sess);

		} else {

			insert_qt(&waitsessqh, sess);

		}

		pthread_mutex_unlock(&sesscntmtx);

	} else { /* the accept failed. */

		memset(sess, 0, sizeof(LGI_SESS_T));

		insert_qt(&freesessqh, sess);

		lib$signal(stat);

	}
}


/*
	Description:

	This routine is invoked when the ICC connecion is disconnected.
	The only thing this routine does is to set the finish flag for the session
	and wake it up.

	INPUT:		session handler

	OUTPUT:		none
*/

void del_sess(LGI_SESS_T *sess) {

	sess->flags.finish = 1;
	pthread_cond_signal(&sess->cndle);

}


/*
	Description:

	This routine wait for new and deleted ICC connections and dispatch
	the session handler to correct routine.

	INPUT:		none

	OUTPUT:		none
*/

void handle_sessions() {

	while (1) {
		LGI_SESS_T		*sess;

		while ((sess = remove_qh(&newsessqh)) != NULL) {

			new_sess(sess);

		}

		while ((sess = remove_qh(&delsessqh)) != NULL) {

			del_sess(sess);

		}

		pthread_cond_wait(&sesscnd, &sessmtx);
	}
}


/*
	Description:

	This is the main routine.
	It initialize some conditions and mutex used by all modules.
	It also initialize the exception handler, utilities, start SSL tunnel
	if ldap_tls_start is not supported.
	It also start threads handling SYSUAF tasks.
	The last initialization is to establish the ICC server before it
	wait for new and deleted ICC sessions.

	INPUT:		number of command line arguments
			array of command line arguments

	OUTPUT:		SS$_NORMAL
*/

int main(int argc, char *argv[]) {

	pthread_cond_init(&sesscnd, NULL);
	pthread_mutex_init(&sessmtx, NULL);
	pthread_mutex_init(&sesscntmtx, NULL);
	pthread_mutex_init(&ldapmtx, NULL);

	exception_init();

	util_init();

#if LDAP_API_VERSION < 2005
	ssl_init();
#endif

	uaf_init();

	icc_init();

	handle_sessions();

	return SS$_NORMAL;
}
