/*
        Copyright (C) 2004 Jonas Lindholm

        This software was developed by Jonas Lindholm, jlhm@usa.net

        History

        V1.0            Jonas Lindholm  2004-05-14

        This program is free software; you can redistribute it and/or modify
        it under the terms of the GNU General Public License as published by
        the Free Software Foundation; either version 2 of the License, or
        (at your option) any later version.

        This program is distributed in the hope that it will be useful,
        but WITHOUT ANY WARRANTY; without even the implied warranty of
        MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
        GNU General Public License for more details.

        You should have received a copy of the GNU General Public License
        along with this program; if not, write to the Free Software
        Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#define	FD_SETSIZE	128

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <fcntl.h>
#include <time.h>

#include <dlfcn.h>

#include <pthread.h>

#include <starlet.h>
#include <descrip.h>
#include <ssdef.h>
#include <stsdef.h>
#include <psldef.h>
#include <cmbdef.h>
#include <dvidef.h>
#include <lnmdef.h>
#include <lckdef.h>
#include <iodef.h>
#include <agndef.h>
#include <syidef.h>
#include <jpidef.h>
#include <rmidef.h>

#include <lib$routines.h>
#include <libvmdef.h>

#include <ldap.h>
#include "lgildap.h"

#include "server.h"

#include <in.h>
#include <inet.h>
#include <socket.h>

#include <openssl/crypto.h>
#include <openssl/ssl.h>
#include <openssl/err.h>



int		(*_CRYPTO_num_locks)();
void		(*_CRYPTO_set_id_callback)(void *);
void		(*_CRYPTO_set_locking_callback)(void *);

void		(*_SHA1_Init)(SHA_CTX *);
void		(*_SHA1_Update)(SHA_CTX *, const void *, long unsigned);
void		(*_SHA1_Final)(char unsigned *, SHA_CTX *);

void		(*_SSL_library_init)();
void		(*_SSL_load_error_strings)();
SSL_METHOD	*(*_SSLv3_method)();
SSL_CTX		*(*_SSL_CTX_new)(SSL_METHOD *);
SSL		*(*_SSL_new)(SSL_CTX *);
void		(*_SSL_free)(SSL *);
int		(*_SSL_set_fd)(SSL *, int);
int		(*_SSL_connect)(SSL *);
int		(*_SSL_pending)(SSL *);
void		(*_SSL_shutdown)(SSL *);
void		(*_SSL_free)(SSL *);
void		(*_SSL_CTX_free)(SSL_CTX *);
int		(*_SSL_write)(SSL *, void *, int);
int		(*_SSL_read)(SSL *, void *, int);
int		(*_SSL_get_error)(SSL *, int);

int		(*_RAND_add)(void *, int);


int unsigned		dq_hdr[4] = { 0, 0, 0, 0 };

pthread_mutex_t		*ssl_lcks;
pthread_t		thrssl;


/*
	Description:

	This routine return a free DQ entry with allocated data are of 1500 bytes.

	INPUT:		none

	OUTPUT:		pointer to free DQ.
*/

DQ_T *get_free_dq() {
        DQ_T		*dq = remove_qh(&dq_hdr);

	if (dq == NULL) {

		if ((dq = alloc_qe(sizeof(DQ_T))) != NULL) {

			dq->data = safe_malloc(1500);

		}
	}

	return dq;	
}


/*
	Description:

	This routine initialize SSL methid and SSL context.
	Should any of these routines fail the program will terminate.

	INPUT:		pointer to SSLv3 method
			pointer to SSL context

	OUTPUT:		updated SSLv3 method pointer
			updated SSL conext pointer
*/

int setup_CTX(SSL_METHOD **meth, SSL_CTX **ctx) {

        if ((*meth = _SSLv3_method()) != NULL) {

        	if ((*ctx = _SSL_CTX_new(*meth)) != NULL) {

			return SS$_NORMAL;

		} else {

			lib$stop(LLSRV_SSLCTXFAIL);

		}

	} else {

		lib$stop(LLSRV_SSLMETHFAIL);

	}

	return SS$_ABORT;
}


/*
	Description:

	This routine will establish a SSL session to the LDAP server.

	INPUT:		SSL

	OUTPUT:		SSL socket or 0 if the connection failed.
*/

int connect_SSL(SSL *ssl) {
        int                     stat;
        int                     ls;
	int			enable = 1;
        struct sockaddr_in      server_addr;
	char			ldaphost[256];

	stat = do_trnlnm("LGI_LDAP_HOST", ldaphost, sizeof(ldaphost) - 1, PSL$C_EXEC);

	if (!$VMS_STATUS_SUCCESS(stat)) {

		lib$stop(LLSRV_LDAPHOSTMISS);

	}

	if ((ls = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP)) > 0) {


		memset(&server_addr, 0, sizeof(server_addr));
        	server_addr.sin_family = AF_INET;
		server_addr.sin_port = htons(636);
        	server_addr.sin_addr.s_addr = inet_addr(ldaphost);

        	if ((stat = connect(ls, (struct sockaddr *)&server_addr, sizeof(server_addr))) == 0) {

			if ((stat = ioctl(ls, FIONBIO, &enable)) == 0) {

        			if ((stat = _SSL_set_fd(ssl, ls)) == 1) {

					do {
						struct fd_set           rs;
						struct fd_set           ws;

				                FD_ZERO(&rs);
				                FD_ZERO(&ws);
                				FD_SET(ls, &rs);
                				FD_SET(ls, &ws);

               					stat = select(FD_SETSIZE, &rs, &ws, NULL, NULL);
						stat = _SSL_connect(ssl);

					} while (stat != 1 &&
						 ((_SSL_get_error(ssl, stat) == SSL_ERROR_WANT_READ) ||
						  (_SSL_get_error(ssl, stat) == SSL_ERROR_WANT_WRITE)));

					if (stat == 1) {

						return ls;

					}
				}

				lib$signal(LLSRV_SSLSETFD, 1, _SSL_get_error(ssl, stat));

			} else {

				lib$signal(LLSRV_SSLSOCKNBIO);

			}

		} else {

			lib$signal(LLSRV_SSLCONN, 2, &ldaphost, (int)636);

		}

		close(ls);

	} else {

		lib$signal(LLSRV_SSLSOCK);

	}

	return 0;
}


/*
	Description:

	This is the thread that handle all SSL session between this VMS host and
	the LDAP server.
	It establish a listen socket and wait for connection from other threads in
	this program to connect. When a client connection is established a new SSL
	session is established with the LDAP server and all data received from the
	client is sent to the LDAP server and responses are send back to the client.

	Note. This thread will create a listen port on 127.0.0.1. This make it possible
	for other processes than this to connect to the LDAP server using this tunnel
	function. There is a risk for a DoS attack but only by processes allowed to run
	on the host. I recommend you upgrade to latest LDAP version that handle TLS.
	In that case this module is never used.

	INPUT:		condtion to signal when the thread is ready to receive connections.

	OUTPUT:		none
*/

void *ssl_thr(void *s) {
        int                     stat;
        int			*cs2ss = safe_malloc(sizeof(int) * FD_SETSIZE + 8);
        int			*ss2cs = safe_malloc(sizeof(int) * FD_SETSIZE + 8);
	SSL			**ss2ssl = safe_malloc(sizeof(void *) * FD_SETSIZE + 8);
        SSL_CTX                 *ctx;
        SSL_METHOD              *meth;
	int unsigned		*cshdr;
	int unsigned		*sshdr;
	pthread_cond_t		*cndw = s;
	int			size = 16 * FD_SETSIZE + 8;
	struct fd_set           rs;
	struct fd_set           ws;

	cshdr = alloc_qe(size);
	sshdr = alloc_qe(size);

	FD_ZERO(&rs);
	FD_ZERO(&ws);

	if ($VMS_STATUS_SUCCESS(setup_CTX(&meth, &ctx))) {
		int			ls;
		int			enable = 1;
		struct sockaddr_in      sa_serv;

		if ((ls = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP)) == 0) {

			lib$stop(LLSRV_SSLLISTSOCK);

		}

		if ((stat = setsockopt(ls, SOL_SOCKET, SO_REUSEADDR, &enable,sizeof(enable))) != 0) {

			lib$stop(LLSRV_SSLLISTOPT);

		}

		memset(&sa_serv, 0, sizeof(sa_serv));

		sa_serv.sin_family = AF_INET;
		sa_serv.sin_addr.s_addr = inet_addr("127.0.0.1");
		sa_serv.sin_port = htons(SSL_PORT);

		if ((stat = bind(ls, (struct sockaddr *)&sa_serv, sizeof(sa_serv))) != 0) {

			lib$stop(LLSRV_SSLLISTBIND);

		}

        	if ((stat = listen(ls, 16)) != 0) {

			lib$stop(LLSRV_SSLLISTLIST);

		}

		FD_SET(ls, &rs);

		lib$signal(LLSRV_SSLTHRSTARTED, 1, SSL_PORT);

		pthread_cond_signal_int_np(cndw);

		while (1) {
			int			s;
			struct fd_set		urs;
			struct fd_set		uws;

			memcpy(&urs, &rs, sizeof(urs));
			memcpy(&uws, &ws, sizeof(uws));
 
			stat = select(FD_SETSIZE, &urs, &uws, NULL, NULL);

			for(s = 2; s < FD_SETSIZE; s++) {

				if (FD_ISSET(s, &uws)) {

					if (cs2ss[s] != 0) { /* data to client */
						DQ_T		*dq = remove_qh(&cshdr[s * 4]);

						if (dq != NULL) {

	                				stat = write(s, dq->data, dq->len);
							insert_qt(&dq_hdr, dq);

						} else {

							FD_CLR(s, &ws);

                				}
					}

					if (ss2ssl[s] != NULL) { /* data to SSL */
						DQ_T		*dq = remove_qh(&sshdr[s * 4]);

						if (dq != NULL) {
			
		                			stat = _SSL_write(ss2ssl[s], dq->data, dq->len);

							if (stat < 0) { /* SSL want read instead */

								insert_qh(&sshdr[s * 4], dq);

							} else {

								insert_qt(&dq_hdr, dq);

							}

		       				} else {

							FD_CLR(s, &ws);

						}
					}
				}

				if (FD_ISSET(s, &urs)) {

					if (s == ls) { /* new connection */
                                		int			cs;
        					struct sockaddr_in      sa_cli;
        					int unsigned            cl = sizeof(sa_cli);

        					if ((cs = accept(ls, (struct sockaddr *)&sa_cli, &cl)) != 0) {
							int		ss;
							int		enable = 1;
							SSL		*ssl = _SSL_new(ctx);

							stat = ioctl(cs, FIONBIO, &enable);

							if ((ss = connect_SSL(ssl)) > 0) {

								cs2ss[cs] = ss;
								ss2ssl[ss] = ssl;
								ss2cs[ss] = cs;
								FD_SET(cs, &rs);
								FD_SET(ss, &rs);

							} else {

								close(cs);
								lib$signal(LLSRV_SSLCONNFAIL, 0);

							}
						}

					} else {

						if (cs2ss[s] != 0) { /* read data from client */
					       		DQ_T		*dq = get_free_dq();

							dq->len = read(s, dq->data, 1500);

							if (dq->len == 0) {

			       					insert_qt(&dq_hdr, dq);

								for(dq = remove_qh(&sshdr[cs2ss[s] * 4]); dq != NULL;
								    insert_qt(&dq_hdr, dq), dq = remove_qh(&sshdr[cs2ss[s] * 4])) {

				                			stat = _SSL_write(ss2ssl[cs2ss[s]], dq->data, dq->len);

								}

								_SSL_shutdown(ss2ssl[cs2ss[s]]);
								_SSL_free(ss2ssl[cs2ss[s]]);

								for(dq = remove_qh(&cshdr[s * 4]); dq != NULL;
								    insert_qt(&dq_hdr, dq), dq = remove_qh(&cshdr[s * 4])) {

			                				stat = write(s, dq->data, dq->len);

								}

								FD_CLR(s, &rs);
								FD_CLR(s, &ws);
								FD_CLR(cs2ss[s], &rs);
								FD_CLR(cs2ss[s], &ws);

								close(s);
								close(cs2ss[s]);
 								ss2cs[cs2ss[s]] = 0;
								ss2ssl[cs2ss[s]] = NULL;
 								cs2ss[s] = 0;
 
							} else {

								if (dq->len != -1 && ss2ssl[cs2ss[s]] != NULL) {

									insert_qt(&sshdr[cs2ss[s] * 4], dq);
                							FD_SET(cs2ss[s], &ws);

 								} else {

									insert_qt(&dq_hdr, dq);

								}
							}
						}

						if (ss2ssl[s] != NULL) { /* read data from SSL */
							DQ_T		*dq = get_free_dq();

	                				dq->len = _SSL_read(ss2ssl[s], dq->data, 1500);

							if (dq->len == 0) {

								insert_qt(&dq_hdr, dq);

								if (_SSL_get_error(ss2ssl[s], dq->len) == SSL_ERROR_ZERO_RETURN) {

									for(dq = remove_qh(&sshdr[s * 4]); dq != NULL;
									    insert_qt(&dq_hdr, dq), dq = remove_qh(&sshdr[s * 4])) {

				                			stat = _SSL_write(ss2ssl[s], dq->data, dq->len);

									}

									_SSL_shutdown(ss2ssl[s]);
									_SSL_free(ss2ssl[s]);

									for(dq = remove_qh(&cshdr[ss2cs[s] * 4]); dq != NULL;
									    insert_qt(&dq_hdr, dq), dq = remove_qh(&cshdr[ss2cs[s] * 4])) {

			                					stat = write(ss2cs[s], dq->data, dq->len);

									}

									FD_CLR(s, &rs);
									FD_CLR(s, &ws);
									FD_CLR(ss2cs[s], &rs);
									FD_CLR(ss2cs[s], &ws);

									close(s);
									close(ss2cs[s]);
									ss2ssl[s] = NULL;
 									cs2ss[ss2cs[s]] = 0;
 									ss2cs[s] = 0;

								}

							} else {

								if (dq->len > 0) {

									insert_qt(&cshdr[ss2cs[s] * 4], dq);
                							FD_SET(ss2cs[s], &ws);

								} else { /* SSL want write instead */

									insert_qt(&dq_hdr, dq);

								}
 							}
						}
					}
				}
			}
		}

		_SSL_CTX_free(ctx);
	}

	pthread_detach(pthread_self());

	return NULL;
}


/*
*/

void SHA1_hash(char *data, char unsigned *md) {
	SHA_CTX			ctx;

	_SHA1_Init(&ctx);
	_SHA1_Update(&ctx, data, strlen(data));
	_SHA1_Final(md, &ctx);
}


/*
	Description:

	This routine initialize an array of random data by creating SHA hash
	from the system time. Ok, not the best random source but at least
	better than nothing.
	This routine is used when password synchronization is enabled and
	a data array is needed with random data used for scramble passwords.

	If SSL is not installed and no SSL is set up to be ok we still
	terminate here because it is not acceptable to store passwords in
	memory without scramble them.

	INPUT:		data array
			length of data array

	OUTPUT:		data array updated with random data
*/

void generate_random(char unsigned *rnddata, int rdnlen) {
	int			stat = SS$_NORMAL;
	int			offset = 0;
	SHA_CTX			ctx;
	int unsigned		curtime[2];
	void			*hndl = NULL;

	if (_SHA1_Init == NULL) { /* No SSL libray loaded */

		if ((hndl = dlopen("SSL$LIBCRYPTO_SHR32", 0)) != NULL) {

			if ((_SHA1_Init = dlsym(hndl, "SHA1_INIT")) == NULL) { stat = SS$_NOSUCHFILE; }
			if ((_SHA1_Update = dlsym(hndl, "SHA1_UPDATE")) == NULL) { stat = SS$_NOSUCHFILE; }
			if ((_SHA1_Final = dlsym(hndl, "SHA1_FINAL")) == NULL) { stat = SS$_NOSUCHFILE; }

			dlclose(hndl);

		}

		if (!$VMS_STATUS_SUCCESS(stat)) { lib$stop(stat); }	
	}

	sys$gettim(&curtime);

	_SHA1_Init(&ctx);

	for (; offset  < rdnlen - SHA_DIGEST_LENGTH; offset += SHA_DIGEST_LENGTH) {

		_SHA1_Update(&ctx, &curtime, sizeof(curtime)); /* the worlds worse random data but better than nothing... */
		_SHA1_Final(&rnddata[offset], &ctx);

	}
	
	_SHA1_Update(&ctx, &curtime, sizeof(curtime));
	_SHA1_Final(&rnddata[rdnlen - SHA_DIGEST_LENGTH], &ctx);
}

/*
	Description:

	This routine lock and unlock OpenSSL locks

	INPUT:		mode, if lock or unlock should be done
			lock id
			file name (for debug)
			line number (for debug)

	OUTPUT:		none
*/

void ssl_lock(int mode, int n, const char *file, int line) {

	if (mode & CRYPTO_LOCK) {

		pthread_mutex_lock(&ssl_lcks[n]);

	} else {

		pthread_mutex_unlock(&ssl_lcks[n]);

	}
}


/*
	Description:

	This routine return the thread id. This is used by OpenSSL.

	INPUT:		none

	OUTPUT:		thread ID
*/

long unsigned get_thread_id() {

	return (long unsigned)pthread_getsequence_np(pthread_self());

}


/*
	Description:

	This routine create locks used by the OpenSSL library.

	INPUT:		none

	OUTPUT:		none
*/

void setup_SSLLock() {
	int			cnt;
	int			reqlcks = _CRYPTO_num_locks();

	ssl_lcks = (pthread_mutex_t *)safe_malloc(sizeof(pthread_mutex_t) * reqlcks);

	for(cnt = 0; cnt < reqlcks; cnt++) {

		pthread_mutex_init(&ssl_lcks[cnt], NULL);

	}

	_CRYPTO_set_id_callback(&get_thread_id);
	_CRYPTO_set_locking_callback(&ssl_lock);
}


/*
	Description:

	This rutine load required OpenSSL library API functions used by this module.
	By not linking the OpenSSL libraries into the image this program can run
	without any SSL functions if no SSL libraries exists on the host.

	If all functions can be loaded the ssl thread is started that handle SSL tunnel
	to the LDAP server.

	INPUT:		none

	OUTPUT:		none
*/

int ssl_load() {
	void			*hndl = NULL;
	
	if ((hndl = dlopen("SSL$LIBCRYPTO_SHR32", 0)) != NULL) {

		if ((_CRYPTO_num_locks = dlsym(hndl, "CRYPTO_NUM_LOCKS")) == NULL) { return SS$_NOSUCHFILE; }
		if ((_CRYPTO_set_id_callback = dlsym(hndl, "CRYPTO_SET_ID_CALLBACK")) == NULL) { return SS$_NOSUCHFILE; }
		if ((_CRYPTO_set_locking_callback = dlsym(hndl, "CRYPTO_SET_LOCKING_CALLBACK")) == NULL) { return SS$_NOSUCHFILE; }

		if ((_SHA1_Init = dlsym(hndl, "SHA1_INIT")) == NULL) { return SS$_NOSUCHFILE; }
		if ((_SHA1_Update = dlsym(hndl, "SHA1_UPDATE")) == NULL) { return SS$_NOSUCHFILE; }
		if ((_SHA1_Final = dlsym(hndl, "SHA1_FINAL")) == NULL) { return SS$_NOSUCHFILE; }

		if ((_RAND_add = dlsym(hndl, "RAND_ADD")) == NULL) { return SS$_NOSUCHFILE; }

		dlclose(hndl);

		if ((hndl = dlopen("SSL$LIBSSL_SHR32", 0)) != NULL) {
			pthread_cond_t		cndw;
			pthread_mutex_t		mtxw;

			if ((_SSL_library_init = dlsym(hndl, "SSL_LIBRARY_INIT")) == NULL) { return SS$_NOSUCHFILE; }
			if ((_SSL_load_error_strings = dlsym(hndl, "SSL_LOAD_ERROR_STRINGS")) == NULL) { return SS$_NOSUCHFILE; }
			if ((_SSLv3_method = dlsym(hndl, "SSLV3_METHOD")) == NULL) { return SS$_NOSUCHFILE; }
			if ((_SSL_CTX_new =  dlsym(hndl, "SSL_CTX_NEW")) == NULL) { return SS$_NOSUCHFILE; }
			if ((_SSL_new = dlsym(hndl, "SSL_NEW")) == NULL) { return SS$_NOSUCHFILE; }
			if ((_SSL_free = dlsym(hndl, "SSL_FREE")) == NULL) { return SS$_NOSUCHFILE; }
			if ((_SSL_set_fd = dlsym(hndl, "SSL_SET_FD")) == NULL) { return SS$_NOSUCHFILE; }
			if ((_SSL_connect = dlsym(hndl, "SSL_CONNECT")) == NULL) { return SS$_NOSUCHFILE; }
			if ((_SSL_pending = dlsym(hndl, "SSL_PENDING")) == NULL) { return SS$_NOSUCHFILE; }
		        if ((_SSL_shutdown = dlsym(hndl, "SSL_SHUTDOWN")) == NULL) { return SS$_NOSUCHFILE; }
			if ((_SSL_CTX_free = dlsym(hndl, "SSL_CTX_FREE")) == NULL) { return SS$_NOSUCHFILE; }
			if ((_SSL_write = dlsym(hndl, "SSL_WRITE")) == NULL) { return SS$_NOSUCHFILE; }
                        if ((_SSL_read = dlsym(hndl, "SSL_READ")) == NULL) { return SS$_NOSUCHFILE; }
                        if ((_SSL_get_error = dlsym(hndl, "SSL_GET_ERROR")) == NULL) { return SS$_NOSUCHFILE; }

			dlclose(hndl);

			_SSL_library_init();
			_SSL_load_error_strings();

			setup_SSLLock();

			lib$signal(LLSRV_SSLLOADED, 0);

			pthread_cond_init(&cndw, NULL);
			pthread_mutex_init(&mtxw, NULL);

			pthread_create(&thrssl, NULL, &ssl_thr, &cndw);

			pthread_cond_wait(&cndw, &mtxw);

			pthread_cond_destroy(&cndw);
			pthread_mutex_destroy(&mtxw);

			use_ssl = 1;

			return SS$_NORMAL;
		}
	}

	return SS$_NOSUCHFILE;
}


/*
	Description:

	This routine try to inilialize the SSL.
	If logical name LGI_LDAP_USE_NO_SSL is defined to SSL is loaded or used.

	If the SSL load fail, i.e. OpenSSL not installed on the host, the profram
	terminates. If the logical name LGI_LDAP_NO_SSL_OK is defined the program
	will continue.

	WARNING. Use of this program without SSL is a major security risk as all
		 data transferred between the host and the LDAP server is in clear
		 text, including passwords! YOU HAVE BEEN WARNED!

	INPUT:		none

	OUTPUT:		SS$_NORMAL if everything was ok
			program aborted if something fails
*/

int ssl_init() {
	char		dummy[256];

	if ($VMS_STATUS_SUCCESS(do_trnlnm("LGI_LDAP_USE_NO_SSL", dummy, sizeof(dummy) - 1, PSL$C_EXEC))) {

		lib$signal(LLSRV_NOSSL, 0, LLSRV_PWDCLEAR, 0);
		return SS$_NORMAL;

	}

	if (!$VMS_STATUS_SUCCESS(ssl_load())) {

		if (!$VMS_STATUS_SUCCESS(do_trnlnm("LGI_LDAP_NO_SSL_OK", dummy, sizeof(dummy) - 1, PSL$C_EXEC))) {

			lib$stop(LLSRV_SSLOADERR, 0);

		}

		lib$signal(LLSRV_NOSSLCONT, 0, LLSRV_PWDCLEAR, 0);
	}

	return SS$_NORMAL;
}
