/*
	Copyright (C) 2004 Jonas Lindholm

	This software was developed by Jonas Lindholm, jlhm@usa.net

	History

	V1.0		Jonas Lindholm	2004-05-14

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
	ldap.h include all standard headers such as string.h
*/
#include <string.h>
#include <ldap.h>

#include <starlet.h>
#include <ssdef.h>
#include <stsdef.h>
#include <psldef.h>
#include <prcdef.h>
#include <lnmdef.h>
#include <jpidef.h>
#include <uaidef.h>
#include <syidef.h>
#include <descrip.h>

#include <lib$routines.h>

#include <rms.h>
#include <fab.h>
#include <rab.h>
#include <fabdef.h>
#include <rabdef.h>
#include <prcdef.h>

#include <dvidef.h>
#include <cmbdef.h>
#include <iodef.h>
#include <nsadef.h>
#include <uaf070def.h>


#include <pthread.h>

#include "lgi.h"
#include "lgildap.h"

#include "server.h"


#define	UAF	uaf070


typedef struct uafact_s {
	int unsigned		qhdr[4];
	int			stat;
	char			username[32 + 1];
	char			password[32 + 1];
	LGI_SESS_T		*sess;
} UAFACT_T;                                                                    


typedef struct prclst_s {
	int unsigned		pid;		/* PID of process */
	int unsigned		starttime[2];	/* start time of process */
	int			pwdidx;		/* current index of passwords */
	char			*pwds[3];	/* pointers to list of password */
} PRCLST_T;


typedef struct uafldap_s {
	struct {
		short		minver;
		short		majver;
	} version;

	int unsigned		magic;   /* magic value for consistency check */
	char			dn[128]; /* LDAP DN used when creating the user */
	char			uid[32]; /* UID used in LDAP for this user */
} UAFLDAP_T;

#define			UAFLDAP_MIN	0
#define			UAFLDAP_MAJ	1
#define			UAFLDAP_MAGIC	0x4c444150  /* LDAP as HEX string */

int unsigned		addqh[4] = { 0, 0, 0, 0 };
int unsigned		verqh[4] = { 0, 0, 0, 0 };
int unsigned		pwdqh[4] = { 0, 0, 0, 0 };
int unsigned		pwdupdqh[4] = { 0, 0, 0, 0 };

PRCLST_T		*prclst;

int			add_user_ok = 0;

pthread_t		thrmod;
pthread_t		thradd;
pthread_t		thrrem;
pthread_t		thrpwdupd;

pthread_cond_t		cndmodw;
pthread_mutex_t		mtxmodw;
pthread_cond_t		cndaddw;
pthread_mutex_t		mtxaddw;
pthread_rwlock_t	rwluai;

char unsigned		rnddata[1024]; /* Random generated data used to scramble password */

/*
	Allowed fids and devices for images that can trigger update of
	user passwords on the LDAP server.
*/

short unsigned		allowed_fid[2][3] = { { 0, 0, 0 }, { 0, 0, 0 } };
char			allowed_dvi[2][65] = { { "" }, { "" } };


/*
	Description:

	This routine create an audit and alarm event.
	It is invoked when a user is added, deleted or modified by this program.
	The DN is logged together with the system service name, LGI LDAP Server.

	This make it possible to keep track of all actions done with this process
	in the SYSUAF.DAT file using RMS routines.

	INPUT:		event subtype
			UAF record
			current DN
			new DN

	OUTPUT:		status of sys$audit
			SS$_ABORT of subtype is invalid
*/

int do_audit(int subtype, char *username, char *curdn, char*newdn) {
	int			stat = SS$_NORMAL;
	int			evtype = NSA$C_MSG_SYSUAF;
	char			auditname[] = "SECURITY";
	char			ssn[] = "LGI LDAP Server";
	char			fn[] = "Distinguished Name";
	ITEM3			itmlst[] = { { sizeof(evtype), NSA$_EVENT_TYPE, &evtype, 0 },
					     { sizeof(subtype), NSA$_EVENT_SUBTYPE, &subtype, 0 },
					     { strlen(auditname), NSA$_AUDIT_NAME, &auditname, 0 },
					     { strlen(auditname), NSA$_ALARM_NAME, &auditname, 0 },
					     { strlen(ssn), NSA$_SYSTEM_SERVICE_NAME, &ssn, 0 },
					     { 4, NSA$_CHAIN, 0, 0 } };

	switch (subtype) {
		case NSA$C_SYSUAF_ADD: {
			ITEM3		itmlst2[] = { { strlen(username), NSA$_UAF_ADD, username, 0 },
						      { strlen(fn), NSA$_FIELD_NAME, fn, 0 },
						      { strlen(newdn), NSA$_NEW_DATA, newdn, 0 },
						      { 0, 0, 0, 0 } };
			itmlst[5].bufadr = &itmlst2;
			stat = sys$audit_eventw(EFN_AUDIT, 0, &itmlst, 0, 0, 0);

			break;
		}

		case NSA$C_SYSUAF_DELETE: {
			ITEM3		itmlst2[] = { { strlen(username), NSA$_UAF_ADD, username, 0 },
						      { strlen(fn), NSA$_FIELD_NAME, fn, 0 },
						      { strlen(curdn), NSA$_ORIGINAL_DATA, curdn, 0 },
						      { 0, 0, 0, 0 } };
			itmlst[5].bufadr = &itmlst2;
			stat = sys$audit_eventw(EFN_AUDIT, 0, &itmlst, 0, 0, 0);

		      	break;
		}

		case NSA$C_SYSUAF_MODIFY: {
			ITEM3		itmlst2[] = { { strlen(username), NSA$_UAF_MODIFY, username, 0 },
						      { strlen(fn), NSA$_FIELD_NAME, fn, 0 },
						      { strlen(curdn), NSA$_ORIGINAL_DATA, curdn, 0 },
						      { strlen(newdn), NSA$_NEW_DATA, newdn, 0 },
						      { 0, 0, 0, 0 } };
			itmlst[5].bufadr = &itmlst2;
			stat = sys$audit_eventw(EFN_AUDIT, 0, &itmlst, 0, 0, 0);

			break;
		}
		default: {
			return SS$_ABORT;
		}
	}

	return stat;
}


/*
	Description:

	This routine is used to open and connect FAB and RAB structures to SYSUAF.DAT.
	It use only executive defined logical names for SYSUAF.DAT.

	INPUT:		FAB
			RAB

	OUTPUT:		RMS$_NORMAL if SYSUAF.DAT open and RAB connected
			any RMS error
*/

int open_sysuaf(struct FAB *fab , struct RAB *rab) {
	int		stat;
	char		sysuafdef[] = "SYS$SYSTEM:.DAT";
	char	       	sysuaf[] = "SYSUAF";

	fab->fab$v_lnm_mode = PSL$C_EXEC;
	fab->fab$l_dna = (char *)&sysuafdef;
	fab->fab$b_dns = strlen(sysuafdef);
	fab->fab$l_fna = (char *)&sysuaf;
	fab->fab$b_fns = strlen(sysuaf);
        fab->fab$b_fac = FAB$M_DEL | FAB$M_GET | FAB$M_PUT | FAB$M_UPD;
        fab->fab$b_shr = FAB$M_SHRDEL | FAB$M_SHRGET | FAB$M_SHRPUT | FAB$M_SHRUPD;

	if ($VMS_STATUS_SUCCESS(stat = sys$open(fab, 0, 0))) {

		rab->rab$l_fab = fab;

		if (!$VMS_STATUS_SUCCESS(stat = sys$connect(rab , 0, 0))) {

			sys$close(fab, 0, 0);

		}
	}

	return stat;
}


/*
	Description:

	This routine disconnect and close SYSUAF.DAT

	INPUT:		FAB
			RAB

	OUTPUT:		none
*/

void close_sysuaf(struct FAB *fab, struct RAB *rab) {

	sys$disconnect(rab , 0, 0);
	sys$close(fab, 0, 0);

}


/*
	Description:

	This routine fetch the owner (cn), default device (homeDirectory), default directory (homeDirectory)
	and UIC (uid & gid).

	INPUT:		pointer to ldap structure
			DN
			uid to fetch
			owner, must be 32 bytes or longer
			defdev, must be 32 bytes or longer
			defdir, must be 64 bytes or longer
			UIC
			GID

	OUTPUT:		SS$_NORMAL, updated owner, defdev, defdir, UIC AND GID if everything is ok
			ldap set to NULL iF any LDAP call fail
			LLSRV_LDAPUSERMISS if DN/uid is missing
*/

get_ldap_user_data(LDAP **ldap, char *dn, char *uid, char *owner, char *defdev, char *defdir, short unsigned uic[], short unsigned *gid) {
	int			stat;
	int			retstat = LLSRV_LDAPUSERMISS;
	int			msgid;
	char			search[255 + strlen(uid) * 2 + 1];
	char			*attrs[] = { "cn", "uidNumber", "gidNumber", "homeDirectory", NULL };
	LDAPMessage		*res;

	if (*ldap == NULL) { return retstat; }

	LDAP_LOCK;

	get_filter(search, sizeof(search) - 1, uid);

	if ((msgid = ldap_search(*ldap, dn, LDAP_SCOPE_BASE, search, attrs, 0)) > 0) {
		struct timeval          to = { 0, 0 };

		while ((stat = ldap_result(*ldap, msgid, 1, &to, &res)) <= 0) {
			struct timespec         wait = { 0, 10000 };

			LDAP_UNLOCK;

			pthread_delay_np(&wait);

			LDAP_LOCK;

		}

		if (stat > 0) {
			LDAPMessage             *e = NULL;

			if ((e = ldap_first_entry(*ldap, res)) != NULL) {
				int		okcnt = 0; /* must reach 4 to be ok, all required attributes ok */
				char 		**attr;

				if ((attr = ldap_get_values(*ldap, e, "cn")) != NULL) {

					strncpy(owner, attr[0], 32);
					ldap_value_free(attr);
					okcnt++;

				}

				if ((attr = ldap_get_values(*ldap, e, "uidNumber")) != NULL) {
					char		offset[7] = "010000";
					int		offs = 010000;

					do_trnlnm("LGI_LDAP_ADD_USER_UID_OFFSET", offset, sizeof(offset) - 1, PSL$C_EXEC);
					stat = sscanf(offset, "%i", &offs);
					if (stat == 0) { lib$signal(LLSRV_INVVALUE, 1, "LGI_LDAP_ADD_USER_UID_OFFSET"); }
					uic[0] = atoi(attr[0]) + offs;
					ldap_value_free(attr);
					okcnt++;

				}

				if ((attr = ldap_get_values(*ldap, e, "gidNumber")) != NULL) {
					char		offset[7] = "01000";
					int		offs = 01000;

					do_trnlnm("LGI_LDAP_ADD_USER_GID_OFFSET", offset, sizeof(offset) - 1, PSL$C_EXEC);
					stat = sscanf(offset, "%i", &offs);
					if (stat == 0) { lib$signal(LLSRV_INVVALUE, 1, "LGI_LDAP_ADD_USER_GID_OFFSET"); }
					uic[1] = atoi(attr[0]) + offs;
					*gid = atoi(attr[0]);
					ldap_value_free(attr);
					okcnt++;

				}

				if ((attr = ldap_get_values(*ldap, e, "homeDirectory")) != NULL) {

					translate_unix2vms(attr[0], defdev, 31, defdir, 63);
					do_trnlnm(defdev, defdev, 31, PSL$C_EXEC);
					ldap_value_free(attr);
					okcnt++;

				}

				if (okcnt == 4) { retstat = SS$_NORMAL; } /* only return if all attributes where fetched */

			} else {

				ldap_unbind(*ldap);
				*ldap = NULL;

			}

			ldap_msgfree(res);

		} else {

			ldap_unbind(*ldap);
			*ldap = NULL;

		}
	}

	LDAP_UNLOCK;

	return retstat;
}


/*
	Description:

	This routine is used to verify the existence of a user in SYSUAF.DAT.
	If the user exists and has a uafldap structure in the user data area
	a check is done to see if the used DN for the session is the same DN
	as was used to create the user.

	If the DN has changed the routine will fetch the cn, uidNumber, gidNumber
	and homeDirectory and update the user. An audit alarm is also generated.

	INPUT:		session handler
			identity (username)

	OUTPUT:		SS$_NORMAL if the user exists in SYSUAF.DAT
			error when user doesn't exist or same other error.
*/

int uaf_verify(LGI_SESS_T *sess, char *identity) {
	int			stat;
	short			udlen;
	short unsigned		curuic[2];
	char			curdefdev[33];
	char			curdefdir[65];
	IOSB			iosb;
	UAFLDAP_T		uafldap;
	ITEM3			itmlstg[] = { {sizeof(uafldap), UAI$_USER_DATA, &uafldap, &udlen },
					      { sizeof(curuic), UAI$_UIC, &curuic, 0 },
					      { sizeof(curdefdev) - 1, UAI$_DEFDEV, &curdefdev, 0 },
					      { sizeof(curdefdir) - 1, UAI$_DEFDIR, &curdefdir, 0 },
					      { 0, 0, 0, 0 } };

	$DESCRIPTOR(usrnamdsc, identity);

	usrnamdsc.dsc$w_length = strlen(identity);

	lock_cluster_cr(LCK_SYSUAF);

	stat = sys$getuai(0, 0, &usrnamdsc, &itmlstg, 0, 0, 0);

	if ($VMS_STATUS_SUCCESS(stat) && (udlen >= sizeof(uafldap)) && (uafldap.magic == UAFLDAP_MAGIC) && strlen(uafldap.dn) > 0) {

		if (strcmp(sess->dn, uafldap.dn) != 0) { /* DN from LDAP server is not the same as in SYSUAF.DAT */
			char			newdefdev[33] = ".USERS"; /* save first byte for length */
			char			newdefdir[65] = ".USER";
			char			newowner[33] = ".Unknown LDAP user";
			short unsigned		newuic[2] = { 01000, 01000 }; /* default UIC */
			short unsigned		gid;
			UAF			uafrec;

			stat = get_ldap_user_data(&sess->ldap, sess->dn, identity, newowner + 1, newdefdev + 1, newdefdir + 1, newuic, &gid);

			if ($VMS_STATUS_SUCCESS(stat)) {
				char		actrtn[256];
				ITEM3		itmlsts[] = { { sizeof(newdefdev) - 1, UAI$_DEFDEV, &newdefdev, 0 },
							      { sizeof(newdefdir) - 1, UAI$_DEFDIR, &newdefdir, 0 },
							      { sizeof(newowner) - 1,  UAI$_OWNER,  &newowner,  0 },
							      { sizeof(newuic),     UAI$_UIC,       &newuic, 0 },
							      { sizeof(uafldap),    UAI$_USER_DATA, &uafldap, 0 },
							      { 0, 0, 0, 0 } };

				newdefdev[0] = strlen(newdefdev + 1);
				newdefdir[0] = strlen(newdefdir + 1);
				newowner[0] = strlen(newowner + 1);

				if ($VMS_STATUS_SUCCESS(do_trnlnm("LGI_LDAP_MOD_USER_ACTION", actrtn, sizeof(actrtn) - 1, PSL$C_EXEC))) {
					char			curuicstr[32];
					char			newuicstr[32];

					sprintf(curuicstr, "[%o,%o]", curuic[1], curuic[0]);
					sprintf(newuicstr, "[%o,%o]", newuic[1], newuic[0]);

					if (!$VMS_STATUS_SUCCESS(stat = do_action(actrtn, 8, uafldap.dn, identity, curdefdev + 1, curdefdir + 1,
										  curuicstr, sess->dn, newdefdev + 1, newdefdir + 1, newuicstr))) {

						lib$signal(LLSRV_MODUSERCMDERR, 2, actrtn, identity);

					}
				}

				do_audit(NSA$C_SYSUAF_MODIFY, identity, uafldap.dn, sess->dn);

				strncpy(uafldap.dn, sess->dn, sizeof(uafldap.dn));
				strncpy(uafldap.uid, identity, sizeof(uafldap.uid));

				stat = sys$setuai(0, 0, &usrnamdsc, &itmlsts, 0, 0, 0);

				if (curuic[0] != newuic[0] || curuic[1] != newuic[1]) { /* modify identifier value if new UIC */

					sys$mod_ident((curuic[0] | (curuic[1] << 16)), 0, 0, 0, (newuic[0] | (newuic[1] << 16)));

				}
			}
		}
	}

	unlock_cluster(LCK_SYSUAF);

	return stat;
}


/*
	Description:

	This routine is used to update the local password hash in SYSUAF.DAT
	for the user.
	The update is done if the password used was correct and the logical
	name LGI_LDAP_PWD_SYNCH is defined.
	The hashed password value is updated only if the hash value for the
	correct password is something else than the currently stored value.

	INPUT:		session handler
			password

	OUTPUT:		SS$_NORMAL is everything was ok
			error from sys$getuai, sys$setuai or sys$hash_password
*/

int uaf_pwd_synch(LGI_SESS_T *sess, char *password) {
	int			stat;
	int unsigned		salt;
	int unsigned		pwdhash[2];
	int unsigned		encalg;
	int unsigned		flags;
	char			username[sizeof(sess->identity)];
	char			*t = sess->identity;
	char			*d = username;
	IOSB			iosb;
	ITEM3			itmlst[] = { { sizeof(pwdhash), UAI$_PWD, &pwdhash, 0}, { sizeof(salt), UAI$_SALT, &salt, 0},
					     { sizeof(flags), UAI$_FLAGS, &flags, 0}, { sizeof(encalg), UAI$_ENCRYPT, &encalg, 0},
					     { 0, 0, 0, 0 } };
	$DESCRIPTOR(usrnamdsc, username);

	/* use username as dummy variable */

	if (!$VMS_STATUS_SUCCESS(do_trnlnm("LGI_LDAP_PWD_SYNCH", (char *)&username, sizeof(username) - 1, PSL$C_EXEC))) {

		return SS$_NORMAL;

	}

	for(; *t; *d = toupper(*t), t++, d++);

	usrnamdsc.dsc$w_length = strlen(sess->identity);

	lock_cluster_cr(LCK_SYSUAF);

	stat = sys$getuai(0, 0, &usrnamdsc, &itmlst, 0, 0, 0);

	if ($VMS_STATUS_SUCCESS(stat)) {
		int unsigned		pwdh[2];
		char			pwd[256];
		$DESCRIPTOR(pwddsc, pwd);

		strncpy(pwd, password, sizeof(pwd) - 1);

		if (osvers < PWDMIX_SUPPORT || !(flags & UAI$M_PWDMIX)) {

			for(d = pwd, t = password; *d = toupper(*t); t++, d++);

		}

		pwddsc.dsc$w_length = strlen(password);
		stat = sys$hash_password(&pwddsc, encalg, salt, &usrnamdsc, &pwdh);

		if ($VMS_STATUS_SUCCESS(stat)) {

			if (pwdh[0] != pwdhash[0] || pwdh[1] != pwdhash[1]) {

				pwdhash[0] = pwdh[0];
				pwdhash[1] = pwdh[1];

				memset(&itmlst[1], 0, sizeof(itmlst[1]));

				stat = sys$setuai(0, 0, &usrnamdsc, &itmlst, 0, 0, 0);

			}
		}
	}

	unlock_cluster(LCK_SYSUAF);

	return stat;
}


/*
	Description:

	This routine create a new user in SYSUAF.DAT using a default user.
	The user is added whenever someone tries to logon and the specified
	user is correct but does not exist in SYSUAF.DAT.
	Also, if a command procedure is defined to be run when a user is added
	the return value from that procedure will determine if the user is
	added or not.
	Also an identifier with the name of the username and UIC is created.

	INPUT:		sess handler
			pointer to ldap handler
			RAB
			username to add

	OUTPUT:		SS$_NORMAL is the user was created
			error if something went wrong.
*/

int uaf_add_user(LGI_SESS_T *sess, LDAP **ldap, struct RAB *rab, char *username) {
	int			stat = SS$_NORMAL;
	char			userkey[33] = "DEFAULT";
	char			defdev[33] = ".USERS"; /* save first byte for length */
	char			defdir[65] = ".USER";
	char			owner[33] = ".Unknown LDAP user";
	char			defusrlog[36];
	short unsigned		uic[2] = { 01000, 01000 }; /* default UIC */
	short unsigned		gid;
	UAF			uafrec;

	if (strlen(username) > 12) { /* username can not be longer than 12 characters */

		lib$signal(LLSRV_UIDTOOLONG, 1, username);
		return SS$_ABORT;

	}

	stat = get_ldap_user_data(ldap, sess->dn, username, owner + 1, defdev + 1, defdir + 1, uic, &gid);

	if (!$VMS_STATUS_SUCCESS(stat)) { return stat; }

	if (!strlen(defdir + 1)) { strcpy(defdir + 1, username); }; /* in case defdir is empty */

	defdev[0] = strlen(defdev + 1);
	defdir[0] = strlen(defdir + 1);
	owner[0] = strlen(owner + 1);

	sprintf(defusrlog, "LGI_LDAP_ADD_USER_DEF_%d", gid);

 	if (!$VMS_STATUS_SUCCESS(do_trnlnm(defusrlog, userkey, sizeof(userkey) - 1, PSL$C_EXEC))) {

	 	do_trnlnm("LGI_LDAP_ADD_USER_DEFAULT", userkey, sizeof(userkey) - 1, PSL$C_EXEC);

	}

	rab->rab$l_kbf = (char *)&userkey;
	rab->rab$b_ksz = strlen(userkey);
	rab->rab$b_krf = 0;
	rab->rab$b_rac = RAB$C_KEY;
	rab->rab$l_rop = RAB$M_NLK;
	rab->rab$l_ubf = (char *)&uafrec;
	rab->rab$w_usz = sizeof(uafrec);

	stat = sys$get(rab, 0, 0); /* get the default user */

	if ($VMS_STATUS_SUCCESS(stat)) {
		char		*d = uafrec.uaf070$t_username;
		char		*s = username;

		memset(uafrec.uaf070$t_username, ' ', sizeof(uafrec.uaf070$t_username));
		for(; *s; *d++ = toupper(*s), s++);

		uafrec.uaf070$w_mem = uic[0];
		uafrec.uaf070$w_grp = uic[1];

		for(d = defdev + 1; *d; *d++ = toupper(*d));
		for(d = defdir + 1; *d; *d++ = toupper(*d));

		lock_cluster_ex(LCK_SYSUAF);

		stat = sys$put(rab, 0, 0); /* insert the new user */

		if ($VMS_STATUS_SUCCESS(stat)) { /* we must now update the new user using sys$setuai */
			int unsigned	flags;
			int unsigned	pwddate[2] = { 0, 0 };
			UAFLDAP_T	uafldap;
			ITEM3		itmlstg[] = { { sizeof(flags),      UAI$_FLAGS,  &flags,  0 },
						      { 0, 0, 0, 0 } };
			ITEM3		itmlsts[] = { { sizeof(defdev) - 1, UAI$_DEFDEV, &defdev, 0 },
						      { sizeof(defdir) - 1, UAI$_DEFDIR, &defdir, 0 },
						      { sizeof(owner) - 1,  UAI$_OWNER,  &owner,  0 },
						      { sizeof(flags),      UAI$_FLAGS,  &flags,  0 },
						      { sizeof(pwddate),    UAI$_PWD_DATE, &pwddate, 0 },
						      { sizeof(uafldap),    UAI$_USER_DATA, &uafldap, 0 },
						      { 0, 0, 0, 0 } };
			$DESCRIPTOR(usrnamdsc, username);
			usrnamdsc.dsc$w_length = strlen(username);

			stat = sys$getuai(0, 0, &usrnamdsc, &itmlstg, 0, 0, 0); /* get current flags */

			if ($VMS_STATUS_SUCCESS(stat)) {
				char		actrtn[256];

				if ($VMS_STATUS_SUCCESS(do_trnlnm("LGI_LDAP_ADD_USER_ACTION", actrtn, sizeof(actrtn) - 1, PSL$C_EXEC))) {
					char		uicstr[32];

					sprintf(uicstr, "[%o,%o]", uic[1], uic[0]);

					if (!$VMS_STATUS_SUCCESS(stat = do_action(actrtn, 8, sess->dn, username, defdev + 1, defdir + 1,
										  uicstr))) {

						lib$signal(LLSRV_ADDUSERCMDERR, 2, actrtn, username);

					}
				}

				if ($VMS_STATUS_SUCCESS(stat)) {

					memset(&uafldap, 0, sizeof(uafldap));

					strncpy(uafldap.dn, sess->dn, sizeof(uafldap.dn));
					strncpy(uafldap.uid, username, sizeof(uafldap.uid));

					do_audit(NSA$C_SYSUAF_ADD, username, NULL, uafldap.dn);

					/*
						If the OpenVMS version is 7.3-2 or later and we password synch is defined
						we set the PWDMIX for the user allowing the user to use mixture of upper
						and lower case for passwords.
					*/

					if (osvers >= PWDMIX_SUPPORT &&
					    $VMS_STATUS_SUCCESS(do_trnlnm("LGI_LDAP_PWD_SYNCH", actrtn, sizeof(actrtn) - 1, PSL$C_EXEC))) {

						flags |= UAI$M_PWDMIX;

					}

					flags &= ~UAI$M_DISACNT;     /* remove disuser flag */
					flags &= ~UAI$M_PWD_EXPIRED; /* password not expired */

					uafldap.version.minver = UAFLDAP_MIN;
					uafldap.version.majver = UAFLDAP_MAJ;
				       	uafldap.magic = UAFLDAP_MAGIC;

					stat = sys$setuai(0, 0, &usrnamdsc, &itmlsts, 0, 0, 0);

					if ($VMS_STATUS_SUCCESS(stat)) {

						stat = sys$add_ident(&usrnamdsc, uafrec.uaf070$l_uic, 0, 0);

						unlock_cluster(LCK_SYSUAF);

						lib$signal(LLSRV_ADDUSER, 3, username, sess->dn, userkey, stat);

						return SS$_NORMAL;
					}
				}
                        }

			/*
				If we come here something went wrong. We must delete the user again.
			*/

			rab->rab$l_rop = 0;
			rab->rab$l_kbf = (char *)&uafrec.uaf070$t_username;
			rab->rab$b_ksz = strlen(username);

			sys$get(rab, 0, 0);    /* lock the new user so we can delete it */
			sys$delete(rab, 0, 0); /* remove the user again */

		}

		unlock_cluster(LCK_SYSUAF);
	}

	if (stat != RMS$_DUP) { lib$signal(LLSRV_ADDUSERERR, 1, username, stat); }

	return stat;
}


/*
	Description:

	This routine verifies that a user exists in LDAP.

	INPUT:		pointer to ldap handler
			DN
			uid

	OUTPUT:		LLSRV_LDAPUSEREXIST if the user exist in LDAP
			LLSRV_LDAPUSERMISS if the user is missing in LDAP
*/

int check_user_exists_in_ldap(LDAP **ldap, char *dn, char *uid) {
	int			stat;
	int			retstat = LLSRV_LDAPUSEREXIST;
	int			msgid;
	char			search[255 + strlen(uid) * 2 + 1];
	char			*attrs[] = { "uid", NULL };
	LDAPMessage		*res = NULL;

	if (*ldap == NULL) { return retstat; }

	LDAP_LOCK;

	get_filter(search, sizeof(search) - 1, uid);

	if ((msgid = ldap_search(*ldap, dn, LDAP_SCOPE_BASE, search, attrs, 0)) > 0) {
		struct timeval          to = { 0, 0 };

		while ((stat = ldap_result(*ldap, msgid, 1, &to, &res)) == 0) {
			struct timespec         wait = { 0, 10000 };

			LDAP_UNLOCK;

			pthread_delay_np(&wait);

			LDAP_LOCK;

		}

		if (stat > 0) {
			LDAPMessage             *e = (void *)1; /* in case ldap_first_entry goes wrong... */

			if ((e = ldap_first_entry(*ldap, res)) == NULL) {

				retstat = LLSRV_LDAPUSERMISS; /* LDAP communication was ok but the DN is missing */

			}

			ldap_msgfree(res);

			LDAP_UNLOCK;

			return retstat;
		}
	}

	ldap_unbind(*ldap);	
	*ldap = NULL;

	LDAP_UNLOCK;

	return retstat;
}


typedef struct remw_s {
	pthread_cond_t	*cndtw;
	int		flag;
} REMW_T;


/*
	Description:

	This AST routine is invoked when the remove user thread should run.

	INPUT:		structure used to wakeup the thread

	OUTPUT:		none
*/

void wake_up_rem(REMW_T *remw) {

	remw->flag = 1;
	pthread_cond_signal_int_np(remw->cndtw);

}


/*
	Description:

	This thread is responsible to verify all users in SYSUAF.DAT that
	was created by this program.
	If the DN for each of these users is missing in LDAP the user is	
	removed.
	A command procedure can be defined to run for every users that is 
	to be removed.

	INPUT:		none

	OUTPUT:		none
*/

void *rem_thr(void *s) {
	int			stat;
	LDAP			*ldap = NULL;
	struct FAB		fab = cc$rms_fab;
	struct RAB		rab = cc$rms_rab;
	pthread_cond_t		cndtw = PTHREAD_COND_INITIALIZER;
	pthread_mutex_t		mtxtw = PTHREAD_MUTEX_INITIALIZER;
	REMW_T			remw;

	lib$signal(LLSRV_THRSTARTED, 1, "THREAD_REM_SYSUAF");

	pthread_cond_init(&cndtw, NULL);
	pthread_mutex_init(&mtxtw, NULL);

	remw.cndtw = &cndtw;

	while (1) {
		int unsigned		abstime[2];
		char			runtime[256] = "TOMORROW + 00:15:00";

		do_trnlnm("LGI_LDAP_REM_USER_RUN", runtime, sizeof(runtime) - 1, PSL$C_EXEC);

		stat = parse_run_time(runtime, abstime);

		if (!$VMS_STATUS_SUCCESS(stat)) {

			lib$signal(LLSRV_REMTERM);

			pthread_detach(pthread_self());
			pthread_exit(NULL);

		}

		remw.flag = 0;

		stat = sys$setimr(EFN_REMTHR, &abstime, &wake_up_rem, &remw, 0);

		while (remw.flag == 0) { pthread_cond_wait(&cndtw, &mtxtw); }

		if (ldap == NULL) {

			if ((ldap = open_ldap()) != NULL) {

				lib$signal(LLSRV_LDAPCONN, 0);

			}
		}

		if (!$VMS_STATUS_SUCCESS(stat = open_sysuaf(&fab, &rab))) {

			lib$signal(LLSRV_ERRACCSYSUAF, 1, "THREAD_REM_SYSUAF", stat);
			pthread_detach(pthread_self());
			pthread_exit(NULL);

		}

		lib$signal(LLSRV_REMUSRRUN, 0);

		do {
			UAFACT_T		*uae;
			UAF			uafrec;

			rab.rab$l_rop = RAB$M_NLK; /* do not lock the record */
			rab.rab$l_ubf = (char *)&uafrec;
			rab.rab$w_usz = sizeof(uafrec);

			stat = sys$get(&rab, 0, 0); /* get the user */

			if (stat == RMS$_NORMAL) {
				int		stat;
				int		uafldaplen;
				short unsigned	uic[2];
				char		defdev[33];
				char		defdir[65];
				UAFLDAP_T	uafldap;
				ITEM3		itmlst[] = { { sizeof(uafldap), UAI$_USER_DATA, &uafldap, &uafldaplen },
		      					     { sizeof(defdev) - 1, UAI$_DEFDEV, &defdev, 0 },
							     { sizeof(defdir) - 1, UAI$_DEFDIR, &defdir, 0 },
							     { sizeof(uic), UAI$_UIC, &uic, 0 },
							     { 0, 0, 0, 0 } };
				$DESCRIPTOR(usrnamdsc, uafrec.uaf070$t_username);

				stat = sys$getuai(0, 0, &usrnamdsc, &itmlst, 0, 0, 0);

				if ($VMS_STATUS_SUCCESS(stat) && uafldaplen > 0 && uafldap.magic == UAFLDAP_MAGIC && strlen(uafldap.dn) > 0) {

					stat = check_user_exists_in_ldap(&ldap, uafldap.dn, uafldap.uid);

					if (stat == LLSRV_LDAPUSERMISS) { /* user does not exist in LDAP anymore with the DN */

						rab.rab$l_rop = 0; /* lock the record this time */
						rab.rab$b_rac = RAB$C_KEY;
						rab.rab$l_kbf = (char *)&uafrec.uaf070$t_username;
						rab.rab$b_ksz = sizeof(uafrec.uaf070$t_username);

						stat = sys$get(&rab, 0, 0); /* get the user and lock it */

						if (stat == RMS$_NORMAL) {
							char		actrtn[256];
                                                        char		username[sizeof(uafrec.uaf070$t_username) + 1];

							if ($VMS_STATUS_SUCCESS(do_trnlnm("LGI_LDAP_REM_USER_ACTION", actrtn,
											  sizeof(actrtn) - 1, PSL$C_EXEC))) {

								char			uicstr[32];

								sprintf(uicstr, "[%o,%o]", uic[1], uic[0]);

								defdev[defdev[0] + 1] = 0;
								defdir[defdir[0] + 1] = 0;

			       	      				stat = do_action(actrtn, 30, uafldap.dn, uafrec.uaf070$t_username, defdev + 1,
										 defdir + 1, uicstr);

							}

							sys$rem_ident((uic[1] << 16) | uic[0]);

							if ($VMS_STATUS_SUCCESS(stat = sys$delete(&rab, 0, 0))) {

								strncpy(username, uafrec.uaf070$t_username, sizeof(uafrec.uaf070$t_username) - 1);
								username[sizeof(uafrec.uaf070$t_username)] = 0;

								do_audit(NSA$C_SYSUAF_DELETE, username, uafldap.dn, NULL);

							}
						}

						rab.rab$b_rac = RAB$C_SEQ;
					}

					stat = RMS$_NORMAL;
				}
			}

		} while (stat == RMS$_NORMAL);

		lib$signal(LLSRV_REMUSRDONE, 0);

		close_sysuaf(&fab, &rab);
	}

	return NULL;
}


/*
	Description:

	This thread is responsible to add users to SYSUAF.DAT.
	
	INPUT:		none

	OUTPUT:		none
*/

void *add_thr(void *s) {
	int		stat;
	char		ldaphost[256];
	char		basedn[256];
	LDAP		*ldap = NULL;
	struct FAB	fab = cc$rms_fab;
	struct RAB	rab = cc$rms_rab;

	lib$signal(LLSRV_THRSTARTED, 1, "THREAD_ADD_SYSUAF");

	if (!$VMS_STATUS_SUCCESS(stat = open_sysuaf(&fab, &rab))) {

		add_user_ok = 0;

		lib$signal(LLSRV_ERRACCSYSUAF, 1, "THREAD_ADD_SYSUAF", stat);

		pthread_detach(pthread_self());
		pthread_exit(NULL);

	}

	while (1) {
		UAFACT_T		*uae;

		pthread_cond_wait(&cndaddw, &mtxaddw);

		if (ldap == NULL) {

			if ((ldap = open_ldap()) != NULL) {

				lib$signal(LLSRV_LDAPCONN, 0);

			}
		}

		while ((uae = remove_qh(&addqh)) != NULL) {

			uae->stat = uaf_add_user(uae->sess, &ldap, &rab, uae->username);

			uae->sess->flags.condok = 1;
			sys$dclast(&cond_wake_ast, &uae->sess->cndw, 0);

		}

	}

	return NULL;
}


/*
	Description:

	This routine scramble the password sent from SYS$HASH_PASSWORD system routine
	and store it for later use when a user change the password.
	The password is only sent when the running image is SYS$SYSTEM:LOGINOUT.EXE or
	SYS$SYSTEM:SETP0.EXE.

	Input:		session handler
			User Action Entry

	Output:		SS$_NORMAL
*/		

int save_pwd(LGI_SESS_T *sess, UAFACT_T *uae) {
	int			stat;
	int			prcidx;
	int unsigned		starttime[2];
	int unsigned		iosb[2];
	ITEM3			itmlst[] = { { sizeof(prcidx), JPI$_PROC_INDEX, &prcidx, 0 },
					     { sizeof(starttime), JPI$_LOGINTIM, &starttime, 0 }, { 0, 0, 0, 0 } };

	if (prclst != NULL && $VMS_STATUS_SUCCESS(stat = sys$getjpiw(EFN_SAVPWD, &sess->pid, 0, &itmlst, &iosb, 0, 0)) &&
	    $VMS_STATUS_SUCCESS(iosb[0])) {

		PRCLST_T	*pi = &prclst[prcidx];

		if (pi != NULL) {

			if (pi->pid == 0 || pi->pid != sess->pid ||
			    (pi->pid != 0 && (pi->starttime[0] != starttime[0] || pi->starttime[1] != starttime[1]))) { /* new PID */

				int		cnt = 0;

				pi->pid = sess->pid;
				pi->starttime[0] = starttime[0];
				pi->starttime[1] = starttime[1];
				pi->pwdidx = 2;

				for (; cnt < 3; cnt++ ) { /* clear old passwords */

					if (pi->pwds[cnt] != NULL) {

						memset(pi->pwds[cnt], 0, sizeof(uae->password));

					}
				}
			}

			pi->pwdidx = (pi->pwdidx + 1) % 3;

			if (pi->pwds[pi->pwdidx] == NULL) { pi->pwds[pi->pwdidx] = safe_malloc(sizeof(uae->password) + 1); }
			memcpy(pi->pwds[pi->pwdidx], uae->password, sizeof(uae->password));

		}
	}

	return SS$_NORMAL;
}

/*
	Description:

	This thread is responsible to handle identity verification and
	password sync.

	INPUT:		none

	OUTPUT:		none
*/

void *mod_thr(void *s) {

	lib$signal(LLSRV_THRSTARTED, 1, "THREAD_MOD_SYSUAF");

	while (1) {
		UAFACT_T		*uae;

		pthread_cond_wait(&cndmodw, &mtxmodw);

		while ((uae = remove_qh(&verqh)) != NULL) {

			uae->stat = uaf_verify(uae->sess, uae->username);
			uae->sess->flags.condok = 1;
			sys$dclast(&cond_wake_ast, &uae->sess->cndw, 0);

		}

		while ((uae = remove_qh(&pwdqh)) != NULL) {

			uae->stat = uaf_pwd_synch(uae->sess, uae->password);
			uae->sess->flags.condok = 1;
			sys$dclast(&cond_wake_ast, &uae->sess->cndw, 0);

		}

		while ((uae = remove_qh(&pwdupdqh)) != NULL) {

			uae->stat = save_pwd(uae->sess, uae);
			uae->sess->flags.condok = 1;
			sys$dclast(&cond_wake_ast, &uae->sess->cndw, 0);

		}
	}

	return NULL;
}


/*
	Description:

	This routine update a users password on the LDAP server.
	Before the update a verification of the validity of the stored
	password data for the process. Both the PID and start time for
	the PID must match with what was stored when the password arrived
	from SYS$HASH_PASSWORD routine.
	A login using the old password is then carried out for the user
	and the password is updated.
	This require the user has update access for the password on the
	LDAP server. This should almost always be the case.

	INPUT:		username
			pid

	OUTPUT:		SS$_NORMAL always
			Problems are reported in the log file only.
*/

int update_user_pwd(char *username, int unsigned pid) {
	int			stat;
	int			prcidx;
	int unsigned		starttime[2];
	IOSB			iosb;
	ITEM3			itmlst[] = { { sizeof(prcidx), JPI$_PROC_INDEX, &prcidx, 0 },
					     { sizeof(starttime), JPI$_LOGINTIM, &starttime, 0 }, { 0, 0, 0, 0 } };

	if (prclst != NULL && $VMS_STATUS_SUCCESS(stat = sys$getjpiw(EFN_UPDUSRPWD, &pid, 0, &itmlst, &iosb, 0, 0)) &&
	    $VMS_STATUS_SUCCESS(iosb.stat)) {

		PRCLST_T	*pi = &prclst[prcidx];

		if (pi != NULL && pi->pid != 0 && pi->pid == pid && pi->starttime[0] == starttime[0] && pi->starttime[1] == starttime[1]) {

			int			udlen = 0;
			LDAP			*ldap = NULL;
			LDAPMod			ldmod[] = { { LDAP_MOD_REPLACE, "userPassword", NULL }, { 0, NULL, NULL } };
			LDAPMod			*ldmods[] = { ldmod, NULL };
			UAFLDAP_T		uafldap;
			ITEM3			itmlst[] = { {sizeof(uafldap), UAI$_USER_DATA, &uafldap, &udlen }, { 0, 0, 0, 0 } };
			$DESCRIPTOR(usrnamdsc, username);

			if (pi->pwds[(pi->pwdidx + 1) %3] == NULL || pi->pwds[pi->pwdidx] == NULL) { /* no LGI LDAP specific hash_password */

				return SS$_ABORT; /* this will cause pwdupd thread to terminate */

			}

			lock_cluster_cr(LCK_SYSUAF);

			usrnamdsc.dsc$w_length = strlen(username);

			stat = sys$getuai(0, 0, &usrnamdsc, &itmlst, 0, 0, 0);

			if ($VMS_STATUS_SUCCESS(stat) && udlen >= sizeof(uafldap) && uafldap.magic == UAFLDAP_MAGIC && strlen(uafldap.dn) > 0) {

				char 	       		pwd0[33];

				if ((ldap = open_ldap()) != NULL) {

					int			cnt;
					int			dolower;

					dolower =  $VMS_STATUS_SUCCESS(do_trnlnm("LGI_LDAP_PWD_IGNORE_CASE", pwd0, sizeof(pwd0) - 1, PSL$C_EXEC));

					LDAP_LOCK;

					for (cnt = 0; (pwd0[cnt] = pi->pwds[(pi->pwdidx + 1) %3][cnt] ^ rnddata[(cnt + pid) % 1024]); cnt++);

                			stat = ldap_simple_bind_s(ldap, uafldap.dn, pwd0); /* try first with uppercase */

					if (stat != 0 && dolower) { /* try with lower case if bind failed and lower is ok to try */

						for(cnt = 0; pwd0[cnt]; pwd0[cnt] = tolower(pwd0[cnt]), cnt++);

	                			stat = ldap_simple_bind_s(ldap, uafldap.dn, pwd0); /* try with lowercase */
					}

					memset(pwd0, 0, sizeof(pwd0)); /* zero out clear text password */

                			if (stat == 0) {

						int			cnt = 0;
						char			pwd1[33];
						char unsigned		md[20];
						char			shapwd[128] = "{SHA}";
						char			*modval[] = { shapwd, NULL };

						dolower =  $VMS_STATUS_SUCCESS(do_trnlnm("LGI_LDAP_PWD_CNV_LOWER", pwd0, sizeof(pwd0) - 1,
											 PSL$C_EXEC));

						for (; (pwd1[cnt] = (pi->pwds[pi->pwdidx][cnt] ^ rnddata[(cnt + pid) % 1024])) != 0; cnt++) {

							pwd1[cnt] = (dolower ? tolower(pwd1[cnt]) : pwd1[cnt]);

						}

						SHA1_hash(pwd1, md);
						memset(pwd1, 0, sizeof(pwd1)); /* zero out clear text password */
						b64_encode(shapwd + 5, (char *)&md, sizeof(md));

						ldmod[0].mod_values = modval;

						if ((stat = ldap_modify_s(ldap, uafldap.dn, ldmods)) == 0) {

							lib$signal(LLSRV_PWDUPD, 1, uafldap.dn);

						} else {

							lib$signal(LLSRV_PWDUPDFAIL, 2, username, uafldap.dn,
								   LLSRV_LDAPERR, 1, ldap_err2string(stat));

						}

					} else {

						lib$signal(LLSRV_PWDUPDFAIL, 2, username, uafldap.dn, LLSRV_LDAPERR, 1, ldap_err2string(stat));

					}

					ldap_unbind_s(ldap);

					LDAP_UNLOCK;

				}

			} else {

				lib$signal(LLSRV_PWDUPDNODN, 1, username);

			}

			unlock_cluster(LCK_SYSUAF);
		}
	}

	return SS$_NORMAL;
}


/*
	Description:

	This routine return the device and FID for a given  file.

	INPUT:		image name
			dvi buffer, must be at least 65 bytes.
			fid

	OUTPUT:		updated dvi buffer
			updated fid
*/

int get_dvi_fid(char *imagename, char *dvi, short unsigned *fid) {
	int			stat;
	struct FAB		fab = cc$rms_fab;
	struct NAM		nam = cc$rms_nam;

	fab.fab$l_fna = imagename;
	fab.fab$b_fns = strlen(imagename);
	fab.fab$l_nam = &nam;
	
	if ($VMS_STATUS_SUCCESS(stat = sys$open(&fab, 0, 0))) {

		sys$close(&fab, 0, 0);

		strncpy(dvi, nam.nam$t_dvi + 1, nam.nam$t_dvi[0]);
		dvi[nam.nam$t_dvi[0]] = 0;

		fid[0] = nam.nam$w_fid[0];
		fid[1] = nam.nam$w_fid[1];
		fid[2] = nam.nam$w_fid[2];

	}

	return stat;
}


/*
	Description:

	This routine check the given image with the images allowed to do
	password updates.

	INPUT:		image name

	OUTPUT:		SS$_NORMAL if image name was allowed do update password
			SS$_ABORT if image was not allowed
*/

int check_image(char *imagename) {
	int			stat;
	short unsigned		fid[3] = { 0, 0, 0 };
	char			dvi[65] = "";

	if ($VMS_STATUS_SUCCESS(stat = get_dvi_fid(imagename, dvi, fid))) {

		if ((strcmp(dvi, allowed_dvi[0]) == 0 &&
		     allowed_fid[0][0] == fid[0] && allowed_fid[0][1] == fid[1] && allowed_fid[0][2] == fid[2]) || 
		    (strcmp(dvi, allowed_dvi[1]) == 0 &&
		     allowed_fid[1][0] == fid[0] && allowed_fid[1][1] == fid[1] && allowed_fid[1][2] == fid[2])) {

			return SS$_NORMAL; /* image is ok to update user password on LDAP server */

		}
	}
	
	return SS$_ABORT;
}


/*
	Description:

	This routine is invoked as an exit handler when the image terminate.
	Its purpose is to disable the AUDIT listener mailbox.

	INPUT:		none

	OUTPUT:		none
*/

int unsigned		exitcode = 0;

void disable_audit_list() {

	do_action("SYS$INPUT", 5, "$ SET PROC/PRIV=ALL", "$ SET AUDIT/NOLISTENER", "$ LOGOUT_");
	lib$signal(LLSRV_DISAUDLIST, 0);

}

/*
	Descriptor:

	This routine listen to audit events and when a SYSUAF modification event
	is received a check is done to see if it was a password change issued by
	a user it self. If it is the new password is sent to the LDAP server.

	INPUT:		none

	OUTPUT:		none

	IMPLICIT:	The password must have been saved when user called the
			SYS$HASH_PASSWORD with LOGINOUT.EXE or SETP0.EXE
*/

void *pwdupd_thr(void *s) {
	int			stat = 0;
	int			accmode = PSL$C_EXEC;
	short unsigned		audchan;
	char unsigned		*nsa;
	IOSB			iosb;
	EXHBLK_T		exhblk = { 0, &disable_audit_list, 0, &exitcode };
	char			mbxnam[65];
	ITEM3			itmlstl[] = { { 0, LNM$_STRING, &mbxnam, 0}, { 0, 0, 0, 0 } };
	$DESCRIPTOR(tblnamdsc, "LNM_LGI_LDAP");
	$DESCRIPTOR(lognamdsc, "LGI_LDAP_AUDIT_MBX");


	lib$signal(LLSRV_THRSTARTED, 1, "THREAD_AUDIT_LISTENER");

	if ($VMS_STATUS_SUCCESS(stat = sys$crembx(0, &audchan, 16384, 16384 * 2, 0xf000, 0, 0, CMB$M_READONLY, 0))) {
		int unsigned		iosb[2];
		ITEM3			itmlst[] = { { sizeof(mbxnam), DVI$_ALLDEVNAM, &mbxnam, &itmlstl[0].buflen }, { 0, 0, 0, 0 } };

		if ($VMS_STATUS_SUCCESS(stat = sys$getdviw(EFN_AUDEVN, audchan, 0, &itmlst, &iosb, 0, 0, 0))) {
			char			cmd[128];

			sprintf(cmd, "$ SET AUDIT/LISTENER=%*.*s", itmlstl[0].buflen, itmlstl[0].buflen, mbxnam);

			if (!$VMS_STATUS_SUCCESS(do_action("SYS$INPUT", 5, "$ SET PROC/PRIV=SECURITY", cmd, "$ LOGOUT_"))) {

				lib$signal(LLSRV_ERRENAUDIT, 1, "THREAD_AUDIT_LISTENER", LLSRV_NOPWDSYNCH, 0, stat);

				pthread_detach(pthread_self());
				pthread_exit(NULL);

			}

			stat = sys$dclexh(&exhblk);
		}
	}

	stat = sys$crelnm(0, &tblnamdsc, &lognamdsc, &accmode, &itmlstl);

	if (!$VMS_STATUS_SUCCESS(stat = get_dvi_fid("SYS$SYSTEM:LOGINOUT.EXE", allowed_dvi[0], allowed_fid[0]))) {

		lib$signal(stat);

	}


	if (!$VMS_STATUS_SUCCESS(stat = get_dvi_fid("SYS$SYSTEM:SETP0.EXE", allowed_dvi[1], allowed_fid[1]))) {

		lib$signal(stat);

	}

	nsa = safe_malloc(16384);

	do {
		
		if ($VMS_STATUS_SUCCESS(stat = sys$qiow(EFN_AUDEVN, audchan, IO$_READVBLK, &iosb, 0, 0, nsa, 16384, 0, 0, 0, 0))) {

			int			datlen = iosb.len - NSA$C_MSG_HDR_LENGTH;
			char unsigned		*nsadat = nsa + NSA$C_MSG_HDR_LENGTH;
			struct nsamsgdef	*nsahdr = (struct nsamsgdef *)nsa;

			if ($VMS_STATUS_SUCCESS(iosb.stat) &&
			    nsahdr->nsa$w_record_type == NSA$C_MSG_SYSUAF && nsahdr->nsa$w_record_subtype == NSA$C_SYSUAF_MODIFY) {

				int unsigned		pid = 0;
				int			pwdchg = 0;
				char			username[33] = "";
				char			uafusr[33] = "";
				char			imagename[1024] = "";

				while (datlen > 0) {

					short unsigned		pkglen = *((short unsigned *)(nsadat));
					short unsigned		pkgtyp = *((short unsigned *)(nsadat + 2));
					char unsigned		*data = (void *)(nsadat + 4);

					switch (pkgtyp) {

						case NSA$C_PKT_IMAGE_NAME:	     strncpy(imagename, (char *)data, pkglen - 4); break;
						case NSA$C_PKT_USERNAME:	     strncpy(username, (char *)data, pkglen - 4); break;
						case NSA$C_PKT_UAF_SOURCE:	     strncpy(uafusr, (char *)data, pkglen - 4); break;
						case NSA$C_PKT_PROCESS_ID:	     pid = *(int unsigned *)(data); break;
						case NSA$C_PKT_SENSITIVE_FIELD_NAME:
							if (pkglen - 4 == 8 && strncmp((char *)data, "Password", pkglen - 4) == 0) {

								pwdchg = 1;

							};
							break;
					}
					
					datlen -= pkglen;
					nsadat += pkglen;
				}

				username[strcspn(username, " ")] = 0; /* remove traling spaces */
				uafusr[strcspn(uafusr, " ")] = 0;

				if (pwdchg && strcmp(username, uafusr) == 0 &&
				    $VMS_STATUS_SUCCESS(check_image(imagename))) { /* Pwd updated by user */

					stat = update_user_pwd(username, pid);

				}
			}
		}

	} while ($VMS_STATUS_SUCCESS(stat));

	safe_free(nsa);

	do_action("SYS$INPUT", 5, "$ SET AUDIT/NOLISTENER", "$ LOGOUT_");

	lib$signal(LLSRV_ERRDAUDIT, 1, "THREAD_AUDIT_LISTENER", LLSRV_NOPWDSYNCH, 0, stat);

	pthread_detach(pthread_self());

	return NULL;
}


/*
	Description:

	This routine is used to execute a User Action Entry.
	The calling thread will continue when the action has been completed.

	INPUT:		sess handler
			User Action Entry
			queue header
			condition variable that wake work thread

	OUTPUT:		status from action routine.
*/

int do_uae(LGI_SESS_T *sess, UAFACT_T *uae, void *qh, pthread_cond_t *cnd) {

	uae->sess = sess;

	insert_qt(qh, uae);

	sess->flags.condok = 0;
	sys$dclast(&cond_wake_ast, cnd, 0);

	while (!sess->flags.condok) { pthread_cond_wait(&sess->cndw, &sess->mtxw); }

	return uae->stat;
}


/*
	Description:

	This routine is invoked to save the password for a user.
	The password is scrambled before stored.

	INPUT:		session handler
			LGI Message 

	OUTPUT:		status from saving the password.
*/

int thread_cmd_password(LGI_SESS_T *sess, LIM_ENTRY_T *le) {
	int			stat;
	int			cnt = 0;
	char			dummy[256];
        UAFACT_T		*uae = alloc_qe(sizeof(UAFACT_T));

	strncpy(uae->username, sess->identity, sizeof(uae->username) - 1);

	/*
		Do some simple scrambling of the password by XOR
		it with some pre-generated bytes.
		We do this to prevent someone with access to SDA to find
		passwords stored in memory.
	*/

	for(; cnt < sizeof(le->lim.request.password.password) && cnt < sizeof(uae->password); cnt++ ) {

		uae->password[cnt] = le->lim.request.password.password[cnt] ^ rnddata[(cnt + sess->pid) % 1024];

	}

	stat = do_uae(sess, uae, &pwdupdqh, &cndmodw);

	return stat;
}

/*
	Description:

	This routine is used to synchronize the password on the host with the password
	on LDAP server.

	INPUT:		sess handler
			credential (password)

	OUTPUT:		status from action routine
*/

int synch_password(LGI_SESS_T *sess, char *credential) {
	int			stat;
        UAFACT_T		*uae = alloc_qe(sizeof(UAFACT_T));

	strncpy(uae->password, credential, sizeof(uae->password) - 1);

	stat = do_uae(sess, uae, &pwdqh, &cndmodw);

	free_qe(sizeof(UAFACT_T), uae);

	return stat;

}


/*
	Description:

	This routine is used to verify the identity (username) on the host.
	The user is added if the DN the user belongs to is valid for the host
	and if the create flag is set.

	INPUT:		sess handler
			credential (password)

	OUTPUT:		status from action routine
*/

int verify_username(LGI_SESS_T *sess, char *identity, int create) {
	int			stat;
	char			dummy[256];
        UAFACT_T		*uae = alloc_qe(sizeof(UAFACT_T));

	strncpy(uae->username, identity, sizeof(uae->username) - 1);

	stat = do_uae(sess, uae, &verqh, &cndmodw);

	if (create && add_user_ok && !$VMS_STATUS_SUCCESS(stat)) {

		stat = do_uae(sess, uae, &addqh, &cndaddw);

		stat = do_uae(sess, uae, &verqh, &cndmodw);

	}

	free_qe(sizeof(UAFACT_T), uae);

	return stat;
}


/*
	Description:

	This routine is inoked when the program is started.

	INPUT:		none

	OUTPUT:		none
*/

int uaf_init() {
	int			stat;
	int			maxprc;
	char			dummy[256];
	ITEM3			itmlst[] = { { sizeof(maxprc), SYI$_MAXPROCESSCNT, &maxprc, 0 }, { 0, 0, 0, 0 } };

	pthread_mutex_init(&mtxmodw, NULL);
	pthread_cond_init(&cndmodw, NULL);
	pthread_rwlock_init(&rwluai, NULL);

	if ($VMS_STATUS_SUCCESS(do_trnlnm("LGI_LDAP_PWD_SYNCH", dummy, sizeof(dummy) - 1, PSL$C_EXEC))) {

		if ($VMS_STATUS_SUCCESS(stat = sys$getsyiw(0, 0, 0, &itmlst, 0, 0, 0))) {

			prclst = (PRCLST_T *)safe_malloc(sizeof(PRCLST_T) * (maxprc + 1)); /* allocate enough memory for all processes */

			memset(prclst, 0, sizeof(PRCLST_T) * (maxprc + 1));

			generate_random(rnddata, sizeof(rnddata));

			pthread_create(&thrpwdupd, NULL, &pwdupd_thr, NULL);
		}
	}

	pthread_create(&thrmod, NULL, &mod_thr, NULL);

	if ($VMS_STATUS_SUCCESS(do_trnlnm("LGI_LDAP_ADD_USER", dummy, sizeof(dummy) - 1, PSL$C_EXEC))) {

		pthread_mutex_init(&mtxaddw, NULL);
		pthread_cond_init(&cndaddw, NULL);

		if (pthread_create(&thradd, NULL, &add_thr, NULL) == 0) { add_user_ok = 1; }

	}

	if ($VMS_STATUS_SUCCESS(do_trnlnm("LGI_LDAP_REM_USER", dummy, sizeof(dummy) - 1, PSL$C_EXEC))) {

		pthread_create(&thrrem, NULL, &rem_thr, NULL);

	}

	return SS$_NORMAL;
}
