/*
	Copyright (C) 2004 Jonas Lindholm

	This software was developed by Jonas Lindholm, jlhm@usa.net

	History

	V1.0		Jonas Lindholm	2004-05-14

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

	This file contain code developed by others, see b64_encode.
*/

#include "ldap.h"

#include <stdarg.h>

#include <starlet.h>
#include <ssdef.h>
#include <stsdef.h>
#include <psldef.h>
#include <lnmdef.h>
#include <syidef.h>
#include <descrip.h>
#include <lckdef.h>
#include <cmbdef.h>
#include <iodef.h>
#include <dvidef.h>
#include <accdef.h>

#include <lib$routines.h>
#include <libdef.h>
#include <libvmdef.h>

#include <rms.h>
#include <prcdef.h>
#include <pthread.h>

#include "lgi.h"
#include "lgildap.h"
#include "server.h"


typedef struct clulck_s {
	int			lckcnt;
	int unsigned		lksb[2];
	pthread_rwlock_t	rwlck;
	pthread_cond_t		cndw;
	pthread_mutex_t		mtxw;
	pthread_mutex_t		mtxl;
} CLULCK_T;


pthread_mutex_t         memmtx;
pthread_mutex_t         getvmmtx;

int unsigned		qe_zone;
int			cluster = 0;
CLULCK_T		lcks[LCK_MAX_INDEX];


/*
	Description:

	This routine is invoked to allocate N bytes using malloc.
	A mutex is used to control access.
	Do not call this routine from AST.

	INPUT:		size of memory to allocate

	OUTPUT:		pointer to allocated memory
			NULL is something went wrong
*/

void *safe_malloc(int size) {
        void            *p;

        pthread_mutex_lock(&memmtx);

        p = malloc(size);

	if (p == NULL) {

		lib$stop(LLSRV_MEMALLOCFAIL, 1, size);

	}

        pthread_mutex_unlock(&memmtx);
        return p;
}


/*
	Description:

	This routine is used to free memory
	A mutex is used to control access.
	Do not call this routine from AST.

	INPUT:		pointer to memory to free

	OUTPUT:		none
*/

void safe_free(void *p) {
        pthread_mutex_lock(&memmtx);

        free(p);

        pthread_mutex_unlock(&memmtx);
}


void *alloc_qe(int size) {
        int             stat;
        void            *qe;

        pthread_mutex_lock(&getvmmtx);

        stat = LIB$GET_VM(&size, &qe, &qe_zone);

        pthread_mutex_unlock(&getvmmtx);

        return qe;
}


void free_qe(int size, void *qe) {
        int             stat;

        pthread_mutex_lock(&getvmmtx);

        stat = LIB$FREE_VM(&size, qe, &qe_zone);

        pthread_mutex_unlock(&getvmmtx);
}


int insert_qt(void *hdr, void *ent) {

        return LIB$INSQTIQ(ent, hdr, 0);
}


int insert_qh(void *hdr, void *ent) {

        return LIB$INSQHIQ(ent, hdr, 0);
}


void *remove_qt(void *qhd) {
        int             stat;
        void            *qe;

        stat = LIB$REMQTIQ(qhd, &qe, 0);
        return ($VMS_STATUS_SUCCESS(stat) ? qe : NULL);
}


void *remove_qh(void *qhd) {
        int             stat;
        void            *qe;

        stat = LIB$REMQHIQ(qhd, &qe, 0);
        return ($VMS_STATUS_SUCCESS(stat) ? qe : NULL);
}


void *get_free_qe(void *qhd, int size) {
        void            *qe = remove_qh(qhd);

        if (qe == NULL) {

                qe = alloc_qe(size);

        }

        return qe;
}


void init_q(void *qhd, int size, int cnt) {

	for(; cnt > 0; cnt--) {

		insert_qt(qhd, alloc_qe(size));

	}
}
                           

/*
	The b64_encode is from Linux pam_ldap.c.

	Copyright (C) 1998-2003 Luke Howard.
	This file is part of the pam_ldap library.
	Contributed by Luke Howard, <lukeh@padl.com>, 1998.

	Portions Copyrighted Andrew Morgan, 1996.  All rights reserved.
	Modified by Alexander O. Yuriev.

	INPUT:		array of bytes
			number of bytes

	OUTPUT:		B64 encoded string of input bytes.
*/

static const char ch[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";

void b64_encode(char *out, char *value, unsigned int vlen)
{
	unsigned int i, j;
	char chin[4], chout[5];

	j = 0;

	chin[0] = chin[1] = chin[2] = 0;
	chout[0] = chout[1] = chout[2] = chout[3] = chout[4] = 0;

	for(i = 0; i < vlen; i++) {

		chin[j++] = value[i];

		chout[0] = ch[(chin[0] >> 2) & 0x3f];
		chout[1] = ch[((chin[0] << 4) & 0x30) | ((chin[1] >> 4) & 0x0f)];
		chout[2] = ch[((chin[1] << 2) & 0x3c) | ((chin[2] >> 6) & 0x03)];

		if(j == 3) {

			chout[3] = ch[chin[2] & 0x3f];
			strcat(out, chout);
			chin[0] = chin[1] = chin[2] = 0;
			chout[0] = chout[1] = chout[2] = chout[3] = 0;
			j = 0;
		}
	}

	if(j == 1) {

		chout[2] = chout[3] = '=';

	} else {

		if(j == 2) { chout[3] = '='; }

	}
	
	strcat(out, chout);
}


/*
	Description:

	This routine translate a logical name found in table LNM_LGI_LDAP.
	The logical name must be defined with access mode <accmode>.

	INPUT:		logical name
			output string
			maximum length of output string
			access mode

	OUTPUT:		updated output string
			SS$_NORMAL if logical name was found
			any other error when not found.
*/

int do_trnlnm(char *lnm, char *trn, int trnlen, int accmode) {
	int			stat;
	int unsigned		mask = LNM$M_CASE_BLIND;
	short unsigned		retlen = 0;
	ITEM3			itmlst[2] = { { trnlen, LNM$_STRING, trn, &retlen }, { 0, 0, 0, 0 } };

	$DESCRIPTOR(tbldsc, "LNM_LGI_LDAP");
	$DESCRIPTOR(lnmdsc, lnm);

	lnmdsc.dsc$w_length = strlen(lnm);

	if (($VMS_STATUS_SUCCESS(stat = sys$trnlnm(&mask, &tbldsc, &lnmdsc, &accmode, &itmlst)))) {

		trn[retlen] = 0;

	}

	return stat;	
}


/*
        Description:

        This AST rutine is invoked when a command procedure times out.
        It signal a condition variable that will wake up the thread waiting
        for the command procedure to finish.

        INPUT:          condition variable

        OUTPUT          none
*/

void cond_wake_ast(pthread_cond_t *cndw) {

        pthread_cond_signal_int_np(cndw);

}


/*
	Description:

	This routine execute a command file and write zero or more text data
	to the created process.
	If the process does not complete in the specified timeout period the
	process is terminated and an error status is returned.

	INPUT:		command procedure to run or SYS$INPUT to let the process
			read commands from text data arguments.
			timout for process e.g. max run time in seconds.
			zeo or more text data arguments.

	OUTPUT:		SS$_ABORT or any other error if something went wrong.
			Final status from process.
*/

int do_action(char *actrtn, int timeout, ...) {
        int                     stat;
        int                     argcnt = 0;
        int unsigned            trmunit;
        int unsigned            pid;
        int unsigned            iosb[2] = { SS$_NORMAL, 0 };
        short unsigned          outchan = 0;
        short unsigned          trmchan = 0;
        char                    mbxnam[65];
        struct timespec         wait = { timeout, 0 };
        struct accdef           acc;
        pthread_cond_t          cndt;
        pthread_mutex_t         mtxt;
        va_list                 ap;
        $DESCRIPTOR(mbxnamdsc, mbxnam);
        $DESCRIPTOR(loginoutdsc, "SYS$SYSTEM:LOGINOUT.EXE");
	$DESCRIPTOR(outdsc, (strcmp(actrtn, "SYS$INPUT") == 0 ? "NLA0:" : "SYS$MANAGER:LGILDAPSRV_USER_ACTION.LOG"));
        ITEM3                   itmlst[] = { { sizeof(mbxnam) - 1, DVI$_ALLDEVNAM, &mbxnam, &mbxnamdsc.dsc$w_length },
                                             { sizeof(trmunit), DVI$_UNIT, &trmunit, 0}, { 0, 0, 0, 0 } };

        outdsc.dsc$w_length = strlen(outdsc.dsc$a_pointer);

        pthread_cond_init(&cndt, NULL);
        pthread_mutex_init(&mtxt, NULL);

        va_count(argcnt);

        if ($VMS_STATUS_SUCCESS(stat = sys$crembx(0, &outchan, 1024, 1024 * argcnt, 0xf000, 0, 0, CMB$M_WRITEONLY, 0))) {

                if ($VMS_STATUS_SUCCESS(stat = sys$crembx(0, &trmchan, ACC$K_TERMLEN * 2, ACC$K_TERMLEN * 2, 0xf000, 0, 0, CMB$M_READONLY, 0))) {

                        if ($VMS_STATUS_SUCCESS(stat = sys$getdviw(EFN_ACTRTN, trmchan, 0, &itmlst, &iosb, 0, 0, 0))) {

                                memset(&itmlst[1], 0, sizeof(itmlst[1]));

                                stat = sys$getdviw(EFN_ACTRTN, outchan, 0, &itmlst, &iosb, 0, 0, 0);

                        }
                }
        }

        if ($VMS_STATUS_SUCCESS(stat) &&
            $VMS_STATUS_SUCCESS(stat = sys$creprc(&pid, &loginoutdsc, &mbxnamdsc, &outdsc, 0, 0, 0, 0, 0, 0, trmunit,
                                                  PRC$M_DETACH, 0, 0, 0))) {

                char                    str[strlen(actrtn) + 2];
                struct timespec         abstime;

                sprintf(str, "@%s", actrtn);
                stat = sys$qiow(EFN_ACTRTN, outchan, IO$_WRITEVBLK | IO$M_NOW, &iosb, 0, 0, &str, strlen(str), 0, 0, 0, 0);

                va_start(ap, timeout);

                while(argcnt-- - 2) {
                        char                    *cmd = va_arg(ap, char *);

                        stat = sys$qiow(EFN_ACTRTN, outchan, IO$_WRITEVBLK | IO$M_NOW, &iosb, 0, 0, cmd, strlen(cmd), 0, 0, 0, 0);

                }

                va_end(ap);

                pthread_get_expiration_np(&wait, &abstime);

                stat = sys$qiow(EFN_ACTRTN, outchan, IO$_WRITEOF | IO$M_NOW, &iosb, 0, 0, 0, 0, 0, 0, 0, 0);

                stat = sys$qio(EFN_ACTRTN, trmchan, IO$_READVBLK, &iosb, &cond_wake_ast, &cndt, &acc, sizeof(acc), 0, 0, 0, 0);

                pthread_cond_timedwait(&cndt, &mtxt, &abstime);

                sys$delprc(&pid, 0, 0); /* in case process still running */

        }

        sys$dassgn(trmchan);
        sys$dassgn(outchan);

        if ($VMS_STATUS_SUCCESS(stat) && !$VMS_STATUS_SUCCESS(iosb[0])) { stat = iosb[0]; }

        if ($VMS_STATUS_SUCCESS(stat)) { stat = acc.acc$l_finalsts; }

        pthread_cond_destroy(&cndt);
        pthread_mutex_destroy(&mtxt);

        return stat;
}


/*
	Description:

	This command procedure translate a Unix directory specification to
	define name and directory structure used by VMS.
	The first directory in the Unix specification is used as device name and
	is target for one logical name translation.
	The remaining directories are translatede to VMS directory specification.

	Example:

	/home/dev/dev1 => HOME:[DEV.TEST1]

	If the exist a logical name for HOME: in logical table LNM_LGI_LDAP
	the equivalence name is used as device instead.

	INPUT:		Unix file specification
			Default device
			Maximum length of default device
			Default directory
			Maximum length of default directory

	OUTPUT:		Updated default device
			Updated default directory
*/

int translate_unix2vms(char *unixdir, char *defdev, int defdevlen, char *defdir, int defdirlen) {
        int                     cnt = 0;
        char                    *p = unixdir;
        char                    *d = defdir;
        char                    *s;

        *d++ = '[';
	
        while (*p != 0 && ((s = strchr(p, '/')) != NULL || (s = strchr(p, 0)) != NULL)) {

                if ((s - p) > 0) {
                        int		len;

                        if (cnt++ == 0) {

				len = MIN((s - p), defdevlen - 2);
                                strncpy(defdev, p, len);
                                defdev[len] = ':';
                                defdev[len + 1] = 0;

                        } else {

                                if (cnt > 2) { *d++ = '.'; };
				*d = 0;
				len = MIN((s - p), (defdirlen - strlen(defdir) - 2));
                                strncpy(d, p, len);
                                d += len;

                        }

                        p += (s - p);
                }

                p += (*p != 0);
        }

        *d++ = ']';
        *d = 0;

        return cnt;
}


LDAP *open_ldap() {
        int             stat;
        char            ldaphost[256];
        char            basedn[256];
        LDAP            *ldap = NULL;

        if (!use_ssl) {

                if (!$VMS_STATUS_SUCCESS(do_trnlnm("LGI_LDAP_HOST", ldaphost, sizeof(ldaphost) - 1, PSL$C_EXEC))) {

                        lib$stop(LLSRV_LDAPHOSTMISS);

                }
        }

        if (!$VMS_STATUS_SUCCESS(do_trnlnm("LGI_LDAP_BASEDN", basedn, sizeof(basedn) - 1, PSL$C_EXEC))) {

                lib$stop(LLSRV_LDAPBASEDNMISS);

        }

	LDAP_LOCK;

        if ((ldap = ldap_init((use_ssl ? "127.0.0.1" : ldaphost), (use_ssl ? SSL_PORT: 389))) != NULL) {
		int             vers = 3;
		int             timeout = 10; /* allow only 10 seconds for LDAP operations */

		if ((stat = ldap_set_option(ldap, LDAP_OPT_PROTOCOL_VERSION, &vers)) == 0 &&
		    (stat = ldap_set_option(ldap, LDAP_OPT_TIMELIMIT, &timeout)) == 0) {

#if LDAP_API_VERSION > 2004
			char		dummy[256];

			if (!$VMS_STATUS_SUCCESS(do_trnlnm("LGI_LDAP_USE_NO_SSL", dummy, sizeof(dummy) - 1, PSL$C_EXEC))) {

				stat = ldap_tls_start(ldap, 1);

				if (stat != 0 &&
				    $VMS_STATUS_SUCCESS(do_trnlnm("LGI_LDAP_NO_SSL_OK", dummy, sizeof(dummy) - 1, PSL$C_EXEC))) {

					stat = 0; /* SSL failed but it is ok ! */

				}
			}
#endif

			if (stat == 0 && (stat = ldap_simple_bind_s(ldap, NULL, NULL)) == 0) {

				LDAP_UNLOCK;

                                return ldap;

			}
                }

		ldap_unbind(ldap);
        }

	LDAP_UNLOCK;

        lib$signal(LLSRV_LDAPCONNERR, 0);

        return NULL;
}


/*
	Description:

	This routine is invoked whne the cluster lock has been
	granted.
	It wakes up the waiting thread

	INPUT:		condition variable

	OUTPUT:		none
*/

void lock_granted(pthread_cond_t *cndl) {

	pthread_cond_signal_int_np(cndl);

}


/*
	Description:

	This routine take a lock on the resource indicated by index <id>.
	If the host is part of a VMS cluster also a cluster lock is used
	to synchronize the access between all instances of the LGI LDAP server.

	INPUT:		lock index <id>
			lock mode, LCK$M_NLMODE, LCK$M_CRMODE or LCK$M_EXMODE.

	OUTPUT:		none.

	IMPLICIT:

	It is invalid to change from a CR to EX lock or vice versa without
	doing a NL lock between the lock modes.
*/

void lock_cluster(int id, int lckmode) {
	int unsigned	stat;

	if (id >= (sizeof(lcks) / sizeof(CLULCK_T))) {

		lib$signal(LLSRV_INTERRLCK, 0); /* internal error */
		return;

	}

	switch (lckmode) {

		case LCK$K_CRMODE: stat = pthread_rwlock_rdlock(&lcks[id].rwlck); break;
		case LCK$K_EXMODE: stat = pthread_rwlock_wrlock(&lcks[id].rwlck); break;

	}

	/*
		The EDEADLK means we have a programming error in our code.
		A thread is not allowed to go directly from RD to WR or from
		WR to RD. It must always unlock between different lock modes.
	*/

	if (stat == EDEADLK) {

		lib$signal(LLSRV_INTERRLCK, 0); /* internal error */
		return;

	}

	/*
		If we are running in a cluster we must also take a cluster lock.
		For RD lock we request CR lock and for WR lock we request EX lock.
		We count all lock requests and when we do the first non NL lock
		we convert our NL lock to CR or EX.
		When we do our last conversion from EX or CR to NL we convert our
		lock to NL.
	*/

	if (cluster) {

		pthread_mutex_lock(&lcks[id].mtxl);

		lcks[id].lckcnt += ((lckmode != LCK$K_NLMODE) | -(lckmode == LCK$K_NLMODE));

		if ((lcks[id].lckcnt == 1 && lckmode != LCK$K_NLMODE) || (lcks[id].lckcnt == 0 && lckmode == LCK$K_NLMODE)) {

			stat = sys$enq(EFN_ENQ, lckmode, &lcks[id].lksb, LCK$M_CONVERT | LCK$M_SYNCSTS, 0, 0, &lock_granted, &lcks[id].cndw,
				       0, 0, 0, 0);

			if (stat != SS$_SYNCH) { while (lcks[id].lksb[0] == 0) pthread_cond_wait(&lcks[id].cndw, &lcks[id].mtxw); }

		}

		pthread_mutex_unlock(&lcks[id].mtxl);
	}

	switch (lckmode) {

		case LCK$K_NLMODE: pthread_rwlock_unlock(&lcks[id].rwlck); break;

	}
}


/*
	Description:

	This routine take a exlusive lock on lock index <id>.

	INPUT:		lock index <id>

	OUTPUT:		none
*/

void lock_cluster_ex(int id) {

	lock_cluster(id, LCK$K_EXMODE);

}


/*
	Description:

	This routine take a read lock on lock index <id>.

	INPUT:		lock index <id>

	OUTPUT:		none
*/

void lock_cluster_cr(int id) {

	lock_cluster(id, LCK$K_CRMODE);

}


/*
	Description:

	This routine unlock a granted lock with index <id>.

	INPUT:		lock index <id>

	OUTPUT:		none
*/

void unlock_cluster(int id) {

	lock_cluster(id, LCK$K_NLMODE);

}


int do_sprintf(char *search, int searchlen, char *filter, char *uid) {
	char			tmp[strlen(filter) + strlen(uid) * 2 + 1];

	lib$establish(lib$sig_to_ret); /* any conditions such as ACCVIO will be translated to return code */

	sprintf(tmp, filter, uid, uid, 1, 2, 3, 4, 5, 6, 7, 8); /* 0 will be translated to text string (null) and we don't want that */

	if (sprintf(search, "%.*s", searchlen, tmp) == strlen(tmp)) {

		return SS$_NORMAL;

	}

	return LIB$_OUTSTRTRU; /* search string was truncated */
}

/*
	Description:

	This routine return the search string to use when filter data
	from the LDAP server.

	Because the filter string can be defined by a logical name and no
	more than 2 %s can be used a check must be implemented that take
	care of more than 2 %s.

	We do that by calling a routine doing the sprintf but also define
	a condition handler than convert access violation to return status.
	We also put several 1,2,3,etc in the sprintf argument list so we don't
	ending up with a random address. The reason for several is in case
	the string contains other conversion characters than %s such as %d,
	%i and %x;

	INPUT:		search string
			maximum length of search string
			uid

	OUTPUT:		updated search string
*/

int get_filter(char *search, int searchlen, char *uid) {
	int			stat;
	char			filter[256] = LDAP_FILTER;

	do_trnlnm("LGI_LDAP_FILTER", filter, sizeof(filter) - 1, PSL$C_EXEC);

	if (!$VMS_STATUS_SUCCESS(stat = do_sprintf(search, searchlen, filter, uid))) {

		lib$signal(LLSRV_INVFILSTR, 2, filter, LDAP_FILTER, stat, 0, 0);

		do_sprintf(search, searchlen, LDAP_FILTER, uid);

	}

	return stat;
}


void create_vmzone() {
        int             stat;
        int             align = 16;
        int             flags = LIB$M_VM_GET_FILL0;

        pthread_mutex_init(&getvmmtx, NULL);

        stat = LIB$CREATE_VM_ZONE(&qe_zone, 0, 0, &flags, 0, 0, 0, &align, 0, 0, 0, 0, 0);

        if (!$VMS_STATUS_SUCCESS(stat)) { lib$stop(stat); }

}


void init_locks() {
	int		stat;
	int		cnt = 0;
	int		osvmaj = 0;
	int		osvmin = 0;
	int		osvsub = 0;
	char		osversion[9];
	ITEM3		itmlst[] = { { sizeof(cluster), SYI$_VAXCLUSTER, &cluster, 0 },
				     { sizeof(osversion), SYI$_VERSION, &osversion, 0}, { 0, 0, 0, 0 } };

	stat = sys$getsyi(0, 0, 0, &itmlst, 0, 0, 0);

	osvers = 0;

	if ((sscanf(osversion, "V%d.%d-%d", &osvmaj, &osvmin, &osvsub) == 3) ||
	    (sscanf(osversion, "V%d.%d", &osvmaj, &osvmin) == 2)) {

		osvers = osvmaj * 10000 + osvmin * 100 + osvsub;

	}

	for(; cnt < sizeof(lcks) / sizeof(CLULCK_T); cnt++) {
		char		lcknam[33];
		$DESCRIPTOR(lcknamdsc, lcknam);

		lcknamdsc.dsc$w_length = sprintf(lcknam, "LGI_LDAP_CLU_LCK_%d", cnt);

		if (!cluster ||
		    ($VMS_STATUS_SUCCESS(stat = sys$enqw(0, LCK$K_NLMODE, &lcks[cnt].lksb, LCK$M_SYSTEM, &lcknamdsc, 0, 0, 0, 0, 0, 0, 0)) &&
		     $VMS_STATUS_SUCCESS(lcks[cnt].lksb[0]))) {

			pthread_cond_init(&lcks[cnt].cndw, NULL);
			pthread_mutex_init(&lcks[cnt].mtxw, NULL);
			pthread_mutex_init(&lcks[cnt].mtxl, NULL);
			pthread_rwlock_init(&lcks[cnt].rwlck, NULL);

		} else {

			lib$stop(stat, lcks[cnt].lksb[0]);

		}
	}
}


int util_init() {
	int		stat;

	init_locks();

	create_vmzone();

	return SS$_NORMAL;
}
