/*****************************************************************************/
/*
                                  VMSeti.c

A Web-server script for displaying the progress and other information for a
VMS-based SETI environment.  SETI@home is a cause worthy of support from two
aspects.  An interesting and seminal use of globally distributed computing
resources (and on a cooperative basis ... ignoring those who cannot adhere to
the agreed conditions of use).  And, the intriguing notion that something just
might be found!  Since my initial interest the VMS contribution rank has gone
from the mid-twenties in mid-1999 to the mid-teens in early 2000.  Not bad for
a "commercial" OS one would expect to be securely locked in major corporate
DP equipment rooms.

With acknowlegement of Edward J.Groth, Physics Dept., Princeton Univeristy, for
the original idea and DCL implementation for this sort of VMS interface.

Should support WASD, OSU and CSWS (VMS Apache), possibly Purveyor environments. 
Can be used both as a CGI and CGIplus (with only activation efficiencies)
script under WASD.

Lots of this is guesswork.  The exact content of the files are apparently
confidential to SETI@home.  Some things are obvious, others less so.  If
anybody can clarify the format of the files I'd be pleased to get the
information.  Of course, if anything is completely SNAFU then please alert me
to that as well.


SKYMAP
------
A work-unit locating skymap is available to browsers that support CSS1 (W3C
Cascading Style Sheets level 1).  This is a very powerful technology allowing
the skymap functionality to be incorporated with a minimum of programming
(couple of hours or so).  This currently does limit it to the more recent
generation of browsers (at least Navigator/Communicator 4 and Internet
Explorer 4 ... it appears as if there are still a few problems with Navigator
4.6 at least).  Anyway, here's the skymap with a minimum of effort!

The skymap images have been converted to byte array header files for
compilation into and serving by the VMSeti script itself without reference to
graphics files located anywhere else.  This is slightly more expensive in terms
of processing overhead but makes configuration a little simpler - only the
executable and it's support procedure need to be placed into the script
directory (/cgi-bin/) and it should all just happen.  The support procedure can
be modified to specify a path for the image files (SKYMAP*.GIFs) themselves if
processing efficiency is a real (or imagined ;^) issue.

The skymap celestial image was sourced and I imagine copyright by SETI@home
(and is used in the spirit-of, but without permission), though a cursory look
through the site shows such legalese to be minimal.


SETUP
-----
Place VMSETI.COM and VMSETI.EXE into the required script directory.

Edit the VMSETI.COM in that script directory, modifying the default location of
VMSETI_... symbols to suit the local environment.  VMSETI_STATE and
VMSETI_WORK_UNIT must be assigned.  VMSETI_USER_INFO, VMSETI_OUTFILE and
VMSETI_SKYMAP are optional and if not assigned personal details, power spikes
and/or the skymap are not displayed. 

Data files can also be located by supplying a path to the script.  The
translation of this path must be the RMS-syntax directory containing the
required data files.  In this way a default location can be provide using the
script wrapping procedure and if the site supports multiple processing streams
these can be selected by supplying a URL containing a path along with the
script.

To include some local information on the report page use the VMSETI_LOCAL
environment variable to supply text (plain or HTML), or if the first character
is a '@' the location of a file containing this text.  Local information is
included towards the bottom of the page.

To access the permanently open and locked OUTFILE containing power spikes and
gaussian peaks of interest the ACP QIO interface is used.  For an unprivileged
HTTP server this requires the script to be installed with SYSPRV.  Without this
privilege it cannot access the file and just reports "(unable to report on
spikes)".  With thanks to James McKinney for demonstrating to me that this
could be done.

  $ INSTALL ADD cgi-bin-dir:VMSETI.EXE /PRIV=(SYSPRV)


ENVIRONMENT VARIABLES (symbols or logicals)
---------------------
VMSETI_STATE               location of state.sah (mandatory)
VMSETI_WORK_UNIT           location of work_unit.sah (mandatory)
VMSETI_USER_INFO           location of user_info.sah (optional)
VMSETI_OUTFILE             location of outfile.sah (optional)
VMSETI_LOCAL               text, or location of local information file
                           (included at bottom of bar graphs, optional)
VMSETI_BACKGROUND          path to background image (optional)
VMSETI_SKYMAP              path to skymap images (optional)
VMSETI_FILE_TYPE           ".SAH" for v2.n


LOGICAL NAMES
-------------
VMSETI$DBUG   turns on all "if (Debug)" statements


BUILD DETAILS
-------------
$ @BUILD_ONE BUILD VMSETI   !for compile and link
$ @BUILD_ONE LINK VMSETI    !for link-only to supplied object file


COPYRIGHT
---------
Copyright (C) 2000-2004 Mark G.Daniel
mark.daniel@wasd.vsm.com.au (mark.daniel@dsto.defence.gov.au)
This program, comes with ABSOLUTELY NO WARRANTY.
This is free software, and you are welcome to redistribute it
under the conditions of the GNU GENERAL PUBLIC LICENSE, version 2.


VERSION HISTORY (update SOFTWAREVN as well)
---------------
23-MAR-2004  MGD  v1.5.2, minor conditional mods to support IA64,
                          SkyMap() adjust RA for occasional >= 24.0
                          (based on comments and suggestions in alt.sci.seti)
24-JAN-2003  WG   v1.5.1, I added size-checks on user-info, especially
                          on email and country, for empty strings would
                          result in an error (404). Empty entries are valid
                          (I think) - anyway, I get them empty again and again
01-JAN-2003  MGD  v1.5.0, (and yes, I'll blame any bugs on the headache)
                          skymap images handled internally by default,
                          bugfix; brain-dead 'LinkUrl' 
30-JAN-2001  MGD  v1.4.2, add estimated completion stats, FFT rate
25-JAN-2001  MGD  v1.4.1, use CGILIB with 'fixbg' for CSWS V1.0-1 (Apache)
27-OCT-2000  MGD  v1.4.0, rework "graphs" a little,
                          remove work-unit information,
                          CGILIB now a self-contained object module,
                          bugfix; major/minor version
22-SEP-2000  MGD  v1.3.1, bugfix; refresh rates
17-JUN-2000  MGD  v1.3.0, file type defaults to .SAH (for SETI@home v2.4),
                          control number of displayed spikes,
                          a little JavaScript for when work units are loading
06-FEB-2000  MGD  v1.2.0, allow for SETI@home v2.n changed file names,
                          bugfix; use ACP-QIO to access locked OUTFILE
18-JAN-2000  MGD  v1.1.0, work-unit skymap using CSS,
                          correct 'declination' to degrees
09-JAN-2000  MGD  v1.0.0, initial (quick hack)
*/

/*****************************************************************************/

#define SOFTWAREVN "1.5.2"
#define SOFTWARENM "VMSETI"
#ifdef __ALPHA
#  define SOFTWAREID SOFTWARENM " AXP-" SOFTWAREVN
#endif
#ifdef __ia64
#  define SOFTWAREID SOFTWARENM " IA64-" SOFTWAREVN
#endif
#ifdef __VAX
   /* yup, I know SETI@home is not supported on VAX ;^) */
   /* but, there's no reason why VMSeti shouldn't!         */
#  define SOFTWAREID SOFTWARENM " VAX-" SOFTWAREVN
#endif

#define FI_LI __FILE__, __LINE__

#define VMSok(x) ((x) & STS$M_SUCCESS)
#define VMSnok(x) !(((x) & STS$M_SUCCESS))

#define MAX_SPIKES 100

/***************************/
/* standard C header files */
/***************************/

#include <ctype.h>
#include <errno.h>
#include <file.h>
#include <stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

/********************/
/* VMS header files */
/********************/

#include <atrdef.h>
#include <descrip.h>
#include <fabdef.h>
#include <fibdef.h>
#include <iodef.h>
#include <namdef.h>
#include <rmsdef.h>
#include <ssdef.h>
#include <stsdef.h>
#include <starlet.h>
#include <syidef.h>

#pragma member_alignment __save
#pragma nomember_alignment

#define FAT$C_UNDEFINED 0
#define FAT$C_FIXED     1
#define FAT$C_VARIABLE  2
#define FAT$C_VFC       3
#define FAT$C_STREAM    4
#define FAT$C_STREAMLF  5
#define FAT$C_STREAMCR  6
#define FAT$C_SEQUENTIAL 0
#define FAT$C_RELATIVE  1
#define FAT$C_INDEXED   2
#define FAT$C_DIRECT    3
#define FAT$M_FORTRANCC 0x1
#define FAT$M_IMPLIEDCC 0x2
#define FAT$M_PRINTCC   0x4
#define FAT$M_NOSPAN    0x8
#define FAT$M_MSBRCW    0x10

struct fatdef
{
   unsigned char fat$b_rtype;
   unsigned char fat$b_rattrib;
   unsigned short int fat$w_rsize;
   unsigned short  fat$w_hiblkh;
   unsigned short  fat$w_hiblkl;
   unsigned short  fat$w_efblkh;
   unsigned short  fat$w_efblkl;
   unsigned short int fat$w_ffbyte;
   unsigned char fat$b_bktsize;
   unsigned char fat$b_vfcsize;
   unsigned short int fat$w_maxrec;
   unsigned short int fat$w_defext;
   unsigned short int fat$w_gbc;
   char fat$b_fill[8];
   unsigned short int fat$versions;
};

/****************************/
/* application header files */
/****************************/

#include <cgilib.h>

/* contains the SKYMAP.GIF image as an array of char */
#include <skymap.h>

/* contains the SKYMAPX.GIF image as an array of char */
#include <skymapx.h>

/******************/
/* global storage */
/******************/

#pragma member_alignment __restore

/* try to access the WU file for 5 minutes (allows for data download!) */
#define WU_RETRY_COUNT 10

int  Debug,
     FftsPerformed,
     IsCgiPlus,
     /* if STATE.SAH is older that 10 minutes processing probably stopped! */
     ProcessingStaleSeconds = 600,
     SkyMapImageHeight = 482,
     SkyMapImageHeightAdjust = +2,
     SkyMapImageWidth = 965,
     SkyMapImageWidthAdjust = +2,
     SkyMapTopBorder = 20,
     SkyMapLeftBorder = 20,
     SkyMapXImageHeight = 40,
     SkyMapXImageWidth = 40,
     StartDecDeg,
     StartDecMin,
     StartDecSec,
     StartHrAgo,
     StartMinAgo,
     StartRaHr,
     StartRaMin,
     StartRaSec;

unsigned long  ProcessingSeconds,
               ProcessingStartTimeSecs;

float  CpuTime,
       GaussianPower,
       GaussianScore,
       PeakPower,
       PeakScore,
       PercentProgress,
       StartDec,
       StartRa,
       SubbandBaseGhz,
       TotalCpuTime;

char  *CgiContentTypePtr,
      *CgiEnvironmentPtr,
      *CgiFormColPtr,
      *CgiFormDecPtr,
      *CgiFormGifPtr,
      *CgiFormRaPtr,
      *CgiFormRefPtr,
      *CgiFormImgPtr,
      *CgiHttpHostPtr,
      *CgiRequestSchemePtr,
      *CgiRequestMethodPtr,
      *CgiPathTranslatedPtr,
      *CgiScriptNamePtr,
      *CgiServerPortPtr,
      *FileNamePtr,
      *FileTypePtr,
      *SkyMapPathPtr,
      *UserInfoFileNamePtr,
      *WarningStalePtr;

char  EmailAddr [128],
      FileName [256],
      LastResultTime [32],
      NumberOfResults [32],
      RegisterTime [32],
      SetiAtHomeVersion [32],
      SoftwareID [48],
      StartTime [32],
      SubbandBase [256],
      TimeRecorded [32],
      UserCountry [64],
      UserName [64];

char  ErrorSkyMapDec [] = "Skymap Dec problem.",
      ErrorSkyMapRa [] = "Skymap RA problem.",
      ErrorSkymapImage [] = "Unknown skymap image request!",
      ErrorSkyMapUnavailable [] = "Skymap is unavailable.",
      ErrorState [] = "Parsing state &nbsp;&quot;%s&quot;.",
      ErrorStateFile [] = "State file environment variable not assigned!\n",
      ErrorVersionFile [] = "Version file environment variable not assigned!",
      ErrorWork [] = "Parsing work unit &nbsp;&quot;%s&quot;.",
      ErrorWorkUnitFile [] = "Work unit file environment variable not assigned!\n",
      ErrorUser [] = "Parsing user information &nbsp;&quot;%s&quot;.",
      MetaAuthor [] = "mark.daniel@wasd.vsm.com.au",
      SetiAtHomeUrl [] = "http://setiathome.ssl.berkeley.edu/",
      VMSetiUrl [] = "http://wasd.vsm.com.au/wasd/";

/***********************/
/* function prototypes */
/***********************/

char* AcpReadFile (char*);
int ParseNumber (char*, void*, char*);
int ParseString (char*, char*, int, char*);
int ProcessRequest ();
int ReportPage ();

/*****************************************************************************/
/*
*/

main (int argc, char* argv[])
       
{
   /*********/
   /* begin */
   /*********/

   sprintf (SoftwareID, "%s (%s)", SOFTWAREID, CgiLibEnvironmentVersion());

   if (argc == 4 && *(unsigned long*)argv[1] == '-f2b')
      File2Bytes (argv[2], argv[3]);

   Debug = (getenv ("VMSETI$DBUG") != NULL);
   if (Debug) fprintf (stdout, "Content-Type: text/plain\n\n");
   CgiLibEnvironmentSetDebug (Debug);

   CgiLibEnvironmentInit (argc, argv, 0);

   CgiLibResponseSetSoftwareID (SoftwareID);
   CgiLibResponseSetErrorMessage ("Reported by VMSeti");

   CgiEnvironmentPtr = CgiLibEnvironmentName ();
   IsCgiPlus = CgiLibEnvironmentIsCgiPlus ();

   if (IsCgiPlus)
   {
      /* block waiting for the next request */
      if (Debug) fprintf (stdout, "Content-Type: text/plain\n\n");
      CgiLibVar ("");
      ProcessRequest ();
      CgiLibCgiPlusEOF ();
   }
   else
      ProcessRequest ();
}

/*****************************************************************************/
/*
Open the work unit file (mandatory), then the state file (mandatory), then the
user information file (optional) and retrieve required fields from each.
*/

ProcessRequest ()
       
{
   int  result,
        status,
        FlkCount,
        MajorVersion,
        MinorVersion;
   unsigned long  TimeSecs;
   float  ScratchFloat;
   char  *cptr;
   char  Line [256];
   FILE  *DataFile;
   stat_t  StatInfo;
   struct tm  *TmPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ProcessRequest()\n");

   CgiRequestMethodPtr = CgiLibVar ("WWW_REQUEST_METHOD");
   if (strcmp (CgiRequestMethodPtr, "GET"))
   {
      fprintf (stdout,
"Status: 501\n\
Content-Type: text/html\n\
\n\
ERROR: method \"%s\" not supported.\n",
               CgiRequestMethodPtr);
      return;
   }

   CgiHttpHostPtr = CgiLibVar ("WWW_HTTP_HOST");
   if (!*CgiHttpHostPtr) CgiHttpHostPtr = CgiLibVar ("WWW_SERVER_NAME");
   CgiScriptNamePtr = CgiLibVar ("WWW_SCRIPT_NAME");
   CgiPathTranslatedPtr = CgiLibVar ("WWW_PATH_TRANSLATED");

   SkyMapPathPtr = getenv ("VMSETI_SKYMAP");
   if (SkyMapPathPtr && !*SkyMapPathPtr) SkyMapPathPtr = NULL;

   CgiFormImgPtr = CgiLibVar ("WWW_FORM_IMG");
   if (CgiFormImgPtr[0]) SkymapImage (CgiFormImgPtr);

   CgiFormRaPtr = CgiLibVar ("WWW_FORM_RA");
   if (CgiFormRaPtr[0])
   {
      CgiFormDecPtr = CgiLibVar ("WWW_FORM_DEC");
      if (CgiFormDecPtr[0])
      {
         /**********/
         /* skymap */
         /**********/

         SkyMap ();
         return;
      }
   }

   if ((FileTypePtr = getenv("VMSETI_FILE_TYPE")) == NULL)
      FileTypePtr = ".SAH";

   /***********/
   /* version */
   /***********/

   if ((FileNamePtr = getenv("VMSETI_VERSION")) == NULL)
   {
      CgiLibResponseError (FI_LI, 0, ErrorVersionFile);
      return;
   }
   if (CgiPathTranslatedPtr[0])
      sprintf (FileNamePtr = FileName, "%sVERSION%s",
               CgiPathTranslatedPtr, FileTypePtr);
   if (Debug) fprintf (stdout, "FileNamePtr |%s|\n", FileNamePtr);
   if ((DataFile = fopen (FileNamePtr, "r", "shr=get", "shr=put")) == NULL)
   {
      CgiLibResponseError (FI_LI, vaxc$errno, FileNamePtr);
      return;
   }

   MajorVersion = MinorVersion = 0;
   while (fgets (Line, sizeof(Line), DataFile) != NULL)
   {
      if (Debug) fprintf (stdout, "Line |%s|\n", Line);
      if (!strncmp (Line, "major_version=", 14))
         MajorVersion = atoi(Line+14);
      if (!strncmp (Line, "minor_version=", 14))
         MinorVersion = atoi(Line+14);
   }
   sprintf (SetiAtHomeVersion, "%d.%0.02d", MajorVersion, MinorVersion);

   fclose (DataFile);

   /*************/
   /* work unit */
   /*************/

   if ((FileNamePtr = getenv("VMSETI_WORK_UNIT")) == NULL)
   {
      CgiLibResponseError (FI_LI, 0, ErrorWorkUnitFile);
      return;
   }
   if (CgiPathTranslatedPtr[0])
      sprintf (FileNamePtr = FileName, "%sWORK_UNIT%s",
               CgiPathTranslatedPtr, FileTypePtr);
   if (Debug) fprintf (stdout, "FileNamePtr |%s|\n", FileNamePtr);

   if ((DataFile = fopen (FileNamePtr, "r", "shr=get", "shr=put")) == NULL)
   {
      /*
         Work units are unavailable when loading from the SETI@home server.
         This slightly more elaborate error message provides for this by
         continually retrying (uses either JavaScript, if browser enabled,
         or falls back to a META refresh tag).
      */

      CgiLibResponseHeader (404, "text/html");
      fprintf (stdout,
"<HTML>\n\
<HEAD>\n\
<META HTTP-EQUIV=\"Refresh\" CONTENT=\"35\">\n\
<META NAME=\"generator\" CONTENT=\"%s\">\n\
<META NAME=\"module\" CONTENT=\"VMSETI\">\n\
<META NAME=\"line\" CONTENT=\"%d\">\n\
<META NAME=\"environment\" CONTENT=\"%s\">\n\
<TITLE>Work unit currently not available!</TITLE>\n\
<SCRIPT LANGUAGE=\"JavaScript\">\n\
<!--\n\
var retrySeconds = 30;\n\
function retryVMSeti () {\n\
   if (--retrySeconds > 0) {\n\
      window.defaultStatus = \"Retry in \" + retrySeconds + \" seconds.\";\n\
      setTimeout(\"retryVMSeti()\", 1000);\n\
      return;\n\
   }\n\
   location.reload();\n\
}\n\
// -->\n\
</SCRIPT>\n\
</HEAD>\n\
<BODY ONLOAD=\"retryVMSeti()\">\n\
<FONT SIZE=+1><B>VMSeti</B> &nbsp;-&nbsp; \
Work unit &quot;%s&quot; currently not available.</FONT>\n\
<P>It may be loading from the SETI@home site.\n\
<BR>Will try again every thirty seconds (see status bar below).\n",
         SoftwareID, __LINE__,
         CgiLibEnvironmentName(),
         FileNamePtr);

      cptr = CgiLib__GetVar ("WWW_SERVER_SIGNATURE", "");
      if (cptr[0])
         fprintf (stdout, "<P><HR WIDTH=85%% ALIGN=left SIZE=2 NOSHADE>\n%s",
                  cptr);

      fprintf (stdout,
"</BODY>\n\
</HTML>\n");

      return;
   }

   fstat (fileno(DataFile), &StatInfo);
   ProcessingStartTimeSecs = StatInfo.st_mtime;
   TmPtr = localtime (&ProcessingStartTimeSecs);
   result = strftime (StartTime, sizeof(StartTime), "%a %b %d %T %Y", TmPtr);
   if (!result) sprintf (StartTime, "ERROR: strftime()\n");
   time (&TimeSecs);
   ProcessingSeconds = (TimeSecs -= ProcessingStartTimeSecs);
   StartHrAgo = TimeSecs / 3600;
   StartMinAgo = (TimeSecs - (StartHrAgo * 3600)) / 60;

   TimeRecorded[0] = '\0';
   SubbandBaseGhz = 0.0;
   StartDecDeg = StartDecMin = StartDecSec =
      StartRaHr = StartRaMin = StartRaSec = 0;

   while (fgets (Line, sizeof(Line), DataFile) != NULL)
   {
      if (Debug) fprintf (stdout, "Line |%s|\n", Line);

      if (!strncmp (Line, "end_seti_header", 15)) break;

      if (!strncmp (Line, "time_recorded=", 14))
         if (!ParseString (Line+14, TimeRecorded, sizeof(TimeRecorded), "()"))
            return (CgiLibResponseError (FI_LI, 0, ErrorWork, Line));
         else
            continue;

      if (!strncmp (Line, "subband_base=", 13))
         if (!ParseNumber (Line+13, &SubbandBaseGhz, "%f"))
            return (CgiLibResponseError (FI_LI, 0, ErrorWork, Line));
         else
         {
            SubbandBaseGhz = SubbandBaseGhz / 1000000000.0;
            continue;
         }

      if (!strncmp (Line, "start_ra=", 9))
         if (!ParseNumber (Line+9, &StartRa, "%f"))
            return (CgiLibResponseError (FI_LI, 0, ErrorWork, Line));
         else
         {
            ScratchFloat = StartRa;
            StartRaHr = (int)ScratchFloat;
            ScratchFloat -= (float)StartRaHr;
            StartRaMin = (int)(60.0 * ScratchFloat);
            ScratchFloat -= (float)StartRaMin / 60.0;
            StartRaSec = (int)(3660.0 * ScratchFloat);
            continue;
         }

      if (!strncmp (Line, "start_dec=", 10))
         if (!ParseNumber (Line+10, &StartDec, "%f"))
            return (CgiLibResponseError (FI_LI, 0, ErrorWork, Line));
         else
         {
            ScratchFloat = StartDec;
            StartDecDeg = (int)ScratchFloat;
            ScratchFloat -= (float)StartDecDeg;
            StartDecMin = (int)(60.0 * ScratchFloat);
            ScratchFloat -= (float)StartDecMin / 60.0;
            StartDecSec = (int)(3660.0 * ScratchFloat);
            continue;
         }
   }

   fclose (DataFile);

   /*****************/
   /* current state */
   /*****************/

   if ((FileNamePtr = getenv("VMSETI_STATE")) == NULL)
   {
      CgiLibResponseError (FI_LI, 0, ErrorStateFile);
      return;
   }
   if (CgiPathTranslatedPtr[0])
      sprintf (FileNamePtr = FileName, "%sSTATE%s",
               CgiPathTranslatedPtr, FileTypePtr);
   if (Debug) fprintf (stdout, "FileNamePtr |%s|\n", FileNamePtr);
   FlkCount = 0;
   for (;;)
   {
      /* occasionally state file will be open/locked at access, try again */
      if ((DataFile = fopen (FileNamePtr, "r", "shr=get", "shr=put")) == NULL)
      {
         if ((status = vaxc$errno) != RMS$_FLK || FlkCount > 5)
         {
            CgiLibResponseError (FI_LI, status, FileNamePtr);
            return;
         }
         FlkCount++;
         sleep (2);
      }
      break;
   }

   fstat (fileno(DataFile), &StatInfo);
   time (&TimeSecs);
   TimeSecs -= StatInfo.st_mtime;
   if (TimeSecs > ProcessingStaleSeconds)
   {
      WarningStalePtr =
"<TR><TH ALIGN=right><FONT COLOR=\"#ff0000\" SIZE=+1>WARNING:&nbsp;</FONT></TH>\
<TD><FONT COLOR=\"#ff0000\">Timestamps indicate processing \
may have stopped!</FONT></TD></TR>\n";
   }
   else
      WarningStalePtr = "";

   FftsPerformed = 0;
   CpuTime = GaussianPower = GaussianScore =
      PeakPower = PeakScore = PercentProgress = 0.0;

   while (fgets (Line, sizeof(Line), DataFile) != NULL)
   {
      if (Debug) fprintf (stdout, "Line |%s|\n", Line);

      if (!strncmp (Line, "bs_power=", 9))
         if (!ParseNumber (Line+9, &PeakPower, "%f"))
            return (CgiLibResponseError (FI_LI, 0, ErrorState, Line));
         else
            continue;

      if (!strncmp (Line, "bs_score=", 9))
         if (!ParseNumber (Line+9, &PeakScore, "%f"))
            return (CgiLibResponseError (FI_LI, 0, ErrorState, Line));
         else
            continue;

      if (!strncmp (Line, "cpu=", 4))
         if (!ParseNumber (Line+4, &CpuTime, "%f"))
            return (CgiLibResponseError (FI_LI, 0, ErrorState, Line));
         else
            continue;

      if (!strncmp (Line, "bg_power=", 9))
         if (!ParseNumber (Line+9, &GaussianPower, "%f"))
            return (CgiLibResponseError (FI_LI, 0, ErrorState, Line));
         else
            continue;

      if (!strncmp (Line, "bg_score=", 9))
         if (!ParseNumber (Line+9, &GaussianScore, "%f"))
            return (CgiLibResponseError (FI_LI, 0, ErrorState, Line));
         else
            continue;

      if (!strncmp (Line, "ncfft=", 6))
         if (!ParseNumber (Line+6, &FftsPerformed, "%d"))
            return (CgiLibResponseError (FI_LI, 0, ErrorState, Line));
         else
            continue;

      if (!strncmp (Line, "prog=", 5))
         if (!ParseNumber (Line+5, &PercentProgress, "%f"))
            return (CgiLibResponseError (FI_LI, 0, ErrorState, Line));
         else
         {
            PercentProgress *= 100.0;
            continue;
         }
   }

   fclose (DataFile);

   /*************/
   /* user info */
   /*************/

   if ((UserInfoFileNamePtr = getenv("VMSETI_USER_INFO")) != NULL)
   {
      FileNamePtr = UserInfoFileNamePtr;
      if (CgiPathTranslatedPtr[0])
         sprintf (FileNamePtr = FileName, "%sUSER_INFO%s",
                  CgiPathTranslatedPtr, FileTypePtr);
      if (Debug) fprintf (stdout, "FileNamePtr |%s|\n", FileNamePtr);
      if ((DataFile = fopen (FileNamePtr, "r", "shr=get", "shr=put")) == NULL)
      {
         CgiLibResponseError (FI_LI, vaxc$errno, FileNamePtr);
         return;
      }

      TotalCpuTime = 0.0;
      EmailAddr[0] = LastResultTime[0] = NumberOfResults[0] =
         RegisterTime[0] = UserCountry[0] = UserName[0] = '\0';

      while (fgets (Line, sizeof(Line), DataFile) != NULL)
      {
         if (Debug) fprintf (stdout, "Line |%s|\n", Line);

         if (!strncmp (Line, "country=", 8) && (strlen (Line) > 9))
            if (!ParseString (Line+8, UserCountry, sizeof(UserCountry), "\0\n"))
               return (CgiLibResponseError (FI_LI, 0, ErrorUser, Line));
            else
               continue;

         if (!strncmp (Line, "email_addr=", 11) && (strlen(Line) > 12))
            if (!ParseString (Line+11, EmailAddr, sizeof(EmailAddr), "\0\n"))
               return (CgiLibResponseError (FI_LI, 0, ErrorUser, Line));
            else
               continue;

         if (!strncmp (Line, "register_time=", 14))
            if (!ParseString (Line+14, RegisterTime, sizeof(RegisterTime), "()"))
               return (CgiLibResponseError (FI_LI, 0, ErrorUser, Line));
            else
               continue;

         if (!strncmp (Line, "last_result_time=", 17))
            if (!ParseString (Line+17, LastResultTime, sizeof(LastResultTime), "()"))
               return (CgiLibResponseError (FI_LI, 0, ErrorUser, Line));
            else
               continue;

         if (!strncmp (Line, "nresults=", 9))
            if (!ParseString (Line+9, NumberOfResults, sizeof(NumberOfResults), "\0\n"))
               return (CgiLibResponseError (FI_LI, 0, ErrorUser, Line));
            else
               continue;

         if (!strncmp (Line, "total_cpu=", 10))
            if (!ParseNumber (Line+10, &TotalCpuTime, "%f"))
               return (CgiLibResponseError (FI_LI, 0, ErrorUser, Line));
            else
               continue;

         if (!strncmp (Line, "name=", 5))
            if (!ParseString (Line+5, UserName, sizeof(UserName), "\0\n"))
               return (CgiLibResponseError (FI_LI, 0, ErrorUser, Line));
            else
               continue;
      }

      fclose (DataFile);
   }

   /****************/
   /* display page */
   /****************/

   ReportPage ();
}

/*****************************************************************************/
/*
Get a string from a string.  'Constraints' of "()" gets a time string (actually
any first parenthesis-delimited string), or "\0\n" gets the rest of the line.
*/

ParseString
(
char *String,
char *ResultString,
int SizeOfResultString,
char *Constraints
)
{
   char  *cptr, *sptr, *zptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ParseString() |%s|\n", String);

   ResultString[0] = '\0';
   if (Constraints[0] == '(')
   {
      for (cptr = String; *cptr != '('; cptr++);
      if (*cptr) cptr++;
   }
   else
      for (cptr = String; isspace(*cptr); cptr++);
   if (!*cptr) return (0);
   zptr = (sptr = ResultString) + SizeOfResultString-1;
   while (*cptr && *cptr != Constraints[1] && sptr < zptr) *sptr++ = *cptr++;
   *sptr = '\0';
   if (Debug) fprintf (stdout, "ResultString |%s|\n", ResultString);
   return (1);
}

/*****************************************************************************/
/*
Get a number from a string.  'FormatString' of "%f" gets a float, or "%d" gets
an integer.
*/

ParseNumber
(
char *String,
void *StoragePtr,
char *FormatString
)
{
   char  *cptr;
   int  result;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ParseNumber() |%s|%s|\n", String, FormatString);

   for (cptr = String; isspace(*cptr); cptr++);
   if (!*cptr) return (0);
   if (*(unsigned short*)FormatString == '%d')
   {
      result = sscanf (cptr, FormatString, (int*)StoragePtr);
      if (Debug) fprintf (stdout, "Number: %d\n", *(int*)StoragePtr);
      return (result);
   }
   else
   if (*(unsigned short*)FormatString == '%f')
   {
      result = sscanf (cptr, FormatString, (float*)StoragePtr);
      if (Debug) fprintf (stdout, "Number: %f\n", *(float*)StoragePtr);
      return (result);
   }
   else
      return (0);
}

/*****************************************************************************/
/*
Generate the HTML page.
*/

ReportPage ()
       
{
   int  result,
        status,
        EstHrTotal,
        EstMinTotal,
        PercentGaussian,
        PercentDone,
        PercentLeft,
        PercentPeak,
        RefreshMinutes,
        ServerPort;

   unsigned long  EstCompletionTimeSecs,
                  EstProcessingSeconds,
                  TimeSecs;

   short  SyiHwNameLength;

   char  *cptr,
         *extPtr,
         *GetEnvPtr,
         *LocalTextPtr;

   char  BarDone [128],
         BarRemaining [128],
         CurrentTime [32],
         EstimatedCompletionTime [64],
         FftRate [32],
         Line [256],
         SyiHwName [32],
         SyiVersion [9];

   FILE  *LocalFile;
   struct tm  *TmPtr;

   struct {
      short  buf_len;
      short  item;
      void  *buf_addr;
      short  *ret_len;
   } SyiItem [] = {
     { sizeof(SyiVersion)-1, SYI$_VERSION, SyiVersion, 0 },
     { sizeof(SyiHwName)-1, SYI$_HW_NAME, SyiHwName, &SyiHwNameLength },
     { 0,0,0,0 }
   };


   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ReportPage()\n");

   time (&TimeSecs);
   TmPtr = localtime (&TimeSecs);
   result = strftime (CurrentTime, sizeof(CurrentTime),
                      "%a %b %d %T %Y", TmPtr);
   if (!result) sprintf (CurrentTime, "ERROR: strftime()\n");

   CgiFormColPtr = CgiLibVar ("WWW_FORM_COL");
   CgiFormRefPtr = CgiLibVar ("WWW_FORM_REF");
   if (CgiFormRefPtr[0])
      RefreshMinutes = atoi(CgiFormRefPtr);
   else
      RefreshMinutes = 0;

   if (!((status = sys$getsyi (0, 0, 0, &SyiItem, 0, 0, 0)) & 1))
      exit (status);

   SyiHwName[SyiHwNameLength] = '\0';
   SyiVersion[sizeof(SyiVersion)-1] = '\0';
   for (cptr = SyiVersion; *cptr && *cptr != ' '; cptr++);
   *cptr = '\0';

   if (PercentDone = (int)PercentProgress)
      sprintf (BarDone, "<TD WIDTH=%d%% BGCOLOR=\"#cc0000\">&nbsp;</TD>",
               PercentDone);
   else
      BarDone[0] = '\0';
   if (PercentLeft = 100 - PercentDone)
      sprintf (BarRemaining, "<TD WIDTH=%d%% BGCOLOR=\"#cccccc\">&nbsp;</TD>",
               PercentLeft);
   else
      BarRemaining[0] = '\0';

   if (PercentProgress > 5.0)
   {
      EstProcessingSeconds =
         (int)((float)ProcessingSeconds * 100.0 / PercentProgress);

      EstHrTotal = EstProcessingSeconds / 3600;
      EstMinTotal = (EstProcessingSeconds - (EstHrTotal * 3600)) / 60;
      EstCompletionTimeSecs = ProcessingStartTimeSecs + EstProcessingSeconds;

      TmPtr = localtime (&EstCompletionTimeSecs);
      result = strftime (EstimatedCompletionTime,
                         sizeof(EstimatedCompletionTime),
                         "%a %b %d %H:%M", TmPtr);
      sprintf (EstimatedCompletionTime+result,
               "&nbsp; (%d hr %d min)", EstHrTotal, EstMinTotal);
      if (!result) sprintf (EstimatedCompletionTime, "ERROR: strftime()\n");

      if ((float)FftsPerformed / (float)ProcessingSeconds < 1.0)
         sprintf (FftRate, "&nbsp; (%.1f Sec/FFT)",
                  (float)ProcessingSeconds / (float)FftsPerformed);
      else
         sprintf (FftRate, "&nbsp; (%.1f FFTs/Sec)",
                  (float)FftsPerformed / (float)ProcessingSeconds);
   }
   else
   {
      EstHrTotal = EstMinTotal = EstProcessingSeconds = 0;
      strcpy (EstimatedCompletionTime, "<I>(too early to determine)</I>");
      FftRate[0] = '\0';
   }

   CgiLibResponseHeader (200, "text/html");

   fprintf (stdout,
"<HTML>\n\
<HEAD>\n\
<META NAME=\"generator\" CONTENT=\"%s\">\n\
<META NAME=\"environment\" CONTENT=\"%s\">\n\
<META NAME=\"author\" CONTENT=\"%s\">\n\
<TITLE>VMSeti @ %s</TITLE>\n\
</HEAD>\n",
      SoftwareID,
      CgiEnvironmentPtr,
      MetaAuthor,
      CgiHttpHostPtr);

   if (RefreshMinutes)
   {
      fprintf (stdout,
"<SCRIPT LANGUAGE=\"JavaScript\">\n\
<!--\n\
\
var refreshPageMinutes = %d;\n\
\
function refreshPage () {\n\
   if (refreshPageMinutes > 0) {\n\
      window.defaultStatus = \"refresh VMSeti in \" + \
refreshPageMinutes + \" minutes\";\n\
      refreshPageMinutes--;\n\
      setTimeout(\"refreshPage()\", 60000);\n\
      return true;\n\
   }\n\
   location.reload(true);\n\
}\n\
\
// -->\n\
</SCRIPT>\n",
         RefreshMinutes);
   }

   fprintf (stdout,
"<BODY%s %s>\n\
<CENTER>\n\
<FONT SIZE=+2><B><A HREF=\"%s\">SETI@home</A></B>@%s</FONT></U>\n\
<TABLE CELLPADDING=0 CELLSPACING=0 BORDER=0 WIDTH=90%%>\n\
<TR><TD ALIGN=center>\n\
<TABLE CELLPADDING=0 CELLSPACING=1 BORDER=0 WIDTH=90%%>\n\
\
<TR><TD>&nbsp;</TD></TR>\n\
<TR><TH ALIGN=right>Recorded:&nbsp;</TH><TD>%s</TD></TR>\n\
<TR><TH ALIGN=right>Base Frequency:&nbsp;</TH><TD>%f GHz</TD></TR>\n\
<TR><TH ALIGN=right>Celestial:&nbsp;</TH>\
<TD><NOBR>%d hr %d min %d sec RA, %d deg %d min %d sec Dec\
&nbsp;&nbsp(<A HREF=\"%s?ra=%f&dec=%f\">skymap</A>)</NOBR></TD></TR>\n\
\
<TR><TD>&nbsp;</TD></TR>\n\
<TR><TH ALIGN=right>As At:&nbsp;</TH><TD>%s</TD></TR>\n\
<TR><TH ALIGN=right>Processed:&nbsp;</TH><TD>%.2f%%</TD></TR>\n\
<TR><TD COLSPAN=2>\n\
<TABLE CELLPADDING=2 CELLSPACING=0 BORDER=0 WIDTH=100%%>\n\
<TR><TD BGCOLOR=\"#000000\" ALIGN=center>\n\
<TABLE CELLPADDING=0 CELLSPACING=0 BORDER=0 WIDTH=100%%>\n\
<TR>%s%s</TR>\n\
</TABLE>\n\
</TD></TR>\n\
</TABLE>\n\
</TD></TR>\n\
<TR><TD>\n\
%s\
<TR><TD>&nbsp;</TD></TR>\n\
\
<TR><TH ALIGN=right>Version:&nbsp;</TH><TD>%s</TD></TR>\n\
<TR><TH ALIGN=right>Platform:&nbsp;</TH><TD>%s, VMS %s</TD></TR>\n\
<TR><TH ALIGN=right>Started:&nbsp;</TH><TD>%s</TD></TR>\n\
<TR><TH ALIGN=right>~Completion:&nbsp;</TH><TD>%s</TD></TR>\n\
<TR><TH ALIGN=right>Elapsed Time:&nbsp;</TH><TD>%d hr %d min</TD></TR>\n\
<TR><TH ALIGN=right>CPU Time:&nbsp;</TH><TD>%d hr %d min</TD></TR>\n\
<TR><TH ALIGN=right>FFTs Performed:&nbsp;</TH><TD>%d%s</TD></TR>\n",
      RefreshMinutes ? " onLoad=\"refreshPage()\"" : "",
      CgiFormColPtr[0] == 'p' ?
"BGCOLOR=\"#ffffff\" TEXT=\"#000000\" LINK=\"#0000cc\" VLINK=\"#0000cc\"" :
"BGCOLOR=\"#000000\" TEXT=\"#ffffff\" LINK=\"#cc99ff\" VLINK=\"#cc99ff\"",
      SetiAtHomeUrl,
      CgiHttpHostPtr,
      TimeRecorded,
      SubbandBaseGhz,
      StartRaHr, StartRaMin, StartRaSec,
      StartDecDeg, StartDecMin, StartDecSec,
      CgiScriptNamePtr, StartRa, StartDec,
      CurrentTime,
      PercentProgress,
      BarDone, BarRemaining,
      WarningStalePtr,
      SetiAtHomeVersion,
      SyiHwName, SyiVersion,
      StartTime,
      EstimatedCompletionTime,
      StartHrAgo, StartMinAgo,
      (int)(CpuTime / 3600.0),
      (int)(((int)CpuTime % 3600) / 60),
      FftsPerformed, FftRate);

   if (PeakPower != 0.0)
   {
      if (GaussianPower != 0.0)
         PercentGaussian = (int)(GaussianPower * 100.0 / PeakPower);
      else
         PercentGaussian = 0;
      PercentPeak = (int)(PeakPower * 100.0 / PeakPower) - PercentGaussian;
   }
   else
      PercentGaussian = PercentPeak = 0;

   fprintf (stdout,
"<TR><TD>&nbsp;</TD></TR>\n\
<TR><TH ALIGN=right>Strongest Power:&nbsp;</TH><TD>%.2f</TD></TR>\n\
<TR><TH ALIGN=right>Strongest Gaussian:&nbsp;</TH><TD>%.2f</TD></TR>\n",
         PeakPower,
         GaussianPower);

   ReportSpikes ();

   fprintf (stdout,
"</TABLE>\n\
</TD></TR>\n\
</TABLE>\n\
</CENTER>\n\
<BR><BR>\n");

   if ((LocalTextPtr = getenv("VMSETI_LOCAL")) != NULL)
   {
      if (LocalTextPtr[0] == '@')
      {
         if ((LocalFile = fopen (LocalTextPtr+1, "r")) == NULL)
            fprintf (stdout,
"<P><FONT COLOR=\"#ff0000\">ERROR opening %s, %%X%08.08X</FONT>\n",
                     LocalTextPtr+1, vaxc$errno);
         else
         {
            while (fgets (Line, sizeof(Line), LocalFile) != NULL)
               fputs (Line, stdout);
            fclose (LocalFile);
         }
      }
      else
         fputs (LocalTextPtr, stdout);
   }

   if (UserInfoFileNamePtr != NULL)
   {
      fprintf (stdout,
"Name:&nbsp;%s\n\
<BR>Email:&nbsp;<A HREF=\"mailto:%s\">%s</A>\n\
<BR>Country:&nbsp;%s\n\
<BR>Total Results:&nbsp;%s\n\
<BR>Total CPU:&nbsp;%d day %d hr %d min\n\
<BR>Last Returned:&nbsp;%s <FONT SIZE=-1><I>(Berkeley time)</I></FONT>\n\
<P>",
         UserName,
         EmailAddr, EmailAddr,
         UserCountry,
         NumberOfResults,
         (int)(TotalCpuTime / 86400.0),
         (int)(((int)TotalCpuTime % 86400) / 3600),
         (int)(((int)TotalCpuTime % 3600) / 60),
         LastResultTime);
   }
   else
      fprintf (stdout, "<P>");

   CgiServerPortPtr = CgiLibVar ("WWW_SERVER_PORT");
   ServerPort = atoi(CgiServerPortPtr);
   CgiRequestSchemePtr = CgiLibVar ("WWW_REQUEST_SCHEME");
   if (!CgiRequestSchemePtr[0]) CgiRequestSchemePtr = "http:";

   /* yes, I'm ashamed of myself ... but most users will probably be in .us! */
   fprintf (stdout,
"Colors: <A HREF=\"%s?col=%s&ref=%d\">%s</A>\n\
<BR>Refresh: \
<A HREF=\"%s?col=%s&ref=5\">5</A>, \
<A HREF=\"%s?col=%s&ref=15\">15</A>, \
<A HREF=\"%s?col=%s&ref=60\">60</A> \
minutes\n\
<BR>SETI@home: <A HREF=\"%s\">%s</A>\n\
<BR>VMSeti script: <A HREF=\"%s\">%s</A>\n\
</BODY>\n\
</HTML>\n",
      /* change colour scheme link */
      CgiScriptNamePtr,
      CgiFormColPtr[0] == 'p' ? "SETI" : "printable",
      RefreshMinutes,
      CgiFormColPtr[0] == 'p' ? "SETI" : "printable",
      /* 5 minutes refresh link */
      CgiScriptNamePtr, CgiFormColPtr,
      /* 15 minutes refresh link */
      CgiScriptNamePtr, CgiFormColPtr,
      /* 60 minutes refresh link */
      CgiScriptNamePtr, CgiFormColPtr,
      /* other links */
      SetiAtHomeUrl, SetiAtHomeUrl,
      VMSetiUrl, VMSetiUrl);
}

/*****************************************************************************/
/*
Read the OUTFILE records to get and "plot" the spikes and gaussians of
interest.
*/

ReportSpikes ()

{
   int  idx,
        result,
        status,
        FftCount,
        PercentLeft,
        PercentSpike,
        SpikeCount;
   float  SpikePower;
   char  SpikeText [128];
   char  *cptr, *sptr,
         *BufferPtr,
         *BarColourPtr,
         *SpikeLeftPtr,
         *SpikeRightPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ReportSpikes()\n");

   if ((FileNamePtr = getenv("VMSETI_OUTFILE")) == NULL)
      return;
   if (CgiPathTranslatedPtr[0])
      sprintf (FileNamePtr = FileName, "%sOUTFILE%s",
               CgiPathTranslatedPtr, FileTypePtr);
   if (Debug) fprintf (stdout, "FileNamePtr |%s|\n", FileNamePtr);

   if ((sptr = BufferPtr = AcpReadFile (FileNamePtr)) == NULL)
   {
      fprintf (stdout,
"<TR><TH></TH><TD ALIGN=right><I>(unable to report on spikes)</I></TD></TR>\n");
      return;
   }

   SpikeCount = 0;
   while (*sptr)
   {
      /* parse a newline-delimited line out at a time, pointed to by 'cptr' */
      cptr = sptr;
      while (*sptr && *sptr != '\n') sptr++;
      if (*sptr) *sptr++ = '\0';
      if (Debug) fprintf (stdout, "cptr |%s|\n", cptr);

      if (!strncmp (cptr, "<ogh", 4))
      {
         if ((cptr = strstr (cptr, "ncfft=")) == NULL) continue;
         for (cptr += 6; *cptr && isspace(*cptr); cptr++);
         result = sscanf (cptr, "%d", &FftCount);
         if (result != 1) FftCount = -1;
         FftCount++;
         continue;
      }

      if (!strncmp (cptr, "spike:", 6))
      {
         if ((cptr = strstr (cptr, "power=")) == NULL) continue;
         for (cptr += 6; *cptr && isspace(*cptr); cptr++);
         result = sscanf (cptr, "%f", &SpikePower);
         if (result != 1) continue;
         BarColourPtr = "#0000cc";
         if (Debug) fprintf (stdout, "SpikePower: %f\n", SpikePower);
      }
      else
      if (!strncmp (cptr, "gaussian:", 9))
      {
         if ((cptr = strstr (cptr, "peak=")) == NULL) continue;
         for (cptr += 5; *cptr && isspace(*cptr); cptr++);
         result = sscanf (cptr, "%f", &SpikePower);
         if (result != 1) continue;
         BarColourPtr = "#cccc00";
         if (Debug) fprintf (stdout, "SpikePower: %f\n", SpikePower);
      }
      else
         continue;

      SpikeCount++;
      if (SpikeCount >= MAX_SPIKES) continue;

      /* this may happen ... I don't know, just ignore it */
      if (SpikePower > PeakPower) SpikePower = PeakPower;

      if (SpikeCount == 1)
         fprintf (stdout,
"<TR><TD COLSPAN=2>\n\
<TABLE CELLPADDING=1 CELLSPACING=0 BORDER=0 WIDTH=100%%>\n\
<TR><TD BGCOLOR=\"#000000\">\n");

      if (PercentSpike = (int)(SpikePower * 100.0 / PeakPower))
      {
         fprintf (stdout,
"<TABLE CELLPADDING=0 CELLSPACING=1 BORDER=0 WIDTH=100%%>\n\
<TR>");

         sprintf (SpikeText,
"&nbsp;<FONT SIZE=-1 COLOR=\"#ffffff\"><I>%.2f&nbsp;(%d)</I></FONT>",
                  SpikePower, FftCount);

         if (PercentSpike > 15)
         {
            SpikeLeftPtr = SpikeText;
            if (PercentSpike == 100)
               SpikeRightPtr = "";
            else
               SpikeRightPtr = "&nbsp;";
         }
         else
         {
            if (PercentSpike == 0)
               SpikeLeftPtr = "";
            else
               SpikeLeftPtr = "&nbsp;";
            SpikeRightPtr = SpikeText;
         }

         if (PercentSpike > 0)
            fprintf (stdout,
"<TD WIDTH=%d%% BGCOLOR=\"%s\" ALIGN=left>%s</TD>",
                     PercentSpike, BarColourPtr, SpikeLeftPtr);

         if (PercentSpike < 100)
            fprintf (stdout,
"<TD WIDTH=%d%% BGCOLOR=\"#cccccc\" ALIGN=left>%s</TD>",
                     100-PercentSpike, SpikeRightPtr);

         fprintf (stdout,
"</TR>\n\
</TABLE>\n");
      }
   }

   if (SpikeCount)
   {
      fprintf (stdout,
"</TD></TR>\n\
</TABLE>\n\
</TD></TR>\n");

      if (SpikeCount > MAX_SPIKES)
         fprintf (stdout,
"<TR><TH></TH><TD ALIGN=right><B>%d spikes not displayed \
(looks like a unusually large number!)</B></FONT></TD></TR>\n",
                  SpikeCount - MAX_SPIKES);
   }
   else
      fprintf (stdout,
"<TR><TH></TH><TD><I>(no spikes to report)</I></TD></TR>\n");

   free (BufferPtr);
}

/*****************************************************************************/
/*
At least for VMS SETI@home v1.3 the OUTFILE is always open and locked to
RMS.  This function uses the ACP QIO interface and SYSPRV to bypass that
locking and read the contents.  Returns a NULL pointer to indicate a problem in
accessing the file (e.g. does not exist, no SYSPRV), a pointer to "" if the
file is empty, or a pointer to a dynamically allocated buffer containing the
stream-LF contents of the file.  This must be parsed a newline-delimited line
at a time by the calling function and the buffer should be free()ed when
finished with.
*/

char* AcpReadFile (char *FileName)

{
#define BLOCK_SIZE 512
#define BLOCK_READ_MAX 64

   int  status,
        bnum,
        EofBlock;
   unsigned short  AcpChannel;
   char  *bptr,
         *BufferPtr;
   char  ExpFileName [256],
         ResFileName [256];
   struct FAB  FileFab;
   struct NAM  FileNam;
   struct fibdef  FileFib; 
   struct fatdef  FileFat;
   $DESCRIPTOR (DeviceDsc, "");
   $DESCRIPTOR (FileFibDsc, "");
   struct {
      unsigned short  Status;
      unsigned short  Count;
      int  Unused;
   } IOsb;
   struct atrdef  FileAtr [] =
   {
#ifndef __VAX
      { sizeof(FileFat), ATR$C_RECATTR, (void*)&FileFat },
#else
      { sizeof(FileFat), ATR$C_RECATTR, (unsigned int)&FileFat },
#endif
      { 0, 0, 0 }
   };

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AcpReadFile()\n");

   /* parse and search for the file to fill out the DID and FID fields */
   FileFab = cc$rms_fab;
   FileFab.fab$l_fna = FileName;
   FileFab.fab$b_fns = strlen(FileName);
   FileFab.fab$l_nam = &FileNam;

   FileNam = cc$rms_nam;
   FileNam.nam$l_esa = ExpFileName;
   FileNam.nam$b_ess = sizeof(ExpFileName)-1;
   FileNam.nam$l_rsa = ResFileName;
   FileNam.nam$b_rss = sizeof(ResFileName)-1;

   status = sys$parse (&FileFab, 0, 0);
   if (Debug) fprintf (stdout, "sys$parse() %%X%08.08X\n", status);
   if (VMSnok (status)) return (NULL);

   status = sys$search (&FileFab, 0, 0);
   if (Debug) fprintf (stdout, "sys$search() %%X%08.08X\n", status);
   if (VMSnok (status)) return (NULL);

   /* allocate a channel to the device */
   DeviceDsc.dsc$a_pointer = FileNam.nam$l_dev;
   DeviceDsc.dsc$w_length = FileNam.nam$b_dev;

   status = sys$assign (&DeviceDsc, &AcpChannel, 0, 0);
   if (Debug) fprintf (stdout, "sys$assign() %%X%08.08X\n", status);
   if (VMSnok (status)) exit (status);

   /* fill out the FIB, override any exclusive locking (requires SYSPRV) */
   memset (&FileFib, 0, sizeof(FileFib));
   FileFib.fib$l_acctl = FIB$M_SEQONLY | FIB$M_NOLOCK;
   FileFib.fib$w_nmctl = FIB$M_FINDFID;
   memcpy(FileFib.fib$w_did, &FileNam.nam$w_did[0], 6);
   memcpy(FileFib.fib$w_fid, &FileNam.nam$w_fid[0], 6);

   FileFibDsc.dsc$a_pointer = (char*)&FileFib;
   FileFibDsc.dsc$w_length = sizeof(FileFib);

   /* open a channel to this file */
   status = sys$qiow (0, AcpChannel, IO$_ACCESS | IO$M_ACCESS, &IOsb, 0, 0,
                      &FileFibDsc, 0, 0, 0, &FileAtr, 0);
   if (Debug)
      fprintf (stdout, "sys$qiow() %%X%08.08X IOsb: %%X%08.08X\n",
               status, IOsb.Status);
   if (VMSok (status) && VMSnok (IOsb.Status)) status = IOsb.Status;
   if (VMSnok (status) && status != SS$_ACCONFLICT) exit (status);

   if (FileFat.fat$b_rtype != FAT$C_STREAM &&
       FileFat.fat$b_rtype != FAT$C_STREAMLF &&
       FileFat.fat$b_rtype != FAT$C_STREAMCR &&
       FileFat.fat$b_rtype != FAT$C_UNDEFINED)
   {
      /* this function is only designed to handle stream(-LF) format files */
      if (Debug) fprintf (stdout, "fat$b_rtype: %d", FileFat.fat$b_rtype); 
      sys$dassgn (AcpChannel);
      return (NULL);
   }

   EofBlock = FileFat.fat$w_efblkl | (FileFat.fat$w_efblkh << 16);
   if (Debug) fprintf (stdout, "EofBlock: %d\n", EofBlock);

   if (!EofBlock)
   {
      /* empty file */
      sys$dassgn (AcpChannel);
      return ("");
   }

   /* allocate sufficient buffer space for the full size of the file */
   BufferPtr = calloc (1, (EofBlock * BLOCK_SIZE) + 1);
   if (BufferPtr == NULL) exit (vaxc$errno);

   /* fill that buffer space, maximum transfer at a time if necessary */
   bptr = BufferPtr;
   bnum = 1;
   while (bnum <= EofBlock)
   {
      if (Debug) fprintf (stdout, "bnum: %d\n", bnum);
      status = sys$qiow (0, AcpChannel, IO$_READVBLK, &IOsb, 0, 0,
                         bptr, bnum+BLOCK_READ_MAX < EofBlock ?
                               BLOCK_READ_MAX * BLOCK_SIZE :
                               (EofBlock-bnum+1) * BLOCK_SIZE,
                         bnum, 0, 0, 0);
      if (Debug)
         fprintf (stdout, "sys$qiow() %%X%08.08X IOsb: %%X%08.08X\n",
                  status, IOsb.Status);
      if (VMSnok (status)) break;

      bptr[IOsb.Count] = '\0';
      bnum += BLOCK_READ_MAX;
      bptr += BLOCK_READ_MAX * BLOCK_SIZE;
   }

   if (VMSok (status) && VMSnok (IOsb.Status)) status = IOsb.Status;
   if (VMSnok (status) && status != SS$_ENDOFFILE) exit (status);

   sys$dassgn (AcpChannel);

   if (Debug) fprintf (stdout, "|%s|\n", BufferPtr);
   return (BufferPtr);
}

/*****************************************************************************/
/*
Use CSS1 <SPAN></SPAN> sections with absolute positioning to overlay an image
of a cross hair, representing the float RA & Dec coordinates passed as form
fields, onto an image of a celestial map.  This is basically for locating the
current work-unit but can also be used for showing any location.
*/

SkyMap ()

{
   static int  TableBorder = 1;

   int  idx,
        result,
        DecPixel,
        RaPixel;
   float  ScratchFloat;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SkyMap() %s %s\n", CgiFormRaPtr, CgiFormDecPtr);

   result = sscanf (CgiFormRaPtr, "%f", &StartRa);

   result = sscanf (CgiFormDecPtr, "%f", &StartDec);

   ScratchFloat = StartRa;
   StartRaHr = (int)ScratchFloat;
   ScratchFloat -= (float)StartRaHr;
   StartRaMin = (int)(60.0 * ScratchFloat);
   ScratchFloat -= (float)StartRaMin / 60.0;
   StartRaSec = (int)(3660.0 * ScratchFloat);

   ScratchFloat = StartDec;
   StartDecDeg = (int)ScratchFloat;
   ScratchFloat -= (float)StartDecDeg;
   StartDecMin = (int)(60.0 * ScratchFloat);
   ScratchFloat -= (float)StartDecMin / 60.0;
   StartDecSec = (int)(3660.0 * ScratchFloat);

   while (StartRa >= 24.0) StartRa -= 24.0;
   while (StartRa < 0.0) StartRa += 24.0;

   if (!result ||
       StartRa < 0.0 ||
       StartRa >= 24.0)
   {
      CgiLibResponseError (FI_LI, 0, ErrorSkyMapRa);
      return;
   }

   if (!result ||
       StartDec < -90.0 ||
       StartDec > +90.0)
   {
      CgiLibResponseError (FI_LI, 0, ErrorSkyMapDec);
      return;
   }

   if (StartRa <= 12.0)
      RaPixel = (SkyMapImageWidth / 2) -
                (int)(StartRa * ((float)SkyMapImageWidth / 24.0));
   else
      RaPixel = SkyMapImageWidth -
                (int)((StartRa - 12.0) * ((float)SkyMapImageWidth / 24.0));

   DecPixel = (SkyMapImageHeight / 2) -
              (int)(StartDec * ((float)SkyMapImageHeight / 180.0));

   if (Debug)
      fprintf (stdout, "%f %f\n%d x %d\n%d x %d\n",
               StartRa, StartDec,
               SkyMapImageWidth, SkyMapImageHeight,
               RaPixel, DecPixel);

   CgiLibResponseHeader (200, "text/html");

   fprintf (stdout,
"<HTML>\n\
<HEAD>\n\
<META NAME=\"generator\" CONTENT=\"%s\">\n\
<META NAME=\"environment\" CONTENT=\"%s\">\n\
<META NAME=\"author\" CONTENT=\"%s\">\n\
<TITLE>VMSeti @ %s</TITLE>\n\
</HEAD>\n\
<BODY BGCOLOR=\"#000000\" TEXT=\"#ffffff\">\n\
<SPAN STYLE=\"POSITION:absolute; \
LEFT:0px; TOP:0px; PADDING:%d; \
WIDTH:%dpx; Z-INDEX:1;\">\n\
<TABLE CELLSPACING=0 CELLPADDING=0 BORDER=%d>\n\
<TR><TD><IMG SRC=\"%s%sskymap.gif\"></TD></TR>\n\
</TABLE>\n\
%d hr %d min %d sec RA, %d deg %d min %d sec Dec\n\
<BR><FONT SIZE=1>\n\
<I><SUP>*</SUP>If the cross hair appears below the skymap then your\n\
browser doesn't support CSS1 and this display will not work for you.\n\
<BR><SUP>**</SUP>Image sourced and copyright: SETI@home</I>\n\
</FONT>\n\
</SPAN>\n\
<SPAN STYLE=\"POSITION:absolute; \
LEFT:%dpx; TOP:%dpx; PADDING:%d; \
Z-INDEX:2;\">\n\
<IMG SRC=\"%s%sskymapx.gif\">\n\
</SPAN>\n\
</BODY>\n\
</HTML>\n",
      SoftwareID,
      CgiEnvironmentPtr,
      MetaAuthor,
      CgiHttpHostPtr,
      /* bottom layer, hosts skymap image */
      SkyMapLeftBorder,
      SkyMapImageWidth + (TableBorder * 2),
      TableBorder,
      SkyMapPathPtr ? SkyMapPathPtr : CgiScriptNamePtr,
      SkyMapPathPtr ? "" : "?img=",
      StartRaHr, StartRaMin, StartRaSec,
      StartDecDeg, StartDecMin, StartDecSec,
      /* top layer, work-unit cross hair */
      SkyMapImageWidthAdjust + TableBorder +
         RaPixel - (SkyMapXImageWidth / 2),
      SkyMapImageHeightAdjust + TableBorder +
         DecPixel - (SkyMapXImageHeight / 2),
      SkyMapLeftBorder,
      SkyMapPathPtr ? SkyMapPathPtr : CgiScriptNamePtr,
      SkyMapPathPtr ? "" : "?img=");
}

/*****************************************************************************/
/*
Return a CGI response containing the skymap GIF image requested.
*/

SkymapImage (char *SkymapName)

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SkymapImage() |%s|\n", SkymapName);

   if (!strcmp (SkymapName, "skymap.gif"))
   {
      CgiLibResponseHeader (200, "image/gif",
"Content-Length: %d\n\
Last-Modified: Fri, 13 Jan 1978 14:00:00 GMT\n",
         sizeof(SkymapBytes));
      write (fileno(stdout), SkymapBytes, sizeof(SkymapBytes));
   }
   else
   if (!strcmp (SkymapName, "skymapx.gif"))
   {
      CgiLibResponseHeader (200, "image/gif",
"Content-Length: %d\n\
Last-Modified: Fri, 13 Jan 1978 14:00:00 GMT\n",
         sizeof(SkymapxBytes));
      write (fileno(stdout), SkymapxBytes, sizeof(SkymapxBytes));
   }
   else
      CgiLibResponseError (FI_LI, 0, ErrorSkymapImage);

   exit (SS$_NORMAL);
}

/*****************************************************************************/
/*
Quick and dirty conversion of a file's contents (e.g. GIF) into a (header) file
as an array of char for inclusion in a C program.
*/

File2Bytes
(
char *fName,
char *bName
)
{
   int  bcnt, ffd, rcnt;
   unsigned char  *cptr;
   unsigned char  buf [4096];
   FILE  *bfh;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "File2Bytes() |%s|%s|\n", fName, bName);

   ffd = open (fName, O_RDONLY, "ctx=bin");
   if (ffd < 0) exit (vaxc$errno);
   bfh = fopen (bName, "w");
   if (!bfh) exit (vaxc$errno);
   fprintf (bfh, "/* %s generated by File2Bytes() %s */\n\nchar %sBytes [] =\n{",
            fName, SOFTWAREID, fName);
   bcnt = 0;
   for (;;)
   {
      rcnt = read (ffd, buf, sizeof(buf));
      if (rcnt <= 0) break;
      cptr = buf;
      while (rcnt--)
      {
         if (bcnt++) fputc (',', bfh);
         if (bcnt % 16 == 1) fputc ('\n', bfh);
         fprintf (bfh, "%3u", *cptr++);
      }
   }
   fprintf (bfh, "\n};\n/* %d bytes */\n", bcnt);
   fclose (bfh);
   close (ffd);

   exit (SS$_NORMAL);
}

/*****************************************************************************/

