{ TAR.PAS -	Manipulates Unix 'tar' format archive files.
!
! Abstract:	A proggie to work with Unix 'tar' (Tape ARchive) files on
!		VMS.  Reads and writes files to/from a file 'TAR_ARCHIVE'
!		(if not specified differently on command line), so if the tape
!		drive is MOUNTed/FOREIGN/RECORD=512/BLOCK=10240 (tar default),
!		and pointed to by a logical 'TAR_ARCHIVE', tar will read from
!		the tape-drive.
!
! Copyright:	Copyright 1989 Victoria College Computer Services.  See
!		AAAREADME.1ST for distribution rights.
!____________________________________________________________________________
! Author:	Tim Cook (timcc@viccol.edu.au)
! Release:	Version 1.0-1
!____________________________________________________________________________}

[INHERIT ('SYS$LIBRARY:STARLET', 'VCDEFS'),
   ENVIRONMENT ('TAR')]

PROGRAM tar (input, output, archive, archive_temp) ;


  CONST
      record_size = 512 ;
      tar_record_size = record_size ;
      VMS_block_size = 512 ;
      tar_multi_block_count = 32 ;	{ Maximum 127 }
      space = ' ' ;
      null = chr (0) ;
      lf = chr (10) ;
      crlf = chr (10) + chr (13) ;
      colon = ':' ;
      dot = '.' ;
      dollar = '$' ;
      underscore = '_' ;

      output_kt = 'OUTPUT' ;
      filespec_kt = 'FILESPEC' ;
      archive_kt = 'ARCHIVE' ;
      tar_archive_kt = 'TAR_ARCHIVE' ;
      confirm_kt = 'CONFIRM' ;
      scan_kt = 'SCAN' ;

   VAR

   { VMS Error status codes for TAR }

   { Success }
      tar__created,
      tar__createdir,
      tar__written,
      tar__appended,
      tar__totcreat,
      tar__totwrite,
      tar__totappend,

   { Informational }
      tar__empty,
      tar__hardlink,
      tar__softlink,

   { Warning }
      tar__nofiles,
      tar__wrapped,
      tar__rectoolong,

   { Error }
      tar__openin,
      tar__close,
      tar__createrr,
      tar__errcredir,

   { Fatal }
      tar__badheader,
      tar__badarchive,
      tar__errread,
      tar__errwrite,
      tar__internerr
         : [EXTERNAL,VALUE] sts_type ;

   TYPE
      fab_rfm_type = [BYTE] (undefined, fixed_length, variable_length,
         variable_fixed_control, stream, stream_lf, stream_cr) ;

      oat_8 = RECORD          { Octal Ascii, Terminated by space and null }
         value : PACKED ARRAY [1..6] OF char ;
         fill_space : char ;
         fill_null : char END ;
      tar_record_type = PACKED RECORD CASE integer OF
0:(      name : PACKED ARRAY [1..100] OF char ;
         mode, uid, gid : oat_8 ;
         siz, mtime : PACKED ARRAY [1..12] OF char ;
         chksum : oat_8 ;
         linkflag : char ;
         linkname : PACKED ARRAY [1..100] OF char ;
         filler : PACKED ARRAY [1..175] OF char) ;
1:(      data : PACKED ARRAY [1..tar_record_size] OF char) END ;

      file_mode_type = RECORD CASE integer OF
0:(      value : unsigned) ;
1:(      mask : PACKED ARRAY [0..31] OF [BIT] boolean) END ;

      tar_record_ptr = ^tar_record_type ;
      tar_file_type = FILE OF tar_record_type ;
      tar_block_type = ARRAY [1..20] OF tar_record_type ;
      broken_time_type = lib_numtim_type ;
      class_protection_type = PACKED RECORD
         noread, nowrite, noexecute, nodelete : [BIT] boolean END ;
      file_protection_type = PACKED RECORD
         system, owner, group, world : class_protection_type END ;
      fixed_string_100 = PACKED ARRAY [1..100] OF char ;
      small_string = VARYING [50] OF char ;
      medium_string = VARYING [255] OF char ;
      large_string = VARYING [4100] OF char ;
      map_mode_type = (prefix_mode, absolute_mode, root_mode, single_dir_mode) ;

   VAR
      archive : tar_file_type ;
      archive_temp : tar_file_type ;
      output_filespec : medium_string ;
      i, j : integer ;                       { miscellaneous counters }
      last_char : integer ;                  { points to last char buffered }
      eof_mark_found : boolean ;             {  by write_temp }
      archive_temp_open : boolean ;
      option : small_string ;
      opening_archive_input : [VOLATILE] boolean ;
      creating_archive_output : [VOLATILE] boolean ;
      validating : [VOLATILE] boolean ;
      full_archive_spec : [VOLATILE] medium_string ;

      default_header : tar_record_type ;

      UNIX_epoch_time : lib_date_type ;
      delta_seconds : integer ;              { Delta time from GMT }

   VALUE
      default_header := (0,
         (100 OF null),             { name }
         ('   644', space, null),   { mode }
         ('     0', space, null),   { uid }
         ('     0', space, null),   { gid }
         '          0 ',            { siz }
         ' 4241462038 ',            { mtime - 10-MAY-1988 12:00 EST }
         ('     0', null, space),   { chksum - nul/spc back to front, I know }

         null,                      { linkflag }
         (100 OF null),             { linkname }
         (175 OF null)) ;           { filler }

      UNIX_epoch_time := (0, %x4BEB4000, %x007C9567) ;
         { which equals 1-JAN-1970 00:00:00.00 }
      validating := false ;
      opening_archive_input := false ;
      archive_temp_open := false ;


   [ASYNCHRONOUS] FUNCTION tar_handler (     { TAR condition handler }
      VAR sigargs : lib_sigargs_type ;
      VAR mechargs: lib_mechargs_type) : sts_type ;

      VAR
         i, j : integer ;
         condition : STS$TYPE ;

      PROCEDURE bad_archive ;
         VAR
            descriptor : [STATIC] PACKED RECORD
               maxlen : lib_word_type ;
               dtype, class : lib_byte_type ;
               pointer : integer END ;

         BEGIN
            sigargs.param_count := 4 ;
            sigargs.condition := tar__badarchive ;
            sigargs.parameter[1] := 1 ;
            descriptor.maxlen := length (full_archive_spec) ;
            descriptor.class := DSC$K_CLASS_VS ;
            descriptor.dtype := DSC$K_DTYPE_VT ;
            descriptor.pointer := iaddress (full_archive_spec) ;
            sigargs.parameter[2] := iaddress (descriptor) END ;

      BEGIN    { tar_handler }
         tar_handler := SS$_RESIGNAL ;
         CASE sigargs.condition OF
            PAS$_ERRDUROPE, PAS$_FILNOTFOU, PAS$_ERRDURREW : BEGIN
               IF opening_archive_input THEN
                  sigargs.condition := tar__openin
               ELSE IF creating_archive_output THEN
                  sigargs.condition := tar__createrr
               ELSE
                  LIB$STOP (tar__internerr, 1, sigargs.condition) ;
               sigargs.parameter[1] := 1 ;
               j := int (sigargs.param_count) - 3 ;
               FOR i := 2 TO j DO
                  sigargs.parameter[i] := sigargs.parameter[i+2] ;
               sigargs.param_count := sigargs.param_count - 2 END ;
            PAS$_ACCMETINC,      { Access method inconsistent }
            PAS$_RECLENINC,      { Record length inconsistent }
            PAS$_RECTYPINC :     { Record type inconsistent }
               IF opening_archive_input THEN
                  bad_archive ;
            PAS$_ERRDURGET :     { Error during GET }
               sigargs.condition := tar__errread ;
            PAS$_INVSYNOCT :     { Invalid syntax in octal value - Something
                                   might have blown up while reading a header }
               IF validating THEN
                  bad_archive ;

            OTHERWISE BEGIN
               condition := (sigargs.condition)::STS$TYPE ;
               IF (condition.STS$V_FAC_NO = PAS$_FACILITY) THEN BEGIN

                { Report condition encountered as "internal error" }

                  LIB$STOP (tar__internerr, 1, sigargs.condition) ;
                  END END ;
            END ;
         END ;    { tar_handler }


   FUNCTION lowercase (
      inp_string : VARYING [n1] OF char ;
      start_pos : integer := 1) : medium_string ;

      VAR
         i : integer ;
         result : medium_string ;

      BEGIN
         result := inp_string ;
         FOR i := start_pos TO n1 DO
            IF inp_string[i] IN ['A'..'Z'] THEN
               result[i] := chr (ord (inp_string[i]) + 32) ;
         lowercase := result ;
         END ;


   FUNCTION uppercase (    { Wrote my own cos it looks neater with "lowercase" }
      inp_string : VARYING [n1] OF char) : medium_string ;

      VAR
         result : medium_string ;
         i : integer ;

      BEGIN
         result := inp_string ;
         FOR i := 1 TO n1 DO
            IF inp_string[i] IN ['a'..'z'] THEN
               result[i] := chr (ord (inp_string[i]) - 32) ;
         uppercase := result ;
         END ;


   PROCEDURE convert_zstr (     { Convert a null-terminated string to VARYING }
      z_string : fixed_string_100 ;
      VAR vs_string : VARYING [n1] OF char) ;

      BEGIN
         vs_string := substr (z_string, 1, index (z_string, null) - 1) ;
         END ;


   FUNCTION checksum (     { Calculate the checksum of a TAR header record }
      check_record : tar_record_type)
         : integer ;

      VAR
         result, i : integer ;

      BEGIN
         result := 0 ;
         FOR i := 1 TO tar_record_size DO
            result := result + ord (check_record.data[i]) ;
         checksum := result END ;


   FUNCTION february_days (year : integer) : integer ;
      BEGIN
         IF year REM 4 = 0 THEN
            IF year REM 100 = 0 THEN
               IF year REM 400 = 0 THEN
                  february_days := 29
               ELSE
                  february_days := 28
            ELSE
               february_days := 29
         ELSE
            february_days := 28 END ;


   FUNCTION get_timezone : integer ;
      VAR
         return, hours, minutes, i : integer ;
         timezone_str : small_string ;

      BEGIN
         return := 1 ;
         IF failure ($TRNLNM (, 'LNM$FILE_DEV', 'TAR_TIMEZONE',,
            lib_item_list (lib_out_item (LNM$_STRING, %DESCR timezone_str))))
         THEN
            IF failure ($TRNLNM (, 'LNM$FILE_DEV', 'SYS$TIME_ZONE',,
                  lib_item_list (lib_out_item (LNM$_STRING,
                  %DESCR timezone_str)))) THEN
               return := 0 ;
         IF return <> 0 THEN BEGIN
            readv (timezone_str, i) ;
            hours := i DIV 100 ;
            minutes := i - hours * 100 ;
            IF (abs (hours) > 18) OR (abs (minutes) > 59) THEN
               return := 0 
            ELSE
               return := minutes * 60 + hours * 3600 END ;
         get_timezone := return ;
         END ;
         

   FUNCTION add_timezone (	{ converts from GMT to local time }
      VAR UNIX_time : unsigned) : boolean ;

      BEGIN
         add_timezone := true ;

         IF delta_seconds < 0 THEN
            IF (-1 * delta_seconds) > UNIX_time THEN BEGIN
               UNIX_time := 0 ;
               add_timezone := false END	{ indicates over/underflow }
            ELSE
               UNIX_time := UNIX_time + delta_seconds
         ELSE
            IF uint (lib_k_maxlong) - delta_seconds > UNIX_time THEN
               UNIX_time := UNIX_time + delta_seconds
            ELSE BEGIN
               UNIX_time := lib_k_maxlong ;
               add_timezone := false END END ;


   FUNCTION subtract_timezone (		{ converts from local time to GMT }
      VAR UNIX_time : unsigned) : boolean ;

      BEGIN
         IF delta_seconds > 0 THEN
            IF delta_seconds < UNIX_time THEN
               UNIX_time := UNIX_time - delta_seconds
            ELSE BEGIN
               UNIX_time := 0 ;
               subtract_timezone := false END
         ELSE
            IF (uint (lib_k_maxlong) - (-1 * delta_seconds)) > UNIX_time THEN
               UNIX_time := UNIX_time - delta_seconds
            ELSE BEGIN
               UNIX_time := lib_k_maxlong ;
               subtract_timezone := false END END ;


   PROCEDURE break_up_UNIX_time (
      UNIX_time : unsigned ;
      VAR time : broken_time_type) ;

      CONST
         mar =	31 ;
         apr =	30 + mar ;
         may =	31 + apr ;
         jun =	30 + may ;
         jul =	31 + jun ;
         aug =	31 + jul ;
         sep =	30 + aug ;
         oct =	31 + sep ;
         nov =	30 + oct ;
         dec =	31 + nov ;
         jan =	31 + dec ;

         seconds_per_day =	86400 ;
         seconds_per_hour =	3600 ;
         seconds_per_minute =	60 ;
         weekday_epoch =	4 ;
         days_to_eoy =		306 ;	{ Days from 1/3 to 1/1 next year }

      VAR
         UNIX_time_l : unsigned ;
         days, m_day, temp : unsigned ;

      BEGIN	{ break_up_UNIX_time }
         UNIX_time_l := UNIX_time ;

         days := int (UNIX_time_l DIV seconds_per_day) ;

         UNIX_time_l := UNIX_time_l - days * seconds_per_day ;
         time.hour := int (UNIX_time_l DIV seconds_per_hour) ;

         UNIX_time_l := UNIX_time_l - time.hour * seconds_per_hour ;
         time.minute := int (UNIX_time_l DIV seconds_per_minute) ;

         time.second := int (UNIX_time_l - time.minute * seconds_per_minute) ;

         days := days + 2133 ;		{ Now relative to 1/3/1964 }

      { Find remainder of days / 365.25 }
         temp := days * 4 DIV 1461 ;
         m_day := days - temp * 1461 DIV 4 ;

      { m_day now contains the day of the year, relative to 1st March }

         time.year := -6 ;		{ Year will then be relative to 1970 }

         IF m_day = 0 THEN BEGIN	{ It's actually the 29th of Feb! }
            time.month := 2 ;
            time.day := 29 END
         ELSE				{ Right, figure out month and day }
            IF m_day > aug THEN
               IF m_day > nov THEN
                  IF m_day > dec THEN BEGIN
                     IF m_day > jan THEN BEGIN		{ February }
                        time.month := 2 ;
                        time.day := int (m_day - jan) END
                     ELSE BEGIN				{ January }
                        time.month := 1 ;
                        time.day := int (m_day - dec) END ;
                     time.year := -5 END
                  ELSE BEGIN
                     time.month := 12 ;
                     time.day := int (m_day - nov) END
               ELSE
                  IF m_day > sep THEN
                     IF m_day > oct THEN BEGIN		{ November }
                        time.month := 11 ;
                        time.day := int (m_day - oct) END
                     ELSE BEGIN				{ October }
                        time.month := 10 ;
                        time.day := int (m_day - sep) END
                  ELSE BEGIN				{ September }
                     time.month := 9 ;
                     time.day := int (m_day - aug) END
            ELSE
               IF m_day > may THEN
                  IF m_day > jun THEN
                     IF m_day > jul THEN BEGIN		{ August }
                        time.month := 8 ;
                        time.day := int (m_day - jul) END
                     ELSE BEGIN				{ July }
                        time.month := 7 ;
                        time.day := int (m_day - jun) END
                  ELSE BEGIN				{ June }
                     time.month := 6 ;
                     time.day := int (m_day - may) END
               ELSE
                  IF m_day > mar THEN
                     IF m_day > apr THEN BEGIN		{ May }
                        time.month := 5 ;
                        time.day := int (m_day - apr) END
                     ELSE BEGIN				{ April }
                        time.month := 4 ;
                        time.day := int (m_day - mar) END
                  ELSE BEGIN				{ March }
                     time.month := 3 ;
                     time.day := int (m_day) END ;

      { Last of all, what year is it? }

         time.year := int (time.year + days * 4 DIV 1461) ;
         END ;


   FUNCTION UNIX_time_to_str (
      UNIX_time : unsigned ;
      delta_seconds : integer := 0) : medium_string ;

      VAR
         time : broken_time_type ;
         UNIX_time_local : unsigned ;
         months : [STATIC] ARRAY [1..12] OF PACKED ARRAY [1..3] OF char :=
            ('Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun',
             'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec') ;
         return_string : VARYING [17] OF char ;

      BEGIN
         add_timezone (UNIX_time) ;
         UNIX_time_local := UNIX_time ;
         break_up_UNIX_time (UNIX_time_local, time) ;
         lib_sigiferr (LIB$SYS_FAO ('!2SL !AS !SL !2ZL:!2ZL',, return_string,
            time.day, %STDESCR (months[time.month]), time.year + 1970,
            time.hour, time.minute)) ;
         UNIX_time_to_str := return_string ;
         END ;


   FUNCTION VMS_to_UNIX_time (
      VMS_time : lib_date_type ;
      delta_seconds : integer := 0) : unsigned ;

      VAR
         VMS_rel_UNIX_epoch : lib_date_type ;    { delta VMS time relative }
         return_value : unsigned ;               {  to 1-JAN-1970 0:0:0.0 }

      BEGIN
         VMS_to_UNIX_time := 0 ;
         IF success (LIB$SUB_TIMES (VMS_time, UNIX_epoch_time,
               VMS_rel_UNIX_epoch)) THEN
            IF success (LIB$CVT_FROM_INTERNAL_TIME (LIB$K_DELTA_SECONDS,
                  return_value, VMS_rel_UNIX_epoch)) THEN BEGIN
               subtract_timezone (return_value) ;
               VMS_to_UNIX_time := return_value END ;
         END ;


{++
!   FUNCTION VMS_to_UNIX_time ( / Calculate seconds from Jan 1 1970, 00:00 GMT /
!      VMS_time : lib_date_type ;
!      delta_seconds : integer := 0) : unsigned ;
!
!      VAR
!         num_time : lib_numtim_type ;
!         time_temp : integer ;
!         days : integer ;
!         feb_days : integer ;
!         temp_result : unsigned ;
!
!      BEGIN
!         lib_sigiferr ($NUMTIM (num_time, VMS_time)) ;
!
!         feb_days := february_days (num_time.year) ;
!
!         CASE num_time.month OF
!            1 : days := num_time.day ;
!            2 : days := num_time.day + 31 ;
!            3 : days := num_time.day + feb_days + 31 ;
!            4 : days := num_time.day + feb_days + 62 ;
!            5 : days := num_time.day + feb_days + 92 ;
!            6 : days := num_time.day + feb_days + 123 ;
!            7 : days := num_time.day + feb_days + 153 ;
!            8 : days := num_time.day + feb_days + 184 ;
!            9 : days := num_time.day + feb_days + 215 ;
!            10 : days := num_time.day + feb_days + 245 ;
!            11 : days := num_time.day + feb_days + 276 ;
!            12 : days := num_time.day + feb_days + 307 ;
!            END ;
!         IF num_time.year < 1970 THEN
!            temp_result := 0
!         ELSE BEGIN
!            temp_result := uint (num_time.second + num_time.minute * 60 +
!               num_time.hour * 3600 + days * 3600 * 24 + delta_seconds) ;
!            num_time.year := num_time.year - 1970 ;
!            temp_result := temp_result +
!               uround (num_time.year * 3600 * 24 * 365.25) ;
!         / Correct to GMT /
!            IF temp_result - delta_seconds < 0 THEN
!               VMS_to_UNIX_time := 0
!            ELSE
!               VMS_to_UNIX_time := temp_result - delta_seconds END ;
!         END ;
!--
}


   FUNCTION parse (
      filespec : VARYING [n1] OF char ;
      default_spec : VARYING [n2] OF char := '' ;
      field : integer := 0) : medium_string ;

      VAR
         return_string : medium_string ;

      BEGIN
         IF field = 0 THEN
            lib_sigiferr (lib_parse (%STDESCR (filespec), return_string,
               %STDESCR (default_spec)))
         ELSE
            lib_sigiferr (lib_parse (%STDESCR (filespec), return_string,
               %STDESCR (default_spec),,, field)) ;
         parse := return_string ;
         END ;


   PROCEDURE get_prompted_string (
      prompt : VARYING [n1] OF char ;
      VAR string : VARYING [n2] OF char ;
      default : VARYING [n3] OF char := '') ;

      VAR
         status : sts_type ;

      BEGIN
         status := LIB$GET_INPUT (string, prompt) ;
         IF status = RMS$_EOF THEN $EXIT
         ELSE lib_sigiferr (status) ;
         IF string = '' THEN string := default END ;


   FUNCTION confirm_operation (
      op_to_confirm : VARYING [n1] OF char ;
      conf_filespec : VARYING [n2] OF char ;
      confirmed : boolean := false) : boolean ;

      VAR
         decision : small_string ;

      BEGIN
         IF confirmed THEN confirm_operation := true
         ELSE BEGIN
            get_prompted_string (lowercase (op_to_confirm, 2) + space +
               conf_filespec + '? (Y/N) [Y]: ', decision, 'Y') ;
            confirm_operation := decision[1] IN ['Y', 'y', ' '] ;
            END ;
         END ;


   FUNCTION VMS_filespec (
      UNIX_filespec : VARYING [n1] OF char ;
      VAR absolute : boolean) : medium_string ;

      VAR
         UNIX_dir : small_string ;
         temp_ch : char ;
         i, j : integer ;
         device, directory, name : medium_string ;
         dot_found, device_name, start_name, more : boolean ;

      BEGIN
         absolute := false ;
         i := n1 ;
         IF i = 0 THEN BEGIN
            name := '' ;
            unix_dir := '' END
         ELSE BEGIN
            more := true ;
            WHILE more DO
               IF i > 0 THEN
                  IF UNIX_filespec[i] <> '/' THEN
                     i := i - 1
                  ELSE
                     more := false
               ELSE
                  more := false ;
            name := substr (UNIX_filespec, i + 1, n1 - i) ;
            IF i > 1 THEN
               UNIX_dir := substr (UNIX_filespec, 1, i - 1)
            ELSE
               UNIX_dir := '' ;
            j := length (name) ;
            dot_found := false ;
            FOR i := j DOWNTO 1 DO
               CASE name[i] OF
                  '.' :
                     IF dot_found THEN
                        name[i] := '-'
                     ELSE
                        dot_found := true ;
                  'a'..'z' :
                     name[i] := chr (ord (name[i]) - 32) ;
                  'A'..'Z', '0'..'9', '$', '_', '-' : ;
                  ',', '+', '~', '#', '*' : name[i] := '_' ;
                  OTHERWISE name[i] := '$' ;
                  END END ;
         device := '' ;
         directory := '' ;
         IF length (UNIX_dir) > 0 THEN BEGIN
            IF substr (UNIX_dir, 1, 2) = './' THEN
               UNIX_dir := substr (UNIX_dir, 3, length (UNIX_dir) - 2) ;
            j := length (UNIX_dir) ;
            start_name := true ;
            device_name := false ;
            i := 0 ;
            WHILE i < j DO BEGIN
               i := i + 1 ;
               CASE UNIX_dir[i] OF
                  '/' :
                     IF device_name THEN BEGIN
                        start_name := true ;
                        device_name := false END
                     ELSE IF i = 1 THEN BEGIN
                        absolute := true ;
                        device_name := true END
                     ELSE BEGIN
                        start_name := true ;
                        directory := directory + '.' END ;
                  'A'..'Z', '0'..'9', '$', '_' : BEGIN
                     IF device_name THEN
                        device := device + UNIX_dir[i]
                     ELSE
                        directory := directory + UNIX_dir[i] ;
                     start_name := false END ;
                  'a'..'z' : BEGIN
                     IF device_name THEN
                        device := device + chr (ord (UNIX_dir[i]) - 32)
                     ELSE
                        directory := directory + chr (ord (UNIX_dir[i]) - 32) ;
                     start_name := false END ;
                  '-' : BEGIN
                     IF start_name OR device_name THEN
                        temp_ch := '_'
                     ELSE
                        temp_ch := '-' ;
                     IF device_name THEN
                        device := device + temp_ch
                     ELSE
                        directory := directory + temp_ch END ;
                  '.' :
                     IF device_name THEN
                        device := device + '_'
                     ELSE
                        IF (j > i) AND (UNIX_dir[i+1] = '.') THEN BEGIN
                           directory := directory + '-' ;
                           i := i + 1 END
                        ELSE
                           directory := directory + '_' ;
                  OTHERWISE BEGIN
                     IF device_name THEN
                        device := device + '$'
                     ELSE
                        directory := directory + '$' ;
                     start_name := false END ;
                  END ;    { CASE }
               END ;    { WHILE }
            END ;     { IF }
         IF device = '' THEN
            IF directory = '' THEN
               VMS_filespec := '[]' + name
            ELSE
               VMS_filespec := '[.' + directory + ']' + name
         ELSE
            VMS_filespec := device + ':[' + directory + ']' + name
         END ;


   FUNCTION UNIX_filespec (
      VMS_filespec : VARYING [n1] OF char ;   { ALL fields expected }
      map_mode : map_mode_type ;
      prefix_length : integer := 0 ;
      retain_version : boolean := false) : medium_string ;

      VAR
         device, name, type_, version : small_string ;
         d, n, t, v, e, i : integer ;
         directory, temp_result : medium_string ;

      BEGIN
         e := length (VMS_filespec) ;
         v := e ;
         WHILE VMS_filespec[v] <> ';' DO
            v := v - 1 ;
         t := v ;
         WHILE VMS_filespec[t] <> '.' DO
            t := t - 1 ;
         n := t ;
         WHILE VMS_filespec[n] <> ']' DO
            n := n - 1 ;
         d := 1 ;
         WHILE VMS_filespec[d] <> ':' DO
            d := d + 1 ;
         device := substr (VMS_filespec, 1, d - 1) ;
         directory := substr (VMS_filespec, d + 2, n - d - 2) ;

   { Take out a leading 000000 directory if present (irrelevant) }

         IF directory.length > 8 THEN
            IF substr (directory, 1, 8) = '000000.' THEN
               directory := substr (directory, 8, length (directory) - 7)
            ELSE
         ELSE
            IF directory = '000000' THEN
               directory := '' ;

         name := substr (VMS_filespec, n + 1, t - n - 1) ;
         type_ := substr (VMS_filespec, t, v - t) ;
         IF type_ = '.' THEN
            type_ := '' ;
         version := substr (VMS_filespec, v, e - v + 1) ;

         temp_result := '' ;
         IF map_mode IN [absolute_mode, prefix_mode] THEN
            temp_result := '/' + device + '/' ;
         IF map_mode <> single_dir_mode THEN BEGIN
            d := length (directory) ;
            IF d > 0 THEN BEGIN
               FOR i := 1 TO d DO
                  IF directory[i] = '.' THEN
                     directory[i] := '/' ;
               temp_result := temp_result + directory + '/' END END ;
         temp_result := temp_result + name + type_ ;
         IF retain_version THEN
            temp_result := temp_result + version ;
         IF map_mode = prefix_mode THEN
            temp_result := substr (temp_result, prefix_length + 1,
               length (temp_result) - prefix_length) ;
         UNIX_filespec := lowercase (temp_result) ;
         END ;


   FUNCTION find_file_spec (
      VAR file_buf : [UNSAFE] text) : medium_string ;

   { This routine returns the full file specification of the file opened with
     the passed file-variable.  This routine does assume that a NAM block has
     been used by Pascal (which is usually the case), and that the file is
     open. }

      VAR
         fab : fab_pointer ;
         nam : nam_pointer ;
         nam_rsa : nam_rsa_type ;

      BEGIN
         fab := PAS$FAB (file_buf) ;
         nam := (fab^.FAB$L_NAM)::nam_pointer ;
         nam_rsa := (nam^.NAM$L_RSA)::nam_rsa_type ;
         find_file_spec := substr (nam_rsa^, 1, nam^.NAM$B_RSL) ;
         END ;


   PROCEDURE open_archive_input (	{ Opens an archive file for input }
      filespec : VARYING [n1] OF char ;
      share : boolean := false) ;

      VAR
         status : sts_type ;

      BEGIN
         opening_archive_input := true ;
         IF share THEN
            open (archive, filespec, history := OLD, sharing := READONLY)
         ELSE
            open (archive, filespec, history := OLD) ;
         reset (archive) ;
         opening_archive_input := false ;
         END ;


   FUNCTION eof_archive : boolean ;
      BEGIN eof_archive := eof (archive) OR eof_mark_found END ;


   PROCEDURE bad_header (header : tar_record_type) ;

      BEGIN
         close (archive) ;
         open (archive, 'SYS$SCRATCH:HEADER.TAR', history := NEW) ;
         rewrite (archive) ;
         write (archive, header) ;
         close (archive) ;
         LIB$STOP (tar__badheader) ;
         END ;


   FUNCTION valid_header (   { Computes and checks the header checksum }
      header_record : tar_record_type ;
      VAR file_spec : VARYING [n1] OF char ;
      VAR file_size : integer ;
      VAR file_mtime : unsigned ;
      VAR file_mode : file_mode_type ;
      VAR directory : boolean) : boolean ;

      VAR
         tar_filespec : medium_string ;
         header_tmp : tar_record_type ;
         header_checksum : integer ;
         i : integer ;
         response : char ;

      BEGIN
         validating := true ;
         valid_header := true ;
         header_tmp := header_record ;
         IF header_record.chksum.value[1] = null THEN BEGIN
               { This may be end of archive }
            valid_header := false ;
            IF checksum (header_tmp) = 0 THEN    { You betcha! }
               eof_mark_found := true END
         ELSE BEGIN

      {  According to TAR(5), chksum is stored as a decimal ascii string, but
         in practice I have found it octal. }

            readv (header_record.chksum.value, header_checksum:oct) ;

      {  In order to calculate the checksum without infinite recursion,
         header_tmp.chksum is set to a known value (as UNIX tar sets it).}

            header_tmp.chksum.value := '      ' ;
            header_tmp.chksum.fill_null := space ;
            header_tmp.chksum.fill_space := space ;
            IF checksum (header_tmp) <> header_checksum THEN
               bad_header (header_record) ;
            convert_zstr (header_record.name, file_spec) ;
            readv (header_record.siz, file_size:oct) ;
            readv (header_record.mtime, file_mtime:oct) ;
            readv (header_record.mode.value, file_mode.value:oct) ;
            directory := false ;
            IF file_mode.mask[10] THEN
               IF file_size = 0 THEN
                  directory := true
               ELSE
                  bad_header (header_record)
            ELSE
               IF (file_spec[file_spec.length] = '/') AND (file_size = 0) THEN
                  directory := true ;
            END ;
         validating := false ;
         END ;


   FUNCTION scan_to_next_header (
      file_size : integer := -1) : boolean ;

      VAR
         skip_count, i : integer ;
         more_to_scan : boolean ;

      BEGIN
         scan_to_next_header := true ;

      { Assume archive^ contains header prior to next header }

         IF file_size = -1 THEN
            readv (archive^.siz, file_size:oct) ;
         IF file_size = 0 THEN skip_count := 0
         ELSE
            skip_count := ((file_size - 1) DIV tar_record_size) + 1 ;
         i := 0 ;
         more_to_scan := true ;
         WHILE more_to_scan DO
            IF i = skip_count THEN more_to_scan := false
            ELSE
               IF eof (archive) THEN BEGIN
                  more_to_scan := false ;
                  scan_to_next_header := false END
               ELSE BEGIN
                  get (archive) ;
                  i := i + 1 END ;
         END ;


   FUNCTION selected (
      candidate_spec : VARYING [n2] OF char ;
      selection_spec : VARYING [n1] OF char := '*') : boolean ;

      BEGIN
         selected :=
            STR$MATCH_WILD (candidate_spec, selection_spec) = STR$_MATCH END ;


   [EXTERNAL] PROCEDURE tar_list (
      list_filespec : VARYING [n1] OF char ;
      archive_filespec : VARYING [n2] OF char ;
      full : boolean) ; external ;

   [EXTERNAL] PROCEDURE tar_extract (
      extract_filespec : VARYING [n1] OF char ;
      archive_filespec : VARYING [n2] OF char ;
      confirm : boolean) ; external ;

   [EXTERNAL] PROCEDURE tar_write (
      write_filespec : VARYING [n1] OF char ;
      archive_filespec : VARYING [n2] OF char := '' ;
      confirm : boolean ;
      scan : boolean ;
      map_mode : map_mode_type ;
      appending : boolean := false) ; external ;

   [EXTERNAL] PROCEDURE tar_append (
      append_filespec : VARYING [n1] OF char ;
      archive_filespec : VARYING [n2] OF char ;
      confirm : boolean ;
      scan : boolean ;
      map_mode : map_mode_type) ; external ;


   FUNCTION get_map_mode : map_mode_type ;
      VAR
         mode_str : small_string ;
      BEGIN
         lib_sigiferr (CLI$GET_VALUE ('MAP_MODE', mode_str)) ;
         CASE mode_str[1] OF
            'P' : get_map_mode := prefix_mode ;
            'A' : get_map_mode := absolute_mode ;
            'R' : get_map_mode := root_mode END END ;


   FUNCTION get_cli_value (
      value_name : VARYING [n1] OF char ;
      value_default : VARYING [n2] OF char := '*') : medium_string ;

      VAR
         return_value : medium_string ;

      BEGIN
         IF cli_present (value_name) THEN BEGIN
            lib_sigiferr (CLI$GET_VALUE (%DESCR value_name, return_value)) ;
            get_cli_value := return_value END
         ELSE
            get_cli_value := value_default ;
         END ;


   BEGIN    { tar }
      establish (tar_handler) ;
      delta_seconds := get_timezone ;		{ Read logical TAR_TIMEZONE }
      IF cli_present (output_kt) THEN BEGIN
         lib_sigiferr (CLI$GET_VALUE (output_kt, output_filespec)) ;
         open (output, output_filespec) ;
         rewrite (output) END ;
      lib_sigiferr (CLI$GET_VALUE ('OPTION', option)) ;
      CASE option[1] OF    { must be an ordinal type }
         'E' : BEGIN
            tar_extract (get_cli_value (filespec_kt), get_cli_value (archive_kt,
               tar_archive_kt), cli_present (confirm_kt)) END ;
         'W' : BEGIN
            tar_write (get_cli_value (filespec_kt),
               get_cli_value (archive_kt, tar_archive_kt),
               cli_present (confirm_kt), cli_present (scan_kt),
               get_map_mode) END ;
         'A' : BEGIN
            tar_append (get_cli_value (filespec_kt),
               get_cli_value (archive_kt, tar_archive_kt),
               cli_present (confirm_kt), cli_present (scan_kt),
               get_map_mode) END ;
         'L' : BEGIN
            tar_list (get_cli_value (filespec_kt), get_cli_value (archive_kt,
               tar_archive_kt), cli_present ('FULL')) END ;
         END ;
      END.
