[INHERIT ('SYS$LIBRARY:STARLET', 'VCDEFS', 'TAR')]

MODULE extract ;

   CONST
      out_buffer_size = 8192 ;

   TYPE
      UNIX_protection_type = [LONG] PACKED RECORD
         others, group, owner : PACKED RECORD
            execute, write, read : [BIT] boolean END END ;
      out_buffer_type = PACKED ARRAY [1..out_buffer_size] OF char ;

   VAR
      out_fab : FAB$TYPE ;
      out_nam : NAM$TYPE ;
      out_xabrdt : XAB$TYPE ;
      out_xabpro : XAB$TYPE ;
      out_rab : RAB$TYPE ;
      default_protection : lib_word_type ;  { from SYS$SETDFPROT }
      output_buffer : out_buffer_type ;
      output_result_spec : PACKED ARRAY [1..NAM$C_MAXRSS] OF char ;

   VALUE
      out_fab := zero ;
      out_nam := zero ;
      out_xabrdt := zero ;
      out_xabpro := zero ;
      out_rab := zero ;


   PROCEDURE open_output_file (
      VAR file_name : VARYING [n1] OF char ;
      file_size : integer) ;

      VAR
         parsed_spec : medium_string ;
         status, secondary_status : sts_type ;

      BEGIN
         out_fab.FAB$B_BID := FAB$C_BID ;
         out_fab.FAB$B_BLN := FAB$C_BLN ;
         out_fab.FAB$B_RFM := FAB$C_VAR ; { Varying-length records }
         out_fab.FAB$V_CR := true ;	{ Carriage-return RAT }
         out_fab.FAB$V_PUT := true ;	{ PUT access }
         out_fab.FAB$V_TEF := true ;	{ Truncate on $CLOSE }
         out_fab.FAB$L_FNA := iaddress (file_name.body) ;
         out_fab.FAB$B_FNS := (file_name.length)::lib_byte_type ;
         IF file_size > 0 THEN
            out_fab.FAB$L_ALQ := (file_size - 1) DIV VMS_block_size + 1 ;

         out_fab.FAB$L_NAM := iaddress (out_nam) ;
         out_nam.NAM$B_BID := NAM$C_BID ;
         out_nam.NAM$B_BLN := NAM$C_BLN ;
         out_nam.NAM$L_RSA := iaddress (output_result_spec) ;
         out_nam.NAM$B_RSS := NAM$C_MAXRSS ;

         status := $CREATE (out_fab) ;

         IF success (status) THEN BEGIN
            out_rab.RAB$B_BID := RAB$C_BID ;
            out_rab.RAB$B_BLN := RAB$C_BLN ;
            out_rab.RAB$B_MBC := tar_multi_block_count ;
               { Multi-block count; specifies how many blocks of a sequential
                 file are transferred per disk access }
            out_rab.RAB$L_FAB := iaddress (out_fab) ;
            file_name := substr (output_result_spec, 1, out_nam.NAM$B_RSL) ;

            status := $CONNECT (out_rab) ;

            IF success (status) THEN BEGIN
               out_rab.RAB$B_RAC := RAB$C_SEQ ;
               out_rab.RAB$L_RBF := iaddress (output_buffer) END
            ELSE
               secondary_status := (out_rab.RAB$L_STV)::sts_type END

         ELSE
            secondary_status := (out_fab.FAB$L_STV)::sts_type ;

         IF failure (status) THEN BEGIN
            parsed_spec := parse (file_name) ;
            LIB$STOP (tar__createrr, 1, %STDESCR (parsed_spec), status,
               secondary_status) END ;

         END ;


   PROCEDURE close_output_file (
      no_records : integer ;
      file_mtime : unsigned ;
      file_mode : [UNSAFE] UNIX_protection_type) ;

      VAR
         created_filespec : medium_string ;
         VMS_protection : file_protection_type ;
         time_vec : lib_numtim_type ;

      BEGIN
         created_filespec :=
            substr (output_result_spec, 1, out_nam.NAM$B_RSL) ;

         out_fab.FAB$L_XAB := iaddress (out_xabrdt) ;

         out_xabrdt.XAB$B_COD := XAB$C_RDT ;
         out_xabrdt.XAB$B_BLN := XAB$C_RDTLEN ;
         out_xabrdt.XAB$L_NXT := iaddress (out_xabpro) ;

   { Here is where the modification date is converted/copied }
         add_timezone (file_mtime) ;
         break_up_UNIX_time (file_mtime, time_vec) ;
         time_vec.year := time_vec.year + 1970 ;
         time_vec.hundredth := 0 ;
         lib_sigiferr (LIB$CVT_VECTIM (time_vec, out_xabrdt.XAB$Q_RDT)) ;

         out_xabpro.XAB$B_COD := XAB$C_PRO ;
         out_xabpro.XAB$B_BLN := XAB$C_PROLEN ;
         lib_sigiferr (LIB$GETJPI (JPI$_UIC,,, out_xabpro.XAB$L_UIC)) ;
         VMS_protection := (default_protection)::file_protection_type ;

   { Here is where the file protection is copied }
         VMS_protection.world.noread := NOT file_mode.others.read ;
         VMS_protection.world.noexecute := NOT file_mode.others.read ;
         VMS_protection.world.nowrite := NOT file_mode.others.write ;

         VMS_protection.group.noread := NOT file_mode.group.read ;
         VMS_protection.group.noexecute := NOT file_mode.group.read ;
         VMS_protection.group.nowrite := NOT file_mode.group.write ;

         VMS_protection.owner.noread := NOT file_mode.owner.read ;
         VMS_protection.owner.noexecute := NOT file_mode.owner.read ;
         VMS_protection.owner.nowrite := NOT file_mode.owner.write ;
         VMS_protection.owner.nodelete := NOT file_mode.owner.write ;

         out_xabpro.XAB$W_PRO := (VMS_protection)::lib_word_type ;

         IF failure ($CLOSE (out_fab)) THEN
            LIB$STOP (tar__close, 1, %STDESCR (created_filespec),
               out_fab.FAB$L_STS, out_fab.FAB$L_STV)
         ELSE
            LIB$SIGNAL (tar__created, 2, %STDESCR (created_filespec),
               no_records) ;
         END ;


   [GLOBAL] PROCEDURE tar_extract (
      extract_filespec : VARYING [n1] OF char ;
      archive_filespec : VARYING [n2] OF char ;
      confirm : boolean) ;

      VAR
         header : boolean ;               { true if current tar_record is one }
         no_records : integer ;           { no of tar records for curr file }
         file_spec : medium_string ;
         file_size : integer ;
         file_mtime : unsigned ;
         file_mode : file_mode_type ;
         bytes_written : integer ;        { bytes written to curr out_file }
         files_created, files_scanned : integer ;
         tar_filespec, upcase_spec : medium_string ;
         more, verbose : boolean ;
         selection : boolean ;
         warned_of_wrap : boolean ;
         absolute : boolean ;
         directory : boolean ;
         protection : unsigned ;          { UNIX style protection }


      PROCEDURE extract_file (
         header_record : tar_record_type ;
         file_spec : VARYING [n1] OF char ;
         file_size : integer ;
         file_mtime : unsigned ;
         file_mode : file_mode_type) ;

         VAR
            more : boolean ;
            VMS_spec : medium_string ;
            absolute : boolean ;


         PROCEDURE write_block (
            block : tar_record_type ;
            VAR bytes_written : integer ;
            VAR no_records : integer) ;

            VAR
               i, status : integer ;
               out_pointer : [STATIC] integer := 0 ;
               more : boolean ;

            BEGIN
               i := 0 ;
               more := true ;
               WHILE more DO BEGIN
                  i := i + 1 ;
                  IF i > record_size THEN more := false
                  ELSE BEGIN
                     bytes_written := bytes_written + 1 ;
                     IF bytes_written = file_size THEN BEGIN    { eof }
                        more := false ;
                        IF block.data[i] <> lf THEN BEGIN
                           out_pointer := out_pointer + 1 ;
                           output_buffer[out_pointer] := block.data[i] END ;
                        out_rab.RAB$W_RSZ := (out_pointer)::lib_word_type ;
                        out_pointer := 0 ;
                        status := $PUT (out_rab) ;
                        IF failure (status) THEN
                           LIB$STOP (tar__errwrite, 1, %STDESCR (VMS_spec),
                              out_rab.RAB$L_STS, out_rab.RAB$L_STV) ;
                        no_records := no_records + 1 END
                     ELSE IF block.data[i] = lf THEN BEGIN        { eoln }
                           out_rab.RAB$W_RSZ := (out_pointer)::lib_word_type ;
                           out_pointer := 0 ;
                           status := $PUT (out_rab) ;
                           IF failure (status) THEN
                              LIB$STOP (tar__errwrite, 1, %STDESCR (VMS_spec),
                                 out_rab.RAB$L_STS, out_rab.RAB$L_STV) ;
                           no_records := no_records + 1 END
                        ELSE IF out_pointer < out_buffer_size THEN BEGIN
                              out_pointer := out_pointer + 1 ;
                              output_buffer[out_pointer] := block.data[i] END
                           ELSE BEGIN              { wrap output records }
                              out_rab.RAB$W_RSZ :=
                                 (out_pointer)::lib_word_type ;
                              status := $PUT (out_rab) ;
                              IF failure (status) THEN
                                 LIB$STOP (tar__errwrite, 1,
                                    %STDESCR (VMS_spec), out_rab.RAB$L_STS,
                                    out_rab.RAB$L_STV) ;
                              out_pointer := 1 ;
                              no_records := no_records + 1 ;
                              output_buffer[out_pointer] := block.data[i] ;
                              IF NOT warned_of_wrap THEN BEGIN
                                 LIB$SIGNAL (tar__wrapped, 2,
                                    %STDESCR (VMS_spec), out_buffer_size) ;
                                 warned_of_wrap := true END ;
                              END ;
                     END ;
                  END ;  { WHILE more }
               END ;  { write_block }


         BEGIN    { extract_file }
            IF file_size = 0 THEN BEGIN
               CASE archive^.linkflag OF
                  '1' :
                     LIB$SIGNAL (tar__hardlink, 1,
                        %STDESCR (file_spec)) ;
                  '2' :
                     LIB$SIGNAL (tar__softlink, 1,
                        %STDESCR (file_spec)) ;
                  OTHERWISE BEGIN
                     VMS_spec := VMS_filespec (file_spec, absolute) ;
                     open_output_file (VMS_spec, file_size) ;
                     no_records := 0 END ;
                  END END
            ELSE BEGIN		{ this is done after each header }
               VMS_spec := VMS_filespec (file_spec, absolute) ;
               no_records := 0 ;
               bytes_written := 0 ;
               open_output_file (VMS_spec, file_size) ;
               more := true ;
               WHILE more DO BEGIN
                  get (archive) ;
                  write_block (archive^, bytes_written, no_records) ;
                  IF bytes_written = file_size THEN
                     more := false END END ;
            close_output_file (no_records, file_mtime, file_mode) END ;


      PROCEDURE make_directory (
         file_spec : medium_string ;
         file_mode : file_mode_type) ;

         VAR
            temp_spec : medium_string ;
            absolute : boolean ;
            status : sts_type ;

         BEGIN
            temp_spec := VMS_filespec (file_spec + 'place.holder', absolute) ;
            temp_spec := parse (temp_spec, '[]') ;
            temp_spec := substr (temp_spec, 1, index (temp_spec, ']')) ;
            create_directory (temp_spec, status) ;
            IF failure (status) THEN
               LIB$STOP (tar__errcredir, 1, %STDESCR (temp_spec), status)
            ELSE
               LIB$SIGNAL (tar__createdir, 1, %STDESCR (temp_spec)) END ;


      BEGIN    { extract }
         open_archive_input (archive_filespec) ;
         full_archive_spec := find_file_spec (archive) ;
         selection := extract_filespec <> '*' ;
         lib_sigiferr (SYS$SETDFPROT (, default_protection)) ;
         files_created := 0 ;
         files_scanned := 0 ;
         header := true ;
         more := true ;
         WHILE more DO BEGIN
            warned_of_wrap := false ;
            IF valid_header (archive^, file_spec, file_size, file_mtime,
                  file_mode, directory) THEN BEGIN
               files_scanned := files_scanned + 1 ;
               IF selected (file_spec, extract_filespec) THEN
                  IF confirm_operation (option, file_spec,
                        (NOT confirm) OR directory) THEN BEGIN
                     IF directory THEN
                        make_directory (file_spec, file_mode)
                     ELSE
                        extract_file (archive^, file_spec, file_size,
                           file_mtime, file_mode) ;
                     files_created := files_created + 1 END
                  ELSE
                     scan_to_next_header (file_size)
               ELSE
                  scan_to_next_header (file_size) ;
               IF eof (archive) THEN
                  more := false
               ELSE
                  get (archive) ;
               END
            ELSE	{ NOT valid_header }
               IF eof_archive THEN
                  more := false
               ELSE
                  bad_header (archive^) END ;   { stops execution }
         LIB$SIGNAL (tar__totcreat, 2, files_created, files_scanned) ;
         close (archive) ;
         END ;    { extract }


   END.
