[INHERIT ('SYS$LIBRARY:STARLET', 'VCDEFS', 'TAR')]

MODULE write ;

   CONST
      in_buffer_size = 8192 ;

   TYPE
      in_buffer_type = PACKED ARRAY [1..in_buffer_size] OF char ;
      filespec_list_type = ARRAY [1..100] OF medium_string ;
      filespec_block_type = RECORD
         file_list : filespec_list_type ;
         next_block : ^filespec_block_type END ;

   VAR
      in_fab : FAB$TYPE ;
      in_xabdat, in_xabfhc, in_xabpro : XAB$TYPE ;
      in_rab : RAB$TYPE ;

      in_buffer : in_buffer_type ;

      file_spec, default_spec, result_spec : VARYING [NAM$C_MAXRSS] OF char ;
      record_format : lib_byte_type ;
      max_record_size, first_free_byte : lib_word_type ;
      eof_block : unsigned ;

      block_pointer : ^filespec_block_type ;
      start_block : ^filespec_block_type ;
      file_index : integer ;
      retrieval_index : integer ;


   VALUE
      in_fab := zero ;
      in_xabdat := zero ;
      in_xabfhc := zero ;
      in_xabpro := zero ;
      in_rab := zero ;
      file_index := 0 ;


   PROCEDURE add_file_to_list (filespec : VARYING [n1] OF char) ;
      BEGIN
         IF file_index = 100 THEN BEGIN
            new (block_pointer^.next_block) ;
            block_pointer := block_pointer^.next_block END
         ELSE IF file_index = 0 THEN BEGIN
            new (block_pointer) ;
            start_block := block_pointer END ;
         file_index := file_index + 1 ;
         block_pointer^.file_list[file_index] := filespec ;
         IF file_index = 100 THEN file_index := 0 END ;


   FUNCTION get_file_from_list : medium_string ;
      VAR
         end_of_list : boolean ;
         return_value : medium_string ;
         temp_pointer : ^filespec_block_type ;

      BEGIN
         end_of_list := false ;
         IF file_index = 0 THEN
            block_pointer := start_block
         ELSE
            IF file_index = 100 THEN
               IF block_pointer^.next_block <> nil THEN BEGIN
                  file_index := 0 ;
                  temp_pointer := block_pointer ;
                  block_pointer := block_pointer^.next_block ;
                  dispose (temp_pointer) END
               ELSE
                  end_of_list := true ;
         IF end_of_list THEN
            return_value := ''
         ELSE BEGIN
            file_index := file_index + 1 ;
            return_value := block_pointer^.file_list[file_index] END ;
         IF return_value = '' THEN
            dispose (block_pointer) ;
         get_file_from_list := return_value END ;


   PROCEDURE collect_filespecs (
      selection_spec : VARYING [n1] OF char ;
      VAR prefix : VARYING [n2] OF char) ;

      VAR
         selection : medium_string ;
         i, j : integer ;
         first_time, more, done : boolean ;
         context : lib_fab_pointer ;
         status : sts_type ;

      BEGIN
         first_time := true ;
         more := true ;
         context := nil ;
         WHILE more DO BEGIN
            status := LIB$FIND_FILE (selection_spec, selection,
               context, %DESCR '.DAT') ;
            IF success (status) THEN BEGIN
               add_file_to_list (selection) ;
               selection := UNIX_filespec (selection, absolute_mode) ;
               IF first_time THEN BEGIN
      { Find a first prefix, but ensure the base file name is'nt part of it }
                  i := length (selection) ;
                  done := false ;
                  WHILE NOT done DO BEGIN
                     IF i = 0 THEN
                        done := true
                     ELSE IF selection[i] = '/' THEN
                        done := true
                     ELSE
                        i := i - 1 END ;
                  prefix := substr (selection, 1, i) ;
                  first_time := false END
               ELSE BEGIN    { find a common prefix to the filenames }
                  j := min (length (selection), length (prefix)) ;
                  i := 1 ;
                  done := false ;
                  WHILE NOT done DO BEGIN
                     IF (i > j) THEN
                        done := true
                     ELSE IF (selection[i] <> prefix[i]) THEN
                        done := true
                     ELSE
                        i := i + 1 END ;
                  prefix.length := i - 1 END END
            ELSE
               IF status = RMS$_FNF THEN
                  LIB$STOP (tar__nofiles, 1, %STDESCR (selection))
               ELSE
                  more := false END ;
         add_file_to_list ('') ;
         file_index := 0 ;

   { Back up enough to ensure that the prefix ends with a '/' }

         i := length (prefix) ;
         done := false ;
         WHILE NOT done DO BEGIN
            IF i = 0 THEN
               done := true
            ELSE IF prefix[i] = '/' THEN
               done := true
            ELSE
               i := i - 1 END ;
         prefix.length := i ;
         LIB$FIND_FILE_END (context) END ;


   FUNCTION open_input_file (
      VAR file_spec : VARYING [n1] OF char ;
      VAR record_format : [UNSAFE] lib_byte_type ;
      VAR record_size : [UNSAFE] lib_word_type ;
      VAR eof_block : [UNSAFE] unsigned ;
      VAR first_free_byte : [UNSAFE] lib_word_type ;
      VAR modification_date : [UNSAFE] lib_date_type ;
      VAR protection : [UNSAFE] lib_word_type) : sts_type ;

      VAR
         status : sts_type ;

      BEGIN
         in_fab.FAB$B_BID := FAB$C_BID ;
         in_fab.FAB$B_BLN := FAB$C_BLN ;
         in_fab.FAB$V_GET := true ;
         in_fab.FAB$V_SHRGET := true ;
         in_fab.FAB$L_XAB := iaddress (in_xabdat) ;
         in_fab.FAB$L_FNA := iaddress (file_spec.body) ;
         in_fab.FAB$B_FNS := (file_spec.length)::lib_byte_type ;

         in_xabdat.XAB$B_COD := XAB$C_DAT ;
         in_xabdat.XAB$B_BLN := XAB$C_DATLEN ;
         in_xabdat.XAB$L_NXT := iaddress (in_xabfhc) ;

         in_xabfhc.XAB$B_COD := XAB$C_FHC ;
         in_xabfhc.XAB$B_BLN := XAB$C_FHCLEN ;
         in_xabfhc.XAB$L_NXT := iaddress (in_xabpro) ;

         in_xabpro.XAB$B_COD := XAB$C_PRO ;
         in_xabpro.XAB$B_BLN := XAB$C_PROLEN ;
 
         status := $OPEN (in_fab) ;

         IF success (status) THEN BEGIN
            record_format := in_fab.FAB$B_RFM ;
            record_size := in_fab.FAB$W_MRS ;
            eof_block := in_xabfhc.XAB$L_EBK ;
            first_free_byte := in_xabfhc.XAB$W_FFB ;
            modification_date := (in_xabdat.XAB$Q_RDT)::lib_date_type ;
            protection := in_xabpro.XAB$W_PRO ;

            in_rab.RAB$B_BID := RAB$C_BID ;
            in_rab.RAB$B_BLN := RAB$C_BLN ;
            in_rab.RAB$B_MBC := tar_multi_block_count ;
               { Multi-block count; specifies how many blocks of a sequential
                 file are read per disk access }
            in_rab.RAB$L_FAB := iaddress (in_fab) ;

            status := $CONNECT (in_rab) ;

            in_rab.RAB$B_RAC := RAB$C_SEQ ;
            in_rab.RAB$L_UBF := iaddress (in_buffer) ;
            CASE record_format OF
               FAB$C_UDF : { undefined, or stream binary }
                  in_rab.RAB$W_USZ := tar_record_size ;
               FAB$C_FIX :
                  IF record_size > in_buffer_size THEN
                     LIB$STOP (tar__openin, 1, %STDESCR (file_spec),
                        tar__rectoolong, 1, (record_size)::unsigned)
                  ELSE
                     in_rab.RAB$W_USZ := record_size ;
               FAB$C_VAR, FAB$C_VFC, FAB$C_STM, FAB$C_STMLF, FAB$C_STMCR :
                  in_rab.RAB$W_USZ := size (in_buffer) ;
               END END ;

         open_input_file := status END ;


   FUNCTION VMS_to_UNIX_protection (VMS_protection : file_protection_type)
      : unsigned ;
      VAR
         return : unsigned ;

      BEGIN
         WITH VMS_protection DO BEGIN
            IF NOT owner.noread THEN return := 256
            ELSE return := 0 ;
            IF NOT owner.nowrite THEN return := return + 128 ;
{           IF NOT owner.noexecute THEN return := return + 64 ; }
            IF NOT group.noread THEN return := return + 32 ;
            IF NOT group.nowrite THEN return := return + 16 ;
{           IF NOT group.noexecute THEN return := return + 8 ; }
            IF NOT world.noread THEN return := return + 4 ;
            IF NOT world.nowrite THEN return := return + 2 ;
{           IF NOT world.noexecute THEN return := return + 1 ; }
            END ;
         VMS_to_UNIX_protection := return END ;


   PROCEDURE build_header (
      head_filespec : VARYING [n1] OF char ;
      filesiz : integer ;
      VAR out_header : tar_record_type ;
      map_mode : map_mode_type ;
      modification_date : lib_date_type ;
      protection : file_protection_type ;
      prefix_length : integer := 0) ;

      VAR
         i : integer ;
         temp_filespec : medium_string ;
         temp_string : small_string ;

      BEGIN
         out_header := default_header ;
         temp_string := oct (VMS_to_UNIX_protection (protection), 6, 3) ;
         out_header.mode.value := temp_string ;
         temp_string := oct (filesiz, 11, 1) + space ;
         out_header.siz := temp_string ;
         temp_string :=
            oct (VMS_to_UNIX_time (modification_date, delta_seconds), 11, 1) +
            space ;
         out_header.mtime := temp_string ;
         temp_filespec := UNIX_filespec (head_filespec, map_mode,
            prefix_length) ;
         FOR i := 1 to length (temp_filespec) DO
            out_header.name[i] := temp_filespec[i] ;
         { I had to do that, to prevent Pascal from blank-padding it }
         out_header.chksum.value := '      ' ;
         out_header.chksum.fill_null := space ;
         out_header.chksum.fill_space := space ;
         writev (temp_string, oct (checksum (out_header), 6, 1)) ;
         out_header.chksum.value := temp_string ;
         out_header.chksum.fill_space := null ; { That's how DYNIX tar does it}
         END ;


   PROCEDURE write_archive (
      VAR file_buf : tar_file_type ;
      VAR out_record : in_buffer_type ;
      VAR record_length : [UNSAFE] lib_word_type ;
      add_lf : boolean := true) ;

      VAR
         i : lib_word_type ;

      BEGIN
         IF add_lf THEN
            IF record_length < in_buffer_size THEN BEGIN
               record_length := record_length + 1 ;
               out_record[record_length] := lf END ;
         FOR i := 1 TO record_length DO BEGIN
            IF last_char = tar_record_size THEN BEGIN
               put (file_buf) ;
               last_char := 1 END
            ELSE
               last_char := last_char + 1 ;
            file_buf^.data[last_char] := out_record[i] END ;
         END ;


   [GLOBAL] PROCEDURE tar_write (
      write_filespec : VARYING [n1] OF char ;
      archive_filespec : VARYING [n2] OF char := '' ;
      confirm : boolean ;
      scan : boolean ;
      map_mode : map_mode_type ;
      appending : boolean := false) ;

      VAR
         not_this_one : boolean ;
         tar_record : tar_record_type ;
         current_file : medium_string ;
         file_size : integer ;
         no_records : integer ;
         files_written : integer ;
         write_message : sts_type ;

         record_format : lib_byte_type ;       { These all correspond to the }
         record_length : lib_word_type ;       {  input file }
         eof_block : integer ;
         first_free_byte : lib_word_type ;
         modification_date : lib_date_type ;
         protection : file_protection_type ;

         status : sts_type ;
         prefix : medium_string ;

      FUNCTION another_file (
         VAR next_file : VARYING [n1] OF char) : boolean ;

         VAR
            context : [STATIC] lib_fab_pointer := NIL ;
            status : sts_type ;

         BEGIN
            IF map_mode = prefix_mode THEN BEGIN
               next_file := get_file_from_list ;
               another_file := next_file <> '' END
            ELSE BEGIN
               another_file := true ;
               status := LIB$FIND_FILE (write_filespec, next_file, context) ;
               IF status <> RMS$_NORMAL THEN BEGIN
                  another_file := false ;
                  IF status <> RMS$_NMF THEN BEGIN
                     IF status = RMS$_FNF THEN BEGIN
                        IF archive_temp_open THEN BEGIN
                           close (archive_temp, disposition := DELETE) ;
                           archive_temp_open := false END ;
                        IF appending THEN BEGIN
                           FOR i := 1 TO record_size DO
                              archive^.data[i] := null ;
                           put (archive) ; put (archive) ; { replace eof-marker }
                           close (archive) END
                        ELSE
                           close (archive, disposition := DELETE) ;
                        LIB$STOP (tar__nofiles, 1, %STDESCR (next_file)) END ;
                     lib_sigiferr (status) END ;
                  lib_sigiferr (LIB$FIND_FILE_END (context)) END END ;
            END ;


      PROCEDURE load_direct ;     { Directly loads a fixed record length }
         BEGIN                    { or stream binary file }
            file_size := (eof_block - 1) * 512 + first_free_byte ;

   { The test below is for cases where the first_free_byte value does not
     point to the first byte in a logical record (I have seen this in .EXE's) }

            IF record_format = FAB$C_FIX THEN
               IF file_size REM record_length > 0 THEN
                  file_size := (file_size DIV record_length + 1) *
                     record_length ;

            build_header (current_file, file_size, archive^, map_mode,
               modification_date, protection, length (prefix)) ;
            put (archive) ;      { writes header }
            status := $GET (in_rab) ;
            WHILE success (status) DO BEGIN
               write_archive (archive, in_buffer, in_rab.RAB$W_RSZ, false) ;
               no_records := no_records + 1 ;
               status := $GET (in_rab) END ;
            IF status <> RMS$_EOF THEN
               LIB$STOP (tar__errread, 3, 0, 0, %STDESCR (current_file),
                  status, in_rab.RAB$L_STV) ;
            IF file_size > 0 THEN               { If non-empty file }
               put (archive) ;                  { like a flush }
            END ;

      PROCEDURE load_after_scan ;   { Scans a var-len file for size then loads }
         BEGIN
            status := $GET (in_rab) ;
            WHILE success (status) DO BEGIN
               file_size := file_size + in_rab.RAB$W_RSZ + 1 ;
               status := $GET (in_rab) END ;
            IF status <> RMS$_EOF THEN
               LIB$STOP (tar__errread, 3, 0, 0, %STDESCR (current_file),
                  status, in_rab.RAB$L_STV) ;

            build_header (current_file, file_size, archive^, map_mode,
               modification_date, protection, length (prefix)) ;
            put (archive) ;      { writes header }

            status := $REWIND (in_rab) ;
            IF failure (status) THEN
               LIB$STOP (tar__errread, 3, 0, 0, %STDESCR (current_file),
                  status, in_rab.RAB$L_STV) ;

            status := $GET (in_rab) ;
            WHILE success (status) DO BEGIN
               write_archive (archive, in_buffer, in_rab.RAB$W_RSZ) ;
               no_records := no_records + 1 ;
               status := $GET (in_rab) END ;
            IF status <> RMS$_EOF THEN
               LIB$STOP (tar__errread, 3, 0, 0, %STDESCR (current_file),
                  status, in_rab.RAB$L_STV) ;
            IF file_size > 0 THEN               { If non-empty file }
               put (archive) ;

            END ;

      PROCEDURE load_from_temp ;
         BEGIN
            IF NOT archive_temp_open THEN BEGIN
               open (archive_temp, history := NEW) ;
               archive_temp_open := true END ;
            rewrite (archive_temp) ;

            status := $GET (in_rab) ;
            WHILE success (status) DO BEGIN
               file_size := file_size + in_rab.RAB$W_RSZ + 1 ;
               write_archive (archive_temp, in_buffer, in_rab.RAB$W_RSZ) ;
               status := $GET (in_rab) END ;
            IF status <> RMS$_EOF THEN
               LIB$STOP (tar__errread, 3, 0, 0, %STDESCR (current_file),
                  status, in_rab.RAB$L_STV) ;
            IF file_size > 0 THEN               { If non-empty file }
               put (archive_temp) ;

            reset (archive_temp) ;
            build_header (current_file, file_size, archive^, map_mode,
               modification_date, protection, length (prefix)) ;
            put (archive) ;      { writes header }
            WHILE NOT eof (archive_temp) DO BEGIN
               read (archive_temp, tar_record) ;
               write (archive, tar_record) END ;
            END ;

      BEGIN    { tar_write }
         IF NOT appending THEN BEGIN
            creating_archive_output := true ;
            open (archive, archive_filespec, history := NEW) ;
            rewrite (archive) ;
            creating_archive_output := false ;
            write_message := tar__written END
         ELSE BEGIN
            truncate (archive) ;
            write_message := tar__appended END ;
         files_written := 0 ;
         prefix := '' ;
         IF map_mode = prefix_mode THEN
            IF (index (write_filespec, ':[') = 0)
                  AND (index (write_filespec, ']') = 0) THEN
               map_mode := single_dir_mode
            ELSE
               collect_filespecs (write_filespec, prefix) ;
         WHILE another_file (current_file) DO
            IF confirm_operation (option, current_file, NOT confirm) THEN BEGIN
               open_input_file (current_file, record_format, record_length,
                  eof_block, first_free_byte, modification_date, protection) ;
               last_char := 0 ;
               no_records := 0 ;
               file_size := 0 ;
               IF record_format IN [FAB$C_FIX, FAB$C_UDF] THEN
                  load_direct
               ELSE
                  IF scan THEN
                     load_after_scan
                  ELSE
                     load_from_temp ;
               files_written := files_written + 1 ;
               LIB$SIGNAL (write_message, 2, %STDESCR (current_file),
                  file_size) ;
               { file built and written }
               status := $CLOSE (in_fab) ;
               IF failure (status) THEN
                  LIB$STOP (tar__close, 1, %STDESCR (current_file),
                     status, in_fab.FAB$L_STV) ;
               END ;  { IF confirmed }

         IF appending THEN
            LIB$SIGNAL (tar__totappend, 1, files_written)
         ELSE
            LIB$SIGNAL (tar__totwrite, 1, files_written) ;
         FOR i := 1 TO record_size DO
            archive^.data[i] := null ;
         put (archive) ; put (archive) ;     { emulate tar's personal eof }
         IF archive_temp_open THEN BEGIN
            close (archive_temp, disposition := DELETE) ;
            archive_temp_open := false END ;
         close (archive) ;
         END ;    { tar_write }


   [GLOBAL] PROCEDURE tar_append (
      append_filespec : VARYING [n1] OF char ;
      archive_filespec : VARYING [n2] OF char ;
      confirm : boolean ;
      scan : boolean ;
      map_mode : map_mode_type) ;

      VAR
         archive_size : integer ;
         record_format : fab_rfm_type ;
         max_record_size : lib_word_type ;
         end_position : integer ;

      BEGIN
         open_archive_input (archive_filespec) ;
         full_archive_spec := parse (archive_filespec, '.DAT') ;
         lib_sigiferr (lib_file_attributes (, archive, lib_item_list (
            lib_out_item (rms__ebk, %DESCR archive_size),
            lib_out_item (rms__rfm, %DESCR record_format),
            lib_out_item (rms__mrs, %DESCR max_record_size)))) ;
         IF (record_format <> fixed_length) OR (max_record_size <> record_size)
               OR (archive_size < 3) THEN
            LIB$STOP (tar__badarchive) ;
         FOR i := 1 TO archive_size - 3 DO
            get (archive) ;
         IF checksum (archive^) <> 0 THEN
            LIB$STOP (tar__badarchive) ;
         tar_write (append_filespec,, confirm, scan, map_mode,
            appending := true) ;
         END ;    { tar_append }


   END.
