*=======================================================================
*     
*  Title:        VAXStation Clock
*                
*  Version:      1-001
*                
*  Abstract:     Creates a realtime clock on the VAXStation as a 
*                separate window.  The clock is updated each minute
*                and can be moved around like any other UIS created
*                window.
* 
*                By creating a detached process running this image,
*                the clock can be a constant part of the VAXStation
*                environment.
*                
*  Environment:  MicroVMS on VAXStation with UIS routines
*                
*  Author:       Eric Andresen of General Research Corporation
*                
*  Date:         15-JUL-1986
*                
*                       C H A N G E   L O G 
*
*     Date     | Name | Description
*--------------+------+-------------------------------------------------
*  15-JUL-1986   ETA    Original Version
*-----------------------------------------------------------------------

      program VAXStation_clock

      implicit none

      parameter window_width = 5.
      parameter window_height = 5.

      include 'sys$library:uisentry'
      include 'sys$library:uisusrdef'

      structure /loc_type/
         integer*4  code_1
         integer*4  attr
         integer*4  code_2
         integer*4  rel_pos
         integer*4  end_list
      end structure

      record /loc_type/ location

      character  ahours*2
      real       hour_x(60), hour_y(60), x(60), y(60)
      real       min_x(60), min_y(60), tick_x(60), tick_y(60)
      real       theta, radius, tx(12), ty(12)
      real       min_r, hour_r, tick_r, t_height, t_width
      integer    ii, kk, hours, mins, vd_id, wd_id, secs, mins_elapsed

C     THE LOCATION OF THE NUMBERS IS STORED IN THESE ARRAYS
C     THEIR POSITION TURNED OUT TO BE AN AESTHETIC PROBLEM NOT
C     A MATHEMATICAL ONE AND I CHOSE TO STORE THE X,Y VALUES
C     INSTEAD OF FIGURING THEM OUT

      data tx / 0.45, 0.75, 0.85, 0.73, 0.43, -0.04, -0.48, -0.83,
     .         -0.94, -0.87, -0.55, -0.08 /

      data ty / 0.87, 0.55, 0.11, -0.32, -0.68, -0.8, -0.68,
     .         -0.32, 0.11, 0.55, 0.87, 0.99 /

C     FIGURE ALL OF THE RADII

      radius = .8              ! RADIUS OF THE CLOCK CIRCLE
      tick_r = radius - .05    ! RADIUS FOR MINUTE TICKS
      min_r =  radius - .1     ! RADIUS FOR HOUR TICKS AND MINUTE HAND
      hour_r = radius - .25    ! RADIUS FOR HOUR HAND

C     STORE ALL OF THE x,y DATA SO WE ONLY HAVE TO FIGURE THIS ONCE
      do ii = 1, 30
         theta = 90. - (6. * ii)
         if ( theta .lt. 0 ) theta = 360 + theta

C        FIGURE THE x,y COORDINATES OF THE CURRENT QUADRANT
         x(ii) = radius * cosd( theta )
         y(ii) = radius * sind( theta )
         min_x(ii) = min_r * cosd( theta )
         min_y(ii) = min_r * sind( theta )
         hour_x(ii) = hour_r * cosd( theta )
         hour_y(ii) = hour_r * sind( theta )
         tick_x(ii) = tick_r * cosd( theta )
         tick_y(ii) = tick_r * sind( theta )

C        DO THE OPPOSITE QUADRANT
         kk = ii + 30
         x(kk) = -x(ii)
         y(kk) = -y(ii)
         min_x(kk) = -min_x(ii)
         min_y(kk) = -min_y(ii)
         hour_x(kk) = -hour_x(ii)
         hour_y(kk) = -hour_y(ii)
         tick_x(kk) = -tick_x(ii)
         tick_y(kk) = -tick_y(ii)
      end do      

C     CREATE THE DISPLAY (NOT ON THE SCREEN AT THIS POINT)
      vd_id = uis$create_display( -1., -1., 1., 1.,
     .                            window_width, window_height )

C     SET UP AN ITEM LIST FOR THE CREATE WINDOW ROUTINE
      location.code_1 =   WDPL$C_ATTRIBUTES
      location.attr =     WDPL$M_NOMENU_ICON
      location.code_2 =   WDPL$C_PLACEMENT
      location.rel_pos =  WDPL$M_TOP .or. WDPL$M_RIGHT
      location.end_list = WDPL$C_END_OF_LIST

C     PUT THE WINDOW UP ON THE SCREEN
      wd_id = uis$create_window( vd_id, 'sys$workstation', '    CLOCK',
     .                           ,,,,,, location )

C     USE THE SMALLEST FONT
      call uis$set_font( vd_id, 0, 1, 'font1' )

C     SET A SECOND LINE WIDTH FOR THE MINUTE HAND AND THE HOUR HAND
      call uis$set_line_width( vd_id, 0, 2, 2., UIS$C_WIDTH_PIXELS )

C     SET AN ATTRIBUTE FOR ERASING THE CLOCK HANDS
      call uis$set_writing_mode( vd_id, 0, 3, UIS$C_MODE_ERAS )
      call uis$set_writing_mode( vd_id, 2, 4, UIS$C_MODE_ERAS )

C     ATTRIBUTE #:
C        0  normal
C        1  small font
C        2  double line width
C        3  erase normal line
C        4  erase doule line width

C     KEEP THE CLOCK RUNNING FOREVER AND EVER
      do while ( .true. ) !forever

C     CREATE THE CLOCK FACE
      call uis$circle( vd_id, 0, 0., 0., radius )

C     PUT THE HOUR TICKS AND THE NUMBERS OUT THERE
      do hours = 1, 12
         mins = hours * 5
         if ( hours .le. 9 ) then
            write( ahours, fmt='(i1)' ) hours
            else
            write( ahours, fmt='(i2)' ) hours
            end if
         call uis$text( vd_id, 1, ahours, tx(hours), ty(hours) )
         call uis$plot( vd_id, 0, x(mins), y(mins), 
     .                  tick_x(mins), tick_y(mins) )
      end do

      mins_elapsed = 0

C     UPDATE THE CLOCK HANDS EVERY 1/4 DAY - THEN WE'LL REDRAW THE 
C     WHOLE CLOCK AGAIN - THIS IS BECAUSE THE HANDS ARE ONLY WRITTEN
C     OVER, NOT ERASED FROM VIRTUAL MEMORY.  THE PROCESS RUNNING THIS
C     IMAGE WOULD FINALLY EXCEED VIRTUAL PAGE COUNT FROM ALL OF THE 
C     HANDS BEING DRAWN.  THE ERASE VD WILL DELETE ALL OF THE HANDS.
      do while ( mins_elapsed .lt. 360 )

         call get_time( hours, mins, secs )

C        DECIDE THE POSITION OF THE MINUTE AND HOUR HANDS
         hours = (hours * 5) + (mins / 12)
         if ( hours .gt. 60 ) hours = hours - 60
         if ( mins .eq. 0 ) mins = 60

C        PLOT THE HANDS
         call uis$plot( vd_id, 2, 0., 0., tick_x(mins), tick_y(mins) )
         call uis$plot( vd_id, 2, 0., 0., hour_x(hours), hour_y(hours) )

C        MAKE A CLICK SOUND
         call uis$sound_click( 'sys$workstation', 7 )

C        WAIT FOR THE NEXT MINUTE
         call lib$wait( 60.-secs )

C        ERASE THE CURRENT HANDS
         call uis$plot( vd_id, 4, 0., 0., tick_x(mins), tick_y(mins) )
         call uis$plot( vd_id, 4, 0., 0., hour_x(hours), hour_y(hours) )

         mins_elapsed = mins_elapsed + 1

      end do !mins_elapsed < 3600

      call uis$erase( vd_id ) ! ERASE THE CLOCK FROM THE VIRTUAL DISPLAY

      end do !forever

      end

      subroutine get_time( hours, minutes, secs )

      implicit   none

      integer    hours, minutes, secs
      character  time*23

      call lib$date_time( time )

      read( time(13:14), fmt='(i2)' ) hours
      read( time(16:17), fmt='(i2)' ) minutes
      read( time(19:20), fmt='(i2)' ) secs

      if ( hours .eq.  0 ) hours = 12
      if ( hours .gt. 12 ) hours = hours - 12

      return
      end
