/* <gcc.c>
 *
 *	A DCL driver for GNU "CC".  It accepts a CLD syntax compatable
 *	with Digital's "VAX C".
 *
 *	Since GCC consists of more than 1 pass we have to do the parsing
 *	here and have a DCL Command (.COM) file do the real work -- so that
 *	multiple images can be invoked WITHOUT having to resort to LIB$SPAWN.
 */
#define GCC_DRIVER_VERSION "2.3.0"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#undef EXIT_FAILURE
#define EXIT_FAILURE 0x1000002C		/* SS$_ABORT|STS$M_INHIB_MSG */

#define MYBUFSIZ 256			/* imposed by lib$set_symbol */

#define LOCAL  0
#define GLOBAL 1
#define VERB_STAT 1	/* verbose */
#define VERB_ECHO 2	/* verify */
#define VERB_SILENT 256 /* suppress GCC-I-foo messages */
#define PROF_FUNC 1	/* function profiling */
#define PROF_BLOC 2	/* block profiling */

enum lang_num {
	lang__default = -1,
	lang_C,
	lang_Cplusplus,
	lang_ObjC,
#if 0
	lang_Chill,
	lang_Fortran,
	lang_Ada,
#endif
	lang_Assembly
};

static char last_resort_basename[] = "_noname_";

static char *Input_File =0;		/* Input file specifications */
static char *Object =0;			/* Object file name	     */
static char *Assembly_File =0;		/* Assembly file name	     */
static char *List_File =0;		/* List file name	     */
static char *Basename =0;		/* Base portion of input name */
static char *Debug =0;			/* /DEBUG specification      */
static char *Target_Machine =0;		/* For cross-compiling	     */
static char *GCC_Version_String =0;	/* Version of gcc we are running */
static int GCC_Version=0;		/* Major version number (1 or 2) */
static int Gas_Version=0;		/* ditto */
static int Optimize=0;			/* Do optimized compile      */
static int G_Float=0;			/* Use G double precision floating */
static int Machine_Code=0;		/* Output assembly lang      */
static int Verbose=0;			/* Make noise while compiling*/
static int Plus=0;			/* Use C++ compiler	     */
static int Show=0;			/* Show options		     */
static int Profile=0;			/* Use compiler profiler     */
static int Warn=0;			/* Generate warning messages */
static int Case_Hack=1;			/* Hack symbols with upper case*/
static int List=0;			/* Save preprocessor output  */
static int Version=0;			/* Show compiler version number */
static int Generate_Object=0;		/* Should we generate obj?   */
static int Standard=0;			/* traditional vs normal vs pedantic */
static int failsafe = 0;		/* flag for overlooking CLD problems */
static enum lang_num Lang;		/* language to compile under */
/*
 * DCL has a limit of 255 characters for a symbol value not obtained via READ.
 */
static char cpp_Options[MYBUFSIZ];	/* Options for the CPP pass  */
static char cc1_Options[MYBUFSIZ];	/* Options for the CC1 pass  */
static char gas_Options[MYBUFSIZ];	/* Options for the GAS pass  */

typedef unsigned long vms_cond;		/* condition code */
typedef unsigned short Uword;		/* unsigned "word", string length */
/* simplified string descriptor, leaving type and class 0 */
typedef struct dsc { Uword len, mbz; char *adr; } Descr;

static int setup_parse(void), cli_negated(void);
static int cli_present(const char *,int,int);
static int cli_get_value(const char *,char *,int);
static int lib_do_command(const char *,int);
static int lib_set_symbol(const char *,const char *);
static void choke(char *);
static void fixup_filename(char **,const char *,int);
static void set_basename(char **);

extern int getpid(void);
extern vms_cond cli$present(const Descr *);
extern vms_cond cli$get_value(const Descr *,Descr *,Uword *);
extern vms_cond cli$dcl_parse(const Descr *,const void *,
			      vms_cond (*f1)(Descr *,const Descr *,Uword *),
			      vms_cond (*f2)(Descr *,const Descr *,Uword *),
			      const Descr *);
extern vms_cond lib$do_command(const Descr *);
extern vms_cond lib$set_symbol(const Descr *,const Descr *,const int *);
extern vms_cond lib$delete_symbol(const Descr *,const int *);
extern vms_cond lib$get_input(Descr *,const Descr *,Uword *);

		/* - assorted utility routines - */

/*
 *	Save a string in dynamic memory
 */
static char *savestr( const char *String )
{
    return strcpy((char *)malloc(strlen(String)+1), String);
}

/*
 *	Append one strings to another, with bounds check.
 */
static void cat( char *string, const char *append )
{
    register char *dst;
    register const char *src;
    unsigned oldlen, newlen;

    oldlen = strlen(string);
    newlen = oldlen + strlen(append);
    if (newlen >= MYBUFSIZ) choke(string);

    dst = string + oldlen;			/* point to nul */
    for (src = append; *src; )
	*dst++ = *src++;
    *dst = '\0';				/* terminate the string */
}

/*
 *	Append some strings to another, doubling quote chars as encountered.
 *	The prefix and suffix arguments are taken as is, no quote handling.
 */
static void q_cat( char *string,
		   const char *prefix,
		   const char *append,
		   const char *suffix )
{
    register char *dst;
    register const char *src;
    unsigned oldlen, newlen;

    oldlen = strlen(string);
    newlen = oldlen;
    if (prefix) newlen += strlen(prefix);
    if (append)
	for (src = append; *src; ++newlen)
	    if (*src++ == '\"') ++newlen;
    if (suffix) newlen += strlen(suffix);
    /* bounds check hack */
    if (newlen >= MYBUFSIZ) choke(string);

    dst = string + oldlen;			/* point to nul */
    if (prefix)
	for (src = prefix; *src; )
	    *dst++ = *src++;
    if (append)
	for (src = append; *src; ) {
	    if (*src == '\"') *dst++ = '\"';	/* double quotes */
	    *dst++ = *src++;
	}
    if (suffix)
	for (src = suffix; *src; )
	    *dst++ = *src++;
    *dst = '\0';				/* terminate the string */
}

/*
 *	Search one string for another
 */
static char *locate( const char *source, const char *target )	/* strstr() */
{
    register const char *pnt = source;
    while (*pnt) {
	if (*pnt == *target && !strncmp(pnt,target,strlen(target)))
	    return (char *)pnt;
	pnt++;
    }
    return (char *)0;
}

/*
 *	Bail out
 */
static void choke( char *buffer )
{
    const char *tag = (buffer == cpp_Options) ? "gcc-cpp" :
		      (buffer == cc1_Options) ? "gcc-cc1" :
		      (buffer == gas_Options) ? "gcc-as"  : "gcc";

    fprintf(stderr,
	    "%%GCC-F-OVERFLOW, `%s' command too long; unable to compile\n",
	    tag);
    exit(EXIT_FAILURE);
    /*NOTREACHED*/
}

		/* - command parsing code - */

/*
 *	Try to figure which compiler to use.
 */
static void identify_language ( void )
{
    failsafe++;
    Lang = lang__default;
    if (cli_present("LANGUAGE",0,GLOBAL)) {
	if (cli_present("LANGUAGE.C",0,GLOBAL))
	    Lang = lang_C;
	else if (cli_present("LANGUAGE.CPLUSPLUS",0,GLOBAL)
	      || cli_present("LANGUAGE.CXX",0,GLOBAL))
	    Lang = lang_Cplusplus;
	else if (cli_present("LANGUAGE.OBJECTIVE_C",0,GLOBAL)
	      || cli_present("LANGUAGE.OBJC",0,GLOBAL)
	      || cli_present("LANGUAGE.OBJ",0,GLOBAL))
	    Lang = lang_ObjC;
#if 0
	else if (cli_present("LANGUAGE.CHILL",0,GLOBAL))
	    Lang = lang_Chill;
	else if (cli_present("LANGUAGE.FORTRAN",0,GLOBAL))
	    Lang = lang_Fortran;
	else if (cli_present("LANGUAGE.ADA",0,GLOBAL))
	    Lang = lang_Ada;
#endif
	else if (cli_present("LANGUAGE.ASSEMBLY",0,GLOBAL))
	    Lang = lang_Assembly;
    } else if (cli_present("PLUS_PLUS",0,GLOBAL)) {
	Lang = lang_Cplusplus;
    }
    --failsafe;
    return;
}

/*
 *	Do the parsing
 */
static void checkswitches( int flag )
{
    char Temp[MYBUFSIZ];

    /*
     *	/CPP_OPTIONS=arbitrary_string
     */
    failsafe++;
    if (cli_present("CPP_OPTIONS",0,flag)) {
	cli_get_value("CPP_OPTIONS",Temp,sizeof Temp);
	q_cat(cpp_Options, Temp, (char *)0, " ");
    }
    --failsafe;
    /*
     *	/CC1_OPTIONS=arbitrary_string
     */
    if (cli_present("CC1_OPTIONS",0,flag)) {
	cli_get_value("CC1_OPTIONS",Temp,sizeof Temp);
	q_cat(cc1_Options, Temp, (char *)0, " ");
    }
    /*
     *	/GAS_OPTIONS=arbitrary_string
     */
    failsafe++;
    if (cli_present("GAS_OPTIONS",0,flag)) {
	cli_get_value("GAS_OPTIONS",Temp,sizeof Temp);
	q_cat(gas_Options, Temp, (char *)0, " ");
    }
    --failsafe;
    /*
     *	/[NO]OBJECT[=file]
     */
    if ((Generate_Object = cli_present("OBJECT",Generate_Object,flag)) != 0) {
	if (Object) free(Object);
	if (cli_get_value("OBJECT",Temp,sizeof Temp))
	    Object = savestr(Temp);
	else
	    Object = 0;
    }
    /*
     *	/[NO]MACHINE_CODE[=file]
     *	    used to save the ".s" assembler intermediate output
     */
    if ((Machine_Code = cli_present("MACHINE_CODE",Machine_Code,flag)) != 0) {
	if (Assembly_File) free(Assembly_File);
	if (cli_get_value("MACHINE_CODE",Temp,sizeof Temp))
	    Assembly_File = savestr(Temp);
	else
	    Assembly_File = 0;
    }
    /*
     *	/[NO]LIST[=file], /PROCESS_ONLY[=file]
     */
    if ((List = cli_present("LIST",List,flag)) != 0
     || (List = cli_present("PREPROCESS_ONLY",List,flag)) != 0) {
	if (List_File) free(List_File),  List_File = 0;
	if (cli_get_value("LIST",Temp,sizeof Temp))
	    List_File = savestr(Temp);
	else if (cli_get_value("PREPROCESS_ONLY",Temp,sizeof Temp))
	    List_File = savestr(Temp);
    }
    /* this needs to be done after /object has been checked */
    if (cli_present("PREPROCESS_ONLY",0,flag)) {
	Generate_Object = Machine_Code = 0;
	cat(cpp_Options, "\"-E\" ");
    }
    /*
     *	/TARGET=[host]
     *	    used for cross-compilation.
     */
    if (cli_present("TARGET",0,flag)) {
	if (Target_Machine) free(Target_Machine);
	if (cli_get_value("TARGET",Temp,sizeof Temp))
	    Target_Machine = savestr(Temp);
	else
	    Target_Machine = 0;
    }

    /*
     *	/INCLUDE_DIRECTORY=(dir1[,dir2,...])
     */
    if (cli_present("INCLUDE_DIRECTORY",0,flag)) {
	while (cli_get_value("INCLUDE_DIRECTORY",Temp,sizeof Temp))
	    q_cat(cpp_Options, "\"-I", Temp, "\" ");
    }
    /*
     *	/UNDEFINE=(macro1[,macro2,...])
     *	    used to cancel built-in macros
     */
    if (cli_present("UNDEFINE",0,flag)) {
	while (cli_get_value("UNDEFINE",Temp,sizeof Temp))
	    q_cat(cpp_Options, "\"-U", Temp, "\" ");
    }
    /*
     *	/DEFINE=(macro1[=value1][,macro2[=value2],...])
     */
    if (cli_present("DEFINE",0,flag)) {
	while (cli_get_value("DEFINE",Temp,sizeof Temp))
	    q_cat(cpp_Options, "\"-D", Temp, "\" ");
    }
    /*
     *	/SCAN=(file1[,file2,...])
     */
    if (cli_present("SCAN",0,flag)) {
	while (cli_get_value("SCAN",Temp,sizeof Temp))
	    q_cat(cpp_Options,
		  (GCC_Version == 1 ? "\"-i" : "\"-imacros\" "),
		  Temp, (GCC_Version == 1 ? "\" " : " "));
    }

    /*
     *	/DEBUG[=ALL|NONE|([NO]TRACEBACK,[NO]SYMBOLS)
     *	    note: only /NODEBUG (the default) and /DEBUG=ALL are supported.
     */
    if (cli_present("DEBUG",0,flag)) {
	/* Get the value (Default = ALL), and save it */
	if (!cli_get_value("DEBUG",Temp,sizeof Temp))
	    strcpy(Temp,"ALL");
	Debug = savestr(Temp);
    }
    /*
     *	/[NO]OPTIMIZE[=anything...]
     */
    if (cli_present("OPTIMIZE",Optimize,flag)) {
	if (!cli_get_value("OPTIMIZE",Temp,sizeof Temp))
	    Optimize = 1;
	else {
	    register char *p;
	    for (p = Temp; *p; p++)
		if (*p < '0' || *p > '9') break;
	    if (!*p)  Optimize = atoi(Temp);
	}
    } else if (cli_negated())
	Optimize = 0;
    /*
     *	/[NO]WARNINGS[=ALL|NONE|([NO]INFORMATIONALS,[NO]WARNINGS)]
     *	    note:  values are ignored
     */
    Warn = cli_present("WARNINGS",Warn,flag);
    /*
     *	/VERSION
     */
    Version = cli_present("VERSION",Version,flag);
    /*
     *	/[NO]G_FLOAT
     */
    G_Float = cli_present("G_FLOAT",G_Float,flag);
    /*
     *	/[NO]VERBOSE
     */
    if (cli_present("VERBOSE",0,flag)) {
	if ((!cli_get_value("VERBOSE",Temp,sizeof Temp))
	 || cli_present("VERBOSE.ALL",0,flag))
	    Verbose = VERB_STAT|VERB_ECHO;	/* all */
	else {
	    if (cli_present("VERBOSE.STATISTICS",0,flag)) Verbose |= VERB_STAT;
	    if (cli_present("VERBOSE.ECHO",0,flag)
	     || cli_present("VERBOSE.VERIFY",0,flag)) Verbose |= VERB_ECHO;
	}
    } else if (cli_negated())
	Verbose = VERB_SILENT;
    /*
     *	/PROFILE[=ALL|BLOCK|FUNCTION]
     */
    if (cli_present("PROFILE",0,flag)) {
	if (!cli_get_value("PROFILE",Temp,sizeof Temp))
	    Profile = PROF_FUNC;	/* function */
	else {
	    if (cli_present("PROFILE.ALL",0,flag))
		Profile = PROF_FUNC|PROF_BLOC;
	    if (cli_present("PROFILE.BLOCK",0,flag)) Profile |= PROF_BLOC;
	    if (cli_present("PROFILE.FUNCTION",0,flag)) Profile |= PROF_FUNC;
	}
    }
    /*
     *	/SHOW[=ALL|RULES|DEFINITIONS]
     */
    if (cli_present("SHOW",0,flag)) {
	if (!cli_get_value("SHOW",Temp,sizeof Temp))
	    Show = 3;		/* all */
	else {
	    if	    (cli_present("SHOW.ALL",0,flag)) Show = 3;
	    else if (cli_present("SHOW.RULES",0,flag)) Show = 2;
	    else if (cli_present("SHOW.DEFINITIONS",0,flag)) Show = 1;
	    else if (cli_present("SHOW.NONE",0,flag)) Show = 0;
	}
    }
    /*
     *	/NAMES={UPPER|LOWER|MIXED|HEX_SUFFIX}
     */
    if (cli_present("NAMES",0,flag)) {
	if (!cli_get_value("NAMES",Temp,sizeof Temp))
	    Case_Hack = 1;	/* HEX */
	else {
	    if	    (cli_present("NAMES.UPPER",0,flag)) Case_Hack = 0;
	    else if (cli_present("NAMES.HEX_SUFFIX",0,flag)) Case_Hack = 1;
	    else if (cli_present("NAMES.LOWER",0,flag)) Case_Hack = 2;
	    else if (cli_present("NAMES.MIXED",0,flag)
		  || cli_present("NAMES.AS_IS",0,flag)) Case_Hack = 3;
	}
    } else {
	/*
	 *  /[NO]CASE_HACK
	 */
	Case_Hack = cli_present("CASE_HACK",Case_Hack,flag);
    }
    /*
     *	/[NO]STANDARD[=[NO]PORTABLE]
     *		/standard=portable	=> -ansi -pedantic
     *		/standard=noportable	=> -ansi
     *		/nostandard		=> -traditional
     *		(default is "none of the above")
     */
    if (cli_present("STANDARD",0,flag)) {
	Standard = cli_present("STANDARD.PORTABLE",0,flag) ? 2 : 1;
	/* explicit /standard=noansi resets to the default behavior */
	if (!cli_present("STANDARD.ANSI",0,flag) && cli_negated()) Standard = 0;
    } else if (cli_negated())
	Standard = -1;
}

/*
 *	Construct GNU syntax gcc-cpp/gcc-cc1/gcc-as commands
 *	from DCL syntax GCC command, then invoke gcc.com to
 *	execute them.
 */
int main( void )
{
    static const char
	*lang_name = 0,
	*C_dflt_type = ".c",
	*I_dflt_type = ".i",
	*S_dflt_type = ".s",
	*O_dflt_type = ".obj";
    const char *gcc_scratch = 0;
    register char *cp,*cp1;
    char Temp[2*MYBUFSIZ];
    int pid = getpid();

    /* initialize DCL context */
    if (!setup_parse()) exit(EXIT_FAILURE);

    cpp_Options[0] = cc1_Options[0] = gas_Options[0] = '\0';
    /*
     *	    Figure out which version of GCC we are running.
     */
    identify_language();
    Plus = (Lang == lang_Cplusplus);
    cp = Plus ? getenv(gcc_scratch = "GNU_CXX_VERSION") : 0;
    if (!cp) cp = getenv(gcc_scratch = "GNU_CC_VERSION");
    if (!cp || (*cp < '1' || *cp > '2')) {
	fprintf(stderr, "\n%s -\n- %s -\n- %s%s%s -\n- %s\n",
		"%GCC-E-UNIDENTIFIED_VERSION,",
		"Cannot determine which version of GCC is being run",
		"because ", gcc_scratch,
		    cp ? " has an invalid value." : " is not defined.",
		"Please check the installation procedure.");
	exit(EXIT_FAILURE);
	/*NOTREACHED*/
    } else {
	switch (Lang) {
	    default:
	    case lang_C:	  lang_name = "C";
				  break;
	    case lang_Cplusplus:  lang_name = "C++";
				  C_dflt_type = ".cc";
				  I_dflt_type = ".ii";
				  break;
	    case lang_ObjC:	  lang_name = "ObjC";
				  C_dflt_type = ".m";
				  break;
	}
	sprintf(Temp, "GNU %s %s", lang_name, cp);
	GCC_Version_String = savestr(Temp);
    }
    GCC_Version = *cp - '0';	/* major version number, 1 or 2 */
    /*
     *	    Also check version of gas.
     */
    cp = getenv("GNU_AS_VERSION");
    Gas_Version = (cp && *cp >= '1' && *cp <= '3') ? (*cp - '0') : GCC_Version;
    /*
     *	    Get user-preference options defined in the environment.
     */
    cp = getenv("GCC_CPP_ENV_OPTIONS");
    if (cp && *cp) q_cat(cpp_Options, (char *)0, cp, " ");
    cp = getenv(Plus ? "GCC_CXX_ENV_OPTIONS" : "GCC_CC1_ENV_OPTIONS");
    if (cp && *cp) q_cat(cc1_Options, (char *)0, cp, " ");
    cp = getenv("GCC_GAS_ENV_OPTIONS");
    if (cp && *cp) q_cat(gas_Options, (char *)0, cp, " ");
    /*
     * create the default name for the scratch files
     */
    gcc_scratch = getenv("GNU_CC_SCRATCH");
    if (!gcc_scratch) gcc_scratch = "sys$scratch:";
    sprintf(Temp, "%sgcc_%08x%s", gcc_scratch, pid, I_dflt_type);
    List_File = savestr(Temp);
    sprintf(Temp, "%sgcc_%08x%s", gcc_scratch, pid, S_dflt_type);
    Assembly_File = savestr(Temp);
    /*
     *	    Get the Input file
     */
    checkswitches(GLOBAL);
    if (cli_get_value("GCC_INPUT",Temp,sizeof Temp)) {
	Input_File = savestr(Temp);
	checkswitches(LOCAL);
    }
    /*
     *	    Handle any version identification request early on.
     */
    if (Version) {
	if (!(Verbose & VERB_SILENT)) printf(
		"%%GCC-I-VERSION, this is version %s of the GCC-VMS driver\n",
					     GCC_DRIVER_VERSION);
	cat(cpp_Options, "-v ");
	cat(cc1_Options, "-version ");
	/* this really should be (gas_major == 2 && gas_minor > 3) */
	if (Gas_Version > 1) cat(gas_Options, "-v ");
    }
    /*
     *	    Arrange to tell gas what compiler was used.
     */
    if (Generate_Object) { /* (this test isn't quite right, but good enough) */
	/* This used to be done inside gcc.com, but to make "-v" reveal
	   gas's version for 2.4 and later without breaking assembly under
	   2.3 and earlier, we need to put "-v" _before_ "-vCompiler-ID".
	   For older versions of gas, the first "-v" will be ignored.  */
	sprintf(Temp, "\"-v%s\" ", GCC_Version_String);
	cat(gas_Options, Temp);
    }
    /*
     *	    Find the base name of the source file
     */
    sprintf(Temp, "%s%s", last_resort_basename, C_dflt_type);
    fixup_filename(&Input_File, Temp, 0);
    set_basename(&Basename);
    /*
     *	    Next check to see if we need to change the name of the list file
     */
    if (Show) {
	const char *ftyp;

	Generate_Object = 0;
	Machine_Code = 0;
	switch (Show) {
	  case 1:   cat(cpp_Options, GCC_Version==1 ? "-d " : "\"-dM\" ");
		    ftyp = ".def";
		    break;
	  case 2:   cat(cpp_Options, "\"-MM\" ");
		    ftyp = ".";     /* ".MAK" perhaps? */
		    break;
	  case 3:   cat(cpp_Options, "\"-M\" ");
		    ftyp = ".";
		    break;
	  default:  ftyp = 0;
		    break;
	}
	if (ftyp && (!List_File || !List)) {
	    if (List_File) free(List_File);
	    strcat(strcpy(Temp, Basename), ftyp);
	    List_File = savestr(Temp);
	}
	List = 1;
    }
    /* Next check and see if we need to supply default file names */
    fixup_filename(&List_File, I_dflt_type, 1);
    fixup_filename(&Assembly_File, S_dflt_type, 1);
    fixup_filename(&Object, O_dflt_type, 1);

    if (Generate_Object || Machine_Code) {
	/* these cc1 options used to be supplied unconditionally by gcc.com */
	if (locate(cc1_Options, "-d") && !locate(cc1_Options, "-dumpbase")) {
	    sprintf(Temp, "-dumpbase %s ", Basename);
	    cat(cc1_Options, Temp);
	}
	if (!locate(cc1_Options, "-mgnu") && !locate(cc1_Options, "-munix"))
	    cat(cc1_Options, "-mgnu ");
    }
    /* check whether this compilation is just performing syntax checking */
    if (!Generate_Object && !Machine_Code && !List) {
	if (!locate(cc1_Options,"-fsyntax-only")) {
	    if (!(Verbose & VERB_SILENT))
		printf("%%GCC-I-NOOUTPUT, no output was requested\n");
#if 0	/* -fsyntax-only disables optimization, hence many useful -W checks */
	    /* -fsyntax-only is available but not reliable under gcc 1.40 */
	    if (GCC_Version > 1) cat(cc1_Options, "-fsyntax-only ");
#endif
	}
	/* preprocess and compile [to the null device] but don't assemble */
	Machine_Code++;
	if (Assembly_File) free(Assembly_File);
	Assembly_File = savestr("_NLA0:");	/* output to /dev/null */
    }

    /*
     *	    Generate gcc and gas switches from assorted DCL qualifiers.
     */
    if (Standard) {
	if (Standard < 0) {
	    cat(cpp_Options, "-traditional ");
	    cat(cc1_Options, "-traditional ");
	} else {
	    cat(cc1_Options, "-ansi ");
	    if (Standard > 1) {
		cat(cpp_Options, "-pedantic \"-D__STRICT_ANSI__\" ");
		cat(cc1_Options, "-pedantic ");
	    }
	}
    }
#if 0		/* not used */
    if (List) cat(cpp_Options, "\"-C\" ");
#endif
#if 0		/* redundant */
    sprintf(Temp, "\"-D__GNUC__=%d\" ", GCC_Version);
    cat(cpp_Options, Temp);
#endif
    if (Lang == lang_Cplusplus) {
	sprintf(Temp, "-+ \"-D__GNUG__=%d\" \"-D__cplusplus\" ", GCC_Version);
	cat(cpp_Options, Temp);
	cat(gas_Options, "-+ ");
    } else if (Lang == lang_ObjC) {
	if (!locate(cpp_Options, "-lang-")) {
	    strcpy(Temp, cpp_Options);
	    strcpy(cpp_Options, "-lang-objc ");
	    cat(cpp_Options, Temp);
	}
	cat(cpp_Options, "\"-D__OBJC__\" ");
	if (!locate(cc1_Options, "-lang-")) {
	    strcpy(Temp, cc1_Options);
	    strcpy(cc1_Options, "-lang-objc ");
	    cat(cc1_Options, Temp);
	}
	if (!locate(cc1_Options, "-fgnu-runtime")
	 && !locate(cc1_Options, "-fnext-runtime"))
	    cat(cc1_Options, "-fgnu-runtime ");
    }
    if (G_Float) {
	cat(cpp_Options, "\"-DCC$gfloat\" ");
	cat(cpp_Options, "\"-D__GFLOAT__\" ");
	cat(cc1_Options, "-mg ");
    }
    /*
     *	    Some version-dependent handling is needed.
     */
    switch (GCC_Version) {
      case 1:	if (Debug) cat(cc1_Options, Plus ? "-g0 " : "-g ");
		if (Optimize) cat(cc1_Options, "\"-O\" ");
		if (Plus) cat(cc1_Options, "-fforce-addr ");
		cat(gas_Options, "-1 ");
		break;
      case 2:	if (Debug) cat(cc1_Options, "-g ");
		if (Optimize) {
		    sprintf(Temp, "\"-O%d\" ", Optimize);
		    cat(cc1_Options, Temp);
		}
		break;
    }
    if (Optimize) {
	if (Debug && !(Verbose & VERB_SILENT)) printf(
      "%%GCC-I-DBGOPT, caution: /debug specified with optimization enabled\n");
	sprintf(Temp,"\"-D__OPTIMIZE__=%d\" ",Optimize);
	cat(cpp_Options, Temp);
    }
    if (Warn) {
	cat(cc1_Options, "\"-Wall\" ");
	cat(cpp_Options, "\"-Wall\" ");
    }
    if (!(Verbose & VERB_STAT)) cat(cc1_Options, "-quiet ");
    else if (Plus) cat(gas_Options, "\"-H\" ");

    if (Profile & PROF_FUNC) cat(cc1_Options, "-p ");
    if (Profile & PROF_BLOC) cat(cc1_Options, "-a ");

    if	    (Case_Hack == 0) cat(gas_Options, "-h0 ");
    else if (Case_Hack == 2) cat(gas_Options, "-h2 ");
    else if (Case_Hack == 3) cat(gas_Options, "-h3 ");

    if ((cp = locate(cc1_Options,"-mpcc-alignment")) != 0) {
	cp1 = cp + (sizeof "-mpcc-alignment" - sizeof "");
	/* If the argument string actually has "\"-mpcc-alignment\"" in it,
	   we need to remove the quotes too, to avoid bogus "\"\"" result.  */
	if (cp > cc1_Options && *(cp - 1) == '\"' && *cp1 == '\"')
	    --cp,  ++cp1;
	while (*cp1) *cp++ = *cp1++;
	*cp = '\0';
	cat(cpp_Options, "\"-DPCC_ALIGNMENT\" ");
    } else if (!locate(cc1_Options,"-mvaxc-alignment")) {
	cat(cc1_Options, "-mvaxc-alignment ");
    }
    if ((cp = locate(cc1_Options,"-mdont-save-r2-r5")) != 0) {
	cp1 = cp + (sizeof "-mdont-save-r2-r5" - sizeof "");
	if (cp > cc1_Options && *(cp - 1) == '\"' && *cp1 == '\"')
	    --cp,  ++cp1;
	while (*cp1) *cp++ = *cp1++;
	*cp = '\0';
	cat(cc1_Options,
	    "-fcall-used-r2 -fcall-used-r3 -fcall-used-r4 -fcall-used-r5 ");
    }

/*
 *	Generate the command string.
 */
    /* Invoke the .COM file */
    strcpy(Temp, "@GNU_CC:[000000]GCC");

    /* P1 = File to compile */
    q_cat(Temp, " \"", Input_File, "\"");
    cp = Temp + strlen(Temp);

    /* P2 = Options */
    *cp++ = ' ';
    *cp++ = '\"';
    if (Lang == lang_C) *cp++ = 'C';
    else if (Lang == lang_Cplusplus) *cp++ = 'P';
    else if (Lang == lang_ObjC) *cp++ = 'O';
    else if (Lang == lang_Assembly) *cp++ = 'S';
    if (List == 1) *cp++ = 'L';
    if (Generate_Object == 1) *cp++ = '*';
    if (Machine_Code) *cp++ = 'M';
    if (Verbose & VERB_ECHO) *cp++ = 'V';
    *cp++ = '\"';

    /* P3 = Name of object file */
    *cp++ = ' ';
    *cp++ = '\"';
    if (Object && Generate_Object) {
	cp1 = Object;
	while (*cp1) *cp++ = *cp1++;
    }
    *cp++ = '\"';

    /* P4 = Name of assembly file */
    *cp++ = ' ';
    *cp++ = '\"';
    if (Assembly_File && (Machine_Code || Generate_Object)) {
	cp1 = Assembly_File;
	while (*cp1) *cp++ = *cp1++;
    }
    *cp++ = '\"';

    /* P5 = Name of listing file */
    *cp++ = ' ';
    *cp++ = '\"';
    if (List_File && (List || Machine_Code || Generate_Object)) {
	cp1 = List_File;
	while (*cp1) *cp++ = *cp1++;
    }
    *cp++ = '\"';

    *cp = '\0';
/*
 * The symbols are assigned in this way, so we do not have to worry about
 * DCL parsing them.  This reduces the complexity, since we do not have to
 * double the single quotes, treble the double quotes, etc.
 *
 * Any single quotes within an include, define, undefine, or scan will be
 * doubled up, since we do require these to be passed along.
 */
    lib_set_symbol("cpp_Options", cpp_Options);
    lib_set_symbol("cc1_Options", cc1_Options);
    lib_set_symbol("gas_Options", gas_Options);

    /* gcc.com uses gnu_cc:['GCC_BinDir'] to find the compiler images */
    lib_set_symbol("GCC_BinDir", Target_Machine ? Target_Machine : "000000");
    /*
     *	    Do it
     */
    return lib_do_command(Temp, cp - Temp);
}


		/* - filename handling - */

#include <fab.h>
#include <nam.h>
#include <rmsdef.h>
#include <devdef.h>
#define NULL_DEVICE "_NLA0"
#define is_null_device(namP)	(((unsigned char)(namP)->nam$t_dvi[0] ==      \
					sizeof NULL_DEVICE - sizeof "")       \
			      && (strncmp(&(namP)->nam$t_dvi[1], NULL_DEVICE, \
					sizeof NULL_DEVICE - sizeof "") == 0))

extern vms_cond sys$parse(struct FAB *,...);
extern vms_cond sys$search(struct FAB *,...);

static struct NAM inam;				/* fully parsed input name */
static char inam_ebuf[NAM$C_MAXRSS + 1];	/* expanded string buffer */
static char inam_rbuf[NAM$C_MAXRSS + 1];	/* resultant string buffer */

/*
 * Apply filename defaulting, but keep expanded name as short as possible.
 */
static void fixup_filename( char **name_ptr,
			    const char *def_name,
			    int output_parse )
{
    vms_cond sts;
    unsigned long *fnb;
    struct NAM *nam_p, onam;
    struct FAB fab;
    char *ptr, onam_ebuf[NAM$C_MAXRSS + 1];

    if (!*name_ptr)
	*name_ptr = savestr(def_name);

    fab = cc$rms_fab;
    fab.fab$b_fns = (unsigned char) strlen( fab.fab$l_fna = *name_ptr );
    fab.fab$b_dns = (unsigned char) strlen( fab.fab$l_dna = (char *)def_name );
    fab.fab$l_fop = output_parse ? FAB$M_OFP : 0;
    fab.fab$l_nam = nam_p = (output_parse ? &onam : &inam);
    *nam_p = cc$rms_nam;
    if (!output_parse) {
	/* input file handling; will usually $search after $parse */
	inam.nam$l_esa = inam_ebuf,  inam.nam$b_ess = sizeof inam_ebuf - 1;
	inam.nam$l_rsa = inam_rbuf,  inam.nam$b_rss = sizeof inam_rbuf - 1;
    } else {
	onam.nam$l_esa = onam_ebuf,  onam.nam$b_ess = sizeof onam_ebuf - 1;
	/* previously parsed input is related file name */
	onam.nam$l_rlf = &inam;
    }
    nam_p->nam$b_nop = NAM$M_PWD;	/* | NAM$M_SYNCHK */
    fnb = &nam_p->nam$l_fnb;

    sts = sys$parse(&fab);
    /* some non-SYNCHK failures still leave sensible data in the NAM block */
    if (sts == RMS$_DNF || sts == RMS$_DEV || sts == RMS$_DNR)	sts = RMS$_SUC;

    if ((sts & 1) && !(*fnb & NAM$M_QUOTED)) {
	nam_p->nam$l_esa[nam_p->nam$b_esl] = '\0';
	if (!output_parse) {	/* setup `inam' for use as related file name */
	    /* $search for non-wildcarded input on files-oriented devices */
	    if ((fab.fab$l_dev & DEV$M_FOD) && !(*fnb & NAM$M_WILDCARD)
	     && (sys$search(&fab) & 1)) {
		nam_p->nam$l_rsa[nam_p->nam$b_rsl] = '\0';
	    } else {
		/* strange or not found; clone resultant name from expanded */
		nam_p->nam$l_rsa = nam_p->nam$l_esa;
		nam_p->nam$b_rss = nam_p->nam$b_ess;
		nam_p->nam$b_rsl = nam_p->nam$b_esl;
	    }
	}
	/* if name or type or both missing, need to fix *name_ptr */
	if ((*fnb & (NAM$M_EXP_NAME|NAM$M_EXP_TYPE))
		!= (NAM$M_EXP_NAME|NAM$M_EXP_TYPE)
	 && ((*fnb & (NAM$M_NODE|NAM$M_EXP_DIR))
		|| (fab.fab$l_dev & DEV$M_FOD))) {
	    /* strip version iff $parse or $search supplied it */
	    if (!(*fnb & NAM$M_EXP_VER)) {
		if (nam_p->nam$b_rsl)  nam_p->nam$b_rsl -= nam_p->nam$b_ver;
		nam_p->nam$b_esl -= nam_p->nam$b_ver;
		nam_p->nam$b_ver = 0;
		*nam_p->nam$l_ver = '\0';
	    }
	    /* keep as little of the path info as feasible */
	    ptr =  (*fnb & NAM$M_NODE) ? nam_p->nam$l_node
		 : (*fnb & NAM$M_EXP_DEV) ? nam_p->nam$l_dev
		 : (*fnb & NAM$M_EXP_DIR) ? nam_p->nam$l_dir
		 : nam_p->nam$l_name;
	    free(*name_ptr);
	    *name_ptr = savestr(ptr);
	}
    }
#if 0
    if (nam_p->nam$l_wcc) {
	/* ordinarily we'd do a syntax-check-only $parse here to cleanup,
	   but we want to keep `inam' as-is for use as related-file-name */
    }
#endif
    return;
}

/* intended to be used _after_ fixup_filename(input_file,*,0) */
static void set_basename( char **base_ptr )
{
    char save_c = 0, *save_p = 0, *name_p;

    if ((name_p = inam.nam$l_name) != 0) {
	/* temporarily strip ".type;version" from midst of inam_%buf[] */
	if ((save_p = inam.nam$l_type) != 0 || (save_p = inam.nam$l_ver) != 0)
	    save_c = *save_p,  *save_p = '\0';
    } else
	name_p = last_resort_basename;

    if (*base_ptr)  free(*base_ptr);
    *base_ptr = savestr(name_p);
    for (name_p = *base_ptr; *name_p; name_p++)
	if (isupper(*name_p))  *name_p = tolower(*name_p);

    if (save_p)  *save_p = save_c;	/* restore inam_ebuf[] */
}


		/* - DCL interface routines - */

/*
 *	Execute the given DCL command
 */
static int lib_do_command( const char *Text, int Size )
{
    Descr cmd_dsc;

    cmd_dsc.adr = (char *) Text;
    cmd_dsc.len = (Uword) Size;
    cmd_dsc.mbz = 0;
    return (int) lib$do_command(&cmd_dsc);
}

/*
 *	Define a local DCL symbol
 */
static int lib_set_symbol( const char *Symbol, const char *Value )
{
    Descr sym_dsc, val_dsc;
    vms_cond sts;
    int sym_typ = 1;		/* LIB$K_CLI_LOCAL_SYM */

    sym_dsc.adr = (char *) Symbol;
    sym_dsc.len = (Uword) strlen(Symbol);
    sym_dsc.mbz = 0;

    if (Value) {
	val_dsc.adr = (char *) Value;
	val_dsc.len = (Uword) strlen(Value);
	val_dsc.mbz = 0;
	sts = lib$set_symbol(&sym_dsc, &val_dsc, &sym_typ);
    } else {
	sts = lib$delete_symbol(&sym_dsc, &sym_typ);
    }
    return (int) sts;
}


/************		DCL PARSING ROUTINES		**********/

#include <ssdef.h>
#ifndef SS$_RESIGNAL
#define SS$_RESIGNAL 0x0918
#endif

#define CLI$_NORMAL	0x30001
#define CLI$_ABSENT	0x381f0
#define CLI$_NEGATED	0x381f8
#define CLI$_LOCNEG	0x38230
#define CLI$_PRESENT	0x3fd19
#define CLI$_DEFAULTED	0x3fd21
#define CLI$_LOCPRES	0x3fd31
#define CLI$_COMMA	0x3fd39

static vms_cond cli_status;

/*
 *	initialize the CLI parse tables.  This hack allows us to invoke
 *	the compiler driver either via a CLI verb (i.e. Set Command) or
 *	via a foreign command (i.e. gcc == "$Dev:[Dir]GCC.Exe".
 *	This code based init_cli() in Joe Meadows' VERB (isn't everyone's)?
 *
 *	21-Mar-1994  Charles Bailey  bailey@genetics.upenn.edu
 */

/* This should be a `globalref' for an arbitrary-sized object, but
   declaring it as if it were a function is an easier way to achieve
   the proper linkage.	[pr]  */
extern void gcc_cmd();

/* condition handling */
extern vms_cond lib$sig_to_ret(void *,void *);
extern vms_cond (*lib$establish(vms_cond (*f)(void *,void *)))(void *,void *);
/* (Note:  since we don't use setjmp/longjmp, we can use
    lib$establish directly instead of using vaxc$establish.)  */

/*
 * Condition handler that translates VMS status to C zero/non-zero semantics.
 */
static vms_cond cli_handler(void *vsig, void *vmech)
{
    unsigned long *sigv = (unsigned long *) vsig,
		  *mechv = (unsigned long *) vmech;

    /* Handle this condition iff the facility is CLI$.	*/
    if ((sigv[1] & 0x0fff0000L) == (CLI$_NORMAL & 0x0fff0000L)) {
	/* convert odd/even status code to 0 or 1 */
	sigv[1] &= 1L;		/* ultimate value for r0 register */
	mechv[3] = sigv[1];
	/* transform the signalled condition into a function return value */
	return lib$sig_to_ret(vsig, vmech);
    } else
	return SS$_RESIGNAL;	/* punt */
}

/*
 * If the command which invoked this image is the first command
 * issued in this process, cli$get_value will generate an ACCVIO
 * when trying to retrieve the non-existent previous command in
 * response to the $VERB key, so we establish a signal handler.
 * - 6-Dec-1994  Charles Bailey  bailey@genetics.upenn.edu
 */
static int cli_get_verb(char *Buffer, int Size)
{
    lib$establish(lib$sig_to_ret);
    return cli_get_value("$VERB",Buffer,Size);
}

static int setup_parse(void)
{
    static char gcc_verb[] = "GCC",
		prompt[] = "GCC>";
    static const Descr Promptdsc = { sizeof prompt - 1, 0, prompt };
    Descr Cmddsc;
    unsigned i;
    char cmdbuf[1024];

    /* were we the last CLI verb parsed? */
    if (!cli_get_verb(cmdbuf,sizeof cmdbuf - 1)) *cmdbuf = '\0';
    if (!strcmp(cmdbuf,gcc_verb)) return 1;

    /* OK, so we used a foreign command - get the entire command line */
    if (!cli_get_value("$LINE",cmdbuf,sizeof cmdbuf - 1)) return 0;

    /* replace the first token on the command line with the CLI verb string.
       BTW, quoting Joe Meadows: "this code assumes that the verb is shorter
       than the foreign symbol (which includes device:[dir], so, should be
       reasonable)."  [That's not really safe.
	 $ define C gnu_cc:[000000]gcc
	 $ gcc == "$C"
	 $ gcc foo.c
       will break.  However, it's not worth bothering with.  pr]  */
    for (i = 0; i < sizeof gcc_verb - 1; i++)
	cmdbuf[i] = gcc_verb[i];
    while (cmdbuf[i] && cmdbuf[i] != '/' && cmdbuf[i] != ' ')
	cmdbuf[i++] = ' ';

    /* Now, set up the parse tables with our command line. */
    Cmddsc.adr = cmdbuf;
    Cmddsc.len = (Uword) strlen(cmdbuf);
    Cmddsc.mbz = 0;
    cli_status = cli$dcl_parse(&Cmddsc, (void *)gcc_cmd, lib$get_input,
			       lib$get_input, &Promptdsc);
    return (int) (cli_status & 1);
}  /* end of setup_parse() */

/*
 *	See if "NAME" is present, absent or negated.
 */
static int cli_present(const char *Name, int oldflag, int gblflag)
{
    Descr Key;

    if (failsafe) lib$establish(cli_handler);
    Key.adr = (char *) Name;
    Key.len = (Uword) strlen(Name);
    Key.mbz = 0;
    cli_status = cli$present(&Key);
    switch (cli_status) {
	case CLI$_ABSENT:
		return oldflag;
	case CLI$_NEGATED:
		if (gblflag == LOCAL) return oldflag;
	case CLI$_LOCNEG:
		return 0;
	case CLI$_PRESENT:
	case CLI$_DEFAULTED:
		if (gblflag == LOCAL) return oldflag;
	case CLI$_LOCPRES:
		return 1;
	default:	/*(shouldn't be possible)*/
		return 0;
    }
}

/*
 *	Return additional information about the last CLI operation.
 */
static int cli_negated(void)
{
    return cli_status == CLI$_NEGATED || cli_status == CLI$_LOCNEG;
}

/*
 *	Get value of `Name'.
 */
static int cli_get_value(const char *Name, char *Buffer, int BufSize)
{
    Descr Key, Value;

    if (failsafe) lib$establish(cli_handler);
    Key.adr = (char *) Name;
    Key.len = (Uword) strlen(Name);
    Key.mbz = 0;
    Value.adr = Buffer;
    Value.len = BufSize - 1;
    Value.mbz = 0;
    Buffer[0] = '\0';	/* in case of failure + cli_handler return */
    cli_status = cli$get_value(&Key, &Value, &Value.len);
    Buffer[Value.len] = '\0';
    return (int) (cli_status & 1);
}
