/* <curses.h>
 *
 *	Curses screen management package for interfacing with VAXCCURSES.
 */
#ifndef _CURSES_H
#define _CURSES_H
# ifdef __cplusplus
extern "C" {
# endif

#include <stdio.h>

#define ERR	0
#define OK	1

#undef TRUE
#define TRUE	1
#undef FALSE
#define FALSE	0

typedef struct _win_st {
	int		_cur_y, _cur_x, _max_y, _max_x, _beg_y, _beg_x;
	short		_flags;
	char		_clear, _leave, _scroll, _wrap;
	/* Misaligned pointers follow;
	   partial struct is sufficient to support the public interface.  */
	unsigned : 16;	/* dummy for padding */
} WINDOW;

typedef struct _kb_st {
	int		_id;
	unsigned char	_flags;
	/* Misaligned structure follows;
	   partial struct is sufficient to support the public interface.  */
	unsigned : 24;	/* dummy for padding */
} _KEYBOARD;

typedef struct _pd_st {
	int		_id;
	int		_rows, _cols;
	void	       *_attr;		/* `union smgdef *_attr;' */
	int		_attr_size;
} _PASTEBOARD;

#define _BOLD		1	/* SMG$M_BOLD	   */
#define _REVERSE	2	/* SMG$M_REVERSE   */
#define _BLINK		4	/* SMG$M_BLINK	   */
#define _UNDERLINE	8	/* SMG$M_UNDERLINE */

#define _NOECHO		0x01
#define _NONL		0x02
#define _NOCRMODE	0x04
#define _NORAW		0x08

/* The `stdstr' and `curstr' screen globals are documented.  Altough not
   mentioned in the VAXCRTL manual, the `stdkb' keyboard global is needed
   for controlling input modes.  The `stdpd' pasteboard global also isn't
   mentioned, and isn't needed by the public interface, but could potentially
   be useful for something like using SMG's broadcast trapping routines.  */

extern WINDOW	   *stdscr	__asm("_$$PsectAttributes_NOSHR$$stdscr");
extern WINDOW	   *curscr	__asm("_$$PsectAttributes_NOSHR$$curscr");
extern _KEYBOARD   *stdkb	__asm("_$$PsectAttributes_NOSHR$$stdkb");
extern _PASTEBOARD *stdpb	__asm("_$$PsectAttributes_NOSHR$$stdpb");
extern int	    LINES	__asm("_$$PsectAttributes_NOSHR$$lines");
extern int	    COLS	__asm("_$$PsectAttributes_NOSHR$$cols");

/* check for alternate representation of doubles */
#if    __GFLOAT__
# define ___gdecl(_func,_arglist) _func _arglist __asm("_vaxc$g" #_func)
#elif  __MIXED_FLOAT__
# define ___gdecl(_func,_arglist) _func _arglist __asm("_vaxc$d" #_func)
#else
# define ___gdecl(_func,_arglist) _func _arglist
#endif

int	waddch(WINDOW *,int);
#define addch(c)	waddch(stdscr,(c))

int	waddstr(WINDOW *,const char *);
#define addstr(s)	waddstr(stdscr,(s))

int	box(WINDOW *,int,int);

/* VAX C doesn't define this pair, but libg++ wants them */
#define cbreak()	crmode()
#define nocbreak()	nocrmode()

int	wclear(WINDOW *);
#define clear()		wclear(stdscr)

#define clearok(w,b)	((w)->_clear = (b))

int	wclrattr(WINDOW *,int);
#define clrattr(a)	wclrattr(stdscr,(a))

int	wclrtobot(WINDOW *);
#define clrtobot()	wclrtobot(stdscr)

int	wclrtoeol(WINDOW *);
#define clrtoeol()	wclrtoeol(stdscr)

#define crmode()	((stdkb->_flags &= ~_NOCRMODE), nonl())
#define nocrmode()	(stdkb->_flags |= _NOCRMODE)

int	wdelch(WINDOW *);
#define delch()		wdelch(stdscr)

int	wdeleteln(WINDOW *);
#define deleteln()	wdeleteln(stdscr)

int	delwin(WINDOW *);

#define echo()		(stdkb->_flags &= ~_NOECHO)
#define noecho()	(stdkb->_flags |= _NOECHO)

void	endwin(void);

int	werase(WINDOW *);
#define erase()		werase(stdscr)

int	wgetch(WINDOW *);
#define getch()		wgetch(stdscr)

int	wgetstr(WINDOW *,char *);
#define getstr(S)	wgetstr(stdscr,(S))

#define getyx(W,Y,X)	((Y) = (W)->_cur_y, (X) = (W)->_cur_x)

char	winch(WINDOW *);
#define inch()		winsh(stdscr)

WINDOW *initscr(void);		/* returns `stdscr'; manual says `void' */

int	winsch(WINDOW *,int);
#define insch(c)	winsch(stdscr,(c))

int	winsertln(WINDOW *);
#define insertln()	winsertln(stdscr)

int	winsstr(WINDOW *,const char *);
#define insstr(s)	winsstr(stdscr,(s))

#define leaveok(w,b)	((w)->_leave = (b))

int	longname(char *__not_used,char *__atleast_64_bytes);

int	wmove(WINDOW *,int,int);
#define move(y,x)	wmove(stdscr,(y),(x))

#define mvwaddch(W,y,x,c)	(wmove((W),(y),(x)) ? waddch((W),(c))	: ERR)
#define mvaddch(y,x,c)	mvwaddch(stdscr,(y),(x),(c))

#define mvwaddstr(W,y,x,s)	(wmove((W),(y),(x)) ? waddstr((W),(s))	: ERR)
#define mvaddstr(y,x,s) mvwaddstr(stdscr,(y),(x),(s))

#define mvcur(y0,x0,y1,x1) move(stdscr,(y1),(x1))

#define mvwdelch(W,y,x)		(wmove((W),(y),(x)) ? wdelch((W))	: ERR)
#define mvdelch(y,x)	mvwdelch(stdscr,(y),(x))

#define mvwgetch(W,y,x)		(wmove((W),(y),(x)) ? wgetch((W))	: ERR)
#define mvgetch(y,x)	mvwgetch(stdscr,(y),(x))

#define mvwgetstr(W,y,x,S)	(wmove((W),(y),(x)) ? wgetstr((W),(S))	: ERR)
#define mvgetstr(y,x,S) mvwgetstr(stdscr,(y),(x),(S))

#define mvwinch(W,y,x)		(wmove((W),(y),(x)) ? winch((W))	: ERR)
#define mvinch(y,x)	mvwinch(stdscr,(y),(x))

#define mvwinsch(W,y,x,c)	(wmove((W),(y),(x)) ? winsch((W),(c))	: ERR)
#define mvinsch(y,x,c)	mvwinsch(stdscr,(y),(x),(c))

#define mvinsstr(y,x,s) mvwinsstr(stdscr,(y),(x),(s))
#define mvwinsstr(W,y,x,s)	(wmove((W),(y),(x)) ? winsstr((W),(s))	: ERR)

int	wvwin(WINDOW *,int,int);

WINDOW *newwin(int,int,int,int);

#define nl()		(stdkb->_flags &= ~_NONL)
#define nonl()		(stdkb->_flags |= _NONL)

int	overlay(WINDOW *,WINDOW *);

int	overwrite(WINDOW *,WINDOW *);

int	___gdecl(wprintw, (WINDOW *,const char *,...));
# ifdef __GNUC_MINOR__
#define printw(fmt,args...) wprintw(stdscr,fmt,##args)
# else
int	___gdecl(printw, (const char *,...));
# endif

#define raw()		(stdkb->_flags &= ~_NORAW)
#define noraw()		(stdkb->_flags |= _NORAW)

int	wrefresh(WINDOW *);
#define refresh()	wrefresh(stdscr)

int	___gdecl(wscanw, (WINDOW *,const char *,...));
# ifdef __GNUC_MINOR__
#define scanw(fmt,args...) wscanw(stdscr,fmt,##args)
# else
int	___gdecl(scanw, (const char *,...));
# endif

int	scroll(WINDOW *);

#define scrollok(w,b)	((w)->_scroll = (b))

int	wsetattr(WINDOW *,int);
#define setattr(a)	wsetattr(stdscr,(a))

int	wstandend(WINDOW *);
#define standend()	wstandend(stdscr)

int	wstandout(WINDOW *);
#define standout()	wstandout(stdscr)

WINDOW *subwin(WINDOW *,int,int,int,int);

int	touchwin(WINDOW *);

#define wrapok(w,b)	((w)->_wrap = (b))

/* VAX C doesn't define these two, but libg++ wants them */
#define _setecho(z)	((z) ? echo() : noecho())
#define _setnonl(z)	((z) ? nl() : nonl())

#undef ___gdecl		/*__GFLOAT__*/

# ifdef __cplusplus
}
# endif
#endif	/*_CURSES_H*/
