c	DBAG7.FOR
c	*********
c
c	Indexed fields handling primitives for the DBAG system
c	======================================================
c
c	Written by Luis Arriaga da Cunha 1985
c	=====================================
c
c	Synopsis of procedures :
c
c	inqx	see if index file contains at least one entry
c	indx	indexes a field of a base
c	noindx	removes indexing for a field of a base
c	opnx	opens an indexed field
c	delx	removes the entry of a record from indexed file
c	insx	inserts an entry for a certain record
c	updx	updates the entry for a certain record
c	clsx	closes the indexed file
c	findx	finds the 1st record with a value in an indexed field
c	thrux	goes thru records with a value in an indexed field
c	startx	goes to the 1st record in the file
c
c
c
	subroutine inqx_ (base,field,empty,error)
c	*****************************************
c
	implicit none
c
	integer base, field, error
	logical empty
c
c	Description
c	===========
c
c	For FIELD of BASE it returns EMPTY = .true. if index file
c	is empty, .false. otherwise, i.e., index file contains at
c	least one entry.
c
c	var
c	===
c
	include 'own:DBAG0.OWN'
c
	integer  istrip_
	external istrip_
c
	integer kanal,rrr
c
c	begin
c	=====
c
c	init error handling
c	-------------------
c
	call errclr_('INQX')
	error=0					!clear error
c
	call clsx_(base,field,error)		!close field if opened, to make
	if (error.ne.0) then			!sure to start at the beg.
	   call errclr_('INQX')			!ignore error
	   error=0
	endif
c
	call opnx_(base,field,error)		!open field
	if (error.ne.0) return			!error, carry
c
	kanal=d$ixio(field,base)		!index file i/o channel
c
	empty=.true.				!assume no index there
	read(kanal,err=100) rrr
	empty=.false.
	goto 100
c
c	Return
c
100	continue
	return
c
c	errors
c	======
c
c	formats
c	=======
c
	end
c
c
c
c
	subroutine indx_ (base,field,akey,error)
c	****************************************
c
	implicit none
c
	integer base, field, akey, error
c
c	Description
c	===========
c
c	For FIELD of BASE it creates an indexed file. The records
c	of this latter have  the  record  number followed  by the
c	field  value  itself ; this  "value"  is the text version
c	stored in disk, call it "internal text format".
c	If AKEY = 1 the  field is a key, ie, no repetitions  will
c	be allowed, otherwise the  field  is  a  normal field, ie,
c	repetitions are allowed.
c	If AKEY = 2  the  field is KWIC indexed, ie, the field is
c	decomposed into "words" and these are indexed.
c
c	var
c	===
c
	include 'own:DBAG0.OWN'
c
	integer  istrip_
	external istrip_
c
	integer kanal,l,l0,l1,lim
	character*60 fspec,fext*4
	integer xtype,xpos1,xpos2,xsize,xlong,xidx
	integer recor,alive,orecor,keymrk
	logical eot,eo,trunc
	integer count,rtmp
c
c	begin
c	=====
c
c	init error handling
c	-------------------
c
	call errclr_('INDX')
	error=0					!clear error
c
	kanal=0
c
	if     (base.le.0.or.base.gt.d$b) then
	   goto 90001				!bad database number
	elseif (d$base(base).le.0) then
	   goto 90002				!not open
	elseif (d$pid(base).le.0) then
	   goto 90002				!or not open for UPDATE
	elseif (field.lt.0.or.
     1   	field.gt.d$nfld(base)) then
	   goto 90003				!field out of bounds
	endif
c
c	field zero is record #, automatically indexed
c
	if (field.eq.0) then
	   return
	endif
c
c	See protection
c
	if (d$prt(base).ne.0) then		!protection ON
	   if (d$prfl(field,base).ne.prtrw) then
	      goto 90003			!protected field
	   endif
	endif
c
	xtype=d$type(field,base)		!field type
	xsize=d$siz (field,base)		!field size
	xpos1=d$pos (field,base)		!field start position
	xpos2=xpos1+xsize-1			!field  end  position
	xlong=d$recl(base)			!record length
c
c	delete old index file if any
c
	kanal=d$ixio(field,base)	!old index file i/o channel
c
	if (kanal.gt.0) then
	   close(unit=kanal,dispose='delete')
	   call freec_(kanal)
	   kanal=0
	   d$ixio(field,base)=0
	endif
c
c	open new indexed file
c
	fspec=d$bfil(base)
	write(fext,'(''.'',i3.3)',err=90010)field!make ext as ".026" for f. 26
	call givext_(fspec,fext)
	call newc_(kanal)		!ask for i/o channel
	if (kanal.le.0) goto 90004	!no more i/o channels
c
	if (akey.eq.0) then
	   keymrk=1			!non-key field
	elseif (akey.eq.1) then
	   keymrk=2			!key field
	elseif (akey.eq.2) then
	   keymrk=4			!KWIC field
	endif
	l0=5
	l1=5+xsize
	open(unit=kanal, file=fspec, status='new',
     1       organization='indexed',  access='keyed',form='unformatted',
     1       recl=l1, key=(1:4:integer,l0:l1:character),
     1       err=90005)
c
c	go thru all records
c
	recor=0
100	continue
	call fnext_(base,recor,alive,d$xbuf,eot,error)
	if (error.ne.0) goto 95000
	if (eot) goto 200
	call uc8to7_(d$xbuf(xpos1:xpos2))
	if (akey.eq.1) then
	   read(kanal,key=d$xbuf(xpos1:xpos2),keyid=1,err=199)orecor
	   if (orecor.ne.recor) goto 90009	!key already exists
199	   continue
	   write(kanal,err=90006)recor,d$xbuf(xpos1:xpos2)
	elseif (akey.eq.2) then
	   count=0
	   eo=.false.
	   call word4_(d$xbuf(xpos1:xpos2),d$cbuf,eo,trunc)
	   do while(.not.eo.or.count.eq.0)
	      l=istrip_(d$cbuf)
	      if (l.gt.0) then
	         rtmp=count*10000000+recor
	         write(kanal,err=90006)rtmp,d$cbuf(1:l)
	         call word4_(d$xbuf(xpos1:xpos2),d$cbuf,eo,trunc)
	         count=count+1
	      else
	         count=1		!don't loop forever if blank field
	      endif
	   enddo
	else
	   write(kanal,err=90006)recor,d$xbuf(xpos1:xpos2)
	endif
	goto 100
200	continue
c
	if (kanal.gt.0) then
	   close(unit=kanal)
	   call freec_(kanal)
	endif
c
c	mark field as indexed in context and disk
c
	d$idx (field,base)=keymrk
	kanal=d$bio(base)			!recover .DBF channel
	read(kanal,rec=22,fmt='(a)',err=90007) d$xbuf(1:xlong)  !index
	if (d$crpt(base).eq.0) call uncript_(d$xbuf,xlong)
c
	xidx=d$idx(field,base)
	call wrivar_(d$xbuf(xpos1:xpos2),xidx,xsize,error)
	if (error.ne.0) goto 90010		!write error
	if (d$crpt(base).eq.0) call cript_(d$xbuf,xlong)
	write(kanal,rec=22,fmt='(a)',err=90008) d$xbuf(1:xlong)  !index
c
	return
c
c	errors
c	======
c	illegal database number
90001	continue
	error=1
	goto 99000
c	base not open or not open for UPDATE
90002	continue
	error=2
	goto 99000
c	field out of bounds or protected
90003	continue
	error=3
	goto 99000
c	no more free i/o channels
90004	continue
	error=4
	goto 99000
c	error opening indexed file
90005	continue
	error=5
	goto 99000
c	error writing indexed file
90006	continue
	error=6
	goto 99000
c	error reading template for index mark, .DBF file
90007	continue
	error=7
	goto 99000
c	error writing template for index mark, .DBF file
90008	continue
	error=8
	goto 99000
c	key already exists
90009	continue
	d$rinf(1:5)='base '
	d$rinf(6:)=d$unam(base)		!tell him witch base
	lim=istrip_(d$rinf)
	d$rinf(lim+1:lim+6)=', fld#'
	write (d$rinf(lim+7:),fmt='(i3)',err=90099) field
90099	continue
	error=9
	goto 99000
c	internal error: read/write
90010	continue
	error=10
	goto 99000
c
99000	continue
	call errset_('INDX',error)		!my own error, now...
	call freec_(kanal)			!release channel
	return
c
c	inherited errors
95000	continue
	return
c
c	formats
c	=======
c
	end
c
c
c
c
	subroutine noindx_ (base,field,akey,error)
c	******************************************
c
	implicit none
c
	integer base, field, akey, error
c
c	Description
c	===========
c
c	For FIELD of BASE it deletes its indexed file and registers
c	that in .DBF file and context.
c	If  AKEY = 1  the  field is a key, ie, no repetitions  will
c	be  allowed, otherwise  the  field  is  a  normal field, ie,
c	repetitions are allowed.
c	If  AKEY = 2  the  field  is KWIC indexed, ie, the field is
c	decomposed into "words" and these are indexed.
c
c	var
c	===
c
	include 'own:DBAG0.OWN'
c
	integer kanal,l,l0,l1
	integer xtype,xpos1,xpos2,xsize,xlong,xidx
	character*60 fspec,fext*4
c
c	begin
c	=====
c
c	init error handling
c	-------------------
c
	call errclr_('NOINDX')
	error=0					!clear error
c
	kanal=0
c
	if     (base.le.0.or.base.gt.d$b) then
	   goto 90001				!bad database number
	elseif (d$base(base).le.0) then
	   goto 90002				!not open
	elseif (d$pid(base).le.0) then
	   goto 90002				!or not open for UPDATE
	elseif (field.lt.0.or.
     1   	field.gt.d$nfld(base)) then
	   goto 90003				!field out of bounds
	endif
c
c	field zero is record #, nothing to do
c
	if (field.eq.0) then
	   return
	endif
c
c	See protection
c
	if (d$prt(base).ne.0) then		!protection ON
	   if (d$prfl(field,base).ne.prtrw) then
	      goto 90003			!protected field
	   endif
	endif
c
	xsize=d$siz (field,base)		!field size
	xpos1=d$pos (field,base)		!field start position
	xpos2=xpos1+xsize-1			!field  end  position
	xlong=d$recl(base)			!record length
c
c	see if call is consistent
c
	if     (akey.eq.1.and.d$idx(field,base).eq.2) then
c	   ok
	elseif (akey.eq.2.and.d$idx(field,base).eq.4) then
c	   ok
	elseif (akey.eq.0.and.d$idx(field,base).gt.0) then
c	   ok
	else
	   goto 90008
	endif
c
c	find old indexed file and delete it !
c
	kanal=d$ixio(field,base)	!index file i/o channel if open
c
	if (kanal.gt.0) then
c
	   close(unit=kanal,dispose='delete')
	   call freec_(kanal)
	   kanal=0
	   d$ixio(field,base)=0
c
	else
c
	   fspec=d$bfil(base)
	   write(fext,'(''.'',i3.3)',err=90009)field!ext as ".026" for f. 26
	   call givext_(fspec,fext)
	   call newc_(kanal)		!ask for i/o channel
	   if (kanal.le.0) goto 90004	!no more i/o channels
c
	   l0=5
	   l1=5+xsize
	   open(unit=kanal, file=fspec, status='old',
     1          organization='indexed',  access='keyed',
     1          form='unformatted',
     1          recl=l1, key=(1:4:integer,l0:l1:character),
     1          err=100)
	   close(unit=kanal,dispose='delete')
100	   continue				!ignore error
	   call freec_(kanal)
c
	endif
c
c	mark field as not any more indexed in context and disk
c
	d$idx (field,base)=0
	d$ixio(field,base)=0
	kanal=d$bio(base)			!recover .DBF channel
	d$xbuf(1:xlong)=' '
	read(kanal,rec=22,fmt='(a)',err=90007) d$xbuf(1:xlong)  !index
	if (d$crpt(base).eq.0) call uncript_(d$xbuf,xlong)
c
	xidx=d$idx(field,base)
	call wrivar_(d$xbuf(xpos1:xpos2),xidx,xsize,error)
	if (error.ne.0) goto 90009		!write error
	if (d$crpt(base).eq.0) call cript_(d$xbuf,xlong)
	write(kanal,rec=22,fmt='(a)',err=90005) d$xbuf(1:xlong)  !index
c
	return
c
c	errors
c	======
c
c	illegal database number
90001	continue
	error=1
	goto 99000
c	base not open or not open for UPDATE
90002	continue
	error=2
	goto 99000
c	field out of bounds or protected
90003	continue
	error=3
	goto 99000
c	no more free i/o channels
90004	continue
	error=4
	goto 99000
c	error writing template for index mark, .DBF file
90005	continue
	error=5
	goto 99000
c	error writing indexed file
90006	continue
	error=6
	goto 99000
c	error reading template for index mark, .DBF file
90007	continue
	error=7
	goto 99000
c	inconsistent call or attempt to remove indexes from non-indexed field
90008	continue
	error=8
	goto 99000
c	internal error: read/write error
90009	continue
	error=9
	goto 99000
c
99000	continue
	call errset_('NOINDX',error)		!my own error, now...
	call freec_(kanal)			!release channel
	return
c
c	inherited errors
95000	continue
	return
c
c	formats
c	=======
c
	end
c
c
c
c
	subroutine opnx_ (base,field,error)
c	***********************************
c
	implicit none
c
	integer base, field, error
c
c	Description
c	===========
c
c
c	var
c	===
c
	include 'own:DBAG0.OWN'
	include 'own:DBAGB.OWN'
c
	external istrip_
	integer istrip_
	integer lim,b2,mast
c
c	begin
c	=====
c
c	init error handling
c	-------------------
c
	call errclr_('OPNX')
	error=0					!clear error
c
	if     (base.le.0.or.base.gt.d$b) then
	   goto 90001				!bad database number
	elseif (d$base(base).le.0) then
	   goto 90002				!not open
	elseif (field.lt.0.or.
     1   	field.gt.d$nfld(base)) then
	   goto 90003				!field out of bounds
	endif
c
	call opx_(base,field,error)		!open field
	if (error.ne.0) return
c
	b2=d$dbio(field,base)
	mast=d$mast(field,base)
	if (b2.gt.0.and.
	1   mast.gt.0) then			!o.d.b. field
	   call opx_(b2,mast,error)		!open master as well
	   if (error.ne.0) then
	      call errclr_('OPNX')		!ignore error
	      error=0
	   endif
	endif
c
	return
c
c	errors
c	======
c	illegal database number
90001	continue
	error=1
	goto 99000
c	base not open
90002	continue
	error=2
	goto 99000
c	field out of bounds or protected
90003	continue
	error=3
	goto 99000
99000	continue
	call errset_('OPNX',error)		!my own error, now...
c
c	formats
c	=======
c
	end
c
c
c
c
	subroutine opx_ (base,field,error)
c	**********************************
c
	implicit none
c
	integer base, field, error
c
c	Description
c	===========
c
c
c	var
c	===
c
	include 'own:DBAG0.OWN'
	include 'own:DBAGB.OWN'
c
	external istrip_
	integer istrip_
	integer kanal,l0,l1,lim
	character*60 fspec,fext*4
	integer xtype,xpos1,xpos2,xsize,xlong,okrec,klrec,akey,ix
c
c	begin
c	=====
c
c	init error handling
c	-------------------
c
	call errclr_('OPX')
	error=0					!clear error
c
	kanal=0
c
	if     (base.le.0.or.base.gt.d$b) then
	   goto 90001				!bad database number
	elseif (d$base(base).le.0) then
	   goto 90002				!not open
	elseif (field.lt.0.or.
     1   	field.gt.d$nfld(base)) then
	   goto 90003				!field out of bounds
	endif
c
	if (.not.(s$set(s$inde))) goto 90007	!indexes are set off
c
c	field zero is record #, automatically indexed
c
	if (field.eq.0) then
	   return
	endif
c
c	See protection
c
	if (d$prt(base).ne.0) then		!protection ON
	   if (d$prfl(field,base).eq.prtno) then
	      goto 90003			!protected field
	   endif
	endif
c
	ix=d$idx(field,base)
c
c	field MUST be indexed
c
	if (ix.le.0) goto 90006
c
c	see if indexed file already open to do nothing
c
	if (d$ixio(field,base).gt.0) return
c
c	shit, work to do...
c
	xtype=d$type(field,base)		!field type
	xsize=d$siz (field,base)		!field size
	xpos1=d$pos (field,base)		!field start position
	xpos2=xpos1+xsize-1			!field  end  position
	xlong=d$recl(base)			!record length
c
c	open indexed file, shared if base opened NOUPDATE
c
	fspec=d$bfil(base)
	write(fext,'(''.'',i3.3)')field	!make ext as ".026" for field 26
	call givext_(fspec,fext)
	call newc_(kanal)		!ask for i/o channel
	if (kanal.le.0) goto 90004	!no more i/o channels
c
	l0=5
	l1=5+xsize
	if (d$pid(base).gt.0) then	!opened update
	   open(unit=kanal, file=fspec, status='old',
     1          organization='indexed',  access='keyed',
     1          form='unformatted',
     1          recl=l1, key=(1:4:integer,l0:l1:character),
     1          err=100)
	else				!read only, share indexes
	   open(unit=kanal, file=fspec, status='old',
     1          organization='indexed',  access='keyed',
     1          form='unformatted',
     1          recl=l1, key=(1:4:integer,l0:l1:character),
     1          shared,readonly,
     1          err=100)
	endif
	goto 500			!ok, proceed
100	continue
c
c	If data base has no alive records, index data base and do open
c
	call zrec2_(base,okrec,klrec,error)
	if (error.ne.0) then
	   call freec_(kanal)		!release channel
	   return			!error, carry
	endif
	if (okrec.gt.0) goto 90005	!give up, indexes should already exist
c
	if     (ix.eq.4) then
	   akey=2			!KWIC field
	elseif (ix.eq.2) then
	   akey=1			!field is keyed
	else
	   akey=0			!normal indexed field
	endif
c
	call indx_ (base,field,akey,error)
	if (error.ne.0) then
	   call freec_(kanal)		!release channel
	   return			!error, carry
	endif
c
	open(unit=kanal, file=fspec, status='old',
     1       organization='indexed',  access='keyed',form='unformatted',
     1       recl=l1, key=(1:4:integer,l0:l1:character),
     1       err=90005)
c
500	continue
c
c	remember channel
c
	d$ixio(field,base)=kanal
c
	return
c
c	errors
c	======
c	illegal database number
90001	continue
	error=1
	goto 99000
c	base not open
90002	continue
	error=2
	goto 99000
c	field out of bounds or protected
90003	continue
	error=3
	goto 99000
c	no more free i/o channels
90004	continue
	error=4
	goto 99000
c	error opening indexed file
90005	continue
	d$rinf(1:5)='base '
	d$rinf(6:)=d$unam(base)		!tell him witch base
	lim=istrip_(d$rinf)
	d$rinf(lim+1:lim+6)=', fld#'
	write (d$rinf(lim+7:),fmt='(i3)',err=90055) field
90055	continue
	error=5
	goto 99000
c	field is not indexed !
90006	continue
	error=6
	goto 99000
c	indexes are SET OFF
90007	continue
	error=7
	goto 99000
99000	continue
	call errset_('OPX',error)		!my own error, now...
	call freec_(kanal)			!error, release channel
	return
c
c	formats
c	=======
c
	end
c
c
c
c
	subroutine clsx_ (base,field,error)
c	***********************************
c
	implicit none
c
	integer base, field, error
c
c	Description
c	===========
c
c
c	var
c	===
c
	include 'own:DBAG0.OWN'
c
	integer b2,mast
c
c	begin
c	=====
c
c	init error handling
c	-------------------
c
	call errclr_('CLSX')
	error=0					!clear error
c
	if     (base.le.0.or.base.gt.d$b) then
	   goto 90001				!bad database number
	elseif (d$base(base).le.0) then
	   goto 90002				!not open
	elseif (field.lt.0.or.
     1   	field.gt.d$nfld(base)) then
	   goto 90003				!field out of bounds
	endif
c
	call clx_(base,field,error)		!close field
	if (error.ne.0) then
	   call errclr_('CLSX')			!ignore error
	   error=0
	endif
c
	b2=d$dbio(field,base)
	mast=d$mast(field,base)
	if (b2.gt.0.and.
	1   mast.gt.0) then			!o.d.b. field
	   call clx_(b2,mast,error)		!close master as well
	   if (error.ne.0) then
	      call errclr_('CLSX')		!ignore error
	      error=0
	   endif
	endif
c
	return
c
c	errors
c	======
c	illegal database number
90001	continue
	error=1
	goto 99000
c	base not open
90002	continue
	error=2
	goto 99000
c	field out of bounds or protected
90003	continue
	error=3
	goto 99000
99000	continue
	call errset_('CLSX',error)		!my own error, now...
	return
c
c	formats
c	=======
c
	end
c
c
c
c
	subroutine clx_ (base,field,error)
c	**********************************
c
	implicit none
c
	integer base, field, error
c
c	Description
c	===========
c
c
c	var
c	===
c
	include 'own:DBAG0.OWN'
c
	integer kanal
	integer xtype,xpos1,xpos2,xsize,xlong
c
c	begin
c	=====
c
c	init error handling
c	-------------------
c
	call errclr_('CLX')
	error=0					!clear error
c
	if     (base.le.0.or.base.gt.d$b) then
	   goto 90001				!bad database number
	elseif (d$base(base).le.0) then
	   goto 90002				!not open
	elseif (field.lt.0.or.
     1   	field.gt.d$nfld(base)) then
	   goto 90003				!field out of bounds
	endif
c
c	field zero is record #, automatically indexed
c
	if (field.eq.0) then
	   return
	endif
c
c	See protection
c
	if (d$prt(base).ne.0) then		!protection ON
	   if (d$prfl(field,base).eq.prtno) then
	      goto 90003			!protected field
	   endif
	endif
c
c	field MUST be indexed
c
	if (d$idx(field,base).le.0) goto 90004
c
c	see if indexed file already closed to do nothing
c
	if (d$ixio(field,base).le.0) return
c
c	recover index io channel and close
c
	kanal=d$ixio(field,base)
	if (kanal.gt.0) then
	   close(unit=kanal)
	   call freec_(kanal)
	endif
	d$ixio(field,base)=0
c
c
	return
c
c	errors
c	======
c	illegal database number
90001	continue
	error=1
	goto 99000
c	base not open
90002	continue
	error=2
	goto 99000
c	field out of bounds or protected
90003	continue
	error=3
	goto 99000
c	field is not indexed !
90004	continue
	error=4
	goto 99000
99000	continue
	call errset_('CLX',error)		!my own error, now...
	return
c
c	formats
c	=======
c
	end
c
c
c
c
	subroutine delx_ (base,field,val,txt,xcode,error)
c	*************************************************
c
	implicit none
c
	integer base, field, val, xcode, error
	character*(*) txt
c
c	Description
c	===========
c
c	The entry in the indexed file for FIELD of record
c	XCODE is  removed. Don't  worry about VAL and TXT.
c	They  will  be  given  back with the "old" values.
c
c	var
c	===
c
	include 'own:DBAG0.OWN'
c
	external istrip_
	integer istrip_
	integer code,kanal,l,key0,m1,m2
	integer xtype,xpos1,xpos2,xsize,xlong
	integer count,rtmp
c
c	begin
c	=====
c
c	init error handling
c	-------------------
c
	call errclr_('DELX')
	error=0					!clear error
c
	if     (base.le.0.or.base.gt.d$b) then
	   goto 90001				!bad database number
	elseif (d$base(base).le.0) then
	   goto 90002				!not open
	elseif (field.lt.0.or.
     1   	field.gt.d$nfld(base)) then
	   goto 90003				!field out of bounds
	endif
c
c	field zero is record #, do nothing
c
	if (field.eq.0) then
	   return
	endif
c
c	See protection
c
	if (d$prt(base).ne.0) then		!protection ON
	   if (d$prfl(field,base).ne.prtrw) then
	      goto 90003			!protected field
	   endif
	endif
c
c	field MUST be indexed
c
	if (d$idx(field,base).le.0) goto 90004
c
c	see if indexed file is open
c
	if (d$ixio(field,base).le.0) goto 90005
c
c	see if code is valid
c
	call ex3in_(base,xcode,code,error)
	if (error.ne.0) goto 90009
	m1=d$unus-d$offs(base)+1
	m2=d$last(base)-d$offs(base)
	if (code.lt.m1.or.code.gt.m2) goto 90008
c
c	recover index io channel and some other information
c
	kanal=d$ixio(field,base)
	xtype=d$type(field,base)	!field type
	xsize=d$siz (field,base)	!field size
	l=len(txt)
	if (l.lt.xsize) xsize=l		!play safe
c
	if (d$idx(field,base).eq.4) then
	   count=0
30	   continue
	      rtmp=count*10000000+xcode
	      read(kanal,key=rtmp,keyid=0,err=31)key0
	      delete(kanal,err=90007)
	      count=count+1
	   goto 30
31	   continue
	else
	   read(kanal,key=xcode,keyid=0,err=90006)key0
	   delete(kanal,err=90007)
	endif
c
	unlock(unit=kanal,err=321)
321	continue
c
	return
c
c	errors
c	======
c	illegal database number
90001	continue
	error=1
	goto 99000
c	base not open
90002	continue
	error=2
	goto 99000
c	field out of bounds or protected
90003	continue
	error=3
	goto 99000
c	field is not indexed !
90004	continue
	error=4
	goto 99000
c	index field file is not open
90005	continue
	error=5
	goto 99000
c	code not found
90006	continue
	error=6
	goto 99000
c	error deleting entry in indexed file
90007	continue
	error=7
	goto 99000
c	invalid record number (out of bounds)
90008	continue
	error=8
	goto 99000
c	invalid record number (wrong check digit)
90009	continue
	error=9
	goto 99000
99000	continue
	call errset_('DELX',error)		!my own error, now...
c
	unlock(unit=kanal,err=3211)
3211	continue
c
	return
c
c	formats
c	=======
c
	end
c
c
c
c
	subroutine insx_ (base,field,val,txt,xcode,error)
c	*************************************************
c
	implicit none
c
	integer base, field, val, xcode, error
	character*(*) txt
c
c	Description
c	===========
c
c	For FIELD of BASE an entry  for  VAL or TXT is recorded,
c	corresponding to record XCODE. For VAL or TXT see above.
c	If  FIELD  is  key, ie, index mark is 2, no repetitions
c	are allowed.
c
c	var
c	===
c
	include 'own:DBAG0.OWN'
c
	external istrip_
	integer  istrip_
	integer code,ocode,kanal,l,m1,m2,lim
	logical iskey,eo,trunc
	integer xtype,xpos1,xpos2,xsize,xlong
	integer count,rtmp
c
c	begin
c	=====
c
c	init error handling
c	-------------------
c
	call errclr_('INSX')
	error=0					!clear error
c
	if     (base.le.0.or.base.gt.d$b) then
	   goto 90001				!bad database number
	elseif (d$base(base).le.0) then
	   goto 90002				!not open
	elseif (field.lt.0.or.
     1   	field.gt.d$nfld(base)) then
	   goto 90003				!field out of bounds
	endif
c
c	field zero is record #, do nothing
c
	if (field.eq.0) then
	   return
	endif
c
c	See protection
c
	if (d$prt(base).ne.0) then		!protection ON
	   if (d$prfl(field,base).ne.prtrw) then
	      goto 90003			!protected field
	   endif
	endif
c
c	field MUST be indexed
c
	if (d$idx(field,base).le.0) then
	   goto 90004
	elseif (d$idx(field,base).eq.2) then
	   iskey=.true.
	elseif (d$idx(field,base).gt.0) then
	   iskey=.false.
	else
	   goto 90009
	endif
c
c	see if indexed file is open
c
	if (d$ixio(field,base).le.0) goto 90005
c
c	see if code is valid
c
	call ex3in_(base,xcode,code,error)
	if (error.ne.0) goto 90008
	m1=d$unus-d$offs(base)+1
	m2=d$last(base)-d$offs(base)
	if (code.lt.m1.or.code.gt.m2) goto 90007
c
c	recover index io channel and some other information
c
	kanal=d$ixio(field,base)
	xtype=d$type(field,base)	!field type
	xsize=d$siz (field,base)	!field size
	l=len(txt)
	if (l.lt.xsize) xsize=l		!play safe
c
c	text or number like argument ?
c
	l=istrip_(txt)
	if (val.ne.0.or.l.eq.0) then	!number like
	   call wrivar_(txt,val,xsize,error)
	   if (error.ne.0) goto 2	!write error, ignore
2	   continue
	endif
	call uc8to7_(txt(1:xsize))
c
	if (iskey) then
cbug	   read(kanal,key=txt(1:xsize),keyid=1,err=200)ocode
	   read(kanal,key=txt(1:xsize),keyid=1,err=200)rtmp
	   ocode=mod(rtmp,10000000)
	   if (ocode.ne.xcode) goto 90010	!key already exists
200	   continue
	endif
	if (d$idx(field,base).eq.4) then
	   count=0
	   eo=.false.
	   call word4_(txt(1:xsize),d$cbuf,eo,trunc)
	   do while(.not.eo.or.count.eq.0)
	      l=istrip_(d$cbuf)
	      if (l.gt.0) then
	         rtmp=count*10000000+xcode
	         write(kanal,err=90006)rtmp,d$cbuf(1:l)
	         call word4_(txt(1:xsize),d$cbuf,eo,trunc)
	         count=count+1
	      else
	         count=1		!don't loop forever if blank field
	      endif
	   enddo
	else
	   write(kanal,err=90006)xcode,txt(1:xsize)
	endif
c
	unlock(unit=kanal,err=321)
321	continue
c
	return
c
c	errors
c	======
c	illegal database number
90001	continue
	error=1
	goto 99000
c	base not open
90002	continue
	error=2
	goto 99000
c	field out of bounds or protected
90003	continue
	error=3
	goto 99000
c	field is not indexed !
90004	continue
	error=4
	goto 99000
c	index field file is not open
90005	continue
	error=5
	goto 99000
c	problems writing indexed file
90006	continue
	error=6
	goto 99000
c	invalid record number (out of bounds)
90007	continue
	error=7
	goto 99000
c	invalid record number (wrong check digit)
90008	continue
	error=8
	goto 99000
c	invalid index mark !!! never happens
90009	continue
	error=9
	goto 99000
c	key already exists
90010	continue
	d$rinf(1:5)='base '
	d$rinf(6:)=d$unam(base)		!tell him witch base
	lim=istrip_(d$rinf)
	d$rinf(lim+1:lim+6)=', fld#'
	write (d$rinf(lim+7:),fmt='(i3)',err=90011) field
90011	continue
	error=10
	goto 99000
99000	continue
	call errset_('INSX',error)		!my own error, now...
c
	unlock(unit=kanal,err=3211)
3211	continue
c
	return
c
c	formats
c	=======
c
	end
c
c
c
c
	subroutine updx_ (base,field,val,txt,xcode,error)
c	*************************************************
c
	implicit none
c
	integer base, field, val, xcode, error
	character*(*) txt
c
c	Description
c	===========
c
c	For FIELD of BASE and record XCODE an update of the entry
c	is  done. The  "new"  values  are  VAL or TXT (see above).
c
c	var
c	===
c
	include 'own:DBAG0.OWN'
c
	external istrip_
	integer  istrip_
	integer code,ocode,kanal,key0,l,m1,m2,lim
	logical iskey,eo,trunc
	integer xtype,xpos1,xpos2,xsize,xlong
	integer count,rtmp
c
c	begin
c	=====
c
c	init error handling
c	-------------------
c
	call errclr_('UPDX')
	error=0					!clear error
c
	if     (base.le.0.or.base.gt.d$b) then
	   goto 90001				!bad database number
	elseif (d$base(base).le.0) then
	   goto 90002				!not open
	elseif (field.lt.0.or.
     1   	field.gt.d$nfld(base)) then
	   goto 90003				!field out of bounds
	endif
c
c	field zero is record #, do nothing
c
	if (field.eq.0) then
	   return
	endif
c
c	See protection
c
	if (d$prt(base).ne.0) then		!protection ON
	   if (d$prfl(field,base).ne.prtrw) then
	      goto 90003			!protected field
	   endif
	endif
c
c	field MUST be indexed
c
	if (d$idx(field,base).le.0) then
	   goto 90004
	elseif (d$idx(field,base).eq.2) then
	   iskey=.true.
	elseif (d$idx(field,base).gt.0) then
	   iskey=.false.
	else
	   goto 90010
	endif
c
c	see if indexed file is open
c
	if (d$ixio(field,base).le.0) goto 90005
c
c	see if code is valid
c
	call ex3in_(base,xcode,code,error)
	if (error.ne.0) goto 90009
	m1=d$unus-d$offs(base)+1
	m2=d$last(base)-d$offs(base)
	if (code.lt.m1.or.code.gt.m2) goto 90008
c
c	recover index io channel and some other information
c
	kanal=d$ixio(field,base)
	xtype=d$type(field,base)	!field type
	xsize=d$siz (field,base)	!field size
	l=len(txt)
	if (l.lt.xsize) xsize=l		!play safe
c
c	text or number like argument ?
c
	l=istrip_(txt)
	if (val.ne.0.or.l.eq.0) then	!number like
	   call wrivar_(txt,val,xsize,error)
	   if (error.ne.0) goto 2	!write error, ignore
2	   continue
	endif
	call uc8to7_(txt(1:xsize))
c
	if (d$idx(field,base).eq.4) then
	   count=0
30	   continue
	      rtmp=count*10000000+xcode
	      read(kanal,key=rtmp,keyid=0,err=31)key0
	      delete(kanal,err=90007)
	      count=count+1
	   goto 30
31	   continue
	else
	   read(kanal,key=xcode,keyid=0,err=90006)key0
	   delete(kanal,err=90007)
	endif
	if (iskey) then
cbug	   read(kanal,key=txt(1:xsize),keyid=1,err=200)ocode
	   read(kanal,key=txt(1:xsize),keyid=1,err=200)rtmp
	   ocode=mod(rtmp,10000000)
	   if (ocode.ne.xcode) goto 90011	!key already exists
200	   continue
	endif
	if (d$idx(field,base).eq.4) then
	   count=0
	   eo=.false.
	   call word4_(txt(1:xsize),d$cbuf,eo,trunc)
	   do while(.not.eo.or.count.eq.0)
	      l=istrip_(d$cbuf)
	      if (l.gt.0) then
	         rtmp=count*10000000+xcode
	         write(kanal,err=90007)rtmp,d$cbuf(1:l)
	         call word4_(txt(1:xsize),d$cbuf,eo,trunc)
	         count=count+1
	      else
	         count=1		!don't loop forever if blank field
	      endif
	   enddo
	else
	   write(kanal,err=90007)xcode,txt(1:xsize)
	endif
c
	unlock(unit=kanal,err=321)
321	continue
c
	return
c
c	errors
c	======
c	illegal database number
90001	continue
	error=1
	goto 99000
c	base not open
90002	continue
	error=2
	goto 99000
c	field out of bounds or protected
90003	continue
	error=3
	goto 99000
c	field is not indexed !
90004	continue
	error=4
	goto 99000
c	index field file is not open
90005	continue
	error=5
	goto 99000
c	code not found
90006	continue
	error=6
	goto 99000
c	error updating indexed file
90007	continue
	error=7
	goto 99000
c	invalid record number (out of bounds)
90008	continue
	error=8
	goto 99000
c	invalid record number (wrong check digit)
90009	continue
	error=9
	goto 99000
c	invalid index mark !!! never happens
90010	continue
	error=10
	goto 99000
c	key already exists
90011	continue
	d$rinf(1:5)='base '
	d$rinf(6:)=d$unam(base)		!tell him witch base
	lim=istrip_(d$rinf)
	d$rinf(lim+1:lim+6)=', fld#'
	write (d$rinf(lim+7:),fmt='(i3)',err=90021) field
90021	continue
	error=11
	goto 99000
99000	continue
	call errset_('UPDX',error)		!my own error, now...
c
	unlock(unit=kanal,err=3211)
3211	continue
c
	return
c
c	formats
c	=======
c
	end
c
c
c
c
	subroutine findx_ (base,field,val,txt,xcode,eox,error)
c	******************************************************
c
	implicit none
c
	integer base, field, val, xcode, error
	character*(*) txt
	logical eox
c
c	Description
c	===========
c
c	For a indexed  FIELD a "value" is looked for. If found
c	the corresponding record XCODE is given back.Otherwise
c	EOX  will  be  true. The "value" perhaps found can the
c	first of a list. See THRUX procedure to know how to go
c	thru  the  list. "Value"  would  normally be a text in
c	"internal" format, ie as written on disk, given in TXT.
c	However, for  number-like  fields (actually all except
c	string  type...), the  "integer" value can be given in
c	VAL  as  long  as TXT(1:)=' ' (you know what I mean ?).
c
c	var
c	===
c
	include 'own:DBAG0.OWN'
c
	external istrip_
	integer istrip_
	integer l,b2,mast,dbcode,myval,xs2,xsize,xt
	integer rtmp
	logical user
	character*10 dbbuf
c
c	begin
c	=====
c
c	init error handling
c	-------------------
c
	call errclr_('FINDX')
	error=0					!clear error
c
	eox=.false.				!let's hope
c
c	field zero is record #, task is trivial
c
	if (field.eq.0) then
	   xcode=val
	   return
	endif
c
	if     (base.le.0.or.base.gt.d$b) then
	   goto 90001				!bad database number
	elseif (d$base(base).le.0) then
	   goto 90002				!not open
	elseif (field.lt.0.or.
     1          field.gt.d$nfld(base)) then
	   goto 90003				!field out of bounds
	endif
c
	b2=d$dbio(field,base)
	mast=d$mast(field,base)
	if (b2.gt.0.and.
	1   mast.gt.0) then			!o.d.b. field
c
c	   See field protection
c
	   if (d$prt(base).ne.0) then		!protection ON
	      if (d$prfl(field,base).eq.prtno) then
	         goto 90003			!protected field
	      endif
	   endif
c
c	   field MUST be indexed
c
	   if (d$idx(field,base).le.0) goto 90004
c
c	   see if indexed file is open
c
	   if (d$ixio(field,base).le.0) goto 90005
c
c	   text or number like argument ?
c
	   l=istrip_(txt)
	   call zkind_(base,field,xt,error)
	   if (error.ne.0) return		!error, carry
c
	   if (xt.eq.l$) then			!THE exception ...
	      if (l.le.0) then
	         if (val.eq.1) then
	            txt='T'
	         else
	            txt='F'
	         endif
	      endif
	   else
	      if (val.ne.0.or.l.eq.0) then	!number like
	         if (d$type(mast,b2).ne.c$) then
	            xs2=d$siz(mast,b2)	!field size
	            call wrivar_(txt,val,xs2,error)
	            if (error.ne.0) goto 2	!write error, ignore
2	            continue
	         endif
	      endif
	   endif
c
	   user=.false.			!internal (disk) format
	   call txtdb_(b2,txt,dbcode,mast,user,eox,error)
	   if (error.ne.0) return	!error, carry
	   if (eox) return		!none found
c
	   xsize=d$siz(field,base)	!field size
	   call wrivar_(dbbuf,dbcode,xsize,error)
	   if (error.ne.0) goto 90008	!write error
c
	   myval=0
	   call fix_ (base,field,myval,dbbuf,xcode,eox,error)
	   if (error.ne.0) return		!error, carry
	else
	   call fix_ (base,field,val,txt,xcode,eox,error)
	   if (error.ne.0) return		!error, carry
	endif
c
	return
c
c
c	errors
c	======
c	illegal database number
90001	continue
	error=1
	goto 99000
c	base not open
90002	continue
	error=2
	goto 99000
c	field out of bounds or protected
90003	continue
	error=3
	goto 99000
c	field is not indexed !
90004	continue
	error=4
	goto 99000
c	index field file is not open
90005	continue
	error=5
	goto 99000
c	*** obsolete *** horror !!! invalid record# number recovered from file
90006	continue
	error=6
	goto 99000
c	*** obsolete *** invalid record number (wrong check digit)
90007	continue
	error=7
	goto 99000
c	? error writing record#
90008	continue
	error=8
	goto 99000
99000	continue
	call errset_('FINDX',error)		!my own error, now...
	return
c
c	formats
c	=======
c
	end
c
c
c
c
	subroutine fix_ (base,field,val,txt,xcode,eox,error)
c	****************************************************
c
	implicit none
c
	integer base, field, val, xcode, error
	character*(*) txt
	logical eox
c
c	Description
c	===========
c
c	For a indexed  FIELD a "value" is looked for. If found
c	the corresponding record XCODE is given back.Otherwise
c	EOX  will  be  true. The "value" perhaps found can the
c	first of a list. See THRUX procedure to know how to go
c	thru  the  list. "Value"  would  normally be a text in
c	"internal" format, ie as written on disk, given in TXT.
c	However, for  number-like  fields (actually all except
c	string  type...), the  "integer" value can be given in
c	VAL  as  long  as TXT(1:)=' ' (you know what I mean ?).
c
c	var
c	===
c
	include 'own:DBAG0.OWN'
c
	external istrip_
	integer istrip_
	integer code,l,kanal,key0,m1,m2
	integer xtype,xpos1,xpos2,xsize,xlong,xt
	integer rtmp
c
c	begin
c	=====
c
c	init error handling
c	-------------------
c
	call errclr_('FIX')
	error=0					!clear error
c
	eox=.false.				!let's hope
c
	if     (base.le.0.or.base.gt.d$b) then
	   goto 90001				!bad database number
	elseif (d$base(base).le.0) then
	   goto 90002				!not open
	elseif (field.lt.0.or.
     1   	field.gt.d$nfld(base)) then
	   goto 90003				!field out of bounds
	endif
c
c	field zero is record #, task is trivial
c
	if (field.eq.0) then
	   xcode=val
	   return
	endif
c
c	See protection
c
	if (d$prt(base).ne.0) then		!protection ON
	   if (d$prfl(field,base).eq.prtno) then
	      goto 90003			!protected field
	   endif
	endif
c
c	field MUST be indexed
c
	if (d$idx(field,base).le.0) goto 90004
c
c	see if indexed file is open
c
	if (d$ixio(field,base).le.0) goto 90005
c
c	recover index io channel and some other information
c
	kanal=d$ixio (field,base)
	xtype=d$type(field,base)	!field type
	xsize=d$siz (field,base)	!field size
	l=len(txt)
	if (l.lt.xsize) xsize=l		!play safe
c
c	text or number like argument ?
c
	l=istrip_(txt)
	call zkind_(base,field,xt,error)
	if (error.ne.0) return		!error, carry
c
	if (xt.eq.l$) then		!THE exception ...
	   if (l.le.0) then
	      if (val.eq.1) then
	         txt='T'
	      else
	         txt='F'
	      endif
	   endif
	else
	   if (val.ne.0.or.l.eq.0) then	!number like
	      if (d$type(field,base).ne.c$) then
	         call wrivar_(txt,val,xsize,error)
	         if (error.ne.0) goto 2	!write error, ignore
2	         continue
	      endif
	   endif
	endif
c
	call uc8to7_(txt(1:xsize))
	read(kanal,key=txt(1:xsize),keyid=1,err=200)rtmp
	xcode=mod(rtmp,10000000)
c
c	play safe, see if code is valid
c
	call ex3in_(base,xcode,code,error)
	if (error.ne.0) goto 90007
	m1=d$unus-d$offs(base)+1
	m2=d$last(base)-d$offs(base)
	if (code.lt.m1.or.code.gt.m2) goto 90006
c
	unlock(unit=kanal,err=321)
321	continue
c
	return
c
c	indexed value not found, assume end of whatever (heavy bet...)
200	continue
	eox=.true.
	unlock(unit=kanal,err=3211)
3211	continue
	return
c
c
c	errors
c	======
c	illegal database number
90001	continue
	error=1
	goto 99000
c	base not open
90002	continue
	error=2
	goto 99000
c	field out of bounds or protected
90003	continue
	error=3
	goto 99000
c	field is not indexed !
90004	continue
	error=4
	goto 99000
c	index field file is not open
90005	continue
	error=5
	goto 99000
c	horror !!! invalid record number recovered from index file
90006	continue
	error=6
	goto 99000
c	invalid record number (wrong check digit)
90007	continue
	error=7
	goto 99000
99000	continue
	call errset_('FIX',error)		!my own error, now...
c
	unlock(unit=kanal,err=32111)
32111	continue
c
	return
c
c	formats
c	=======
c
	end
c
c
c
c
	subroutine thrux_ (base,field,val,txt,xcode,eox,error)
c	******************************************************
c
	implicit none
c
	integer base, field, val, xcode, error
	character*(*) txt
	logical eox
c
c	Description
c	===========
c
c	For  a  indexed   FIELD  after  a  FINDX  with success,
c	records with the same "value"  are given in succession
c	(perhaps  none)  until  EOX  becomes true.
c	Again "value"  would  normally be a text in "internal"
c	format, ie as written on disk, given in TXT.  However,
c	for  number-like  fields (actually all except 	string
c	type...), the  "integer" value can be given in 	VAL as
c	long  as TXT(1:)=' ' (you know what I mean ?).
c
c
c	var
c	===
c
	include 'own:DBAG0.OWN'
c
	external istrip_
	integer  istrip_
	integer code,l,kanal,m1,m2,b2,mast,dbcode
	integer xtype,xpos1,xpos2,xsize,xlong,xt
	logical sequen,odb,user
	integer rtmp
	character*10 dbbuf
c
c	begin
c	=====
c
c	init error handling
c	-------------------
c
	call errclr_('THRUX')
	if (error.eq.-1) then
	   sequen=.true.
	else
	   sequen=.false.
	endif
	error=0					!clear error
c
	eox=.false.				!let's hope
c
	if     (base.le.0.or.base.gt.d$b) then
	   goto 90001				!bad database number
	elseif (d$base(base).le.0) then
	   goto 90002				!not open
	elseif (field.lt.0.or.
     1   	field.gt.d$nfld(base)) then
	   goto 90003				!field out of bounds
	endif
c
c	field zero is record #, eox guaranteed
c
	if (field.eq.0) then
	   eox=.true.
	   return
	endif
c
c	See protection
c
	if (d$prt(base).ne.0) then		!protection ON
	   if (d$prfl(field,base).eq.prtno) then
	      goto 90003			!protected field
	   endif
	endif
c
c	field MUST be indexed
c
	if (d$idx(field,base).le.0) goto 90004
c
c	see if indexed file is open
c
	if (d$ixio(field,base).le.0) goto 90005
c
c	recover index io channel and some other information
c
	kanal=d$ixio(field,base)
	xtype=d$type(field,base)	!field type
	xsize=d$siz (field,base)	!field size
	l=len(txt)
	if (l.lt.xsize) xsize=l		!play safe
c
c	text or number like argument ?
c
	l=istrip_(txt)
	call zkind_(base,field,xt,error)
	if (error.ne.0) return			!error, carry
c
	if (xt.eq.l$) then	!THE exception ...
	   if (l.le.0) then
	      if (val.eq.1) then
	         txt='T'
	      else
	         txt='F'
	      endif
	   endif
	else
	   if (val.ne.0.or.l.eq.0) then	!number like
	      if (d$type(field,base).ne.c$) then
	         call wrivar_(txt,val,xsize,error)
	         if (error.ne.0) goto 2	!write error, ignore
2	         continue
	      endif
	   endif
	endif
	call uc8to7_(txt(1:xsize))
c
	b2=d$dbio(field,base)
	mast=d$mast(field,base)
	odb=.false.
	if (b2.gt.0.and.
	1   mast.gt.0) then			!o.d.b. field
c
c	   See field protection
c
	   if (d$prt(b2).ne.0) then		!protection ON
	      if (d$prfl(mast,b2).eq.prtno) then
	         goto 90003			!protected field
	      endif
	   endif
c
c	   field MUST be indexed
c
	   if (d$idx(mast,b2).le.0) goto 90004
c
c	   see if indexed file is open
c
	   if (d$ixio(mast,b2).le.0) goto 90005
c
	   odb=.true.
c
	   user=.false.			!internal (disk) format
	   call txtdb_(b2,txt,dbcode,mast,user,eox,error)
	   if (error.ne.0) return	!error, carry
	   if (eox) return		!none found
c
	   xsize=d$siz(field,base)		!field size
	   call wrivar_(dbbuf,dbcode,xsize,error)
	   if (error.ne.0) goto 90008	!write error
c
	endif
c
	read(kanal,end=300,err=300)rtmp,d$xbuf(1:xsize)
	xcode=mod(rtmp,10000000)
c
c	play safe, see if code is valid
c
	call ex3in_(base,xcode,code,error)
	if (error.ne.0) goto 90007
	m1=d$unus-d$offs(base)+1
	m2=d$last(base)-d$offs(base)
	if (code.lt.m1.or.code.gt.m2) goto 90006
c
	if (odb) then
	   if (dbbuf(1:xsize).ne.d$xbuf(1:xsize)) goto 200	!end of thing
	else
	   if (txt(1:xsize).ne.d$xbuf(1:xsize)) goto 200	!end of thing
	endif
c
	unlock(unit=kanal,err=321)
321	continue
c
	return
c
c	here no more, don't panic
c
200	continue
	eox=.true.
	if (.not.odb) then
	   txt(1:xsize)=d$xbuf(1:xsize)
	else
c	   nothing (not yet...)
	endif
	unlock(unit=kanal,err=3211)
3211	continue
	return
c
c	here end of all, don't panic
c
300	continue
	eox=.true.
	if (sequen) then
	   error=-1
	endif
	unlock(unit=kanal,err=32111)
32111	continue
	return
c
c	errors
c	======
c	illegal database number
90001	continue
	error=1
	goto 99000
c	base not open
90002	continue
	error=2
	goto 99000
c	field out of bounds or protected
90003	continue
	error=3
	goto 99000
c	field is not indexed !
90004	continue
	error=4
	goto 99000
c	index field file is not open
90005	continue
	error=5
	goto 99000
c	horror !!! invalid record number recovered from index file
90006	continue
	error=6
	goto 99000
c	invalid record number (wrong check digit)
90007	continue
	error=7
	goto 99000
c	? error writing record#
90008	continue
	error=8
	goto 99000
99000	continue
	call errset_('THRUX',error)		!my own error, now...
c
	unlock(unit=kanal,err=322)
322	continue
c
	return
c
c	formats
c	=======
c
	end
c
c
c
c
	subroutine startx_ (base,field,val,txt,xcode,error)
c	***************************************************
c
	implicit none
c
	integer base, field, val, xcode, error
	character*(*) txt
c
c	Description
c	===========
c
c	For an indexed  FIELD  the procedure positions itself
c	at  the  very  beginning  of  the  first block giving
c	back the corresponding "value".
c	Again "value"  would  normally be a text in "internal"
c	format,  ie  as  written  on disk, given back  in TXT.
c	However, for number-like  fields (actually all except
c	string 	type...), the  "integer" value is also  given
c	back in VAL.
c
c
c	var
c	===
c
	include 'own:DBAG0.OWN'
c
	external istrip_
	integer istrip_
	integer code,l,kanal,m1,m2
	integer xtype,xpos1,xpos2,xsize,xlong
c
c	begin
c	=====
c
c	init error handling
c	-------------------
c
	call errclr_('STARTX')
	error=0					!clear error
c
	if     (base.le.0.or.base.gt.d$b) then
	   goto 90001				!bad database number
	elseif (d$base(base).le.0) then
	   goto 90002				!not open
	elseif (field.lt.0.or.
     1   	field.gt.d$nfld(base)) then
	   goto 90003				!field out of bounds
	endif
c
c	field zero is record #
c
	if (field.eq.0) then
	   return
	endif
c
c	See protection
c
	if (d$prt(base).ne.0) then		!protection ON
	   if (d$prfl(field,base).eq.prtno) then
	      goto 90003			!protected field
	   endif
	endif
c
c	field MUST be indexed
c
	if (d$idx(field,base).le.0) goto 90004
c
c	see if indexed file is open
c
	if (d$ixio(field,base).le.0) goto 90005
c
c	recover index io channel and some other information
c
	kanal=d$ixio(field,base)
	xtype=d$type(field,base)	!field type
	xsize=d$siz (field,base)	!field size
	l=len(txt)
	if (l.lt.xsize) xsize=l		!play safe
c
	rewind(kanal)
	read(kanal,end=90008)xcode,txt(1:xsize)
c
c	text or number like argument ?
c
	l=istrip_(txt)
	if (val.ne.0.or.l.eq.0) then	!number like
	   call rdivar_(txt,val,xsize,error)
	   if (error.ne.0) goto 2	!read error, ignore
2	   continue
	endif
c
c	play safe, see if code is valid
c
	call ex3in_(base,xcode,code,error)
	if (error.ne.0) goto 90007
c
	unlock(unit=kanal,err=321)
321	continue
c
	return
c
c	errors
c	======
c	illegal database number
90001	continue
	error=1
	goto 99000
c	base not open
90002	continue
	error=2
	goto 99000
c	field out of bounds or protected
90003	continue
	error=3
	goto 99000
c	field is not indexed !
90004	continue
	error=4
	goto 99000
c	index field file is not open
90005	continue
	error=5
	goto 99000
c	horror !!! invalid record number recovered from index file
90006	continue
	error=6
	goto 99000
c	invalid record number (wrong check digit)
90007	continue
	error=7
	goto 99000
c	indexed file is empty
90008	continue
	error=7
	goto 99000
99000	continue
	call errset_('STARTX',error)		!my own error, now...
c
	unlock(unit=kanal,err=3211)
3211	continue
c
	return
c
c	formats
c	=======
c
	end
c
c
c
c
