/*
UNOBFUSC.C

Andrew Schulman
andrew@ora.com
http://www.ora.com/windows/
ftp://ftp.ora.com/pub/examples/windows/win95.update/schulman.html
August 1995

Based on code supplied by Thuan Ewe.

In Windows 95, the GetCurrentProcessID() and GetCurrentThreadID()
functions return "obfuscated" IDs.  KERNEL32 takes the actual address
of the data structure and XORs it with a magic number to generate the
value returned from GetCurrentXXXID():

KERNEL32!GetCurrentProcessId:
    mov     eax, [PTR_CURRENT_PROCESS_ID]
    push    dword ptr [eax]
    call    OBFUSCATOR
    ret
OBFUSCATOR:
    mov     eax, [MAGIC_XOR_PATTERN]    ; e.g., 7EAE7049h
    test    eax, eax
    jnz     do_xor
    xor     eax, eax
    jmp     do_ret
do_xor:
    xor     eax, [esp+4]
do_ret:
    ret     4

This module contains TIDToTDB() and PIDToPDB() to turn the values
returned from these functions back into addresses of the data
structures.  The code is based on the fact that if a XOR b EQU c,
then a XOR c EQU b.

Note: the de-obfuscation code is fragile: it depends on specific code
patterns for GetCurrentXXXId in debug and retail versions.

To do:
Need structure for Ring 3 TCB; also use to verify that TIDToTDB works!
Similar obfuscation pattern for modules?
Rely on fact that process and thread IDs use the same XOR pattern?

For standalone test:  cl -DSTANDALONE unobfusc.c
*/

#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>
#define  WIN32_LEAN_AND_MEAN
#include "windows.h"
#include "unobfusc.h"

static DWORD GetXORPattern(char *funcname)
{
    #define GET_PROC(mod, func) \
        GetProcAddress(GetModuleHandle(mod), (func))
    DWORD (WINAPI *func)(VOID) = GET_PROC("KERNEL32", funcname);
    BYTE *p = (BYTE *) func;
    DWORD id = (*func)();
    DWORD cb, ***pppcb;
    #define PUSH_CODE   0x6A
    #define MOV_CODE    0xA1
    if (*p == PUSH_CODE) 
        p += 7;         // adjust for debug kernel: PUSH 1Ah, CALL XXX
    if (*p != MOV_CODE)
        return 0;       // something wrong!
    p ++;               // skip past MOV to [XXXXXXXX]
    pppcb = (DWORD ***) p;
    cb = ***pppcb;
    return (id ^ cb);
}

DWORD WINAPI PIDToPDB(DWORD pid)
{
    static DWORD xor_patt = 0;
    if (! xor_patt)  // one-time static init
    {
        xor_patt = GetXORPattern("GetCurrentProcessId");
        if (! xor_patt)  // still
            return 0;    // couldn't find it
    }
    return (pid ^ xor_patt);    // should verify that really is process!
}

struct PDB_t * WINAPI PIDToPDBStruct(DWORD pid)
{
    return (struct PDB_t *) PIDToPDB(pid);
}

DWORD WINAPI TIDToTDB(DWORD tid)
{
    static DWORD xor_patt = 0;
    if (! xor_patt)     // one-time static init
    {
        xor_patt = GetXORPattern("GetCurrentThreadId");
        if (xor_patt == 0)  // still
            return 0;       // couldn't find it
    }
    return (tid ^ xor_patt);    // should verify that really is Ring 3 TCB!
}

#ifdef STANDALONE
/*
Sample output:
    GetCurrentProcessID ==> FFFDA44Dh
    PIDToPDB ==>            815335B4h
    XOR pattern ==>         7EAE91F9h
    PSP: 2BAFh
    Mem Context: C10919ECh
    Win16 TDB: 0BC6h

    GetCurrentThreadID ==>  FFFCF4E1h
    TIDToTDB ==>            81526518h
    XOR pattern ==>         7EAE91F9h
*/
main(int argc, char *argv[])
{
    DWORD pid = GetCurrentProcessId();
    DWORD pdb = PIDToPDB(pid);
    struct PDB_t *pdbstr = PIDToPDBStruct(pid);
    DWORD tid = GetCurrentThreadId();
    DWORD tdb = TIDToTDB(tid);
    
    printf("GetCurrentProcessID ==> %08lXh\n", pid);
    printf("PIDToPDB ==>            %08lXh\n", pdb);
    if (pdb)
    {
        // Grab some fields from process data structure to confirm
        // that we've really got it
        printf("XOR pattern ==>         %08lXh\n", pid ^ pdb);
        printf("PSP: %04Xh\n", pdbstr->PSPSelector);
        printf("Mem Context: %08lXh\n", pdbstr->MemContext);
        printf("Win16 TDB: %04Xh\n", pdbstr->K16TDBSel);
#if 1
{
		typedef struct PDB_t PDBSTR;
		printf("Thread List (offset %04Xh): %08lXh\n",
			offsetof(PDBSTR, ThreadList),
			pdbstr->ThreadList);
}
#endif		
    }
    
    printf("\n");
    printf("GetCurrentThreadID ==>  %08lXh\n", tid);
    printf("TIDToTDB ==>            %08lXh\n", tdb);
    if (tdb)
        printf("XOR pattern ==>         %08lXh\n", tid ^ tdb);

    if (argc > 1)
        system(getenv("COMSPEC"));  // for testing
}
#endif

