 /*   *   MATRIX DECOMPOSITION MODULE  *
  *   Author:    *       Kenneth S. Kundert   *       UC Berkeley  *  *   Advising Professor:(  *       Alberto Sangiovanni-Vincentelli  *  *L  *   This module contains the routines to decompose the matrix into LU form.  *<  *   >>> User accessable functions contained in this module:  *   OrderAndDecomposeMatrix  *   DecomposeMatrix  *   MatrixRoundoffError  *,  *   >>> Functions contained in this module:  *   OrderAndDecomposeMatrix  *   DecomposeMatrix  *   EstimateLargestElement.  *   CreateMarkowitzVectorss  *   CountMarkowitza  *   MarkowitzProducts  *   SearchForPivoti  *   SearchForSingleton   *   QuicklySearchDiagonal  *   SearchDiagonal*  *   SearchEntireMatrix:  *   FindLargestInCol   *   FindLargestInColExclude  *   ExchangeRowsAndCols  *   ExchangeRowse  *   ExchangeCols   *   ExchangeElementsInCol  *   ExchangeElementsInRow  *   RealRowColElimination  *   ComplexRowColEliminationr  *   UpdateMarkowitzNumbers   *   CreateFillIn   *   MatrixRoundoffError  *   WriteStatus  */*                 b /*  *   IMPORTS  *  *   >>> Imported Files:  *   stdio.h  *       Standard C IO Library.l  *   MtrxPers.h :  *       Macros that customize the sparse matrix routines.  *   MtrxExpt.h)9  *       Macros for the matrix routines used by the user.F  *   MtrxDefs.hn,  *       Matrix type and macro definitions.   *   MtrxError.h)  *       Matrix error macro definitions. e  */l   #include <stdio.h> #include "MtrxPers.h"* #include "MtrxExpt.h"  #include "MtrxDefs.h"( #include "MtrxError.h"             i /*  *   GLOBAL VARIABLESo  *  *   >>> Global variables:  *   gMatrix  (MatrixPointer) 2  *       Global pointer to matrix data structures.$  *   LargestAfterDecompose  (double)G  *       Contains the value of the lergest element in the matrix after )  *       decomposition.E%  *   LargestBeforeDecompose  (double) J  *       Contains an estimate of the lergest element in the matrix before   *       decomposition.e  *   LargestInCol  (double) K  *       Contains the magnitude of the largest element in the column of the 3  *       current pivot, excluding the pivot itself.   *   LargestPivot  (double)1)  *       Magnitude of the largest pivot.    *   LowerThreshold  (double)aP  *       The minimum magniude an element must have to be considered as a pivot.   *   PivotsOriginalCol  (int)p'  *       Column pivot was chosen from. t  *   PivotsOriginalRow  (int) $  *       Row pivot was chosen from. !  *   PivotSelectionMethod  (char)hL  *       Character that indicates which pivot search method was successful.   *   Singletons  (int)N  *       The number of singletons available for pivoting.  Note that if row I L  *       and column I both contain singletons, only one of them is counted.   *   Size  (int)  *       Size of matrix. i  *   SmallestPivot  (double)*  *       Magnitude of the smallest pivot.   *   Step  (int)G  *       Indicates which row and column elimination is currently being t  *       performed.h  *   Threshold  (double)N  *       The minimum magniude an element must have relative to other elements 3  *       in its column to be considered as a pivot.l  */n   MatrixPointer  gMatrix;aA double  LargestPivot, SmallestPivot, Threshold, LowerThreshold;   F double  LargestInCol, LargestBeforeDecompose, LargestAfterDecompose;  C int  Size, Step, Singletons, PivotsOriginalRow, PivotsOriginalCol;   char  PivotSelectionMethod;n                 i /*  *   ORDER AND DECOMPOSE MATRIXo  *H  *   This routine chooses a pivot order for the matrix and decomposes itH  *   into LU form.  It handles both the initial decompose and subsequentJ  *   decomposes when a reordering is desired.  This is handled in a manner&  *   that is transparrent to the user.  *  *   >>> Returned:C  *   The error code is returned.  Possible errors are listed below.r  *  *   >>> Arguments:   *   Matrix  <input>  (char *)  *       Pointer to matrix. !  *   Source  <input>  (double []) N  *       Representitive source vector that is used to determine pivoting orderP  *       when the right hand side vector is sparse.  If Source is a NULL pointerH  *       then the Source vector is assumed to be full and it is not used.  *       when determining the pivoting order. "  *   lThreshold  <input>  (double)M  *       This number determines what the pivot threshold will be.  It should nP  *       be between zero and one.  If it is one then the pivoting method becomesO  *       complete pivoting, which is very slow and tends to fill up the matrix.rP  *       If it is set close to zero the pivoting method becomes strict MarkowitzL  *       with no threshold.  The pivot threshold is used to eliminate pivot P  *       candidates that would cause excessive element growth if they were used.P  *       Element growth is the cause of roundoff error.  Element growth usually O  *       occurs in well-conditioned matrices.  Setting the Threshold large willhL  *       reduce element growth and roundoff error, but setting it too large N  *       will cause execution time to be excessive and will result in a large H  *       number of fill-ins.  If this occurs, accuracy will actually be M  *       decreased because of the large number of operations required on the aL  *       matrix due to the large number of fill-ins.  A good value seems to N  *       be 0.001.  The default is chosen by giving a value larger than one orM  *       less than or equal to zero.  This value should be increased and the dK  *       matrix resolved if growth is found to be excessive.  Changing the dO  *       pivot threshold does not improve performance on matrices where growth eH  *       is low, as is usually the case with ill-conditioned matrices.  !  *   Growth  <output>  (double *)gL  *       Local version of gMatrix->Growth.  Ratio of the largest element in M  *       the matrix after decomposition to the largest element in the matrix PN  *       before decomposition.  Growth is a measure of the amount of roundoff H  *       error incurred during the decomposition. If the growth becomes L  *       large, the pivot threshold should be lowered.  Calculation of this M  *       number requires a significant amount of time, if this number is not nB  *       wanted, the macro MONITOR_ROUNDOFF should be set false.  *  *   PseudoCondition  <output>  (double *)K  *       A very rough and easily fooled estimation of the condition of the >N  *       matrix.  If this number is large, then the matrix is ill-conditioned G  *       and the accuracy of the answer is suspect.  Reducing the pivottP  *       threshold has little effect on this problem.  Ill-conditioning suggestsK  *       that something is wrong with the problem formulation. This number mI  *       is not calculated unless CALCULATE_PSEUDOCONDITION is set true. d*  *   LargestElement  <output>  (double *) M  *       An estimate of the largest element in either the original matrix or *L  *       the decomposed matrix.  Not computed unless MONITOR_ROUNDOFF is set  *       true.  * i  *   >>> Local variables:a  *   pPivot  (ElementPointer)r6  *       Pointer to the element being used as a pivot."  *   ReorderingRequired  (BOOLEAN)<  *       Flag that indicates whether reordering is required.  *  *   >>> Global variables:  *   gMatrix  <set> "  *   LargestAfterDecompose  <used>#  *   LargestBeforeDecompose  <used>c  *   LargestInCol  <set>  *   Size  <set>  *   Step  <set>  *   Threshold  <set>E  *  *   >>> Possible errors:e  *   NO_MEMORY
  *   RANGE
  *   SINGULARC  *   ILL_CONDITIONED'  *   Error is cleared in this function.e  */l  = OrderAndDecomposeMatrix (Matrix, Source, lThreshold, Growth,  <                             PseudoCondition, LargestElement)    register  MatrixPointer  Matrix;I double  Source[], lThreshold, *Growth, *PseudoCondition, *LargestElement;r                    p {)! register  ElementPointer  pPivot;u int   ReorderingRequired;   ElementPointer SearchForPivot(); double FindLargestInCol(); ELEMENT_MAG_DECLARATIONS;M   /* Begin. */     gMatrix = Matrix;      Matrix->Error = NO_ERROR;"     if (Matrix == NULL)t     {   Matrix->Error = RANGE;         goto  FatalError;      }      Size = Matrix->Size;/     Matrix->Threshold = Threshold = lThreshold;a     ReorderingRequired = FALSE;G       EstimateLargestElement();c"     if (Matrix->Error == SINGULAR)         goto FatalError;       if (NOT Matrix->FirstTime) i     {aG /* Matrix has been decomposed before and reordering is not required. */b,         for (Step = 1; Step <= Size; Step++)(         {   pPivot = Matrix->Diag[Step];?             LargestInCol = FindLargestInCol(pPivot->NextInCol); A             if ((LargestInCol * Threshold > ELEMENT_MAG(pPivot)))d$             {   if (Matrix->Complex)5                     ComplexRowColElimination(pPivot);s                 else2                     RealRowColElimination(pPivot);
             }e             else  *             {   ReorderingRequired = TRUE;%                 break; /* for loop */r
             } 	         }l#         if (NOT ReorderingRequired)s         {      #if (MONITOR_ROUNDOFF)U             Matrix->Growth = *Growth = LargestAfterDecompose /LargestBeforeDecompose; M             Matrix->LargestElement = *LargestElement = LargestAfterDecompose;  #endif #if (CALCULATE_PSEUDOCONDITION)UV             Matrix->PseudoCondition = *PseudoCondition = LargestPivot / SmallestPivot; #endif  "             return(Matrix->Error);	         }          else	         {oM /* A pivot was not large enough to maintain accuracy, so a partial reordering     is required. */  % #if (ANNOTATE >= ON_STRANGE_BEHAVIOR)*6             printf("Reordering,  Step = %1d\n", Step);- #endif                                        	         }t,     } /* End of if(NOT Matrix->FirstTime) */     else     {eO /* This is the first time the matrix has been decomposed.  These few statementseL    indicate to the rest of the code that a full reodering is required ratherF    than a partial reordering, which occurs during a failure of a fast     decompose. */           Step = 1;C#         if (NOT Matrix->RowsLinked)t             LinkRows();r!         CreateMarkowitzVectors();c'         if (Matrix->Error == NO_MEMORY)              return(NO_MEMORY);     }m  & /* Form initial Markowitz products. */     CountMarkowitz(Source);      MarkowitzProducts();  + /* Perform reordering and decomposition. */e+     for (Step = Step; Step <= Size; Step++)e"     {   pPivot = SearchForPivot();&         if (Matrix->Error == SINGULAR)             goto FatalError;$         ExchangeRowsAndCols(pPivot);           if (Matrix->Complex)-             ComplexRowColElimination(pPivot);o         else*             RealRowColElimination(pPivot);  '         if (Matrix->Error == NO_MEMORY)e             return(NO_MEMORY);'         UpdateMarkowitzNumbers(pPivot);g   #if (ANNOTATE == FULL) r         WriteStatus(); #endif     }t       Matrix->FirstTime = FALSE;     Matrix->Decomposed = TRUE;   #if (MONITOR_ROUNDOFF)L     Matrix->Growth = *Growth = LargestAfterDecompose/LargestBeforeDecompose;E     Matrix->LargestElement = *LargestElement = LargestAfterDecompose;= #endif #if (CALCULATE_PSEUDOCONDITION) N     Matrix->PseudoCondition = *PseudoCondition = LargestPivot / SmallestPivot; #endif       return(Matrix->Error);   FatalError:w#     Matrix->Growth = *Growth = 1.0; >     Matrix->PseudoCondition = *PseudoCondition = LARGEST_REAL;     return (Matrix->Error);e }                        g /*  *   DECOMPOSE MATRIX   *N  *   This routine is the companion routine to OrderAndDecomposeMatrix.  UnlikeM  *   OrderAndDecomposeMatrix, DecomposeMatrix cannot change the ordering.  ItiL  *   is slightly faster, but its main utility is stems from the fact that itH  *   will not change the ordering if a small pivot is encountered.  The <  *   standard way to use these two routines is to first use H  *   OrderAndDecomposeMatrix for the initial decompose.  For subsequent K  *   decomposes, DecomposeMatrix is used.  The value of Growth returned be  B  *   MatrixDecompose is monitored, and if it becomes significantly;  *   larger than the value of Growth initially returned by tL  *   OrderAndDecomposeMatrix, then the matrix should be reordered by callingL  *   OrderAndDecomposeMatrix.  If DecomposeMatrix is called for the initial >  *   decompose of the matrix, then it will automatically call 8  *   OrderAndDecomposeMatrix with the default threshold.  *  *   >>> Returned:C  *   The error code is returned.  Possible errors are listed below.   *  *   >>> Arguements:  *   Matrix  <input>  (char *)  *       Pointer to matrix. !  *   Growth  <output>  (double *)oL  *       Local version of gMatrix->Growth.  Ratio of the largest element in M  *       the matrix after decomposition to the largest element in the matrix mN  *       before decomposition.  Growth is a measure of the amount of roundoff O  *       error incurred during the decomposition. If the growth becomes large, oL  *       the pivot threshold should be lowered.  Calculation of this number N  *       requires a significant amount of time, if this number is not wanted, :  *       the macro MONITOR_ROUNDOFF should be set false.  *  *   PseudoCondition  <output>  (double *)K  *       A very rough and easily fooled estimation of the condition of the LN  *       matrix.  If this number is large, then the matrix is ill-conditioned G  *       and the accuracy of the answer is suspect.  Reducing the pivot P  *       threshold has little effect on this problem.  Ill-conditioning suggestsK  *       that something is wrong with the problem formulation. This number nI  *       is not calculated unless CALCULATE_PSEUDOCONDITION is set true. i*  *   LargestElement  <output>  (double *) M  *       An estimate of the largest element in either the original matrix or  L  *       the decomposed matrix.  Not computed unless MONITOR_ROUNDOFF is set  *       true.  * o  *   >>> Local variables:=  *   pPivot  (ElementPointer)r6  *       Pointer to the element being used as a pivot.  *  *   >>> Global variables:  *   gMatrix  <set>n"  *   LargestAfterDecompose  <used>#  *   LargestBeforeDecompose  <used>x  *   LargestInCol  <set>  *   Size  <set>  *   Step  <set>  *  *   >>> Possible errors:l
  *   RANGE
  *   SINGULARt'  *   Error is cleared in this function.   */s  A DecomposeMatrix (Matrix, Growth, PseudoCondition, LargestElement)r    register  MatrixPointer  Matrix;3 double  *Growth, *PseudoCondition, *LargestElement;b                    o {,! register  ElementPointer  pPivot;e double FindLargestInCol(); ELEMENT_MAG_DECLARATIONS;o   /* Begin. */     gMatrix = Matrix;*     Matrix->Error = NO_ERROR;w     if (Matrix == NULL)t     {   Matrix->Error = RANGE;         goto FatalError;     }e     if (Matrix->FirstTime)P         return(OrderAndDecomposeMatrix(Matrix, NULL, DEFAULT_THRESHOLD, Growth, O                                              PseudoCondition, LargestElement));h   #if (MONITOR_GROWTH)     EstimateLargestElement();e #endif  6 #if (CALCULATE_PSEUDOCONDITION AND NOT MONITOR_GROWTH)G /* Initialize LargestPivot and SmallestPivot for use in PseudoConditiona  * calculation.  */<     LargestPivot = 0.0; !     SmallestPivot = LARGEST_REAL;t #endif       Size = Matrix->Size;     if (Matrix->Complex),     {   for (Step = 1; Step <= Size; Step++)(         {   pPivot = Matrix->Diag[Step]; #if (MONITOR_GROWTH)?             LargestInCol = FindLargestInCol(pPivot->NextInCol);m #endif-             ComplexRowColElimination(pPivot);p	         }o     }l     else,     {   for (Step = 1; Step <= Size; Step++)(         {   pPivot = Matrix->Diag[Step]; #if (MONITOR_GROWTH)?             LargestInCol = FindLargestInCol(pPivot->NextInCol);* #endif*             RealRowColElimination(pPivot);	         }      }i       Matrix->Decomposed = TRUE;   #if (MONITOR_ROUNDOFF)L     Matrix->Growth = *Growth = LargestAfterDecompose/LargestBeforeDecompose;E     Matrix->LargestElement = *LargestElement = LargestAfterDecompose;  #endif #if (CALCULATE_PSEUDOCONDITION)lN     Matrix->PseudoCondition = *PseudoCondition = LargestPivot / SmallestPivot; #endif       if (Matrix->Error < FATAL)         return(Matrix->Error);   FatalError:R#     Matrix->Growth = *Growth = 1.0;S>     Matrix->PseudoCondition = *PseudoCondition = LARGEST_REAL;     return(Matrix->Error); }o                   t /*8  *   ESTIMATE THE LARGEST ELEMENT IN THE ORIGINAL MATRIX  *N  *   Scans Matrix to determine the approximate magnitude of the largest of theO  *   previously chosen pivots.  If this is the first time the decomposition is uJ  *   run then the diagonal elements are assumed to be the pivots.  If all L  *   previously chosen pivots are zero then the whole matrix is scanned and 9  *   the magnitude of the largest element is determined.    *  *   >>> Local variables:   *   Largest  (double)2  *       Absolute value of largest element found.   *   Magnitude  (double)-  *       Absolute value of diagonal element. *  *   pElement  (ElementPointer) L  *       Points to elements in the matrix and is used to scan matrix to find  *       largest element.  o  *   ppDiag  (ElementPointer *)sK  *       Pointer into the array of pointers to the Diagonal elements.  Thisi@  *       variable is used to quickly scan the diagonal elements.  *  *   >>> Global Varaibles:  *   gMatrix  <used>!  *   LargestAfterDecompose  <set> "  *   LargestBeforeDecompose  <set>  *   LargestPivot  <set>  *   LowerThreshold  <set>  *   Size  <used>   *   SmallestPivot  <set>S  *  *   >>> Possible errors:U  *   SINGULAR   */U   static EstimateLargestElement()   {r. register  ElementPointer  pElement, *ppDiag;   register  int  I;e double  Magnitude, Largest;t ELEMENT_MAG_DECLARATIONS;i   /* Begin. */! /* Find largest pivot element. */      Largest = 0.0;  . /* Use ppDiag to scan through the diagonals.*/     ppDiag = gMatrix->Diag;a     for (I = 1; I <= Size; I++)R      {   if (*(++ppDiag) != NULL)-         {   Magnitude = ELEMENT_MAG(*ppDiag);($             if (Largest < Magnitude)$                 Largest = Magnitude;	         }G     }a  b     if (Largest != 0.0)n     {LO /* Just to avoid trouble on some degenerate matrices with very small diagonals,n@  * scan first column in search for largest element in matrix. */*         pElement = gMatrix->FirstInCol[1];          while (pElement != NULL).         {   Magnitude = ELEMENT_MAG(pElement);+             pElement = pElement->NextInCol;p$             if (Largest < Magnitude)$                 Largest = Magnitude;	         }i  J /* Assume that matrix is diagonally dominant so that largest element is on    diagonal.  */)         LargestBeforeDecompose = Largest;r     }A     else     {u  B /* All previously chosen pivots are zero; search entire matrix for    largest element. */#         for (I = 1; I <= Size; I++)R.         {   pElement = gMatrix->FirstInCol[I];$             while (pElement != NULL)2             {   Magnitude = ELEMENT_MAG(pElement);/                 pElement = pElement->NextInCol;h(                 if (Largest < Magnitude)(                     Largest = Magnitude;
             } 	         }e)         LargestBeforeDecompose = Largest;h     }s  N /* Set LargestAfterDecompose equal to LargestBeforeDecompose so Growth >= 1 */3     LargestAfterDecompose = LargestBeforeDecompose;d$     LowerThreshold = Largest * FUZZ;                  if (Largest == 0.0) #     {   gMatrix->Error = SINGULAR; u         return;      }  l	           G /* Initialize LargestPivot and SmallestPivot for use in PseudoCondition     calculation.  */e     LargestPivot = 0.0;s!     SmallestPivot = LARGEST_REAL;        return;e }                               * /*1  *   INITIALIZE COLUMN AND ROW ELIMINATED VECTORSD  *O  *   Initalizes the vectors that keep track of which columns and rows have beene"  *   eliminated and in what order.  *  *   >>> Local variables:r  *   SizePlusOne  (unsigned),  *       Size of the arrays to be allocated.  *   SizePlusTwo  (unsigned)H  *       This is the size of the MarkowitzProduct array.  The two extra &  *       elements are needed because: E  *       1) arrays in C start with index 0 but first index used is 1.eM  *       2) the last element in the array is set to zero to act as an end of _+  *           array flag for a while loop.  s  *  *   >>> Global variables:  *   gMatrix  <used>  *   Size  <used>   *  *   >>> Possible errors:g  *   NO_MEMORY    */    static CreateMarkowitzVectors()   {+# unsigned  SizePlusOne, SizePlusTwo;d char  *malloc();   /* Begin. */ /* Create Markowitz arrays. */'     SizePlusOne = (unsigned)(Size + 1); '     SizePlusTwo = (unsigned)(Size + 2);0T     if (( gMatrix->MarkowitzRow = (int *)malloc(SizePlusOne * sizeof(int))) == NULL)#         gMatrix->Error = NO_MEMORY;*T     if (( gMatrix->MarkowitzCol = (int *)malloc(SizePlusOne * sizeof(int))) == NULL)#         gMatrix->Error = NO_MEMORY; V     if (( gMatrix->MarkowitzProd = (long*)malloc(SizePlusTwo * sizeof(long))) == NULL)#         gMatrix->Error = NO_MEMORY;s  9 /* Create Intermediate vectors for use in MatrixSolve. */e]     if (( gMatrix->IntermediateReal = (double*)malloc(SizePlusOne * sizeof(double))) == NULL)B#         gMatrix->Error = NO_MEMORY;        if (gMatrix->Complex) H     {   if (( gMatrix->IntermediateImag = (double*)malloc(SizePlusOne * P                                                        sizeof(double))) == NULL)#         gMatrix->Error = NO_MEMORY;w     }   )     gMatrix->InitialDecompStarted = TRUE;e     return;  }g       t /*  *   COUNT MARKOWITZ )  *L  *   Scans Matrix to determine the Markowitz counts for each row and column.  *  *   >>> Argument:!  *   Source  <input>  (double [])vN  *       Representitive source vector that is used to determine pivoting orderP  *       when the right hand side vector is sparse.  If Source is a NULL pointerH  *       then the Source vector is assumed to be full and it is not used.  *       when determining the pivoting order.   *  *   >>> Local variables:F  *   Count  (int)v&  *       Temporary counting variable.   *   pElement  (ElementPointer) %  *       Pointer to matrix elements. o  *  *   >>> Global variables:  *   gMatrix - (used)=  *   Size - (used)  *   Step - (used)  */    static CountMarkowitz( Source ) register  double  Source[];r                    a {x register  int Count, I;g# register  ElementPointer  pElement;    /* Begin. *// /* Generate MarkowitzRow Count for each row. */;$     for (I = Step; I <= Size; I++)       {pE /* Set Count to -1 initially to remove count due to pivot element. */t         Count = -1;**         pElement = gMatrix->FirstInRow[I];          while (pElement != NULL)         {   Count++;+             pElement = pElement->NextInRow; 	         }   4 /* Include nonzero elements in the Source vector. */         if (Source != NULL)*!         {   if (Source[I] != 0.0)g             Count++;	         } )         gMatrix->MarkowitzRow[I] = Count;E     }   6 /* Generate the MarkowitzCol count for each column. */"     for (I = Step; I <= Size; I++)     {;E /* Set Count to -1 initially to remove count due to pivot element. */s         Count = -1; *         pElement = gMatrix->FirstInCol[I];          while (pElement != NULL)         {   Count++;+             pElement = pElement->NextInCol;r	         }O)         gMatrix->MarkowitzCol[I] = Count;t     }o     return;  }i                     /*  *   MARKOWITZ PRODUCTSE  *M  *   Calculates MarkowitzProduct for each diagonal element from the Markowitz)  *   counts.  *  *   Local Varaibles:E   *   pMarkowtizProduct  (long *)E  *       Pointer that points into MarkowitzProdoct array. Is used to  -  *       sequentially access entries quickly.)  *   pMarkowtizRow  (int *)EN  *       Pointer that points into MarkowitzRow array. Is used to sequentially    *       access entries quickly.  *   pMarkowtizCol  (int *) M  *       Pointer that points into MarkowitzCol array. Is used to sequentiallyE   *       access entries quickly.  *   Product  (long)2  *       Temporary storage for Markowitz product./  *  *   >>> Global variables:  *   gMatrix  <used>  *   Singletons  <set>  *   Size  <used>l  *   Step  <used>A  */n   static MarkowitzProducts()l      { 1 register  int  I, *pMarkowitzRow, *pMarkowitzCol; , register  long  Product, *pMarkowitzProduct;   /* Begin. */     Singletons = 0;o  8     pMarkowitzProduct = &(gMatrix->MarkowitzProd[Step]);3     pMarkowitzRow = &(gMatrix->MarkowitzRow[Step]);e3     pMarkowitzCol = &(gMatrix->MarkowitzCol[Step]);r  "     for (I = Step; I <= Size; I++)6     {   Product = *pMarkowitzRow++ * *pMarkowitzCol++;2         if ((*pMarkowitzProduct++ = Product) == 0)             Singletons++;:     }g     return;d }                        /*  *   SEARCH FOR BEST PIVOT  *J  *   Performs a search to determine the element with the lowest Markowitz K  *   Product that is also acceptable.  An acceptable element is one that is L  *   larger than the LowerThreshold and at least as large as Threshold timesL  *   the largest element in the same column.  The first step is to look for M  *   singletons if any exist.  If none are found, then all the diagonals are ;N  *   searched. The diagonal is searched once quickly using the assumption thatK  *   elements on the diagonal are large compared to other elements in theireO  *   column, and so the pivot can be chosen soley on the basis of the MarkowitzzL  *   criterion.  After a element has been chosen to be pivot on the basis ofI  *   its Markowitz product, it is checked to see if it is large enough.   L  *   Waiting to the end of the Markowitz search to check the size of a pivotI  *   candidate saves considerable time, but is not guatanteed to find an eM  *   acceptable pivot.  Thus if unsucessful a second pass of the diagonal is mP  *   made.  This second pass checks to see if an element is large enough during K  *   the search, not after it.  If still no acceptable pivot candidate has  ?  *   been found, the search expands to cover the entire matrix.   *  *   >>> Returned:K  *   A pointer to the element chosen to be pivot.  If every element in the o+  *   matrix is zero, then NULL is returned.P  *  *   >>> Local variables: "  *   ChosenPivot  (ElementPointer)A  *       Pointer to element that has been chosen to be the pivot.R  *  *   >>> Global variables:  *   gMatrix  <used>!  *   PivotSelectionMethod  <set> a  *   Singletons <used>  *  *   >>> Possible errors: 
  *   SINGULARc  *   ILL_CONDITIONED z  */O   static ElementPointer  z SearchForPivot() b   {o% register ElementPointer  ChosenPivot;*% ElementPointer  SearchForSingleton();z( ElementPointer  QuicklySearchDiagonal();! ElementPointer  SearchDiagonal(); % ElementPointer  SearchEntireMatrix();t   /* Begin. */  G /*  If singletons exist, look for an acceptable one to use as pivot. */s     if (Singletons) +     {   ChosenPivot = SearchForSingleton();           if (ChosenPivot != NULL)'         {   PivotSelectionMethod = 's';M              return(ChosenPivot);	         }*     }t  L /*  Either no singletons exist or they weren't acceptable.  Take quick first"  *  pass at searching diagonal. */*     ChosenPivot = QuicklySearchDiagonal();     if (ChosenPivot != NULL)#     {   PivotSelectionMethod = 'q';          return(ChosenPivot);     }   B /*  Quick search of diagonal failed, carefully search diagonal. */#     ChosenPivot = SearchDiagonal();      if (ChosenPivot != NULL)#     {   PivotSelectionMethod = 'd';n         return(ChosenPivot);     }r  ; /*  No acceptable pivot found yet, search entire matrix. */r'     ChosenPivot = SearchEntireMatrix();s     PivotSelectionMethod = 'e';l       return(ChosenPivot);   }f                         s /*)  *   SEARCH FOR SINGLETON TO USE AS PIVOT   *K  *   Performs a search to find a singleton to use as the pivot.  The first oN  *   acceptable singleton is used.  A singleton is acceptable if it is larger L  *   in magnitude than the LowerThreshold.  A singleton does not need to be $  *   on the diagonal to be selected.  *  *   >>> Returned:H  *   A pointer to the singleton chosen to be pivot.  In no singleton is   *   acceptable, return NULL.e  *  *   >>> Local variables:n$  *   ChosenPivot  (ElementPointer)  B  *       Pointer to element that has been chosen to be the pivot.   *   lSingletons  (int)eJ  *       A local version of Singletons, which is a count of the number of /  *       singletons that can be used as pivots.i   *   pMarkowitzProduct  (long *)P  *       Pointer that points into MarkowitzProduct array. It is used to quickly .  *       access succesive Markowitz products.   *  *   >>> Global variables:  *   gMatrix  <used>  *   LowerThreshold  <used>L  *   Size  <used>   *   Singletons  <set>  *   Step  <used>t  */e   static ElementPointer  t SearchForSingleton() b   {h( register  ElementPointer  ChosenPivot;   register  int  I;g% register  long  *pMarkowitzProduct;  o int  lSingletons;n ELEMENT_MAG_DECLARATIONS;>   /* Begin. */I /* Initialize pointer that is to scan through MarkowitzProduct vector. */z8     pMarkowitzProduct = &(gMatrix->MarkowitzProd[Step]);  I /* Decrement the count of available singletons, on the assumption that anc#    acceptable one will be found. */i     lSingletons = Singletons--;h  G /* Assure that following while loop will always terminate, this is justcG    preventative medicine, if things are working right this should nevero    be needed. */+         gMatrix->MarkowitzProd[Size+1] = 0;        while (lSingletons-- > 0)'     { " /* Singletons exist, find them. */  H /* This is tricky.  Am using a pointer to sequentially step through the J  * MarkowitzProduct array.  Search terminates when singleton (Product = 0)>  * is found.  Note that the conditional in the while statementH  * ( *pMarkowitzProduct ) is true as long as the MarkowitzProduct is notF  * equal to zero.  The row (and column) index on the diagonal is then O  * calculated by subtracting the pointer to the Markowitz product of the first  N  * diagonal from the pointer to the Markowitz product of the desired element,   * the singleton. */  &         while ( *pMarkowitzProduct++ )'         {   /* Just passing through. */l	         } ;         I = pMarkowitzProduct - gMatrix->MarkowitzProd - 1;    /* Assure that I is valid. */s         if (I > Size)              return(NULL);*  B /* Singleton has been found in either/both row or/and column I. */5         if ((ChosenPivot = gMatrix->Diag[I]) != NULL)i	         {*% /* Singleton lies on the diagonal. */a;             if (ELEMENT_MAG(ChosenPivot) >= LowerThreshold)n$                 return(ChosenPivot);	         }m         else	         {i2 /* Singleton does not lie on diagonal, find it. */.             if (gMatrix->MarkowitzCol[I] == 0)5             {   ChosenPivot = gMatrix->FirstInCol[I];tK                 while ((ChosenPivot != NULL) AND (ChosenPivot->Row < Step))e9                     ChosenPivot = ChosenPivot->NextInCol; ?                 if (ELEMENT_MAG(ChosenPivot) >= LowerThreshold)i(                     return(ChosenPivot);                 else6                 {   if (gMatrix->MarkowitzRow[I] == 0)=                     {   ChosenPivot = gMatrix->FirstInRow[I];aP                         while((ChosenPivot != NULL) AND (ChosenPivot->Col<Step))A                             ChosenPivot = ChosenPivot->NextInRow; G                         if (ELEMENT_MAG(ChosenPivot) >= LowerThreshold)t0                             return(ChosenPivot);                     }=                 }o
             }d             else5             {   ChosenPivot = gMatrix->FirstInRow[I]; K                 while ((ChosenPivot != NULL) AND (ChosenPivot->Col < Step))s9                     ChosenPivot = ChosenPivot->NextInRow;t?                 if (ELEMENT_MAG(ChosenPivot) >= LowerThreshold)O(                     return(ChosenPivot);
             }s	         } 8 /* Singleton not acceptable (too small), try another. */'     } /* end of while(lSinglatons>0) */d  ) /* All singletons were not acceptable. */=L /* Restore Singletons count. Initial assumption that an acceptable singleton    would be found was wrong. */i     Singletons++;u     return(NULL);g }h                                r #if (MODIFIED_MARKOWITZ) /*I  *   QUICK SEARCH OF DIAGONAL FOR PIVOT WITH MODIFIED MARKOWITZ CRITERIONt  *J  *   Searches the diagonal looking for the best pivot.  For a pivot to be L  *   acceptable it must be larger than the pivot Threshold times the largestJ  *   element in its reduced column.  Amoung the acceptable diagonals, the J  *   one with the smallest MarkowitzProduct is sought.  Search terminates I  *   early if a diagonal is found with a MarkowitzProduct of one and its  I  *   magnitude is larger than the other elements in its row and column.  MK  *   Since its MarkowitzProduct is one, there is only one other element in  I  *   both its row and column, and, as a condition for early termination, /I  *   these elements must be located symmetricly in the matrix.  If a tie !M  *   occurs between elements of equal MarkowitzProduct, then the element withLL  *   the largest ratio between its magnitude and the largest element in its L  *   column is used.  The search will be terminated after a given number of M  *   ties have occured and the best (largest ratio) of the tied element will  J  *   be used as the pivot.  The number of ties that will trigger an early '  *   termination is MAX_MARKOWITZ_TIES.   *  *   >>> Returned:M  *   A pointer to the diagonal element chosen to be pivot.  If no diagonal is)$  *   acceptable, a NULL is returned.  *  *   >>> Local variables: "  *   ChosenPivot  (ElementPointer)E  *       Pointer to the element that has been chosen to be the pivot. %  *   LargestInColOfAccepted  (double) K  *       The magnitude of the largest element in the column (excluding the  L  *       diagonal) for the column that contains the favored pivot candidate.!  *   LargestOffDiagonal  (double)>L  *       Magnitude of the largest of the off-diagonal terms associated with 8  *       a diagonal with MarkowitzProduct equal to one.   *   Magnitude  (double),  *       Absolute value of diagonal element.  *   MaxRatio  (double)nN  *       Among the elements tied with the smallest Markowitz product, MaxRatioO  *       is the best (smallest) ratio of LargestInCol to the diagonal MagnitudeTJ  *       found so far.  The smaller the ratio, the better numerically the "  *       element will be as pivot.   *   MinMarkowitzProduct  (long)L  *       Smallest Markowitz product found of pivot candidates that lie along  *       diagonal.  *   NumberOfTies  (int)M  *       A count of the number of Markowitz ties that have occured at current   *       MarkowitzProduct.  *   pDiag  (ElementPointer)  .  *       Pointer to current diagonal element.    *   pMarkowitzProduct  (long *)P  *       Pointer that points into MarkowitzProduct array. It is used to quickly .  *       access succesive Markowitz products.   *   Ratio  (double)L  *       For the current pivot candidate, Ratio is the ratio of the largest D  *       element in its column (excluding itself) to its magnitude. %  *   TiedElements  (ElementPointer[])tE  *       Array of pointers to the elements with the minimun Markowitz>  *       product. "  *   pOtherInCol  (ElementPointer)I  *       When there is only one other element in a column other than the dL  *       diagonal, pOtherInCol is used to point to it.  Used when Markowitz I  *       product is to determine if off diagonals are placed symmetricly.w"  *   pOtherInRow  (ElementPointer)O  *       When there is only one other element in a row other than the diagonal, L  *       pOtherInRow is used to point to it.  Used when Markowitz product is>  *       to determine if off diagonals are placed symmetricly.  *  *   >>> Global variables:  *   gMatrix  <used>  *   LargestInCol  <set>  *   Size  <used>-  *   Step  <used>t  *   Threshold  <used>  */w   static ElementPointer  c QuicklySearchDiagonal()    {t7 register long  MinMarkowitzProduct, *pMarkowitzProduct; ; register  ElementPointer  pDiag, pOtherInRow, pOtherInCol; l int  I, NumberOfTies;sB ElementPointer  ChosenPivot, TiedElements[MAX_MARKOWITZ_TIES + 1];O double  Magnitude, Ratio, MaxRatio, LargestOffDiagonal, LargestInColOfAccepted;r" double  FindLargestInColExclude(); ELEMENT_MAG_DECLARATIONS;O   /* Begin. */     NumberOfTies = -1;/     MinMarkowitzProduct = LARGEST_LONG_INTEGER;e:     pMarkowitzProduct = &(gMatrix->MarkowitzProd[Step-1]);  = /* Assure that following while loop will always terminate. */t'     gMatrix->MarkowitzProd[Size+1] = 0;i  O /* This is tricky.  Am using a pointer in the inner while loop to sequentially oH  * step through the MarkowitzProduct array.  Search terminates when the O  * Markowitz product of zero placed at location Size+1 is found.  The row (and iO  * column) index on the diagonal is then calculated by subtracting the pointer  J  * to the Markowitz product of the first diagonal from the pointer to the L  * Markowitz product of the desired element. The outer for loop is infinite #  * and is broken by using break. */   $     for(;;)  /* Endless for loop. */<     {   while (MinMarkowitzProduct < *(++pMarkowitzProduct))'         {   /* Just passing through. */d	         }p  7         I = pMarkowitzProduct - gMatrix->MarkowitzProd;n  < /* Assure that I is valid; if I > Size, terminate search. */         if (I > Size)w)             break; /* Endless for loop */*  /         if ((pDiag = gMatrix->Diag[I]) == NULL)R,             continue; /* Endless for loop */>         if ((Magnitude = ELEMENT_MAG(pDiag)) < LowerThreshold),             continue; /* Endless for loop */  $         if (*pMarkowitzProduct == 1)	         { O /* Case where only one element exists in row and column other than diagonal. */u  ! /* Find off diagonal elements. */e+             pOtherInRow = pDiag->NextInRow;m+             pOtherInCol = pDiag->NextInCol;a<             if (pOtherInRow == NULL AND pOtherInCol == NULL)6             {    pOtherInRow = gMatrix->FirstInRow[I];+                  while(pOtherInRow != NULL)eL                  {   if (pOtherInRow->Col >= Step AND pOtherInRow->Col != I)                          break;I:                      pOtherInRow = pOtherInRow->NextInRow;                  }6                  pOtherInCol = gMatrix->FirstInCol[I];+                  while(pOtherInCol != NULL)*L                  {   if (pOtherInCol->Row >= Step AND pOtherInCol->Row != I)                          break;o:                      pOtherInCol = pOtherInCol->NextInCol;                  }
             }t  L /* Accept diagonal as pivot if diagonal is larger than off diagonals and the+  * off diagonals are placed symmetricly. */e>             if (pOtherInRow != NULL  AND  pOtherInCol != NULL)9             {   if (pOtherInRow->Col == pOtherInCol->Row)iF                 {   LargestOffDiagonal = MAX(ELEMENT_MAG(pOtherInRow),P                                                       ELEMENT_MAG(pOtherInCol));8                     if (Magnitude >= LargestOffDiagonal)                     {t> /* Accept pivot, it is unlikely to contribute excess error. */@                         LargestInCol = ELEMENT_MAG(pOtherInCol);&                         return(pDiag);                     }l                 }r
             }f	         }s  5         if (*pMarkowitzProduct < MinMarkowitzProduct)t	         {oP /* Notice strict inequality in test. This is a new smallest MarkowitzProduct. */$             TiedElements[0] = pDiag;5             MinMarkowitzProduct = *pMarkowitzProduct;e             NumberOfTies = 0;i	         }m         else	         {o' /* This case handles Markowitz ties. */P1             TiedElements[++NumberOfTies] = pDiag;t3             if (NumberOfTies >= MAX_MARKOWITZ_TIES)t-                 break; /* Endless for loop */r	         }e$     } /* End of endless for loop. */  A /* Test to see if any element was chosen as a pivot candidate. */b     if (NumberOfTies < 0)u         return(NULL);t  ; /* Determine which of tied elements is best numerically. */      ChosenPivot = NULL;A     MaxRatio = 1.0 / Threshold;c  '     for (I = 0; I <= NumberOfTies; I++)       {   pDiag = TiedElements[I];'         Magnitude = ELEMENT_MAG(pDiag);e6         LargestInCol = FindLargestInColExclude(pDiag);)         Ratio = LargestInCol / Magnitude;*         if (Ratio < MaxRatio)b          {   ChosenPivot = pDiag;             MaxRatio = Ratio; 2             LargestInColOfAccepted = LargestInCol;	         }n     }a*     LargestInCol = LargestInColOfAccepted;     return(ChosenPivot); }i                   e# #else  /* Not MODIFIED_MARKOWITZ */  /*D  *   QUICK SEARCH OF DIAGONAL FOR PIVOT WITH CONVENTIONAL MARKOWITZ   *   CRITERION  *J  *   Searches the diagonal looking for the best pivot.  For a pivot to be L  *   acceptable it must be larger than the pivot Threshold times the largestJ  *   element in its reduced column.  Amoung the acceptable diagonals, the J  *   one with the smallest MarkowitzProduct is sought.  Search terminates I  *   early if a diagonal is found with a MarkowitzProduct of one and its  I  *   magnitude is larger than the other elements in its row and column.  eK  *   Since its MarkowitzProduct is one, there is only one other element in  I  *   both its row and column, and, as a condition for early termination,  @  *   these elements must be located symmetricly in the matrix.    *  *   >>> Returned:M  *   A pointer to the diagonal element chosen to be pivot.  If no diagonal is $  *   acceptable, a NULL is returned.  *  *   >>> Local variables:r"  *   ChosenPivot  (ElementPointer)E  *       Pointer to the element that has been chosen to be the pivot. !  *   LargestOffDiagonal  (double)=L  *       Magnitude of the largest of the off-diagonal terms associated with 8  *       a diagonal with MarkowitzProduct equal to one.   *   Magnitude  (double),  *       Absolute value of diagonal element.   *   MinMarkowitzProduct  (long)H  *       Smallest Markowitz product found of pivot candidates which are   *       acceptable.  *   pDiag  (ElementPointer) k.  *       Pointer to current diagonal element.    *   pMarkowitzProduct  (long *)P  *       Pointer that points into MarkowitzProduct array. It is used to quickly .  *       access succesive Markowitz products. "  *   pOtherInCol  (ElementPointer)I  *       When there is only one other element in a column other than the oL  *       diagonal, pOtherInCol is used to point to it.  Used when Markowitz I  *       product is to determine if off diagonals are placed symmetricly. "  *   pOtherInRow  (ElementPointer)O  *       When there is only one other element in a row other than the diagonal, L  *       pOtherInRow is used to point to it.  Used when Markowitz product is>  *       to determine if off diagonals are placed symmetricly.  *  *   >>> Global variables:  *   gMatrix  <used>  *   LargestInCol  <set>  *   Size  <used>   *   Step  <used>D  *   Threshold  <used>  */R   static ElementPointer  I QuicklySearchDiagonal()    { 7 register long  MinMarkowitzProduct, *pMarkowitzProduct;p  register  ElementPointer  pDiag; int  I; 6 ElementPointer  ChosenPivot, pOtherInRow, pOtherInCol;& double  Magnitude, LargestOffDiagonal;" double  FindLargestInColExclude(); ELEMENT_MAG_DECLARATIONS;    /* Begin. */     ChosenPivot = NULL; /     MinMarkowitzProduct = LARGEST_LONG_INTEGER;r:     pMarkowitzProduct = &(gMatrix->MarkowitzProd[Step-1]);  = /* Assure that following while loop will always terminate. */*'     gMatrix->MarkowitzProd[Size+1] = 0;   O /* This is tricky.  Am using a pointer in the inner while loop to sequentially rH  * step through the MarkowitzProduct array.  Search terminates when the O  * Markowitz product of zero placed at location Size+1 is found.  The row (and nO  * column) index on the diagonal is then calculated by subtracting the pointer *J  * to the Markowitz product of the first diagonal from the pointer to the M  * Markowitz product of the desired element. The outer for loop is infinite,    * broken by using break. */  %     for (;;)  /* Endless for loop. */g=     {   while (*(++pMarkowitzProduct) >= MinMarkowitzProduct)s'         {   /* Just passing through. */E	         }i  7         I = pMarkowitzProduct - gMatrix->MarkowitzProd;r  < /* Assure that I is valid; if I > Size, terminate search. */         if (I > Size)e)             break; /* Endless for loop */a  /         if ((pDiag = gMatrix->Diag[I]) == NULL) ,             continue; /* Endless for loop */>         if ((Magnitude = ELEMENT_MAG(pDiag)) < LowerThreshold),             continue; /* Endless for loop */  $         if (*pMarkowitzProduct == 1)	         { O /* Case where only one element exists in row and column other than diagonal. */w  ! /* Find off diagonal elements. */ +             pOtherInRow = pDiag->NextInRow; +             pOtherInCol = pDiag->NextInCol;n<             if (pOtherInRow == NULL AND pOtherInCol == NULL)6             {    pOtherInRow = gMatrix->FirstInRow[I];+                  while(pOtherInRow != NULL)lL                  {   if (pOtherInRow->Col >= Step AND pOtherInRow->Col != I)                          break;p:                      pOtherInRow = pOtherInRow->NextInRow;                  }6                  pOtherInCol = gMatrix->FirstInCol[I];+                  while(pOtherInCol != NULL)nL                  {   if (pOtherInCol->Row >= Step AND pOtherInCol->Row != I)                          break; :                      pOtherInCol = pOtherInCol->NextInCol;                  }
             }e  L /* Accept diagonal as pivot if diagonal is larger than off diagonals and the+  * off diagonals are placed symmetricly. */e>             if (pOtherInRow != NULL  AND  pOtherInCol != NULL)9             {   if (pOtherInRow->Col == pOtherInCol->Row)iF                 {   LargestOffDiagonal = MAX(ELEMENT_MAG(pOtherInRow),P                                                       ELEMENT_MAG(pOtherInCol));8                     if (Magnitude >= LargestOffDiagonal)                     {i> /* Accept pivot, it is unlikely to contribute excess error. */@                         LargestInCol = ELEMENT_MAG(pOtherInCol);&                         return(pDiag);                     }o                 } 
             } 	         }r  1         MinMarkowitzProduct = *pMarkowitzProduct;o         ChosenPivot = pDiag;%     }  /* End of endless for loop. */a       if (ChosenPivot != NULL)<         LargestInCol = FindLargestInColExclude(ChosenPivot);     return(ChosenPivot); }p #endif                 n /*@  *   SEARCH DIAGONAL FOR PIVOT WITH MODIFIED MARKOWITZ CRITERION  *J  *   Searches the diagonal looking for the best pivot.  For a pivot to be L  *   acceptable it must be larger than the pivot Threshold times the largestJ  *   element in its reduced column.  Amoung the acceptable diagonals, the H  *   one with the smallest MarkowitzProduct is sought.  If a tie occurs F  *   between elements of equal MarkowitzProduct, then the element withL  *   the largest ratio between its magnitude and the largest element in its L  *   column is used.  The search will be terminated after a given number of N  *   ties have occured and the best (smallest ratio) of the tied element will J  *   be used as the pivot.  The number of ties that will trigger an early '  *   termination is MAX_MARKOWITZ_TIES.   *  *   >>> Returned:M  *   A pointer to the diagonal element chosen to be pivot.  If no diagonal is($  *   acceptable, a NULL is returned.  *  *   >>> Local variables:I"  *   ChosenPivot  (ElementPointer)E  *       Pointer to the element that has been chosen to be the pivot.e%  *   LargestInColOfAccepted  (double)lP  *       Magnitude of the largest element in the column of the diagonal element 9  *       that is currently the pivot candidate of choice.t  *   lSize  (int)>O  *       Local version of size which is placed in a register to increase speed.   *   Magnitude  (double),  *       Absolute value of diagonal element.   *   MinMarkowitzProduct  (long)N  *       Smallest Markowitz product found of those pivot candidates which are   *       acceptable.  *   NumberOfTies  (int)M  *       A count of the number of Markowitz ties that have occured at current   *       MarkowitzProduct.  *   pDiag  (ElementPointer) t.  *       Pointer to current diagonal element.   *   pMarkowitzProduct  (long*)MP  *       Pointer that points into MarkowitzProduct array. It is used to quickly .  *       access succesive Markowitz products.   *   Ratio  (double)6  *       For the current pivot candidate, Ratio is theF  *       Ratio of the largest element in its column to its magnitude.   *   RatioOfAccepted  (double)J  *       For the best pivot candidate found so far, RatioOfAccepted is theF  *       Ratio of the largest element in its column to its magnitude.   *  *   >>> Global variables:  *   gMatrix  <used>  *   LargestInCol  <set>  *   Size  <used>   *   Step  <used>t  *   Threshold  <used>  */*   static ElementPointer  n SearchDiagonal() a   {m register  int  I, lSize;7 register long  MinMarkowitzProduct, *pMarkowitzProduct;e  register  ElementPointer  pDiag; int  NumberOfTies; ElementPointer  ChosenPivot;* double  Magnitude, Ratio, RatioOfAccepted; double  LargestInColOfAccepted;e" double  FindLargestInColExclude(); ELEMENT_MAG_DECLARATIONS;R   /* Begin. */     ChosenPivot = NULL;k     lSize = Size;*/     MinMarkowitzProduct = LARGEST_LONG_INTEGER;y:     pMarkowitzProduct = &(gMatrix->MarkowitzProd[Step-1]);   /* Begin search of diagonal. */ #     for (I = Step; I <= lSize; I++)*     {s9         if (*(++pMarkowitzProduct) > MinMarkowitzProduct)c$             continue; /* for loop *//         if ((pDiag = gMatrix->Diag[I]) == NULL) $             continue; /* for loop */>         if ((Magnitude = ELEMENT_MAG(pDiag)) < LowerThreshold)$             continue; /* for loop */  8 /* Test to see if diagonal's magnitude is acceptable. */6         LargestInCol = FindLargestInColExclude(pDiag);1         if (Magnitude < Threshold * LargestInCol) $             continue; /* for loop */  5         if (*pMarkowitzProduct < MinMarkowitzProduct)l	         {nP /* Notice strict inequality in test. This is a new smallest MarkowitzProduct. */              ChosenPivot = pDiag;5             MinMarkowitzProduct = *pMarkowitzProduct;r7             RatioOfAccepted = LargestInCol / Magnitude;>2             LargestInColOfAccepted = LargestInCol;             NumberOfTies = 0; 	         }u         else	         {s' /* This case handles Markowitz ties. */n             NumberOfTies++; -             Ratio = LargestInCol / Magnitude;p(             if (Ratio < RatioOfAccepted)$             {   ChosenPivot = pDiag;(                 RatioOfAccepted = Ratio;6                 LargestInColOfAccepted = LargestInCol;             }   3             if (NumberOfTies >= MAX_MARKOWITZ_TIES)n6             {   LargestInCol = LargestInColOfAccepted;$                 return(ChosenPivot);
             }i	         }u     } /* End of for(Step) */*     LargestInCol = LargestInColOfAccepted;     return(ChosenPivot); }p                     /*(  *   SEARCH ENTIRE MATRIX FOR BEST PIVOT  *I  *   Performs a search over the entire matrix looking for the acceptable tJ  *   element with the lowest MarkowitzProduct.  If there are several that L  *   are tied for the smallest MarkowitzProduct, the tie is broken by using N  *   the ratio of the magnitude of the element being considered to the largestO  *   element in the same column.  If no element is acceptable then the largest rC  *   element in the reduced submatrix is used as the pivot and the kJ  *   matrix is declared to be ILL_CONDITIONED.  If the largest element is 1  *   zero, the matrix is declared to be SINGULAR.   *  *   >>> Returned:L  *   A pointer to the diagonal element chosen to be pivot.  If no element is=  *   found, then NULL is returned and the matrix is SINGULAR.   *  *   >>> Local variables: "  *   ChosenPivot  (ElementPointer)E  *       Pointer to the element that has been chosen to be the pivot. %  *   LargestInColOfAccepted  (double) I  *       Magnitude of the largest element in the column of the currently )D  *       favored pivot candidate.  The pivot candidate is excluded.    *   LargestElementMag  (double)M  *       Magnitude of the largest element yet found in the reduced submatrix.T  *   lSize  (int)r?  *       Local version of Size; placed in a register for speed.e  *   Magnitude  (double),  *       Absolute value of diagonal element.   *   MinMarkowitzProduct  (long)H  *       Smallest Markowitz product found of pivot candidates which are   *       acceptable.  *   NumberOfTies  (int)M  *       A count of the number of Markowitz ties that have occured at currentn  *       MarkowitzProduct.   *   pElement  (ElementPointer) %  *       Pointer to current element.  &  *   pLargestElement  (ElementPointer)K  *       Pointer to the largest element yet found in the reduced submatrix.s  *   Product  (long):  *       Markowitz product for the current row and column.  *   Ratio  (double)6  *       For the current pivot candidate, Ratio is theF  *       Ratio of the largest element in its column to its magnitude.   *   RatioOfAccepted  (double)J  *       For the best pivot candidate found so far, RatioOfAccepted is theF  *       Ratio of the largest element in its column to its magnitude.   *  *   >>> Global variables:  *   gMatrix  <used>  *   LargestInCol  <set>  *   Size  <used>   *   Step  <used>   *   Threshold  <used>  *  *   >>> Possible errors: 
  *   SINGULARe  *   ILL_CONDITIONED z  */c   static ElementPointer  * SearchEntireMatrix() t   {i register  int  I, lSize;# register  ElementPointer  pElement;e int  NumberOfTies;# long  Product, MinMarkowitzProduct; - ElementPointer  ChosenPivot, pLargestElement; = double  Magnitude, LargestElementMag, Ratio, RatioOfAccepted;h  double  LargestInColOfAccepted;  double  FindLargestInCol();i" double  FindLargestInColExclude(); ELEMENT_MAG_DECLARATIONS;t       ChosenPivot = NULL;e     lSize = Size;n     LargestElementMag = 0.0;/     MinMarkowitzProduct = LARGEST_LONG_INTEGER;l  7 /* Begin search of matrix on column by column basis. */t#     for (I = Step; I <= lSize; I++) *     {   pElement = gMatrix->FirstInCol[I];  9         while (pElement != NULL AND pElement->Row < Step)c+             pElement = pElement->NextInCol;o  >         if((LargestInCol = FindLargestInCol(pElement)) == 0.0)$             continue; /* for loop */            while (pElement != NULL)	         { L /* Check to see if element is the largest encountered so far.  If so, record     its magnitude and address. */H             if ((Magnitude = ELEMENT_MAG(pElement)) > LargestElementMag).             {   LargestElementMag = Magnitude;+                 pLargestElement = pElement; 
             }a+ /* Calculate element's MarkowitzProduct. */e=             Product = gMatrix->MarkowitzRow[pElement->Row] * n;                       gMatrix->MarkowitzCol[pElement->Col];d  @ /* Test to see if element is acceptable as a pivot candidate. */4             if ((Product <= MinMarkowitzProduct) AND:                 (Magnitude > Threshold * LargestInCol) AND-                 (Magnitude > LowerThreshold)) 
             {aN /* Test to see if element has lowest MarkowitzProduct yet found, or whether it,    is tied with an element found earlier. */2                 if (Product < MinMarkowitzProduct)                 {iP /* Notice strict inequality in test. This is a new smallest MarkowitzProduct. */+                     ChosenPivot = pElement;e2                     MinMarkowitzProduct = Product;?                     RatioOfAccepted = LargestInCol / Magnitude;h:                     LargestInColOfAccepted = LargestInCol;%                     NumberOfTies = 0;e                 }t                 else                 {*' /* This case handles Markowitz ties. */s#                     NumberOfTies++;>5                     Ratio = LargestInCol / Magnitude;t0                     if (Ratio < RatioOfAccepted)/                     {   ChosenPivot = pElement; 0                         RatioOfAccepted = Ratio;>                         LargestInColOfAccepted = LargestInCol;                     }   ;                     if (NumberOfTies >= MAX_MARKOWITZ_TIES)g>                     {   LargestInCol = LargestInColOfAccepted;-                         return(ChosenPivot); m                     }c                 }a
             }e+             pElement = pElement->NextInCol; /         }  /* End of while(pElement != NULL) */O     } /* End of for(Step) */     if (ChosenPivot != NULL)         return(ChosenPivot);  !     if (LargestElementMag == 0.0)G"     {   gMatrix->Error = SINGULAR;         return(NULL);.     }   <     LargestInCol = FindLargestInColExclude(pLargestElement);%     gMatrix->Error = ILL_CONDITIONED;n     return(pLargestElement); }p                       = /*@  *   DETERMINE THE MAGNITUDE OF THE LARGEST ELEMENT IN A COLUMN   *L  *   This routine searches a column and returns the magnitude of the largestJ  *   element.  This routine begins the search at the element pointed to by  *   pElement, the parameter.s  *O  *   The search is conducted by starting at the element specified by a pointer,gM  *   which should be one below the diagonal, and moving down the column.  On  O  *   the way down the column, the magnitudes of the elements are tested to see  '  *   if they are the largest yet found.f  *  *   >>> Returned:L  *   The magnitude of the largest element in the column below and including /  *   the one pointed to by the input parameter.   *  *   >>> Arguements:(  *   pElement  <input>  (ElementPointer)K  *       The pointer to the first element to be tested.  Also, used by the I<  *       routine to access all lower elements in the column.  *  *   >>> Local variables:c  *   Largest  (double).  *       The magnitude of the largest element.  *   Magnitude  (double)7  *       The magnitude of the currently active element.b  */n  
 static doubleo FindLargestInCol( pElement )  # register  ElementPointer  pElement;  {e! double  Magnitude, Largest = 0.0;m ELEMENT_MAG_DECLARATIONS;l   /* Begin. */= /* Search column for largest element beginning at Element. */      while (pElement != NULL):     {   if ((Magnitude = ELEMENT_MAG(pElement)) > Largest)              Largest = Magnitude;'         pElement = pElement->NextInCol;r     }        return(Largest); }b                   n /*@  *   DETERMINE THE MAGNITUDE OF THE LARGEST ELEMENT IN A COLUMN   *   EXCLUDING AN ELEMENTs  *L  *   This routine searches a column and returns the magnitude of the largestJ  *   element.  One given element is specifically excluded from the search.  *K  *   The search is conducted by starting at the first element in the columnvO  *   and moving down the column until the active part of the matrix is entered, K  *   i.e. the reduced submatrix.  The rest of the column is then traversed  %  *   looking for the largest element.e  *  *   >>> Returned:N  *   The magnitude of the largest element in the active portion of the column,2  *   excluding the specified element, is returned.  *  *   >>> Arguements:(  *   pElement  <input>  (ElementPointer)N  *       The pointer to the element that is to be excluded from search. ColumnI  *       to be searched is one that contains this element.  Also used to e+  *       access the elements in the column.e  *  *   >>> Local variables:=  *   Col  (int)IL  *       The number of the column to be searched.  Also the column number of1  *       the element to be avoided in the search.r  *   Largest  (double).  *       The magnitude of the largest element.  *   lStep  (int)mN  *       A local version of Step, which indicates where the active part of the.  *       matrix begins.  Made local for speed.  *   Magnitude  (double)7  *       The magnitude of the currently active element.o  *   Row  (int)dB  *       The row number of element to be excluded from the search.  *  *   >>> Global variables:        *   gMatrix  <used>  *   Step  <used>   */e  
 static double # FindLargestInColExclude( pElement )y  # register  ElementPointer  pElement;g {e register  int  Row, lStep;	 int  Col;o double  Largest, Magnitude;t ELEMENT_MAG_DECLARATIONS;e   /* Begin. */     lStep = Step;      Row = pElement->Row;     Col = pElement->Col;(     pElement = gMatrix->FirstInCol[Col];  < /* Travel down column until reduced submatrix is entered. */:     while ((pElement != NULL) AND (pElement->Row < lStep))'         pElement = pElement->NextInCol;u  & /* Initialize the variable Largest. */     if (pElement->Row != Row) (         Largest = ELEMENT_MAG(pElement);     else         Largest = 0.0;  C /* Search rest of column for largest element, avoiding diagonal. */ 4     while ((pElement = pElement->NextInCol) != NULL):     {   if ((Magnitude = ELEMENT_MAG(pElement)) > Largest)%         {   if (pElement->Row != Row)-$                 Largest = Magnitude;	         }      }        return(Largest); }R                     /*  *   EXCHANGE ROWS AND COLUMNS  *N  *   Exchanges two rows and two columns so that the selected pivot is moved to6  *   the upper left corner of the remaining submatrix.  *  *   >>> Argument:&  *   pPivot  <input>  (ElementPointer)&  *       Pointer to the current pivot.  *  *   >>> Local variables:m  *   Col  (int)l7  *       Column that the pivot was originally found in.   *   Row  (int) 4  *       Row that the pivot was originally found in.!  *   OldMarkowitzProd_Col  (long)eJ  *       Markowitz product associated with the diagonal element in the row   *       the pivot was found in.!  *   OldMarkowitzProd_Row  (long)tM  *       Markowitz product associated with the diagonal element in the columne   *       the pivot was found in."  *   OldMarkowitzProd_Step  (long)M  *       Markowitz product associated with the diagonal element that is beinguL  *       moved so that the pivot can be placed in the upper left-hand corner"  *       of the reduced submatrix.  *  *   >>> Global variables:  *   gMatrix  <used>  *   PivotsOriginalCol  <set>w  *   PivotsOriginalRow  <set>   *   Singletons  <set>  *   Step  <used>e  */    static ExchangeRowsAndCols(pPivot)r  c ElementPointer  pPivot;a {O register  int   Row, Col;MH long  OldMarkowitzProd_Step, OldMarkowitzProd_Row, OldMarkowitzProd_Col;" ElementPointer FindElementInCol();   /* Begin. */     Row = pPivot->Row;     Col = pPivot->Col;     PivotsOriginalRow = Row;     PivotsOriginalCol = Col;  (     if ((Row == Step) AND (Col == Step))         return;/    /* Exchange rows and columns. */     if (Row == Col)p!     {   ExchangeRows (Step, Row);n!         ExchangeCols (Step, Col); N         SWAP (long, gMatrix->MarkowitzProd[Step],gMatrix->MarkowitzProd[Row]);E         SWAP (ElementPointer,gMatrix->Diag[Row],gMatrix->Diag[Step]);b     }s     else     {_  < /* Initialize variables that hold old Markowitz products. */=         OldMarkowitzProd_Step = gMatrix->MarkowitzProd[Step]; ;         OldMarkowitzProd_Row = gMatrix->MarkowitzProd[Row]; ;         OldMarkowitzProd_Col = gMatrix->MarkowitzProd[Col];e   /* Exchange rows. */         if (Row != Step)%         {   ExchangeRows (Step, Row); G             gMatrix->MarkowitzProd[Row] = gMatrix->MarkowitzRow[Row] * tN                                                    gMatrix->MarkowitzCol[Row];   /* Update singleton count. */eN             if ((gMatrix->MarkowitzProd[Row]==0) != (OldMarkowitzProd_Row==0)).             {   if (OldMarkowitzProd_Row == 0)!                     Singletons--;                  else!                     Singletons++;t
             }i	         }h   /* Exchange columns. */          if (Col != Step)%         {   ExchangeCols (Step, Col);mG             gMatrix->MarkowitzProd[Col] = gMatrix->MarkowitzCol[Col] *  N                                                    gMatrix->MarkowitzRow[Col];   /* Update singleton count. */uN             if ((gMatrix->MarkowitzProd[Col]==0) != (OldMarkowitzProd_Col==0)).             {   if (OldMarkowitzProd_Col == 0)!                     Singletons--;a                 else!                     Singletons++;t
             }u5                                                      rN             gMatrix->Diag[Col] = FindElementInCol(&(gMatrix->FirstInCol[Col]),P                                                                  Col,Col,FALSE);	         }e         if (Row != Step)O             gMatrix->Diag[Row] = FindElementInCol(&(gMatrix->FirstInCol[Row]), tP                                                                  Row,Row,FALSE);M         gMatrix->Diag[Step] = FindElementInCol(&(gMatrix->FirstInCol[Step]), sP                                                                Step,Step,FALSE);   /* Update singleton count. */ E         gMatrix->MarkowitzProd[Step] = gMatrix->MarkowitzCol[Step] * OP                                                     gMatrix->MarkowitzRow[Step];L         if ((gMatrix->MarkowitzProd[Step]==0) != (OldMarkowitzProd_Step==0))+         {   if (OldMarkowitzProd_Step == 0)                  Singletons--;=             else                 Singletons++;d	         }/       }( }                  x /*  *   EXCHANGE ROWS  *M  *   Performs all required operations to exchange two rows. Those operations eJ  *   include: swap FirstInRow pointers, fixing up the NextInCol pointers, L  *   swapping row indexes in MatrixElements, and swapping Markowitz row and   *   column counts.k  *  *   >>> Arguements:  *   Row1  <input>  (int)MC  *       Row index of one of the rows, becomes the smallest index. >  *   Row2  <input>  (int) ?  *       Row index of the other row, becomes the largest index.a  *  *   Local variables:o  *   Column  (int)E  *       Column in which row elements are currently being exchanged.    *   Row1Ptr  (ElementPointer)(  *       Pointer to an element in Row1.   *   Row2Ptr  (ElementPointer)(  *       Pointer to an element in Row2.   *   Element1  (ElementPointer)t9  *       Pointer to the element in Row1 to be exchanged.    *   Element2  (ElementPointer) 9  *       Pointer to the element in Row2 to be exchanged.    *  *   >>> Global variables:  *   gMatrix  <used>  */h   static ExchangeRows( Row1, Row2 )  ( int  Row1, Row2; { + register  ElementPointer  Row1Ptr, Row2Ptr;M int  Column;# ElementPointer  Element1, Element2;o   /* Begin. */,     if (Row1 > Row2)  SWAP(int, Row1, Row2);  (     Row1Ptr = gMatrix->FirstInRow[Row1];(     Row2Ptr = gMatrix->FirstInRow[Row2];.     while (Row1Ptr != NULL OR Row2Ptr != NULL)     {eC /* Exchange elements in rows while traveling from left to right. */          if (Row1Ptr == NULL)"         {   Column = Row2Ptr->Col;             Element1 = NULL;             Element2 = Row2Ptr;E)             Row2Ptr = Row2Ptr->NextInRow;n	         }o!         else if (Row2Ptr == NULL)_"         {   Column = Row1Ptr->Col;             Element1 = Row1Ptr;e             Element2 = NULL;)             Row1Ptr = Row1Ptr->NextInRow;C	         }T-         else if (Row1Ptr->Col < Row2Ptr->Col)t"         {   Column = Row1Ptr->Col;             Element1 = Row1Ptr;              Element2 = NULL;)             Row1Ptr = Row1Ptr->NextInRow;r	         } -         else if (Row1Ptr->Col > Row2Ptr->Col) "         {   Column = Row2Ptr->Col;             Element1 = NULL;             Element2 = Row2Ptr;e)             Row2Ptr = Row2Ptr->NextInRow;E	         }l1         else   /* Row1Ptr->Col == Row2Ptr->Col */="         {   Column = Row1Ptr->Col;             Element1 = Row1Ptr;R             Element2 = Row2Ptr;a)             Row1Ptr = Row1Ptr->NextInRow;L)             Row2Ptr = Row2Ptr->NextInRow; 	         }O  G         ExchangeElementsInCol( Row1, Element1, Row2, Element2, Column);e=     }  /* end of while(Row1Ptr != NULL OR Row2Ptr != NULL) */n  I     SWAP( int, gMatrix->MarkowitzRow[Row1], gMatrix->MarkowitzRow[Row2]);)P     SWAP( ElementPointer, gMatrix->FirstInRow[Row1], gMatrix->FirstInRow[Row2]);W     SWAP( int, gMatrix->RowEliminationOrder[Row1], gMatrix->RowEliminationOrder[Row2]);        return;t }o                   /*  *   EXCHANGE COLUMNSd  *P  *   Performs all required operations to exchange two columns. Those operations J  *   include: swap FirstInCol pointers, fixing up the NextInRow pointers, O  *   swapping column indexes in MatrixElements, and swapping Markowitz row and t  *   column counts.   *  *   >>> Arguements:  *   Col1  <input>  (int) I  *       Column index of one of the columns, becomes the smallest index. i  *   Col2  <input>  (int)nE  *       Column index of the other column, becomes the largest index P  *  *   Local variables:t  *   Row  (int)RE  *       Row in which column elements are currently being exchanged. u  *   Col1Ptr  (ElementPointer)(  *       Pointer to an element in Col1.   *   Col2Ptr  (ElementPointer)(  *       Pointer to an element in Col2.   *   Element1  (ElementPointer)g9  *       Pointer to the element in Col1 to be exchanged.    *   Element2  (ElementPointer)19  *       Pointer to the element in Col2 to be exchanged. w  *  *   >>> Global variables:  *   gMatrix  <used>  */n   static ExchangeCols( Col1, Col2 )  i int  Col1, Col2; {(+ register  ElementPointer  Col1Ptr, Col2Ptr;r	 int  Row; # ElementPointer  Element1, Element2;    /* Begin. */,     if (Col1 > Col2)  SWAP(int, Col1, Col2);  (     Col1Ptr = gMatrix->FirstInCol[Col1];(     Col2Ptr = gMatrix->FirstInCol[Col2];.     while (Col1Ptr != NULL OR Col2Ptr != NULL)     {2C /* Exchange elements in rows while traveling from top to bottom. */          if (Col1Ptr == NULL)         {   Row = Col2Ptr->Row;=             Element1 = NULL;             Element2 = Col2Ptr;l)             Col2Ptr = Col2Ptr->NextInCol;t	         }w!         else if (Col2Ptr == NULL)w         {   Row = Col1Ptr->Row;              Element1 = Col1Ptr;              Element2 = NULL;)             Col1Ptr = Col1Ptr->NextInCol;o	         }2-         else if (Col1Ptr->Row < Col2Ptr->Row)          {   Row = Col1Ptr->Row;              Element1 = Col1Ptr;;             Element2 = NULL;)             Col1Ptr = Col1Ptr->NextInCol; 	         }1-         else if (Col1Ptr->Row > Col2Ptr->Row)          {   Row = Col2Ptr->Row;O             Element1 = NULL;             Element2 = Col2Ptr;t)             Col2Ptr = Col2Ptr->NextInCol;r	         } 1         else   /* Col1Ptr->Row == Col2Ptr->Row */k         {   Row = Col1Ptr->Row;z             Element1 = Col1Ptr;i             Element2 = Col2Ptr;g)             Col1Ptr = Col1Ptr->NextInCol;g)             Col2Ptr = Col2Ptr->NextInCol;>	         }i  D         ExchangeElementsInRow( Col1, Element1, Col2, Element2, Row);=     }  /* end of while(Col1Ptr != NULL OR Col2Ptr != NULL) */a  I     SWAP( int, gMatrix->MarkowitzCol[Col1], gMatrix->MarkowitzCol[Col2]);iP     SWAP( ElementPointer, gMatrix->FirstInCol[Col1], gMatrix->FirstInCol[Col2]);W     SWAP( int, gMatrix->ColEliminationOrder[Col1], gMatrix->ColEliminationOrder[Col2]);l       return;t }                  e /*&  *   EXCHANGE TWO ELEMENTS IN A COLUMN  *K  *   Performs all required operations to exchange two elements in a column. O  *   Those operations are: restring NextInCol pointers and swapping row indexes   *   in the MatrixElements.m  *  *   >>> Arguements:  *   Row1  <input>  (int) -  *       Row of top element to be exchanged.  (  *   Element1  <input>  (ElementPointer)1  *       Pointer to top element to be exchanged. =  *   Row2  <input>  (int)]0  *       Row of bottom element to be exchanged. (  *   Element2  <input>  (ElementPointer)4  *       Pointer to bottom element to be exchanged.   *   Column <input>  (int)3  *       Column that exchange is to take place in.    *  *   >>> Local variables:e)  *   ElementAboveRow1  (ElementPointer *)/N  *       Location of pointer which points to the element above Element1. This K  *       pointer is modified so that it points to correct element on exit.  )  *   ElementAboveRow2  (ElementPointer *)tN  *       Location of pointer which points to the element above Element2. This K  *       pointer is modified so that it points to correct element on exit. n'  *   ElementBelowRow1  (ElementPointer)f,  *       Pointer to element below Element1. '  *   ElementBelowRow2  (ElementPointer)x,  *       Pointer to element below Element2.   *   pElement  (ElementPointer) .  *       Pointer used to traverse the column.   *  *   >>> Global variables:  *   gMatrix  <used>  */e   static' ExchangeElementsInCol( Row1, Element1,  /                        Row2, Element2, Column )r   - register  ElementPointer  Element1, Element2;i int  Row1, Row2, Column; {h5 ElementPointer  *ElementAboveRow1, *ElementAboveRow2;i3 ElementPointer  ElementBelowRow1, ElementBelowRow2; # register  ElementPointer  pElement;    /* Begin. */* /* Search to find the ElementAboveRow1. */6     ElementAboveRow1 = &(gMatrix->FirstInCol[Column]);!     pElement = *ElementAboveRow1;i      while (pElement->Row < Row1)2     {   ElementAboveRow1 = &(pElement->NextInCol);%         pElement = *ElementAboveRow1;*     }e     if (Element1 != NULL) /     {   ElementBelowRow1 = Element1->NextInCol;e         if (Element2 == NULL)l	         {,: /* Element2 does not exist, move Element1 down to Row2. */L             if ( ElementBelowRow1 != NULL AND ElementBelowRow1->Row < Row2 )
             {o: /* Element1 must be removed from linked list and moved. */5                 *ElementAboveRow1 = ElementBelowRow1;x   /* Search column for Row2. */r,                 pElement = ElementBelowRow1;                 do>                 {   ElementAboveRow2 = &(pElement->NextInCol);1                     pElement = *ElementAboveRow2;nF                 }   while (pElement != NULL AND pElement->Row < Row2);   /* Place Element1 in Row2. */i-                 *ElementAboveRow2 = Element1;:/                 Element1->NextInCol = pElement; 4                 *ElementAboveRow1 =ElementBelowRow1;
             } !             Element1->Row = Row2;b	         }.         else	         {iB /* Element2 does exist, and the two elements must be exchanged. *//             if ( ElementBelowRow1->Row == Row2) 
             {t5 /* Element2 is just below Element1, exchange them. */C:                 Element1->NextInCol = Element2->NextInCol;/                 Element2->NextInCol = Element1;i-                 *ElementAboveRow1 = Element2;i
             }e             else
             { C /* Element2 is not just below Element1 and must be searched for. */ ,                 pElement = ElementBelowRow1;                 do>                 {   ElementAboveRow2 = &(pElement->NextInCol);1                     pElement = *ElementAboveRow2; 1                 }   while (pElement->Row < Row2);P  t7                 ElementBelowRow2 = Element2->NextInCol;E  # /* Switch Element1 and Element2. */ -                 *ElementAboveRow1 = Element2;p7                 Element2->NextInCol = ElementBelowRow1;e-                 *ElementAboveRow2 = Element1;:7                 Element1->NextInCol = ElementBelowRow2;n
             }e!             Element1->Row = Row2;,!             Element2->Row = Row1; 	         }t     }m     else     {  /* Element1 does not exist. */$         ElementBelowRow1 = pElement;   /* Find Element2. */*         if (ElementBelowRow1->Row != Row2)         {   do:             {   ElementAboveRow2 = &(pElement->NextInCol);-                 pElement = *ElementAboveRow2;&-             }   while (pElement->Row < Row2);*  /         ElementBelowRow2 = Element2->NextInCol;)   /* Move Element2 to Row1. */4             *ElementAboveRow2 = Element2->NextInCol;)             *ElementAboveRow1 = Element2; 3             Element2->NextInCol = ElementBelowRow1; 	         }m         Element2->Row = Row1;x     }>     return;t }                      2 /*#  *   EXCHANGE TWO ELEMENTS IN A ROWl  *H  *   Performs all required operations to exchange two elements in a row.K  *   Those operations are: restring NextInRow pointers and swapping column 1#  *   indexes in the MatrixElements.   *  *   >>> Arguements:  *   Col1  <input>  (int)P3  *       Col of left-most element to be exchanged. l(  *   Element1  <input>  (ElementPointer)7  *       Pointer to left-most element to be exchanged. o  *   Col2  <input>  (int) 4  *       Col of right-most element to be exchanged. (  *   Element2  <input>  (ElementPointer)8  *       Pointer to right-most element to be exchanged.   *   Row <input>  (int) 0  *       Row that exchange is to take place in.   *  *   >>> Local variables:1*  *   ElementLeftOfCol1  (ElementPointer *)H  *       Location of pointer which points to the element to the left of I  *       Element1. This pointer is modified so that it points to correct    *       element on exit. *  *   ElementLeftOfRow2  (ElementPointer *)H  *       Location of pointer which points to the element to the left of I  *       Element2. This pointer is modified so that it points to correct t  *       element on exit. )  *   ElementRightOfRow1  (ElementPointer)R/  *       Pointer to element right of Element1.  )  *   ElementRightOfRow2  (ElementPointer)n/  *       Pointer to element right of Element2. x  *   pElement  (ElementPointer)o+  *       Pointer used to traverse the row. h  *  *   >>> Global variables:  *   gMatrix  <used>  */i   static' ExchangeElementsInRow( Col1, Element1, (,                        Col2, Element2, Row )  o int  Col1, Col2, Row; , register ElementPointer  Element1, Element2;   {d7 ElementPointer  *ElementLeftOfCol1, *ElementLeftOfCol2;l7 ElementPointer  ElementRightOfCol1, ElementRightOfCol2; $ register   ElementPointer  pElement;   /* Begin. */+ /* Search to find the ElementLeftOfCol1. */n4     ElementLeftOfCol1 = &(gMatrix->FirstInRow[Row]);"     pElement = *ElementLeftOfCol1;      while (pElement->Col < Col1)3     {   ElementLeftOfCol1 = &(pElement->NextInRow); &         pElement = *ElementLeftOfCol1;     }x     if (Element1 != NULL)E1     {   ElementRightOfCol1 = Element1->NextInRow;          if (Element2 == NULL) 	         {B> /* Element2 does not exist, move Element1 to right to Col2. */P             if ( ElementRightOfCol1 != NULL AND ElementRightOfCol1->Col < Col2 )
             { : /* Element1 must be removed from linked list and moved. */8                 *ElementLeftOfCol1 = ElementRightOfCol1;   /* Search Row for Col2. */.                 pElement = ElementRightOfCol1;                 do?                 {   ElementLeftOfCol2 = &(pElement->NextInRow);o2                     pElement = *ElementLeftOfCol2;F                 }   while (pElement != NULL AND pElement->Col < Col2);   /* Place Element1 in Col2. */ .                 *ElementLeftOfCol2 = Element1;/                 Element1->NextInRow = pElement;*7                 *ElementLeftOfCol1 =ElementRightOfCol1;*
             }>!             Element1->Col = Col2;t	         }          else	         {eB /* Element2 does exist, and the two elements must be exchanged. */1             if ( ElementRightOfCol1->Col == Col2) 
             {x8 /* Element2 is just right of Element1, exchange them. */:                 Element1->NextInRow = Element2->NextInRow;/                 Element2->NextInRow = Element1;h.                 *ElementLeftOfCol1 = Element2;
             }              else
             {eF /* Element2 is not just right of Element1 and must be searched for. */.                 pElement = ElementRightOfCol1;                 do?                 {   ElementLeftOfCol2 = &(pElement->NextInRow);e2                     pElement = *ElementLeftOfCol2;1                 }   while (pElement->Col < Col2);n  9                 ElementRightOfCol2 = Element2->NextInRow;t  # /* Switch Element1 and Element2. */ .                 *ElementLeftOfCol1 = Element2;9                 Element2->NextInRow = ElementRightOfCol1;>.                 *ElementLeftOfCol2 = Element1;9                 Element1->NextInRow = ElementRightOfCol2; 
             }C!             Element1->Col = Col2;,!             Element2->Col = Col1;e	         },     }t     else     {P /* Element1 does not exist. */&         ElementRightOfCol1 = pElement;   /* Find Element2. */,         if (ElementRightOfCol1->Col != Col2)         {   do;             {   ElementLeftOfCol2 = &(pElement->NextInRow);L.                 pElement = *ElementLeftOfCol2;-             }   while (pElement->Col < Col2);L  5             ElementRightOfCol2 = Element2->NextInRow;.   /* Move Element2 to Col1. */5             *ElementLeftOfCol2 = Element2->NextInRow;o*             *ElementLeftOfCol1 = Element2;5             Element2->NextInRow = ElementRightOfCol1;l	         }          Element2->Col = Col1;l     }      return;e }t                     w /*6  *   PERFORM ROW AND COLUMN ELIMINATION ON REAL MATRIX  *N  *   Eliminates a single row and column of the matrix and leaves single row ofL  *   the upper triangular matrix and a single column of the lower triangular/  *   matrix in its wake.  Uses Gauss's method.    *  *   >>> Argument:&  *   pPivot  <input>  (ElementPointer)&  *       Pointer to the current pivot.  *  *   >>> Local variables:m  *   ColPtr  (ElementPointer) >  *       Points to matrix element in lower triangular column. +  *   LargestInEliminatedRowAndCol  (double)-M  *       Contains the value of the largest element in the row or column being H  *       eliminated.  Used in calculation of element growth for roundoff  *       error estimation.  *   pElement (ElementPointer)'  *       Points to elements in matrix.    *   PivotMagnitude  (double)m6  *       Magnitude of the element being used as pivot.  *   Row  (int)   *       Row index.   *   RowPtr  (ElementPointer)m;  *       Points to matrix element in upper triangular row. o  *   UpperTri  (double)22  *       Current upper triangular matrix element.   *  *   >>> Global variable:/  *   gMatrix  <used>!  *   LargestAfterDecompose  <set>n  *   Step  <used>   *  *   >>> Possible errors:e  *   NO_MEMORY    */    static RealRowColElimination(pPivot)      ! register  ElementPointer  pPivot;n {g
 #if (REAL)  3 register  ElementPointer  pElement, ColPtr, RowPtr;  register  int  Row; @ double  UpperTri, PivotMagnitude,  LargestInEliminatedRowAndCol; ELEMENT_MAG_DECLARATIONS;L ElementPointer  CreateFillIn();o   /* Begin. */ /* Perform elimination. */ o!     pPivot = gMatrix->Diag[Step]; (     PivotMagnitude = ABS(pPivot->Real);    /* Test for zero pivot. */     if (PivotMagnitude == 0.0)"     {   gMatrix->Error = SINGULAR;         return;>     }o   #if (MONITOR_ROUNDOFF)E     LargestInEliminatedRowAndCol = MAX(PivotMagnitude, LargestInCol);n #endif   #if (CALCULATE_PSEUDOCONDITION)pG     if (PivotMagnitude < SmallestPivot) SmallestPivot = PivotMagnitude;CE     if (PivotMagnitude > LargestPivot) LargestPivot = PivotMagnitude;e #endif       RowPtr = pPivot;0     while ((RowPtr = RowPtr->NextInRow) != NULL)     {n  ) /* Calculate upper triangular element. */f2         UpperTri = (RowPtr->Real /= pPivot->Real);   #if (MONITOR_ROUNDOFF)9         if (ABS(UpperTri) > LargestInEliminatedRowAndCol)t9             LargestInEliminatedRowAndCol = ABS(UpperTri);l #endif                          pElement = RowPtr;         ColPtr = pPivot;4         while ((ColPtr = ColPtr->NextInCol) != NULL)         {   Row = ColPtr->Row;  I /* Find element in row being eliminated that lines up with current lower s    triangular element. */r<             while ((pElement = pElement->NextInCol) != NULL))             {   if (pElement->Row == Row)t+                     goto RealElementExists;<-                 else if (pElement->Row > Row)h+                     break; /* while loop */s
             }   4 /* Desired element was not found, create fill-in. */6             pElement = CreateFillIn(Row, RowPtr->Col);,             if (gMatrix->Error == NO_MEMORY)                 return;    RealElementExists:6             pElement->Real -= UpperTri * ColPtr->Real;	         } ?     }  /* end of while((RowPtr = RowPtr->NextInRow) != NULL) */e   #if (MONITOR_ROUNDOFF)F /* Update the value of the largest element in the matrices L and U. */=     if (LargestInEliminatedRowAndCol > LargestAfterDecompose)R=         LargestAfterDecompose = LargestInEliminatedRowAndCol;m #endif       return;t   #endif  /* REAL */ }p                 * /*9  *   PERFORM ROW AND COLUMN ELIMINATION ON COMPLEX MATRIXG  *N  *   Eliminates a single row and column of the matrix and leaves single row ofL  *   the upper triangular matrix and a single column of the lower triangular/  *   matrix in its wake.  Uses Gauss's method. l  * P  *   >>> Argument:&  *   pPivot  <input>  (ElementPointer)&  *       Pointer to the current pivot.  *  *   >>> Local variables:i  *   ColPtr  (ElementPointer)t>  *       Points to matrix element in lower triangular column. +  *   LargestInEliminatedRowAndCol  (double)aM  *       Contains the value of the largest element in the row or column being1H  *       eliminated.  Used in calculation of element growth for roundoff  *       error estimation.  *   pElement (ElementPointer)'  *       Points to elements in matrix. e  *   PivotMagnitude  (double) 6  *       Magnitude of the element being used as pivot.  *   Row  (int)L  *       Row index.   *   RowPtr  (ElementPointer)C;  *       Points to matrix element in upper triangular row.    *   UpperTriImag  (double)oG  *       Imaginary portion of current upper triangular matrix element.    *   UpperTriReal  (double)lB  *       Real portion of current upper triangular matrix element.   *  *   >>> Global variables:  *   gMatrix  <used>!  *   LargestAfterDecompose  <set>O  *   Step  <used>   *  *   Possible errors:t  *   NO_MEMORY  */e   static  ComplexRowColElimination(pPivot)  ! register  ElementPointer  pPivot;o { 
 #if (COMPLEX)   3 register  ElementPointer  pElement, ColPtr, RowPtr;e register  int  Row;  double  Magnitude;  A double  UpperTriReal, UpperTriImag, LargestInEliminatedRowAndCol;e ELEMENT_MAG_DECLARATIONS;f CMPLX_DIVIDE_DECLARATIONS; ElementPointer  CreateFillIn();    /* Begin. */ /* Perform elimination. */ O!     pPivot = gMatrix->Diag[Step]; %     Magnitude = ELEMENT_MAG(pPivot); f   /* Test for zero pivot. */     if (Magnitude == 0.0))"     {   gMatrix->Error = SINGULAR;         return;N     }w   #if (CALCULATE_PSEUDOCONDITION)t"     if (Magnitude < SmallestPivot)"         SmallestPivot = Magnitude;!     if (Magnitude > LargestPivot)f!         LargestPivot = Magnitude;) #endif   #if (MONITOR_ROUNDOFF)?     LargestInEliminatedRowAndCol = MAX(Magnitude,LargestInCol);; #endif       RowPtr = pPivot;0     while ((RowPtr = RowPtr->NextInRow) != NULL)     {M  ) /* Calculate upper triangular element. */ 1 /* Complex divide : UpperTri = RowPtr / pPivot */i?         CMPLX_DIVIDE(UpperTriReal, UpperTriImag, RowPtr->Real, i@                       RowPtr->Imag, pPivot->Real, pPivot->Imag);   #if (MONITOR_ROUNDOFF):         Magnitude = ABS(UpperTriReal) + ABS(UpperTriImag);5         if (Magnitude > LargestInEliminatedRowAndCol)r5             LargestInEliminatedRowAndCol = Magnitude;t #endif                u% /* Store upper triangular element. */:$         RowPtr->Real = UpperTriReal;$         RowPtr->Imag = UpperTriImag;                >         pElement = RowPtr;         ColPtr = pPivot;4         while ((ColPtr = ColPtr->NextInCol) != NULL)         {   Row = ColPtr->Row;  I /* Find element in row being eliminated that lines up with current lower h3    triangular element and subtract off LowerTri. */ <             while ((pElement = pElement->NextInCol) != NULL))             {   if (pElement->Row == Row)*.                     goto ComplexElementExists;-                 else if (pElement->Row > Row)h+                     break; /* while loop */t
             }w  4 /* Desired element was not found, create fill-in. */6             pElement = CreateFillIn(Row, RowPtr->Col);,             if (gMatrix->Error == NO_MEMORY)                 return;g   ComplexElementExists:*8 /* Complex expression : pElement -= UpperTri * ColPtr */P             CMPLX_MULT_SUBT_ASSIGN(pElement->Real, pElement->Imag, UpperTriReal,H                               UpperTriImag, ColPtr->Real, ColPtr->Imag);  	         }o?     }  /* end of while((RowPtr = RowPtr->NextInRow) != NULL) */i   #if (MONITOR_ROUNDOFF)F /* Update the value of the largest element in the matrices L and U. */=     if (LargestInEliminatedRowAndCol > LargestAfterDecompose) =         LargestAfterDecompose = LargestInEliminatedRowAndCol;C #endif       return;R   #endif  /* COMPLEX */C }A         m /*  *   UPDATE MARKOWITZ NUMBERSB  *O  *   Updates the Markowitz numbers after a row and column have been eliminated.t"  *   Also updates singleton count.  *  *   >>> Argument:&  *   pPivot  <input>  (ElementPointer)&  *       Pointer to the current pivot.  *  *   >>> Local variables:D  *   Row  (int)   *       Row index.   *   Col  (int)   *       Column index. ;  *   ColPtr  (ElementPointer)v>  *       Points to matrix element in upper triangular column.   *   RowPtr  (ElementPointer)a;  *       Points to matrix element in lower triangular row. o  *  *   >>> Global variables:  *   gMatrix  <used>  *   Singletons  <set>  */u   static UpdateMarkowitzNumbers(pPivot)   ElementPointer  pPivot;w {  register  int  Row, Col;) register  ElementPointer  ColPtr, RowPtr;-   /* Begin. */ /* Update Markowitz numbers. */v     ColPtr = pPivot;0     while ((ColPtr = ColPtr->NextInCol) != NULL)     {   Row = ColPtr->Row;E         gMatrix->MarkowitzProd[Row] = --gMatrix->MarkowitzRow[Row] * iO                                                     gMatrix->MarkowitzCol[Row];e,         if (gMatrix->MarkowitzRow[Row] == 0)             Singletons++;      }=       RowPtr = pPivot;0     while ((RowPtr = RowPtr->NextInRow) != NULL)     {   Col = RowPtr->Col;`         gMatrix->MarkowitzProd[Col] = gMatrix->MarkowitzRow[Col] * --gMatrix->MarkowitzCol[Col];T         if ((gMatrix->MarkowitzCol[Col] == 0) AND (gMatrix->MarkowitzRow[Col] != 0))             Singletons++;      }p     return;I }l                     /*  *   CREATE FILL-IN=  *E  *   This routine is used to create fill-ins and splice them into thei  *   matrix.  *  *   >>> Returns:   *   Pointer to fill-in.  *  *   >>> Arguements:  *   Col  <input>  (int)#  *       Column index for element.    *   Row  <input>  (int)   *       Row index for element.   *  *   >>> Local variables:   *   pElement  (ElementPointer)l/  *       Pointer to an element in the matrix.  T'  *   ppElementAbove  (ElementPointer *) P  *       This contains the address of the pointer to the element just above the M  *       one being created. It is used to speed the search and it is updated (/  *       with address of the created element.  o  *  *   >>> Global variables:  *   gMatrix - (used)e  *   Singletons - (set)   *  *   >>> Possible errors:/  *   NO_MEMORY  */    static ElementPointer  CreateFillIn (Row, Col)    register int  Row;	 int  Col;e    {g4 register  ElementPointer  pElement, *ppElementAbove;  ElementPointer  CreateElement();   /* Begin. */! /* Find Element above fill-in. */u/     ppElementAbove = &gMatrix->FirstInCol[Col];i     pElement = *ppElementAbove;P     while (pElement != NULL)      {   if (pElement->Row < Row)2         {   ppElementAbove = &pElement->NextInCol;'             pElement = *ppElementAbove;r	         }N$         else break; /* while loop */     }>  ( /* End of search, create the element. */7     pElement = CreateElement(Row, Col, ppElementAbove);m  + /* Update Markowitz counts and products. */(\     gMatrix->MarkowitzProd[Row] = ++gMatrix->MarkowitzRow[Row] * gMatrix->MarkowitzCol[Row];P     if ((gMatrix->MarkowitzRow[Row] == 1) AND (gMatrix->MarkowitzCol[Row] != 0))         Singletons--;i\     gMatrix->MarkowitzProd[Col] = gMatrix->MarkowitzRow[Col] * ++gMatrix->MarkowitzCol[Col];P     if ((gMatrix->MarkowitzRow[Col] != 0) AND (gMatrix->MarkowitzCol[Col] == 1))         Singletons--;=	          x     return (pElement); }o                     /*(  *   CALCULATE ROUNDOFF ERROR ESTIMATION  *M  *   Using the growth that occurred in the matrix, the largest element in theMK  *   matrix and the resolution of the machine, an estimation of the maximum J  *   amount of roundoff error present in the triangular matrices L an U isM  *   made.  Note that this is not an estimation of the amount of error in thedE  *   solution, which is a function of the conditioning of the matrix.   *  *   >>> Returned:F  *   An estimate of the maximum amount of roundoff present in L and U.  *  *   >>> Arguements:  *   Matrix  <input>  (char *)K  *       A pointer to the matrix for which the error estimation is desired.s  *  *   >>> Local variables:   *   Roundoff  (double)t  *       The error estimate.  */*   double   MatrixRoundoffError (Matrix) (    register  MatrixPointer  Matrix; {. #if (MONITOR_ROUNDOFF) double  Roundoff;i   /* Begin. */V      Roundoff = 3.01 * (Matrix->Size) * (Matrix->LargestElement) * MACHINE_RESOLUTION;      return (Roundoff);x   #else       return(0.0);l #endif   /* MONITOR_ROUNDOFF */o }i                 h /*  *  *   WRITE STATUSs  *?  *   Write a summary of important variables to standard output.r  *  *   >>> Global variables:  *   gMatrix  <used>!  *   PivotSelectionMethod  <used>o  *   PivotsOriginalCol  <used>  *   PivotsOriginalRow  <used>  *   Size  <used>i  *   Singletons  <used>O  *   Step  <used>E  */t   static
 WriteStatus()o   {l #if (ANNOTATE == FULL)  	 int  I;      /* Begin. */"     printf("Step = %1d   ", Step);\     printf("Pivot Originally found at %1d,%1d using ", PivotsOriginalRow,PivotsOriginalCol);      switch(PivotSelectionMethod)9     {   case 's': printf("SearchForSingleton\n");  break;l<         case 'q': printf("QuicklySearchDiagonal\n");  break;5         case 'd': printf("SearchDiagonal\n");  break; 9         case 'e': printf("SearchEntireMatrix\n");  break;e     }w&                                       "     printf("MarkowitzRow     = ");     for (I = 1; I <= Size; I++)i1         printf("%2d  ",gMatrix->MarkowitzRow[I]);      printf("\n");e  "     printf("MarkowitzCol     = ");     for (I = 1; I <= Size; I++) 1         printf("%2d  ",gMatrix->MarkowitzCol[I]);i     printf("\n");   "     printf("MarkowitzProduct = ");     for (I = 1; I <= Size; I++)o2         printf("%2d  ",gMatrix->MarkowitzProd[I]);     printf("\n");-  ,     printf("Singletons = %2d\n",Singletons);  "     printf("RowElimOrder     = ");     for (I = 1; I <= Size; I++) 8         printf("%2d  ",gMatrix->RowEliminationOrder[I]);     printf("\n");   "     printf("ColElimOrder     = ");     for (I = 1; I <= Size; I++),8         printf("%2d  ",gMatrix->ColEliminationOrder[I]);     printf("\n\n");P   /*  PrintMatrix(FALSE); */         return;/    #endif   /* ANNOTATE == FULL */  }R                  