; /* deflate.c -- compress data using the deflation algorithm ,  * Copyright (C) 1995-1996 Jean-loup Gailly.J  * For conditions of distribution and use, see copyright notice in zlib.h   */    /*
  *  ALGORITHM   *J  *      The "deflation" process depends on being able to identify portionsH  *      of the input text which are identical to earlier input (within aL  *      sliding window trailing behind the input currently being processed).  *J  *      The most straightforward technique turns out to be the fastest forJ  *      most input files: try all possible matches and select the longest.L  *      The key feature of this algorithm is that insertions into the stringK  *      dictionary are very simple and thus fast, and deletions are avoided M  *      completely. Insertions are performed at each input character, whereas M  *      string matches are performed only when the previous match ends. So it M  *      is preferable to spend more time in matches to allow very fast string H  *      insertions and avoid deletions. The matching algorithm for smallM  *      strings is inspired from that of Rabin & Karp. A brute force approach I  *      is used to find longer strings when a small match has been found. J  *      A similar algorithm is used in comic (by Jan-Mark Wams) and freeze  *      (by Leonid Broukhis). N  *         A previous version of this file used a more sophisticated algorithmL  *      (by Fiala and Greene) which is guaranteed to run in linear amortizedN  *      time, but has a larger average cost, uses more memory and is patented.I  *      However the F&G algorithm may be faster for some highly redundant O  *      files if the parameter max_chain_length (described below) is too large.   *  *  ACKNOWLEDGEMENTS  *K  *      The idea of lazy evaluation of matches is due to Jan-Mark Wams, and :  *      I found it in 'freeze' written by Leonid Broukhis.:  *      Thanks to many people for bug reports and testing.  *  *  REFERENCES  *G  *      Deutsch, L.P.,"'Deflate' Compressed Data Format Specification". F  *      Available in ftp.uu.net:/pub/archiving/zip/doc/deflate-1.1.doc  *J  *      A description of the Rabin and Karp algorithm is given in the book>  *         "Algorithms" by R. Sedgewick, Addison-Wesley, p252.  *#  *      Fiala,E.R., and Greene,D.H. N  *         Data Compression with Finite Windows, Comm.ACM, 32,4 (1989) 490-595  *  */   : /* $Id: deflate.c,v 1.8 1995/05/03 17:27:08 jloup Exp $ */   #include "deflate.h"  P char deflate_copyright[] = " deflate 1.0 Copyright 1995-1996 Jean-loup Gailly "; /*H   If you use the zlib library in a product, an acknowledgment is welcomeE   in the documentation of your product. If for some reason you cannot G   include such an acknowledgment, I would appreciate that you keep this 5   copyright string in the executable of your product.   */   N /* ===========================================================================  *  Function prototypes.  */ 1 local void fill_window    OF((deflate_state *s)); < local int  deflate_stored OF((deflate_state *s, int flush));< local int  deflate_fast   OF((deflate_state *s, int flush));< local int  deflate_slow   OF((deflate_state *s, int flush));1 local void lm_init        OF((deflate_state *s)); A local int longest_match   OF((deflate_state *s, IPos cur_match)); 9 local void putShortMSB    OF((deflate_state *s, uInt b)); / local void flush_pending  OF((z_stream *strm)); J local int read_buf        OF((z_stream *strm, charf *buf, unsigned size)); #ifdef ASMV ?       void match_init OF((void)); /* asm code initialization */  #endif   #ifdef DEBUGE local  void check_match OF((deflate_state *s, IPos start, IPos match, )                             int length));  #endif  N /* ===========================================================================
  * Local data   */   
 #define NIL 0  /* Tail of hash chains */    #ifndef TOO_FAR  #  define TOO_FAR 4096 #endifI /* Matches of length 3 are discarded if their distance exceeds TOO_FAR */   - #define MIN_LOOKAHEAD (MAX_MATCH+MIN_MATCH+1) D /* Minimum amount of lookahead, except at the end of the input file.4  * See deflate.c for comments about the MIN_MATCH+1.  */   ? typedef int (*compress_func) OF((deflate_state *s, int flush));  /* Compressing function */  K /* Values for max_lazy_match, good_match and max_chain_length, depending on K  * the desired pack level (0..9). The values given below have been tuned to N  * exclude worst case performance for pathological files. Better values may be  * found for specific files.  */  typedef struct config_s { D    ush good_length; /* reduce lazy search above this match length */L    ush max_lazy;    /* do not perform lazy search above this match length */=    ush nice_length; /* quit search above this match length */     ush max_chain;     compress_func func;	 } config;   ( local config configuration_table[10] = { /*      good lazy nice chain */ > /* 0 */ {0,    0,  0,    0, deflate_stored},  /* store only */O /* 1 */ {4,    4,  8,    4, deflate_fast}, /* maximum speed, no lazy matches */ * /* 2 */ {4,    5, 16,    8, deflate_fast},* /* 3 */ {4,    6, 32,   32, deflate_fast},  > /* 4 */ {4,    4, 16,   16, deflate_slow},  /* lazy matches */* /* 5 */ {8,   16, 32,   32, deflate_slow},* /* 6 */ {8,   16, 128, 128, deflate_slow},* /* 7 */ {8,   32, 128, 256, deflate_slow},+ /* 8 */ {32, 128, 258, 1024, deflate_slow}, F /* 9 */ {32, 258, 258, 4096, deflate_slow}}; /* maximum compression */  M /* Note: the deflate() code requires max_lazy >= MIN_MATCH and max_chain >= 4 L  * For deflate_fast() (levels <= 3) good is ignored and lazy has a different  * meaning.   */    #define EQUAL 0 ( /* result of memcmp for equal strings */  A struct static_tree_desc_s {int dummy;}; /* for buggy compilers */   N /* ===========================================================================0  * Update a hash value with the given input byteG  * IN  assertion: all calls to to UPDATE_HASH are made with consecutive K  *    input characters, so that a running hash key can be computed from the ?  *    previous key instead of complete recalculation each time.   */ L #define UPDATE_HASH(s,h,c) (h = (((h)<<s->hash_shift) ^ (c)) & s->hash_mask)    N /* ===========================================================================N  * Insert string str in the dictionary and set match_head to the previous headH  * of the hash chain (the most recent string with same hash key). Return)  * the previous length of the hash chain. I  * IN  assertion: all calls to to INSERT_STRING are made with consecutive E  *    input characters and the first MIN_MATCH bytes of str are valid @  *    (except for the last MIN_MATCH-1 bytes of the input file).  */ + #define INSERT_STRING(s, str, match_head) \ A    (UPDATE_HASH(s, s->ins_h, s->window[(str) + (MIN_MATCH-1)]), \ B     s->prev[(str) & s->w_mask] = match_head = s->head[s->ins_h], \     s->head[s->ins_h] = (str))  N /* ===========================================================================H  * Initialize the hash table (avoiding 64K overflow for 16 bit systems).)  * prev[] will be initialized on the fly.   */  #define CLEAR_HASH(s) \ $     s->head[s->hash_size-1] = NIL; \L     zmemzero((charf *)s->head, (unsigned)(s->hash_size-1)*sizeof(*s->head));  O /* ========================================================================= */ 3 int deflateInit_(strm, level, version, stream_size)      z_stream *strm;      int level;     const char *version;     int stream_size; { K     return deflateInit2_(strm, level, Z_DEFLATED, MAX_WBITS, DEF_MEM_LEVEL, . 			 Z_DEFAULT_STRATEGY, version, stream_size);<     /* To do: ignore strm->next_in if we use it as window */ }   O /* ========================================================================= */ F int deflateInit2_(strm, level, method, windowBits, memLevel, strategy, 		  version, stream_size)      z_stream *strm;      int  level;      int  method;     int  windowBits;     int  memLevel;     int  strategy;     const char *version;     int stream_size; {      deflate_state *s;      int noheader = 0;         ushf *overlay; L      /* We overlay pending_buf and d_buf+l_buf. This works since the averageI       * output size for (length,distance) codes is <= 24 bits (worst casec       * is 15+15+13=33).       */  =     if (version == Z_NULL || version[0] != ZLIB_VERSION[0] ||**         stream_size != sizeof(z_stream)) { 	return Z_VERSION_ERROR;     } .     if (strm == Z_NULL) return Z_STREAM_ERROR;       strm->msg = Z_NULL; !     if (strm->zalloc == Z_NULL) {e 	strm->zalloc = zcalloc; 	strm->opaque = (voidpf)0;     }i4     if (strm->zfree == Z_NULL) strm->zfree = zcfree;  2     if (level == Z_DEFAULT_COMPRESSION) level = 6;  J     if (windowBits < 0) { /* undocumented feature: suppress zlib header */         noheader = 1;*!         windowBits = -windowBits;d     }aK     if (memLevel < 1 || memLevel > MAX_MEM_LEVEL || method != Z_DEFLATED ||aF         windowBits < 8 || windowBits > 15 || level < 0 || level > 9 ||- 	strategy < 0 || strategy > Z_HUFFMAN_ONLY) {          return Z_STREAM_ERROR;     }aA     s = (deflate_state *) ZALLOC(strm, 1, sizeof(deflate_state));t(     if (s == Z_NULL) return Z_MEM_ERROR;1     strm->state = (struct internal_state FAR *)s;.     s->strm = strm;c       s->noheader = noheader;g     s->w_bits = windowBits;      s->w_size = 1 << s->w_bits;a     s->w_mask = s->w_size - 1;        s->hash_bits = memLevel + 7;%     s->hash_size = 1 << s->hash_bits;e$     s->hash_mask = s->hash_size - 1;<     s->hash_shift =  ((s->hash_bits+MIN_MATCH-1)/MIN_MATCH);  B     s->window = (Bytef *) ZALLOC(strm, s->w_size, 2*sizeof(Byte));?     s->prev   = (Posf *)  ZALLOC(strm, s->w_size, sizeof(Pos));eB     s->head   = (Posf *)  ZALLOC(strm, s->hash_size, sizeof(Pos));  G     s->lit_bufsize = 1 << (memLevel + 6); /* 16K elements by default */   D      overlay = (ushf *) ZALLOC(strm, s->lit_bufsize, sizeof(ush)+2);'      s->pending_buf = (uchf *) overlay;   H     if (s->window == Z_NULL || s->prev == Z_NULL || s->head == Z_NULL ||#         s->pending_buf == Z_NULL) { )         strm->msg = ERR_MSG(Z_MEM_ERROR);P         deflateEnd (strm);         return Z_MEM_ERROR;      }i5      s->d_buf = overlay + s->lit_bufsize/sizeof(ush);d@      s->l_buf = s->pending_buf + (1+sizeof(ush))*s->lit_bufsize;       s->level = level;      s->strategy = strategy;c     s->method = (Byte)method;*       return deflateReset(strm); }*  O /* ========================================================================= */*7 int deflateSetDictionary (strm, dictionary, dictLength)x     z_stream *strm;e     const Bytef *dictionary;     uInt  dictLength;C {r     deflate_state *s;p     uInt length = dictLength;e     uInt n;      IPos hash_head;o  J     if (strm == Z_NULL || strm->state == Z_NULL || dictionary == Z_NULL ||A         strm->state->status != INIT_STATE) return Z_STREAM_ERROR;e       s = strm->state;?     strm->adler = adler32(strm->adler, dictionary, dictLength);=  (     if (length < MIN_MATCH) return Z_OK;     if (length > MAX_DIST(s)) {* 	length = MAX_DIST(s);# 	dictionary += dictLength - length;e     }t4     zmemcpy((charf *)s->window, dictionary, length);     s->strstart = length;n"     s->block_start = (long)length;  L     /* Insert all strings in the hash table (except for the last two bytes).J      * s->lookahead stays null, so s->ins_h will be recomputed at the next      * call of fill_window.c      */)     s->ins_h = s->window[0];+     UPDATE_HASH(s, s->ins_h, s->window[1]);f/     for (n = 0; n <= length - MIN_MATCH; n++) {e  	INSERT_STRING(s, n, hash_head);     }b     return Z_OK; }   O /* ========================================================================= */d int deflateReset (strm)      z_stream *strm;( {l     deflate_state *s;t     2     if (strm == Z_NULL || strm->state == Z_NULL ||O         strm->zalloc == Z_NULL || strm->zfree == Z_NULL) return Z_STREAM_ERROR;=  )     strm->total_in = strm->total_out = 0; K     strm->msg = Z_NULL; /* use zfree if we ever allocate msg dynamically */e      strm->data_type = Z_UNKNOWN;  %     s = (deflate_state *)strm->state;R     s->pending = 0;O$     s->pending_out = s->pending_buf;       if (s->noheader < 0) {G         s->noheader = 0; /* was set to -1 by deflate(..., Z_FINISH); */h     }A6     s->status = s->noheader ? BUSY_STATE : INIT_STATE;     strm->adler = 1;     s->last_flush = Z_NO_FLUSH;        _tr_init(s);     lm_init(s);t       return Z_OK; },  O /* ========================================================================= */t( int deflateParams(strm, level, strategy)     z_stream *strm;l     int level;     int strategy;f {s     deflate_state *s;y     compress_func func;   G     if (strm == Z_NULL || strm->state == Z_NULL) return Z_STREAM_ERROR;h     s = strm->state;  )     if (level == Z_DEFAULT_COMPRESSION) {  	level = 6;c     }hN     if (level < 0 || level > 9 || strategy < 0 || strategy > Z_HUFFMAN_ONLY) { 	return Z_STREAM_ERROR;o     }f.     func = configuration_table[s->level].func;  /     if (func != configuration_table[level].funct" 	&& strm->state->lookahead != 0) {   	/* Flush the last buffer: */t- 	(void)(*func)(strm->state, Z_PARTIAL_FLUSH);*     }      if (s->level != level) { 	s->level = level;; 	s->max_lazy_match   = configuration_table[level].max_lazy;_> 	s->good_match       = configuration_table[level].good_length;> 	s->nice_match       = configuration_table[level].nice_length;< 	s->max_chain_length = configuration_table[level].max_chain;     }s     s->strategy = strategy;,     return Z_OK; };  L /* =========================================================================K  * Put a short in the pending buffer. The 16-bit value is put in MSB order.oH  * IN assertion: the stream state is correct and there is enough room in  * pending_buf.o  */a local void putShortMSB (s, b)r     deflate_state *s;      uInt b;c {i      put_byte(s, (Byte)(b >> 8));"     put_byte(s, (Byte)(b & 0xff)); }     L /* =========================================================================F  * Flush as much pending output as possible. All deflate() output goesC  * through this function so some applications may wish to modify it I  * to avoid allocating a large strm->next_out buffer and copying into it.   * (See also read_buf()).h  */> local void flush_pending(strm)     z_stream *strm;= {=(     unsigned len = strm->state->pending;  5     if (len > strm->avail_out) len = strm->avail_out;y     if (len == 0) return;p  ;     zmemcpy(strm->next_out, strm->state->pending_out, len);      strm->next_out  += len;e%     strm->state->pending_out  += len;*     strm->total_out += len;t     strm->avail_out  -= len;      strm->state->pending -= len;$     if (strm->state->pending == 0) {<         strm->state->pending_out = strm->state->pending_buf;     }u }i  O /* ========================================================================= */_ int deflate (strm, flush)A     z_stream *strm;p     int flush; {sG     int old_flush; /* value of flush param for previous deflate call */      deflate_state *s;=  G     if (strm == Z_NULL || strm->state == Z_NULL) return Z_STREAM_ERROR;a          s = strm->state;  #     if (strm->next_out == Z_NULL ||l;         (strm->next_in == Z_NULL && strm->avail_in != 0) || 4 	(s->status == FINISH_STATE && flush != Z_FINISH)) {)         ERR_RETURN(strm, Z_STREAM_ERROR);i     }-<     if (strm->avail_out == 0) ERR_RETURN(strm, Z_BUF_ERROR);  &     s->strm = strm; /* just in case */     old_flush = s->last_flush;     s->last_flush = flush;       /* Write the zlib header */r"     if (s->status == INIT_STATE) {  =         uInt header = (Z_DEFLATED + ((s->w_bits-8)<<4)) << 8;F-         uInt level_flags = (s->level-1) >> 1;s  -         if (level_flags > 3) level_flags = 3;i%         header |= (level_flags << 6);=- 	if (s->strstart != 0) header |= PRESET_DICT;=%         header += 31 - (header % 31);i           s->status = BUSY_STATE;,         putShortMSB(s, header);,  1 	/* Save the adler32 of the preset dictionary: */  	if (s->strstart != 0) {/ 	    putShortMSB(s, (uInt)(strm->adler >> 16));t2 	    putShortMSB(s, (uInt)(strm->adler & 0xffff)); 	    strm->adler = 1L; 	}     }n  2     /* Flush as much pending output as possible */     if (s->pending != 0) {         flush_pending(strm);.         if (strm->avail_out == 0) return Z_OK;  I     /* Make sure there is something to do and avoid duplicate consecutive E      * flushes. For repeated and useless calls with Z_FINISH, we keepe6      * returning Z_STREAM_END instead of Z_BUFF_ERROR.      */ ;     } else if (strm->avail_in == 0 && flush <= old_flush &&s 	       flush != Z_FINISH) {&         ERR_RETURN(strm, Z_BUF_ERROR);     }s  B     /* User must not provide more input after the first FINISH: */;     if (s->status == FINISH_STATE && strm->avail_in != 0) {e&         ERR_RETURN(strm, Z_BUF_ERROR);     }m  5     /* Start a new block or continue the current one.=      */ 3     if (strm->avail_in != 0 || s->lookahead != 0 ||v=         (flush != Z_NO_FLUSH && s->status != FINISH_STATE)) {          int quit;<            if (flush == Z_FINISH) {%             s->status = FINISH_STATE;g	         }A: 	quit = (*(configuration_table[s->level].func))(s, flush);  6         if (quit || strm->avail_out == 0) return Z_OK;B         /* If flush != Z_NO_FLUSH && avail_out == 0, the next callF          * of deflate should use the same flush parameter to make sureD          * that the flush is complete. So we don't have to output anF          * empty block here, this will be done at next call. This alsoG          * ensures that for a very small output buffer, we emit at most           * one empty block.>          */_7         if (flush != Z_NO_FLUSH && flush != Z_FINISH) {s+             if (flush == Z_PARTIAL_FLUSH) {                  _tr_align(s);_3             } else { /* FULL_FLUSH or SYNC_FLUSH */C5                 _tr_stored_block(s, (char*)0, 0L, 0);eH                 /* For a full flush, this empty block will be recognized9                  * as a special marker by inflate_sync().s                  */ ,                 if (flush == Z_FULL_FLUSH) {C                     CLEAR_HASH(s);             /* forget history */                  } 
             }_              flush_pending(strm);2             if (strm->avail_out == 0) return Z_OK;	         }u     }r(     Assert(strm->avail_out > 0, "bug2");  '     if (flush != Z_FINISH) return Z_OK;s)     if (s->noheader) return Z_STREAM_END;   *     /* Write the zlib trailer (adler32) */.     putShortMSB(s, (uInt)(strm->adler >> 16));1     putShortMSB(s, (uInt)(strm->adler & 0xffff));=     flush_pending(strm);D     /* If avail_out is zero, the application will call deflate again      * to flush the rest.s      */d8     s->noheader = -1; /* write the trailer only once! */1     return s->pending != 0 ? Z_OK : Z_STREAM_END;  }   O /* ========================================================================= */= int deflateEnd (strm)m     z_stream *strm;I {T     int status;E  G     if (strm == Z_NULL || strm->state == Z_NULL) return Z_STREAM_ERROR;d  5     /* Deallocate in reverse order of allocations: */e-     TRY_FREE(strm, strm->state->pending_buf);l&     TRY_FREE(strm, strm->state->head);&     TRY_FREE(strm, strm->state->prev);(     TRY_FREE(strm, strm->state->window);  !     status = strm->state->status;o     ZFREE(strm, strm->state);      strm->state = Z_NULL;e  6     return status == BUSY_STATE ? Z_DATA_ERROR : Z_OK; }h  O /* ========================================================================= */d int deflateCopy (dest, source)     z_stream *dest;A     z_stream *source;> {dH     if (source == Z_NULL || dest == Z_NULL || source->state == Z_NULL) {         return Z_STREAM_ERROR;     }Z     *dest = *source;2     return Z_STREAM_ERROR; /* to be implemented */ #if 0=/     dest->state = (struct internal_state FAR *)m2         (*dest->zalloc)(1, sizeof(deflate_state));2     if (dest->state == Z_NULL) return Z_MEM_ERROR;  &     *(dest->state) = *(source->state);     return Z_OK; #endif }=  N /* ===========================================================================F  * Read a new buffer from the current input stream, update the adler32D  * and total number of bytes read.  All deflate() input goes throughD  * this function so some applications may wish to modify it to avoid?  * allocating a large strm->next_in buffer and copying from it.;  * (See also flush_pending()).  */a# local int read_buf(strm, buf, size)t     z_stream *strm;-     charf *buf;O     unsigned size; {t"     unsigned len = strm->avail_in;       if (len > size) len = size;=     if (len == 0) return 0;=       strm->avail_in  -= len;n  !     if (!strm->state->noheader) {)?         strm->adler = adler32(strm->adler, strm->next_in, len);      }t%     zmemcpy(buf, strm->next_in, len);      strm->next_in  += len;     strm->total_in += len;       return (int)len; }   N /* ===========================================================================@  * Initialize the "longest match" routines for a new zlib stream  */O local void lm_init (s)     deflate_state *s;u {='     s->window_size = (ulg)2L*s->w_size;        CLEAR_HASH(s);  0     /* Set the default configuration parameters:      */ A     s->max_lazy_match   = configuration_table[s->level].max_lazy;TD     s->good_match       = configuration_table[s->level].good_length;D     s->nice_match       = configuration_table[s->level].nice_length;B     s->max_chain_length = configuration_table[s->level].max_chain;       s->strstart = 0;     s->block_start = 0L;     s->lookahead = 0;h3     s->match_length = s->prev_length = MIN_MATCH-1;      s->match_available = 0;      s->ins_h = 0;  #ifdef ASMV=/     match_init(); /* initialize the asm code */= #endif }=  N /* ===========================================================================H  * Set match_start to the longest match starting at the given string andL  * return its length. Matches shorter or equal to prev_length are discarded,F  * in which case the result is equal to prev_length and match_start is  * garbage.;I  * IN assertions: cur_match is the head of the hash chain for the current=L  *   string (strstart) and its distance is <= MAX_DIST, and prev_length >= 1D  * OUT assertion: the match length is not greater than s->lookahead.  */  #ifndef ASMVM /* For 80x86 and 680x0, an optimized version will be provided in match.asm or_5  * match.S. The code will be functionally equivalent.t  */*% local int longest_match(s, cur_match)a     deflate_state *s;(C     IPos cur_match;                             /* current match */u {pK     unsigned chain_length = s->max_chain_length;/* max hash chain length */eH     register Bytef *scan = s->window + s->strstart; /* current string */E     register Bytef *match;                       /* matched string */ M     register int len;                           /* length of current match */sN     int best_len = s->prev_length;              /* best match length so far */O     int nice_match = s->nice_match;             /* stop if match long enough */m2     IPos limit = s->strstart > (IPos)MAX_DIST(s) ?.         s->strstart - (IPos)MAX_DIST(s) : NIL;B     /* Stop when cur_match becomes <= limit. To simplify the code,<      * we prevent matches with the string of window index 0.      */f     Posf *prev = s->prev;|     uInt wmask = s->w_mask;    #ifdef UNALIGNED_OKnH     /* Compare two bytes at a time. Note: this is not always beneficial.4      * Try with and without -DUNALIGNED_OK to check.      */_E     register Bytef *strend = s->window + s->strstart + MAX_MATCH - 1; +     register ush scan_start = *(ushf*)scan; 8     register ush scan_end   = *(ushf*)(scan+best_len-1); #else A     register Bytef *strend = s->window + s->strstart + MAX_MATCH; 0     register Byte scan_end1  = scan[best_len-1];.     register Byte scan_end   = scan[best_len]; #endif  O     /* The code is optimized for HASH_BITS >= 8 and MAX_MATCH-2 multiple of 16.)?      * It is easy to get rid of this optimization if necessary.       */ E     Assert(s->hash_bits >= 8 && MAX_MATCH == 258, "Code too clever");t  E     /* Do not waste too much time if we already have a good match: */p*     if (s->prev_length >= s->good_match) {         chain_length >>= 2;r     }rM     /* Do not look for matches beyond the end of the input. This is necessaryg%      * to make deflate deterministic.i      */{=     if (nice_match > s->lookahead) nice_match = s->lookahead;0  O     Assert((ulg)s->strstart <= s->window_size-MIN_LOOKAHEAD, "need lookahead");s       do {5         Assert(cur_match < s->strstart, "no future");I&         match = s->window + cur_match;  A         /* Skip to next match if the match length cannot increase 1          * or if the match length is less than 2:F          */ / #if (defined(UNALIGNED_OK) && MAX_MATCH == 258) D         /* This code assumes sizeof(unsigned short) == 2. Do not use?          * UNALIGNED_OK if your compiler uses a different size.R          */,5         if (*(ushf*)(match+best_len-1) != scan_end ||o2             *(ushf*)match != scan_start) continue;  M         /* It is not necessary to compare scan[2] and match[2] since they are=L          * always equal when the other bytes match, given that the hash keysJ          * are equal and that HASH_BITS >= 8. Compare 2 bytes at a time atL          * strstart+3, +5, ... up to strstart+257. We check for insufficientL          * lookahead only every 4th comparison; the 128th check will be madeH          * at strstart+257. If MAX_MATCH-2 is not a multiple of 8, it isI          * necessary to put more guard bytes at the end of the window, or :          * to check more often for insufficient lookahead.          */ 0         Assert(scan[2] == match[2], "scan[2]?");         scan++, match++;         do {;         } while (*(ushf*)(scan+=2) == *(ushf*)(match+=2) &&h;                  *(ushf*)(scan+=2) == *(ushf*)(match+=2) && ;                  *(ushf*)(scan+=2) == *(ushf*)(match+=2) &&L;                  *(ushf*)(scan+=2) == *(ushf*)(match+=2) &&                    scan < strend);G         /* The funny "do {}" generates better code on most compilers */   /         /* Here, scan <= window+strstart+257 */ L         Assert(scan <= s->window+(unsigned)(s->window_size-1), "wild scan");$         if (*scan == *match) scan++;  3         len = (MAX_MATCH - 1) - (int)(strend-scan); &         scan = strend - (MAX_MATCH-1);   #else /* UNALIGNED_OK */  -         if (match[best_len]   != scan_end  ||--             match[best_len-1] != scan_end1 ||F-             *match            != *scan     ||r8             *++match          != scan[1])      continue;  I         /* The check at best_len-1 can be removed because it will be madet=          * again later. (This heuristic is not always a win.) I          * It is not necessary to compare scan[2] and match[2] since they B          * are always equal when the other bytes match, given that;          * the hash keys are equal and that HASH_BITS >= 8.           */=         scan += 2, match++;=-         Assert(*scan == *match, "match[2]?");n  I         /* We check for insufficient lookahead only every 8th comparison;t8          * the 256th check will be made at strstart+258.          */*         do {>         } while (*++scan == *++match && *++scan == *++match &&>                  *++scan == *++match && *++scan == *++match &&>                  *++scan == *++match && *++scan == *++match &&>                  *++scan == *++match && *++scan == *++match &&                   scan < strend);  L         Assert(scan <= s->window+(unsigned)(s->window_size-1), "wild scan");  /         len = MAX_MATCH - (int)(strend - scan);="         scan = strend - MAX_MATCH;   #endif /* UNALIGNED_OK */_           if (len > best_len) {r'             s->match_start = cur_match;t             best_len = len;=)             if (len >= nice_match) break;  #ifdef UNALIGNED_OKs1             scan_end = *(ushf*)(scan+best_len-1);n #else *             scan_end1  = scan[best_len-1];(             scan_end   = scan[best_len]; #endif	         }):     } while ((cur_match = prev[cur_match & wmask]) > limit%              && --chain_length != 0);   2     if (best_len <= s->lookahead) return best_len;     return s->lookahead; }= #endif /* ASMV */=   #ifdef DEBUGN /* ===========================================================================9  * Check that the match at match_start is indeed a match.   */i/ local void check_match(s, start, match, length)a     deflate_state *s;a     IPos start, match;     int length;  {;0     /* check that the match is indeed a match */+     if (zmemcmp((charf *)s->window + match,m?                 (charf *)s->window + start, length) != EQUAL) {s;         fprintf(stderr, " start %u, match %u, length %d\n",e 		start, match, length);         do {E 	    fprintf(stderr, "%c%c", s->window[match++], s->window[start++]);  	} while (--length != 0);n!         z_error("invalid match");(     }r     if (verbose > 1) {9         fprintf(stderr,"\\[%d,%d]", start-match, length);eG         do { putc(s->window[start++], stderr); } while (--length != 0);=     }= }= #else=. #  define check_match(s, start, match, length) #endif  N /* ===========================================================================;  * Fill the window when the lookahead becomes insufficient. "  * Updates strstart and lookahead.  **  * IN assertion: lookahead < MIN_LOOKAHEAD8  * OUT assertions: strstart <= window_size-MIN_LOOKAHEADB  *    At least one byte has been read, or avail_in == 0; reads areJ  *    performed for at least two bytes (required for the zip translate_eol$  *    option -- not supported here).  */i local void fill_window(s)      deflate_state *s;; {      register unsigned n, m;m     register Posf *p;eJ     unsigned more;    /* Amount of free space at the end of the window. */     uInt wsize = s->w_size;*       do {O         more = (unsigned)(s->window_size -(ulg)s->lookahead -(ulg)s->strstart);=  (         /* Deal with !@#$% 64K limit: */A         if (more == 0 && s->strstart == 0 && s->lookahead == 0) {r             more = wsize;e  ,         } else if (more == (unsigned)(-1)) {M             /* Very unlikely, but possible on 16 bit machine if strstart == 0 ?              * and lookahead == 1 (input done one byte at time)               */r             more--;s  L         /* If the window is almost full and there is insufficient lookahead,O          * move the upper half to the lower one to make room in the upper half.           */v6         } else if (s->strstart >= wsize+MAX_DIST(s)) {  A             zmemcpy((charf *)s->window, (charf *)s->window+wsize,c$                    (unsigned)wsize);$             s->match_start -= wsize;K             s->strstart    -= wsize; /* we now have strstart >= MAX_DIST */a  +             s->block_start -= (long) wsize;i  H             /* Slide the hash table (could be avoided with 32 bit values/                at the expense of memory usage):               */e             n = s->hash_size;              p = &s->head[n];             do {                 m = *--p;;7                 *p = (Pos)(m >= wsize ? m-wsize : NIL);e             } while (--n);               n = wsize;             p = &s->prev[n];             do {                 m = *--p; 7                 *p = (Pos)(m >= wsize ? m-wsize : NIL);cH                 /* If n is not on any hash chain, prev[n] is garbage but0                  * its value will never be used.                  */v             } while (--n);               more += wsize;	         }a+         if (s->strm->avail_in == 0) return;a  #         /* If there was no sliding:hO          *    strstart <= WSIZE+MAX_DIST-1 && lookahead <= MIN_LOOKAHEAD - 1 &&s8          *    more == window_size - lookahead - strstartJ          * => more >= window_size - (MIN_LOOKAHEAD-1 + WSIZE + MAX_DIST-1)/          * => more >= window_size - 2*WSIZE + 2s;          * In the BIG_MEM or MMAP case (not yet supported),[:          *   window_size == input_size + MIN_LOOKAHEAD  &&L          *   strstart + s->lookahead <= input_size => more >= MIN_LOOKAHEAD.:          * Otherwise, window_size == 2*WSIZE so more >= 2.K          * If there was sliding, more >= WSIZE. So in all cases, more >= 2.2          */ &         Assert(more >= 2, "more < 2");  N         n = read_buf(s->strm, (charf *)s->window + s->strstart + s->lookahead,                      more);>         s->lookahead += n;  D         /* Initialize the hash value now that we have some input: */(         if (s->lookahead >= MIN_MATCH) {.             s->ins_h = s->window[s->strstart];?             UPDATE_HASH(s, s->ins_h, s->window[s->strstart+1]);z #if MIN_MATCH != 35             Call UPDATE_HASH() MIN_MATCH-3 more times< #endif	         }uN         /* If the whole input has less than MIN_MATCH bytes, ins_h is garbage,N          * but this is not important since only literal bytes will be emitted.          */   E     } while (s->lookahead < MIN_LOOKAHEAD && s->strm->avail_in != 0);T }   N /* ===========================================================================8  * Flush the current block, with given end-of-file flag.A  * IN assertion: strstart is set to the end of the current match.f  */c$ #define FLUSH_BLOCK_ONLY(s, eof) { \/    _tr_flush_block(s, (s->block_start >= 0L ? \cD                    (charf *)&s->window[(unsigned)s->block_start] : \&                    (charf *)Z_NULL), \. 		(ulg)((long)s->strstart - s->block_start), \ 		(eof)); \ "    s->block_start = s->strstart; \    flush_pending(s->strm); \     Tracev((stderr,"[FLUSH]")); \ }o  1 /* Same but force premature exit if necessary. */e #define FLUSH_BLOCK(s, eof) { \M    FLUSH_BLOCK_ONLY(s, eof); \+    if (s->strm->avail_out == 0) return 1; \a }b  N /* ===========================================================================M  * Copy without compression as much as possible from the input stream, return I  * true if processing was terminated prematurely (no more input or output G  * space).  This function does not insert new strings in the dictionarytJ  * since uncompressible data is probably not useful. This function is used+  * only for the level=0 compression option.mB  * NOTE: this function should be optimized to avoid extra copying.  */{" local int deflate_stored(s, flush)     deflate_state *s;*     int flush; {o     for (;;) {2         /* Fill the window as much as possible: */          if (s->lookahead <= 1) {  9             Assert(s->strstart < s->w_size+MAX_DIST(s) ||-: 		   s->block_start >= (long)s->w_size, "slide too late");               fill_window(s); C             if (s->lookahead == 0 && flush == Z_NO_FLUSH) return 1;c  G             if (s->lookahead == 0) break; /* flush the current block */|	         } , 	Assert(s->block_start >= 0L, "block gone");   	s->strstart += s->lookahead;t 	s->lookahead = 0;  8         /* Stored blocks are limited to 0xffff bytes: */7         if (s->strstart == 0 || s->strstart > 0xffff) {sF 	    /* strstart == 0 is possible when wraparound on 16-bit machine */) 	    s->lookahead = s->strstart - 0xffff;t 	    s->strstart = 0xffff; 	}  1 	/* Emit a stored block if it is large enough: */ @         if (s->strstart - (uInt)s->block_start >= MAX_DIST(s)) {             FLUSH_BLOCK(s, 0); 	}     }l&     FLUSH_BLOCK(s, flush == Z_FINISH);     return 0; /* normal exit */m }   N /* ===========================================================================E  * Compress as much as possible from the input stream, return true ifsI  * processing was terminated prematurely (no more input or output space).aH  * This function does not perform lazy evaluation of matches and insertsH  * new strings in the dictionary only for unmatched strings or for short=  * matches. It is used only for the fast compression options.   */t  local int deflate_fast(s, flush)     deflate_state *s;      int flush; {_6     IPos hash_head = NIL; /* head of the hash chain */D     int bflush;           /* set if current block must be flushed */       for (;;) {A         /* Make sure that we always have enough lookahead, except(@          * at the end of the input file. We need MAX_MATCH bytesA          * for the next match, plus MIN_MATCH bytes to insert the +          * string following the next match.a          */ +         if (s->lookahead < MIN_LOOKAHEAD) {              fill_window(s); N             if (s->lookahead < MIN_LOOKAHEAD && flush == Z_NO_FLUSH) return 1;  G             if (s->lookahead == 0) break; /* flush the current block */=	         }C  B         /* Insert the string window[strstart .. strstart+2] in theG          * dictionary, and set hash_head to the head of the hash chain:s          */;(         if (s->lookahead >= MIN_MATCH) {5             INSERT_STRING(s, s->strstart, hash_head);*	         }+  C         /* Find the longest match, discarding those <= prev_length. @          * At this point we have always match_length < MIN_MATCH          */hI         if (hash_head != NIL && s->strstart - hash_head <= MAX_DIST(s)) {sG             /* To simplify the code, we prevent matches with the stringdH              * of window index 0 (in particular we have to avoid a matchI              * of the string with itself at the start of the input file).;              */h0             if (s->strategy != Z_HUFFMAN_ONLY) {?                 s->match_length = longest_match (s, hash_head);=
             }=2             /* longest_match() sets match_start */	         } +         if (s->match_length >= MIN_MATCH) {sI             check_match(s, s->strstart, s->match_start, s->match_length);O  ?             bflush = _tr_tally(s, s->strstart - s->match_start,t<                                s->match_length - MIN_MATCH);  ,             s->lookahead -= s->match_length;  L             /* Insert new strings in the hash table only if the match lengthJ              * is not too large. This saves time but degrades compression.              */ :             if (s->match_length <= s->max_insert_length &&,                 s->lookahead >= MIN_MATCH) {Q                 s->match_length--; /* string at strstart already in hash table */r                 do {"                     s->strstart++;=                     INSERT_STRING(s, s->strstart, hash_head); K                     /* strstart never exceeds WSIZE-MAX_MATCH, so there are 4                      * always MIN_MATCH bytes ahead.                      */ 1                 } while (--s->match_length != 0);e                 s->strstart++;               } else {/                 s->strstart += s->match_length;s$                 s->match_length = 0;2                 s->ins_h = s->window[s->strstart];C                 UPDATE_HASH(s, s->ins_h, s->window[s->strstart+1]);z #if MIN_MATCH != 39                 Call UPDATE_HASH() MIN_MATCH-3 more timesw #endifN                 /* If lookahead < MIN_MATCH, ins_h is garbage, but it does notK                  * matter since it will be recomputed at next deflate call.                   */a
             }i         } else {1             /* No match, output a literal byte */ ;             Tracevv((stderr,"%c", s->window[s->strstart])); >             bflush = _tr_tally (s, 0, s->window[s->strstart]);             s->lookahead--;o             s->strstart++; ;	         } &         if (bflush) FLUSH_BLOCK(s, 0);     } &     FLUSH_BLOCK(s, flush == Z_FINISH);     return 0; /* normal exit */> }v  N /* ===========================================================================@  * Same as above, but achieves better compression. We use a lazyF  * evaluation for matches: a match is finally adopted only if there is/  * no better match at the next window position.   */   local int deflate_slow(s, flush)     deflate_state *s;      int flush; {i5     IPos hash_head = NIL;    /* head of hash chain */hG     int bflush;              /* set if current block must be flushed */D  "     /* Process the input block. */     for (;;) {A         /* Make sure that we always have enough lookahead, except+@          * at the end of the input file. We need MAX_MATCH bytesA          * for the next match, plus MIN_MATCH bytes to insert the +          * string following the next match.L          */ +         if (s->lookahead < MIN_LOOKAHEAD) {_             fill_window(s);.N             if (s->lookahead < MIN_LOOKAHEAD && flush == Z_NO_FLUSH) return 1;  G             if (s->lookahead == 0) break; /* flush the current block */ 	         }t  B         /* Insert the string window[strstart .. strstart+2] in theG          * dictionary, and set hash_head to the head of the hash chain:           */d(         if (s->lookahead >= MIN_MATCH) {5             INSERT_STRING(s, s->strstart, hash_head);o	         }I  C         /* Find the longest match, discarding those <= prev_length.           */,I         s->prev_length = s->match_length, s->prev_match = s->match_start;l&         s->match_length = MIN_MATCH-1;  E         if (hash_head != NIL && s->prev_length < s->max_lazy_match && 5             s->strstart - hash_head <= MAX_DIST(s)) {pG             /* To simplify the code, we prevent matches with the stringhH              * of window index 0 (in particular we have to avoid a matchI              * of the string with itself at the start of the input file).               */e0             if (s->strategy != Z_HUFFMAN_ONLY) {?                 s->match_length = longest_match (s, hash_head);d
             }C2             /* longest_match() sets match_start */  E             if (s->match_length <= 5 && (s->strategy == Z_FILTERED ||c1                  (s->match_length == MIN_MATCH &&\=                   s->strstart - s->match_start > TOO_FAR))) {   J                 /* If prev_match is also MIN_MATCH, match_start is garbage?                  * but we will ignore the current match anyway.e                  */ .                 s->match_length = MIN_MATCH-1;
             } 	         }vD         /* If there was a match at the previous step and the current:          * match is not better, output the previous match:          */sO         if (s->prev_length >= MIN_MATCH && s->match_length <= s->prev_length) {nE             uInt max_insert = s->strstart + s->lookahead - MIN_MATCH;oB             /* Do not insert strings in hash table beyond this. */  I             check_match(s, s->strstart-1, s->prev_match, s->prev_length);   A             bflush = _tr_tally(s, s->strstart -1 - s->prev_match,v;                                s->prev_length - MIN_MATCH);   K             /* Insert in hash table all strings up to the end of the match. L              * strstart-1 and strstart are already inserted. If there is notI              * enough lookahead, the last two strings are not inserted ino              * the hash table.              */i-             s->lookahead -= s->prev_length-1;               s->prev_length -= 2;             do {2                 if (++s->strstart <= max_insert) {=                     INSERT_STRING(s, s->strstart, hash_head);"                 }+,             } while (--s->prev_length != 0);#             s->match_available = 0;f*             s->match_length = MIN_MATCH-1;             s->strstart++;  *             if (bflush) FLUSH_BLOCK(s, 0);  (         } else if (s->match_available) {G             /* If there was no match at the previous position, output aoI              * single literal. If there was a match but the current matchoJ              * is longer, truncate the previous match to a single literal.              */u=             Tracevv((stderr,"%c", s->window[s->strstart-1]));==             if (_tr_tally (s, 0, s->window[s->strstart-1])) {='                 FLUSH_BLOCK_ONLY(s, 0); 
             }s             s->strstart++;             s->lookahead--;a2             if (s->strm->avail_out == 0) return 1;         } else {C             /* There is no previous match to compare with, wait fori'              * the next step to decide.r              */s#             s->match_available = 1;s             s->strstart++;             s->lookahead--; 	         }t     } .     Assert (flush != Z_NO_FLUSH, "no flush?");     if (s->match_available) { 9         Tracevv((stderr,"%c", s->window[s->strstart-1]));s3         _tr_tally (s, 0, s->window[s->strstart-1]);e         s->match_available = 0;c     } &     FLUSH_BLOCK(s, flush == Z_FINISH);
     return 0;H }t  