/*
 * Handle reading and parsing of rule file.
 *
 *    int http_parse_elements ( int limit, char *line, string *elements );
 *    int http_read_rules ( char *rule_file );
 *    int http_multihomed_hostname ( unsigned long address, access_info acc );
 *    int http_multihome_scan ( int ndx, char **name, unsigned char **addr );
 *
 * Revised:  6-APR-1994			 Added redirect and exec support.
 * Revised: 26-MAY-1994			Added protect rule. and AddType syn.
 * Revised: 27-MAY-1994			Make rule names case insensitive
 * Revised: 2-JUN-1994			Log stats of rule file.
 * Revised: 7-JUN-1994			Added validity checks to read_rule_file
 * Revised: 27-JUN-1994
 * Revised:  2-AUG-1994			Support decnet_searchlist module.
 * Revised: 24-AUG-1994			Define taskname for presentation rules.
 * Revised:  4-NOV-1994			Initialize index file search list.
 * Revised:  5-NOV-1994			Support welcome rule.
 * Revised: 16-NOV-1994			Support DirAccess rule.
 * Revised:  6-JAN-1994			Support scriptname for search.
 * Revised: 12-JAN-1995			Added George Carrette's mods for
 *					hostname, localhost (gjc@village.com).
 * Revised: 25-JAN-1995			Fixed bug in tracelevel rule.
 * Revised: 24-FEB-1995			Support port configuration rule.
 * Revised: 26-FEB-1995			Support usermap directive.
 * Revised: 24-MAR-1995			Added mapimage rule.
 * Revised: 21-APR-1995			Added counter rule.
 * Revised: 11-MAY-1995			Hack for OSF version.
 * Revised: 15-MAY-1995			Conditionally compile counters.
 *					(skip if SCRIPTSERVER defined).
 * Revised:  3-JUN-1995			Add threadpool and service rules
 * Revised: 12-JUN-1995			Support continuation lines.
 * Revised:  15-JUL-1995		Add multihomed hacks.
 * Revised:  24-NOV-1995		Add scan routine.
 * Revised:  19-FEB-1996		Add Timelimit rule.
 * Revised:   1-MAR-1996		whitespace must precede comment delim.
 * Revised:  18-JUN-1996		Support 'pre-processor' (.define,
 *					.expand, .ignore, .iterate, .next ).
 * Revised:  24-JUL-1996		Support MANAGE directive.
 * Revised:  11-AUG-1996		Support METHOD directive.
 * Revised:  16-AUG-1996		Remove default PUT method, give each
 *					include file its own namespace for
 *					pre-processor directives.
 * Revised:  24-aug-1996		Flag when localaddress argument begins
 *					with '@'.
 */
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include "ctype_np.h"
#include "access.h"
#include "ident_map.h"
#include "decnet_searchlist.h"
#include "tutil.h"
#ifndef SCRIPTSERVER
#include "tserver_tcp.h"
#include "counters.h"
#include "script_manage.h"
#endif
/*
 * Work around problems with DECC and forbidden (non-ANSI) names.
 */
#ifdef __DECC
#ifndef fgetname
#define fgetname decc$fgetname
char *fgetname ( FILE *f, char *fname );
#endif
#else
#ifdef VMS
char *fgetname();
#else
#define fgetname(a,b) tu_strcpy(b,"rule_file")
#endif
#endif

#ifdef VMS
#define VMS_FOPEN_OPT , "mbc=64", "dna=.conf"
#else
#define VMS_FOPEN_OPT
#ifndef vaxc$errno
#define vaxc$errno 20
#endif
#endif

typedef struct { int l; char *s; } string;
char *http_search_script;		/* GLobal variable, search enable. */
char **http_index_filename;		/* Filename for index files */
char *http_scheme_fixup;		/* hack for https: redirects */
int http_dir_access;
int http_ports[2];			/* TCP/IP port numbers */
int http_request_time_limit;		/* timeout for reading request */
int http_response_time_limit;		/* timeout for sending data */
int http_keepalive_limit;		/* Max keepalives allowed or 0 */
int http_keepalive_time;		/* time limit for keepalive */
char *http_dir_access_file;		/* Directory access check file */
char *http_authenticator_image;		/* Global variable, level 2 access */
char *http_default_host;		/* server name to advertise */
int http_multihomed;			/* true if multihomed */
int http_dns_enable;			/* flag to control host name lookups */
int http_manage_port;			/* Manage port number */
char *http_manage_host;			/* Management host */
int http_log_level, tlog_putlog();	/* Global variable, logger detail level */

int http_define_suffix(), http_define_presentation(), http_define_ident();
int http_load_dynamic_service();
int http_define_method();

static int index_file_alloc=0, index_file_count;
static int multihome_count=0;
static struct host_def {
    struct host_def *next;
    char *name;			/* host name to use */
    union {
        unsigned long number;
	unsigned char octet[4];
    } a;
    int accesslog;
} *multihome_list, *multihome_last;
static char *default_index_file[] = { "index.html", "index.htmlx", (char *) 0 };
static int preprocess_rule ( int count, int limit, string *elem );
/***********************************************************************/
/* Parse line into whitespace-delimited tokens, trimming leading and trailing
 * whitespace.  Each token parse is explicitly null-terminataed.  Function 
 * value returned is number of elements found.  May be 1 more than limit if 
 * last element does not end line.
 */
int http_parse_elements (
	int limit,	    /* Max number of element to delimit */
	char *line,	    /* input line to parse */
	string *elem )	    /* Output array. */
{
    int tcnt, in_token, length, i;
    char *ptr;
    /* Delimit up to three tokens */
    for ( in_token = tcnt = length = 0, ptr = line; *ptr; ptr++ ) {
	if ( in_token ) {
	    if ( isspace ( *ptr ) ) {
		/* End current token */
		*ptr = '\0';  /* terminate string */
		elem[tcnt++].l = length;
		in_token = 0;
	    }
	    else length++;
	} else {
	    if ( !isspace ( *ptr ) ) {
		/* start next token */
		if ( tcnt >= limit ) {
		    /* more tokens than expected */
		    tcnt++;
		    break;
		}
		elem[tcnt].s = ptr;
		in_token = 1;
		length = 1;
	    }
        }
    }
    /*
     * Make final adjust to element count and make remaining elment null.
     */
    if ( in_token ) { elem[tcnt++].l = length; }
    for ( i = tcnt; i < limit; i++ ) { elem[i].l = 0; elem[i].s = ptr; }
    return tcnt;
}
/***********************************************************************/
/* Read next line from nested stack of input file.  Trim line of comments
 * and final linefeed and concatentate continuation lines if
 * last character is a backslash (\).
 */
static char *read_nested 
	( char *line, int linesize, FILE **fp, int *sp, int *lines )
{
    char *result;
    int filled, trim;
    for ( *lines = filled = 0; filled < (linesize-1); ) {
	result = fgets ( &line[filled], linesize-filled, fp[*sp] );
	if ( !result ) {
	    /*
	     * End-of-file.
	     */
	    if ( filled > 0 ) break;		/* return what we got. */
	    if ( *sp == 0 ) return result;	/* final EOF */
	    /*
	     * Resume with previous file, restore pre-processor name space.
	     */
	    *lines = 0;
	    fclose ( fp[*sp] );
	    *sp = *sp - 1;
	    preprocess_rule ( -1, 0, (string *) 0);
	} else {
	    *lines = *lines + 1;
	    /*
	     * Trim comments and check for continuation line.
	     */
	    for (trim=(-1); line[filled] && (line[filled]!='\n'); filled++) {
		if ( (line[filled] == '#') && (trim < 0) ) {
		    if ( filled <= 0 ) trim = filled;
		    else if ( isspace(line[filled-1]) ) trim = filled;
		}
	    }
	    if ( filled > 0 ) if ( line[filled-1] == '\\' )  {
		filled = ( trim < 0 ) ? (filled-1) : trim;
		for ( ; filled > 0; --filled ) 
			if ( !isspace(line[filled-1]) ) break;
		continue;
	    }
	    if ( trim >= 0 ) filled = trim;
	    break;
	}
    }
    line[filled] = '\0';
    return line;
}
/***********************************************************************/
/* Convert argument of timelimit command into seconds.  Argument may be:
 *    1. 'nn', decimal number of seconds
 *    2. '[hh:]mm:ss', hours, minutes, seconds
 *    3. 'symbol', getenv symbols, value is retrieve and must be form 1 or 2.
 */
static int parse_timelimit ( char *arg, char *temp ) 
{
    int secs, i, j;
    if ( (*arg < '0') || (*arg  > '9') ) {
	/* Assume argument isenvironment variable. */
	char *val = getenv(arg);
	if ( !val ) tlog_putlog(0,
		"Invalid variable name in TimeLimit rule: '!AZ'!/", arg );
	arg = val ? arg : "0";
    }
    /*
     * Scan for colons, using to delimit a radix 60 number.
     */
    secs = 0;
    for ( j = i = 0; i < 11 && arg[i]; i++ ) {
	temp[j] = arg[i];
	if ( arg[i] == ':' ) {
	    temp[j+1] = '\0';
	    secs = (secs*60) + atoi(temp);
	    j = 0;
        } else if ( (arg[i] < '0') || (arg[i]  > '9') ) {
	    tlog_putlog ( 0, 
	      	"Invalid time value in TimeLimit rule: '!AZ'!/", arg );
	    return 0;
	} else j++;
    }
    temp[j] = '\0';
    secs = (secs*60) + atoi(temp);
    return secs;
}
/***********************************************************************/
/*  Append filename to list of index files (welcome pages) to search for
 *  when doing directory listings.
 */
static void add_index_file ( char *fname )
{
    /*  Allocate enough for all names plus a null */
    if ( index_file_count+2 >= index_file_alloc ) {
	index_file_alloc += 20;
	if ( index_file_alloc == 20 ) 		/* first call */
	    http_index_filename = (char **) malloc ( sizeof(char *)*index_file_alloc );
	else
	    http_index_filename = (char **) realloc ( http_index_filename, 
		sizeof(char *)*index_file_alloc );

	if ( !http_index_filename ) {
	   tlog_putlog ( 0, "error allocating index file list!/");
	   index_file_alloc = 0;
	   http_index_filename = default_index_file;
	   return;
	}
    }
    /*
     * Copy string and append to list.  Mark end of list with null.
     */
    http_index_filename[index_file_count] = malloc (tu_strlen(fname)+1);
    tu_strcpy ( http_index_filename[index_file_count], fname );
    index_file_count++;
    http_index_filename[index_file_count] = (char *) 0;
}
/***********************************************************************/
/*
 * Initialize rules and load rules database from file.
 */
int http_read_rules ( char *rule_file )
{
    FILE *rf[20];
    string token[8];
    char line[512], tline[512], temp[300], fname[256];
    int length, count, colon, id_count, sfx_count, lnum[20], pass_count, sp;
    int lcnt, acc_log_count, acc_local_log;
    /*
     * Initialize global variables.
     */
    http_search_script = NULL;
    http_authenticator_image = NULL;
    http_scheme_fixup = NULL;
    http_dns_enable = 0;
    http_index_filename = default_index_file;
    http_dir_access = 0;		/* Unrestricted access */
    http_dir_access_file = ".www_browsable";
    http_request_time_limit = 0;
    http_response_time_limit = 0;
    http_keepalive_limit = 0;
    http_keepalive_time = 10;
    acc_log_count = 1;			/* base for log index. */
    acc_local_log = 0;			/* true if inside localaddress block*/
    http_manage_port = 0;		/* No management host */
    http_manage_host = (char *) 0;	/* Default management host  */
    /*
     * See if rule file specified.
     */
    if ( !(*rule_file) ) {
	tlog_putlog ( 0, "No rule file specified, abort.!/" );
	return 20;
    } else {
	if ( http_log_level > 1 ) tlog_putlog ( 0,
		"Loading configuration/rules file !AZ!/", rule_file );
	id_count = sfx_count = pass_count = 0;
	sp = lnum[0] = 0;
	rf[0] = fopen ( rule_file, "r" VMS_FOPEN_OPT );
	if ( !rf[0] ) {
	    tlog_putlog ( 0, "Error opening rule file '!AZ'!/", rule_file );
	    return  (vaxc$errno);
	} else while (read_nested ( line, sizeof(line)-1, rf, &sp, &lcnt )) {
	    /*
	     * Trim line of comments and line-feed and parse into tokens.
	     */
	    lnum[sp]+=lcnt;			/* track position in file */
	    for ( length = 0; line[length] /* && (line[length] != '#') &&
			(line[length] != '\n') */; 
			length++ ) tline[length] = line[length];
	    tline[length] = line[length] = '\0';
	    if ( length > 0 ) count = http_parse_elements ( 7, tline, token );
	    else count = 0;
	    /*
	     * Process record based upon first token.
	     */
	    if ( count > 0 ) {
		tu_strupcase ( token[0].s, token[0].s );
		if ( http_log_level > 14 ) tlog_putlog ( 15,
			"Processing rule '!AZ', tokens: !SL!/", token[0].s,
			count );
	        while ( (count > 0) && (token[0].s[0] == '.') ) {
		    /* Rescan line */
		    count = preprocess_rule ( count, 7, token );
		    if ( count < 0 ) tlog_putlog (0,
			"Error in line !SL of rule file !AZ: !AZ!/",
			lnum[sp], sp ? fgetname(rf[sp],fname) : rule_file,
			line );
		}
	    }
	    if ( count <= 0 ) {
		/* Ignore null line */

	    } else if ( (tu_strncmp ( token[0].s, "SUFFIX", 7 ) == 0) ||
			(tu_strncmp ( token[0].s, "ADDTYPE", 8 ) == 0) ) {
		/*
		 * Define attributes of suffix, syntax:
		 *    suffix <sfx> <representation> encoding [qual]
		 */
		if ( count < 3 ) {
		    tlog_putlog ( 0, "Insufficient arguments in suffix command: !AZ!/",
			line );
		}
		http_define_suffix ( token[1].s, token[2].s, token[3].s,
			token[4].s ); sfx_count++;

	    } else if ( tu_strncmp ( token[0].s, "MAP", 7 ) == 0 ) {
		http_define_ident ( token[1].s, 1, token[2].s ); id_count++;

	    } else if ( tu_strncmp ( token[0].s, "PASS", 7 ) == 0 ) {
		http_define_ident ( token[1].s, 2, token[2].s ); 
		id_count++; pass_count++;

	    } else if ( tu_strncmp ( token[0].s, "FAIL", 7 ) == 0 ) {
		http_define_ident ( token[1].s, 3, token[2].s ); id_count++;

	   } else if ( tu_strncmp ( token[0].s, "REDIRECT", 9 ) == 0 ) {
		http_define_ident ( token[1].s, 4, token[2].s ); id_count++;

	    } else if ( tu_strncmp ( token[0].s, "HTBIN", 7 ) == 0 ) {
		/* Make exec entry for compatibility. */
		http_define_ident ( "/htbin/*", 5, token[1].s ); 
		dnetx_define_task ( token[1].s );
		id_count++; pass_count++;

	    } else if ( tu_strncmp ( token[0].s, "EXEC", 7 ) == 0 ) {
		http_define_ident ( token[1].s, 5, token[2].s ); 
		if ( token[2].s[0] != '%' ) dnetx_define_task ( token[2].s );
		id_count++; pass_count++;

	    } else if ( tu_strncmp( token[0].s, "MAPIMAGE", 9) == 0 ) {
#ifndef SCRIPTSERVER
		int length;
		/*
		 * Save argument for fake 'exec' call and parse new line
		 * for creating dynamic service.
		 */
		tu_strcpy ( temp, "%mapimage:" );
		tu_strnzcpy ( &temp[10], token[1].s, sizeof(temp)-11 );
		tu_strcpy ( tline, "SERVICE mapimage builtin=mapimage" );
		count = http_parse_elements ( 7, tline, token );
		if ( 0 == http_load_dynamic_service ( token, count, temp ) ) {
		    tlog_putlog (0, 
			"!AZ, line !SL of rule file !AZ!/", temp,
			lnum[sp], sp ? fgetname(rf[sp],fname) : rule_file );
		}
		else {
		    /* Success */
		    http_define_ident ( "/$mapimage/*", 5, temp );
		}
#endif
	    } else if ( (tu_strncmp ( token[0].s, "DYNAMAP", 8 ) == 0 ) ||
			(tu_strncmp ( token[0].s, "USERMAP", 8 ) == 0 ) ) {
		/* Dynamically defined usermap */
#ifdef SCRIPTSERVER
		int length;
		temp[0] = '(';
		tu_strnzcpy ( &temp[1], token[3].s, 256 );
		length = tu_strlen ( temp );
		temp[length++] = ')';
		tu_strnzcpy ( &temp[length], token[2].s, 299-length );
		http_define_ident ( token[1].s, IDENT_MAP_RULE_USERMAP, temp ); 
		id_count++; pass_count++;
#endif
	    } else if ( tu_strncmp ( token[0].s, "USERDIR", 8) == 0 ) {
		/* Set template to match /~ */
		http_define_ident ( "/~*", 6, token[1].s );
		id_count++; pass_count++;

	    } else if ( tu_strncmp ( token[0].s, "PROTECT", 8) == 0 ) {
		/* level 2 protection check record.  Prepend '+' to setup */
		char l2_setup[512];
		l2_setup[0] = '+';
		tu_strnzcpy ( &l2_setup[1], token[2].s, 500 );
		http_define_ident ( token[1].s, 7, l2_setup ); id_count++;

	    } else if ( tu_strncmp ( token[0].s, "DEFPROT", 8) == 0 ) {
		/* protection check record */
		http_define_ident ( token[1].s, 8, token[2].s ); id_count++;

	    } else if ( tu_strncmp ( token[0].s, "HOSTPROT", 9) == 0 ) {
		/* protection check record */
		http_define_ident ( token[1].s, 7, token[2].s ); id_count++;

	    } else if ( tu_strncmp ( token[0].s, "SEARCH", 7 ) == 0 ) {
		http_search_script = malloc ( token[1].l+1 );
		tu_strnzcpy ( http_search_script, token[1].s, token[1].l );
		dnetx_define_task ( http_search_script );

	    } else if ( tu_strncmp ( token[0].s, "PRESENTATION", 12) == 0 ) {
		http_define_presentation ( token[1].s, token[2].s ); id_count++;
		dnetx_define_task ( token[2].s );

	    } else if ( tu_strncmp ( token[0].s, "METHOD", 7) == 0 ) {
		if ( (tu_strncmp ( token[1].s, "GET", 4 ) == 0) ||
		    (tu_strncmp ( token[1].s, "HEAD", 5 ) == 0) ) {
		    tlog_putlog(0,"Error, !AZ method cannot be re-defined!/",
				token[1].s );
		} else {
		    int status;
		    status = http_define_method ( token[1].s, token[2].s );
		    if ( status == 3 ) tlog_putlog ( 0, 
			"Previous definition for !AZ method replaced!/", 
			token[1].s);
		    dnetx_define_task ( token[2].s );
		}
 	    } else if ( tu_strncmp ( token[0].s, "AUTHENTICATOR", 14 ) == 0 ){
		http_authenticator_image = malloc ( token[1].l+1 );
		tu_strnzcpy (http_authenticator_image, token[1].s, token[1].l);

	    } else if ( tu_strncmp ( token[0].s, "INCLUDE", 10 ) == 0 ) {
		if ( count < 2 ) {
		    tlog_putlog ( 0, "Missing filename in Include rule!/" );
		} else if ( sp > 18 ) {
		    tlog_putlog ( 0, "Nesting level too deep to include file!/");
		} else {
		    lnum[++sp] = 0;
		    if ( !(rf[sp] = fopen (token[1].s, "r" VMS_FOPEN_OPT)) ) {
			tlog_putlog ( 0, "Error openning include file '!AZ'!/",
				token[1].s );
			--sp;
		    } else {
			/* Let pre-processor know we are at new file */
			preprocess_rule ( count, 7, token );
		    }
		}	
	    } else if ( tu_strncmp ( token[0].s, "ACCESSLOG", 10 ) == 0 ) {
		int status, ndx, tlog_initlog();
		tu_strnzcpy ( temp, token[1].s, token[1].l );
		if ( acc_local_log ) {	/* we are inside localaddres block */
		    acc_log_count++;
		    ndx = 0 - acc_log_count;
		    multihome_last->accesslog = ndx;
		} else ndx = -1;
		status = tlog_initlog ( ndx, temp );
		if ( (status&1) == 0 ) tlog_putlog ( 0, 
			"Error opening access log file, '!AZ'!/", temp );
		else if ( status == 3 ) tlog_putlog ( 0,
			"Access log already active, '!AZ' not openned!/", temp);
		else tlog_putlog ( 0, 
			"Opened access log file, '!AZ', level: !SL!/",
			temp, http_log_level );
	    } else if ( tu_strncmp ( token[0].s, "TRACELEVEL", 10 ) == 0 ) {
		int status, tlog_initlog();
		if ( count < 2 ) {
		    tlog_putlog ( 0, "Missing argument for TraceLevel rule!/" );
		    http_log_level = 1;
		} else {
		    tu_strnzcpy ( temp, token[1].s, token[1].l );
		    if ( (temp[0] < '0') || (temp[0] > '9') ) {
			char *val = getenv(temp);
			if ( !val ) tlog_putlog(0,
			    "Invalid variable name in TraceLevel rule: '!AZ'!/",
			    temp );
			tu_strcpy ( temp, val ? val : "1" );
		    }
		    http_log_level = atoi ( temp );
		}
		tu_strnzcpy ( temp, token[2].s, (count > 2) ? token[2].l : 0 );
		status = tlog_initlog ( http_log_level, temp );
		if ( (status&1) == 0 ) tlog_putlog ( 0, 
			"Error opening trace log file, '!AZ'!/", temp );
		else if ( status == 3 ) tlog_putlog ( 0,
			"Trace log already active, '!AZ' not openned!/", temp);
		else tlog_putlog ( 0, "Opened trace file, '!AZ', level: !SL!/",
			temp, http_log_level );
	    } else if ( tu_strncmp ( token[0].s, "DNSLOOKUP", 10 ) == 0 ) {
		tu_strupcase ( token[1].s, token[1].s );
		if ( tu_strncmp ( token[1].s, "OFF", 4 ) == 0 ) {
		    http_dns_enable = 0;
		} else if ( tu_strncmp ( token[1].s, "ON", 3 ) == 0 ) {
		    http_dns_enable = 1;
		} else {
		    tlog_putlog (0, 
			"Bad value for DNSLookup directive, line !SL of rule file !AZ!/",
			lnum[sp], sp ? fgetname(rf[sp],fname) : rule_file );
		}
	    } else if ( tu_strncmp ( token[0].s, "WELCOME", 8 ) == 0 ) {
		if ( (index_file_alloc == 0) || (token[1].l > 0) ) {
		    add_index_file ( token[1].s );
		    if ( token[1].l == 0 ) {
			free ( http_index_filename[0] );
			http_index_filename[0] = (char *) 0;
			tlog_putlog (1,"Disabled search for WELCOME files.!/" );
		    }
		} else {
		   tlog_putlog ( 0, 
		   "Missing filename on WELCOME directive, line !SL of rule file !AZ!/",
			lnum[sp], sp ? fgetname(rf[sp],fname) : rule_file );
		}
	    } else if ( tu_strncmp ( token[0].s, "DIRACCESS", 10 ) == 0 ) {
		tu_strupcase ( token[1].s, token[1].s );
		if ( tu_strncmp ( token[1].s, "OFF", 4 ) == 0 ) {
		    http_dir_access = 2;	/* fully restricted (none) */
		} else if ( tu_strncmp ( token[1].s, "ON", 3 ) == 0 ) {
		    http_dir_access = 0;
		} else if ( tu_strncmp ( token[1].s, "SELECTIVE", 10 ) == 0 ) {
		    http_dir_access = 1;
		    if ( token[2].l > 0 ) {
		 	/* Override filename that flags browsable directories */
			http_dir_access_file = malloc ( token[2].l+1 );
			tu_strnzcpy(http_dir_access_file,token[2].s,token[2].l);
		    }
		} else {
		    tlog_putlog (0, 
			"Bad value for DirAccess directive, line !SL of rule file !AZ!/",
			lnum[sp], sp ? fgetname(rf[sp],fname) : rule_file );
		}
	    } else if ( tu_strncmp ( token[0].s, "PORT", 5 ) == 0 ) {
		/* set port numbers */
		int i;
		char *slash;
		if ( count < 2 ) {
		    tlog_putlog ( 0, "Missing argument for Port rule!/" );
		} else for ( i = 1; (i < count) && (i < 3); i++ ) {
		    tu_strnzcpy ( temp, token[i].s, token[i].l );
		    slash = tu_strstr ( temp, "/" );
		    if ( slash ) {
			http_scheme_fixup = malloc ( tu_strlen(temp) + 1 );
			tu_strcpy ( http_scheme_fixup, temp );
			*slash++ = '\0';
		    }
		    if ( (temp[0] < '0') || (temp[0] > '9') ) {
			char *val = getenv(temp);
			if ( !val ) tlog_putlog(0,
			    "Invalid variable name in Port rule: '!AZ'!/",
			    temp );
			tu_strcpy ( temp, val ? val : "1" );
		    }
		    http_ports[i-1] = atoi ( temp );
		}
	    } else if ( tu_strncmp ( token[0].s, "LOCALADDRESS", 13 ) == 0 ) {
#ifndef SCRIPTSERVER
		int status;
		if ( count > 1 ) {
		    /*
		     * Call TCP module to set listen address.
		     */
		    tu_strnzcpy ( temp, token[1].s, token[1].l );
		    status = ts_set_local_addr(temp);
		    if (status == 1 ) tlog_putlog ( 1, temp[0] == '@' ?
			"TCP driver option '!AZ'!/" :
			"Set local ip address to, '!AZ'!/", temp );
		    else tlog_putlog ( 0, 
			"Error setting local ip address to, '!AZ'!/",
			temp);
		    /*
		     * If hostname present, go into multihomed mode
		     * and being block of address-specific rules.
		     */
		    if ( count > 2 ) {
			/*
			 * Add to list of host defs so we can get
			 * name back from address.
			 */
			struct host_def *new; unsigned long a[4];
			http_multihomed = 1;
			new = (struct host_def *) 
				malloc(sizeof(struct host_def));
			new->next = (struct host_def *) 0;
			new->name = malloc(token[2].l+1);
			tu_strcpy ( new->name, token[2].s );
			/*
			 * Decode 'dot' address.
			 */
			sscanf (temp, "%d.%d.%d.%d", &a[0],&a[1],&a[2],&a[3]);
			new->a.octet[0] = a[0]; new->a.octet[1] = a[1];
			new->a.octet[2] = a[2]; new->a.octet[3] = a[3];
			new->accesslog = -1;
			if ( multihome_count == 0 ) {
			    multihome_list = multihome_last = new;
			} else {
			    multihome_last->next = new;
			    multihome_last = new;
			}
			multihome_count++;
			http_define_ident ( "*", 9, new->name );
			acc_local_log = 1;
		    }
		} else if ( http_multihomed ) {
		    /* End localaddress block. */
		    http_define_ident ( "*", 9, "" );
		    acc_local_log = 0;
		}
#endif
	    } else if ( tu_strncmp( token[0].s, "HOSTNAME", 9) == 0 ) {
	        http_default_host = malloc ( token[1].l + 1);
		tu_strnzcpy (http_default_host, token[1].s, token[1].l );
	    } else if ( tu_strncmp( token[0].s, "EVENTCOUNTER", 13) == 0 ) {
#ifndef SCRIPTSERVER
		tu_strupcase ( token[1].s, token[1].s );
		if ( tu_strncmp ( token[1].s, "CLIENTS", 10 ) == 0 ) {
		    http_enable_active_counters();
		    tlog_putlog (1,"Enabled counting of client threads.!/" );
		} else if ( tu_strncmp ( token[1].s, "HOSTCLASS", 10 ) == 0 ) {
		    if ( count >= 4 ) {
			http_define_host_counter ( token[2].s, token[3].s );
		    } else {
		        tlog_putlog ( 0, 
			    "Missing argument(s) for EventCounter rule!/" );
		    }
		} else {
		    tlog_putlog (0, 
			"Bad value for EventCounter directive, line !SL of rule file !AZ!/",
			lnum[sp], sp ? fgetname(rf[sp],fname) : rule_file );
		}
#endif
	    } else if ( tu_strncmp( token[0].s, "THREADPOOL", 13) == 0 ||
		tu_strncmp ( token[0].s, "SERVICE", 9 ) == 0 ) {
#ifndef SCRIPTSERVER
		if ( 0 == http_load_dynamic_service ( token, count, temp ) ) {
		    tlog_putlog (0, 
			"!AZ, line !SL of rule file !AZ!/", temp,
			lnum[sp], sp ? fgetname(rf[sp],fname) : rule_file );
		}
#endif
	    } else if ( tu_strncmp( token[0].s, "TIMELIMIT", 10) == 0 ) {
#ifndef SCRIPTSERVER
		int tlimit, climit;
		if ( count < 3 ) {
		    tlog_putlog ( 0, "Insufficient arguments in suffix command: !AZ!/",
			line );
		    continue;
		}
		tlimit = parse_timelimit ( token[2].s, temp );
		tu_strupcase ( temp, token[1].s );
		if ( tu_strncmp ( temp, "REQUEST", 8 ) == 0 ) {
		    http_request_time_limit = tlimit;
		} else if ( tu_strncmp ( temp, "RESPONSE", 10 ) == 0 ) {
		    http_response_time_limit = tlimit;
		} else if ( tu_strncmp ( temp, "KEEPALIVE", 10 ) == 0  ||
			tu_strncmp ( temp, "KEEP-ALIVE", 11 ) == 0 ) {
		    http_keepalive_time = tlimit;
		    if ( count > 3 ) {
			climit = atoi ( token[3].s );
			if ( climit > 1 ) http_keepalive_limit = climit;
		    }
		} else {
		    tlog_putlog ( 0, "Bad TimeLimit keyword: !AZ!/", 
			token[1].s );
		}
#endif
	    } else if ( tu_strncmp( token[0].s, "MANAGE", 7) == 0 ) {
		/*
		 * Set management parameters:
		 *     MANAGE PORT nnn
		 *     MANAGE HOST nnn.nnn.nnn.nnn
		 *     MANAGE SCRIPT script
		 */
#ifndef SCRIPTSERVER
		int tlimit, climit, status;
		if ( count < 3 ) {
		    tlog_putlog ( 0, "Insufficient arguments in manage command: !AZ!/",
			line );
		    continue;
		} else if ( tu_strlen(token[1].s ) >= sizeof(temp) ) {
		    tlog_putlog ( 0, "Option name too long: !AZ!/", line );
		    continue;
	        }
		tu_strupcase ( temp, token[1].s );
		status = http_configure_manage_port ( temp,
			token[2].s, token[3].s, token[4].s );
#endif
	    } else {
		tlog_putlog (0,
			"Unrecognized data, line !SL of rule file !AZ: !AZ!/",
			lnum[sp], sp ? fgetname(rf[sp],fname) : rule_file,
			line );
	    }
	}
	if ( http_log_level > 1 ) tlog_putlog ( 1,
		"Rule file has !SL suffix def!%S., !SL translation rule!%S!/",
		sfx_count, id_count );
	while ( sp >= 0 ) fclose ( rf[sp--] );
	if ( pass_count == 0 ) {
	    tlog_putlog ( 0, "Rule file contains no 'pass' actions, !AZ!/",
		"all requests would be _ruled_ out." );
	    return 2160;
	}
    }
    /*
     * Provide default rules if not found in rule file.
     */
    if ( sfx_count == 0 ) {
	tlog_putlog ( 0, "Adding default suffix rules!/" );
	http_define_suffix ( ".gif", "image/gif", "BINARY", "1.0" );
	http_define_suffix ( ".txt", "text/plain", "8BIT", "0.5" );
	http_define_suffix ( ".com", "text/plain", "8BIT", "0.5" );
	http_define_suffix ( ".html", "text/html", "8BIT", "1.0" );
    }
    return 1;
}
/*****************************************************************************/
/* Return multihomed info.  Return status is 0 if not-multihomed or
 * index out of range, 1 on success.  Index starts at 0.
 */
int http_multihome_scan ( int ndx, 	/* position in list, 0..n-1 */
	char **name,			/* Returns hostname */
	unsigned char **address )	/* IP address */
{
    int i;
    struct host_def *ptr;
    if ( !http_multihomed || (ndx < 0) ) return 0;	/* list empty */
    for ( ptr = multihome_list; ptr; ptr = ptr->next ) {
	--ndx;
	if ( ndx < 0 ) {
	    *name = ptr->name;
	    *address = ptr->a.octet;
	    return 1;
	}
    }
    return 0;		/* index out of range */
}
/*****************************************************************************/
/* Lookup IP address in multihomed list and set local_address element in
 * access info structure to point to corresponding host name.  If no match,
 * field is set to http_default_host.
 *
 * If an accesslog rule was present in the localaddress block, return
 * the tlog index number for it (negative number).
 *
 * Return value:
 *	1		Success.
 *	0		Lookup error, acc loaded with default hostname.
 */
int http_multihomed_hostname ( unsigned long address, access_info acc,
	int *accesslog )
{
    struct host_def *blk;
    if ( multihome_count > 0 ) {
	for ( blk = multihome_list; blk; blk = blk->next ) {
	    if ( blk->a.number == address ) {
		acc->local_address = blk->name;
		*accesslog = blk->accesslog;
		return 1;
	    }
	}
    }
    acc->local_address = http_default_host;	/* not found */
    return 0;
}
/*****************************************************************************/
/* Handle preprocessor operations, rewriting element list.
 * Directives:
 *    .DEFINE symbol elem1 [elem2 [elem3...]]
 *    .EXPAND [string|$symbol] ...
 *    .IGNORE directive [directive [directive...]]
 *    .ITERATE [string|$dummyarg] ...
 *     .NEXT arg ...
 *	INCLUDE		! push operations, note no leading period
 *
 * A negative count forces a pop operations.
 *
 * Function value returned is new size of resulting list or -1 for error.
 */
static int preprocess_rule ( int count, int limit, string *elem )
{
    struct keydef {
	struct keydef *next;
	int count;
        string elem[1];
    } *sym;
    struct keysave {
	struct keysave *next;
	struct keydef *symlist, *ignore_list;
    } *env;
    static struct keysave *keystack = (struct keysave *) 0;
    static struct keydef *symlist = (struct keydef *) 0;
    static struct keydef *ignore_list = (struct keydef *) 0;
    static work_limit = 0, pattern_count = 0;
    static string *work_elem, *pattern;
    int i, j, k;

    if ( limit > work_limit ) {
	if ( http_log_level > 14 ) tlog_putlog(14,
		"Expanding pre-processor work list to !SL!/", limit );
	if ( work_limit == 0 ) work_elem = malloc(sizeof(string)*limit);
	else work_elem = realloc(work_elem,sizeof(string)*limit);
	if ( work_limit == 0 ) pattern = malloc(sizeof(string)*limit);
	else work_elem = realloc(pattern,sizeof(string)*limit);
	work_limit = limit;
    }
    /*
     * Check for stack operations. 
     */
    if ( count < 0 ) {
	/*
	 * Restore next symlist and ignore_list saved on keystack.
	 */
	if ( !keystack ) return -1;
	symlist = keystack->symlist;
	ignore_list = keystack->ignore_list;
	env = keystack;
	keystack = keystack->next;
	free ( env );
	return 0;
    } else if ( tu_strncmp ( elem[0].s, "INCLUDE", 8) == 0 ) {
	/*
	 * Save current symlist and ignore lists and reset to empty.
	 */
	env = (struct keysave *) malloc ( sizeof(struct keysave) );
	if ( !env ) return -1;
	env->next = keystack;
	keystack = env;
	env->symlist = symlist;
	env->ignore_list = ignore_list;
	
	symlist = ignore_list = (struct keydef *) 0;
	return 0;
    }
    /*
     * Process standard directives.
     */
    if ( (count > 1 ) && (tu_strncmp(elem[0].s,".DEFINE", 8) == 0) ) {
	/*
	 * make duplicate of elem list and add to symbol table.
	 */
	tu_strupcase ( elem[1].s, elem[1].s );
	sym = (struct keydef *) malloc ( sizeof(struct keydef) +
		sizeof(string)*count-2 );
	sym->next = symlist;
	symlist = sym;
	sym->count = count-1;

	for ( i = 0; i < count-1; i++ ) {
	    sym->elem[i].l = elem[i+1].l;
	    sym->elem[i].s = (char *) malloc ( elem[i+1].l + 1 );
	    tu_strcpy ( sym->elem[i].s, elem[i+1].s );
	}
	return 0;
    } else if ( tu_strncmp(elem[0].s,".EXPAND",8) == 0 ) {
	/*
	 * Copy elements to temporary work array.
	 *   i - input index, j - output index.
	 */
	for ( i =1, j = 0; i < count; i++ ) {
	    if ( elem[i].s[0] == '$' ) {
		/*
		 * String following $ is symbol name to match.
		 */
		tu_strupcase(elem[i].s, elem[i].s);
		for ( sym = symlist; sym; sym = sym->next ) {
		    if ( tu_strncmp(sym->elem[0].s, &elem[i].s[1],
				sym->elem[0].l) == 0 ) {
			if ( '\0' == elem[i].s[1+sym->elem[0].l] ) break;
		    }
		}
		if ( !sym ) {
		    tlog_putlog(0, "Undefined symbol in expansion: !AZ!/",
			elem[i].s );
		    return -1;		/* error */
		}
		/*
		 * append to output, null definition terminates expansion
		 */
		if ( sym->count < 1 ) break;
		if ( sym->elem[1].l <= 0 ) break;
		for ( k = 1; k < sym->count; k++ ) {
		    work_elem[j++] = sym->elem[k];
		}
	    } else {
		/* Copy element with no expansion */
		work_elem[j++] = elem[i];
	    }
	}
	/*
	 * Copy result back, re-upcase first token.
	 */
	for ( i = 0; i < j; i++ ) elem[i] = work_elem[i];
	if ( j > 0 ) tu_strupcase ( elem[0].s, elem[0].s );
	for ( i = j; i < limit; i++ ) elem[j].l = 0;

	if ( http_log_level > 14 ) {
	   tlog_putlog(14,"Pre-processor expansion:" );
	    for ( i = 0; i < j; i++ ) tlog_putlog(14," '!AZ'", elem[i].s );
	    tlog_putlog(14,"!/");
	}
	return j;
    } else if ( tu_strncmp(elem[0].s,".NEXT",8) == 0 ) {
	/*
	 * Build work array drawing each element from pattern.
	 */
	for ( j = 0, i = 1; j < pattern_count; j++ ) {
	    if ( pattern[j].s[0] == '$' ) {
		/* Truncate pattern if null element */
		if ( i >= count ) break;
		work_elem[j] = elem[i++];
	    } else {
		work_elem[j] = pattern[j];
	    }
	}
	/*
	 * Copy result back, re-upcase first token.
	 */
	for ( i = 0; i < j; i++ ) elem[i] = work_elem[i];
	if ( j > 0 ) tu_strupcase ( elem[0].s, elem[0].s );
	for ( i = j; i < limit; i++ ) elem[j].l = 0;

	if ( http_log_level > 14 ) {
	   tlog_putlog(14,"Pre-processor iteration:" );
	    for ( i = 0; i < j; i++ ) tlog_putlog(14," '!AZ'", elem[i].s );
	    tlog_putlog(14,"!/");
	}
	return j;
    } else if ( tu_strncmp(elem[0].s,".ITERATE",8) == 0 ) {
	/*
	 * Save elements in pattern buffer.
	 */
	pattern_count = count - 1;
	for ( i = 0; i < pattern_count; i++ ) {
	    pattern[i].l = elem[i+1].l;
	    pattern[i].s = (char *) malloc ( elem[i+1].l + 1 );
	    tu_strcpy ( pattern[i].s, elem[i+1].s );
	}
	return 0;
    } else if ( tu_strncmp(elem[0].s,".IGNORE",8) == 0 ) {
	/*
	 * Append symbols to ignore_list.
	 */
	tu_strupcase ( elem[1].s, elem[1].s );
	sym = (struct keydef *) malloc ( sizeof(struct keydef) +
		sizeof(string)*count-1 );
	sym->next = ignore_list;
	ignore_list = sym;
	sym->count = count-1;

	for ( i = 0; i < count-1; i++ ) {
	    sym->elem[i].l = elem[i+1].l;
	    sym->elem[i].s = (char *) malloc ( elem[i+1].l + 1 );
	    tu_strupcase ( sym->elem[i].s, elem[i+1].s );
	}
	return 0;
    } else {
	/*
	 * See if directive was on one of the ignore lists and return 0 if
	 * found, which effectively ignores the line.
	 */
	for ( sym = ignore_list; sym; sym = sym->next ) {
	    for ( i = 0; i < sym->count; i++ ) {
		if ( 0 == tu_strncmp ( elem[0].s, sym->elem[i].s,
			sym->elem[i].l+1 ) ) return 0;
	    }
	}
	return -1;		/* unknown directive */
    }
    return count;
}
