/*
 * SSL server high-level interface.  Applications using this interface
 * supply callback routines that this layer will call to perform raw
 * input and output to the client connection.  The get/put callbacks
 * match the semantics of tserver_tcp's interface.
 *
 */
#include "pthread_1c_np.h"
#include "tutil.h"

#include <stdio.h>
#include <stdlib.h>
#include <ssdef.h>
#include "ssl_server_v3.h"

static ssl_context free_ctx;
static pthread_mutex_t context_db;   /* Mutex to guard access to global data */
static flush_outbuf ( ssl_context ctx );	/* forward reference */

/*****************************************************************************/
/* Function tssl_read_rec handles interpreting the record layer header for
 * SSL data from the client.
 */
int tssl_read_rec ( ssl_context ctx )
{
    int status, length, i, initial_read;
    struct ssl_rec *rec;
    /*
     * Read first 2 bytes to get length and protocol, assume using
     * version 2 protocol if first byte greater than 127 (i.e. all
     * message types for versions after 2.0 will be less than 128) and that
     * no 'security escapes' were ever defined for SSL 2.0.
     */
    initial_read = 2;
    if ( ctx->ssl_version > 2 ) initial_read = 5;
    rec = &ctx->in_rec;
    for ( rec->filled = 0; rec->filled < initial_read; rec->filled+=length ) {
       status = (*ctx->get) ( ctx->tcp_ctx, &rec->hdr.raw[rec->filled],
		initial_read-rec->filled, &length );
       if ( (status&1) == 0 ) return status;
    }
    if ( ctx->ssl_version == 0 ) {
	/*
	 * New connection, try to determine prototol.
	 */
	if ( rec->hdr.raw[0] > 127 ) ctx->ssl_version = 2;
	else {
	    /*
	     * Read remaining 3 bytes of header and get protocol from it.
	     */
	    for (; rec->filled < sizeof(rec->hdr.plain); rec->filled+=length) {
                status = (*ctx->get) ( ctx->tcp_ctx, &rec->hdr.raw[rec->filled],
		    sizeof(rec->hdr.plain)-rec->filled, &length );
                if ( (status&1) == 0 ) return status;
	    }
	}
    }
    if ( ctx->ssl_version <= 2 ) {
	/*
	 * Version 2 format, length is extracted from 1st 2or 3 bytes.
	 */
	char sts_line[80];
	if ( rec->hdr.raw[0] > 127 ) {
	    /* 2-byte header */
	    rec->frag_length = ((rec->hdr.raw[0]&0x7f)<<8) | rec->hdr.raw[1];
	    rec->v2_pad = 0;
	} else {
           status = (*ctx->get) ( ctx->tcp_ctx, &rec->hdr.raw[rec->filled],
		1, &length );
           if ( (status&1) == 0 ) return status;
	   rec->frag_length = ((rec->hdr.raw[0]&0x3f)<<8) | rec->hdr.raw[1];
	   rec->v2_pad = rec->hdr.v2_hdr.pad;
	}
sprintf(sts_line,"V2 SSL header detected, fragment length: %d\n",
rec->frag_length );
(*ctx->error) ( 1, sts_line );
    } else {
	/*
	 * Version is >2 length follows type and version.
	 */
        rec->frag_length = (rec->hdr.plain.len_msb << 8)+rec->hdr.plain.len_lsb;
    }
    /*
     * We have length, ensure data buffer is large enough.
     */
    if ( (rec->frag_length+1024) > rec->data_alloc ) {
	if ( rec->data_alloc > 0 ) {	/* release current buffer */
	    LOCK_C_RTL
	    free ( rec->data );
	    UNLOCK_C_RTL
	}
	/*
	 * Choose base size that is next higher multiple of 8K, then add 1K
	 * The 1K headroom is an allowance for decompression.
	 */
	for ( rec->data_alloc = 0; rec->data_alloc < rec->frag_length;
		rec->data_alloc += 8192 );
	rec->data_alloc += 1024;
	LOCK_C_RTL
	rec->data = malloc ( rec->data_alloc );
	if ( !rec->data ) {
	    (*ctx->error) (SS$_INSFMEM,"Failure to allocate rec layer buffer");
	    return SS$_INSFMEM;
	}
    }
    /*
     * FInally, read the data fragment from the client.
     */
    for ( rec->filled = 0; rec->filled < rec->frag_length; 
		rec->filled += length) {
	int rd_size;
	rd_size = rec->frag_length - rec->filled;
	if ( rd_size > ctx->max_io ) rd_size = ctx->max_io;
       status = (*ctx->get) 
		( ctx->tcp_ctx, &rec->data[rec->filled], rd_size, &length );
       if ( (status&1) == 0 ) break;
    }
    rec->sequence++;
    return status;
}
/***************************************************************************/
/* Function tssl_initalize() should be called once per image invocation.
 * It handles any global configuration needed by the ssl_server module.
 */
int tssl_initialize(int (*error) ( int, char *) )
{
    /*
     * Initialize ssl_context lookaside list and mutex that guards it.
     */
    free_ctx = (ssl_context) 0;
    INITIALIZE_MUTEX ( &context_db );

    (*error) ( 1, "SSL-V3 initialized" );
    return SS$_NORMAL;
}

/***************************************************************************/
/* Initialize context for new client connect.  SSL negoitation takes place
 * at this point.
 */
int tssl_init_server_context ( ssl_context *ctx,	/* New context */
    int (*get) ( void *, char *, int, int * ),
    int (*put) ( void *, char *, int ),
    int (*error) ( int code, char *text ),
    void *io_ctx,		/* I/O context for get() and put() calls */
    int max_io_size )		/* Max I/O length per get or put call */
{
    int status;
    /*
     * Allocate a structure.
     */
    pthread_mutex_lock ( &context_db );
    if ( free_ctx ) {
	*ctx = free_ctx;
	free_ctx = (*ctx)->next;
	pthread_mutex_unlock ( &context_db );
     } else {
	pthread_mutex_unlock ( &context_db );
	LOCK_C_RTL
	*ctx = (ssl_context) malloc ( sizeof(struct ssl_control) );
	UNLOCK_C_RTL
	if ( !*ctx ) {
	    (*error) ( SS$_INSFMEM, "Failure to allocate ssl_context" );
	    return SS$_INSFMEM;
	}
    }
    (*ctx)->next = (ssl_context) 0;
    /*
     * Initialize TCP callback interface.
     */
    (*ctx)->ssl_version = (*ctx)->cnx_state = 0;
    (*ctx)->tcp_ctx = io_ctx;
    (*ctx)->get = get;
    (*ctx)->put = put;
    (*ctx)->error = error;
    (*ctx)->max_io = max_io_size;
    /*
     * Initialize record layer.
     */
    (*ctx)->in_rec.data_alloc = 0;
    (*ctx)->out_rec.data_alloc = 0;
    /*
     * Read first record and perform handshake
     */
    status = tssl_read_rec ( *ctx );
    if ( (status&1) == 1 ) {
	TRY {
	    status = tssl_handshake ( *ctx );
	}
	CATCH_ALL {
	    (*(*ctx)->error) ( SS$_ABORT, 
		"Fatal error while performing SSL handshake" );
#ifdef PTHREAD_USE_D4
	    exc_report ( THIS_CATCH );
#endif
	    status = SS$_ABORT;
	}
	ENDTRY
    }

    if ( (status&1) == 0 ) {
	/*
	 * If we return error status, caller won't call rundown so we must.
	 */
	tssl_rundown_context ( *ctx, 0 );	/* free memory */
	*ctx = (ssl_context) 0;
    }

    return status;
}

/*****************************************************************************/
/*  The tssl_put_app_data function encrypts the application data and sends
 *  it to the client.  The application data may be buffered prior to
 *  encryption.
 */
int tssl_put_app_data ( ssl_context ctx,  unsigned char *data, int length )
{
    int status, i, j;
    /*
     * Save data in output buffer.
     */
    return SS$_NORMAL;
}

/*****************************************************************************/
/* Tssl_get_app_data flushes any pending output data and reads sll records
 * until more application data arrives.  The function returns a pointer to
 * the decrypted application data.
 */
int tssl_get_app_data ( ssl_context ctx, int *length, unsigned char **data )
{
    int status;

    return 0;
}

/*****************************************************************************/
/* Cleanup context (session IDs for context may still be cached).  If flags
 * bit 0 is set, rundown will flush pending output data before performing
 * rundown.
 */
int tssl_rundown_context ( ssl_context ctx, int flags )
{
    int status;
    /*
     * Flush remaining outbound data if requested.
     */
    status = SS$_NORMAL;
    /*
     * Put structure back on free list.
     */
    pthread_mutex_lock ( &context_db );
    ctx->next = free_ctx;
    free_ctx = ctx;
    pthread_mutex_unlock ( &context_db );
    return 0;
}

/*****************************************************************************/
/* Send output in max_io-sized chunks to client.
 */
int flush_outbuf ( ssl_context ctx )
{
    int status, segment, i;

   status = 1;
    return status;
}
