/*  CALLMON - A Call Monitor for OpenVMS Alpha
 *
 *  File:     CALLMON$RELOC.C
 *  Author:   Thierry Lelegard
 *  Version:  1.0
 *  Date:     24-JUL-1996
 *
 *  Abstract: This module processes the relocations inside an image.
 */


#include "callmon$private.h"


/*******************************************************************************
 *
 *  This routine modifies the relocations of one routine inside its
 *  image. Inside a shareable image, the relocations are references
 *  inside the same image. We explore each relocation table (they
 *  were generated by the linker). When we find a relocation which
 *  was adjusted by the image activator to point to the old routine,
 *  we modify it to point to the jacket routine.
 *
 *  There are 2 types of relocations:
 *    - quadword relocations
 *    - longword relocations
 */

void callmon$$process_relocations (routine_t* routine)
{
    int64 old_entry, old_pdsc, new_entry, new_pdsc;
    pdsc_t* old_pdsc_p;
    rel_t* rel;
    uint32 bit;
    int64* squad;
    int32* slong;
    int hook_count = 0;

    /* We cannot modify resident and protected images */

    if (routine->image->resident || routine->image->protected)
        return;

    /* If there is no relocation in this image, nothing to do */

    if (routine->image->eiaf == NULL)
        return;

    /* If the routine cannot be relocated, ignore it */

    if (routine->unrelocatable) {

        if (callmon$$own.trace_flags & TRACE_UNRELOCATED)
            callmon$$putmsg (CALLMON$_NOTREL, 1, routine->name);

        return;
    }

    /* Compute the address of the old and new routine */

    old_pdsc_p = (pdsc_t*) routine->lkp.lkp$q_proc_value;
    old_pdsc   = routine->lkp.lkp$q_proc_value;
    old_entry  = routine->lkp.lkp$q_entry;
    new_pdsc   = (int64) routine->jacket;
    new_entry  = *(int64*) routine->jacket->pdsc$q_entry;

    /* Trace message when necessary */

    if (callmon$$own.trace_flags & TRACE_RELOCATION)
        callmon$$putmsg (CALLMON$_DORELOC, 6, routine->name,
            routine->image->logname, old_pdsc, old_entry, new_pdsc, new_entry);

    /* Modify all quadword relocations */

    if (routine->image->eiaf->eiaf$l_qrelfixoff != 0) {

        rel = (rel_t*) ((char*) routine->image->eiaf +
            routine->image->eiaf->eiaf$l_qrelfixoff);

        while (rel->rel$l_count != 0) {

            squad = (int64*) (
                (char*) routine->image->imcb->imcb$l_base_address +
                rel->rel$l_addr);

            for (bit = 0; bit < rel->rel$l_count; bit++) {
                if (rel->rel$t_bitmap [bit / 8] & (1 << (bit % 8))) {

                    /* A relocation was performed here during image fixup */

                    if (squad [bit] == old_pdsc) {

                        /* The relocation points to the procedure descriptor
                         * of the intercepted routine. Modify to point to
                         * the new routine. Note that one of these relocations
                         * is inside the symbol vector. */

                        if (callmon$$own.trace_flags & TRACE_RELOCATION)
                            callmon$$putmsg (CALLMON$_QPREL, 1, squad + bit);

                        if (callmon$$writeable (squad + bit, sizeof (int64))) {
                            squad [bit] = new_pdsc;
                            hook_count++;
                        }
                    }

                    else if (squad [bit] == old_entry &&
                        squad + bit != (int64*) &old_pdsc_p->pdsc$l_entry) {

                        /* The relocation points to the procedure entry code
                         * of the intercepted routine. Modify to point to
                         * the new routine. Note that we must NOT modify the
                         * address if it is part of the original procedure
                         * descriptor (we must keep it intact). */

                        if (callmon$$own.trace_flags & TRACE_RELOCATION)
                            callmon$$putmsg (CALLMON$_QEREL, 1, squad + bit);

                        if (callmon$$writeable (squad + bit, sizeof (int64))) {
                            squad [bit] = new_entry;
                            hook_count++;
                        }
                    }
                }
            }

            rel = (rel_t*) (rel->rel$t_bitmap + rel->rel$l_count / 8);
        }
    }

    /* Modify all longword relocations */

    if (routine->image->eiaf->eiaf$l_lrelfixoff != 0) {

        rel = (rel_t*) ((char*) routine->image->eiaf +
            routine->image->eiaf->eiaf$l_lrelfixoff);

        while (rel->rel$l_count != 0) {

            slong = (int32*) (
                (char*) routine->image->imcb->imcb$l_base_address +
                rel->rel$l_addr);

            for (bit = 0; bit < rel->rel$l_count; bit++) {
                if (rel->rel$t_bitmap [bit / 8] & (1 << (bit % 8))) {

                    /* A relocation was performed here during image fixup */

                    if (slong [bit] == old_pdsc) {

                        if (callmon$$own.trace_flags & TRACE_RELOCATION)
                            callmon$$putmsg (CALLMON$_LPREL, 1, slong + bit);

                        if (callmon$$writeable (slong + bit, sizeof (int32))) {
                            slong [bit] = new_pdsc;
                            hook_count++;
                        }
                    }

                    else if (slong [bit] == old_entry &&
                        slong + bit != (int32*) old_pdsc_p->pdsc$l_entry) {

                        if (callmon$$own.trace_flags & TRACE_RELOCATION)
                            callmon$$putmsg (CALLMON$_LEREL, 1, slong + bit);

                        if (callmon$$writeable (slong + bit, sizeof (int32))) {
                            slong [bit] = new_entry;
                            hook_count++;
                        }
                    }
                }
            }

            rel = (rel_t*) (rel->rel$t_bitmap + rel->rel$l_count / 8);
        }
    }

    if ((callmon$$own.trace_flags & TRACE_REFERENCES) && hook_count != 0)
        callmon$$putmsg (CALLMON$_RELOCCNT, 3, hook_count, routine->name,
            routine->image->logname);
}
