/* command line handling for rstsflx */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "flx.h"
#include "fldef.h"
#include "silfmt.h"
#include "scancmd.h"
#include "fip.h"
#include "diskio.h"
#include "filename.h"
#include "doalloc.h"
#include "doget.h"
#include "doident.h"
#include "dolist.h"
#include "dotype.h"
#include "dodump.h"
#include "doput.h"
#include "dodelete.h"
#include "dorename.h"
#include "dorts.h"
#include "dodir.h"
#include "doprot.h"
#include "doinit.h"
#include "dohook.h"
#include "docomp.h"

swtab	sw;			/* switch flag table */

char	*progname;		/* name of program (argv[0]) */
char	*cmdname;		/* command name (argv[1]) */
char	cmdbuf[LSIZE];		/* command line buffer */
char	*(*fargv)[] = NULL;	/* array of pointers to arguments */
int	fargc;			/*  and count of how many we found */
char	*defdevice = NULL;	/* default device name */
char	*defsize = NULL;	/*  and size, as a string */

typedef struct {
	const char	*kw;
	void		(*hand)(int, char *[]);
} cmdent;

/* This routine is used to ask (if -query was specified) whether a file
 * should be processed or not.  It returns the answer, which is a character
 * "y", "n" or "q".  Note that the answer "a" is returned as "y" and
 * further prompting is turned off.
 */

char doquery (void (*printfile)(void *), void *name) {
	char	answer[200];
	char	first;

	if (sw.query == NULL) return ('y');
	else {
		for (;;) {
			printf (" %s ", cmdname);
			(*printfile) (name);
			printf (" (y,n,a,q)? ");
			fflush (stdout);
			gets (answer);
			first = tolower (answer[0]);
			if (first == 'q' || first == 'n' || first == 'y') return (first);
			if (first == 'a') {
				sw.query = NULL;
				return ('y');
			}
			printf ("Invalid answer\n");
		}
	}
}

/* This routine is used as the generic main loop for many command
 * processing routines.  The first two arguments define the list of
 * command (file) arguments.  The third argument controls what to do
 * if the name part of the filespec is null:
 *	NOTNULL		error -- "missing filename"
 *	NULLISWILD	ok, substitute wildcard ("?") for the missing name
 *	NULLISNULL	ok, leave the name part null
 *
 * -query switch processing is provided in this routine.  If a command
 * does not want -query to have effect, it should override the switch to
 * be absent (sw.query = NULL).
 */

void dofiles (int argc, char *argv[], commandaction action, int nullflag)
{
	int	fnum, j;
	firqb	f;
	int	matches;
	char	answer;

	if (argc == 0) printf ("Usage: %s %s files...\n", progname, cmdname);
	for (fnum = 0; fnum < argc; fnum++) {
		if (!parse (argv[fnum], &f)) {
			printf ("Invalid filespec %s\n", argv[fnum]);
			continue;
		}
		if (nullflag == NOTNULL && (f.flags & f_name) == 0) {
			printf ("Missing filename %s\n", argv[fnum]);
			continue;
		} else if (nullflag == NULLISWILD) {
			if ((f.flags & f_name) == 0) 
				for (j = 0; j < 6; j++) f.name[j] = '?';
				f.flags |= f_name | f_namw;
			if ((f.flags & f_ext) == 0) {
				f.name[7] = '?';
				f.name[8] = '?';
				f.name[9] = '?';
				f.flags |= f_ext | f_extw;
			}
		}
		initfilescan (&f, gfddcntbl);	/* setup file scan */
		matches = 0;
		answer = 'y';			/* dummy */
		while (nextfile (&f)) {
			matches++;			/* found another */
			answer = doquery ((void (*)())printcurname, &f);
			if (answer == 'q') break;
			if (answer == 'n') continue;
			(*action) (&f);			/* do whatever */
		}
		if (answer == 'q') break;		/* out two levels... */
		if (matches == 0) {
			printf ("No files matching ");
			printfqbname (&f);
			printf ("\n");
		}
	}
}

void dodisk (int argc, char *argv[])
{
	long	rsize, tsize;
	int	flag;
	
	if (defdevice != NULL) free (defdevice);
	defdevice = NULL;
	if (argc > 0) {
		if (argc > 1) {			/* size also supplied */
			if (defsize != NULL) free (defsize);
			getsize (argv[1], &rsize, &tsize, &flag);
			if (rsize == 0) {
				printf ("Invalid size %s\n", argv[1]);
				return;
			}
			defsize = malloc (strlen (argv[1]) + 1);
			defdevice = malloc (strlen (argv[0]) + 1);
			if (defdevice == NULL || defsize == NULL) rabort(NOMEM);
			strcpy (defdevice, argv[0]);
			strcpy (defsize, argv[1]);
		} else {
			defdevice = malloc (strlen (argv[0] + 1));
			if (defdevice == NULL) rabort(NOMEM);
			strcpy (defdevice, argv[0]);
		}
	}
	if (sw.verbose != NULL) {
		setrname ();
		if (defsize != NULL)
			printf ("Default RSTS disk name is %s, size %ld\n",
				rname, rsize);
		else	printf ("Default RSTS disk name is %s\n", rname);
	}
}

void doexit (int argc, char *argv[])
{
	exit (EXIT_SUCCESS);
}

const cmdent commands[] = {
	{ "list", dolist },
	{ "directory", dolist },
	{ "ls", dolist },
	{ "type", dotype },
	{ "cat", dotype },
	{ "dump", dodump },
	{ "exit", doexit },
	{ "quit", doexit },
	{ "bye", doexit },
	{ "get", doget },
	{ "put", doput },
	{ "delete", dodelete },
	{ "rm", dodelete },
	{ "protect", doprot },
	{ "rts", dorts },
	{ "runtime", dorts },
	{ "rename", dorename },
	{ "mv", dorename },
	{ "move", dorename },
	{ "hook", dohook },
	{ "mkdir", domkdir },
	{ "rmdir", dormdir },
	{ "identify", doident },
	{ "allocation", doalloc },
	{ "initialize", doinit },
	{ "dskint", doinit },
	{ "disk", dodisk },
	{ "compress", docomp },
	{ NULL, NULL } };

typedef struct {
	char	*kw;
	char	**flag;
	int	arg;		/* TRUE if switch takes an argument */
} swname;

#define sd(x,y)		{ #x, &sw.y, FALSE }	/* switch w/o arg */
#define sda(x,y)	{ #x, &sw.y, TRUE }	/* switch with arg */

const swname switches[] = {
	sd(ascii,ascii),
	sd(binary,bswitch),
	sd(brief,bswitch),
	sda(clustersize,clusiz),
	sd(confirm,query),
	sd(contiguous,contig),
	sda(create,create),
	sda(disk,rstsdevice),
	sd(Debug,debug),
	sda(end,end),
	sda(filesize,size),
	sd(full,full),
	sd(image,bswitch),
	sd(long,full),
	sda(merge,merge),
	sd(oattributes,oattr),
	sd(protect,prot),
	sd(query,query),
	sd(replace,replace),
	sd(r,rmsvar),
	sda(rf,rmsfix),
	sd(rv,rmsvar),
	sd(rs,rmsstm),
	sda(size,size),
	sda(Size,disksize),
	sda(start,start),
	sd(tree,tree),
	sd(unprotect,unprot),
	sd(verbose,verbose),
	sd(wide, wide),
	sd(Write,overwrite),
	sd(1column,narrow),
	sd(user,user),
	sd(hex,hex),
	{ "", NULL, FALSE } };

commandhandler scanargs(int argc, char *argv[])
{
	int		n, l;
	const cmdent	*c;
	const swname 	*s, *swn;
	commandhandler	command;

	memset (&sw, 0, sizeof (swtab));	/* Indicate no switches yet */
	if (fargv != NULL) free (fargv);
	if ((fargv = malloc (argc * sizeof(char *))) == NULL) rabort (NOMEM);
	fargc = 0;				/* No arguments yet */
	c = commands;
	command = NULL;				/* Nothing found yet */
	cmdname = argv[1];			/* save command name ptr */
	l = strlen (cmdname);			/* get length of command */
	while (c->kw != NULL) { 		/* look for matching command */
		if (strncmp (cmdname, c->kw, l) == 0) {
			if (strlen (c->kw) == l) {
				command = c->hand;
				break;		/* stop now on exact match */
			}
			if (command == NULL) command = c->hand;
			else if (command != c->hand) {
				printf ("Ambiguous command %s\n", cmdname);
				return (NULL);
			}
		}
		c++;
	}
	if (command == NULL) {
		printf ("Unrecognized command %s\n", cmdname);
		return (NULL);
	}
	for (n = 2; n < argc; n++) {		/* scan the command arguments */
		if (argv[n][0] == '-') {	/* it's a switch */
			s = switches; 		/* point to switch names */
			l = strlen (argv[n]) - 1;
			swn = NULL;		/* no match yet */
			while (s->flag != NULL) {
				if (strncmp (&argv[n][1], s->kw, l) == 0) {
					if (strlen (s->kw) == l) {
						swn = s;
						break;	/* exact match */
					}
					if (swn == NULL) swn = s;
					else if (s->flag != swn->flag) {
						printf ("Ambiguous switch %s\n", 
							argv[n]);
						return (NULL);
					}
				}
				s++;
			}
			if (swn == NULL) {
				printf ("Unrecognized switch %s\n", argv[n]);
				return (FALSE);
			}
			*(swn->flag) = argv[n];
			if (swn->arg) {			/* switch takes an argument */
				if (++n == argc) {	/* skip it */
					printf ("Missing argument for switch %s\n", argv[n - 1]);
					return (FALSE);
				}
				*(swn->flag) = argv[n];	/* point to that */
			}
		} else (*fargv)[fargc++] = argv[n];
	}
	return (command);
}

