/*
 *	vms.c - dynamic loading support for TCL under OpenVMS
 *
 *	from Eckart Meyer (meyer@ifn.ing.tu-bs.de)
 *
 *  -----------------------------------------------------------------
 *    Copyright 1993 D.I.S. - Universita` di Pavia - Italy
 *  -----------------------------------------------------------------
 *
 *  Permission to  use,  copy,   modify,   distribute  this  software
 *  and  its  documentation for any purpose is hereby granted without
 *  fee, provided that the above copyright  notice   appear   in  all
 *  copies   and  that both that copyright notice and this permission
 *  notice appear in supporting documentation, and that the  name  of
 *  D.I.S.   not  be  used  in advertising or publicity pertaining to
 *  distribution of the software without specific, written prior per-
 *  mission.   D.I.S.  makes no representations about the suitability
 *  of this software for any purpose.  It is provided "as is" without
 *  express or implied warranty.
 *
 *  D.I.S. DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, IN-
 *  CLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 *  NO EVENT SHALL D.I.S.  BE LIABLE FOR  ANY  SPECIAL,  INDIRECT  OR
 *  CONSEQUENTIAL  DAMAGES  OR  ANY DAMAGES WHATSOEVER RESULTING FROM
 *  LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION  OF  CONTRACT,
 *  NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNEC-
 *  TION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <lib$routines.h>
#include <libdef.h>
#include <ssdef.h>
#include <rmsdef.h>
#include <errno.h>

#include "tcl.h"

static void setupp(char *s);


/*
 *----------------------------------------------------------------------
 *
 * cond_handler --
 *
 *	Catches for the boring "compilation warnings" ins shared
 *	images. Convert this message into SS$_NORMAL. Re-signal
 *	all other messages.
 *
 *----------------------------------------------------------------------
 */

static long
cond_handler(long *sigargs, long *mechargs)
{
    /*
    	We try not to output crud when possible.
    */
    switch (sigargs[1]) {
      case LIB$_EOMWARN:
    	return(SS$_NORMAL);
      case RMS$_FNF:
      case 0x1512ba:		/* Variation of RMS$_FNF, extra bits set */
      case LIB$_KEYNOTFOU:
      case 0x1582fa:		/* Some variation of LIB$_KEYNOTFOU */
	return(SS$_CONTINUE);
      default:
	return(SS$_RESIGNAL);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * TclLoadFile --
 *
 *	Dynamically loads a binary code file into memory and returns
 *	the addresses of two procedures within that file, if they
 *	are defined.
 *
 * Results:
 *	A standard Tcl completion code.  If an error occurs, an error
 *	message is left in interp->result.  *proc1Ptr and *proc2Ptr
 *	are filled in with the addresses of the symbols given by
 *	*sym1 and *sym2, or NULL if those symbols can't be found.
 *
 * Side effects:
 *	New code suddenly appears in memory.
 *
 *----------------------------------------------------------------------
 */

int
TclLoadFile(interp, fileName, sym1, sym2, proc1Ptr, proc2Ptr)
    Tcl_Interp *interp;		/* Used for error reporting. */
    char *fileName;		/* Name of the file containing the desired
				 * code. */
    char *sym1, *sym2;		/* Names of two procedures to look up in
				 * the file's symbol table. */
    Tcl_PackageInitProc **proc1Ptr, **proc2Ptr;
				/* Where to return the addresses corresponding
				 * to sym1 and sym2. */
{
    char *s, initName[256];
    char libName[256];
    long status;
    struct {
	int	l;
    	void	*a;
    } dsc_libname, dsc_initname;

    /*
    	The following code always appends "SHR" to the name
    	of the file passed in. That means you won't  be
    	able to explicitly specify what to load.
    */
    strcpy(libName, fileName);
    strcat(libName, "SHR");
    setupp(libName);
    dsc_libname.a = libName;
    dsc_libname.l = strlen(libName);

    lib$establish(cond_handler);
    dsc_initname.a = sym1;
    dsc_initname.l = strlen(sym1);
    *proc1Ptr = 0;
    status = lib$find_image_symbol(&dsc_libname, &dsc_initname, proc1Ptr);
    if (!(status & 1)) {
	Tcl_AppendResult(interp, "could not load file \"", fileName,
    	 		 "\": ", strerror(EVMSERR, status), (char *) NULL);
	lib$revert();
   	return TCL_ERROR;
    }

    dsc_initname.a = sym2;
    dsc_initname.l = strlen(sym2);
    *proc2Ptr = 0;
    status = lib$find_image_symbol(&dsc_libname, &dsc_initname, proc2Ptr);
    lib$revert();
    return TCL_OK;
}

static void
setupp(char *s)
{
    char c;

    while (c = *s) {
    	if (islower(c))
	    *s = toupper(c);
	s++;
    }
}
